/************************************************************************/
/*									*/
/*  Paragraph manipulation routines.					*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>

#   include	"docBuf.h"
#   include	"docParaString.h"
#   include	"docEdit.h"

#   define	DEB_PARTICULES		0
#   define	SHOW_SELECTION_RANGE	0

/************************************************************************/

static int docEditParaFindHeadPositions(
			int *				pPartFrom,
			int *				pStroff0,
			int *				pSplitHead,
			int *				pMergeHead,
			const DocumentPosition *	dpHead,
			const int			partHead,
			int				addedAttrNr )
    {
    int				partFrom= partHead;
    const BufferItem *		paraBi= dpHead->dpBi;
    int				stroff0= dpHead->dpStroff;

    const TextParticule *	tp;

    int				splitHead= 0;
    int				mergeHead= 0;

    /*  B  */
    partFrom= partHead;
    stroff0= dpHead->dpStroff;
    if  ( partFrom < paraBi->biParaParticuleCount )
	{
	tp= paraBi->biParaParticules+ partFrom;

	if  ( tp->tpKind == DOCkindSPAN 	&&
	      addedAttrNr == tp->tpTextAttrNr	)
	    { stroff0= tp->tpStroff;	}

	if  ( partFrom > 0					&&
	      tp[-1].tpKind == DOCkindSPAN			&&
	      dpHead->dpStroff == tp->tpStroff			&&
	      addedAttrNr == tp[-1].tpTextAttrNr		)
	    {
	    mergeHead= 1; partFrom--;
	    stroff0= tp[-1].tpStroff;
	    }
	else{
	    /*  C  */
	    if  ( dpHead->dpStroff > tp->tpStroff		&&
		  addedAttrNr != tp->tpTextAttrNr		)
		{
		if  ( tp->tpKind != DOCkindSPAN )
		    { LDEB(tp->tpKind); return -1;	}

		splitHead= 1;
		}
	    }
	}
    else{
	partFrom= paraBi->biParaParticuleCount;
	tp= paraBi->biParaParticules+ partFrom;

	if  ( stroff0 != docParaStrlen( paraBi ) )
	    { LLDEB(stroff0,docParaStrlen(paraBi));	}

	if  ( paraBi->biParaParticuleCount > 0	&&
	      tp[-1].tpKind == DOCkindSPAN	&&
	      addedAttrNr == tp[-1].tpTextAttrNr)
	    {
	    partFrom--;
	    stroff0= tp[-1].tpStroff;
	    }
	}

    *pPartFrom= partFrom;
    *pStroff0= stroff0;
    *pSplitHead= splitHead;
    *pMergeHead= mergeHead;
    return 0;
    }

static int docEditParaFindTailPositions(
			int *				pPartUpto,
			int *				pStroff1,
			int *				pSplitTail,
			int *				pMergeTail,
			const BufferItem *		paraBi,
			const int			partTail,
			const int			stroffTail,
			int				addedAttrNr )
    {
    int				partUpto= partTail;
    int				stroff1= stroffTail;

    const TextParticule *	tp;

    int				splitTail= 0;
    int				mergeTail= 0;

    if  ( partUpto < paraBi->biParaParticuleCount )
	{
	tp= paraBi->biParaParticules+ partUpto;

	/* include next? */
	if  ( partUpto < paraBi->biParaParticuleCount- 1	&&
	      tp[1].tpKind == DOCkindSPAN			&&
	      stroffTail == tp->tpStroff+ tp->tpStrlen		&&
	      addedAttrNr == tp[1].tpTextAttrNr			)
	    {
	    mergeTail= 1; partUpto++;
	    stroff1= tp[1].tpStroff+ tp[1].tpStrlen;
	    }
	else{
	    if  ( tp->tpKind == DOCkindSPAN )
		{
		/*  E  */
		if  ( stroffTail >= tp->tpStroff		&&
		      addedAttrNr != tp->tpTextAttrNr		)
		    {
		    if  ( stroffTail > tp->tpStroff )
			{ splitTail= 1;	}
		    }
		else{
		    partUpto++;
		    stroff1= tp->tpStroff+ tp->tpStrlen;
		    }
		}
	    else{
		if  ( stroffTail != tp->tpStroff )
		    {
		    LLLDEB(stroffTail,tp->tpStroff,tp->tpStrlen);
		    return -1;
		    }
		}
	    }
	}
    else{
	partUpto= paraBi->biParaParticuleCount;
	if  ( stroff1 != docParaStrlen( paraBi ) )
	    { LLDEB(stroff1,docParaStrlen(paraBi));	}
	}

    *pPartUpto= partUpto;
    *pStroff1= stroff1;
    *pSplitTail= splitTail;
    *pMergeTail= mergeTail;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Replace text in a paragraph.					*/
/*									*/
/*  A)  It turns out that setting the irrelevant text attribute of an	*/
/*	empty insert to that of the text after the replacement makes	*/
/*	the code simpler. The -1 case is a paranoia attempt to obtain a	*/
/*	valid value in the empty paragraph case below.			*/
/*  B)  If we are at the head of the particule and the previous one	*/
/*	has the same text attribute as the insertion source include	*/
/*	the previous particule in the reformat. This prevents the	*/
/*	creation of adjacent particules with the same text attribute.	*/
/*  C)  If only part of the head particule is covered, decide to split	*/
/*	it.								*/
/*  D)  If we are at the tail of the particule and the next one has the	*/
/*	same text attribute as the insertion source include the next	*/
/*	particule in the reformat. (Why? See B). Remember that unless	*/
/*	the length of a particule equals zero, we should never be at	*/
/*	the end.							*/
/*  E)  If only part of the head particule is covered, decide to split	*/
/*	it.								*/
/*  F)  Clean all particules affected. Field particules should have	*/
/*	been cleaned by the caller. Remember that replacements never	*/
/*	contain only part of a field or bookmark. The caller takes care	*/
/*	of this.							*/
/*  G)  Insert the byes of the string into the paragraph.		*/
/*  H)  Redivide the new bytes and the old bytes from (B,D) in		*/
/*	particules.							*/
/*  I)  Delete any particules left over. Make sure that at the		*/
/*	paragraph has at least one particule.				*/
/*  J)  Shift the offsets of any particules after the replacement.	*/
/*  K)  Return tail position and particule numbers to the caller. If	*/
/*	the head or tail particules were merged, recalculate the	*/
/*	particule numbers. As the positions are inside a series of	*/
/*	spans, the special cases for different kinds of particules do	*/
/*	not apply.							*/
/*									*/
/************************************************************************/

int docParaReplaceText(	EditOperation *			eo,
			DocumentPosition *		dpTail,
			int *				pPartTail,
			int				paraNr,
			const DocumentPosition *	dpHead,
			int				partHead,
			unsigned int			stroffTail,
			int				partTail,
			const char *			addedString,
			unsigned int			addedStrlen,
			int				addedAttrNr )
    {
    int				rval= 0;

    int				partFrom;
    int				partUpto;

    int				splitHead= 0;
    int				splitTail= 0;
    int				mergeHead= 0;
    int				mergeTail= 0;

    int				stroff0;
    int				stroff1;
    int				partsFree;
    int				partsMade;

    BufferItem *		paraBi= dpHead->dpBi;
    int				stroffShift;

    /*  Paranoia: Check conventions  */
    if  ( partHead < 0 || partHead > paraBi->biParaParticuleCount )
	{
	LLDEB(partHead,paraBi->biParaParticuleCount);
	rval= -1; goto ready;
	}
    if  ( partTail < 0 || partTail > paraBi->biParaParticuleCount )
	{
	LLDEB(partTail,paraBi->biParaParticuleCount);
	rval= -1; goto ready;
	}
    if  ( partHead < paraBi->biParaParticuleCount		&&
	  paraBi->biParaParticules[partHead].tpStrlen > 0	&&
	  dpHead->dpStroff ==
		paraBi->biParaParticules[partHead].tpStroff+
		paraBi->biParaParticules[partHead].tpStrlen	)
	{ LLDEB(dpHead->dpStroff,partHead); rval= -1; goto ready;	}
    if  ( partTail < paraBi->biParaParticuleCount		&&
	  paraBi->biParaParticules[partTail].tpStrlen > 0	&&
	  stroffTail ==
		paraBi->biParaParticules[partTail].tpStroff+
		paraBi->biParaParticules[partTail].tpStrlen	)
	{ LDEB(partTail); rval= -1; goto ready;	}

    /*  A  */
    if  ( addedStrlen == 0 || addedAttrNr < 0 )
	{
	if  ( partTail < paraBi->biParaParticuleCount )
	    {
	    addedAttrNr= paraBi->biParaParticules[partTail   ].tpTextAttrNr;
	    }
	else{
	    addedAttrNr= paraBi->biParaParticules[partTail- 1].tpTextAttrNr;
	    }
	}

    /*  B,C  */
    if  ( docEditParaFindHeadPositions( &partFrom, &stroff0,
					    &splitHead, &mergeHead,
					    dpHead, partHead, addedAttrNr ) )
	{ LDEB(partHead); rval= -1; goto ready;	}

    /*  D,E  */
    if  ( docEditParaFindTailPositions( &partUpto, &stroff1,
					    &splitTail, &mergeTail,
					    paraBi, partTail,
					    stroffTail, addedAttrNr ) )
	{ LDEB(partTail); rval= -1; goto ready;	}

    /*  C  */
    if  ( splitHead )
	{
	TextParticule *		tp;
	TextParticule *		tpPart;

	if  ( docSplitTextParticule( &tpPart, &tp, paraBi,
						partHead, dpHead->dpStroff ) )
	    { LDEB(partHead); rval= -1; goto ready;	}

	partHead++; partFrom= partHead;
	partTail++; partUpto++;

	/* redivide will take care of splitting the tail */
	if  ( dpHead->dpStroff == stroffTail )
	    { splitTail= 0;	}
	}

    /*  E  */
    if  ( splitTail )
	{
	TextParticule *		tp;
	TextParticule *		tpNext;

	if  ( docSplitTextParticule( &tp, &tpNext, paraBi,
						    partTail, stroffTail ) )
	    { LDEB(partTail); rval= -1; goto ready;	}
	stroff1= stroffTail;
	partUpto= partTail+ 1;
	}

    /*  G  */
    stroffShift= 0;
    if  ( docParaStringReplace( &stroffShift, paraBi,
				    dpHead->dpStroff, stroffTail,
				    addedString, addedStrlen ) )
	{ LDEB(addedStrlen); rval= -1; goto ready;	}

    /*  F  */
    if  ( docEditCleanParticules( eo, paraBi, partFrom, partUpto ) )
	{ LLDEB(partFrom,partUpto); rval= -1; goto ready;	}

    /*  H  */
    partsFree= partUpto- partFrom;
    stroff1 += stroffShift;
    partsMade= docRedivideStringInParticules( paraBi,
					    stroff0, stroff1- stroff0,
					    partFrom, partsFree, addedAttrNr );
    if  ( partsMade < 0 )
	{ LDEB(partsMade); rval= -1; goto ready;	}

    /*  I  */
    if  ( partsMade < partsFree )
	{
	docDeleteParticules( paraBi, partFrom+ partsMade, partsFree- partsMade );

	if  ( paraBi->biParaParticuleCount == 0 )
	    {
	    TextParticule *	tp;

	    if  ( partFrom != 0 || docParaStrlen( paraBi ) != 0 )
		{ LLDEB(partFrom,docParaStrlen(paraBi));	}

	    tp= docInsertTextParticule( paraBi, partFrom,
					    docParaStrlen( paraBi ), 0,
					    DOCkindSPAN, addedAttrNr );
	    if  ( ! tp )
		{ XDEB(tp); rval= -1; goto ready;	}
	    partsMade++;

	    /* No.. Gives problems in the incremental formatter!
	    docInvalidateParagraphLayout( paraBi );
	    */
	    }
	}

    partTail += partsMade- partsFree;

    /*  J  */
    if  ( docEditShiftParticuleOffsets( eo, paraBi, paraNr,
			    partFrom+ partsMade, paraBi->biParaParticuleCount,
			    dpHead->dpStroff, stroffShift ) )
	{ LDEB(stroffShift);	}

#   if DEB_PARTICULES
    {
    TextParticule *	tp;
    int			p;

    appDebug( "\n" );

    p= 0; tp= paraBi->biParaParticules;
    while( p < paraBi->biParaParticuleCount )
	{
	const int	indent= 0;
	docListParticule( indent, "===", p, paraBi, tp );
	p++; tp++;
	}
    }
#   endif

    /*  K  */
    docSetDocumentPosition( dpTail, paraBi, dpHead->dpStroff+ addedStrlen );

    if  ( eo->eoHeadDp.dpBi == dpHead->dpBi && mergeHead )
	{
	const int	lastOne= 1;
	int		part;

	if  ( docFindParticuleOfPosition( &part, &(eo->eoHeadDp), lastOne ) )
	    { LDEB(eo->eoHeadDp.dpStroff); rval= -1; goto ready;	}
	if  ( eo->eoHeadDp.dpStroff >= docParaStrlen( eo->eoHeadDp.dpBi ) )
	    { part= eo->eoHeadDp.dpBi->biParaParticuleCount;	}

	eo->eoHeadParticule= part;
	}
    if  ( mergeTail )
	{
	const int	lastOne= 1;

	if  ( docFindParticuleOfPosition( &partTail, dpTail, lastOne ) )
	    { LDEB(dpTail->dpStroff); rval= -1; goto ready;	}
	if  ( dpTail->dpStroff >= docParaStrlen( dpTail->dpBi )	)
	    { partTail= dpTail->dpBi->biParaParticuleCount;	}
	}

    *pPartTail= partTail;

  ready:

    return rval;
    }

/************************************************************************/
/*									*/
/*  Delete text: Replace with nothing.					*/
/*									*/
/************************************************************************/

int docParaDeleteText(		EditOperation *			eo,
				DocumentPosition *		dpTail,
				int *				pPartTail,
				int				paraNr,
				const DocumentPosition *	dpHead,
				int				partHead,
				unsigned int			stroffTail,
				int				partTail )
    {
    return docParaReplaceText( eo, dpTail, pPartTail,
					paraNr, dpHead, partHead,
					stroffTail, partTail,
					(const char *)0, 0, -1 );
    }


/************************************************************************/
/*									*/
/*  Insert/Append the tail of a paragraph into another paragraph.	*/
/*									*/
/*  A)  Never copy the head field of the source.			*/
/*  B)  If the head of the source is a span with the same text		*/
/*	attributes as the target insert it as text. This takes care of	*/
/*	the correct distribution of the text over the spans.		*/
/*  C)  Manually copy a partial particule (Not handled by B)		*/
/*	Unless the arguments are silly, this only occurs for SPAN	*/
/*	particules. (As the implementation is identical to that of (B)	*/
/*	the cases are merged.)						*/
/*  D)  If the tail of the source is a span with the same text		*/
/*	attributes as the target we will insert it as text.		*/
/*  E)  Copy particules not covered by B,C,D=F.				*/
/*  F)  Insert tail as text (D). This takes care of the correct		*/
/*	distribution of the text over the spans. There is no partial	*/
/*	span case here as we completely copy the tail of the source.	*/
/*									*/
/************************************************************************/

int docParaInsertTail(		DocumentCopyJob *		dcj,
				DocumentPosition *		dpTail,
				int *				pPartTail,
				int				paraNrTo,
				const DocumentPosition *	dp_To,
				int				partTo,
				const DocumentPosition *	dp_From,
				int				partFrom )
    {
    EditOperation *		eo= dcj->dcjEditOperation;
    DocumentPosition		dpTo= *dp_To;
    DocumentPosition		dpFrom= *dp_From;
    int				rval= 0;

    BufferItem *		biTo= dpTo.dpBi;
    TextParticule *		tpTo;
    const BufferItem *		biFrom= dpFrom.dpBi;

    int				textAttrTo;

    int				copyLastAsText= 0;
    int				particulesCopied;

    /*  A  */
    if  ( biFrom->biParaListOverride > 0 )
	{
	DocumentField *		dfHead= (DocumentField *)0;
	DocumentSelection	dsInsideHead;
	DocumentSelection	dsAroundHead;
	int			partHead= -1;
	int			partTail= -1;

	if  ( docDelimitParaHeadField( &dfHead, &dsInsideHead, &dsAroundHead,
			    &partHead, &partTail, biFrom, dcj->dcjBdFrom ) )
	    { LDEB(biFrom->biParaListOverride);	}
	else{
	    if  ( dpFrom.dpStroff < dsAroundHead.dsTail.dpStroff )
		{ dpFrom.dpStroff=  dsAroundHead.dsTail.dpStroff;	}
	    if  ( partFrom < partTail+ 1 )
		{ partFrom=  partTail+ 1;	}
	    }
	}

    if  ( partFrom >= biFrom->biParaParticuleCount )
	{ goto done;	}

    /*  B,C  */
    if  ( partTo < biTo->biParaParticuleCount )
	{
	tpTo= biTo->biParaParticules+ partTo;

	if  ( dpTo.dpStroff > tpTo->tpStroff )
	    {
	    TextParticule *	tp;

	    if  ( docSplitTextParticule( &tp, &tpTo, dpTo.dpBi,
						    partTo, dpTo.dpStroff ) )
		{ LDEB(partTo); rval= -1; goto ready;	}

	    partTo++;
	    }
	}
    else{ tpTo= biTo->biParaParticules+ biTo->biParaParticuleCount- 1;	}

    {
    const TextParticule * const	tpFrom= biFrom->biParaParticules+ partFrom;

    textAttrTo= docMapTextAttributeNumber( dcj, tpFrom->tpTextAttrNr );

    if  ( tpFrom->tpKind == DOCkindSPAN			&&
	  tpTo->tpKind == DOCkindSPAN			&&
	  ( textAttrTo == tpTo->tpTextAttrNr	||		/*B*/
	    dpFrom.dpStroff > tpFrom->tpStroff	)	)	/*C*/
	{
	char *		addedString;
	int		addedStrlen;

	addedString= (char *)docParaString( biFrom, dpFrom.dpStroff );
	addedStrlen= tpFrom->tpStroff+ tpFrom->tpStrlen- dpFrom.dpStroff;

	if  ( docParaReplaceText( eo, &dpTo, &partTo,
				    paraNrTo, &dpTo, partTo,
				    dpTo.dpStroff, partTo,
				    addedString, addedStrlen, textAttrTo ) )
	    { LDEB(addedStrlen); rval= -1; goto ready;	}

	docSetDocumentPosition( &dpFrom, dpFrom.dpBi,
					tpFrom->tpStroff+ tpFrom->tpStrlen );
	partFrom++;
	}
    }

    if  ( partFrom >= biFrom->biParaParticuleCount )
	{ goto done;	}

    /*  D  */
    if  ( dpTo.dpStroff < docParaStrlen( dpTo.dpBi ) )
	{
	const TextParticule * const	tpFrom=
		    biFrom->biParaParticules+ biFrom->biParaParticuleCount- 1;

	tpTo= biTo->biParaParticules+ partTo;

	textAttrTo= docMapTextAttributeNumber( dcj, tpFrom->tpTextAttrNr );

	copyLastAsText= 0;

	if  ( partTo < biTo->biParaParticuleCount	&&
	      tpFrom->tpKind == DOCkindSPAN		&&
	      tpTo->tpKind == DOCkindSPAN		&&
	      textAttrTo == tpTo->tpTextAttrNr		)
	    { copyLastAsText= 1;	}
	}

    /*  E  */
    particulesCopied= biFrom->biParaParticuleCount- partFrom- copyLastAsText;
    if  ( particulesCopied > 0 )
        {
	int		particulesInserted= 0;
	int		charactersInserted= 0;

        if  ( docCopyParticules( dcj, biTo, biFrom, partTo, partFrom,
                                    particulesCopied,
                                    &particulesInserted,
                                    &charactersInserted ) )
            { LDEB(1); rval= -1; goto ready;       }

        partTo += particulesInserted;
        dpTo.dpStroff += charactersInserted;

	/* May point outside source */
	partFrom += particulesInserted;
        dpFrom.dpStroff += charactersInserted;
        }
    else{
        if  ( particulesCopied < 0 )
            { LDEB(particulesCopied); particulesCopied= 0;      }
        }

    /*  F  */
    if  ( copyLastAsText > 0 )
	{
	char *		addedString;
	int		addedStrlen;

	addedString= (char *)docParaString( biFrom, dpFrom.dpStroff );
	addedStrlen= docParaStrlen( biFrom )- dpFrom.dpStroff;

	if  ( docParaReplaceText( eo, &dpTo, &partTo,
				    paraNrTo, &dpTo, partTo,
				    dpTo.dpStroff, partTo,
				    addedString, addedStrlen, textAttrTo ) )
	    { LDEB(addedStrlen); rval= -1; goto ready;	}

	/* May point outside source */
	partFrom++; dpFrom.dpStroff += addedStrlen;
	}

  done:

    *dpTail= dpTo;
    *pPartTail= partTo;

  ready:

    return rval;
    }

static void docEditShiftSplitReferences(EditOperation *		eo,
					int			splitLevel )
    {
    const BufferItem *	oldParaBi= eo->eoTailDp.dpBi;
    const int		paraNr= docNumberOfParagraph( oldParaBi );
    const int		stroffFrom= eo->eoTailDp.dpStroff;
    const int		paraShift= 1;
    const int		stroffShift= -stroffFrom;
    const int		sectShift= splitLevel <= DOClevSECT;

    docEditShiftReferences( eo, &(eo->eoSelectionScope),
				    paraNr, stroffFrom,
				    sectShift, paraShift, stroffShift );

    eo->eoSectionsAdded += sectShift;

    return;
    }

static int docMoveTailToNew(	EditOperation *		eo,
				DocumentCopyJob *	dcj,
				BufferItem *		newBi,
				BufferItem *		oldBi )
    {
    TextParticule *	newTp;
    int			part= eo->eoTailParticule;
    int			stroff= eo->eoTailDp.dpStroff;

    int			stroffShift= 0;
    int			newStrlen= 0;
    int			partShift= 0;
    TextParticule *	tp;

    int			truncatedParticuleCount= part;

    /*  2  */
    if  ( docParaStringReplace( &stroffShift, newBi,
					docParaStrlen( newBi ),
					docParaStrlen( newBi ),
					(char *)docParaString( oldBi, stroff ),
					docParaStrlen( oldBi )- stroff ) )
	{ LLDEB(docParaStrlen(oldBi),stroff); return -1; }

    /*  4  */
    newStrlen= 0;
    tp= oldBi->biParaParticules+ part;
    while( part < oldBi->biParaParticuleCount )
	{
	switch( tp->tpKind )
	    {
	    default:
		LDEB(tp->tpKind);
		newStrlen += tp->tpStrlen; part++; tp++;
		continue;

	    case DOCkindFIELDEND:
		/*  5  */
		break;

	    case DOCkindFIELDSTART:
		break;

	    case DOCkindOBJECT:
	    case DOCkindSPAN:
	    case DOCkindTAB:
	    case DOCkindLINEBREAK:
	    case DOCkindPAGEBREAK:
	    case DOCkindCOLUMNBREAK:
		break;
	    }

	if  ( docCopyParticuleAndData( &newTp, dcj, newBi, partShift++,
					newStrlen, tp->tpStrlen, oldBi, tp ) )
	    { LDEB(part); return -1;	}

	newStrlen += tp->tpStrlen; part++; tp++;
	}

    oldBi->biParaParticuleCount= truncatedParticuleCount;
    docParaSetStrlen( oldBi, stroff );

    if  ( newBi->biParaParticuleCount == 0 )
	{
	tp= oldBi->biParaParticules+ oldBi->biParaParticuleCount- 1;

	newTp= docInsertTextParticule( newBi, -1, 0, 0,
				    DOCkindSPAN, tp->tpTextAttrNr );
	if  ( ! newTp )
	    { LDEB(newBi->biParaParticuleCount); return -1; }
	}

    if  ( oldBi->biParaParticuleCount == 0 )
	{
	tp= oldBi->biParaParticules;

	newTp= docInsertTextParticule( oldBi, -1, 0, 0,
				    DOCkindSPAN, tp->tpTextAttrNr );
	if  ( ! newTp )
	    { LDEB(oldBi->biParaParticuleCount); return -1; }
	}
    else{ docCheckNoBreakAsLast( eo, oldBi );	}


    return 0;
    }

/************************************************************************/
/*									*/
/*  Split a paragraph.							*/
/*									*/
/*  a)  Find the first particule that ends on or after the given	*/
/*	position. If several particules apply, choose the one with the	*/
/*	lowest level of field nesting.					*/
/*  1)  Insert a paragraph AFTER the current one.			*/
/*  1a) Copy indents and tab stops.					*/
/*  2)  Allocate space for the string of the successor.			*/
/*  3)  If the position is in the middle of a particule, split it.	*/
/*  4)  Move the rest of the particules to the new paragraph.		*/
/*  5)  Keep field ends at the split position in the old paragraph.	*/
/*	So do not insert field ends in the new paragraph. They are	*/
/*	reinserted at the end of the old paragraph.			*/
/*									*/
/************************************************************************/

int docSplitParaItem(		BufferItem **		pNewParaBi,
				EditOperation *		eo,
				int			splitLevel )
    {
    int				rval= 0;

    BufferItem *		oldParaBi= eo->eoTailDp.dpBi;
    BufferItem *		newParaBi;

    PropertyMask		ppChgMask;
    PropertyMask		ppUpdMask;

    DocumentCopyJob		dcj;

    docInitDocumentCopyJob( &dcj );

    if  ( docSet1DocumentCopyJob( &dcj, eo ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( ! eo->eoTailAtPartHead )
	{
	TextParticule *		tp;
	TextParticule *		newTp;

	if  ( docSplitTextParticule( &tp, &newTp, eo->eoTailDp.dpBi,
				eo->eoTailParticule, eo->eoTailDp.dpStroff ) )
	    { LDEB(1); rval= -1; goto ready;	}

	eo->eoTailParticule++;
	eo->eoTailAtPartHead= 1;
	}

#   if SHOW_SELECTION_RANGE
    appDebug( "SPLIT     %3d(%3d)\n",
		    docNumberOfParagraph( oldParaBi ), eo->eoTailDp.dpStroff );
#   endif

    /*  1  */
    newParaBi= docInsertItem( eo->eoBd, oldParaBi->biParent,
			oldParaBi->biNumberInParent+ 1, oldParaBi->biLevel );
    if  ( ! newParaBi )
	{ XDEB(newParaBi); rval= -1; goto ready;	}

    if  ( splitLevel < DOClevPARA )
	{
	BufferItem *		insBi;
	BufferItem *		aftBi;

	if  ( docSplitGroupItem( eo->eoBd, &insBi, &aftBi, oldParaBi->biParent,
				 oldParaBi->biNumberInParent+ 1, splitLevel ) )
	    { LDEB(1); return -1;	}

	docEditShiftSplitReferences( eo, splitLevel );

	if  ( aftBi && aftBi->biParent )
	    { docEditIncludeItemInReformatRange( eo, aftBi->biParent );	}
	else{ XDEB(aftBi);						}
	}
    else{
	docEditShiftSplitReferences( eo, splitLevel );
	}

    /*  2  */
    if  ( docMoveTailToNew( eo, &dcj, newParaBi, oldParaBi ) )
	{
	LLDEB(docParaStrlen(oldParaBi),eo->eoTailDp.dpStroff);
	/* Try to undo! */
	docDeleteItems( eo->eoBd, eo->eoEi, oldParaBi->biParent,
					oldParaBi->biNumberInParent+ 1, 1 );
	rval= -1; goto ready;
	}

    if  ( splitLevel >= DOClevPARA )
	{
	/*  2, 4  */
	docEditIncludeItemInReformatRange( eo, oldParaBi );
	docEditIncludeItemInReformatRange( eo, newParaBi );
	}

    utilPropMaskClear( &ppChgMask );
    utilPropMaskClear( &ppUpdMask );
    utilPropMaskFill( &ppUpdMask, PPprop_FULL_COUNT );

    /*  1a  */
    if  ( docEditUpdParaProperties( eo, &ppChgMask, newParaBi,
				&ppUpdMask, &(oldParaBi->biParaProperties),
				&(dcj.dcjAttributeMap) ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( oldParaBi->biParaListOverride > 0 )
	{ dcj.dcjBulletsCopied++;	}
    if  ( newParaBi->biParaListOverride > 0 )
	{
	eo->eoFieldUpdate |= FIELDdoLISTTEXT;
	eo->eoBulletsChanged++;
	}

    docInvalidateParagraphLayout( newParaBi );
    docInvalidateParagraphLayout( oldParaBi );

    *pNewParaBi= newParaBi;

  ready:

    docCleanDocumentCopyJob( &dcj );

    return rval;
    }

/************************************************************************/
/*									*/
/*  Insert a tab or another kind of special particule for something	*/
/*  that takes up some space in the document.				*/
/*									*/
/*  1)  Insert a space in the text string.				*/
/*  2)  Replace the empty particule in an empty paragraph.		*/
/*  3)  Can be inserted before the current particule.			*/
/*  4)  Can be inserted after the current particule.			*/
/*  5)  First split the particule,					*/
/*  6)  Then insert between the two halves.				*/
/*  7)  Shift particile offsets in the rest of the paragraph.		*/
/*									*/
/************************************************************************/

TextParticule * docEditParaSpecialParticule(
					EditOperation *		eo,
					int			kind )
    {
    TextParticule *	tp;
    int			partShift= 0;
    int			stroffShift= 0;

    tp= docParaSpecialParticule( eo, kind, &partShift, &stroffShift );

    docExtendParagraphAdjust( eo, eo->eoTailDp.dpBi, stroffShift );
    eo->eoTailDp.dpStroff += stroffShift;
    eo->eoTailParticule += partShift;

    return tp;
    }

TextParticule * docParaSpecialParticule(
				EditOperation *			eo,
				int				kind,
				int *				pPartShift,
				int *				pStroffShift )
    {
    const DocumentPosition *	dp= &(eo->eoTailDp);
    int				part= eo->eoTailParticule;

    TextParticule *		tp;
    TextParticule *		tpRet;
    int				stroffShift= 0;

    BufferItem *		paraBi= dp->dpBi;
    int				stroff= dp->dpStroff;
    int				paraNr= docNumberOfParagraph( paraBi );

    /*  1  */
    if  ( docParaStringReplace( &stroffShift, paraBi, stroff, stroff, " ", 1 ) )
	{ LDEB(stroff); return (TextParticule *)0;	}

    if  ( part == paraBi->biParaParticuleCount )
	{
	tp= paraBi->biParaParticules+ part- 1;

	tpRet= docInsertTextParticule( paraBi, part, stroff, 1,
					    kind, tp->tpTextAttrNr );
	if  ( ! tpRet )
	    { LXDEB(part,tpRet); return (TextParticule *)0;	}

	tpRet->tpObjectNumber= -1; /* set by caller if needed */

	*pPartShift= 1; *pStroffShift= stroffShift;
	return tpRet;
	}

    if  ( part < 0 || part >= paraBi->biParaParticuleCount )
	{
	LLDEB(part,paraBi->biParaParticuleCount);
	LLDEB(stroff,docParaStrlen( paraBi ));
	return (TextParticule *)0;
	}

    tp= paraBi->biParaParticules+ part;

    /*  2  */
    if  ( paraBi->biParaParticuleCount == 1			&&
	  docParaStrlen( paraBi ) == 1				&&
	  paraBi->biParaParticules->tpKind == DOCkindSPAN	)
	{
	paraBi->biParaParticules->tpKind= kind;
	paraBi->biParaParticules->tpStrlen= 1;

	*pPartShift= 0; *pStroffShift= stroffShift;
	part= 1;

	if  ( docEditShiftParticuleOffsets( eo, paraBi, paraNr,
			    part, paraBi->biParaParticuleCount, stroff, 1 ) )
	    { LDEB(1);	}

	return paraBi->biParaParticules;
	}

    /*  3  */
    if  ( stroff == tp->tpStroff )
	{
	tpRet= docInsertTextParticule( paraBi, part, stroff, 1,
					    kind, tp->tpTextAttrNr );
	if  ( ! tpRet )
	    { LXDEB(part,tpRet); return (TextParticule *)0;	}

	tpRet->tpObjectNumber= -1; /* set by caller if needed */

	*pPartShift= 1;
	part++;
	}
    else{
	/*  4  */
	if  ( stroff == tp->tpStroff+ tp->tpStrlen )
	    {
	    tpRet= docInsertTextParticule( paraBi, part+ 1, stroff, 1,
					    kind, tp->tpTextAttrNr );
	    if  ( ! tpRet )
		{ LXDEB(part,tpRet); return (TextParticule *)0;	}

	    tpRet->tpObjectNumber= -1; /* set by caller if needed */

	    *pPartShift= 2;
	    part += 2;
	    }
	else{
	    TextParticule *	tpNew;

	    /*  5  */
	    if  ( docSplitTextParticule( &tp, &tpNew, paraBi, part, stroff ) )
		{ LLDEB(part,stroff); return (TextParticule *)0;	}

	    /*  6  */
	    tpRet= docInsertTextParticule( paraBi, part+ 1, stroff, 1,
					    kind, tp->tpTextAttrNr );
	    if  ( ! tpRet )
		{ LXDEB(part+1,tpRet); return (TextParticule *)0;	}

	    tpRet->tpObjectNumber= -1; /* set by caller if needed */

	    *pPartShift= 2;
	    part += 2;
	    }
	}

    /*  7  */
    if  ( docEditShiftParticuleOffsets( eo, paraBi, paraNr,
			    part, paraBi->biParaParticuleCount, stroff, 1 ) )
	{ LDEB(1);	}

    *pStroffShift= stroffShift;

    return tpRet;
    }

/************************************************************************/
/*									*/
/*  Make a paragraph empty by deleting its contents.			*/
/*									*/
/************************************************************************/

int docEditMakeParagraphEmpty(		BufferItem *		paraBi,
					EditOperation *		eo )
    {
    TextParticule *	tp= paraBi->biParaParticules;
    int			textAttributeNumber= tp->tpTextAttrNr;

    int			bulletsDeleted= 0;
    int			paragraphCount= 0;

    docCleanItemObjects( &bulletsDeleted, &paragraphCount,
		eo->eoEi, eo->eoBd, paraBi, eo->eoCloseObject );

    eo->eoBulletsChanged += bulletsDeleted;

    docParaSetStrlen( paraBi, 0 );
    docDeleteParticules( paraBi, 0, paraBi->biParaParticuleCount );
    docDeleteLines( paraBi, 0, paraBi->biParaLineCount );

    if  ( ! docInsertTextParticule( paraBi, 0, 0, 0, DOCkindSPAN,
							textAttributeNumber ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Fixup for paragraphs that start with a \page or \column:		*/
/*  Remove the particule and remember the fact as a paragraph property.	*/
/*									*/
/************************************************************************/

int docEditFixParaBreakKind(	EditOperation *		eo,
				DocumentTree *		ei,
				BufferDocument *	bd,
				int			paraNr )
    {
    BufferItem *	paraBi= docGetParagraphByNumber( ei, paraNr );

    int			partHead= 0;
    int			stroffHead= 0;
    int			stroffShift= 0;

    int			part;
    TextParticule *	tp;

    if  ( ! paraBi )
	{ LXDEB(paraNr,paraBi); return -1;	}

    if  ( paraBi->biParaListOverride > 0 )
	{
	DocumentField *		dfHead= (DocumentField *)0;
	DocumentSelection	dsInsideHead;
	DocumentSelection	dsAroundHead;
	int			partBegin= -1;
	int			partEnd= -1;

	if  ( docDelimitParaHeadField( &dfHead, &dsInsideHead, &dsAroundHead,
					&partBegin, &partEnd, paraBi, bd ) )
	    { LDEB(paraBi->biParaListOverride);		}
	else{
	    stroffHead= dsAroundHead.dsTail.dpStroff;
	    partHead= partEnd+ 1;
	    }
	}

    tp= paraBi->biParaParticules+ partHead;
    for ( part= partHead; part < paraBi->biParaParticuleCount; tp++, part++ )
	{
	const int	paraShift= 0;

	if  ( tp->tpStroff < stroffHead )
	    { LLDEB(tp->tpStroff,stroffHead); continue;	}
	if  ( tp->tpStroff > stroffHead )
	    { break;					}

	switch( tp->tpKind )
	    {
	    case DOCkindPAGEBREAK:
		paraBi->biParaBreakKind= DOCibkPAGE;
		break;

	    case DOCkindCOLUMNBREAK:
		paraBi->biParaBreakKind= DOCibkCOL;
		break;

	    default:
		continue;
	    }

	/* remove the space from the paragraph contents */
	docParaStringReplace( &stroffShift, paraBi,
				tp->tpStroff, tp->tpStroff+ tp->tpStrlen,
				(char *)0, 0 );

	docShiftEditRange( &(eo->eoSelectedRange),
			    paraNr, tp->tpStroff, paraShift, stroffShift );

	docEditShiftParticuleOffsets( eo, paraBi, paraNr, 
				    part+ 1, paraBi->biParaParticuleCount,
				    tp->tpStroff+ tp->tpStrlen, stroffShift );

	docDeleteParticules( paraBi, part, 1 );
	break;
	}

    return 0;
    }

