/************************************************************************/
/*									*/
/*  Buffer manipulation routines.					*/
/*									*/
/************************************************************************/

#   include	"docBufConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>

#   include	<appDebugon.h>

#   include	<appUnit.h>

#   include	<docBuf.h>
#   include	<docParaString.h>
#   include	"docEdit.h"

#   define	DEB_PARTICULES		0
#   define	SHOW_SELECTION_RANGE	0

/************************************************************************/
/*									*/
/*  Replace a selection with a piece of text.				*/
/*									*/
/*  1)  If the selection spans more than one paragraph...		*/
/*  2)  Replace the tail of the first paragraph with the new text.	*/
/*  2b) Make sure that the bullet of the last paragraph is not included	*/
/*	in the tail that is to be appended to the first paragraph.	*/
/*  3)  Merge the two paragraphs. I.E: append what remains of the last	*/
/*	paragraph to the first paragraph.				*/
/*  4)  Erase all paragraphs after the beginning of the selection	*/
/*	upto and including the end of the selection.			*/
/*  5)  Otherwise just replace the text inside the selected paragraph.	*/
/*									*/
/************************************************************************/

int docReplaceSelection(EditOperation *			eo,
			const char *			addedString,
			int				addedStrlen,
			int				addedAttrNr )
    {
    int			rval= 0;
    DocumentCopyJob	dcjTail;
    int			paraNrBegin;

    paraNrBegin= docNumberOfParagraph( eo->eoHeadDp.dpBi );

    docInitDocumentCopyJob( &dcjTail );

    if  ( docEditDeleteReplacedFields( eo ) )
	{ LDEB(1);	}

    if  ( eo->eoHeadDp.dpStroff == 0			&&
	  eo->eoHeadDp.dpBi->biParaListOverride > 0	)
	{
	if  ( docRemoveParagraphFromList( eo->eoHeadDp.dpBi,
						    eo->eoEi, eo->eoBd ) )
	    { LDEB(1);	}
	}

    /*  1  */
    if  ( eo->eoHeadDp.dpBi != eo->eoTailDp.dpBi )
	{
	int			headPartCount;

	DocumentPosition	tailDp= eo->eoTailDp;
	int			tailParticule= eo->eoTailParticule;

	if  ( eo->eoHeadDp.dpBi->biParaTableNesting > 0		&&
	      eo->eoTailDp.dpBi->biParaTableNesting == 0	)
	    { docFlattenRow( eo->eoHeadDp.dpBi );	}

	/*  2  */
	headPartCount= eo->eoHeadDp.dpBi->biParaParticuleCount;
	if  ( docParaReplaceText( eo,
			&tailDp, &tailParticule,
			paraNrBegin, &(eo->eoHeadDp), eo->eoHeadParticule,
			docParaStrlen( eo->eoHeadDp.dpBi ), headPartCount,
			addedString, addedStrlen, addedAttrNr ) )
	    { LDEB(addedStrlen); rval= -1; goto ready;	}

	/*  3  */
	if  ( eo->eoTailParticule < eo->eoTailDp.dpBi->biParaParticuleCount )
	    {
	    if  ( docSet1DocumentCopyJob( &dcjTail, eo ) )
		{ LDEB(1); rval= -1; goto ready;	}

	    dcjTail.dcjCopyFields= 1;

	    /*  tail of EO is after insert */
	    if  ( docParaInsertTail( &dcjTail,
				&(eo->eoTailDp), &(eo->eoTailParticule),
				paraNrBegin, &tailDp, tailParticule,
				&(eo->eoTailDp), eo->eoTailParticule ) )
		{ LDEB(eo->eoTailParticule); rval= -1; goto ready;	}
	    }
	else{
	    eo->eoTailDp= tailDp;
	    eo->eoTailParticule= tailParticule;
	    }

	docSetEditPosition( &(eo->eoSelectedRange.erTail),
					    tailDp.dpBi, tailDp.dpStroff );

	/*  4  */
	if  ( docRemoveSelectionTail( eo ) )
	    { LDEB(1); rval= -1; goto ready;	}

	eo->eoTailDp= tailDp;
	eo->eoTailParticule= tailParticule;

	docCheckNoBreakAsLast( eo, eo->eoHeadDp.dpBi );

	docEditIncludeItemInReformatRange( eo, eo->eoHeadDp.dpBi );
	}
    else{
	int		tailStroffOld= eo->eoTailDp.dpStroff;
	int		tailStroffNew;

	/*  5  */
	if  ( docParaReplaceText( eo,
			&(eo->eoTailDp), &(eo->eoTailParticule),
			paraNrBegin, &(eo->eoHeadDp), eo->eoHeadParticule,
			eo->eoTailDp.dpStroff, eo->eoTailParticule,
			addedString, addedStrlen, addedAttrNr ) )
	    { LDEB(addedStrlen); rval= -1; goto ready;	}

	docCheckNoBreakAsLast( eo, eo->eoHeadDp.dpBi );

	tailStroffNew= eo->eoTailDp.dpStroff;
	docSetParagraphAdjust( eo, eo->eoHeadDp.dpBi,
				tailStroffNew- tailStroffOld, tailStroffNew );
	}

    docEditFinishStep( eo );

  ready:

    docCleanDocumentCopyJob( &dcjTail );

    return rval;
    }

int docDeleteSelection(	EditOperation *			eo )
    {
    return docReplaceSelection( eo, (const char *)0, 0, -1 );
    }

void docEditFinishStep(	EditOperation *			eo )
    {
    eo->eoTailAtPartHead= 0;
    if  ( eo->eoTailParticule >= eo->eoTailDp.dpBi->biParaParticuleCount )
	{ eo->eoTailAtPartHead= 1;	}
    else{
	const BufferItem *	tailBi;
	const TextParticule *	tailTp;

	tailBi= eo->eoTailDp.dpBi;
	tailTp= tailBi->biParaParticules+ eo->eoTailParticule;

	if  ( tailTp->tpStrlen > 0				&&
	      eo->eoTailDp.dpStroff == docParaStrlen( tailBi )	)
	    {
	    eo->eoTailParticule++;
	    eo->eoTailAtPartHead= 1;
	    }
	else{
	    eo->eoTailAtPartHead=
			    ( eo->eoTailDp.dpStroff == tailTp->tpStroff );
	    }
	}

    eo->eoHeadDp= eo->eoTailDp;
    eo->eoHeadParticule= eo->eoTailParticule;

    return;
    }

/************************************************************************/
/*									*/
/*  Make sure that the ranges on the EditOperation do not include the	*/
/*  deleted paragraphs.							*/
/*									*/
/************************************************************************/

void docEditAvoidDeletedParagraphs(	EditOperation *		eo,
					const SelectionScope *	ssRoot,
					int			paraFrom,
					int			paraUpto )
    {
    int			prune= 0;

    if  ( ! docSelectionSameScope( &(eo->eoSelectionScope), ssRoot ) )
	{ return;	}

    if  ( eo->eoSelectedRange.erHead.epParaNr >= paraFrom	&&
	  eo->eoSelectedRange.erHead.epParaNr <  paraUpto	)
	{ prune= 1;	}
    if  ( eo->eoSelectedRange.erTail.epParaNr >= paraFrom	&&
	  eo->eoSelectedRange.erTail.epParaNr <  paraUpto	)
	{ prune= 1;	}
    if  ( eo->eoReformatRange.erHead.epParaNr >= paraFrom	&&
	  eo->eoReformatRange.erHead.epParaNr <  paraUpto	)
	{ prune= 1;	}
    if  ( eo->eoReformatRange.erTail.epParaNr >= paraFrom	&&
	  eo->eoReformatRange.erTail.epParaNr <  paraUpto	)
	{ prune= 1;	}

    if  ( ! prune )
	{ return;	}

    if  ( paraFrom <= 1 )
	{
	EditPosition		epAfter;

	epAfter.epParaNr= paraUpto;
	epAfter.epStroff= 0;

	if  ( eo->eoSelectedRange.erHead.epParaNr >= paraFrom	&&
	      eo->eoSelectedRange.erHead.epParaNr <  paraUpto	)
	    { eo->eoSelectedRange.erHead= epAfter;	}
	if  ( eo->eoSelectedRange.erTail.epParaNr >= paraFrom	&&
	      eo->eoSelectedRange.erTail.epParaNr <  paraUpto	)
	    { eo->eoSelectedRange.erTail= epAfter;	}
	if  ( eo->eoReformatRange.erHead.epParaNr >= paraFrom	&&
	      eo->eoReformatRange.erHead.epParaNr <  paraUpto	)
	    { eo->eoReformatRange.erHead= epAfter;	}
	if  ( eo->eoReformatRange.erTail.epParaNr >= paraFrom	&&
	      eo->eoReformatRange.erTail.epParaNr <  paraUpto	)
	    { eo->eoReformatRange.erTail= epAfter;	}
	}
    else{
	DocumentPosition	dpBefore;
	EditPosition		epBefore;

	dpBefore.dpBi= docGetParagraphByNumber( eo->eoEi, paraFrom- 1 );
	if  ( ! dpBefore.dpBi || docLastPosition( &dpBefore, dpBefore.dpBi ) )
	    { LDEB(paraFrom); return;	}
	epBefore.epParaNr= paraFrom- 1;
	epBefore.epStroff= dpBefore.dpStroff;

	if  ( eo->eoSelectedRange.erHead.epParaNr >= paraFrom	&&
	      eo->eoSelectedRange.erHead.epParaNr <  paraUpto	)
	    { eo->eoSelectedRange.erHead= epBefore;	}
	if  ( eo->eoSelectedRange.erTail.epParaNr >= paraFrom	&&
	      eo->eoSelectedRange.erTail.epParaNr <  paraUpto	)
	    { eo->eoSelectedRange.erTail= epBefore;	}
	if  ( eo->eoReformatRange.erHead.epParaNr >= paraFrom	&&
	      eo->eoReformatRange.erHead.epParaNr <  paraUpto	)
	    { eo->eoReformatRange.erHead= epBefore;	}
	if  ( eo->eoReformatRange.erTail.epParaNr >= paraFrom	&&
	      eo->eoReformatRange.erTail.epParaNr <  paraUpto	)
	    { eo->eoReformatRange.erTail= epBefore;	}
	}
    }

/************************************************************************/
/*									*/
/*  Shift references after an insertion.				*/
/*									*/
/*  1)  If a paragraph is split, only shift the list number references	*/
/*	from the next paragraph.					*/
/*									*/
/************************************************************************/

void docEditShiftReferences(		EditOperation *		eo,
					const SelectionScope *	ssRoot,
					int			paraFrom,
					int			stroffFrom,
					int			sectShift,
					int			paraShift,
					int			stroffShift )
    {
    DocumentTree *		ei= eo->eoEi;
    int				sectFrom= -1;

#   if SHOW_SELECTION_RANGE
    appDebug( "REFS FROM %3d(%3d) BY %2d(%2d) <%s>\n",
			    paraFrom, stroffFrom,
			    paraShift, stroffShift,
			    docExternalKindStr( ssRoot->ssInExternalItem) );
#   endif

    if  ( docSelectionSameScope( &(eo->eoSelectionScope), ssRoot ) )
	{
#	if SHOW_SELECTION_RANGE
	appDebug( "> SEL     %3d(%3d) .. %3d(%3d)\n",
			    eo->eoSelectedRange.erHead.epParaNr,
			    eo->eoSelectedRange.erHead.epStroff,
			    eo->eoSelectedRange.erTail.epParaNr,
			    eo->eoSelectedRange.erTail.epStroff );
	appDebug( "> FORMAT  %3d(%3d) .. %3d(%3d)\n",
			    eo->eoReformatRange.erHead.epParaNr,
			    eo->eoReformatRange.erHead.epStroff,
			    eo->eoReformatRange.erTail.epParaNr,
			    eo->eoReformatRange.erTail.epStroff );
#	endif

	if  ( eo->eoParaAdjustParagraphNumber >= 0		&&
	      eo->eoParaAdjustParagraphNumber >= paraFrom	)
	    { eo->eoParaAdjustParagraphNumber += paraShift;	}

	docShiftEditRange( &(eo->eoReformatRange),
			    paraFrom, stroffFrom, paraShift, stroffShift );
	docShiftEditRange( &(eo->eoSelectedRange),
			    paraFrom, stroffFrom, paraShift, stroffShift );

#	if SHOW_SELECTION_RANGE
	appDebug( "< SEL     %3d(%3d) .. %3d(%3d)\n",
			    eo->eoSelectedRange.erHead.epParaNr,
			    eo->eoSelectedRange.erHead.epStroff,
			    eo->eoSelectedRange.erTail.epParaNr,
			    eo->eoSelectedRange.erTail.epStroff );
	appDebug( "< FORMAT  %3d(%3d) .. %3d(%3d)\n",
			    eo->eoReformatRange.erHead.epParaNr,
			    eo->eoReformatRange.erHead.epStroff,
			    eo->eoReformatRange.erTail.epParaNr,
			    eo->eoReformatRange.erTail.epStroff );
#	endif
	}

    if  ( ssRoot->ssInExternalItem == DOCinBODY )
	{ sectFrom= ssRoot->ssSectNr;	}

    docShiftFieldReferences( ei, sectFrom, paraFrom, stroffFrom,
					sectShift, paraShift, stroffShift );

    if  ( ssRoot->ssInExternalItem == DOCinBODY )
	{
	if  ( paraShift != 0 )
	    {
	    int				listParaFrom= paraFrom;

	    /*  1  */
	    if  ( paraShift > 0 && stroffFrom > 0 )
		{ listParaFrom++;	}

	    docShiftListTreeReferences( &(ei->eiListNumberTrees),
						    listParaFrom, paraShift );
	    docShiftListNodeReferences( &(ei->eiOutlineTree),
						    listParaFrom, paraShift );
	    }
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Shift offsets in particules, selection and notes to reflect a	*/
/*  change in the paragraph string.					*/
/*									*/
/************************************************************************/

int docEditShiftParticuleOffsets(	EditOperation *		eo,
					BufferItem *		paraBi,
					int			paraNr,
					int			partFrom,
					int			partUpto,
					int			stroffFrom,
					int			stroffShift )
    {
    BufferDocument *		bd= eo->eoBd;
    int				rval;

    docAdjustEditPositionOffsetB( &(eo->eoSelectedRange.erHead),
					    paraNr, stroffFrom, stroffShift );
    docAdjustEditPositionOffsetE( &(eo->eoSelectedRange.erTail),
					    paraNr, stroffFrom, stroffShift );

    rval= docShiftParticuleOffsets( bd, paraBi,
					    partFrom, partUpto, stroffShift );

    return rval;
    }

/************************************************************************/
/*									*/
/*  Clean particules that will no longer be needed.			*/
/*									*/
/************************************************************************/

int docEditCleanParticules(	EditOperation *		eo,
				BufferItem *		paraBi,
				int			partFrom,
				int			partUpto )
    {
    int			part;
    TextParticule *	tp;

    /*  11  */
    tp= paraBi->biParaParticules+ partFrom;
    for ( part= partFrom; part < partUpto; tp++, part++ )
	{
#	if DEB_PARTICULES
	const int	indent= 0;
	docListParticule( indent, "OLD", part, paraBi, tp );
#	endif

	if  ( eo->eoCloseObject )
	    { (*eo->eoCloseObject)( eo->eoBd, tp ); }

	docCleanParticuleObject( eo->eoBd, tp );

	if  ( tp->tpKind == DOCkindFIELDSTART	||
	      tp->tpKind == DOCkindFIELDEND	)
	    {
	    const DocumentField *	df;

	    /* Not yet deleted? */
	    df= docGetFieldByNumber( &(eo->eoBd->bdFieldList),
						    tp->tpObjectNumber );
	    if  ( df && df->dfFieldNumber >= 0 )
		{
		LSLDEB(part,docKindStr(tp->tpKind),df->dfFieldNumber);
		docListFieldParticule( 0, "FLD", part, paraBi, tp, df );
		return -1;
		}
	    }

	tp->tpObjectNumber= -1;
	tp->tpKind= DOCkindUNKNOWN;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Merge the selection into one paragraph.				*/
/*									*/
/************************************************************************/

static int docIsIndentationParticule(	const BufferItem *	bi,
					const TextParticule *	tp )
    {
    const unsigned char *	s= docParaString( bi, tp->tpStroff );
    int				i;

    if  ( tp->tpKind == DOCkindTAB )
	{ return 1;	}

    if  ( tp->tpKind != DOCkindSPAN )
	{ return 0;	}

    for ( i= 0; i < tp->tpStrlen; i++ )
	{
	if  ( *(s++) != ' ' )
	    { return 0;	}
	}

    return 1;
    }

int docMergeParagraphsInSelection(	EditOperation *	eo )
    {
    int				rval= 0;
    BufferDocument *		bd= eo->eoBd;

    BufferItem *		biTo;
    BufferItem *		biFrom;

    int				beforeEnd;

    DocumentCopyJob		dcj;

    docInitDocumentCopyJob( &dcj );

    beforeEnd= docParaStrlen( eo->eoTailDp.dpBi )-
					eo->eoSelectedRange.erTail.epStroff;

    if  ( docSet1DocumentCopyJob( &dcj, eo ) )
	{ LDEB(1); rval= -1; goto ready;	}

    biTo= eo->eoHeadDp.dpBi;
    biFrom= docNextParagraph( biTo );
    if  ( ! biTo )
	{ XDEB(biTo); rval= -1; goto ready;	}

    for (;;)
	{
	int	partFrom= 0;

	int	particulesInserted= 0;
	int	charactersCopied= 0;

	if  ( biFrom->biParaListOverride > 0 )
	    {
	    DocumentField *	dfHead= (DocumentField *)0;
	    DocumentSelection	dsInsideHead;
	    DocumentSelection	dsAroundHead;
	    int			partBegin= -1;
	    int			partEnd= -1;

	    if  ( docDelimitParaHeadField( &dfHead,
					&dsInsideHead, &dsAroundHead,
					&partBegin, &partEnd, biFrom, bd ) )
		{ LDEB(biFrom->biParaListOverride);	}
	    else{
		if  ( partFrom <= partEnd )
		    { partFrom= partEnd+ 1;	}
		}
	    }

	while( partFrom < biFrom->biParaParticuleCount- 1		&&
	       docIsIndentationParticule( biFrom,
				biFrom->biParaParticules+ partFrom )	)
	    { partFrom++; }

	if  ( partFrom < biFrom->biParaParticuleCount )
	    {
	    int			toCount= biTo->biParaParticuleCount;
	    int			toStrlen= docParaStrlen( biTo );
	    const unsigned char *
				toString= docParaString( biTo, 0 ); /*SHAME*/

	    if  ( toCount > 0						&&
		  biTo->biParaParticules[toCount-1].tpKind ==
							DOCkindSPAN	&&
		  toString[toStrlen-1] != ' '				)
		{
		int		stroffShift= 0;

		if  ( docParaStringReplace( &stroffShift, biTo,
					    toStrlen, toStrlen, " ", 1 ) )
		    { LDEB(toStrlen); rval= -1; goto ready;	}

		biTo->biParaParticules[toCount-1].tpStrlen++;
		}

	    if  ( docCopyParticules( &dcj, biTo, biFrom,
			biTo->biParaParticuleCount, partFrom,
			biFrom->biParaParticuleCount- partFrom,
			&particulesInserted, &charactersCopied ) )
		{ LDEB(biFrom->biParaParticuleCount); rval= -1; goto ready; }
	    }

	if  ( biFrom == eo->eoTailDp.dpBi )
	    { break;	}

	biFrom= docNextParagraph( biFrom );
	if  ( ! biFrom )
	    { XDEB(biFrom); rval= -1; goto ready;	}
	}

    biTo= eo->eoHeadDp.dpBi;
    if  ( ! biTo )
	{ XDEB(biTo); rval= -1; goto ready;	}

    if  ( docLastPosition( &(eo->eoTailDp), eo->eoHeadDp.dpBi ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( docRemoveSelectionTail( eo ) )
	{ LDEB(1); rval= -1; goto ready;	}

    eo->eoNotesAdded += dcj.dcjNotesCopied;
    eo->eoBulletsChanged += dcj.dcjBulletsCopied;

    docEditIncludeItemInReformatRange( eo, biTo );

  ready:

    docCleanDocumentCopyJob( &dcj );

    return rval;
    }

/************************************************************************/
/*									*/
/*  Delete all fields that are embedded in the content that we will	*/
/*  replace in this edit operation.					*/
/*									*/
/************************************************************************/

int docEditDeleteReplacedFields(	EditOperation *		eo )
    {
    DocumentSelection	dsBalanced;
    int			headPart= eo->eoHeadParticule;
    int			tailPart= eo->eoTailParticule;

    dsBalanced.dsHead= eo->eoHeadDp;
    dsBalanced.dsTail= eo->eoTailDp;

    if  ( docDeleteFieldsInRange( eo, headPart, tailPart, &dsBalanced ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

