/************************************************************************/
/*									*/
/*  Text Editor Buffer structure.					*/
/*									*/
/************************************************************************/

#   ifndef	DOC_BUF_H
#   define	DOC_BUF_H

#   include	<time.h>

#   include	<bitmap.h>
#   include	<utilDocFont.h>
#   include	<psPostScriptFontList.h>
#   include	"docShape.h"
#   include	<geo2DInteger.h>
#   include	<sioGeneral.h>
#   include	<utilMemoryBuffer.h>
#   include	<utilTextAttributeAdmin.h>

#   include	"docBorderPropertyAdmin.h"
#   include	"docItemShadingAdmin.h"
#   include	"docFramePropertiesAdmin.h"

#   include	"docParaProperties.h"
#   include	"docRowProperties.h"
#   include	"docSectProperties.h"
#   include	"docStyleSheet.h"
#   include	"docDocumentProperties.h"
#   include	"docShapeProperties.h"
#   include	"docDocumentField.h"
#   include	"docDocumentFieldList.h"
#   include	"docDrawingShapeList.h"
#   include	"docField.h"
#   include	"docObject.h"
#   include	"docSelect.h"
#   include	"docExternalItem.h"
#   include	"docDocumentNote.h"
#   include	"docListTable.h"
#   include	"docTextParticule.h"
#   include	"docTextLine.h"

#   include	"docEditPosition.h"
#   include	"docEditRange.h"

#   include	"docParaItem.h"
#   include	"docCellItem.h"
#   include	"docRowItem.h"
#   include	"docSectItem.h"

/************************************************************************/
/*									*/
/*  Forward declarations of structs.					*/
/*									*/
/************************************************************************/

struct BufferDocument;
struct BufferItem;

/************************************************************************/
/*									*/
/*  A BufferItem.							*/
/*									*/
/*  A BufferItem is a node in what we fashionably call the document	*/
/*  object model. In a typical XML type DOM, the buffer items would be	*/
/*  element nodes. An item at level DOClevPARA is very much comparable	*/
/*  to a <div> in HTML terms. The HTML equivalent of a <span> does not	*/
/*  exist inside the Ted source code. Spans are implemented by a series	*/
/*  of text particules of kind DOCkindSPAN with equal text attributes.	*/
/*									*/
/*  1)  Fields for managing the hierarchy.				*/
/*  2)  Geometry information that is set by the formatter.		*/
/*	The insets are an amount of space that the item claims for	*/
/*	itself. The content of the item (children, or lines) appears	*/
/*	between the insets. Typically, the insets are used for spacing	*/
/*	or borders.							*/
/*  3)  Information that is specific to the kind of item.		*/
/*									*/
/************************************************************************/

typedef struct BufferItem
    {
				/*  1  */
    unsigned char		biLevel;
    unsigned char		biInExternalItem;
    struct BufferItem *		biParent;
    struct BufferItem **	biChildren;
    short int			biChildCount;
    short int			biNumberInParent;
    int				biLeftParagraphs;

				/*  2  */
    LayoutPosition		biTopPosition;
    LayoutPosition		biBelowPosition;

				/*  3  */
    union			BufferItemUnion
	{
	ParaItem		biuPara;
	CellItem		biuCell;
	SectItem		biuSect;
	RowItem			biuRow;
	} BIU;

    } BufferItem;

# define docIsCellItem( bi )	( (bi)->biLevel == DOClevCELL )
# define docIsParaItem( bi )	( (bi)->biLevel == DOClevPARA )
# define docIsSectItem( bi )	( (bi)->biLevel == DOClevSECT )
# define docIsRowItem( bi )	( (bi)->biLevel == DOClevROW && \
				  (bi)->biRowCellCount > 0 )

/************************************************************************/
/*									*/
/*  The document as a whole.						*/
/*									*/
/************************************************************************/

typedef struct NotesList
    {
    DocumentNote *		nlNotes;
    int				nlNoteCount;
    } NotesList;

typedef struct BufferDocument
    {
    NumberedPropertiesList	bdTextAttributeList;
    NumberedPropertiesList	bdBorderPropertyList;
    NumberedPropertiesList	bdItemShadingList;
    NumberedPropertiesList	bdFramePropertyList;
    PagedList			bdObjectList;

    DocumentTree		bdBody;
    DocumentProperties		bdProperties;
				/**
				 * Default font for plain text runs. Is not 
				 * the ducument property itself, but a 
				 * derived attribute.
				 */
    int				bdDefaultFont;

    DocumentStyleSheet		bdStyleSheet;
    DocumentFieldList		bdFieldList;
    DrawingShapeList		bdShapeList;

    NotesList			bdNotesList;

    DocumentTree		bdEiFtnsep;
    DocumentTree		bdEiFtnsepc;
    DocumentTree		bdEiFtncn;

    DocumentTree		bdEiAftnsep;
    DocumentTree		bdEiAftnsepc;
    DocumentTree		bdEiAftncn;

    int				bdAutoHyphenate;
				    /************************************/
				    /*  Used to smuggle property into	*/
				    /*  line formatter.			*/
				    /************************************/
    } BufferDocument;

/************************************************************************/
/*									*/
/*  Statistics about a document. Used in the 'Document Properties'	*/
/*  dialog.								*/
/*									*/
/************************************************************************/

typedef struct DocumentStatistics
    {
    int		dsPageCount;
    int		dsParagraphCount;
    int		dsLineCount;
    int		dsWordCount;
    int		dsCharacterCount;
    } DocumentStatistics;

/************************************************************************/
/*									*/
/*  Routine declarations.						*/
/*									*/
/************************************************************************/

#   include	"docDebug.h"

extern int docGetDefaultFont(		BufferDocument *	bd );

extern BufferDocument *	docPlainReadFile( SimpleInputStream *	sis,
				    int *			pMxL,
				    const DocumentGeometry *	dg );

extern BufferDocument * docNewFile(
				TextAttribute *			taDefault,
				const char *			defaultFontName,
				int				defaultFontSize,
				const PostScriptFontList *	psfl,
				const DocumentGeometry * 	dg );

extern int docHtmlSaveDocument(		SimpleOutputStream *	sos,
					BufferDocument *	bd,
					int			asMimeAggr,
					const char *		mimeBoundary,
					const char *		filename );

extern int docPlainSaveDocument(SimpleOutputStream *		sos,
				BufferDocument *		bd,
				const DocumentSelection *	ds,
				int				fold,
				int				closed );

extern TextLine * docInsertTextLine(	BufferItem *	bi,
					int		line );

extern void docCleanItem(	BufferDocument *	bd,
				DocumentTree *		ei,
				BufferItem *		bi );

extern void docFreeItem(	BufferDocument *	bd,
				DocumentTree *		ei,
				BufferItem *		bi );

extern void docFreeDocument( BufferDocument *	bd );
extern BufferDocument * docNewDocument( void );

extern BufferItem * docInsertItem(	const BufferDocument *	bd,
					BufferItem *		parent,
					int			n,
					ItemLevel		level );

extern void docDeleteItems(		BufferDocument *	bd,
					DocumentTree *		ei,
					BufferItem *		bi,
					int			first,
					int			count );

extern TextParticule *	docInsertTextParticule(	BufferItem *	bi,
						int		n,
						int		off,
						int		len,
						int		kind,
						int		textAttrNr );

extern TextParticule * docCopyParticule(	BufferItem *		bi,
						int			n,
						int			off,
						int			len,
						int			kind,
						const TextParticule *	from );

extern int docSplitTextParticule(
				TextParticule **		pTpPart,
				TextParticule **		pTpNext,
				BufferItem *			paraBi,
				int				part,
				int				stroff );

extern void docDeleteParticules(	BufferItem *	bi,
					int		first,
					int		count );

extern void docDeleteLines(		BufferItem *	bi,
					int		first,
					int		count );

extern void docInitTableRectangle(	TableRectangle *	tr );

extern int docSplitGroupItem(	BufferDocument *	bd,
				BufferItem **		pNewBi,
				BufferItem **		pOldBi,
				BufferItem *		oldBi,
				int			n,
				int			level );

extern int docMergeGroupItems(		BufferItem *	to,
					BufferItem *	from );

extern BufferItem * docNextParagraph(	BufferItem *	bi );
extern BufferItem * docPrevParagraph(	BufferItem *	bi );

extern BufferItem * docNextSection(	BufferItem *	bi );
extern BufferItem * docPrevSection(	BufferItem *	bi );

extern int docNextPosition(	DocumentPosition *	dp );

extern int docPrevPosition(	DocumentPosition *	dp );

extern int docNextWord(		DocumentPosition *	dp );
extern int docPrevWord(		DocumentPosition *	dp );

extern int docParaBegin(	DocumentPosition *	dp );
extern int docParaEnd(		DocumentPosition *	dp );

extern int docLineDown(		TextLine **		pTl,
				int *			pLine,
				DocumentPosition *	dp,
				int			line );

extern int docLineUp(		TextLine **		pTl,
				int *			pLine,
				DocumentPosition *	dp,
				int			line );

extern int docFirstPosition(	DocumentPosition *	dp,
				BufferItem *		bi );

extern int docLastPosition(	DocumentPosition *	dp,
				BufferItem *		bi );

extern int docFirstDocumentPosition(	DocumentPosition *	dp,
					BufferDocument *	bd );

extern void docAvoidParaHeadField(	DocumentPosition *	dp,
					int *			pPart,
					const BufferDocument *	bd );

extern int docLastDocumentPosition(	DocumentPosition *	dp,
					BufferDocument *	bd );

extern void docSetParaSelection( DocumentSelection *	ds,
				BufferItem *		bi,
				int			direction,
				int			start,
				int			length );

extern int docComparePositions(	const DocumentPosition *	dp1,
				const DocumentPosition *	dp2 );

extern int docCompareItemPositions(	const BufferItem *	bi1,
					const BufferItem *	bi2 );

extern int docSetDocumentPosition(	DocumentPosition *	dp,
					BufferItem *		bi,
					int			stroff );

extern int docFindParticuleOfPosition(
				int *				pPart,
				const DocumentPosition *	dp,
				int				lastOne );

extern int docFindLineOfPosition(
				int *				pLine,
				const DocumentPosition *	dp,
				int				lastOne );

extern void docDeleteItem(		BufferDocument *	bd,
					DocumentTree *		ei,
					BufferItem *		bi );

extern void docDeleteExternalItem(	BufferDocument *	bd,
					DocumentTree *		ei );

extern void docInitItem(	BufferItem *		bi,
				BufferItem *		parent,
				const BufferDocument *	bd,
				int			numberInParent,
				int			level,
				int			inExternalItem );

extern TextParticule *	docAppendObject(	BufferDocument *	bd,
						BufferItem *		paraBi,
						const TextAttribute *	ta );

extern InsertedObject * docGetObject(	const BufferDocument *	bd,
					int			n );

extern void docCleanParticuleObject(	BufferDocument *	bd,
					TextParticule *		tp );

extern InsertedObject *	docClaimObject(	int *			pNr,
					BufferDocument *	bd );

extern InsertedObject * docClaimObjectCopy(
					BufferDocument *	bd,
					int *			pNr,
					const InsertedObject *	ioFrom );

extern int docDelimitTable(	BufferItem *		paraBi,
				BufferItem **		pSectBi,
				int *			pCol,
				int *			pRow0,
				int *			pRow,
				int *			pRow1 );

extern void docTableDetermineCellspans(	int *			pRowspan,
					int *			pColspan,
					const BufferItem *	rowBi,
					int			col );

extern int docSaveSpecialParticule(	BufferDocument *	bd,
					BufferItem *		paraBi,
					const TextAttribute *	ta,
					int			kind );

extern void docLineSelection(	DocumentSelection *	dsLine,
				int *			pPartLineBegin,
				int *			pPartLineEnd,
				const BufferItem *	bi,
				int			line );

extern void docWordSelection(
			DocumentSelection *		dsWord,
			int *				pIsObject,
			const DocumentPosition *	dpAround );

extern void docInitDocumentStatistics(		DocumentStatistics *	ds );

extern void docCollectDocumentStatistics(	DocumentStatistics *	ds,
						const BufferDocument *	bd );

extern int docWhatPageHeader(	DocumentTree **			pEi,
				int *				pIsEmpty,
				BufferItem *			sectBi,
				int				page,
				const BufferDocument *		bd );

extern int docWhatPageFooter(	DocumentTree **			pEi,
				int *				pIsEmpty,
				BufferItem *			sectBi,
				int				page,
				const BufferDocument *		bd );

extern int docWhatPagesForHeaderFooter(
				const DocumentProperties *	dp,
				const SectionProperties *	sp,
				int				which );

extern BufferItem * docGetSelectionRoot(
				DocumentTree **			pEi,
				BufferItem **			pBodySectBi,
				BufferDocument *		bd,
				const DocumentSelection *	ds );

extern int docGetHeaderFooter(	DocumentTree **			pEi,
				BufferItem **			pBodySectBi,
				const DocumentSelection *	ds,
				BufferDocument *		bd,
				int				which );

extern int docInquireHeadersFooters(
				int *				pDocHasHeaders,
				int *				pDocHasFooters,
				const BufferItem *		docBi );

extern int docMakeExternalItem(
				BufferDocument *		bd,
				DocumentTree *			ei,
				const SelectionScope *		ss,
				const SectionProperties *	sp );

extern BufferItem * docMakeExternalParagraph(
				BufferDocument *		bd,
				DocumentTree *			ei,
				const BufferItem *		bodyBi,
				int				txtAttrNr,
				int				ownerNumber,
				int				inExtIt );

extern BufferItem * docMakeExternalSection(
				BufferDocument *		bd,
				DocumentTree *			ei,
				const BufferItem *		bodyBi,
				int				ownerNumber,
				int				inExtIt );

extern BufferItem * docInsertEmptyParagraph(
				BufferDocument *	bd,
				BufferItem *		bi,
				int			textAttributeNumber );

extern int docIntersectTableRectangle(	TableRectangle *		tr,
					const TableRectangle *		tr1,
					const TableRectangle *		tr2 );

extern void docExpandTableRectangleToWholeTable( TableRectangle *	tr );
extern void docExpandTableRectangleToWholeRows(	TableRectangle *	tr );
extern void docExpandTableRectangleToWholeColumns( TableRectangle *	tr );

extern int docShiftTableRectangleByRows( TableRectangle *	tr,
					int			rows );
extern int docShiftTableRectangleByColumns( TableRectangle *	tr,
					int			columns );

extern int docTableRectangleSelection(	DocumentSelection *	ds,
					BufferItem **		pSelParentBi,
					BufferDocument *	bd,
					const TableRectangle *	tr );

extern int docMergeDocumentLists(
				int **				pFontMap,
				int **				pListStyleMap,
				const int *			colorMap,
				BufferDocument *		bdTo,
				BufferDocument *		bdFrom );

extern int docMergeColorTables(
				int **				pColorMap,
				BufferDocument *		bdTo,
				const BufferDocument *		bdFrom );

extern int docFindBookmarkInDocument(	DocumentSelection *	dsInside,
					int *			pHeadPart,
					int *			pTailPart,
					BufferDocument *	bd,
					const char *		markName,
					int			markSize );

extern int docSurroundTextSelectionByField(
				DocumentField **		pDf,
				DocumentSelection *		dsInside,
				DocumentSelection *		dsAround,
				int *				pHeadPart,
				int *				pTailPart,
				BufferDocument *		bd,
				DocumentTree *			ei,
				int				singlePara,
				const DocumentSelection *	ds,
				const PropertyMask *		taSetMask,
				const TextAttribute *		taSet );

extern DocumentField * docMakeField(
				BufferDocument *		bd,
				DocumentTree *			ei,
				const DocumentSelection *	dsInput,
				int				part0,
				int				part1,
				int				attr0,
				int				attr1 );

extern void docConstrainSelectionToOneParagraph( int *		pHeadMoved,
					int *			pTailMoved,
					DocumentSelection *	ds );

extern int docParaNextWord(		const BufferItem *	paraBi,
					int			stroff );

extern int docParaPrevWord(		const BufferItem *	paraBi,
					int			stroff );

extern int docParaHeadOfWord(		int *				pStroff,
					const BufferItem *		paraBi,
					int				part );

extern int docParaTailOfWord(		int *				pStroff,
					const BufferItem *		paraBi,
					int				part );

extern int docParaPastLastNonBlank(	const BufferItem *	paraBi,
					int			from,
					int			upto );

extern int docInsertNote(		DocumentNote **		pDn,
					BufferDocument *	bd,
					DocumentField *		dfNote,
					int			autoNumber );

extern int docMakeNote(	DocumentNote **			pDn,
			BufferDocument *		bd,
			DocumentField *			dfNote,
			const BufferItem *		bodyBi,
			int				extItKind,
			int				fieldKind );

extern DocumentField * docGetSelectedNote(
				DocumentNote **			pDn,
				BufferDocument *		bd,
				const DocumentSelection *	ds );

extern void docRenumberNotes(		int *			pChanged,
					BufferDocument *	bd );

extern DocumentTree *	docSectionHeaderFooter(	BufferItem *	sectBi,
						int		which );

DocumentTree *	docDocumentNoteSeparator(
					BufferDocument *	bd,
					int			which );

extern int docGetRootForItem(		DocumentTree **		pEi,
					BufferItem **		pBodySectBi,
					BufferDocument *	bd,
					BufferItem *		bi );

extern int docGetRootOfSelectionScope(	DocumentTree **		pEi,
					BufferItem **		pBodySectBi,
					BufferDocument *	bd,
					const SelectionScope *	ss );

extern int docGetTopOfColumn(	DocumentPosition *		dp,
				int *				pLineTop,
				int *				pPartTop,
				BufferDocument *		bd,
				int				page,
				int				column );

extern int docGetFirstInColumnForItem(
				DocumentPosition *		dp,
				int *				pLineTop,
				int *				pPartTop,
				BufferItem *			bi,
				int				page,
				int				column );

extern int docParaGetFirstInColumn(
				DocumentPosition *		dp,
				int *				pLineTop,
				int *				pPartTop,
				BufferItem *			paraBi,
				int				lineUpto,
				int				page,
				int				column );

extern int docGetBottomOfColumn(DocumentPosition *		dp,
				int *				pPartBot,
				BufferDocument *		bd,
				int				page,
				int				column );

extern int docGetLastInColumnForItem(
				DocumentPosition *		dp,
				int *				pLineBot,
				int *				pPartBot,
				BufferItem *			bi,
				int				page,
				int				column );

extern BufferItem * docInsertRowItem(
				BufferDocument *	bd,
				BufferItem *		sectBi,
				int			n,
				const RowProperties *	rp,
				int			textAttributeNumber );

extern int docSectionHeaderFooterFirstPage(
				int *				pUsedInDocument,
				const BufferItem *		bodySectBi,
				int				which,
				const DocumentProperties *	dp );

extern int docSelectWholeSection(	DocumentSelection *	bs,
					BufferDocument *	bd,
					int			direction );

extern int docSelectWholeTree(		DocumentSelection *	ds,
					BufferDocument *	bd );

extern int docNumberOfParagraph(	const BufferItem *	paraBi );

extern BufferItem * docGetParagraphByNumber(	const DocumentTree *	ei,
						int			n );

extern int docShiftParticuleOffsets(	BufferDocument *	bd,
					BufferItem *		paraBi,
					int			partFrom,
					int			partUpto,
					int			stroffShift );

extern void docCleanNote(		BufferDocument *	bd,
					DocumentNote *		dn );

extern int docCountNotes(		const BufferDocument *	bd );

extern DocumentField * docGetFirstNoteFromColumn(
					DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			page,
					int			column,
					int			extItKind );

extern DocumentField * docGetFirstNoteInColumn(
					DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			page,
					int			column,
					int			extItKind );

extern DocumentField * docGetLastNoteInColumn(
					DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			page,
					int			column,
					int			extItKind );

extern DocumentField * docGetFirstNoteOnPage(
					DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			page,
					int			extItKind );

extern DocumentField * docGetFirstNoteOfSection(
					DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			sect,
					int			extItKind );

extern DocumentField * docGetFirstNoteOfDocument(
					DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			extItKind );

extern DocumentField * docGetLastNoteOfDocument(
					DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			extItKind );

extern DocumentField * docGetNextNoteInSection(
					DocumentNote **		pDn,
					const BufferDocument *	bd,
					int			sect,
					DocumentField *		df,
					int			extItKind );

extern DocumentField * docGetNextNoteInDocument(
					DocumentNote **		pDn,
					const BufferDocument *	bd,
					DocumentField *		df,
					int			extItKind );

extern DocumentField * docGetPrevNoteInDocument(
					DocumentNote **		pDn,
					const BufferDocument *	bd,
					DocumentField *		df,
					int			extItKind );

extern void docSetExternalItemKind(	BufferItem *		bi,
					int			extItKind );

extern int docCheckNoteSeparatorItem(	BufferDocument *	bd,
					int			extItKind );

extern int docCountParticulesInField(	const BufferItem *	paraBi,
					int *			pClosed,
					int			part,
					int			partUpto );

extern int docGetSolidRgbShadeOfItem(	int *			pIsFilled,
					RGB8Color *		rgb8,
					const BufferDocument *	bd,
					const ItemShading *	is );

extern void docDelimitTables(		BufferItem *		parentBi );

extern void docGetCellTopBorder(
			BorderProperties *		bpTop,
			int *				pUseAbove,
			const BufferDocument *		bd,
			const BufferItem *		rowBi,
			int				col,
			int				atRowTop );

extern void docGetCellBottomBorder(
			BorderProperties *		bpBottom,
			int *				pUseBelow,
			const BufferDocument *		bd,
			const BufferItem *		rowBi,
			int				col,
			int				atRowBottom );

extern int docTextAttributeNumber(	BufferDocument *	bd,
					const TextAttribute *	ta );

extern int docInsertAdminParticule(	BufferDocument *	bd,
					BufferItem *		paraBi,
					int			n,
					int			off,
					int			objectNumber,
					int			kind,
					const TextAttribute *	ta );

extern TextParticule * docMakeSpecialParticule(
				BufferItem *			paraBi,
				int				n,
				int				stroff,
				int				kind,
				int				textAttrNr );

extern int docBalanceFieldSelection(	DocumentField **	pLeftField,
					DocumentField **	pRightField,
					int *			pBeginMoved,
					int *			pEndMoved,
					int *			pHeadPart,
					int *			pTailPart,
					DocumentPosition *	dpBegin,
					DocumentPosition *	dpEnd,
					DocumentTree *		ei,
					BufferDocument *	bd );

extern int docGetListOfParagraph(	ListOverride **		pLo,
					DocumentList **		pDl,
					int			ls,
					const BufferDocument *	bd );

extern int docGetListLevelOfParagraph(
				int *				startPath,
				int *				formatPath,
				ListOverride **			pLo,
				DocumentList **			pDl,
				const DocumentListLevel **	pDll,
				const ParagraphProperties *	pp,
				const BufferDocument *		bd );

extern int docGetListOfOverride(	ListOverride *			lo,
					const DocumentListTable *	dlt );

extern int docListGetFormatPath(int *				startPath,
				int *				formatPath,
				const DocumentListLevel **	pDll,
				int				ilvl,
				const DocumentList *		dl,
				const ListOverride *		lo );

extern int docInsertListtextField(
				DocumentField **	pField,
				DocumentSelection *	dsInsideField,
				DocumentSelection *	dsAroundField,
				int *			pPartBegin,
				int *			pPartEnd,
				BufferItem *		paraBi,
				DocumentTree *		ei,
				BufferDocument *	bd );

extern int docRemoveParagraphFromList(	BufferItem *		paraBi,
					DocumentTree *		ei,
					BufferDocument *	bd );

extern int docParaHeadFieldKind(	const BufferItem *	paraBi,
					const BufferDocument *	bd );

extern int docDelimitParaHeadField(
				DocumentField **	pField,
				DocumentSelection *	dsInsideField,
				DocumentSelection *	dsAroundField,
				int *			pPartBegin,
				int *			pPartEnd,
				const BufferItem *	paraBi,
				const BufferDocument *	bd );

extern int docFindListOfSelection(
				int *				pLs,
				int *				pLevel,
				int *				pMultiList,
				int *				pMultiLevel,
				int *				pParaNr,
				const DocumentSelection *	ds );

extern int docApplyListRuler(	const DocumentList *		dl,
				const ListOverride *		lo,
				BufferDocument *		bd );

extern void docGetSelectionAttributes(
				BufferDocument *		bd,
				const DocumentSelection *	ds,
				PropertyMask *			pUpdMask,
				TextAttribute *			pTaNew );

extern int docInsertParaHeadField(
				DocumentField **		pField,
				DocumentSelection *		dsInsideField,
				DocumentSelection *		dsAroundField,
				int *				pHeadPart,
				int *				pTailPart,
				BufferItem *			paraBi,
				BufferDocument *		bd,
				DocumentTree *			ei,
				int				fieldKind,
				int				textAttrNr );

extern int docDelimitFieldInDoc(DocumentSelection *		dsInside,
				DocumentSelection *		dsAround,
				int *				pPart0,
				int *				pPart1,
				const BufferDocument *		bd,
				const DocumentField *		df );

extern int docDelimitFieldInTree(
				DocumentSelection *		dsInside,
				DocumentSelection *		dsAround,
				int *				pPart0,
				int *				pPart1,
				const BufferDocument *		bd,
				const DocumentTree *		ei,
				const DocumentField *		df );

extern int docDeleteFieldFromParent(
				DocumentTree *			ei,
				DocumentField *			df );

extern int docDeleteField(	BufferItem *			paraBiHead,
				BufferItem *			paraBiTail,
				int *				pPartHead,
				int *				pPartTail,
				int				partHead,
				int				partTail,
				DocumentTree *			ei,
				BufferDocument *		bd,
				DocumentField *			df );

extern int docNextSimilarRoot(	DocumentPosition *		dp,
				int *				pPage,
				int *				pColumn,
				BufferDocument *		bd );

extern int docPrevSimilarRoot(	DocumentPosition *		dp,
				int *				pPage,
				int *				pColumn,
				BufferDocument *		bd );

extern int docNewList(		int *				pLs,
				int				lsExample,
				BufferDocument *		bd,
				const PropertyMask *		taSetMask,
				const TextAttribute *		taSet );

extern int docGetBitmapForObjectData(
				int				kind,
				void **				pPrivate,
				const MemoryBuffer *		mb );

extern int docAdaptParagraphToListLevel(
				int *				pChanged,
				BufferItem *			paraBi,
				const BufferDocument *		bd );

extern void docInvalidateParagraphLayout(	BufferItem *	paraBi );

extern int docMakeCapsString(	char **			pUpperString,
				int **			pSegments,
				int *			pSegmentCount,
				const TextAttribute *	ta,
				const char *		printString,
				int			len );

extern int docChangeRowProperties(
				PropertyMask *			rpDonePask,
				BufferItem *			rowBi,
				const PropertyMask *		rpSetMask,
				const RowProperties *		rpFrom );

extern BufferItem * docGetBodySectBiOfScope(
					const SelectionScope *	ss,
					const BufferDocument *	bd );

extern const BufferItem * docGetBodySectBi(
					BufferItem *		bi,
					const BufferDocument *	bd );

extern int docSelectionForEditPositionsInDoc(
				DocumentSelection *		ds,
				const BufferDocument *		bd,
				const SelectionScope *		ss,
				const EditPosition *		epHead,
				const EditPosition *		epTail );

extern int docSelectionForEditPositionsInTree(
				DocumentSelection *		ds,
				const DocumentTree *		ei,
				const EditPosition *		epHead,
				const EditPosition *		epTail );

extern int docPositionForEditPosition(	DocumentPosition *		dpNew,
					const EditPosition *		ep,
					const DocumentTree *		ei );

extern DocumentNote *	docGetNoteOfField(
					const DocumentField *		dfNote,
					const BufferDocument *		bd );

extern void docRememberFontUsed(	BufferDocument *		bd,
					const TextAttribute *		ta );

extern void docShiftFieldReferences(	DocumentTree *		ei,
					int			sectFrom,
					int			paraFrom,
					int			stroffFrom,
					int			sectShift,
					int			paraShift,
					int			stroffShift );

extern int docGetCellRight(		int *			pColspan,
					const BufferItem *	rowBi,
					int			col );

extern int docGetMatchingCell(		int *			pColspan,
					const BufferItem *	rowBi,
					int			l,
					int			r );

extern void docDeleteFieldFromDocument(	BufferDocument *	bd,
					DocumentField *		df );

extern int docDeleteFieldRange(		int *			pBulletsDeleted,
					int *			pNotesDeleted,
					BufferDocument *	bd,
					const EditRange *	er,
					ChildFields *		rootFields,
					DocumentField *		dfLeft,
					DocumentField *		dfRight );

extern void docDeleteChildFields(	int *			pBulletsDeleted,
					int *			pNotesDeleted,
					BufferDocument *	bd,
					ChildFields *		cf );

extern void docRemoveUnbalancedTocBookmarks(
					BufferDocument *	bdDoc );

extern int docSetTocBookmarks(		BufferDocument *	bd );

extern int docSetParaTocBookmark(	BufferDocument *	bd,
					DocumentTree *		ei,
					BufferItem *		paraBi );

extern int docFindSetPattern(		void **			pProg,
					const char *		pattern,
					int			useRegex );

extern BufferItem * docGetCellItem(		BufferItem *		bi );
extern BufferItem * docGetRowItem(		BufferItem *		bi );
extern BufferItem * docGetRowLevelItem(		BufferItem *		bi );
extern BufferItem * docGetSectItem(		BufferItem *		bi );

extern int docRowNesting(			const BufferItem *	bi );
extern int docTableNesting(			const BufferItem *	bi );
extern void docSetParaTableNesting(		BufferItem *	paraBi );

extern int docValidChildLevel(		int		parentLevel,
					int		childLevel );

extern int docGetCharsUsed(	BufferDocument *		bd );

extern void docPlainTextAttribute(	TextAttribute *		ta,
					BufferDocument *	bd );

extern int docScapsAttributeNumber(		BufferDocument *	bd,
						const TextAttribute *	ta );

extern void docCleanExternalItemObjects( int *			pBulletsDeleted,
					int *			pParagraphCount,
					DocumentTree *		ei,
					BufferDocument *	bd,
					DOC_CLOSE_OBJECT	closeObject );

extern void docCleanItemObjects(	int *			pBulletsDeleted,
					int *			pParagraphCount,
					DocumentTree *		ei,
					BufferDocument *	bd,
					BufferItem *		bi,
					DOC_CLOSE_OBJECT	closeObject );

extern void docCleanDocumentObjects(	BufferDocument *	bd,
					DOC_CLOSE_OBJECT	closeObject );

extern int docChangeSectionProperties(	PropertyMask *		pSpChgMask,
					int *			pRedraw,
					BufferDocument *	bd,
					BufferItem *		sectBi,
					const PropertyMask *	spSetMask,
					const SectionProperties * spNew );

extern void docInvalidateNoteLayoutForSection(
				BufferDocument *		bd,
				int				sect );

extern void docCleanParaItem(	BufferDocument *		bd,
				DocumentTree *			ei,
				BufferItem *			bi );

extern void docInitParaItem(	BufferItem *			bi );

extern int docGetTextAttributeForInsert(TextAttribute *			ta,
					const BufferDocument *		bd,
					int				IBar,
					const DocumentPosition *	dp );

#   endif
