/************************************************************************/
/*									*/
/*  Basic structures for a Gui document manipulator application.	*/
/*									*/
/************************************************************************/

#   ifndef	APP_FRAME_H
#   define	APP_FRAME_H

#   include	<stdio.h>

#   include	<appGuiBase.h>
#   include	<appGuiResource.h>

#   include	<geo2DInteger.h>
#   include	<utilDocumentGeometry.h>
#   include	"utilPrinter.h"
#   include	<appIcons.h>
#   include	<appTool.h>
#   include	<appInspector.h>
#   include	<appDraw.h>
#   include	<guiWidget.h>
#   include	<drawDrawingSurface.h>

# ifdef __cplusplus
extern "C"
    {
# endif

struct PrintGeometry;
struct PrintJob;

/************************************************************************/
/*									*/
/*  For configuring texts of 'Really' dialogs.				*/
/*									*/
/************************************************************************/

typedef struct AppFileMessageResources
    {
					/********************************/
					/*  Format for 'New' title	*/
					/********************************/
    char *		afmrNamelessTitleFormat;
					/********************************/
					/*  Really Close? Dialog.	*/
					/********************************/
    char *		afmrReallyCloseQuestion;
    char *		afmrReallyCloseSaveIt;
    char *		afmrReallyCloseDontSave;
    char *		afmrReallyCloseCancel;
					/********************************/
					/*  Really Quit? Dialog.	*/
					/********************************/
    char *		afmrReallyQuitQuestion;
    char *		afmrReallyQuitReview;
    char *		afmrReallyQuitAnyway;
    char *		afmrReallyQuitCancel;
					/********************************/
					/*  Messages about  a file.	*/
					/********************************/
    char *		afmrFileNoAccess;
    char *		afmrFileReadOnly;
    char *		afmrNoSuchFileMessage;
    char *		afmrFileNotWritable;
    char *		afmrMakeItQuestion;
    } AppFileMessageResources;

/************************************************************************/
/*  A menu, menu item in an application.				*/
/************************************************************************/

typedef enum MenuItemType
    {
    ITEMtyOPTION= 0,
    ITEMtyTOGGLE_OFF,
    ITEMtyTOGGLE_ON,
    ITEMtySEPARATOR
    } MenuItemType;

typedef struct AppMenuItem
    {
    const char *	amiTextResName;
    const char *	amiItemText;

    const char *	amiKeyResName;
    const char *	amiKey;

    const char *	amiKeyTextResName;
    const char *	amiKeyText;

    MenuItemType	amiItemType;
    APP_MENU_CALLBACK_T	amiCallback;
    APP_WIDGET		amiOptionWidget;
    } AppMenuItem;

/************************************************************************/
/*									*/
/*  Kinds of files that an application can open.			*/
/*									*/
/************************************************************************/

#   define	APPFILE_CAN_OPEN	0x01
#   define	APPFILE_CAN_SAVE	0x02
#   define	APPFILE_IS_BASIC_TYPE	0x04
#   define	APPFILE_HIDE_OPEN	0x08

typedef struct AppFileExtension
    {
    const char *	afeId;
    const char *	afeFilter;
    const char *	afeDescription;
    const char *	afeExtension;
    unsigned int	afeUseFlags;
    } AppFileExtension;

/************************************************************************/
/*									*/
/*  Describes a document generically.					*/
/*									*/
/************************************************************************/

typedef struct DocumentWidget
    {
    APP_WIDGET			dwWidget;
    AppColors			dwColors;
    APP_INPUT_CONTEXT		dwInputContext;
    AppDrawingData		dwDrawingData;

				/****************************************/
				/*  Event handling.			*/
				/*  Note that this is not a full GUI	*/
				/*  toolkit!				*/
				/****************************************/
    void *			dwOwner;
    void			(*dwGotString)(	void *		owner,
						const char *	value,
						int		length );
    void			(*dwGotKey)(	void *		owner,
						int		key,
						unsigned int	state );
    } DocumentWidget;

typedef struct SelectRectangle
    {
    int				srDirection;
#				define DOCselNONE	0

#				define DOCselN		1
#				define DOCselE		2
#				define DOCselS		3
#				define DOCselW		4

    DocumentRectangle		srSelected;
    DocumentRectangle		srLTM; /* left/top margins */
    DocumentRectangle		srRBM; /* right/bottom margins */
    } SelectRectangle;

typedef struct EditDocument
    {
    struct EditApplication *	edApplication;

    char *			edFilename;
    char *			edTitle;
    int				edFormat;
    int				edFileReadOnly;
    unsigned int		edDocumentId;

    AppToplevel			edToplevel;

    APP_WIDGET			edMenuBar;
    APP_WIDGET			edMainWindow;
    APP_WIDGET				edFileMenu;
    APP_WIDGET				edFileMenuButton;
    APP_WIDGET				edEditMenu;
    APP_WIDGET				edEditMenuButton;
    APP_WIDGET				edWindowMenu;
    APP_WIDGET				edWindowMenuButton;
    APP_WIDGET				edHelpMenu;
    APP_WIDGET				edHelpMenuButton;

    APP_WIDGET			edToolbar;
    APP_WIDGET			edScrolledWindow;
    APP_WIDGET				edVerticalScrollbar;
    APP_WIDGET				edHorizontalScrollbar;
#   ifdef USE_GTK
    GtkAdjustment *			edVerticalAdjustment;
    GtkAdjustment *			edHorizontalAdjustment;
#   endif
    DocumentWidget			edDocumentWidget;
    ScreenWidget			edScreenWidget;
    DrawingSurface			edDrawingSurface;

    APP_WIDGET				edLeftRulerWidget;
    APP_WIDGET				edTopRulerWidget;
    APP_WIDGET				edRightRulerWidget;
    APP_WIDGET				edBottomRulerWidget;

    void *				edLeftRuler;
    void *				edTopRuler;
    void *				edRightRuler;
    void *				edBottomRuler;

    int					edLeftRulerWidePixels;
    int					edTopRulerHighPixels;
    int					edRightRulerWidePixels;
    int					edBottomRulerHighPixels;

    int				edShellExtraWidth;
    int				edShellExtraHeight;

    int				edHasBeenChanged;
    int				edIsReadonly;
    int				edIsVisible;

    DocumentRectangle		edVisibleRect;

    void *			edPrivateData;

    struct AppSelectionTargetType *
				edTargetTypes;
    int				edTargetTypeCount;

    int				edMapped;	/*  Struggle with fvwm	*/
    int				edNotYetDrawn;	/*  For FirstVisible	*/

    SelectRectangle		edSelectRectangle;
    } EditDocument;

/************************************************************************/
/*									*/
/*  Special calls to the aplication.					*/
/*									*/
/************************************************************************/

typedef struct SpecialCall
    {
    const char *	scCall;
    int			(*scExecuteCall)(
					struct EditApplication *	ea,
					const char *			prog,
					const char *			call,
					int				argc,
					char **				argv );
				/****************************************/
				/*  Returns the number of parameters	*/
				/*  consumed or -1 on failure.		*/
				/****************************************/
    } SpecialCall;

/************************************************************************/
/*									*/
/*  Describes the application.						*/
/*									*/
/*  1)  Descriptive members.						*/
/*  2)  Allocated at run time.						*/
/*									*/
/************************************************************************/

typedef struct EditApplication
    {
    /*  1  */
    const char *	eaApplicationName;
    const char *	eaOptionalComponents;
    const char *	eaNameAndVersion;
    const char *	eaReference;
    const char *	eaPlatformCompiled;
    const char *	eaHostDateCompiled;

    const char *	eaDocumentWidgetName;
    const char *	eaPageToolName;
    const char *	eaPrintDialogName;
    const char *	eaMessageDialogName;
    const char *	eaMainIcon;
    const char *	eaMainPicture;
    AppFileExtension *	eaFileExtensions;
    int			eaFileExtensionCount;
    const char *	eaDefaultFileFilter;
    const char *	eaDocumentIcon;
    void *			eaResourceData;
    AppConfigurableResource *	eaResourceTable;
    int				eaResourceCount;
    AppConfigurableResource *	eaFileMessageResourceTable;
    int				eaFileMessageResourceCount;
    const SpecialCall *		eaSpecialCalls;
    int				eaSpecialCallCount;
    int				eaCreateNewFromCommand;

    int			eaScreenPixelsWide;
    int			eaScreenPixelsHigh;
    float		eaMagnification;

    int			eaLeftRulerWidthMM;
    int			eaTopRulerHeightMM;
    int			eaRightRulerWidthMM;
    int			eaBottomRulerHeightMM;
    int			eaPageGapMM;

    int			eaLeftRulerWidthMultiple;
    int			eaTopRulerHeightMultiple;
    int			eaRightRulerWidthMultiple;
    int			eaBottomRulerHeightMultiple;

    const char *	eaUnitString;
    const char *	eaPaperString;
    const char *	eaLeftMarginString;
    const char *	eaTopMarginString;
    const char *	eaRightMarginString;
    const char *	eaBottomMarginString;
    int			eaUnitInt;
    DocumentGeometry	eaDefaultDocumentGeometry;

    char **		eaAppFileMenuText;
    AppMenuItem *	eaAppFileMenuItems;
    int			eaAppFileMenuItemCount;

    char **		eaAppWinMenuText;
    AppMenuItem *	eaAppWinMenuItems;
    int			eaAppWinMenuItemCount;

    char **		eaDocFileMenuText;
    AppMenuItem *	eaDocFileMenuItems;
    int			eaDocFileMenuItemCount;

    char **		eaDocEditMenuText;
    AppMenuItem *	eaDocEditMenuItems;
    int			eaDocEditMenuItemCount;

    char **		eaDocWindowMenuText;
    AppMenuItem *	eaDocWindowMenuItems;
    int			eaDocWindowMenuItemCount;

    char **		eaDocHelpMenuText;
    AppMenuItem *	eaDocHelpMenuItems;
    int			eaDocHelpMenuItemCount;

    char **		eaAppHelpMenuText;
    AppMenuItem *	eaAppHelpMenuItems;
    int			eaAppHelpMenuItemCount;

    void *		(*eaMakePrivateData)( void );

    int			(*eaMakeDocumentWidget)(
				struct EditApplication *	ea,
				EditDocument *			ed );
    int			(*eaOpenDocument)(
				struct EditApplication *	ea,
				void *				privateData,
				int *				pFormat,
				APP_WIDGET			relative,
				APP_WIDGET			option,
				const char *			filename );
    int			(*eaNewDocument)(
				struct EditApplication *	ea,
				EditDocument *			ed,
				const char *			filename );
    int			(*eaLayoutDocument)(
				void *				privateData,
				int				format,
				AppDrawingData *		add,
				const DocumentGeometry *	defDg );
    int			(*eaFinishDocumentSetup)(
				EditDocument *			ed );
    void		(*eaDocumentFirstVisible)(
				EditDocument *			ed );
    int			(*eaCanSaveDocument)(
				const void *			privateData,
				int				format );
    int			(*eaSaveDocument)(
				const void *			privateData,
				int				format,
				const char *			applicationId,
				const char *			documentTitle,
				const char *			filename );
    void		(*eaFreeDocument)(
				void *				privateData,
				int				format );
    void		(*eaSuggestNup)(
				struct PrintGeometry *		pg,
				void *				privateData );
    int			(*eaPrintDocument)(
				SimpleOutputStream *		sos,
				const struct PrintJob *		pj,
				const struct PrintGeometry *	pg );
    void		(*eaDrawRectangle)(
				EditDocument *			ed,
				DocumentRectangle *		drClip,
				int				ox,
				int				oy );

    void		(*eaVisibleDocumentCountChanged)(
				    struct EditApplication *	ea,
				    int				from,
				    int				to );

    void		(*eaMakePrivateApplicationMenus)(
				    struct EditApplication *	ea,
				    APP_WIDGET			menubar );

    void		(*eaMakePrivateDocumentMenus)(
				    struct EditApplication *	ea,
				    EditDocument *		ed,
				    APP_WIDGET			menubar );

    void		(*eaDocCopy)( EditDocument *);
    void		(*eaDocCut)( EditDocument *);
    void		(*eaDocSelAll)( EditDocument * );

    void		(*eaSetPageLayout)(
				    struct EditDocument *	ed,
				    const DocumentGeometry *	dgNew,
				    const PropertyMask *	setMask,
				    int				wholeDocument );

			/************************************************/
			/*  User input on the document widget:		*/
			/************************************************/
    APP_EVENT_HANDLER_T	eaDocumentMouseHandler;
    APP_EVENT_HANDLER_T	eaDocumentScrollHandler;
    void		(*eaDocGotString)(	void *		voided,
						const char *	value,
						int		length );
    void		(*eaDocGotKey)(		void *		voided,
						int		key,
						unsigned int	state );
    APP_EVENT_HANDLER_T	eaObserveFocus;

    APP_SCROLLBAR_CALLBACK_T	eaDocHorizontalScrollbarCallback;
    APP_SCROLLBAR_CALLBACK_T	eaDocVerticalScrollbarCallback;

    void		(*eaScrollTopRuler) (	void *, APP_WIDGET, int );
    void		(*eaSetTopRulerRange)(	void *		voidtr,
						APP_WIDGET	w,
						int		docVisX0,
						int		docVisX1,
						int		docBackX1 );

    void		(*eaScrollLeftRuler) (	void *, APP_WIDGET, int );
    void		(*eaSetLeftRulerRange)(	void *		voidtr,
						APP_WIDGET	w,
						int		docVisY0,
						int		docVisY1,
						int		docBackY1 );

    void		(*eaScrollRightRuler) (	void *, APP_WIDGET, int );
    void		(*eaSetRightRulerRange)(	void *	voidtr,
						APP_WIDGET	w,
						int		docVisY0,
						int		docVisY1,
						int		docBackY1 );

    void		(*eaScrollBottomRuler) ( void *, APP_WIDGET, int );
    void		(*eaSetBottomRulerRange)( void *	voidtr,
						APP_WIDGET	w,
						int		docVisY0,
						int		docVisY1,
						int		docBackY1 );

    int			(*eaSetTopRuler)(	EditDocument *	ed );
    int			(*eaSetLeftRuler)(	EditDocument *	ed );
    int			(*eaSetRightRuler)(	EditDocument *	ed );
    int			(*eaSetBottomRuler)(	EditDocument *	ed );

    void		(*eaFreeTopRuler)(
					void *			topRuler );
    void		(*eaFreeLeftRuler)(
					void *			leftRuler );
    void		(*eaFreeRightRuler)(
					void *			rightRuler );
    void		(*eaFreeBottomRuler)(
					void *			bottomRuler );

    struct AppSelectionType *	eaDocSelectionTypes;
    int				eaDocSelectionTypeCount;

    struct AppSelectionType *	eaAppSelectionTypes;
    int				eaAppSelectionTypeCount;

    PrintDestination *	eaPrintDestinations;
    int			eaPrintDestinationCount;
    int			eaDefaultPrintDestination;
    int			eaPrintDestinationsCollected;

    NamedPicture *	eaNamedPictures;
    int			eaNamedPictureCount;

    APP_ATOM		eaCloseAtom;

    unsigned int	eaNextDocumentId;

    /*  2  */
#   ifdef USE_MOTIF
    XtAppContext		eaContext;
#   endif

    AppToplevel			eaToplevel;
	APP_INPUT_METHOD	eaInputMethod;
	APP_CURSOR		eaDocumentCursor;
    APP_WIDGET			eaMainWindow;
    APP_WIDGET			eaMenuBar;
    APP_WIDGET			eaFileMenu;
    APP_WIDGET			eaFileMenuButton;
    APP_WIDGET			eaWinMenu;
    APP_WIDGET			eaWinMenuButton;
    APP_WIDGET			eaHelpMenu;
    APP_WIDGET			eaHelpMenuButton;

    void *			eaPageTool;
    void *			eaPrintDialog;

    int				eaArgc;
    char **			eaArgv;

    EditDocument *		eaCurrentDocument;
    EditDocument **		eaOpenDocuments;
    int				eaOpenCount;
    int				eaVisibleDocumentCount;
    int				eaMainVisibleAsAbout;

    char *			eaAfmDirectory;
    char *			eaFontDirectory;
    char *			eaGhostscriptFontmap;
    char *			eaGhostscriptFontToXmapping;
    int				eaGhostscriptMappingsRead;
    char *			eaDefaultFont;
    char *			eaRulerFont;
    char *			eaPrintDialogFont;
    char *			eaCustomPrintCommand;
    char *			eaCustomPrinterName;
    char *			eaCustomPrintCommand2;
    char *			eaCustomPrinterName2;
    char *			eaAuthor;
    char *			eaFocusColor;

    int				eaHideSaveToOption;
    int				eaUsePostScriptFilters;
    int				eaUsePostScriptIndexedImages;
    int				ea7BitsPostScript;
    int				eaSkipEmptyPages;
    int				eaSkipBlankPages;
    int				eaOmitHeadersOnEmptyPages;
    int				eaAvoidFontconfig;
    int				eaPreferBase35Fonts;
    int				eaEmbedFonts;

    int				eaGotPaste;
    AppFileMessageResources	eaFileMessageResources;
    char *			eaMagnificationString;
    char *			eaHideSaveToOptionString;
    char *			eaUsePostScriptFiltersString;
    char *			eaUsePostScriptIndexedImagesString;
    char *			ea7BitsPostScriptString;
    char *			eaSkipEmptyPagesString;
    char *			eaSkipBlankPagesString;
    char *			eaOmitHeadersOnEmptyPagesString;
    char *			eaAvoidFontconfigString;
    char *			eaPreferBase35FontsString;
    char *			eaEmbedFontsString;
    char *			eaCustomPsSetupFilename;


    PostScriptFontList		eaPostScriptFontList;

    void *			eaSystemProperties;
    void *			eaUserProperties;

    int				eaGotResourceTable;
    int				eaGotFileMessageResourceTable;
    int				eaGotApplicationResources;
    } EditApplication;

typedef int (*APP_OPEN_DOCUMENT)(	EditApplication *	ea,
					void *			through,
					APP_WIDGET		relative,
					APP_WIDGET		option,
					const char *		filename );

typedef int (*APP_SAVE_DOCUMENT)(	EditDocument *	ed,
					void *		through,
					APP_WIDGET	relative,
					APP_WIDGET	option,
					int		format,
					const char *	filename );

/************************************************************************/
/*									*/
/*  Describes types of content for Copy/Paste.				*/
/*  The order is the order of desirability for the application.		*/
/*									*/
/************************************************************************/

typedef void (*APP_FORGET_COPY)( APP_WIDGET		w,
				void *			through,
				APP_EVENT *		event );

typedef struct AppSelectionTargetType
    {
    const char *	asttTargetString;		/*  ContentType	*/
    APP_ATOM		asttTargetAtom;
    APP_PASTE_REPLY	asttUsePaste;
    APP_GIVE_COPY	asttGiveCopy;
    } AppSelectionTargetType;

typedef struct AppSelectionType
    {
    const char *		astSelectionString;	/*  What	*/
    APP_ATOM			astSelectionAtom;

    AppSelectionTargetType *	astTargetTypes;
    int				astTargetTypeCount;

    APP_FORGET_COPY		astForgetCopy;
    } AppSelectionType;

/************************************************************************/
/*									*/
/*  Possible responses in one of Question dialogs.			*/
/*									*/
/************************************************************************/

#   define	AQDrespFAILURE		-1
#   define	AQDrespNONE		0
#   define	AQDrespYES		1
#   define	AQDrespNO		2
#   define	AQDrespCANCEL		3
#   define	AQDrespOK		4

/************************************************************************/
/*									*/
/*  Routine declarations.						*/
/*									*/
/************************************************************************/

extern APP_WIDGET appMakeMenu(		APP_WIDGET *		pButton,
					AppToplevel *		at,
					EditApplication *	ea,
					APP_WIDGET		parent,
					const char *		itemText,
					int			isHelp,
					AppMenuItem *		items,
					int			itemCount,
					void *			through );

extern EditDocument * appOpenDocument(	EditApplication *	ea,
					APP_WIDGET		relative,
					APP_WIDGET		option,
					int			read_only,
					const char *		filename );

extern int appNewDocument(	EditApplication *	ea,
				const char *		filename );

extern int appMain(	EditApplication *	ea,
			int			argc,
			char *			argv[] );

extern void appSetDocument(	EditApplication *	ea,
				EditDocument *		ed );

extern void appRemoveDocument(	EditApplication *	ea,
				EditDocument *		ed );

extern void appCloseDocument(	EditApplication *	ea,
				EditDocument *		ed );

extern void appSetCurrentDocument(	EditApplication *	ea,
					EditDocument *		ed );

extern void appDocToFront(	APP_WIDGET		option,
				void *			voided,
				void *			e );

extern void appDocumentChanged(	EditDocument *		ed,
				int			changed );

extern int appDocSaveDocumentByName(	EditDocument *		ed,
					APP_WIDGET		option,
					int			interactive,
					int			format,
					const char *		filename );

void appQuitApplication(	APP_WIDGET		option,
				APP_WIDGET		relative,
				EditApplication *	ea );

extern APP_MENU_CALLBACK_H( appAppFileQuit, option, voidea, e );

extern void appDocVisible(	EditApplication *	ea,
				EditDocument *		ed,
				int			visible );

extern APP_MENU_CALLBACK_H( appAppFileOpen, option, voidea, e );
extern APP_MENU_CALLBACK_H( appAppFileNew, option, voidea, e );

extern void appDocSetScrollbarValues(	EditDocument *	ed );

extern void appMouseWheelUp(		EditDocument *	ed );
extern void appMouseWheelDown(		EditDocument *	ed );
extern APP_EVENT_HANDLER_H( appScrollEventHandler, w, voided, scrollEvent );

extern APP_MENU_CALLBACK_H( appDocFileSaveAs, option, voided, e );

extern void appSetShellConstraints(	EditDocument *		ed );

extern APP_MENU_CALLBACK_H( appDocFileNew, option, voided, e );
extern APP_MENU_CALLBACK_H( appDocFileOpen, option, voided, e );
extern APP_MENU_CALLBACK_H( appDocFileSave, option, voided, e );
extern APP_MENU_CALLBACK_H( appDocFileClose, option, voided, e );
extern APP_MENU_CALLBACK_H( appDocFileQuit, option, voided, e );
extern APP_MENU_CALLBACK_H( appDocEditCopy, option, voided, e );
extern APP_MENU_CALLBACK_H( appDocEditCut, option, voided, e );
extern APP_MENU_CALLBACK_H( appDocEditPaste, option, voided, e );
extern APP_MENU_CALLBACK_H( appDocEditSelAll, option, voided, e );

extern APP_CLOSE_CALLBACK_H( appDocFileCloseCallback, w, voided );
extern APP_CLOSE_CALLBACK_H( appAppWmClose, w, voidea );

extern void appRunReallyCloseDialog(	APP_WIDGET		option,
					EditDocument *		ed );

extern int appRunReallyQuitDialog(	APP_WIDGET		option,
					APP_WIDGET		relative,
					EditApplication *	ea );

extern void appMakeDocVisible(	EditApplication *	ea,
				EditDocument *		ed );

extern void appShowDefaultsEditor(	APP_WIDGET	prefOption,
					void *		voidea );

extern int appGetImagePixmap(	EditApplication *	ea,
				const char *		name,
				APP_BITMAP_IMAGE *	pImage,
				APP_BITMAP_MASK *	pMask );

extern void appDiscardImagePixmaps(	EditApplication *	ea );

extern int appSetDocumentFilename(	EditDocument *		ed,
					const char *		filename );

extern int appSetDocumentTitle(		EditDocument *		ed,
					const char *		title );

extern void appGetFactors(	EditApplication *	ea,
				double *		pHorPixPerMM,
				double *		pVerPixPerMM,
				double *		pXfac,
				double *		pYfac );

extern int appFinishDocumentSetup(	EditDocument *		ed );

extern int appMakeDocumentWindow(	EditDocument **		pEd,
					EditApplication *	ea,
					int			read_only,
					const char *		title,
					const char *		filename );

extern int appSetupDocument(	EditApplication *	ea,
				EditDocument *		ed );

extern void appRenameDocumentOptions(	EditApplication *	ea,
					EditDocument *		ed,
					char *			title );

extern int appQuestionRunSubjectYesNoCancelDialog(
					EditApplication *	ea,
					APP_WIDGET		relative,
					APP_WIDGET		option,
					const char *		subject,
					const char *		question );

extern int appQuestionRunYesNoCancelDialog(
					EditApplication *	ea,
					APP_WIDGET		relative,
					APP_WIDGET		option,
					const char *		question );

extern void appQuestionRunSubjectErrorDialog(
					EditApplication *	ea,
					APP_WIDGET		relative,
					APP_WIDGET		option,
					const char *		subject,
					const char *		message );

extern void appQuestionRunErrorDialog(	EditApplication *	ea,
					APP_WIDGET		relative,
					APP_WIDGET		option,
					const char *		message );

extern int appQuestionRunOkCancelDialog( EditApplication * ea,
					APP_WIDGET		relative,
					APP_WIDGET		option,
					const char *		question );

extern int appQuestionRunSubjectOkCancelDialog( EditApplication * ea,
					APP_WIDGET		relative,
					APP_WIDGET		option,
					const char *		subject,
					const char *		question );

extern int appQuestionRunSubjectYesNoDialog( EditApplication *	ea,
					APP_WIDGET		relative,
					APP_WIDGET		option,
					const char *		subject,
					const char *		message );

extern void * appMakePageTool(	EditApplication *	ea,
				APP_WIDGET		printOption,
				const char *		pixmapName );

extern void appShowPageTool(		EditApplication *	ea );

extern void appPageToolSetProperties(	void *				voidapt,
					const DocumentGeometry *	dg );

extern void appEnablePageTool(		void *		voidapt,
					int		enabled );

extern void appRunPrintDialog(		EditDocument *		ed,
					const DocumentGeometry * dgDocument,
					int			pageCount,
					int			firstSelected,
					int			lastSelected,
					APP_WIDGET		printOption,
					const char *		pixmapName );

extern int appPrintDocument(	int				printer,
				const struct PrintJob *		pj );

extern void appDocPrintToFile(	APP_WIDGET			option,
				APP_WIDGET			panel,
				EditDocument *			ed,
				const struct PrintGeometry *	pg );

APP_MENU_CALLBACK_H( appDocAbout, option, voided, e );

extern APP_EVENT_HANDLER_H( appDocCopyPasteHandler, w, voided, event );
extern APP_EVENT_HANDLER_H( appAppGotPasteCall, w, voided, event );

# ifdef USE_GTK

extern void appDocGotPasteReplyGtk(	GtkWidget *		w,
					GtkSelectionData *	gsd,
					guint			time,
					void *			voided );

extern void appAppGotPasteReplyGtk(	GtkWidget *		w,
					GtkSelectionData *	gsd,
					guint			time,
					void *			voidea );
# endif

extern int appDocReleaseSelection(	EditDocument *		ed,
					const char *		selection );

extern int appDocOwnSelection(	EditDocument *			ed,
				const char *			selection,
				AppSelectionTargetType * 	targets,
				int				targetCount );

extern int appDocAskForPaste(	EditDocument *		ed,
				const char *		selection );

extern int appAppAskForPaste(	EditApplication *	ea,
				const char *		selection );

APP_SCROLLBAR_CALLBACK_H( appDocHorizontalScrollbarCallback, bar, voided, e );
APP_SCROLLBAR_CALLBACK_H( appDocVerticalScrollbarCallback, bar, voided, e );

extern void appMakeButtonInRow(	APP_WIDGET *		pButton,
				APP_WIDGET		buttonRow,
				const char *		text,
				APP_BUTTON_CALLBACK_T	callback,
				void *			through,
				int			position,
				int			colspan,
				int			showAsDefault );

extern void appMakeButtonInColumn(
				APP_WIDGET *		pButton,
				APP_WIDGET		column,
				const char *		buttonText,
				APP_BUTTON_CALLBACK_T	callback,
				void *			through,
				int			showAsDefault );

extern void appMakeLabelInRow(		APP_WIDGET *		pLabel,
					APP_WIDGET		row,
					int			column,
					int			colspan,
					const char *		labelText );

extern void appMakeLabelInHBox(		APP_WIDGET *		pLabel,
					APP_WIDGET		hbox,
					const char *		labelText );

extern void appMakeTextInRow(		APP_WIDGET *		pText,
					APP_WIDGET		row,
					int			column,
					int			colspan,
					int			textColumns,
					int			textEnabled );

extern void appMakeTextInHBox(		APP_WIDGET *		pText,
					APP_WIDGET		hbox,
					int			textColumns,
					int			textEnabled );

extern void appMakeLabelAndTextRow(	APP_WIDGET *		pRow,
					APP_WIDGET *		pLabel,
					APP_WIDGET *		pText,
					APP_WIDGET		parent,
					const char *		labelText,
					int			textColumns,
					int			textEnabled );

extern APP_WIDGET appMakeToggleInRow(	APP_WIDGET		buttonRow,
					const char *		text,
					APP_TOGGLE_CALLBACK_T	callback,
					void *			through,
					int			col,
					int			colspan );

extern void appMakeToggleAndTextRow(	APP_WIDGET *		pRow,
					APP_WIDGET *		pToggle,
					APP_WIDGET *		pText,
					APP_WIDGET		parent,
					const char *		labelText,
					APP_TOGGLE_CALLBACK_T	callback,
					void *			through,
					int			textColumns,
					int			textEnabled );

extern void appReportSaveFailure(	EditApplication *	ea,
					APP_WIDGET		option,
					APP_WIDGET		relative,
					const char *		filename );

extern void appMakeColumnFrameInColumn(	APP_WIDGET *		pFrame,
					APP_WIDGET *		pPaned,
					APP_WIDGET		parent,
					const char *		title );

extern void appMakeLabelInColumn(	APP_WIDGET *		pLabel,
					APP_WIDGET		row,
					const char *		labelText );

extern void appMakeTextInColumn(	APP_WIDGET *	pText,
					APP_WIDGET	column,
					int		textColumns,
					int		textEnabled );

extern void appMakeColumnToggle(	APP_WIDGET *		pToggle,
					APP_WIDGET		column,
					APP_TOGGLE_CALLBACK_T	callback,
					void *			through,
					const char *		labelText,
					int			set );

extern void appMakeColumnDrawing(	APP_WIDGET *		pButton,
					APP_WIDGET		column,
					APP_DRAW_BUTTON_CALLBACK exposeCallback,
					APP_BUTTON_CALLBACK_T	pushedCallback,
					void *			through,
					int			width );

extern void appMakeVerticalDialog(	AppDialog *		ad,
					APP_WIDGET *		pPaned,
					EditApplication *	ea,
					APP_CLOSE_CALLBACK_T	closeCallback,
					APP_DESTROY_CALLBACK_T	destroyCallback,
					void *			through,
					const char *		widgetName );

extern char * 	appWidgetName(		char *			file,
					int			line );

extern int appGetLengthFromTextWidget(	APP_WIDGET	w,
					int *		pValue,
					int *		pChanged,
					int		defaultUnit,
					int		minValue,
					int		adaptToMin,
					int		maxValue,
					int		adaptToMax );

extern int appGetIntegerFromTextWidget(	APP_WIDGET	w,
					int *		pValue,
					int		minValue,
					int		adaptToMin,
					int		maxValue,
					int		adaptToMax );

extern int appGetDoubleFromTextWidget(	APP_WIDGET	w,
					double *	pValue,
					double		minValue,
					int		adaptToMin,
					double		maxValue,
					int		adaptToMax );

extern void appEnableText(		APP_WIDGET	text,
					int		enabled );

extern void appRefuseTextValue(		APP_WIDGET	w );

extern void appDrawPageDiagram(	APP_WIDGET			w,
				AppDrawingData *		add,
				double				widgetHighCm,
				const DocumentGeometry *	dg );

extern void appDrawNupDiagram(	APP_WIDGET			w,
				AppDrawingData *		add,
				int				screenFont,
				double				widgetHighCm,
				const DocumentGeometry *	dgPage,
				const struct PrintGeometry *	pg );

extern int appCallPrintFunction( SimpleOutputStream *		sos,
				const struct PrintJob *		pj );

extern void appStringToTextWidget(	APP_WIDGET		w,
					const char *		s );

extern void appIntegerToTextWidget(	APP_WIDGET		w,
					int			n );

extern void appDoubleToTextWidget(	APP_WIDGET		w,
					double			d );

extern void appLengthToTextWidget(	APP_WIDGET		w,
					int			l,
					int			unit );

extern int appImgMakeFileExtensions(	AppFileExtension **	pAfeList,
					int *			pAfeCount );

extern void appInspectorMakeButtonRow(	APP_WIDGET *		pRow,
					APP_WIDGET		parent,
					APP_WIDGET *		pLeftButton,
					APP_WIDGET *		pRightButton,
					const char *		leftLabel,
					const char *		rightLabel,
					APP_BUTTON_CALLBACK_T	leftCallback,
					APP_BUTTON_CALLBACK_T	rightCallback,
					void *			through );

extern void appInspectorMake3ButtonRow( APP_WIDGET *		pRow,
					APP_WIDGET		parent,
					APP_WIDGET *		pLeftButton,
					APP_WIDGET *		pMiddleButton,
					APP_WIDGET *		pRightButton,
					const char *		leftLabel,
					const char *		middleLabel,
					const char *		rightLabel,
					APP_BUTTON_CALLBACK_T	leftCallback,
					APP_BUTTON_CALLBACK_T	middleCallback,
					APP_BUTTON_CALLBACK_T	rightCallback,
					void *			through );

extern void appInspectorMakeToggleRow(	APP_WIDGET *		pRow,
					APP_WIDGET		parent,
					APP_WIDGET *		pLeftToggle,
					APP_WIDGET *		pRightToggle,
					const char *		leftText,
					const char *		rightText,
					APP_TOGGLE_CALLBACK_T	leftCallback,
					APP_TOGGLE_CALLBACK_T	rightCallback,
					void *			through );

extern void appInspectorMakeToggleAndLabelRow(
					APP_WIDGET *		pRow,
					APP_WIDGET		parent,
					APP_WIDGET *		pToggle,
					APP_WIDGET *		pLabel,
					const char *		toggleText,
					const char *		labelText,
					APP_TOGGLE_CALLBACK_T	toggleCallback,
					void *			through );

extern void appInspectorMakeLabelAndToggleRow(
					APP_WIDGET *		pRow,
					APP_WIDGET		parent,
					APP_WIDGET *		pLabel,
					APP_WIDGET *		pToggle,
					const char *		labelText,
					const char *		toggleText,
					APP_TOGGLE_CALLBACK_T	toggleCallback,
					void *			through );

extern AppInspector * 	appMakeInspector(
				    EditApplication *		ea,
				    APP_WIDGET			option,
				    const char *		pixmapName,
				    const char *		widgetName,
				    InspectorSubjectResources * isr,
				    int				subjectCount,
				    AppToolDestroy		closeInspector,
				    void *			through );

extern void appInspectorMakeMenuRow(	APP_WIDGET *		pRow,
					AppOptionmenu *		aom,
					APP_WIDGET *		pLabel,
					APP_WIDGET		parent,
					const char *		labelText );

extern void appFillInspectorMenu(	int			count,
					int			current,
					APP_WIDGET *		items,
					const char * const *	texts,
					AppOptionmenu *		aom,
					APP_OITEM_CALLBACK_T	callBack,
					void *			target );

extern int appMakeDocumentWidget(	EditApplication *	ea,
					EditDocument *		ed );

extern APP_EVENT_HANDLER_H( appDocExposeHandler, w, d, exposeEvent );

extern int appFileConvert(	EditApplication *	ea,
				const char *		fromName,
				const char *		toName );

extern int appPrintToFile(	EditApplication *	ea,
				const char *		fromName,
				const char *		toName,
				const char *		paperString );

extern int appPrintToPrinter(	EditApplication *	ea,
				const char *		fromName,
				const char *		toName,
				const char *		paperString );

APP_DESTROY_CALLBACK_H( appDestroyEditDocument, w, voided );

extern APP_WIDGET appMakePageDrawing(	APP_WIDGET		parent,
					EditApplication *	ea,
					int			mmHigh,
					APP_EVENT_HANDLER_T	redraw,
					void *			through );

extern void appScrollToRectangle(	EditDocument *		ed,
					int			x0,
					int			y0,
					int			x1,
					int			y1,
					int *			pScrolledX,
					int *			pScrolledY );

extern void appMakeVerticalTool( APP_WIDGET *		pShell,
				APP_WIDGET *		pPaned,
				EditApplication *	ea,
				APP_BITMAP_IMAGE	iconPixmap,
				APP_BITMAP_MASK		iconMask,
				const char *		widgetName,
				int			userResizable,
				APP_WIDGET		option,
				APP_CLOSE_CALLBACK_T	closeCallback,
				void *			through );

extern APP_WIDGET appMakeRowInColumn(	APP_WIDGET	parent,
					int		columnCount,
					int		heightResizable );

extern void appMakeColumnInRow(		APP_WIDGET *	pColumn,
					APP_WIDGET	row,
					int		position,
					int		colspan );

extern void appMakeHBoxInRow(		APP_WIDGET *	pHbox,
					APP_WIDGET	row,
					int		position,
					int		colspan );

extern void appMakeRowFrameInColumn(	APP_WIDGET *		pFrame,
					APP_WIDGET *		pRow,
					APP_WIDGET		parent,
					int			columnCount,
					const char *		title );

extern void appTextSelectContents(	APP_WIDGET	w,
					int		from,
					int		upto );

extern char *	appGetStringFromTextWidget(	APP_WIDGET	w );
extern void	appFreeStringFromTextWidget(	char *		s );

extern char *	appGetTextFromMenuOption(	APP_WIDGET	w );
extern void	appFreeTextFromMenuOption(	char *		s );

extern void appGuiGetResourceValues(
				int *				pGotResources,
				EditApplication *		ea,
				void *				pValues,
				AppConfigurableResource *	acr,
				int				acrCount );

extern void appSetResourceDefaults(
				EditApplication *		ea,
				AppConfigurableResource *	acr,
				int				acrCount );

extern int appCountChangedDocuments(	EditApplication *	ea );

extern int appGuiInitApplication(	EditApplication *	ea,
					int *			pArgc,
					char ***		pArgv );

extern void appExitApplication(	EditApplication *	ea );

extern void appAskCloseDocuments(	APP_WIDGET		option,
					EditApplication *	ea );

extern void appGuiInsertColumnInWindow(	APP_WIDGET *	pColumn,
					APP_WIDGET	parent );

extern void appGuiInsertMenubarInColumn( APP_WIDGET *	pMenubar,
					APP_WIDGET	parent );

extern void appMakeImageInColumn(	APP_WIDGET *		pLabel,
					APP_WIDGET		column,
					APP_BITMAP_IMAGE	pixmap,
					APP_BITMAP_MASK		mask );

extern void appGuiSetToggleLabel(	APP_WIDGET		toggle,
					const char *		text );

extern void appGuiSetToggleState(	APP_WIDGET		toggle,
					int			set );

extern int appGuiGetToggleState(	APP_WIDGET		toggle );

extern void appSetOptionmenu(		AppOptionmenu *		aom,
					int			num );

extern void appGuiEnableWidget(		APP_WIDGET		w,
					int			on_off );

extern void appGuiSetWidgetVisible(	APP_WIDGET		w,
					int			on_off );

extern void appShowShellWidget(		EditApplication *	ea,
					APP_WIDGET		shell );
extern void appHideShellWidget(		APP_WIDGET		shell );
extern void appDestroyShellWidget(	APP_WIDGET		shell );
extern void appIconifyShellWidget(	APP_WIDGET		shell );

extern void appGuiInsertSeparatorInColumn(	APP_WIDGET *	pSeparator,
						APP_WIDGET	parent );

extern void appOptionmenuSetWidthMotif(	APP_WIDGET		w,
					int			newWidth );

extern void appOptionmenuRefreshWidth(	AppOptionmenu *		aom );

extern void appEmptyOptionmenu(		AppOptionmenu *		aom );
extern void appEmptyParentWidget(	APP_WIDGET		parent );

extern void appGuiEnableOptionmenu(	AppOptionmenu *		aom,
					int			sensitive );

extern APP_WIDGET appAddItemToOptionmenu( AppOptionmenu *	aom,
					const char *		label,
					APP_OITEM_CALLBACK_T	callBack,
					void *			target );

extern void appDeleteItemFromOptionmenu( AppOptionmenu *	aom,
					APP_WIDGET		item );

extern void appSetShellTitle(		APP_WIDGET		shell,
					APP_WIDGET		option,
					const char *		title );

extern void appGuiLowerShellWidget(	APP_WIDGET		shell );

/************************************************************************/
/*									*/
/*  Event handler to set minimum size.					*/
/*									*/
/************************************************************************/

extern APP_EVENT_HANDLER_H( appSetSizeAsMinimum, w, through, event );

/************************************************************************/
/*									*/
/*  Names of the X11 events.						*/
/*									*/
/************************************************************************/

extern const char * const APP_X11EventNames[];

extern void appGuiChangeLabelText(	APP_WIDGET		labelWidget,
					const char *		label );

extern void appGuiChangeButtonText(	APP_WIDGET		labelWidget,
					const char *		label );

extern void appInspectorMakePageParent( AppInspector *	ai );

extern int appMakeVerticalInspectorPage(	APP_WIDGET *	pPage,
						APP_WIDGET *	pMenuitem,
						AppInspector *	ai,
						const char *	label );

extern void appInspectorEnablePage(	AppInspector *	ai,
					int		pageNumber,
					int		enabled );

extern void appMakeOptionmenuInColumn(		AppOptionmenu *	aom,
						APP_WIDGET	parent );

extern void appMakeOptionmenuInRow(	AppOptionmenu *	aom,
					APP_WIDGET	row,
					int		column,
					int		colspan );

# ifdef USE_GTK
    extern int appGuiGetMenuToggleStateFromCallbackGtk(
					APP_WIDGET		toggle );
    # define appGuiGetMenuToggleStateFromCallback( t, e ) \
			    appGuiGetMenuToggleStateFromCallbackGtk( (t) )
# else
    extern int appGuiGetMenuToggleStateFromCallback(
					APP_WIDGET		toggle,
					void *			voidcbs );
# endif

# ifdef USE_GTK
    extern int appGuiGetScrollbarValueFromCallbackGtk(
					APP_WIDGET		scrollbar );

    # define appGuiGetScrollbarValueFromCallback( bar, e ) \
			    appGuiGetScrollbarValueFromCallbackGtk( (bar) )
# else
    extern int appGuiGetScrollbarValueFromCallback(
					APP_WIDGET		scrollbar,
					void *			e );
# endif

extern void appGuiMakeListInColumn(	APP_WIDGET *		pList,
					APP_WIDGET		column,
					int			visibleItems,
					APP_LIST_CALLBACK_T	callback,
					APP_BUTTON_CALLBACK_T	actionCallback,
					void *			through );

extern void appGuiSetTypingCallbackForText(
					APP_WIDGET		text,
					APP_TXTYPING_CALLBACK_T	callBack,
					void *			through );

extern void appGuiSetGotValueCallbackForText(
				    APP_WIDGET			text,
				    APP_TXACTIVATE_CALLBACK_T	callBack,
				    void *			through );

extern void appGuiGetScrollbarValues(	int *		pValue,
					int *		pSliderSize,
					APP_WIDGET	scrollbar );

extern void appGuiSetScrollbarValues(	APP_WIDGET	scrollbar,
					int		value,
					int		sliderSize );

extern void appInitEditDocument(	EditApplication *	ea,
					EditDocument *		ed );

extern void appDocumentRulerWidth(	EditApplication *	ea,
					EditDocument *		ed );

extern int appFinishDocumentWindow(	EditDocument *		ed );

extern void appAdaptToDocumentSize(	EditDocument *	ed,
					int		width,
					int		height );

extern void appGuiSetShellTitle(	APP_WIDGET		shell,
					const char *		title );

extern void appGuiSetIconTitle(		APP_WIDGET		shell,
					const char *		title );

extern void appGuiEmptyListWidget(	APP_WIDGET		list );

extern void appGuiAddValueToListWidget(	APP_WIDGET		list,
					int			pos,
					const char *		value );

extern void appGuiReplaceValueInListWidget(
					APP_WIDGET		list,
					int			pos,
					const char *		value );

extern void appGuiDeleteValueFromListWidget(
					APP_WIDGET		list,
					int			pos );

extern void appGuiSelectPositionInListWidget(	APP_WIDGET	list,
						int		position );

extern void appGuiRemoveSelectionFromListWidget(	APP_WIDGET	list );

extern APP_WIDGET appSetMenuItem(	APP_WIDGET		menu,
					AppToplevel *		at,
					AppMenuItem *		ami,
					void *			target );

extern APP_WIDGET appSetMenuSeparator(	APP_WIDGET		menu,
					AppToplevel *		at,
					AppMenuItem *		ami,
					void *			target );

extern APP_WIDGET appSetToggleMenuItem(	APP_WIDGET		menu,
					AppToplevel *		at,
					AppMenuItem *		ami,
					void *			target );

extern APP_WIDGET appMakeMenuInParent(	APP_WIDGET *		pButton,
					AppToplevel *		at,
					APP_WIDGET		menuBar,
					const char *		itemText,
					int			isHelp );

extern char * appGuiGetStringFromListCallback(
					    APP_WIDGET		list,
					    APP_LIST_CHOICE *	voidlcs );

extern void appFreeStringFromListCallback(	char *		s );

extern void appGuiRunDialog( 		AppDialog *		ad,
					int			initial,
					EditApplication *	ea );

extern void appGuiBreakDialog( 		AppDialog *		ad,
					int			response );

extern void appGuiSetDefaultButtonForDialog(	AppDialog *	ad,
						APP_WIDGET	button );

extern void appGuiSetCancelButtonForDialog(	AppDialog *	ad,
						APP_WIDGET	button );

extern void appGuiShowDialog(			EditApplication * ea,
						AppDialog *	ad,
						APP_WIDGET	relative );

extern void appGuiHideDialog(			AppDialog *	ad );

extern void appMotifTurnOfSashTraversal(	APP_WIDGET	column );

extern void appDocumentSetInputContext(		APP_INPUT_METHOD	im,
						DocumentWidget *	dw );

extern void appGuiMakeDrawingAreaInColumn( APP_WIDGET *		pDrawing,
				APP_WIDGET			column,
				int				wide,
				int				high,
				int				heightResizable,
				APP_EVENT_HANDLER_T		redraw,
				void *				through );

extern void appDocFillMenu(		EditDocument *		ed );

void appInspectorChoosePage(	AppInspector *		ai,
				int			andMenu,
				int			pageNumber );

extern void appInitOptionmenu(		AppOptionmenu *		aom );

extern void appRunDragLoop(	APP_WIDGET		w,
				EditApplication *	ea,
				APP_EVENT *		downEvent,
				APP_EVENT_HANDLER_T	upHandler,
				APP_EVENT_HANDLER_T	moveHandler,
				int			timerInterval,
				APP_TIMER_CALLBACK	timerHandler,
				void *			through );

extern void appGuiSetFocusChangeHandler( APP_WIDGET		shell,
					APP_EVENT_HANDLER_T	handler,
					void *			through );

extern void appGuiSetToggleItemState(	APP_WIDGET		toggle,
					int			set );

extern void appSetDestroyCallback(	APP_WIDGET		shell,
					APP_DESTROY_CALLBACK_T	destroyCallback,
					void *			through );

extern void appSetCloseCallback(	APP_WIDGET		shell,
					EditApplication *	ea,
					APP_CLOSE_CALLBACK_T	closeCallback,
					void *			through );

extern void appGuiSetToggleItemLabel(	APP_WIDGET		toggle,
					const char *		label );

extern int appGuiGtkGetChildLabel(	char **			pLabel,
					APP_WIDGET		w );

extern int appGuiGtkSetChildLabel(	APP_WIDGET		w,
					const char *		s );

extern void appGuiGetResourceValuesGtkX(
				EditApplication *		ea,
				void *				pValues,
				AppConfigurableResource *	acr,
				int				acrCount );

extern int appFormatDocumentTitle(	const char **		pWindowTitle,
					const char **		pIconTitle,
					EditApplication *	ea,
					const char *		title );

extern APP_EVENT_HANDLER_H( appDocConfigure, w, voided, event );

extern void appDocumentCalculateExtraSize(	EditDocument *	ed );

extern void appSetWindowsItemState(	APP_WIDGET	menu,
					EditDocument *	ed,
					int		changed );

extern void appRemoveWindowsOption(	APP_WIDGET		menu,
					EditDocument *		oldEd );

extern void appRenameWindowsOption(	APP_WIDGET		menu,
					EditDocument *		ed,
					const char *		title );

extern void appAllocateCopyPasteTargetAtoms(	EditApplication *	ea );

extern APP_GIVE_COPY( appDocReplyToCopyRequest, w, gsd, voided );

extern void appCopyPixmapValue(	APP_SELECTION_EVENT *	event,
				APP_BITMAP_IMAGE	pixmapCopied );

extern void appGetApplicationResourceValues(	EditApplication *	ea );

extern void appDocExposeRectangle(
				const EditDocument *		ed,
				const DocumentRectangle *	drChanged,
				int				scrolledX,
				int				scrolledY );

extern void appPrintJobForEditDocument(	struct PrintJob *		pj,
					EditDocument *			ed,
					const struct PrintGeometry *	pg );

extern void appApplicationSettingsToPrintGeometry(
					struct PrintGeometry *	pg,
					EditApplication *	ea );

extern int appDocumentTestCanSave(
				EditApplication *		ea,
				const AppFileExtension *	afe,
				void *				privateData,
				unsigned int			useFlags,
				int				format );

extern int appDocumentGetSaveFormat(
				EditApplication *		ea,
				const char *			filename,
				void *				privateData,
				unsigned int			useFlags,
				int				format );

extern void appGuiMakeSliderInRow(	APP_WIDGET *		pSlider,
					APP_WIDGET		row,
					int			column,
					int			colspan,
					APP_BUTTON_CALLBACK_T	callback,
					void *			through );

extern void appGuiSetSlider(		APP_WIDGET		slider,
					int			minVal,
					int			maxVal,
					int			curVal );

extern void appGuiSetSliderValue(	APP_WIDGET		slider,
					int			curVal );

extern int appGuiGetValueFromSliderCallback(	int *		pValue,
						APP_WIDGET	w,
						void *		voidscs );

extern int appGuiGetLabelFont(		AppDrawingData *	add,
					APP_WIDGET		w );

extern int appGuiGetPositionFromListCallback(	APP_WIDGET	list,
						void *		voidlcs );

extern void appOptionmenuItemSetVisibility(	APP_WIDGET	w,
						int		visible );

extern int appPostScriptFontCatalog(		EditApplication *	ea );

extern int appGetPrintDestinations(		EditApplication *	ea );

extern int appReadSystemProperties(	EditApplication *	ea );
extern int appReadUserProperties(	EditApplication *	ea );
extern int appSetUserProperty(		EditApplication *	ea,
					const char *		name,
					const char *		value );

extern int appSetSystemProperty(	EditApplication *	ea,
					const char *		name,
					const char *		value );

extern int appGuiSetFrameTitle(		APP_WIDGET		w,
					const char *		title );

extern void appDocScrollToY(		EditDocument *		ed,
					int			y );
extern void appDocScrollToX(		EditDocument *		ed,
					int			x );

extern void appGuiFocusToWidget(	APP_WIDGET		w );

extern const AppSelectionType * appDocGetSelectionType(
				const EditApplication *		ea,
				const char *			selection );

extern void appInitSelectRectangle(	SelectRectangle *	sr );

extern void appHideMenuOption(		APP_WIDGET		w );

extern void appHideMenuChild(		APP_WIDGET		menu,
					int			n );

# ifdef __cplusplus
    }
# endif

#   endif
