/************************************************************************/
/*									*/
/*  Ted: Table manipulation.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	"tedEdit.h"
#   include	"docPageGrid.h"

#   include	<appDebugon.h>


/************************************************************************/
/*									*/
/*  Table related menu option callbacks.				*/
/*									*/
/************************************************************************/

void tedDocTableSelectTableRectangle(	EditDocument *		ed,
					const TableRectangle *	tr )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    DocumentSelection		dsNew;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    BufferItem *		selParentBi;

    int				scrolledX= 0;
    int				scrolledY= 0;

    const int			lastLine= 0;

    if  ( tedGetSelection( &dsNew, &sg, &sd,
			    (DocumentTree **)0, (const BufferItem **)0, td  ) )
	{ LDEB(1); return;	}

    if  ( docTableRectangleSelection( &dsNew, &selParentBi, bd, tr ) )
	{ LDEB(1); return;	}

    tedSetSelection( ed, &dsNew, lastLine, &scrolledX, &scrolledY );

    tedAdaptToolsToSelection( ed );

    return;
    }


/************************************************************************/
/*									*/
/*  Insert a table in the document.					*/
/*									*/
/*  0)  Also expose the table borders.					*/
/*  1)  If a range of data was selected, first discard it.		*/
/*  2)  If the selection was at the end of a paragraph, try to move to	*/
/*	the beginning of the next one.					*/
/*  3)  If the IBar is in the middle of a paragraph, split it.		*/
/*	tedSplitParaItem() does the layout of both halves.		*/
/*  4)  If the paragraph is not the first one in the 'cell' that	*/
/*	contains it, split the cell.					*/
/*  5)  If the 'cell' is not the first one in the 'row' that contains	*/
/*	it, split the row.						*/
/*									*/
/************************************************************************/

int tedInsertTable(		EditDocument *		ed,
				int			rows,
				int			columns )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    BufferItem *		sectBi;
    BufferItem *		refRowBi;
    int				col;
    int				paraNr;

    int				textAttributeNumber;

    int				wide;

    RowProperties		rp;

    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    BufferItem *		bi;
    DocumentRectangle		dr;

    DocumentPosition		dpTail;
    DocumentPosition		dpFirst;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    /*  0  */
    teo.teoChangedRect.drY0--;

    /*  1  */
    if  ( tedEditDeleteSelection( &teo ) )
	{ LDEB(1); return -1;	}

    dpTail= teo.teoEo.eoTailDp;

    /*  2  */
    bi= dpTail.dpBi;
    if  ( dpTail.dpStroff > 0				&&
	  dpTail.dpStroff == docParaStrlen( bi )	)
	{
	DocumentPosition	dpNext;

	dpNext= dpTail;
	docNextPosition( &dpNext );

	if  ( docPositionsInsideCell( &dpNext, &(dpTail) ) )
	    { dpTail= dpNext; bi= dpTail.dpBi; }
	}

    /*  3  */
    if  ( dpTail.dpStroff != 0 )
	{
	BufferItem *	newParaBi;
	int		onNewPage= 0;

	if  ( tedSplitParaItem( &newParaBi, &teo, DOClevROW, onNewPage ) )
	    { LDEB(1); return -1;	}

	bi= newParaBi;
	}
    else{
	BufferItem *		aftBi;

	/* 4,5 */
	if  ( bi->biNumberInParent > 0			||
	      bi->biParent->biNumberInParent > 0	)
	    {
	    BufferItem *	newBi;

	    if  ( docSplitGroupItem( bd, &newBi, &aftBi, bi->biParent,
					bi->biNumberInParent, DOClevROW ) )
		{ LDEB(bi->biNumberInParent); return -1;	}

	    if  ( aftBi && aftBi->biParent )
		{
		docEditIncludeItemInReformatRange( &(teo.teoEo),
							    aftBi->biParent );
		}
	    else{ XDEB(aftBi);	}
	    }
	}

    textAttributeNumber= bi->biParaParticules[0].tpTextAttrNr;

    docParaBlockFrameRectangle( &dr, bi, teo.teoEo.eoBodySectBi, bd,
					    bi->biTopPosition.lpPage,
					    bi->biTopPosition.lpColumn );

    wide= dr.drX1- dr.drX0;

    docInitRowProperties( &rp );

    {
    TextAttribute	ta;

    utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
							textAttributeNumber );

    rp.rpHalfGapWidthTwips= 5* ta.taFontSizeHalfPoints;
    rp.rpLeftIndentTwips=  -5* ta.taFontSizeHalfPoints;
    }

    for ( col= 0; col < columns; col++ )
	{
	CellProperties		cp;

	docInitCellProperties( &cp );
	cp.cpRightBoundaryTwips= ( ( col+ 1 )* wide )/ columns;

	if  ( docInsertRowColumn( &rp, col, &cp,
					(const DocumentAttributeMap *)0 ) )
	    { LDEB(col); return -1;	}

	docCleanCellProperties( &cp );
	}

    refRowBi= (BufferItem *)0;
    sectBi= bi->biParent->biParent->biParent;
    paraNr= docNumberOfParagraph( bi );

    if  ( docInsertTableRows( &dpFirst, &(teo.teoEo), sectBi, refRowBi,
		    &rp, textAttributeNumber,
		    bi->biParent->biParent->biNumberInParent, paraNr, rows ) )
	{ LDEB(rows); return -1;	}

    docCleanRowProperties( &rp );

    tedEditFinishIBarSelection( &teo, dpFirst.dpBi, dpFirst.dpStroff );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert a number of rows in a table.					*/
/*									*/
/*  1)  Check that the section is not empty. Ted avoids empty items at	*/
/*	any price. Every Item is supposed to contain at least one child	*/
/*	or at least one particule.					*/
/*  2)  Sanity check.							*/
/*  5)  Set an I-Bar selection of the first position in the first	*/
/*	(empty) cell of the first fresh row.				*/
/*									*/
/************************************************************************/

static int tedInsertRowsInTable(
			BufferItem *			sectBi,
			TedEditOperation *		teo,
			int				paraNr,
			int				textAttributeNumber,
			BufferItem *			refRowBi,
			EditDocument *			ed,
			int				row,
			int				rows )
    {
    EditOperation *		eo= &(teo->teoEo);
    const RowProperties *	rp;

    DocumentPosition		dpFirst;
    /*  1  */
    if  ( sectBi->biChildCount == 0 )
	{ LDEB(sectBi->biChildCount); return -1;	}

    /*  2  */
    if  ( ! docIsRowItem( refRowBi ) )
	{
	SLDEB(docLevelStr(refRowBi->biLevel),refRowBi->biRowCellCount);
	return -1;
	}

    rp= &(refRowBi->biRowProperties);
    if  ( docInsertTableRows( &dpFirst, eo, sectBi, refRowBi, rp,
				textAttributeNumber, row, paraNr, rows ) )
	{ LDEB(rows); return -1;	}

    /*  5  */
    tedEditFinishIBarSelection( teo, dpFirst.dpBi, dpFirst.dpStroff );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert a row before the current row in a table.			*/
/*									*/
/*  1)  Start edit operation.						*/
/*  2)  Find the position of the beginning of the selection in the	*/
/*	table.								*/
/*  3)  Get the row that serves as a template.				*/
/*  4)  Paragraph number of the paragraph that comes directly below	*/
/*	the fresh rows.							*/
/*  5)  Finish..							*/
/*									*/
/************************************************************************/

int tedInsertRowInTable(	EditDocument *		ed )
    {
    BufferItem *		sectBi;
    BufferItem *		refRowBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    const int			rows= 1;

    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    DocumentPosition		dpRef;
    int				part;
    const int			lastOne= 1;
    int				paraNr;
    int				textAttributeNumber;

    /*  1  */
    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    /*  2  */
    if  ( docDelimitTable( teo.teoEo.eoHeadDp.dpBi, &sectBi,
						&col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    /*  3  */
    refRowBi= sectBi->biChildren[row];
    if  ( docFirstPosition( &dpRef, refRowBi ) )
	{ LDEB(row); return -1;	}
    if  ( docFindParticuleOfPosition( &part, &dpRef, lastOne ) )
	{ LDEB(dpRef.dpStroff); return -1;	}

    /*  4  */
    paraNr= docNumberOfParagraph( dpRef.dpBi );
    textAttributeNumber= dpRef.dpBi->biParaParticules[part].
						    tpTextAttrNr;

    /*  5  */
    if  ( tedInsertRowsInTable( sectBi, &teo,
		    paraNr, textAttributeNumber, refRowBi, ed, row, rows ) )
	{ LDEB(row); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert a row after the current row in a table.			*/
/*									*/
/*  1)  Start edit operation.						*/
/*  2)  Find the position of the end of the selection in the table	*/
/*  3)  Get the row that serves as a template.				*/
/*  4)  Paragraph number of the paragraph that comes directly below	*/
/*	the fresh rows.							*/
/*  5)  Finish..							*/
/*									*/
/************************************************************************/

int tedAppendRowToTable(	EditDocument *		ed )
    {
    BufferItem *		sectBi;
    BufferItem *		refRowBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    const int			rows= 1;

    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    DocumentPosition		dpRef;
    int				part;
    const int			lastOne= 0;
    int				paraNr;
    int				textAttributeNumber;

    /*  1  */
    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    /*  2  */
    if  ( docDelimitTable( teo.teoEo.eoTailDp.dpBi, &sectBi,
						&col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    /*  3  */
    refRowBi= sectBi->biChildren[row];
    if  ( docLastPosition( &dpRef, refRowBi ) )
	{ LDEB(row); return -1;	}
    if  ( docFindParticuleOfPosition( &part, &dpRef, lastOne ) )
	{ LDEB(dpRef.dpStroff); return -1;	}

    /*  4  */
    paraNr= docNumberOfParagraph( dpRef.dpBi )+ 1;
    textAttributeNumber= dpRef.dpBi->biParaParticules[part].
							tpTextAttrNr;

    /*  5  */
    if  ( tedInsertRowsInTable( sectBi, &teo,
		    paraNr, textAttributeNumber, refRowBi, ed, row+ 1, rows ) )
	{ LDEB(row); return -1;	}
    
    return 0;
    }

/************************************************************************/
/*									*/
/*  Delete rows from a table.						*/
/*									*/
/************************************************************************/

static int tedDeleteRowsFromTableOperation(
					TedEditOperation *	teo,
					int			row0,
					int			row1 )
    {
    EditOperation *		eo= &(teo->teoEo);

    int				rval= 0;

    BufferItem *		selSectBi;

    int				col;
    int				row;
    int				tabRow0;
    int				tabRow1;

    DocumentPosition		dpNew;

    int				paraNr;
    int				sectionsDeleted= 0;
    int				firstParaDeleted= -1;
    int				paragraphsDeleted= 0;

    PropertyMask		ppDoneMask;
    PropertyMask		ppSetMask;

    DocumentSelection		dsRows;

    ParagraphProperties		pp;

    docInitParagraphProperties( &pp );

    if  ( docDelimitTable( teo->teoEo.eoHeadDp.dpBi, &selSectBi,
					    &col, &tabRow0, &row, &tabRow1 ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( row0 > row1 )
	{ LLDEB(row0,row1); rval= -1; goto ready;	}
    if  ( row0 < tabRow0 || row0 > tabRow1 )
	{ LLLDEB(tabRow0,row0,tabRow1); rval= -1; goto ready;	}
    if  ( row1 < tabRow0 || row1 > tabRow1 )
	{ LLLDEB(tabRow0,row1,tabRow1); rval= -1; goto ready;	}

    docInitDocumentSelection( &dsRows );
    if  ( docFirstPosition( &dsRows.dsHead, selSectBi->biChildren[row0] ) )
	{ LLDEB(row0,row1); rval= -1; goto ready;	}
    if  ( docLastPosition( &dsRows.dsTail, selSectBi->biChildren[row1] ) )
	{ LLDEB(row0,row1); rval= -1; goto ready;	}

    paraNr= docNumberOfParagraph( dsRows.dsTail.dpBi )+ 1;

    utilPropMaskClear( &ppDoneMask );
    utilPropMaskClear( &ppSetMask );
    utilPropMaskFill( &ppSetMask, PPprop_FULL_COUNT );
    PROPmaskUNSET( &ppSetMask, PPpropTABLE_NESTING );

    if  ( docUpdParaProperties( &ppDoneMask, &pp,
			    &ppSetMask, &(dsRows.dsTail.dpBi->biParaProperties),
			    (const DocumentAttributeMap *)0 ) )
	{ LDEB(1); rval= -1; goto ready;	}

    docInitDocumentPosition( &dpNew );
    dpNew= dsRows.dsTail;
    if  ( docNextPosition( &dpNew ) )
	{
	dpNew= dsRows.dsHead;
	if  ( docPrevPosition( &dpNew ) )
	    { docInitDocumentPosition( &dpNew );	}
	}

    tedEditIncludeRowsInRedraw( teo, selSectBi, row0, row1 );

    if  ( docEditDeleteFieldsForBlockDelete( eo, &dsRows ) )
	{ LDEB(1); rval= -1; goto ready;	}

    docEditDeleteItems( eo, &sectionsDeleted,
				&firstParaDeleted, &paragraphsDeleted,
				selSectBi, row0, row1- row0+ 1 );

    docEditIncludeItemInReformatRange( eo, selSectBi );

    if  ( ! dpNew.dpBi )
	{
	const int	sectShift= 0;
	EditDocument *	ed= teo->teoEditDocument;
	TedDocument *	td= (TedDocument *)ed->edPrivateData;

	if  ( docSectionParagraph( eo, &(dpNew.dpBi),
				    selSectBi, sectShift,
				    &pp, td->tdCurrentTextAttributeNumber ) )
	    { LDEB(1); rval= -1; goto ready;	}

	docEditIncludeItemInReformatRange( eo, selSectBi );

	if  ( docFirstPosition( &dpNew, dpNew.dpBi ) )
	    { LDEB(1);	}
	}

    docDelimitTables( selSectBi );

    tedEditFinishIBarSelection( teo, dpNew.dpBi, dpNew.dpStroff );

  ready:

    docCleanParagraphProperties( &pp );

    return rval;
    }

/************************************************************************/
/*									*/
/*  Roll rows in a table.						*/
/*									*/
/************************************************************************/

static int tedRollRowsInTableOperation(
				EditDocument *			ed,
				TedEditOperation *		teo,
				const SelectionDescription  *	sd,
				int				row0,
				int				row1,
				int				rowsdown )
    {
    EditOperation *		eo= &(teo->teoEo);
    int				rval= 0;

    BufferItem *		selSectBi;

    int				col;
    int				row;
    int				tabRow0;
    int				tabRow1;
    int				rowsHigh= row1- row0+ 1;

    DocumentPosition		dp;

    EditRange			selectedRange= eo->eoSelectedRange;
    int				paraNr;
    int				paraOff0;
    int				paraOff1;
    int				selRow0= sd->sdTableRectangle.trRow0;
    int				selRow1= sd->sdTableRectangle.trRow1;

    if  ( docDelimitTable( teo->teoEo.eoHeadDp.dpBi, &selSectBi,
					    &col, &tabRow0, &row, &tabRow1 ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( rowsdown < 0 )
	{ rowsdown= rowsHigh+ rowsdown;	}

    if  ( row0 >= row1 )
	{ LLDEB(row0,row1); rval= -1; goto ready;	}
    if  ( row0 < tabRow0 || row0 > tabRow1 )
	{ LLLDEB(tabRow0,row0,tabRow1); rval= -1; goto ready;	}
    if  ( row1 < tabRow0 || row1 > tabRow1 )
	{ LLLDEB(tabRow0,row1,tabRow1); rval= -1; goto ready;	}
    if  ( rowsdown < 1 || rowsdown >= rowsHigh )
	{ LLDEB(rowsdown,rowsHigh); rval= -1; goto ready;	}

    docInitDocumentPosition( &dp );
    if  ( docFirstPosition( &dp, selSectBi->biChildren[selRow0] ) )
	{ LDEB(selRow0); rval= -1; goto ready;	}
    paraNr= docNumberOfParagraph( dp.dpBi );
    paraOff0= selectedRange.erHead.epParaNr- paraNr;
    if  ( docFirstPosition( &dp, selSectBi->biChildren[selRow1] ) )
	{ LDEB(selRow1); rval= -1; goto ready;	}
    paraNr= docNumberOfParagraph( dp.dpBi );
    paraOff1= selectedRange.erTail.epParaNr- paraNr;
    if  ( paraOff0 < 0 || paraOff1 < 0 )
	{ LLDEB(paraOff0,paraOff1); rval= -1; goto ready;	}

    tedEditIncludeRowsInRedraw( teo, selSectBi, row0, row1 );

    if  ( docRollItemChildren( eo, selSectBi, row0, row1+ 1, rowsdown ) )
	{ LLLDEB(row0,row1,rowsdown); rval= -1; goto ready;	}

    docEditIncludeItemInReformatRange( eo, selSectBi );

    docDelimitTables( selSectBi );

    selRow0 += rowsdown;
    if  ( selRow0 > row1 )
	{ selRow0 -= rowsHigh;	}
    selRow1 += rowsdown;
    if  ( selRow1 > row1 )
	{ selRow1 -= rowsHigh;	}

    if  ( docFirstPosition( &dp, selSectBi->biChildren[selRow0] ) )
	{ LDEB(selRow0); rval= -1; goto ready;	}
    paraNr= docNumberOfParagraph( dp.dpBi );
    selectedRange.erHead.epParaNr= paraNr+ paraOff0;
    if  ( docFirstPosition( &dp, selSectBi->biChildren[selRow1] ) )
	{ LDEB(selRow1); rval= -1; goto ready;	}
    paraNr= docNumberOfParagraph( dp.dpBi );
    selectedRange.erTail.epParaNr= paraNr+ paraOff1;
    if  ( selectedRange.erHead.epParaNr < 0	||
	  selectedRange.erTail.epParaNr < 0	)
	{ LDEB(1); rval= -1; goto ready;	}

    eo->eoSelectedRange= selectedRange;
    tedEditFinishOldSelection( teo );

  ready:

    return rval;
    }

/************************************************************************/

int tedDeleteRowsFromTable(	EditDocument *		ed,
				int			row0,
				int			row1 )
    {
    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    if  ( tedDeleteRowsFromTableOperation( &teo, row0, row1 ) )
	{ LDEB(1); return -1;	}

    appDocumentChanged( ed, 1 );

    return 0;
    }

int tedRollRowsInTable(		EditDocument *		ed,
				int			rowsdown )
    {
    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    int				row0;
    int				row1;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    row0= sd.sdTableRectangle.trRow0;
    row1= sd.sdTableRectangle.trRow1;

    if  ( rowsdown < 0 )
	{ row0 += rowsdown;	}
    if  ( rowsdown > 0 )
	{ row1 += rowsdown;	}

    if  ( tedRollRowsInTableOperation( ed, &teo, &sd, row0, row1, rowsdown ) )
	{ LDEB(1); return -1;	}

    appDocumentChanged( ed, 1 );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Insert cells/columns in tables.					*/
/*									*/
/************************************************************************/

static int tedSplitColumnInRows(	TedEditOperation *	teo,
					BufferItem *		sectBi,
					int			row0,
					int			row,
					int			row1,
					int			col,
					int			after )
    {
    EditOperation *		eo= &(teo->teoEo);
    BufferItem *		newParaBi= (BufferItem *)0;

    tedEditIncludeRowsInRedraw( teo, sectBi, row0, row1 );

    if  ( docSplitColumnInRows( &newParaBi, eo,
				    sectBi, row0, row, row1, col, after ) )
	{ LDEB(1); return -1;	}

    docDelimitTables( sectBi );

    tedEditFinishIBarSelection( teo, newParaBi, 0 );

    return 0;
    }

int tedInsertColumnInTable(	EditDocument *		ed )
    {
    BufferItem *		sectBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    int				after= 0;

    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    if  ( docDelimitTable( teo.teoEo.eoHeadDp.dpBi, &sectBi,
					    &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    return tedSplitColumnInRows( &teo, sectBi, row0, row, row1, col, after );
    }

int tedAppendColumnToTable(	EditDocument *		ed )
    {
    BufferItem *		sectBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    int				after= 1;

    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    if  ( docDelimitTable( teo.teoEo.eoHeadDp.dpBi, &sectBi,
					    &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    return tedSplitColumnInRows( &teo, sectBi, row0, row, row1, col, after );
    }

static int tedDeleteColumnsFromRowsOperation(
					TedEditOperation *	teo,
					int			delRow0,
					int			delRow1,
					int			delCol0,
					int			delCol1 )
    {
    EditOperation *		eo= &(teo->teoEo);

    BufferItem *		sectBi;
    BufferItem *		rowBi;

    int				col;
    int				row;
    int				row0;
    int				row1;

    DocumentPosition		dpNew;

    if  ( docDelimitTable( teo->teoEo.eoHeadDp.dpBi, &sectBi,
					    &col, &row0, &row, &row1 ) )
	{ LDEB(1); return -1;	}

    tedEditIncludeRowsInRedraw( teo, sectBi, delRow0, delRow1 );

    docInitDocumentPosition( &dpNew );
    rowBi= sectBi->biChildren[delRow1];
    if  ( delCol1 >= rowBi->biChildCount- 1			||
	  docLastPosition( &dpNew, rowBi->biChildren[delCol1] )	||
	  docNextPosition( &dpNew )				)
	{
	docInitDocumentPosition( &dpNew );
	rowBi= sectBi->biChildren[delRow0];
	if  ( delCol0 <= 0						||
	      docFirstPosition( &dpNew, rowBi->biChildren[delCol0] )	||
	      docPrevPosition( &dpNew )					)
	    { docInitDocumentPosition( &dpNew ); }
	}

    if  ( docDeleteColumnsFromRows( eo, sectBi,
				    delRow0, delRow1, delCol0, delCol1 ) )
	{ LDEB(1); return -1;	}

    tedEditFinishIBarSelection( teo, dpNew.dpBi, dpNew.dpStroff );

    return 0;
    }

int tedDeleteColumnsFromRows(	EditDocument *	ed,
				int		delRow0,
				int		delRow1,
				int		delCol0,
				int		delCol1 )
    {
    int				rval;

    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    rval= tedDeleteColumnsFromRowsOperation( &teo,
					delRow0, delRow1, delCol0, delCol1 );

    appDocumentChanged( ed, 1 );

    return rval;
    }

/************************************************************************/
/*									*/
/*  Implementation of the 'Change' functionality of the table related	*/
/*  pages of the format tool.						*/
/*									*/
/*  1)  Retrieve the current selection of the document.			*/
/*  2)  Obtain a selection for the table rectangle.			*/
/*  3)  Were the colspan/rowspan of a cell to be changed?		*/
/*  4)  Remove spans from mask.						*/
/*  5)  Update other properties.					*/
/*									*/
/************************************************************************/

void tedDocSetTableProperties(	EditDocument *		ed,
				const TableRectangle *	trChange,
				const PropertyMask *	rpSetMask,
				const PropertyMask *	cpSetMask,
				const RowProperties *	rpFrom )
    {
    TedDocument *		td;
    BufferDocument *		bd;

    BufferItem *		selParentBi;
    int				row1Reformat;

    DocumentSelection		dsNew;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    PropertyMask		cellSpanMask;
    PropertyMask		cpSetMaskStripped;

    td= (TedDocument *)ed->edPrivateData;
    bd= td->tdDocument;

    /*  1  */
    docInitDocumentSelection( &dsNew );
    if  ( tedGetSelection( &dsNew, &sg, &sd, 
			    (DocumentTree **)0, (const BufferItem **)0, td ) )
	{ LDEB(1); return;	}

    /*  2  */
    if  ( docTableRectangleSelection( &dsNew, &selParentBi, bd, trChange ) )
	{ LDEB(1); return;	}

    row1Reformat= trChange->trRow1;

    /*  3  */
    utilPropMaskClear( &cellSpanMask );
    PROPmaskADD( &cellSpanMask, CLprop_ROWSPAN );
    PROPmaskADD( &cellSpanMask, CLprop_COLSPAN );

    utilPropMaskAnd( &cellSpanMask, &cellSpanMask, cpSetMask );

    if  ( ! utilPropMaskIsEmpty( &cellSpanMask ) )
	{
	int		row1SpanChanged= trChange->trRow0- 1;

	if  ( docChangeCellSpans( &row1SpanChanged, selParentBi,
					    trChange->trRow0, trChange->trCol0,
					    trChange->trCellRowspan,
					    trChange->trCellColspan ) )
	    { LDEB(1); return;	}

	if  ( row1Reformat < row1SpanChanged )
	    { row1Reformat=  row1SpanChanged;	}
	}

    /*  4  */
    cpSetMaskStripped= *cpSetMask;
    PROPmaskUNSET( &cpSetMaskStripped, CLprop_ROWSPAN );
    PROPmaskUNSET( &cpSetMaskStripped, CLprop_COLSPAN );

    /*  5  */
    tedChangeTableLayout( ed, selParentBi,
				    trChange->trRow0, trChange->trRow1,
				    trChange->trCol0, trChange->trCol1,
				    row1Reformat,
				    rpSetMask, &cpSetMaskStripped, rpFrom );

    tedAdaptToolsToSelection( ed );

    if  ( ! utilPropMaskIsEmpty( rpSetMask )	||
	  ! utilPropMaskIsEmpty( cpSetMask )	)
	{ appDocumentChanged( ed, 1 );	}

    return;
    }

void tedAppSetTableProperties(	EditApplication *	ea,
				const TableRectangle *	trChange,
				const PropertyMask *	rpSetMask,
				const PropertyMask *	cpSetMask,
				const RowProperties *	rpFrom )
    {
    EditDocument *		ed= ea->eaCurrentDocument;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    tedDocSetTableProperties( ed, trChange, rpSetMask, cpSetMask, rpFrom );

    return;
    }

/************************************************************************/
/*									*/
/*  Select a rectangle in a table.					*/
/*									*/
/************************************************************************/

void tedAppSetTableSelection(	EditDocument *		ed,
				const TableRectangle *	trSet )
    {
    TedDocument *		td;

    TableRectangle		tr;

    int				tableRectangle= 0;

    DocumentSelection		ds;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    td= (TedDocument *)ed->edPrivateData;

    tedGetSelection( &ds, &sg, &sd,
			    (DocumentTree **)0, (const BufferItem **)0, td );

    if  ( ! docGetTableRectangle( &tr, &ds ) )
	{ tableRectangle= 1;	}

    if  ( ! tableRectangle )
	{ LDEB(tableRectangle); return;	}

    docExpandTableRectangleToWholeTable( &tr );

    if  ( ! docIntersectTableRectangle( &tr, &tr, trSet ) )
	{ return;	}

    tedDocTableSelectTableRectangle( ed, &tr );

    return;
    }

/************************************************************************/
/*									*/
/*  Change table layout from various table related tools.		*/
/*									*/
/*  1)  Reformat at least all rows that are touched.			*/
/*  2)  Start edit operation.						*/
/*  3)  Include all rows touched by the operation in the reformat.	*/
/*  4)  If the 'IsTableHeader' property is changed, the whole table	*/
/*	must be reformatted.						*/
/*  5)  Apply changes.							*/
/*  6)  Finish edit operation. I.E: Redo layout where needed and expose	*/
/*	the affected area to be redrawn.				*/
/*									*/
/************************************************************************/

void tedChangeTableLayout(	EditDocument *		ed,
				BufferItem *		sectBi,
				int			row0Change,
				int			row1Change,
				int			col0Change,
				int			col1Change,
				int			row1Reformat,
				const PropertyMask *	rpSetMask,
				const PropertyMask *	cpSetMask,
				const RowProperties *	rpSet )
    {
    TedEditOperation		teo;
    EditOperation *		eo= &(teo.teoEo);
    SelectionGeometry		sg;
    SelectionDescription	sd;

    /*  1  */
    if  ( row1Reformat < row1Change )
	{ row1Reformat=  row1Change;	}

    /*  2  */
    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    /*  3  */
    docEditIncludeRowsInReformatRange( eo,
				    sectBi, row0Change, row1Reformat );
    tedEditIncludeRowsInRedraw( &teo,
				    sectBi, row0Change, row1Reformat );

    /*  4  */
    if  ( PROPmaskISSET( rpSetMask, RPpropIS_TABLE_HEADER ) )
	{
	const BufferItem *	rowBi= sectBi->biChildren[row0Change];

	docEditIncludeRowsInReformatRange( eo,
				sectBi, row0Change, rowBi->biRowTablePast- 1 );
	tedEditIncludeRowsInRedraw( &teo,
				sectBi, row0Change, rowBi->biRowTablePast- 1 );
	}

    /*  5  */
    docChangeTableLayout( eo, sectBi,
			    row0Change, row1Change, col0Change, col1Change,
			    rpSetMask, cpSetMask, rpSet );

    /*  6  */
    if  ( tedEditFinishOldSelection( &teo ) )
	{ LDEB(1);	}

    return;
    }

/************************************************************************/
/*									*/
/*  Delete a slice from a table. Either horizontal or vertical or both:	*/
/*  the table as a whole.						*/
/*									*/
/*  1)  See whether the current selection is a table slice.		*/
/*  2)  If it is a row slice (Whole rows) delete the rows. This covers	*/
/*	deleteing the whole table as well. [ Handling it in the column	*/
/*	branch would leave empty rows.]					*/
/*  3)  If it is a column slice (Whole columns) delete the columns.	*/
/*  4)  Impossible!							*/
/*									*/
/************************************************************************/

int tedDeleteTableSliceSelection(	EditDocument *		ed )
    {
    int				isRowSlice= 0;
    int				isColSlice= 0;
    TableRectangle		tr;

    TedEditOperation		teo;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    int				ret;

    if  ( ed->edIsReadonly )
	{ return 1;	}

    tedStartEditOperation( &teo, &sg, &sd, ed, 1 );

    /*  1  */
    {
    DocumentSelection		ds;

    docEditOperationGetSelection( &ds, &(teo.teoEo) );

    if  ( docGetTableSliceSelection( &isRowSlice, &isColSlice, &tr, &ds ) )
	{ return 1;	}
    }

    /*  2  */
    if  ( isRowSlice )
	{
	ret= tedDeleteRowsFromTableOperation( &teo, tr.trRow0, tr.trRow1 );
	if  ( ret )
	    { LDEB(ret); return -1;	}

	appDocumentChanged( ed, 1 );
	return 0;
	}

    /*  3  */
    if  ( isColSlice )
	{
	ret= tedDeleteColumnsFromRowsOperation( &teo,
				tr.trRow0, tr.trRow1, tr.trCol0, tr.trCol1 );
	if  ( ret )
	    { LDEB(ret); return -1;	}

	appDocumentChanged( ed, 1 );
	return 0;
	}

    /*  4  */
    LLDEB(isRowSlice,isColSlice); return -1;
    }

