/************************************************************************/
/*									*/
/*  Open/Close objects inserted in the document.			*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>

#   include	<sioMemory.h>
#   include	<sioHex.h>
#   include	<sioPipe.h>
#   include	<appSystem.h>

#   include	"tedDraw.h"
#   include	"tedLayout.h"
#   include	"docScreenLayout.h"
#   include	<appImage.h>
#   include	<appWinMeta.h>
#   include	<guiWidgetPixmap.h>

#   include	"docPageGrid.h"
#   include	"docLayout.h"
#   include	"docPixels.h"
#   include	"tedApp.h"
#   include	<appFileChooser.h>

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Clip and draw the visible part of a pixmap.				*/
/*									*/
/*  X11 can get in serious trouble if we draw to negative coordinates	*/
/*  so really avoid this. To hide the similarity to MS-Windows we say	*/
/*  this is an optimization.						*/
/*									*/
/*  1)  Destination rectangle.						*/
/*  2)  Clip if needed.							*/
/*  3)  Shift by scrollbars.						*/
/*  4)  Do not draw what protrudes on the top and left sides of the	*/
/*	destination. (Clipping usually takes care op right/bottom).	*/
/*  5)  Draw if something left. The source coordinates reflect what	*/
/*	has been cut off on the top and left by clipping.		*/
/*									*/
/************************************************************************/

static void tedDrawObjectPixmap( const InsertedObject *		io,
				int				x0,
				int				y0,
				const DocumentRectangle *	drClip,
				const LayoutContext *		lc )
    {
    DocumentRectangle	drDest;
    int			destX0= x0- lc->lcOx;
    int			destY0= y0- lc->lcOy;

    /*  1  */
    drDest.drX0= x0;
    drDest.drY0= y0;
    drDest.drX1= drDest.drX0+ io->ioPixelsWide- 1;
    drDest.drY1= drDest.drY0+ io->ioPixelsHigh- 1;

    /*  2  */
    if  ( drClip )
	{
	if  ( ! geoIntersectRectangle( &drDest, &drDest, drClip ) )
	    { return;	}
	}

    /*  3  */
    drDest.drX0 -= lc->lcOx;
    drDest.drY0 -= lc->lcOy;
    drDest.drX1 -= lc->lcOx;
    drDest.drY1 -= lc->lcOy;

    /*  4  */
    if  ( drDest.drX0 < 0 )
	{ drDest.drX0=  0;	}
    if  ( drDest.drY0 < 0 )
	{ drDest.drY0=  0;	}

    /*  5  */
    if  ( drDest.drX1 >= drDest.drX0 && drDest.drY1 >= drDest.drY0 )
	{
	tedDrawPixmap( lc, io->ioScreenPixmap,
			drDest.drX0- destX0,		/*  src x	*/
			drDest.drY0- destY0,		/*  src y	*/
			drDest.drX1- drDest.drX0+ 1,	/*  wide	*/
			drDest.drY1- drDest.drY0+ 1,	/*  high	*/
			drDest.drX0, drDest.drY0 );	/*  dest: x,y	*/
	}
    }

/************************************************************************/
/*									*/
/*  Draw an object.							*/
/*									*/
/************************************************************************/

int tedDrawObject(	const TextParticule *		tp,
			int				x0PixelsShifted,
			int				baseLinePixels,
			const DocumentRectangle *	drClip,
			const LayoutContext *		lc )
    {
    AppDrawingData *		add= lc->lcAdd;
    const InsertedObject *	io;
    int				x0;
    int				y0;

    io= docGetObject( lc->lcDocument, tp->tpObjectNumber );
    if  ( ! io )
	{ LPDEB(tp->tpObjectNumber,io); return 0;	}

    x0= x0PixelsShifted;
    y0= baseLinePixels- io->ioPixelsHigh;

    if  ( ! io->ioInline )
	{
	x0= X_PIXELS( add, io->ioX0Twips );
	y0= LP_YPIXELS( add, &(io->ioY0Position) );
	}

    if  ( io->ioKind == DOCokDRAWING_SHAPE )
	{ return 0;	}

    if  ( io->ioKind == DOCokPICTWMETAFILE	||
	  io->ioKind == DOCokPICTEMFBLIP	||
	  io->ioKind == DOCokMACPICT		||
	  io->ioKind == DOCokPICTPNGBLIP	||
	  io->ioKind == DOCokPICTJPEGBLIP	)
	{
	tedDrawObjectPixmap( io, x0, y0, drClip, lc );
	return 0;
	}

    if  ( io->ioKind == DOCokOLEOBJECT				&&
	  ( io->ioResultKind == DOCokPICTWMETAFILE	||
	    io->ioResultKind == DOCokPICTEMFBLIP	||
	    io->ioResultKind == DOCokMACPICT		||
	    io->ioResultKind == DOCokPICTPNGBLIP	||
	    io->ioResultKind == DOCokPICTJPEGBLIP	)	)
	{
	tedDrawObjectPixmap( io, x0, y0, drClip, lc );
	return 0;
	}

    if  ( io->ioKind == DOCokEPS_FILE			)
	{
	tedDrawObjectPixmap( io, x0, y0, drClip, lc );
	return 0;
	}

    appDrawSetLineAttributes( add,
			    1, LINEstyleSOLID, LINEcapBUTT, LINEjoinMITER,
			    (const unsigned char *)0, 0 );

    appDrawDrawRectangle( add,
		x0- lc->lcOx, baseLinePixels- io->ioPixelsHigh- lc->lcOy,
		io->ioPixelsWide, io->ioPixelsHigh );

    return 0;
    }

typedef int (*PLAY_META_X11)(	SimpleInputStream *	sis,
				void **			pPrivate,
				AppDrawingData *	add,
				ScreenPixmap		sp,
				int			mapMode,
				int			xExt,
				int			yExt,
				int			pixelsWide,
				int			pixelsHigh,
				int			twipsWide,
				int			twipsHigh );

static int tedOpenMetafileObject(
				void **				pPrivate,
				ScreenPixmap *			pSp,
				const PictureProperties *	pip,
				int				pixelsWide,
				int				pixelsHigh,
				PLAY_META_X11			playMetaX11,
				const LayoutContext *		lc,
				const MemoryBuffer *		mb )
    {
    SimpleInputStream *	sisMem;
    SimpleInputStream *	sisMeta;

    void *		ownData= (void *)0;
    ScreenPixmap	sp= (ScreenPixmap)0;

    sisMem= sioInMemoryOpen( mb );
    if  ( ! sisMem )
	{ XDEB(sisMem); return -1;	}

    sisMeta= sioInHexOpen( sisMem );
    if  ( ! sisMeta )
	{ XDEB(sisMem); return -1;	}

    sp= guiMakePixmapForWidget( lc->lcScreenWidget, pixelsWide, pixelsHigh );
    if  ( ! sp )
	{ XDEB(sp); return -1;	}

    if  ( (*playMetaX11)( sisMeta, &ownData, lc->lcAdd, sp, pip->pipMapMode,
				    pip->pip_xWinExt, pip->pip_yWinExt,
				    pixelsWide, pixelsHigh,
				    pip->pipTwipsWide, pip->pipTwipsHigh ) )
	{ LDEB(1);	}

    sioInClose( sisMeta );
    sioInClose( sisMem );

    *pPrivate= ownData;
    *pSp= sp;
    return 0;
    }

static int tedOpenBitmapObject(	void **				pPrivate,
				ScreenPixmap *			pSp,
				int				pixelsWide,
				int				pixelsHigh,
				bmReadBitmap			readBitmap,
				const LayoutContext *		lc,
				const MemoryBuffer *		mb )
    {
    AppBitmapImage *	abi= (AppBitmapImage *)malloc( sizeof(AppBitmapImage) );

    SimpleInputStream *	sisMem;
    SimpleInputStream *	sisBitmap;

    ScreenPixmap	sp;

    if  ( ! abi )
	{ XDEB(abi); return -1;	}
    bmInitBitmapImage( abi );

    sisMem= sioInMemoryOpen( mb );
    if  ( ! sisMem )
	{ XDEB(sisMem); return -1;	}

    sisBitmap= sioInHexOpen( sisMem );
    if  ( ! sisBitmap )
	{ XDEB(sisMem); return -1;	}

    if  ( (*readBitmap)( &abi->abiBitmap, &(abi->abiBuffer), sisBitmap ) )
	{ LDEB(1); return -1;	}

    sioInClose( sisBitmap );
    sioInClose( sisMem );

    sp= guiMakePixmapForImageAndWidget( lc->lcScreenWidget,
		abi, (const DocumentRectangle *)0, pixelsWide, pixelsHigh );
    if  ( ! sp )
	{
	PDEB(sp);
	bmCleanBitmapImage( abi ); free( abi );
	return -1;
	}

    *pPrivate= (void *)abi;
    *pSp= sp;
    return 0;
    }

static int tedOpenPixmap(	void **				pPrivate,
				ScreenPixmap *			pSp,
				const PictureProperties *	pip,
				int				kind,
				int				pixelsWide,
				int				pixelsHigh,
				const LayoutContext *		lc,
				const MemoryBuffer *		mb )
    {
    switch( kind )
	{
	case DOCokPICTWMETAFILE:
	    if  ( tedOpenMetafileObject( pPrivate, pSp, pip,
					pixelsWide, pixelsHigh,
					appMetaPlayWmfX11, lc, mb ) )
		{ LDEB(1); return 0;	}

	    return 0;

	case DOCokPICTEMFBLIP:
	    if  ( tedOpenMetafileObject( pPrivate, pSp, pip,
					pixelsWide, pixelsHigh,
					appMetaPlayEmfX11, lc, mb ) )
		{ LDEB(1); return 0;	}
	    return 0;

	case DOCokMACPICT:
	    if  ( tedOpenMetafileObject( pPrivate, pSp, pip,
					pixelsWide, pixelsHigh,
					appMacPictPlayFileX11, lc, mb ) )
		{ LDEB(1); return 0;	}
	    return 0;

	case DOCokPICTPNGBLIP:
	    if  ( tedOpenBitmapObject( pPrivate, pSp,
				    pixelsWide, pixelsHigh,
				    bmPngReadPng, lc, mb )	)
		{ LDEB(1); return 0;	}
	    return 0;

	case DOCokPICTJPEGBLIP:
	    if  ( tedOpenBitmapObject( pPrivate, pSp,
				    pixelsWide, pixelsHigh,
				    bmJpegReadJfif, lc, mb ) )
		{ LDEB(1); return 0;	}
	    return 0;

	default:
	    LDEB(kind); return 0;
	}
    }

static int tedOpenEpsfObject(		InsertedObject *	io,
					AppDrawingData *	add,
					const MemoryBuffer *	mb )
    {
#   if 0
    SimpleInputStream *		sisMem;
    SimpleInputStream *		sisEps;
    SimpleOutputStream *	sosGs;

    char			command[1000];

    static Atom			XA_GHOSTVIEW= None;

    if  ( XA_GHOSTVIEW == None )
	{
	XA_GHOSTVIEW= XInternAtom( add->addDisplay, "GHOSTVIEW", False );
	if  ( XA_GHOSTVIEW == None )
	    { XDEB(XA_GHOSTVIEW); return -1;	}
	}

    io->ioScreenPixmap= appMakePixmap( add, io->ioPixelsWide, io->ioPixelsHigh );
    if  ( ! io->ioScreenPixmap )
	{ XDEB(io->ioScreenPixmap); return -1;	}

    sprintf( command, "0 0 %d %d %d %d %f %f",
				/* llx	*/ 0,
				/* lly	*/ 0,
				/* urx	*/ io->ioTwipsWide/ 20,
				/* ury	*/ io->ioTwipsHigh/ 20,
				/* xdpi	*/ 72.0,
				/* ydpi	*/ 72.0 );

    XChangeProperty( add->addDisplay, add->addDrawable,
			XA_GHOSTVIEW, XA_STRING, 8,
			PropModeReplace,
			(unsigned char *)command,
			strlen(command) );

    XSync( add->addDisplay, False );

    sprintf( command,
	    "GHOSTVIEW='%ld %ld' gs -dQUIET -sDEVICE=x11 -dNOPAUSE -c quit -",
	    add->addDrawable, io->ioScreenPixmap );
    sosGs= sioOutPipeOpen( command );

    sisMem= sioInMemoryOpen( mb );
    if  ( ! sisMem )
	{ XDEB(sisMem); return -1;	}

    sisEps= sioInHexOpen( sisMem );
    if  ( ! sisEps )
	{ XDEB(sisMem); return -1;	}

    for (;;)
	{
	int		got;

	got= sioInReadBytes( sisEps,
				(unsigned char *)command, sizeof( command ) );
	if  ( got < 1 )
	    { break;	}
	if  ( sioOutWriteBytes( sosGs, (unsigned char *)command, got ) != got )
	    { LDEB(got); break;	}
	}

    sioInClose( sisEps );
    sioInClose( sisMem );

    sioOutPrintf( sosGs, " showpage\n" );
    sioOutClose( sosGs );

#   endif
    return 0;
    }

static int tedOpenShapePixmaps(	DrawingShape *			ds,
				const DocumentRectangle *	drTwips,
				const LayoutContext *		lc )
    {
    AppDrawingData *		add= lc->lcAdd;
    double			xfac= add->addMagnifiedPixelsPerTwip;

    int				rval= 0;
    const PictureProperties *	pip= &(ds->dsPictureProperties);
    int				pixelsWide;
    int				pixelsHigh;

    pixelsWide= X_PIXELS( add, drTwips->drX1 )-
		X_PIXELS( add, drTwips->drX0 )+ 1;
    pixelsHigh= COORDtoGRID( xfac, drTwips->drY1 )-
		COORDtoGRID( xfac, drTwips->drY0 )+ 1;

    if  ( ds->dsChildCount > 0 )
	{
	int			child;
	double			xscale;
	double			yscale;

	DocumentRectangle	drHere= *drTwips;

	if  ( DSflipHORIZONTAL( ds ) )
	    { int sw= drHere.drX0; drHere.drX0= drHere.drX1; drHere.drX1= sw; }

	if  ( DSflipVERTICAL( ds ) )
	    { int sw= drHere.drY0; drHere.drY0= drHere.drY1; drHere.drY1= sw; }

	xscale= ( 1.0* ( drHere.drX1- drHere.drX0 ) )/
			    ( ds->dsGroupRect.drX1- ds->dsGroupRect.drX0 );
	yscale= ( 1.0* ( drHere.drY1- drHere.drY0 ) )/
			    ( ds->dsGroupRect.drY1- ds->dsGroupRect.drY0 );

	for ( child= 0; child < ds->dsChildCount; child++ )
	    {
	    DrawingShape *	dsChild= ds->dsChildren[child];
	    DocumentRectangle	drChild;

	    drChild.drX0= drHere.drX0+ xscale* 
			( dsChild->dsRelRect.drX0- ds->dsGroupRect.drX0 );
	    drChild.drY0= drHere.drY0+ yscale*
			( dsChild->dsRelRect.drY0- ds->dsGroupRect.drY0 );
	    drChild.drX1= drHere.drX0+ xscale* 
			( dsChild->dsRelRect.drX1- ds->dsGroupRect.drX0 );
	    drChild.drY1= drHere.drY0+ yscale*
			( dsChild->dsRelRect.drY1- ds->dsGroupRect.drY0 );

	    if  ( tedOpenShapePixmaps( dsChild, &drChild, lc ) )
		{ LDEB(child); rval= -1;	}
	    }
	}

    switch( pip->pipType )
	{
	case DOCokUNKNOWN:
	    break;

	case DOCokPICTWMETAFILE:
	case DOCokPICTEMFBLIP:
	case DOCokMACPICT:
	case DOCokPICTPNGBLIP:
	case DOCokPICTJPEGBLIP:
	    if  ( ! ds->dsPrivate					&&
		  tedOpenPixmap( &(ds->dsPrivate), &(ds->dsScreenPixmap),
				    pip, pip->pipType, pixelsWide, pixelsHigh,
				    lc, &(ds->dsPictureData) )	)
		{ LDEB(1); rval= -1; break;	}
	    break;

	default:
	    LDEB(pip->pipType); rval= -1; break;
	}

    return rval;
    }

int tedOpenObject(	InsertedObject *	io,
			const LayoutContext *	lc )
    {
    AppDrawingData *	add= lc->lcAdd;
    PictureProperties *	pip= &(io->ioPictureProperties);

    if  ( io->ioKind != DOCokDRAWING_SHAPE )
	{
	if  ( io->ioTwipsWide < 1 || io->ioTwipsHigh < 1 )
	    { LLLDEB(io->ioKind,io->ioTwipsWide,io->ioTwipsHigh); return -1; }
	}

    switch( io->ioKind )
	{
	case DOCokPICTWMETAFILE:
	case DOCokPICTEMFBLIP:
	    if  ( ! io->ioPrivate					&&
		  tedOpenPixmap( &(io->ioPrivate), &(io->ioScreenPixmap),
			pip, io->ioKind, io->ioPixelsWide, io->ioPixelsHigh,
			lc, &(io->ioObjectData) )	)
		{ LDEB(1); return 0;	}

	    if  ( io->ioPrivate )
		{ io->ioPictureProperties.pipMetafileIsBitmap= 1;	}

	    return 0;

	case DOCokMACPICT:
	case DOCokPICTPNGBLIP:
	case DOCokPICTJPEGBLIP:
	    if  ( ! io->ioPrivate					&&
		  tedOpenPixmap( &(io->ioPrivate), &(io->ioScreenPixmap),
			pip, io->ioKind, io->ioPixelsWide, io->ioPixelsHigh,
			lc, &(io->ioObjectData) )	)
		{ LDEB(1); return 0;	}
	    return 0;

	case DOCokOLEOBJECT:
	    if  ( io->ioResultKind == DOCokPICTWMETAFILE	||
		  io->ioResultKind == DOCokPICTEMFBLIP		)
		{
		if  ( ! io->ioPrivate					&&
		      tedOpenPixmap( &(io->ioPrivate), &(io->ioScreenPixmap),
					pip, io->ioResultKind,
					io->ioPixelsWide, io->ioPixelsHigh,
					lc, &(io->ioResultData) )	)
		    { LDEB(1); return 0;	}

		if  ( io->ioPrivate )
		    { io->ioPictureProperties.pipMetafileIsBitmap= 1;	}

		return 0;
		}

	    if  ( io->ioResultKind == DOCokMACPICT	||
		  io->ioResultKind == DOCokPICTPNGBLIP	||
		  io->ioResultKind == DOCokPICTJPEGBLIP	)
		{
		if  ( ! io->ioPrivate					&&
		      tedOpenPixmap( &(io->ioPrivate), &(io->ioScreenPixmap),
					pip, io->ioResultKind,
					io->ioPixelsWide, io->ioPixelsHigh,
					lc, &(io->ioResultData) )	)
		    { LDEB(1); return 0;	}
		}

	    return 0;

	case DOCokEPS_FILE:
	    if  ( tedOpenEpsfObject( io, add, &io->ioResultData ) )
		{ LDEB(1); return 0;	}

	    return 0;

	case DOCokDRAWING_SHAPE:
	    {
	    DrawingShape *		ds= io->ioDrawingShape;
	    const ShapeProperties *	sp= &(ds->dsShapeProperties);

	    if  ( ! ds )
		{ XDEB(ds); return 0;	}

	    if  ( tedOpenShapePixmaps( ds, &(sp->spRect), lc ) )
		{ LDEB(1); return 0;	}
	    }
	    return 0;

	default:
	    LDEB(io->ioKind); return 0;
	}
    }

static int tedOpenParaObjects(	BufferItem *		bi,
				const LayoutContext *	lc )
    {
    int			part;
    TextParticule *	tp;

    tp= bi->biParaParticules;
    for ( part= 0; part < bi->biParaParticuleCount; tp++, part++ )
	{
	InsertedObject *	io;

	if  ( tp->tpKind != DOCkindOBJECT )
	    { continue;	}

	io= docGetObject( lc->lcDocument, tp->tpObjectNumber );
	if  ( ! io )
	    { LPDEB(tp->tpObjectNumber,io); continue;	}

	if  ( tedOpenObject( io, lc )	)
	    { LDEB(part); continue;	}
	}

    return 0;
    }

int tedOpenExternalItemObjects(	DocumentTree *		ei,
				const LayoutContext *	lc )
    {
    if  ( ei->eiRoot				&&
	  tedOpenItemObjects( ei->eiRoot, lc )	)
	{ LDEB(1); return -1;	}

    return 0;
    }

int tedOpenItemObjects(	BufferItem *		bi,
			const LayoutContext *	lc )
    {
    int			i;
    BufferDocument *	bd= lc->lcDocument;

    switch( bi->biLevel )
	{
	case DOClevBODY:

	    if  ( tedOpenExternalItemObjects( &(bd->bdEiFtnsep), lc ) )
		{ LDEB(1); return -1;	}
	    if  ( tedOpenExternalItemObjects( &(bd->bdEiFtnsepc), lc ) )
		{ LDEB(1); return -1;	}
	    if  ( tedOpenExternalItemObjects( &(bd->bdEiFtncn), lc ) )
		{ LDEB(1); return -1;	}

	    if  ( tedOpenExternalItemObjects( &(bd->bdEiAftnsep), lc ) )
		{ LDEB(1); return -1;	}
	    if  ( tedOpenExternalItemObjects( &(bd->bdEiAftnsepc), lc ) )
		{ LDEB(1); return -1;	}
	    if  ( tedOpenExternalItemObjects( &(bd->bdEiAftncn), lc ) )
		{ LDEB(1); return -1;	}

	    break;

	case DOClevSECT:

	    if  ( tedOpenExternalItemObjects( &(bi->biSectFirstPageHeader), lc ) )
		{ LDEB(1); return -1;	}
	    if  ( tedOpenExternalItemObjects( &(bi->biSectLeftPageHeader), lc ) )
		{ LDEB(1); return -1;	}
	    if  ( tedOpenExternalItemObjects( &(bi->biSectRightPageHeader), lc ) )
		{ LDEB(1); return -1;	}

	    if  ( tedOpenExternalItemObjects( &(bi->biSectFirstPageFooter), lc ) )
		{ LDEB(1); return -1;	}
	    if  ( tedOpenExternalItemObjects( &(bi->biSectLeftPageFooter), lc ) )
		{ LDEB(1); return -1;	}
	    if  ( tedOpenExternalItemObjects( &(bi->biSectRightPageFooter), lc ) )
		{ LDEB(1); return -1;	}

	    break;

	case DOClevROW:
	case DOClevCELL:
	    break;

	case DOClevPARA:
	    if  ( tedOpenParaObjects( bi, lc ) )
		{ LDEB(0); return -1;	}
	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    for ( i= 0; i < bi->biChildCount; i++ )
	{
	if  ( tedOpenItemObjects( bi->biChildren[i], lc ) )
	    { LDEB(i); return -1;	}
	}

    return 0;
    }

static void tedCloseDrawingShape(	DrawingShape *		ds )
    {
    int		child;

    for ( child= 0; child < ds->dsChildCount; child++ )
	{
	DrawingShape *	dsChild= ds->dsChildren[child];

	tedCloseDrawingShape( dsChild );
	}

    if  ( ds->dsPrivate )
	{
	AppBitmapImage *	abi= (AppBitmapImage *)ds->dsPrivate;

	if  ( ds->dsScreenPixmap )
	    {
	    guiFreeScreenPixmap( ds->dsScreenPixmap );
	    ds->dsScreenPixmap= (ScreenPixmap)0;
	    }

	if  ( abi )
	    { bmCleanBitmapImage( abi ); free( abi );	}

	ds->dsPrivate= (void *)0;
	}

    return;
    }

void docScreenCloseObject(	const BufferDocument *	bd,
				const TextParticule *	tp )
    {
    InsertedObject *	io;

    if  ( tp->tpKind != DOCkindOBJECT )
	{ return;	}

    io= docGetObject( bd, tp->tpObjectNumber );
    if  ( ! io )
	{ LPDEB(tp->tpObjectNumber,io); return;	}

    switch( io->ioKind )
	{
	case DOCokPICTWMETAFILE:
	case DOCokPICTEMFBLIP:
	case DOCokPICTPNGBLIP:
	case DOCokPICTJPEGBLIP:
	case DOCokOLEOBJECT:
	case DOCokMACPICT:
	    if  ( io->ioPrivate )
		{
		AppBitmapImage *	abi= (AppBitmapImage *)io->ioPrivate;

		if  ( abi )
		    { bmCleanBitmapImage( abi ); free( abi );	}

		io->ioPrivate= (void *)0;
		}

	    if  ( io->ioScreenPixmap )
		{
		guiFreeScreenPixmap( io->ioScreenPixmap );
		io->ioScreenPixmap= (ScreenPixmap)0;
		}

	    break;

	case DOCokEPS_FILE:
	    if  ( io->ioScreenPixmap )
		{
		guiFreeScreenPixmap( io->ioScreenPixmap );
		io->ioScreenPixmap= (ScreenPixmap)0;
		}
	    break;

	case DOCokDRAWING_SHAPE:
	    tedCloseDrawingShape( io->ioDrawingShape );
	    break;

	default:
	    LDEB(io->ioKind); break;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Called to resize an inserted object.				*/
/*									*/
/************************************************************************/

int tedReopenObject(	TextParticule *		tp,
			const LayoutContext *	lc )
    {
    InsertedObject *		io;

    io= docGetObject( lc->lcDocument, tp->tpObjectNumber );
    if  ( ! io )
	{ LPDEB(tp->tpObjectNumber,io); return -1;	}

    if  ( lc->lcCloseObject )
	{ (*lc->lcCloseObject)( lc->lcDocument, tp );	}

    if  ( tedOpenObject( io, lc ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

static void tedScaleObjectToParagraph(
				const BufferDocument *		bd,
				const BufferItem *		bodySectBi,
				BufferItem *			paraBi,
				double				xfac,
				InsertedObject *		io )
    {
    ParagraphFrame		pf;

    PictureProperties *		pip= &(io->ioPictureProperties);

    BlockFrame			bf;
    int				changed= 0;

    int				pageHigh;

    docBlockFrameTwips( &bf, paraBi, bodySectBi, bd,
				    paraBi->biTopPosition.lpPage,
				    paraBi->biTopPosition.lpColumn );

    docParagraphFrameTwips( &pf, &bf, paraBi );

    pageHigh= bf.bfPageGeometry.dgPageHighTwips- 
			    bf.bfPageGeometry.dgTopMarginTwips-
			    bf.bfPageGeometry.dgBottomMarginTwips;

    docLayoutScaleObjectToFitParagraphFrame( &changed, io, pageHigh, &pf );

    pip->pipScaleXUsed= io->ioScaleXUsed;
    pip->pipScaleYUsed= io->ioScaleYUsed;

    io->ioPixelsWide= COORDtoGRID( xfac,
				( io->ioScaleXUsed* pip->pipTwipsWide )/ 100 );
    io->ioPixelsHigh= COORDtoGRID( xfac,
				( io->ioScaleYUsed* pip->pipTwipsHigh )/ 100 );
    if  ( io->ioPixelsWide < 1 )
	{ io->ioPixelsWide=  1;	}
    if  ( io->ioPixelsHigh < 1 )
	{ io->ioPixelsHigh=  1;	}

    pip->pip_xWinExt= (int) ( ( 100000.0* pip->pipTwipsWide )/
						    ( 20* POINTS_PER_M ) );
    pip->pip_yWinExt= (int) ( ( 100000.0* pip->pipTwipsHigh )/
						    ( 20* POINTS_PER_M ) );

    return;
    }

void tedScaleObjectToSelectedParagraph(	EditDocument *		ed,
					InsertedObject *	io )
    {
    AppDrawingData *		add= &(ed->edDocumentWidget.dwDrawingData);
    double			xfac= add->addMagnifiedPixelsPerTwip;

    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    DocumentSelection		ds;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    const BufferItem *		bodySectBi= (const BufferItem *)0;

    if  ( tedGetSelection( &ds, &sg, &sd,
				    (DocumentTree **)0, &bodySectBi, td ) )
	{ LDEB(1); return;	}

    tedScaleObjectToParagraph( bd, bodySectBi, ds.dsHead.dpBi, xfac, io );

    return;
    }

/************************************************************************/
/*									*/
/*  Save the picture of an object for Copy/Paste.			*/
/*									*/
/************************************************************************/

int tedSaveObjectPicture(	AppBitmapImage *	abiTo,
				InsertedObject *	io )
    {
    AppBitmapImage *	abiFrom= (AppBitmapImage *)0;
    unsigned char *	buffer;

    if  ( ( io->ioKind == DOCokPICTWMETAFILE	||
	    io->ioKind == DOCokPICTEMFBLIP	||
	    io->ioKind == DOCokMACPICT		||
	    io->ioKind == DOCokPICTPNGBLIP	||
	    io->ioKind == DOCokPICTJPEGBLIP	)	&&
	  io->ioPrivate					)
	{ abiFrom= (AppBitmapImage *)io->ioPrivate; }

    if  ( io->ioKind == DOCokOLEOBJECT				&&
	  ( io->ioResultKind == DOCokPICTWMETAFILE	||
	    io->ioResultKind == DOCokPICTEMFBLIP	)	&&
	  io->ioPrivate						)
	{ abiFrom= (AppBitmapImage *)io->ioPrivate; }

    if  ( ! abiFrom )
	{ LXXDEB(io->ioKind,io->ioPrivate,abiFrom); return -1;	}

    buffer= (unsigned char *)malloc( abiFrom->abiBitmap.bdBufferLength );
    if  ( ! buffer )
	{ LXDEB(abiFrom->abiBitmap.bdBufferLength,buffer); return -1; }

    if  ( bmCopyDescription( &(abiTo->abiBitmap), &(abiFrom->abiBitmap) ) )
	{ LDEB(1); free( buffer ); return -1;	}

    memcpy( buffer, abiFrom->abiBuffer, abiFrom->abiBitmap.bdBufferLength );
    abiTo->abiBuffer= buffer;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Used from the 'Paste' for images.					*/
/*									*/
/*  9)  Not needed as the document is marked as changed anyway.		*/
/*									*/
/************************************************************************/

InsertedObject * tedObjectMakeBitmapObject(
					int *			pObjectNumber,
					EditDocument *		ed,
					BufferItem *		bi,
					const LayoutContext *	lc,
					AppBitmapImage *	abi )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    InsertedObject *		io;
    PictureProperties *		pip;

    MemoryBuffer		mb;
    SimpleOutputStream *	sosMem;
    SimpleOutputStream *	sosHex;

    int				objectNumber;
    int				includedAsBitmap= 0;

    const char *		typeId="";

    io= docClaimObject( &objectNumber, bd );
    if  ( ! io )
	{ XDEB(io); return (InsertedObject *)0;	}
    pip= &(io->ioPictureProperties);

    utilInitMemoryBuffer( &mb );

    sosMem= sioOutMemoryOpen( &mb );
    if  ( ! sosMem )
	{ XDEB(sosMem); return (InsertedObject *)0;	}

    sosHex= sioOutHexOpen( sosMem );
    if  ( ! sosHex )
	{ XDEB(sosHex); return (InsertedObject *)0;	}

    if  ( abi->abiFormat >= 0 )
	{ typeId= bmFileFormats[abi->abiFormat].bffFileType->bftTypeId; }

    if  ( abi->abiFormat >= 0			&&
	  ! includedAsBitmap			&&
	  ! strcmp( typeId, "pngFile" )		)
	{
	if  ( bmPngWritePng( &(abi->abiBitmap), abi->abiBuffer, sosHex ) )
	    { LDEB(1); return (InsertedObject *)0;	}

	io->ioKind= DOCokPICTPNGBLIP;
	pip->pipType= DOCokPICTPNGBLIP;
	includedAsBitmap= 1;
	}

    if  ( abi->abiFormat >= 0				&&
	  ! includedAsBitmap				&&
	  ( ! strcmp( typeId, "jpgFile" )	||
	    ! strcmp( typeId, "jpegFile" )	)	)
	{
	if  ( bmJpegWriteJfif( &abi->abiBitmap, abi->abiBuffer, sosHex ) )
	    { LDEB(1); return (InsertedObject *)0;	}

	io->ioKind= DOCokPICTJPEGBLIP;
	pip->pipType= DOCokPICTJPEGBLIP;
	includedAsBitmap= 1;
	}

    if  ( ! includedAsBitmap )
	{
	if  ( bmWmfWriteWmf( &(abi->abiBitmap), abi->abiBuffer, sosHex ) )
	    { LDEB(1); return (InsertedObject *)0;	}

	pip->pipMetafileIsBitmap= 1;
	pip->pipMetafileBitmapBpp= abi->abiBitmap.bdBitsPerPixel;

	io->ioKind= DOCokPICTWMETAFILE;
	pip->pipType= DOCokPICTWMETAFILE;
	pip->pipMapMode= 8;
	}

    sioOutClose( sosHex );
    sioOutClose( sosMem );

    io->ioObjectData= mb;

    bmImageSizeTwips( &(pip->pipTwipsWide), &(pip->pipTwipsHigh),
							&(abi->abiBitmap) );
    io->ioTwipsWide= pip->pipTwipsWide;
    io->ioTwipsHigh= pip->pipTwipsHigh;

    io->ioPrivate= abi;

    tedScaleObjectToSelectedParagraph( ed, io );

    io->ioScreenPixmap= guiMakePixmapForImageAndWidget( lc->lcScreenWidget,
				    abi, (const DocumentRectangle *)0,
				    io->ioPixelsWide, io->ioPixelsHigh );
    if  ( ! io->ioScreenPixmap )
	{ PDEB(io->ioScreenPixmap); return (InsertedObject *)0; }

    /*  9
    io->ioBliptag= appGetTimestamp();
    */

    *pObjectNumber= objectNumber;
    return io;
    }

/************************************************************************/
/*									*/
/*  Adapt (scale) the current image.					*/
/*									*/
/************************************************************************/

void tedObjectSetImageProperties(
				PropertyMask *			pipDoneMask,
				EditDocument *			ed,
				const BufferItem *		bodySectBi,
				InsertedObject *		io,
				const DocumentPosition *	dpObject,
				int				partObject,
				const PositionGeometry *	pgObject,
				const PropertyMask *		pipSetMask,
				const PictureProperties *	pipFrom )
    {
    AppDrawingData *		add= &(ed->edDocumentWidget.dwDrawingData);
    double			xfac= add->addMagnifiedPixelsPerTwip;

    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    PictureProperties *		pipTo= &(io->ioPictureProperties);

    if  ( docUpdPictureProperties( pipDoneMask, pipTo, pipSetMask, pipFrom ) )
	{ LDEB(1); return;	}

    if  ( utilPropMaskIsEmpty( pipDoneMask ) )
	{ return;	}

    if  ( PROPmaskISSET( pipDoneMask, PIPpropPICSCALE_X ) )
	{
	io->ioScaleXSet= io->ioScaleXUsed=
			    pipTo->pipScaleXUsed= pipTo->pipScaleXSet;
	}
    if  ( PROPmaskISSET( pipDoneMask, PIPpropPICSCALE_Y ) )
	{
	io->ioScaleYSet= io->ioScaleYUsed=
			    pipTo->pipScaleYUsed= pipTo->pipScaleYSet;
	}

    docObjectAdaptToPictureGeometry( io, pipTo );

    tedScaleObjectToParagraph( bd, bodySectBi, dpObject->dpBi, xfac, io );

    if  ( tedEditReopenObject( ed, partObject, dpObject, pgObject ) )
	{ LDEB(1);	}

    return;
    }

static void tedDocSetImageProperties(
				EditDocument *			ed,
				const PropertyMask *		pipSetMask,
				const PictureProperties *	pipFrom )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    InsertedObject *		io;
    DocumentPosition		dpObject;
    int				part;

    PropertyMask		pipDoneMask;
    const BufferItem *		bodySectBi;

    utilPropMaskClear( &pipDoneMask );

    if  ( tedGetObjectSelection( td, &bodySectBi, &part, &dpObject, &io ) )
	{ LDEB(1); return;	}

    tedObjectSetImageProperties( &pipDoneMask, ed, bodySectBi,
		    io, &dpObject, part,
		    &(td->tdSelectionGeometry.sgBegin), pipSetMask, pipFrom );

    if  ( utilPropMaskIsEmpty( &pipDoneMask ) )
	{ return;	}

    tedAdaptToolsToSelection( ed );

    appDocumentChanged( ed, 1 );

    return;
    }

void tedAppSetImageProperties(	EditApplication *		ea,
				const PropertyMask *		pipSetMask,
				const PictureProperties *	pip )
    {
    EditDocument *	ed= ea->eaCurrentDocument;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    tedDocSetImageProperties( ed, pipSetMask, pip );

    return;
    }

/************************************************************************/
/*									*/
/*  Draw the eight blocks around a selected object.			*/
/*									*/
/************************************************************************/

void tedGetObjectRectangle(	DocumentRectangle *		drObject,
				APP_POINT *			xp,
				const InsertedObject *		io,
				const PositionGeometry *	pg,
				const LayoutContext *		lc,
				const TedDocument *		td )
    {
    docGetPixelRectForPos( drObject, lc,
			    pg->pgXTwips, pg->pgXTwips+ io->ioTwipsWide,
			    &(pg->pgTopPosition), &(pg->pgBasePosition) );

    /* Do not trust pg->pgXTwips yet */
    drObject->drX0= pg->pgXPixels;
    drObject->drX1= drObject->drX0+ io->ioPixelsWide- 1;

    switch( td->tdObjectResizeCorner )
	{
	case -1:
	    break;

	case RESIZE_BOTTOM_LEFT:
	case RESIZE_MIDDLE_LEFT:
	case RESIZE_TOP_LEFT:
	    drObject->drX0 += td->tdObjectCornerMovedX;
	    break;

	case RESIZE_BOTTOM_RIGHT:
	case RESIZE_MIDDLE_RIGHT:
	case RESIZE_TOP_RIGHT:
	    drObject->drX1 += td->tdObjectCornerMovedX;
	    break;

	case RESIZE_BOTTOM_MIDDLE:
	case RESIZE_TOP_MIDDLE:
	    break;

	default:
	    LDEB(td->tdObjectResizeCorner);
	    break;
	}

    if  ( drObject->drX1 <= drObject->drX0 )
	{ drObject->drX1= drObject->drX0+ 1;	}

    switch( td->tdObjectResizeCorner )
	{
	case -1:
	    break;

	case RESIZE_TOP_LEFT:
	case RESIZE_TOP_MIDDLE:
	case RESIZE_TOP_RIGHT:
	    drObject->drY0 += td->tdObjectCornerMovedY;
	    break;

	case RESIZE_BOTTOM_LEFT:
	case RESIZE_BOTTOM_MIDDLE:
	case RESIZE_BOTTOM_RIGHT:
	    drObject->drY1 += td->tdObjectCornerMovedY;
	    break;

	case RESIZE_MIDDLE_LEFT:
	case RESIZE_MIDDLE_RIGHT:
	    break;

	default:
	    LDEB(td->tdObjectResizeCorner);
	    break;
	}

    if  ( drObject->drY1 <= drObject->drY0 )
	{ drObject->drY1= drObject->drY0+ 1;	}

    if  ( xp )
	{
	int		xl, xm, xr;
	int		yb, ym, yt;

	xl= drObject->drX0;
	xm= ( drObject->drX0+ drObject->drX1 )/ 2- RESIZE_BLOCK/2;
	xr= drObject->drX1- RESIZE_BLOCK;

	yt= drObject->drY0;
	ym= ( drObject->drY0+ drObject->drY1 )/ 2- RESIZE_BLOCK/2;
	yb= drObject->drY1- RESIZE_BLOCK;

	xp[RESIZE_BOTTOM_LEFT].x= xl;
	xp[RESIZE_BOTTOM_LEFT].y= yb;

	xp[RESIZE_BOTTOM_MIDDLE].x= xm;
	xp[RESIZE_BOTTOM_MIDDLE].y= yb;

	xp[RESIZE_BOTTOM_RIGHT].x= xr;
	xp[RESIZE_BOTTOM_RIGHT].y= yb;

	xp[RESIZE_MIDDLE_LEFT].x= xl;
	xp[RESIZE_MIDDLE_LEFT].y= ym;

	xp[RESIZE_MIDDLE_RIGHT].x= xr;
	xp[RESIZE_MIDDLE_RIGHT].y= ym;

	xp[RESIZE_TOP_LEFT].x= xl;
	xp[RESIZE_TOP_LEFT].y= yt;

	xp[RESIZE_TOP_MIDDLE].x= xm;
	xp[RESIZE_TOP_MIDDLE].y= yt;

	xp[RESIZE_TOP_RIGHT].x= xr;
	xp[RESIZE_TOP_RIGHT].y= yt;
	}

    return;
    }

