/************************************************************************/
/*									*/
/*  Ted: Handle user input.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	"docScreenLayout.h"
#   include	"tedApp.h"
#   include	"tedRuler.h"
#   include	"tedLayout.h"

#   include	"appGuiKeys.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Just log events that pass by for debugging purposes.		*/
/*									*/
/*  NOTE the silly constuction to do away with the 'unused' compiler	*/
/*	 warning.							*/
/*									*/
/************************************************************************/

#   ifdef USE_MOTIF_AND_NEEDED

static void tedLogEvent(	Widget		w,
				void *		voided,
				XEvent *	event,
				Boolean *	pRefused )
    {
    EditDocument *		ed= (EditDocument *)voided;

    appDebug( "EVENT \"%s\": %s\n",
			ed->edTitle, APP_X11EventNames[event->type] );

    *pRefused= 1;

    if  ( ! event )
	{ return;	}
    if  ( ! event )
	{ tedLogEvent( w, voided, event, pRefused );	}
    }

#   endif

/************************************************************************/
/*									*/
/*  Handle mouse clicks.						*/
/*									*/
/************************************************************************/

typedef struct DraggingContext
    {
    int				dcMouseX;
    int				dcMouseY;
    DocumentPosition		dcAnchorPosition;
    BufferItem *		dcSelRootItem;
    BufferItem *		dcBodyItem;
    BufferItem *		dcBodySectBi;
    EditDocument *		dcEd;
    DocumentTree *		dcTree;
    LayoutContext		dcLayoutContext;
    } DraggingContext;

/************************************************************************/
/*									*/
/*  Find the document position under the mouse and extend the selection	*/
/*  to that position.							*/
/*									*/
/************************************************************************/

static int tedExtendSelectionToXY( DraggingContext *		dc )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    EditDocument *		ed= dc->dcEd;

    DocumentPosition		dpTo;
    PositionGeometry		pgTo;

    int				ox= ed->edVisibleRect.drX0;
    int				oy= ed->edVisibleRect.drY0;

    int				pageFound;
    int				columnFound;

    dpTo= dc->dcAnchorPosition;

    if  ( tedFindPosition( &dpTo, &pgTo, &pageFound, &columnFound,
			    dc->dcSelRootItem, dc->dcBodyItem,
			    lc, dc->dcMouseX+ ox, dc->dcMouseY+ oy ) )
	{ /*LLDEB(mouseX,mouseY);*/ return 0; }

    if  ( tedExtendSelectionToPosition( ed, &(dc->dcAnchorPosition), &dpTo ) )
	{ return -1;	}

    if  ( dpTo.dpBi && dpTo.dpBi->biInExternalItem == DOCinBODY )
	{
	BufferItem *	bi= docGetSectItem( dpTo.dpBi );

	if  ( ! bi )
	    { XDEB(bi);			}
	else{ dc->dcBodyItem= bi;	}
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Supposing te click is in a diffrent document tree, reformat that	*/
/*  tree for the context of the click to determine where the location	*/
/*  of the click in the document hierarchy.				*/
/*									*/
/************************************************************************/

static int tedLayoutSelectedRoot(	const LayoutContext *	lc,
					DocumentTree *		eiSet,
					BufferItem *		rootBiSet,
					BufferItem *		bodySectBiSet,
					int			page,
					int			column )
    {
    DocumentPosition		dp;
    int				inHeadFoot;
    DocumentRectangle		drChanged;

    docInitDocumentPosition( &dp );

    inHeadFoot= rootBiSet->biInExternalItem;

    switch( inHeadFoot )
	{
	case DOCinBODY:
	case DOCinFOOTNOTE:
	case DOCinENDNOTE:
	    break;

	case DOCinFIRST_HEADER:
	case DOCinLEFT_HEADER:
	case DOCinRIGHT_HEADER:

	case DOCinFIRST_FOOTER:
	case DOCinLEFT_FOOTER:
	case DOCinRIGHT_FOOTER:

	    if  ( ! eiSet )
		{ XDEB(eiSet); return 1;	}
	    if  ( eiSet->eiPageFormattedFor != page )
		{
		const int	column0= 0;

		if  ( docLayoutExternalItem( eiSet, &drChanged,
			    page, column0, eiSet->eiY0UsedTwips,
			    bodySectBiSet, lc,
			    docInitScreenLayoutExternalItem ) )
		    { LDEB(page); return 1; }

		eiSet->eiPageSelectedUpon= page;
		eiSet->eiColumnSelectedIn= 0;
		}
	    break;

	case DOCinFTNSEP:
	case DOCinFTNSEPC:
	case DOCinFTNCN:
	case DOCinAFTNSEP:
	case DOCinAFTNSEPC:
	case DOCinAFTNCN:

	    if  ( ! eiSet )
		{ XDEB(eiSet); return 1;	}

	    {
	    int			changed= 0;
	    DocumentRectangle	drExternalSetX;
	    BufferItem *	bodySectBiSetX;

	    eiSet->eiPageSelectedUpon= page;
	    eiSet->eiColumnSelectedIn= column;

	    if  ( docCheckPageOfSelectedExtItem( &changed, &bodySectBiSetX,
				&drExternalSetX, eiSet, lc,
				docInitScreenLayoutExternalItem ) )
		{ LDEB(1);	}

	    if  ( changed )
		{ geoUnionRectangle( &drChanged, &drChanged, &drExternalSetX ); }
	    }
	    break;

	default:
	    LDEB(rootBiSet->biInExternalItem);
	}

    if  ( eiSet )
	{
	eiSet->eiPageSelectedUpon= page;
	eiSet->eiColumnSelectedIn= column;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Handle a multi-click.						*/
/*									*/
/************************************************************************/

static void tedHandleMultiClick(EditDocument *			ed,
				const DocumentSelection *	dsWord,
				const DocumentPosition *	dpClick,
				const PositionGeometry *	pgClick,
				int				seq )
    {
    DocumentSelection	dsNew;
    int			scrolledX= 0;
    int			scrolledY= 0;

    docInitDocumentSelection( &dsNew );

    docSetIBarSelection( &dsNew, dpClick );

    if  ( seq > 2 )
	{
	int		partLineHead;
	int		partLineTail;

	docLineSelection( &dsNew, &partLineHead, &partLineTail,
					    dpClick->dpBi, pgClick->pgLine );
	}
    else{ dsNew= *dsWord;	}

    tedSetSelection( ed, &dsNew, pgClick->pgAtLineHead,
						    &scrolledX, &scrolledY );

    }

/************************************************************************/
/*									*/
/*  Select the object that we just clicked.				*/
/*									*/
/************************************************************************/

static int tedSelectObject(	EditDocument *			ed,
				const BufferItem *		bodySectBi,
				const LayoutContext *		lc,
				const DocumentSelection *	dsWord,
				int				docX )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    int				partObject;
    DocumentPosition		dpObject;
    InsertedObject *		io;

    if  ( ! docGetObjectSelection( dsWord, bd, &partObject, &dpObject, &io ) )
	{
	DocumentRectangle	drObject;
	PositionGeometry	pgObject;
	const int		lastOne= 1;
	int			marg= 0;
	int			wide;

	tedPositionGeometry( &pgObject, &dpObject, bodySectBi, lastOne, lc );

	tedGetObjectRectangle( &drObject, (APP_POINT *)0,
						    io, &pgObject, lc, td );

	wide= drObject.drX1- drObject.drX0+ 1;
	if  ( wide > 10 )
	    { marg++;	}
	if  ( wide > 20 )
	    { marg++;	}

	if  ( docX >= drObject.drX0+ marg	&&
	      docX <= drObject.drX1- marg	)
	    {
	    int			scrolledX= 0;
	    int			scrolledY= 0;

	    tedSetSelection( ed, dsWord, pgObject.pgAtLineHead,
						    &scrolledX, &scrolledY );
	    return 0;
	    }
	}

    return 1;
    }

/************************************************************************/
/*									*/
/*  Locate a mouse click.						*/
/*									*/
/************************************************************************/

static int tedFindMousePosition( APP_WIDGET			w,
				EditDocument *			ed,
				DraggingContext *		dc,
				const DocumentSelection *	dsOld,
				int *				pSameRoot,
				int *				pSameInstance,
				PositionGeometry *		pgClick,
				int *				pDocX,
				int *				pSeq,
				unsigned int *			pKeyState,
				APP_EVENT *			downEvent )
    {
    AppDrawingData *		add= &(ed->edDocumentWidget.dwDrawingData);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    int				ox= ed->edVisibleRect.drX0;
    int				oy= ed->edVisibleRect.drY0;

    int				mouseX;
    int				mouseY;
    int				docX;
    int				docY;

    int				sameRoot= 1;
    int				sameInstance= 1;

    int				button;
    int				upDown;
    int				seq;
    unsigned int		keyState= 0;

    int				page;
    int				column;
    BufferItem *		rootBiSet;

    if  ( appGetCoordinatesFromMouseButtonEvent( &mouseX, &mouseY,
					    &button, &upDown, &seq, &keyState,
					    w, downEvent ) )
	{ LDEB(1); return 1;	}

    if  ( mouseX < -ox )
	{ mouseX=  -ox;	}
    if  ( mouseY < -oy )
	{ mouseY=  -oy;	}

    dc->dcEd= ed;
    tedSetLayoutContext( &(dc->dcLayoutContext), ed );
    dc->dcMouseX= mouseX;
    dc->dcMouseY= mouseY;

    docX= mouseX+ ox;
    docY= mouseY+ oy;

    dc->dcSelRootItem= bd->bdBody.eiRoot;
    dc->dcBodyItem= bd->bdBody.eiRoot;
    dc->dcTree= &(bd->bdBody);

    if  ( tedFindRootForPosition( &(dc->dcTree), &rootBiSet,
					    &(dc->dcBodySectBi),
					    &page, &column, ed, docX, docY ) )
	{ /*LLDEB(docX,docY);*/ return 1;	}

    dc->dcSelRootItem= rootBiSet;
    dc->dcBodyItem= dc->dcBodySectBi;

    sameRoot= docSelectionSameRoot( dsOld, rootBiSet );

    if  ( sameRoot )
	{ sameInstance= docSelectionSameInstance( rootBiSet, page, column ); }

    if  ( ! sameRoot || ! sameInstance )
	{
	if  ( tedLayoutSelectedRoot( &(dc->dcLayoutContext),
		    dc->dcTree, rootBiSet, dc->dcBodySectBi, page, column ) )
	    { LDEB(page); return 1;	}

	appExposeDrawingData( add );
	}

    if  ( tedFindPosition( &(dc->dcAnchorPosition), pgClick, &page, &column,
					dc->dcSelRootItem, dc->dcBodyItem,
					&(dc->dcLayoutContext), docX, docY ) )
	{ LLDEB(docX,docY); return 1;	}

    *pSameRoot= sameRoot;
    *pSameInstance= sameInstance;
    *pDocX= docX;
    *pSeq= seq;
    *pKeyState= keyState;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Avoid readonly fields.						*/
/*									*/
/************************************************************************/

static void tedAvoidReadonlyField(	DraggingContext *	dc,
					PositionGeometry *	pgClick,
					int			docX )
    {
    DocumentField *		dfHead;
    DocumentField *		dfTail;
    int				headMoved= 0;
    int				tailMoved= 0;
    int				headPart= -1;
    int				tailPart= -1;

    DocumentPosition		dpHead= dc->dcAnchorPosition;
    DocumentPosition		dpTail= dc->dcAnchorPosition;

    docBalanceFieldSelection( &dfHead, &dfTail, &headMoved, &tailMoved,
					    &headPart, &tailPart,
					    &dpHead, &dpTail, dc->dcTree,
					    dc->dcLayoutContext.lcDocument );

    if  ( headMoved )
	{
	PositionGeometry	pg;

	tedPositionGeometry( &pg, &dpHead,
				dc->dcBodySectBi, 1, &(dc->dcLayoutContext) );

	if  ( docX <= pg.pgXPixels )
	    {
	    dc->dcAnchorPosition= dpHead;
	    *pgClick= pg;
	    return;
	    }
	}

    if  ( tailMoved )
	{
	PositionGeometry	pg;

	tedPositionGeometry( &pg, &dpTail,
				dc->dcBodySectBi, 0, &(dc->dcLayoutContext) );

	if  ( docX >= pg.pgXPixels )
	    {
	    dc->dcAnchorPosition= dpTail;
	    *pgClick= pg;
	    return;
	    }
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Select the current position of the mouse.				*/
/*									*/
/************************************************************************/

static int tedSelectMousePosition(
				unsigned int *			pKeyState,
				APP_WIDGET			w,
				EditDocument *			ed,
				DraggingContext *		dc,
				APP_EVENT *			downEvent )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    int				docX;

    int				sameRoot= 1;
    int				sameInstance= 1;

    int				scrolledX= 0;
    int				scrolledY= 0;

    int				seq;
    unsigned int		keyState= 0;

    DocumentSelection		dsOld;
    SelectionGeometry		sg;
    SelectionDescription	sd;
    const BufferItem *		bodySectBi;

    PositionGeometry		pgClick;
    DocumentSelection		dsWord;
    int				wordIsObject= 0;

    if  ( tedGetSelection( &dsOld, &sg, &sd,
				(DocumentTree **)0, &bodySectBi, td ) )
	{ LDEB(1); return 1;	}

    if  ( tedFindMousePosition( w, ed, dc, &dsOld, &sameRoot, &sameInstance,
				&pgClick, &docX, &seq, &keyState, downEvent ) )
	{ return 1;	}

    tedAvoidReadonlyField( dc, &pgClick, docX );

    docWordSelection( &dsWord, &wordIsObject, &(dc->dcAnchorPosition) );

    if  ( seq > 1 )
	{
	tedHandleMultiClick( ed, &dsWord,
				    &(dc->dcAnchorPosition), &pgClick, seq );
	goto ready;
	}

    if  ( sameRoot && sameInstance && ( keyState & KEY_SHIFT_MASK ) )
	{
	dc->dcAnchorPosition= dsOld.dsAnchor;

	if  ( tedExtendSelectionToXY( dc ) )
	    { LDEB(1); return 1;	}

	goto ready;
	}

    docAvoidParaHeadField( &(dc->dcAnchorPosition), (int *)0, bd );

    if  ( wordIsObject							&&
	  ! tedSelectObject( ed, bodySectBi, &(dc->dcLayoutContext),
						      &dsWord, docX )	)
	{ goto ready;	}

    if  ( ! docIsIBarSelection( &dsOld )				||
	  ! docSamePosition( &(dsOld.dsHead), &(dc->dcAnchorPosition) )	)
	{
	tedSetSelectedPosition( ed, &(dc->dcAnchorPosition),
						    pgClick.pgAtLineHead,
						    &scrolledX, &scrolledY );
	}

  ready:
    tedAdaptToolsToSelection( ed );

    *pKeyState= keyState;
    return 0;
    }

#   define	TED_DRAG_INTERVAL	(150L)

static APP_EVENT_HANDLER_H( tedInputDragMouseMove, w, vdc, event )
    {
    DraggingContext *	dc= (DraggingContext *)vdc;
    EditDocument *	ed= dc->dcEd;

    int			ox= ed->edVisibleRect.drX0;
    int			oy= ed->edVisibleRect.drY0;

    int			mouseX;
    int			mouseY;

    if  ( appGetCoordinatesFromMouseMoveEvent( &mouseX, &mouseY, w, event ) )
	{ return;	}

    if  ( mouseX < -ox )
	{ mouseX=  -ox;	}
    if  ( mouseY < -oy )
	{ mouseY=  -oy;	}

    dc->dcMouseX= mouseX;
    dc->dcMouseY= mouseY;

    if  ( tedExtendSelectionToXY( dc ) )
	{ LDEB(1); return;	}

    return;
    }

static APP_EVENT_HANDLER_H( tedInputDragMouseUp, w, vdc, event )
    {
    DraggingContext *		dc= (DraggingContext *)vdc;
    EditDocument *		ed= dc->dcEd;

    tedDescribeSelection( ed );
    tedAdaptToolsToSelection( ed );

    return;
    }

static APP_TIMER_HANDLER( tedTick, voiddc )
    {
    DraggingContext *		dc= (DraggingContext *)voiddc;
    EditDocument *		ed= dc->dcEd;
    AppDrawingData *		add= &(ed->edDocumentWidget.dwDrawingData);

    int				ox= ed->edVisibleRect.drX0;
    int				oy= ed->edVisibleRect.drY0;

    int				mouseX;
    int				mouseY;

    int				scrolledX= 0;
    int				scrolledY= 0;

#   ifdef USE_MOTIF
    {
    Window		root;
    Window		child;
    int			rootX;
    int			rootY;
    int			winX;
    int			winY;
    unsigned int	mask;

    XQueryPointer( add->addDisplay, add->addDrawable,
						    &root, &child,
						    &rootX, &rootY,
						    &winX, &winY, &mask );
    mouseX= winX; mouseY= winY;
    }
#   endif

#   ifdef USE_GTK
    {
    gint		winX;
    gint		winY;
    GdkModifierType	mask;

    gdk_window_get_pointer( add->addDrawable, &winX, &winY, &mask );

    mouseX= winX; mouseY= winY;
    }
#   endif

#   ifdef USE_QT
    mouseX= mouseY= 0;
#   endif

    if  ( mouseX < -ox )
	{ mouseX=  -ox;	}
    if  ( mouseY < -oy )
	{ mouseY=  -oy;	}
    if  ( mouseX > add->addBackRect.drX1- ox )
	{ mouseX=  add->addBackRect.drX1- ox;	}
    if  ( mouseY > add->addBackRect.drY1- oy )
	{ mouseY=  add->addBackRect.drY1- oy;	}

    appScrollToRectangle( ed, mouseX+ ox, mouseY+ oy,
					mouseX+ ox, mouseY+ oy,
					&scrolledX, &scrolledY );

    if  ( scrolledX || scrolledY )
	{
	if  ( tedExtendSelectionToXY( dc ) )
	    { LDEB(1);	}
	}

#   ifdef USE_MOTIF
    return;
#   endif
#   ifdef USE_GTK
    return 0;
#   endif
    }

static void tedButton1Pressed(	APP_WIDGET			w,
				EditDocument *			ed,
				APP_EVENT *			downEvent )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    DraggingContext		dc;
    unsigned int		keyState;

    int				startedWithObject;

    if  ( tedSelectMousePosition( &keyState, w, ed, &dc, downEvent ) )
	{ return;	}

    startedWithObject= td->tdSelectionDescription.sdIsObjectSelection;

    if  ( ed->edFileReadOnly || ( keyState & KEY_CONTROL_MASK ) )
	{
	DocumentField *		dfHyperlink;

	dfHyperlink= docFindTypedFieldForPosition( bd,
				    &(dc.dcAnchorPosition), DOCfkHYPERLINK );

	if  ( dfHyperlink )
	    {
	    const char *	fileName= (const char *)0;
	    int			fileSize= 0;
	    const char *	markName= (const char *)0;
	    int			markSize= 0;

	    if  ( ! docFieldGetHyperlink( dfHyperlink,
				&fileName, &fileSize, &markName, &markSize ) )
		{
		tedDocFollowLink( (APP_WIDGET)0, ed,
				    fileName, fileSize, markName, markSize );
		return;
		}
	    }
	}

    appRunDragLoop( w, ed->edApplication, downEvent,
				tedInputDragMouseUp, tedInputDragMouseMove,
				TED_DRAG_INTERVAL, tedTick, (void *)&dc );

    if  ( startedWithObject					&&
	  ! td->tdSelectionDescription.sdIsObjectSelection	)
	{ tedHideObjectWindows( ed );	}

    return;
    }

/************************************************************************/
/*									*/
/*  Handle mouse button down events for the document widget.		*/
/*									*/
/************************************************************************/

static void tedButton3Pressed(	APP_WIDGET			w,
				EditDocument *			ed,
				APP_EVENT *			downEvent )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    int				ox= ed->edVisibleRect.drX0;
    int				oy= ed->edVisibleRect.drY0;

    int				mouseX;
    int				mouseY;
    int				docX;
    int				docY;

    int				otherRoot= 0;

    int				button;
    int				upDown;
    int				seq;
    unsigned int		keyState= 0;

    DraggingContext		dc;

    int				selectMousePosition= 0;

    DocumentSelection		dsOld;
    SelectionGeometry		sg;
    SelectionDescription	sd;

    int				page;
    int				column;
    DocumentTree *		eiSet= (DocumentTree *)0;
    BufferItem *		bodySectBiSet;
    BufferItem *		rootBiSet;

    LayoutContext		lc;

    layoutInitContext( &lc );
    tedSetLayoutContext( &lc, ed );

    if  ( tedGetSelection( &dsOld, &sg, &sd,
			    (DocumentTree **)0, (const BufferItem **)0, td ) )
	{ LDEB(1); return;	}

    if  ( appGetCoordinatesFromMouseButtonEvent( &mouseX, &mouseY,
					    &button, &upDown, &seq, &keyState,
					    w, downEvent ) )
	{ LDEB(1); return;	}

    if  ( mouseX < -ox )
	{ mouseX=  -ox;	}
    if  ( mouseY < -oy )
	{ mouseY=  -oy;	}

    docX= mouseX+ ox;
    docY= mouseY+ oy;

    if  ( tedFindRootForPosition( &eiSet, &rootBiSet, &bodySectBiSet,
					    &page, &column, ed, docX, docY ) )
	{ LLDEB(docX,docY); return;	}

    otherRoot= ! docSelectionSameRoot( &dsOld, rootBiSet );

    if  ( otherRoot )
	{ selectMousePosition= 1;	}
    else{
	DocumentPosition	dpClick;
	PositionGeometry	pgClick;

	int			cmp1;
	int			cmp2;

	if  ( tedFindPosition( &dpClick, &pgClick, &page, &column,
						rootBiSet, bodySectBiSet,
						&lc, docX, docY ) )
	    { LLDEB(docX,docY); return;	}

	cmp1= docComparePositions( &dpClick, &(dsOld.dsHead) );
	cmp2= docComparePositions( &dpClick, &(dsOld.dsTail) );

	if  ( cmp1 < 0 || cmp2 > 0 )
	    { selectMousePosition= 1;	}
	}

    if  ( selectMousePosition )
	{
	if  ( tedSelectMousePosition( &keyState, w, ed, &dc, downEvent ) )
	    { return;	}
	}

    tedShowFormatTool( td->tdToolsFormatToolOption, ed->edApplication );

    tedAdaptFormatToolToDocument( ed, 1 );

    return;
    }

APP_EVENT_HANDLER_H( tedMouseButtonPressed, w, voided, downEvent )
    {
    EditDocument *	ed= (EditDocument *)voided;
    TedDocument *	td= (TedDocument *)ed->edPrivateData;
    int			button= MOUSE_BUTTON_FROM_EVENT( downEvent );

    tedStopCursorBlink( ed );

    switch( button )
	{
	case MOUSE_BUTTON_1:
#	    ifdef USE_MOTIF
	    if  ( ! ed->edFileReadOnly		&&
		  downEvent->xbutton.subwindow	)
		{
		int	res= tedObjectDrag( w, ed, downEvent );

		if  ( res < 1 )
		    { return;	}
		}
#	    endif

#	    ifdef USE_GTK
	    if  ( ! ed->edFileReadOnly )
		{
		int	res= tedObjectDrag( w, ed, downEvent );

		if  ( res < 1 )
		    { return;	}
		}
#	    endif

	    tedButton1Pressed( w, ed, downEvent );
	    break;

	case MOUSE_BUTTON_2:
	    appDocAskForPaste( ed, "PRIMARY" );
	    break;

	case MOUSE_BUTTON_3:
	    tedButton3Pressed( w, ed, downEvent );
	    break;

#	ifdef USE_MOTIF
	case Button4:
	case Button5:
	    break;
#	endif

#	ifdef USE_GTK
#	if GTK_MAJOR_VERSION < 2

	case SCROLL_UP:
	    appMouseWheelUp( ed );
	    break;

	case SCROLL_DOWN:
	    appMouseWheelDown( ed );
	    break;

#	endif
#	endif

	default:
	    LDEB(button);
	    break;
	}

    if  ( tedHasIBarSelection( td ) )
	{ tedStartCursorBlink( ed );	}

    return;
    }

