/************************************************************************/
/*									*/
/*  Geometry calculations when looking for a position in the document.	*/
/*									*/
/*  Find the object in the document for an x,y position.		*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<limits.h>
#   include	<math.h>

#   include	"tedApp.h"
#   include	"tedLayout.h"
#   include	<docParaString.h>
#   include	"docScreenLayout.h"
#   include	"docPixels.h"
#   include	"docDraw.h"
#   include	<docParaString.h>

#   include	<appDebugon.h>

/************************************************************************/

typedef struct PositionFindJob
    {
    DocumentPosition		pfjDocumentPosition;
    PositionGeometry		pfjPositionGeometry;
    int				pfjPage;
    int				pfjColumn;

    int				pfjFound;
    int				pfjDistance;

    int				pfjDocX;
    int				pfjDocY;
    } PositionFindJob;

/************************************************************************/
/*									*/
/*  Probe a line to find out whether the position is inside the line.	*/
/*  If it is, remember the documentposition.				*/
/*									*/
/*  0)  Avoid shifted table headers.					*/
/*  1)  Determine rectangle (in pixels) for the line.			*/
/*	As pfjDocX,Y have been corrected for scrolling, scrolling is	*/
/*	irrelevant here. The shifts are for table headers &c that are	*/
/*	repeated on a different page.					*/
/*  2)  x, y inside rectangle?						*/
/*  3)  Above line, if near.. select head of line.			*/
/*									*/
/************************************************************************/

static void tedSetPositionFound(	PositionFindJob *	pfj,
					const TextLine *	tl,
					int			distance )
    {
    pfj->pfjPage= tl->tlTopPosition.lpPage;
    pfj->pfjColumn= tl->tlTopPosition.lpColumn;

    pfj->pfjFound= 1;
    pfj->pfjDistance= distance;

    return;
    }

static int tedFindPositionInTextLine(
				BufferItem *			paraBi,
				int				line,
				const ParagraphFrame *		pf,
				const DocumentRectangle *	drLinePixels,
				void *				vpfj,
				DrawingContext *		dc,
				const BlockOrigin *		bo )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;
    PositionFindJob *		pfj= (PositionFindJob *)vpfj;

    const TextLine *		tl= paraBi->biParaLines+ line;

    int				x0FrameShifted;

    x0FrameShifted= X_PIXELS( add, pf->pfParaContentRect.drX0+ bo->boXShift );

    /*  0  */
    if  ( bo->boOverrideFrame )
	{ return tl->tlParticuleCount;	}

    /*  2  */
    if  ( geo2DIXYInBox( pfj->pfjDocX, pfj->pfjDocY, drLinePixels ) )
	{
	const int	d= 0;

	if  ( tedFindPositionInLine( &(pfj->pfjDocumentPosition),
		    &(pfj->pfjPositionGeometry),
		    lc, paraBi, line, dc->dcBodySectBi, pfj->pfjDocX ) )
	    { LDEB(pfj->pfjDocX);			}
	else{ tedSetPositionFound( pfj, tl, d );	}

	return tl->tlParticuleCount;
	}

    if  ( paraBi->biInExternalItem == DOCinBODY )
	{
	int			yc= ( drLinePixels->drY0+ drLinePixels->drY1 )/ 2;
	int			dx;
	int			dy;
	int			d;

	DocumentRectangle	drTry;

	if  ( pfj->pfjDocY > yc )
	    { dy= pfj->pfjDocY- drLinePixels->drY0;	}
	else{ dy= drLinePixels->drY1- pfj->pfjDocY;	}

	/*  left */
	drTry.drX0= 2* drLinePixels->drX0- 1* drLinePixels->drX1;
	drTry.drX1= drLinePixels->drX0;
	drTry.drY0= 2* drLinePixels->drY0- 1* drLinePixels->drY1;
	drTry.drY1= 2* drLinePixels->drY1- 1* drLinePixels->drY0;

	dx= drLinePixels->drX0- pfj->pfjDocX;
	d= (int)sqrt( (double)(dx*dx)+ (double)(dy*dy) );

	if  ( d < pfj->pfjDistance					&&
	      geo2DIXYInBox( pfj->pfjDocX, pfj->pfjDocY, &drTry ) 	)
	    {
	    const int	lastOne= 1;

	    pfj->pfjDocumentPosition.dpBi= (BufferItem *)paraBi;
	    pfj->pfjDocumentPosition.dpStroff= tl->tlStroff;

	    tedPositionGeometry( &(pfj->pfjPositionGeometry),
					    &(pfj->pfjDocumentPosition),
					    dc->dcBodySectBi, lastOne, lc );

	    tedSetPositionFound( pfj, tl, d );
	    return tl->tlParticuleCount;
	    }

	/*  right */
	drTry.drX0= drLinePixels->drX1;
	drTry.drX1= 2* drLinePixels->drX1- drLinePixels->drX0;
	drTry.drY0= 2* drLinePixels->drY0- 1* drLinePixels->drY1;
	drTry.drY1= 2* drLinePixels->drY1- 1* drLinePixels->drY0;

	dx= pfj->pfjDocX- drLinePixels->drX1;
	d= (int)sqrt( (double)(dx*dx)+ (double)(dy*dy) );

	if  ( d < pfj->pfjDistance					&&
	      geo2DIXYInBox( pfj->pfjDocX, pfj->pfjDocY, &drTry ) 	)
	    {
	    const int	lastOne= 0;

	    pfj->pfjDocumentPosition.dpBi= (BufferItem *)paraBi;
	    pfj->pfjDocumentPosition.dpStroff= tl->tlStroff+ tl->tlStrlen;

	    tedPositionGeometry( &(pfj->pfjPositionGeometry),
					    &(pfj->pfjDocumentPosition),
					    dc->dcBodySectBi, lastOne, lc );

	    tedSetPositionFound( pfj, tl, d );
	    return tl->tlParticuleCount;
	    }

	/*  above, below */
	drTry.drX0= drLinePixels->drX0;
	drTry.drX1= drLinePixels->drX1;
	drTry.drY0= 2* drLinePixels->drY0- 1* drLinePixels->drY1;
	drTry.drY1= 2* drLinePixels->drY1- 1* drLinePixels->drY0;

	d= dy;

	if  ( d < pfj->pfjDistance					&&
	      geo2DIXYInBox( pfj->pfjDocX, pfj->pfjDocY, &drTry ) 	)
	    {
	    if  ( tedFindPositionInLine( &(pfj->pfjDocumentPosition),
			&(pfj->pfjPositionGeometry),
			lc, paraBi, line, dc->dcBodySectBi, pfj->pfjDocX ) )
		{ LDEB(pfj->pfjDocX);			}
	    else{ tedSetPositionFound( pfj, tl, d );	}

	    return tl->tlParticuleCount;
	    }
	}

    return tl->tlParticuleCount;
    }

/************************************************************************/
/*									*/
/*  Find the document position corresponding to the x coordinate in	*/
/*  pixels in a line of text.						*/
/*  Initially, we try a binary search for a string offset that results	*/
/*  in an X position <= docXPixels. But instead of the final comparison	*/
/*  we compare the X to the center of the character.			*/
/*									*/
/*  NOTE: As docXPixels has been corrected for the scrollbars,		*/
/*	scrolling is irrelevant.					*/
/*									*/
/************************************************************************/

int tedFindPositionInLine(	DocumentPosition *	dp,
				PositionGeometry *	pg,
				const LayoutContext *	lc,
				const BufferItem *	paraBi,
				int			line,
				const BufferItem *	bodySectBi,
				int			docXPixels )
    {
    const TextLine *		tl= paraBi->biParaLines+ line;
    int				upto= docParaStrlen( paraBi );

    int				l= tl->tlStroff;
    int				r= tl->tlStroff+ tl->tlStrlen;
    int				m= (l+ r)/ 2;

    const int			lastOne= 0;
    DocumentPosition		dpM;
    DocumentPosition		dpR;
    PositionGeometry		pgM;
    PositionGeometry		pgR;

    docInitPositionGeometry( &pgM );
    docInitPositionGeometry( &pgR );

    dpM.dpBi= (BufferItem *)paraBi; /* remove const */
    dpM.dpStroff= l;

    while( l != m )
	{
	dpM.dpStroff= m= docParaFixStroff( paraBi, m );
	if  ( l == m )
	    { break;	}

	tedPositionGeometry( &pgM, &dpM, bodySectBi, lastOne, lc );

	if  ( pgM.pgXPixels < docXPixels )
	    { l= m;	}
	else{ r= m;	}

	m= (l+ r)/ 2;
	}

    dpM.dpStroff= docParaFixStroff( paraBi, m );
    tedPositionGeometry( &pgM, &dpM, bodySectBi, lastOne, lc );
    dpR= dpM;
    pgR= pgM;

    if  ( dpR.dpStroff < upto )
	{
	dpR.dpStroff= docParaNextStroff( paraBi, dpR.dpStroff );
	if  ( dpR.dpStroff < 0 )
	    { LDEB(dpR.dpStroff); 	}
	else{
	    tedPositionGeometry( &pgR, &dpR, bodySectBi, lastOne, lc );

	    if  ( docXPixels > ( pgM.pgXPixels+ pgR.pgXPixels )/ 2 )
		{
		*dp= dpR;
		*pg= pgR;
		return 0;
		}
	    }
	}

    *dp= dpM;
    *pg= pgM;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Translate a Window coordinate to a position in a text buffer.	*/
/*									*/
/************************************************************************/

int tedFindPosition(	DocumentPosition *		dp,
			PositionGeometry *		pg,
			int *				pPage,
			int *				pColumn,
			BufferItem *			selRootBi,
			const BufferItem *		bodySectBi,
			const LayoutContext *		lc,
			int				docXPixels,
			int				docY )
    {
    AppDrawingData *	add= lc->lcAdd;
    int			page= docY/ add->addPageStepPixels;
    DocumentRectangle	drPage;

    PositionFindJob	pfj;
    DrawingContext	dc;

    LayoutPosition	lpBelow;

    docInitLayoutPosition( &lpBelow );

    drPage.drX0= 0;
    drPage.drX1= INT_MAX; /* irrelevant */
    drPage.drY0= page* add->addPageStepPixels;
    drPage.drY1= drPage.drY0+ add->addPageStepPixels- 1;

    docInitDrawingContext( &dc );

    dc.dcLayoutContext= *lc;
    dc.dcDrawTableGrid= 0;
    dc.dcClipRect= &drPage;

    dc.dcFirstPage= page;
    dc.dcLastPage= page;

    docInitDocumentPosition( &(pfj.pfjDocumentPosition) );
    docInitPositionGeometry( &(pfj.pfjPositionGeometry) );
    pfj.pfjPage= -1;
    pfj.pfjColumn= -1;
    pfj.pfjFound= 0;
    pfj.pfjDistance= INT_MAX;

    pfj.pfjDocX= docXPixels;
    pfj.pfjDocY= docY;

    dc.dcDrawTextLine= tedFindPositionInTextLine;
    dc.dcDrawExternalItems= 0;
    dc.dcPostponeHeadersFooters= 0;

    dc.dcCurrentColorSet= 0;
    dc.dcCurrentTextAttributeSet= 0;

    if  ( selRootBi->biInExternalItem != DOCinBODY	||
	  selRootBi->biLevel != DOClevBODY		)
	{
	if  ( bodySectBi && bodySectBi->biLevel == DOClevSECT )
	    { dc.dcBodySectBi= bodySectBi;				}
	else{ XLDEB(bodySectBi,bodySectBi?bodySectBi->biLevel:1);	}
	}

    if  ( docDrawItem( &lpBelow, selRootBi, (void *)&pfj, &dc ) )
	{ LDEB(1); return 1;	}

    if  ( ! pfj.pfjFound )
	{ return 1;	}

    *dp= pfj.pfjDocumentPosition;
    *pg= pfj.pfjPositionGeometry;
    *pPage= pfj.pfjPage;
    *pColumn= pfj.pfjColumn;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find the root item corresponding to a mouse click.			*/
/*									*/
/*  The essence of tedFindRootForPosition() is to intercept clicks in	*/
/*  items that are used on different pages: Before the regular find	*/
/*  code is called, the item must be reformatted for a particular	*/
/*  page (headers,footers) or a particular column (footnote separators)	*/
/*  So we have to know what item is clicked on what page in order to	*/
/*  format it for that particular page and then find the exact position	*/
/*  of the click.							*/
/*									*/
/*  1)  Calculate the page number from the coordinates.			*/
/*  2)  Find the first section corresponding to this page.		*/
/*	For finding headers and footers this is sufficient. For finding	*/
/*	the actual body section, it is not: Below a more precise	*/
/*	attempt is made. Also note that though this is the section that	*/
/*	determines the layout of headers and footers, it does not	*/
/*	determine the division of the page in columns.			*/
/*  3)  If no section can be found, the click might have been in a note	*/
/*	protruding beyond the the last page with regular text.		*/
/*  4)  Is the click in the header corresponding to this section?	*/
/*  5)  Is the click in the footer corresponding to this section?	*/
/*  6)  Is the click in a note on this page?				*/
/*  7)  Is the click in the footnote separator on this page.		*/
/*  8)  Is the click in the endnote separator on this page.		*/
/*									*/
/*  9)  Not in an exceptional place.. assume it is in the body.		*/
/*									*/
/************************************************************************/

/************************************************************************/
/*									*/
/*  Find an item for a particular position. Use the regular algoritm.	*/
/*									*/
/************************************************************************/

static int tedFindSectForPosition(	BufferItem **		pSelSectBi,
					int *			pPage,
					int *			pColumn,
					BufferItem *		selRootBi,
					BufferItem *		bodySectBi,
					const LayoutContext *	lc,
					int			docXPixels,
					int			docY )
    {
    DocumentPosition	dp;
    PositionGeometry	pg;
    BufferItem *	sectBi;

    int			pageFound;
    int			columnFound;

    if  ( tedFindPosition( &dp, &pg, &pageFound, &columnFound,
			    selRootBi, bodySectBi, lc, docXPixels, docY ) )
	{ return 1;	}

    sectBi= docGetSectItem( dp.dpBi );
    if  ( ! sectBi )
	{ XDEB(sectBi); return 1;	}

    *pSelSectBi= sectBi;
    *pPage= pageFound;
    *pColumn= columnFound;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Find an item for a particular position.				*/
/*									*/
/************************************************************************/

int tedFindRootForPosition(		DocumentTree **		pEi,
					BufferItem **		pSelSectBi,
					BufferItem **		pBodySectBi,
					int *			pPage,
					int *			pColumn,
					EditDocument *		ed,
					int			docXPixels,
					int			docY )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    AppDrawingData *		add= &(ed->edDocumentWidget.dwDrawingData);

    int				page;
    int				sectNr;

    int				pageY;

    BufferItem *		bodySectBi= (BufferItem *)0;

    int				inHeaderFooter;
    DocumentTree *		ei;
    DocumentNote *		dn;

    const int			justUsed= 0;
    DocumentRectangle		drExtern;

    DocumentField *		dfNote;
    int				notePage= -1;
    int				noteColumn= -1;

    BufferItem *		bodyBi= bd->bdBody.eiRoot;

    LayoutContext		lc;

    layoutInitContext( &lc );
    tedSetLayoutContext( &lc, ed );

    /*  1  */
    page= docY/ add->addPageStepPixels;
    pageY= docY- ( page* add->addPageStepPixels );

    /*  2  */
    for ( sectNr= 0; sectNr < bodyBi->biChildCount; sectNr++ )
	{
	bodySectBi= bodyBi->biChildren[sectNr];

	if  ( bodySectBi->biTopPosition.lpPage <= page		&&
	      bodySectBi->biBelowPosition.lpPage >= page	)
	    { break;	}
	}

    if  ( sectNr < bodyBi->biChildCount )
	{
	const DocumentGeometry *	dgSect;
	int				isEmpty;

	dgSect= &(bodySectBi->biSectDocumentGeometry);

	/*  4  */
	ei= (DocumentTree *)0;
	inHeaderFooter= docWhatPageHeader( &ei, &isEmpty,
						    bodySectBi, page, bd );
	if  ( ei && ei->eiRoot )
	    {
	    const int		headerColumn= 0;

	    if  ( docGetExternalItemBox( &drExtern, bodySectBi, ei, justUsed,
						page, headerColumn, &lc ) )
		{ LDEB(1);	}
	    else{
		if  ( docY >= drExtern.drY0 && docY <= drExtern.drY1 )
		    {
		    *pEi= ei;
		    *pSelSectBi= ei->eiRoot;
		    *pBodySectBi= bodySectBi;
		    *pPage= page;
		    *pColumn= headerColumn;

		    return 0;
		    }
		}
	    }

	/*  5  */
	ei= (DocumentTree *)0;
	inHeaderFooter= docWhatPageFooter( &ei, &isEmpty,
						    bodySectBi, page, bd );
	if  ( ei && ei->eiRoot )
	    {
	    const int		footerColumn= 0;

	    if  ( docGetExternalItemBox( &drExtern, bodySectBi, ei, justUsed,
						page, footerColumn, &lc ) )
		{ LDEB(1);	}
	    else{
		if  ( docY >= drExtern.drY0 && docY <= drExtern.drY1 )
		    {
		    *pEi= ei;
		    *pSelSectBi= ei->eiRoot;
		    *pBodySectBi= bodySectBi;
		    *pPage= page;
		    *pColumn= footerColumn;

		    return 0;
		    }
		}
	    }
	}

    /*  6  */
    for ( dfNote= docGetFirstNoteOfDocument( &dn, bd, -1 );
	  dfNote;
	  dfNote= docGetNextNoteInDocument( &dn, bd, dfNote, -1 ) )
	{
	DocumentTree *		eiBody;
	int			andSeparator= 0;

	ei= &(dn->dnDocumentTree);

	if  ( ! ei->eiRoot )
	    { XDEB(ei->eiRoot); continue;	}

	if  ( docGetRootOfSelectionScope( &eiBody, &bodySectBi, bd,
						&(dfNote->dfSelectionScope) ) )
	    { LDEB(1); return -1;	}

	if  ( notePage != ei->eiRoot->biBelowPosition.lpPage		||
	      noteColumn != ei->eiRoot->biBelowPosition.lpColumn	)
	    {
	    notePage= ei->eiRoot->biBelowPosition.lpPage;
	    noteColumn= ei->eiRoot->biBelowPosition.lpColumn;
	    andSeparator= 1;
	    }

	if  ( ei->eiRoot->biBelowPosition.lpPage < page )
	    { continue;	}
	if  ( ei->eiRoot->biTopPosition.lpPage > page )
	    { continue;	}

	/*  7,8  */
	if  ( andSeparator )
	    {
	    int			sepItKind= DOCinFTNSEP;
	    int			ret;
	    DocumentTree *	eiNoteSep;
	    int			y0Twips;

	    if  ( dn->dnNoteProperties.npExternalItemKind == DOCinENDNOTE )
		{ sepItKind= DOCinAFTNSEP;	}

	    ret= docNoteSeparatorRectangle( &drExtern, &eiNoteSep, &y0Twips,
					    bodySectBi, dn, sepItKind, &lc );

	    if  ( ret < 0 )
		{ LDEB(ret); return -1;	}
	    if  ( ret == 0 && geo2DIXYInBox( docXPixels, docY, &drExtern ) )
		{
		*pEi= eiNoteSep;
		*pSelSectBi= eiNoteSep->eiRoot;
		*pBodySectBi= bodySectBi;
		*pPage= notePage;
		*pColumn= noteColumn;
		return 0;
		}
	    }

	if  ( ! tedFindSectForPosition( pSelSectBi, pPage, pColumn,
					    ei->eiRoot, bodySectBi,
					    &lc, docXPixels, docY ) )
	    {
	    *pEi= ei;
	    *pBodySectBi= bodySectBi;
	    return 0;
	    }
	}

    /*  9  */
    if  ( ! tedFindSectForPosition( pSelSectBi, pPage, pColumn,
				    bodyBi, (BufferItem *)0,
				    &lc, docXPixels, docY ) )
	{
	if  ( (*pSelSectBi)->biInExternalItem != DOCinBODY )
	    { LDEB((*pSelSectBi)->biInExternalItem); return 1;	}

	*pEi= &(bd->bdBody);
	*pBodySectBi= *pSelSectBi;
	return 0;
	}

    return 1;
    }
