/************************************************************************/
/*									*/
/*  Editor functionality.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	"docScreenLayout.h"
#   include	"docPixels.h"
#   include	"docEvalField.h"
#   include	"tedEdit.h"
#   include	"tedLayout.h"

#   include	<appDebugon.h>

#   define	VALIDATE_TREE		0

/************************************************************************/
/*									*/
/*  Redo document layout after editing.					*/
/*									*/
/************************************************************************/

static int tedEditRefreshLayout(	TedEditOperation *	teo )
    {
    const LayoutContext  *	lc= &(teo->teoLayoutContext);
    EditOperation *		eo= &(teo->teoEo);
    EditRange *			er= &(eo->eoReformatRange);

    EditDocument *		ed= teo->teoEditDocument;
    AppDrawingData *		add= &(ed->edDocumentWidget.dwDrawingData);

    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    RecalculateFields		rf;

    DocumentTree *		ei;
    BufferItem *		bodyBi= bd->bdBody.eiRoot;
    BufferItem *		bodySectBi;

    const SelectionScope *	ss= &(eo->eoSelectionScope);

    if  ( docGetRootOfSelectionScope( &ei, &bodySectBi, bd, ss ) )
	{ LDEB(1); return -1;	}

    if  ( docEditFixParaBreakKind( eo, ei, bd, er->erHead.epParaNr ) )
	{ LDEB(er->erHead.epParaNr);	}
    if  ( docEditFixParaBreakKind( eo, ei, bd, er->erTail.epParaNr ) )
	{ LDEB(er->erTail.epParaNr);	}

    docInitRecalculateFields( &rf );

    rf.rfBd= bd;
    rf.rfEi= ei;
    rf.rfCloseObject= lc->lcCloseObject;
    rf.rfUpdateFlags= eo->eoFieldUpdate;
    rf.rfFieldsUpdated= 0;

    if  ( eo->eoBulletsChanged > 0 )
	{ rf.rfUpdateFlags |= FIELDdoLISTTEXT; }

    if  ( eo->eoNotesDeleted > 0	||
	  eo->eoNotesAdded > 0		||
	  eo->eoSectionsDeleted > 0	||
	  eo->eoSectionsAdded > 0	)
	{
	int			changed= 0;

	docRenumberNotes( &changed, bd );

	if  ( changed )
	    { rf.rfUpdateFlags |= FIELDdoCHFTN;	}
	}

    if  ( rf.rfUpdateFlags )
	{
	rf.rfSelBegin= eo->eoSelectedRange.erHead;
	rf.rfSelEnd= eo->eoSelectedRange.erTail;

	if  ( docRecalculateTextLevelFields( &rf, bodyBi ) )
	    { LDEB(1);	}

	eo->eoSelectedRange.erHead= rf.rfSelBegin;
	eo->eoSelectedRange.erTail= rf.rfSelEnd;
	}

    if  ( rf.rfFieldsUpdated > 0 )
	{
	DocumentSelection	dsLayout;

	if  ( docLayoutInvalidateRange( &dsLayout, ei, er ) )
	    { LDEB(1); return -1;	}

	if  ( tedLayoutDocumentTree( td, &(teo->teoLayoutContext) ) )
	    { LDEB(1); return -1;	}

	if  ( tedOpenItemObjects( bodyBi, lc ) )
	    { LDEB(1); return -1;	}

	tedIncludeRectangleInChange( teo, &(add->addBackRect) );
	}
    else{
	if  ( eo->eoNotesDeleted > 0	||
	      eo->eoNotesAdded > 0	||
	      eo->eoSectionsDeleted > 0	||
	      eo->eoSectionsAdded > 0	)
	    {
	    if  ( docScreenLayoutItem( bodyBi, lc, &(teo->teoChangedRect) ) )
		{ LDEB(eo->eoNotesDeleted); return -1;	}

	    if  ( tedOpenItemObjects( bodyBi, lc ) )
		{ LDEB(1); return -1;	}
	    }
	else{
	    if  ( eo->eoParaAdjustParagraphNumber >= 0 )
		{
		int		paraNr= eo->eoParaAdjustParagraphNumber;
		BufferItem *	paraBi;

		if  ( er->erHead.epParaNr != paraNr )
		    { LLDEB(er->erHead.epParaNr,paraNr); }

		if  ( er->erTail.epParaNr != paraNr )
		    { LLDEB(er->erTail.epParaNr,paraNr); }

		paraBi= docGetParagraphByNumber( ei, paraNr );

		if  ( tedAdjustParagraphLayout( teo, paraBi ) )
		    { LDEB(eo->eoParaAdjustParagraphNumber); }

		if  ( tedOpenItemObjects( paraBi, lc ) )
		    { LDEB(1); return -1;	}
		}
	    else{
		BufferItem *		selRootBi;
		DocumentSelection	dsLayout;

		if  ( docLayoutInvalidateRange( &dsLayout, ei, er ) )
		    { LDEB(1); return -1;	}

		selRootBi= docGetSelectionRoot( &ei, &bodySectBi,
							    bd, &dsLayout );
		if  ( ! selRootBi )
		    { XDEB(selRootBi); return -1;	}

		while( selRootBi && selRootBi->biLevel > er->erBottomLevel )
		    { selRootBi= selRootBi->biParent;	}
		if  ( ! selRootBi )
		    { XLDEB(selRootBi,er->erBottomLevel); return -1;	}

		if  ( selRootBi->biInExternalItem != DOCinBODY	&&
		      ! selRootBi->biParent			)
		    {
		    int				page;
		    int				column;

		    if  ( selRootBi->biLevel != DOClevSECT )
			{
			SDEB(docLevelStr(selRootBi->biLevel));
			return -1;
			}

		    page= selRootBi->biSectExternalUseForPage;
		    column= selRootBi->biSectExternalUseForColumn;
		    ei->eiPageFormattedFor= -1;
		    ei->eiColumnFormattedFor= -1;

		    docLayoutExternalItem( ei, &(teo->teoChangedRect),
					    page, column, ei->eiY0UsedTwips,
					    bodySectBi, 
					    &(teo->teoLayoutContext),
					    docInitScreenLayoutExternalItem );

		    if  ( tedOpenExternalItemObjects( ei, lc ) )
			{ LDEB(1); return -1;	}
		    }
		else{
		    if  ( docScreenLayoutItem( selRootBi,
					    &(teo->teoLayoutContext),
					    &(teo->teoChangedRect) ) )
			{ LDEB(1);	}

		    if  ( tedOpenItemObjects( selRootBi, lc ) )
			{ LDEB(1); return -1;	}
		    }
		}
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Do the last steps that set an I-Bar selection after an editing	*/
/*  action.								*/
/*									*/
/*  1)  The order in which the size of the window is adapted and the	*/
/*	window is scrolled to the selection depends on the direction	*/
/*	of the change, as the scrollbars complain when the corrent	*/
/*	position is not in range.					*/
/*  2)  Do not place the IBar in a bullet or note number.		*/
/*									*/
/************************************************************************/

static int tedEditFinishIBarSelection2(
			    TedEditOperation *		teo,
			    BufferItem *		paraBi,
			    int				stroff )
    {
    EditDocument *		ed= teo->teoEditDocument;
    AppDrawingData *		add= &(ed->edDocumentWidget.dwDrawingData);
    BufferDocument *		bd= teo->teoEo.eoBd;

    int				scrolledX= 0;
    int				scrolledY= 0;

    const int			lastLine= 0;

    /*  1  */
    if  ( add->addBackRect.drY1 > teo->teoOldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    if  ( paraBi )
	{
	DocumentPosition	dpIbar;

	/*  2  */
	docSetDocumentPosition( &dpIbar, paraBi, stroff );
	docAvoidParaHeadField( &dpIbar, (int *)0, bd );

	tedSetIBarSelection( ed, &dpIbar, lastLine, &scrolledX, &scrolledY );
	}
    else{ XDEB(paraBi);	}


    /*  1  */
    if  ( add->addBackRect.drY1 < teo->teoOldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    appDocExposeRectangle( ed, &(teo->teoChangedRect), scrolledX, scrolledY );

    tedStartCursorBlink( ed );

    return 0;
    }

int tedEditFinishIBarSelection(	TedEditOperation *		teo,
				BufferItem *			bi,
				int				stroff )
    {
    if  ( tedEditRefreshLayout( teo ) )
	{ LDEB(1); /* return -1; */	}

    return tedEditFinishIBarSelection2( teo, bi, stroff );
    }

/************************************************************************/
/*									*/
/*  Adapt to a formatting change.					*/
/*									*/
/************************************************************************/

static int tedEditFinishSelection2(	TedEditOperation *		teo,
					const DocumentSelection *	dsNew )
    {
    EditDocument *		ed= teo->teoEditDocument;
    AppDrawingData *		add= &(ed->edDocumentWidget.dwDrawingData);

    int				scrolledX= 0;
    int				scrolledY= 0;

    int				lastLine= 0;

    if  ( docIsIBarSelection( dsNew ) )
	{
	return tedEditFinishIBarSelection2( teo,
			    dsNew->dsHead.dpBi, dsNew->dsHead.dpStroff );
	}

    if  ( add->addBackRect.drY1 > teo->teoOldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    tedSetSelection( ed, dsNew, lastLine, &scrolledX, &scrolledY );
    tedAdaptToolsToSelection( ed );

    if  ( add->addBackRect.drY1 < teo->teoOldBackY1 )
	{
	appDocSetScrollbarValues( ed );
	appSetShellConstraints( ed );
	}

    appDocExposeRectangle( ed, &(teo->teoChangedRect), scrolledX, scrolledY );

    return 0;
    }


int tedEditFinishSelection(	TedEditOperation *		teo,
				const DocumentSelection *	dsNew )
    {
    if  ( tedEditRefreshLayout( teo ) )
	{ LDEB(1); return -1;	}

    return tedEditFinishSelection2( teo, dsNew );
    }

int tedEditFinishOldSelection(		TedEditOperation *		teo )
    {
    EditOperation *		eo= &(teo->teoEo);
    const EditRange *		er= &(teo->teoEo.eoSelectedRange);
    int				rval;

    EditDocument *		ed= teo->teoEditDocument;
    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    DocumentSelection		dsNew;

    if  ( tedEditRefreshLayout( teo ) )
	{ LDEB(1); return -1;	}

    docInitDocumentSelection( &dsNew );

    if  ( docSelectionForEditPositionsInTree( &dsNew, eo->eoEi,
					&(er->erHead), &(er->erTail) ) )
	{ LDEB(1); return -1;	}

    rval= tedEditFinishSelection2( teo, &dsNew );

    td->tdCurrentTextAttribute= teo->teoSavedTextAttribute;
    td->tdCurrentTextAttributeNumber= teo->teoSavedTextAttributeNumber;

    return rval;
    }

int tedEditFinishOldSelectionEnd(	TedEditOperation *		teo )
    {
    int				rval;

    EditOperation *		eo= &(teo->teoEo);
    DocumentPosition		dpNew;

    docPositionForEditPosition( &dpNew,
				    &(eo->eoSelectedRange.erTail), eo->eoEi );

    rval= tedEditFinishIBarSelection( teo, dpNew.dpBi, dpNew.dpStroff );

    return rval;
    }

/************************************************************************/
/*									*/
/*  1)  Start an Edit Operation.					*/
/*  2)  Finish an Edit Operation.					*/
/*									*/
/*  The reference positions in the Start and Finish calls should be	*/
/*  the same position in the document from a logical point of view.	*/
/*									*/
/************************************************************************/

static void tedInitEditOperation(	TedEditOperation *	teo )
    {
    docInitEditOperation( &(teo->teoEo) );

    teo->teoEditDocument= (EditDocument *)0;
    layoutInitContext( &(teo->teoLayoutContext) );

    geoInitRectangle( &(teo->teoChangedRect) );
    teo->teoChangedRectSet= 0;
    teo->teoOldBackY1= 0;
    }

/*  1  */
int tedStartEditOperation(	TedEditOperation *	teo,
				SelectionGeometry *	sg,
				SelectionDescription *	sd,
				EditDocument *		ed,
				int			fullWidth )
    {
    EditOperation *		eo= &(teo->teoEo);
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;
    AppDrawingData *		add= &(ed->edDocumentWidget.dwDrawingData);

    DocumentSelection		ds;

#   if VALIDATE_TREE
    {
    const BufferDocument *	bd= td->tdDocument;
    LDEB(1);
    if  ( docCheckItem( &(bd->bdItem) ) )
	{ LDEB(2); docListItem( 0, &(bd->bdItem) ); abort(); }
    }
#   endif

    tedInitEditOperation( teo );

    if  ( tedGetSelection( &ds, sg, sd, &(eo->eoEi), &(eo->eoBodySectBi), td ) )
	{ LDEB(1); return -1;	}

    if  ( docStartEditOperation( eo, &ds, bd ) )
	{ LDEB(1); return -1;	}

    teo->teoSavedTextAttribute= td->tdCurrentTextAttribute;
    teo->teoSavedTextAttributeNumber= td->tdCurrentTextAttributeNumber;

    eo->eoIBarSelectionOld= sd->sdIsIBarSelection;
    eo->eoMultiParagraphSelectionOld= ! sd->sdIsSingleParagraph;

    tedIncludeRectangleInChange( teo, &(sg->sgRectangle) );
    teo->teoChangedRect.drX1= add->addBackRect.drX1;

    if  ( fullWidth )
	{ teo->teoChangedRect.drX0= add->addBackRect.drX0; }

    teo->teoOldBackY1= add->addBackRect.drY1;

    /**/
    teo->teoEditDocument= ed;

    tedSetLayoutContext( &(teo->teoLayoutContext), ed );
    eo->eoCloseObject= teo->teoLayoutContext.lcCloseObject;

    return 0;
    }

/************************************************************************/
/*									*/
/*  Widen redraw rectangle to whole BufferItems.			*/
/*									*/
/************************************************************************/

int tedEditIncludeItemInRedraw(	TedEditOperation *	teo,
				const BufferItem *	bi )
    {
    const LayoutContext  *	lc= &(teo->teoLayoutContext);
    AppDrawingData *		add= lc->lcAdd;
    DocumentRectangle		drLocal;

    drLocal.drX0= add->addBackRect.drX0;
    drLocal.drX1= add->addBackRect.drX1;
    drLocal.drY0= BI_TOP_PIXELS( add, bi )- 1;
    drLocal.drY1= BI_BELOW_PIXELS( add, bi )- 1;

    if  ( docIsRowItem( bi ) )
	{
	drLocal.drY1= LP_YPIXELS( add, &(bi->biRowBelowAllPosition) )- 1;
	}

    tedIncludeRectangleInChange( teo, &drLocal );

    return 0;
    }

int tedEditIncludeRowsInRedraw(	TedEditOperation *	teo,
				const BufferItem *	sectBi,
				int			row0,
				int			row1 )
    {
    const LayoutContext  *	lc= &(teo->teoLayoutContext);
    AppDrawingData *		add= lc->lcAdd;
    const BufferItem *		rowBi;
    DocumentRectangle		drLocal;

    drLocal.drX0= add->addBackRect.drX0;
    drLocal.drX1= add->addBackRect.drX1;

    rowBi= sectBi->biChildren[row0];
    drLocal.drY0= BI_TOP_PIXELS( add, rowBi )- 1;

    rowBi= sectBi->biChildren[row1];
    drLocal.drY1= LP_YPIXELS( add, &(rowBi->biRowBelowAllPosition) )- 1;

    tedIncludeRectangleInChange( teo, &drLocal );

    return 0;
    }

void tedIncludeRectangleInChange(	TedEditOperation *		teo,
					const DocumentRectangle *	dr )
    {
    if  ( teo->teoChangedRectSet )
	{
	geoUnionRectangle( &(teo->teoChangedRect), &(teo->teoChangedRect), dr );
	return;
	}
    else{
	teo->teoChangedRect= *dr;
	teo->teoChangedRectSet= 1;
	}
    }

