/************************************************************************/
/*									*/
/*  Ted: screen specific shape drawing.					*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<math.h>

#   include	"tedApp.h"
#   include	"docScreenLayout.h"
#   include	"tedDraw.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Draw a Word 97+ drawing shape.					*/
/*									*/
/************************************************************************/

static int tedShapeDrawArrowHead(	AppDrawingData *	add,
					int			xTo,
					int			yTo,
					int			xFrom,
					int			yFrom,
					int			widthPixels,
					const ShapeArrow *	sa )
    {
    double		xfac= add->addMagnifiedPixelsPerTwip;
    AffineTransform2D	at;
    AffineTransform2D	att;
    APP_POINT		points[5];

    int			length;
    int			length2;
    int			width;

    docShapeArrowSizesTwips( &length, &length2, &width, sa );

    length= X_PIXELS( add, length );
    length2= X_PIXELS( add, length2 );
    width= COORDtoGRID( xfac, width );

    geoRotationAffineTransform2DAtan( &at, yTo- yFrom, xTo- xFrom );
    geoTranslationAffineTransform2D( &att, xTo, yTo );
    geoAffineTransform2DProduct( &at, &att, &at );

    points[0].x= xTo;
    points[0].y= yTo;

    switch( sa->saArrowHead )
	{
	case DSarrowNONE:
	    return 0;
	case DSarrowARROW:
	    points[1].x= AT2_X( -length,  width, &at );
	    points[1].y= AT2_Y( -length,  width, &at );
	    points[2].x= AT2_X( -length, -width, &at );
	    points[2].y= AT2_Y( -length, -width, &at );
	    points[3]= points[0];
	    appDrawFillPolygon( add, points, 3 );
	    break;

	case DSarrowSTEALTH_ARROW:
	    points[1].x= AT2_X( -length2,  width, &at );
	    points[1].y= AT2_Y( -length2,  width, &at );
	    points[2].x= AT2_X( -length,  0, &at );
	    points[2].y= AT2_Y( -length,  0, &at );
	    points[3].x= AT2_X( -length2, -width, &at );
	    points[3].y= AT2_Y( -length2, -width, &at );
	    points[4]= points[0];
	    appDrawFillPolygon( add, points, 4 );
	    break;

	case DSarrowDIAMOND:
	    points[1].x= AT2_X( -length/ 2,  width, &at );
	    points[1].y= AT2_Y( -length/ 2,  width, &at );
	    points[2].x= AT2_X( -length,  0, &at );
	    points[2].y= AT2_Y( -length,  0, &at );
	    points[3].x= AT2_X( -length/ 2, -width, &at );
	    points[3].y= AT2_Y( -length/ 2, -width, &at );
	    points[4]= points[0];
	    appDrawFillPolygon( add, points, 4 );
	    break;

	case DSarrowOPEN_ARROW:
	    points[0].x= AT2_X( -length,  width, &at );
	    points[0].y= AT2_Y( -length,  width, &at );
	    points[1].x= AT2_X( 0, 0, &at );
	    points[1].y= AT2_Y( 0, 0, &at );
	    points[2].x= AT2_X( -length, -width, &at );
	    points[2].y= AT2_Y( -length, -width, &at );

	    appDrawSetLineAttributes( add,
		    widthPixels, LINEstyleSOLID, LINEcapBUTT, LINEjoinMITER,
		    (unsigned char *)0, 0 );

	    appDrawDrawLines( add, points, 3, 0 );
	    break;

	case DSarrowOVAL:
	    points[0].x= AT2_X( -length/ 2,  width, &at );
	    points[0].y= AT2_Y( -length/ 2,  width, &at );
	    points[1].x= AT2_X(  length/ 2, -width, &at );
	    points[1].y= AT2_Y(  length/ 2, -width, &at );
	    if  ( points[0].x > points[1].x )
		{
		int x= points[0].x;
		       points[0].x= points[1].x;
		                    points[1].x= x;
		}
	    if  ( points[0].y > points[1].y )
		{
		int y= points[0].y;
		       points[0].y= points[1].y;
		                    points[1].y= y;
		}
	    appDrawFillArc( add, points[0].x, points[0].y,
					    points[1].x- points[0].x+ 1,
					    points[1].y- points[0].y+ 1,
					    64* 0, 64* 360 );
	    break;

	case DSarrowCHEVRON_ARROW:
	case DSarrow2CHEVRON_ARROW:
	default:
	    LDEB(sa->saArrowHead); return 0;
	}

    return 0;
    }

static APP_POINT * tedDrawGetVertices(
				const DocumentRectangle *	drTwips,
				const DrawingShape *		ds,
				const LayoutContext *		lc,
				ScreenDrawingData *		sdd )
    {
    double		xfac= lc->lcAdd->addMagnifiedPixelsPerTwip;
    int			x0Ref= drTwips->drX0;
    int			y0Ref= drTwips->drY0;

    int			i;
    const ShapeVertex *	sv;

    APP_POINT *		xp;
    APP_POINT *		xpret;

    xp= xpret= malloc( ( ds->dsVertexCount+ 1)* sizeof(APP_POINT) );
    if  ( ! xp )
	{ LXDEB(ds->dsVertexCount,xp); return (APP_POINT *)0;	}

    sv= ds->dsVertices;
    for ( i= 0; i < ds->dsVertexCount; xp++, sv++, i++ )
	{
	xp->x= X_PIXELS( lc->lcAdd, x0Ref+ sv->svX )- lc->lcOx;
	xp->y= COORDtoGRID( xfac, y0Ref+ sv->svY )- lc->lcOy;
	}

    if  ( ds->dsVertexCount > 0 )
	{ xpret[ds->dsVertexCount]= xpret[0]; }

    return xpret;
    }

static void tedShapeGetLine(	int *				pLine,
				int *				pWidthPixels,
				const DrawingShape *		ds,
				DrawingContext *		dc,
				ScreenDrawingData *		sdd )
    {
    docShapeGetLine( pLine, ds, dc, sdd );

    if  ( *pLine )
	{
	const LayoutContext *	lc= &(dc->dcLayoutContext);
	AppDrawingData *	add= lc->lcAdd;
	int			widthTwips= EMUtoTWIPS( ds->dsLineWidthEmu );
	int			widthPixels= X_PIXELS( add, widthTwips );

	static unsigned char	dash[]= { 3, 2 };
	static unsigned char	dot[]= { 1, 2 };
	static unsigned char	dashdot[]= { 2, 2, 1, 2 };
	static unsigned char	dashdotdot[]= { 2, 2, 1, 2, 1, 2 };

	const unsigned char *	dashList= (const unsigned char *)0;
	int			dashCount= 0;
	int			lineStyle= LINEstyleSOLID;
	int			capStyle= LINEcapBUTT;
	int			joinStyle= LINEjoinMITER;

	if  ( widthPixels < 1 )
	    { widthPixels=  1;	}

	switch( ds->dsLineDashing )
	    {
	    case DSdashSOLID:
		dashList= (const unsigned char *)0;
		dashCount= 0;
		lineStyle= LINEstyleSOLID;
		capStyle= LINEcapBUTT;
		joinStyle= LINEjoinMITER;
		break;

	    case DSdashDASHED:
	    case DSdashDASHED_X:
	    case DSdashDASHED_L:
		dashList= dash;
		dashCount= sizeof( dash );
		lineStyle= LINEstyleON_OFF_DASH;
		capStyle= LINEcapBUTT;
		joinStyle= LINEjoinMITER;
		break;

	    case DSdashDOT:
	    case DSdashDOT_X:
		dashList= dot;
		dashCount= sizeof( dot );
		lineStyle= LINEstyleON_OFF_DASH;
		capStyle= LINEcapBUTT;
		joinStyle= LINEjoinMITER;
		break;

	    case DSdashDASHDOT:
	    case DSdashDASHDOT_X:
	    case DSdashDASHDOT_L:
		dashList= dashdot;
		dashCount= sizeof( dashdot );
		lineStyle= LINEstyleON_OFF_DASH;
		capStyle= LINEcapBUTT;
		joinStyle= LINEjoinMITER;
		break;

	    case DSdashDASHDOTDOT:
	    case DSdashDASHDOTDOT_L:
		dashList= dashdotdot;
		dashCount= sizeof( dashdotdot );
		lineStyle= LINEstyleON_OFF_DASH;
		capStyle= LINEcapBUTT;
		joinStyle= LINEjoinMITER;
		break;

	    default:
		LDEB(ds->dsLineDashing);
		dashList= (const unsigned char *)0;
		dashCount= 0;
	    }

	appDrawSetLineAttributes( add,
		    widthPixels, lineStyle, capStyle, joinStyle,
		    dashList, dashCount );

	*pWidthPixels= widthPixels;
	}
    }

static int tedShapeDrawPoints(	const DrawingShape *		ds,
				const APP_POINT *		xp,
				int				np,
				int				closed,
				DrawingContext *		dc,
				ScreenDrawingData *		sdd )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;

    int			line= 0;
    int			widthPixels;

    if  ( closed )
	{
	int		fill= 0;

	docShapeGetFill( &fill, ds, dc, sdd );
	if  ( fill )
	    { appDrawFillPolygon( add, xp, np ); }
	}

    tedShapeGetLine( &line, &widthPixels, ds, dc, sdd );
    if  ( line )
	{ appDrawDrawLines( add, xp, np, closed ); }

    if  ( line && ! closed && np >= 2 )
	{
	if  ( ds->dsLineStartArrow.saArrowHead != DSarrowNONE )
	    {
	    tedShapeDrawArrowHead( add, xp[0].x, xp[0].y,
					xp[1].x, xp[1].y,
					widthPixels,
					&(ds->dsLineStartArrow) );
	    }
	if  ( ds->dsLineEndArrow.saArrowHead != DSarrowNONE )
	    {
	    tedShapeDrawArrowHead( add, xp[np-1].x, xp[np-1].y,
					xp[np-2].x, xp[np-2].y,
					widthPixels,
					&(ds->dsLineEndArrow) );
	    }
	}

    return 0;
    }

static void tedDrawSetShapePath(APP_POINT *			xp,
				const DrawingShape *		ds,
				const ShapePath *		sp,
				const DocumentRectangle *	dr )
    {
    int			np= sp->spVertexCount;
    const ShapeVertex *	sv= sp->spVertices;
    int			i;

    if  ( ds->dsRotation == 0 )
	{
	int			xs= sp->spXSize;
	int			ys= sp->spYSize;

	int			x0= dr->drX0;
	int			x1= dr->drX1;
	int			y0= dr->drY0;
	int			y1= dr->drY1;

	if  ( DSflipHORIZONTAL( ds ) )
	    { int swap= x0; x0= x1; x1= swap;	}
	if  ( DSflipVERTICAL( ds ) )
	    { int swap= y0; y0= y1; y1= swap;	}

	for ( i= 0; i < np; sv++, i++ )
	    {
	    xp[i].x= ( ( xs- sv->svX )* x0+ sv->svX* x1 )/ xs;
	    xp[i].y= ( ( ys- sv->svY )* y0+ sv->svY* y1 )/ ys;
	    }
	}
    else{
	AffineTransform2D	at;
	double			xs= sp->spXSize;
	double			ys= sp->spYSize;
	double			xm= ( dr->drX1+ dr->drX0 )/ 2.0;
	double			ym= ( dr->drY1+ dr->drY0 )/ 2.0;

	docShapeInternalTransform( &at, ds );

	for ( i= 0; i < np; sv++, i++ )
	    {
	    double	x;
	    double	y;

	    x= ( ( xs- sv->svX )* dr->drX0+ sv->svX* dr->drX1 )/ xs -xm;
	    y= ( ( ys- sv->svY )* dr->drY0+ sv->svY* dr->drY1 )/ ys- ym;

	    xp[i].x= AT2_X( x, y, &at )+ xm;
	    xp[i].y= AT2_Y( x, y, &at )+ ym;
	    }
	}

    xp[np]= xp[0];

    return;
    }

static int tedDrawShapePath(	const DrawingShape *		ds,
				DrawingContext *		dc,
				ScreenDrawingData *		sdd,
				const DocumentRectangle *	drPixels,
				const ShapePath *		sp )
    {
    int			rval= 0;
    int			np= sp->spVertexCount;
    APP_POINT *		xp;

    xp= malloc( ( np+ 1 )* sizeof(APP_POINT) );
    if  ( ! xp )
	{ LXDEB(np,xp); return -1;	}

    tedDrawSetShapePath( xp, ds, sp, drPixels );

    if  ( tedShapeDrawPoints( ds, xp, np, sp->spClosed, dc, sdd ) )
	{ LDEB(np); rval= -1;	}

    free( xp );

    return rval;
    }

static int tedDrawPictureFrame( DrawingShape *			ds,
				DrawingContext *		dc,
				ScreenDrawingData *		sdd,
				const DocumentRectangle *	drPixels )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;

    const ShapePath *	sp= &SP_RECTANGLE;
    int			np= sp->spVertexCount;
    APP_POINT *		xp;

    int			fill= 0;
    int			line= 0;
    int			widthPixels= 0;

    xp= malloc( ( np+ 1 )* sizeof(APP_POINT) );
    if  ( ! xp )
	{ LXDEB(np,xp); return -1;	}

    tedDrawSetShapePath( xp, ds, sp, drPixels );

    docShapeGetFill( &fill, ds, dc, sdd );
    if  ( fill )
	{ appDrawFillPolygon( add, xp, np ); }

    tedDrawPixmap( lc, ds->dsScreenPixmap,
		0, 0, /* src: x,y */
		drPixels->drX1- drPixels->drX0+ 1, /* wide */
		drPixels->drY1- drPixels->drY0+ 1, /* high */
		drPixels->drX0, drPixels->drY0 ); /* dest: x,y */

    tedShapeGetLine( &line, &widthPixels, ds, dc, sdd );
    if  ( line )
	{ appDrawDrawLines( add, xp, np+ 1, 0 ); }

    free( xp );

    return 0;
    }

static int tedDrawOnlineStorage( DrawingShape *			ds,
				DrawingContext *		dc,
				ScreenDrawingData *		sdd,
				const DocumentRectangle *	drPixels )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;

    const ShapePath *	sp= &SP_RECTANGLE;
    int			np= sp->spVertexCount;
    APP_POINT *		xp;

    int			fill= 0;
    int			line= 0;
    int			widthPixels= 0;

    int			wide;
    int			high;
    int			rx0;
    int			rx1;

    xp= malloc( ( np+ 1 )* sizeof(APP_POINT) );
    if  ( ! xp )
	{ LXDEB(np,xp); return -1;	}

    tedDrawSetShapePath( xp, ds, sp, drPixels );

    wide= drPixels->drX1- drPixels->drX0;
    high= drPixels->drY1- drPixels->drY0;
    rx0= ( 5* drPixels->drX0+ 1* drPixels->drX1 )/ 6;
    rx1= ( 1* drPixels->drX0+ 5* drPixels->drX1 )/ 6;

    docShapeGetFill( &fill, ds, dc, sdd );
    if  ( fill )
	{
	appDrawFillArc( add, drPixels->drX0, drPixels->drY0,
			    ( wide+ 2 )/ 3, high+ 1,
			    64* 90, 64* 180 );

	appDrawFillRectangle( add, rx0, drPixels->drY0,
			    rx1- rx0+ 1, high );

	/*
	appDrawFillArc( add, drPixels->drX0,
			    ( 1* drPixels->drY0+ 2* drPixels->drY1 )/ 3,
			    wide+ 1, ( high+ 2 )/ 3,
			    64* 180, 64* 180 );
	*/
	}
    tedShapeGetLine( &line, &widthPixels, ds, dc, sdd );
    if  ( line )
	{
	appDrawDrawArc( add, drPixels->drX0, drPixels->drY0,
			    ( wide+ 2 )/ 3, high+ 1,
			    64* 90, 64* 180 );

	appDrawDrawLine( add, rx0,  drPixels->drY0, rx1,  drPixels->drY0 );
	appDrawDrawLine( add, rx0,  drPixels->drY1, rx1,  drPixels->drY1 );

	appDrawDrawArc( add, ( 1* drPixels->drX0+ 2* drPixels->drX1 )/ 3,
			    drPixels->drY0,
			    ( wide+ 2 )/ 3, high+ 1,
			    64* 90, 64* 180 );
	}

    free( xp );

    return 0;
    }

static int tedDrawMagneticDisk( DrawingShape *			ds,
				DrawingContext *		dc,
				ScreenDrawingData *		sdd,
				const DocumentRectangle *	drPixels )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;

    const ShapePath *		sp= &SP_RECTANGLE;
    int				np= sp->spVertexCount;
    APP_POINT *			xp;

    int				fill= 0;
    int				line= 0;
    int				widthPixels= 0;

    int				wide;
    int				high;
    int				ry0;
    int				ry1;

    xp= malloc( ( np+ 1 )* sizeof(APP_POINT) );
    if  ( ! xp )
	{ LXDEB(np,xp); return -1;	}

    tedDrawSetShapePath( xp, ds, sp, drPixels );

    wide= drPixels->drX1- drPixels->drX0;
    high= drPixels->drY1- drPixels->drY0;
    ry0= ( 5* drPixels->drY0+ 1* drPixels->drY1 )/ 6;
    ry1= ( 1* drPixels->drY0+ 5* drPixels->drY1 )/ 6;

    docShapeGetFill( &fill, ds, dc, sdd );
    if  ( fill )
	{
	appDrawFillArc( add, drPixels->drX0, drPixels->drY0,
			    wide+ 1, ( high+ 2 )/ 3,
			    64* 0, 64* 360 );

	appDrawFillRectangle( add, drPixels->drX0, ry0,
			    wide, ry1- ry0+ 1 );

	appDrawFillArc( add, drPixels->drX0,
			    ( 1* drPixels->drY0+ 2* drPixels->drY1 )/ 3,
			    wide+ 1, ( high+ 2 )/ 3,
			    64* 180, 64* 180 );
	}

    tedShapeGetLine( &line, &widthPixels, ds, dc, sdd );
    if  ( line )
	{
	appDrawDrawArc( add, drPixels->drX0, drPixels->drY0,
			    wide+ 1, ( high+ 2 )/ 3,
			    64* 0, 64* 360 );

	appDrawDrawLine( add, drPixels->drX0, ry0,
			    drPixels->drX0, ry1 );
	appDrawDrawLine( add, drPixels->drX1, ry0,
			    drPixels->drX1, ry1 );

	appDrawDrawArc( add, drPixels->drX0,
			    ( 1* drPixels->drY0+ 2* drPixels->drY1 )/ 3,
			    wide+ 1, ( high+ 2 )/ 3,
			    64* 180, 64* 180 );
	}

    free( xp );
    return 0;
    }


static int tedDrawCallout(	DrawingShape *			ds,
				DrawingContext *		dc,
				ScreenDrawingData *		sdd,
				const DocumentRectangle *	drPixels )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;

    const ShapePath *	sp= &SP_RECTANGLE;
    int			np= sp->spVertexCount;
    APP_POINT *		xp;

    int			fill= 0;
    int			line= 0;
    int			widthPixels= 0;

    if  ( np != 4 )
	{ LDEB(np); return -1;	}

    xp= malloc( ( np+ 1+ 2+ 1 )* sizeof(APP_POINT) );
    if  ( ! xp )
	{ LXDEB(np,xp); return -1;	}

    tedDrawSetShapePath( xp, ds, sp, drPixels );

    docShapeGetFill( &fill, ds, dc, sdd );
    if  ( fill )
	{ appDrawFillPolygon( add, xp, np ); }

    tedShapeGetLine( &line, &widthPixels, ds, dc, sdd );
    if  ( line )
	{
	int		pp0= 5;
	int		pp1= pp0;
	long		a0, a2;

	const int	SC= 21600; /* adjust to geoLeft etc? */

	a2= ds->dsAdjustValue; a0= SC- a2;
	xp[pp1].x= ( a0* xp[0].x+ a2* xp[2].x )/ SC;
	a2= ds->dsAdjust2Value; a0= SC- a2;
	xp[pp1].y= ( a0* xp[0].y+ a2* xp[2].y )/ SC;
	pp1++;

	a2= ds->dsAdjust3Value; a0= SC- a2;
	xp[pp1].x= ( a0* xp[0].x+ a2* xp[2].x )/ SC;
	a2= ds->dsAdjust4Value; a0= SC- a2;
	xp[pp1].y= ( a0* xp[0].y+ a2* xp[2].y )/ SC;
	pp1++;

	appDrawDrawLines( add, xp+ pp0, pp1- pp0, 0 );
	}

    free( xp );

    return 0;
    }

int tedDrawDrawingShape(	const DocumentRectangle *	drTwips,
				int				page,
				DrawingShape *			ds,
				DrawingContext *		dc,
				void *				vsdd )
    {
    const LayoutContext *	lc= &(dc->dcLayoutContext);
    AppDrawingData *		add= lc->lcAdd;
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;

    int				rval= 0;

    DocumentRectangle		drPixels;
    int				fill= 0;
    int				line= 0;
    int				widthPixels= 0;

    int				wide;
    int				high;

    APP_POINT *			xp= (APP_POINT *)0;

    docGetPixelRect( &drPixels, lc, drTwips, page );

    if  ( dc->dcClipRect						&&
	  ! geoIntersectRectangle( (DocumentRectangle *)0,
					    &drPixels, dc->dcClipRect ) )
	{ return 0;	}

    drPixels.drX0 -= lc->lcOx;
    drPixels.drX1 -= lc->lcOx;
    drPixels.drY0 -= lc->lcOy;
    drPixels.drY1 -= lc->lcOy;

    switch( ds->dsShapeType )
	{
	case SHPtyGROUP:
	    /* Done by docDrawDrawingShape(). */
	    LSDEB(ds->dsShapeType,docShapeTypeString(ds->dsShapeType));
	    break;

	case 33:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_33 ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case 34:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_34 ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyPICTURE_FRAME:
	    if  ( tedDrawPictureFrame( ds, dc, sdd, &drPixels ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyCALLOUT_1:
	case SHPtyCALLOUT_90:
	    if  ( tedDrawCallout( ds, dc, sdd, &drPixels ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyRECTANGLE:
	case SHPtyFLOW_CHART_PROCESS:
	case SHPtyACCENT_BORDER_CALLOUT_90:
	case SHPtyTEXT_BOX:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_RECTANGLE ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyROUND_RECTANGLE:
	case SHPtyFLOW_CHART_ALTERNATE_PROCESS:
	    {
	    const int	w= ( drPixels.drX1- drPixels.drX0 )/ 4;
	    const int	h= ( drPixels.drY1- drPixels.drY0 )/ 4;

	    docShapeGetFill( &fill, ds, dc, sdd );
	    if  ( fill )
		{ appDrawFillRoundRectX11( add, &drPixels, w, h );	}
	    tedShapeGetLine( &line, &widthPixels, ds, dc, sdd );
	    if  ( line )
		{ appDrawDrawRoundRectX11( add, &drPixels, w, h );	}
	    }
	    break;

	case SHPtyELLIPSE:
	case SHPtyFLOW_CHART_CONNECTOR:
	    wide= drPixels.drX1- drPixels.drX0;
	    high= drPixels.drY1- drPixels.drY0;
	    docShapeGetFill( &fill, ds, dc, sdd );
	    if  ( fill )
		{
		appDrawFillArc( add, drPixels.drX0, drPixels.drY0,
					    wide+ 1, high+ 1,
					    64* 0, 64* 360 );
		}
	    tedShapeGetLine( &line, &widthPixels, ds, dc, sdd );
	    if  ( line )
		{
		appDrawDrawArc( add, drPixels.drX0, drPixels.drY0,
					    wide+ 1, high+ 1,
					    64* 0, 64* 360 );
		}
	    break;

	case SHPtyARC:
	    wide= drPixels.drX1- drPixels.drX0;
	    high= drPixels.drY1- drPixels.drY0;
	    docShapeGetFill( &fill, ds, dc, sdd );
	    if  ( fill )
		{
		appDrawFillArc( add, drPixels.drX0- wide, drPixels.drY0,
					    2* wide+ 1, 2* high+ 1,
					    64* 0, 64* 90 );
		}
	    tedShapeGetLine( &line, &widthPixels, ds, dc, sdd );
	    if  ( line )
		{
		appDrawDrawArc( add, drPixels.drX0- wide, drPixels.drY0,
					    2* wide+ 1, 2* high+ 1,
					    64* 0, 64* 90 );
		}
	    break;

	case SHPtyDIAMOND:
	case SHPtyFLOW_CHART_DECISION:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_DIAMOND ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyISOSCELES_TRIANGLE:
	case SHPtyFLOW_CHART_EXTRACT:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels,
						    &SP_ISOSCELES_TRIANGLE ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyFLOW_CHART_MERGE:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels,
						    &SP_FLOW_CHART_MERGE ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyRIGHT_TRIANGLE:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels,
						    &SP_RIGHT_TRIANGLE ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyPARALLELOGRAM:
	case SHPtyFLOW_CHART_INPUT_OUTPUT:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels,
						    &SP_PARALLELOGRAM ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyTRAPEZOID:
	case SHPtyFLOW_CHART_MANUAL_OPERATION:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_TRAPEZOID ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyHEXAGON:
	case SHPtyFLOW_CHART_PREPARATION:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_HEXAGON ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyOCTAGON:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_OCTAGON ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyPLUS_SIGN:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_PLUS_SIGN ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyARROW:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_ARROW ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyNOTCHED_RIGHT_ARROW:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_NOTCHED_RIGHT_ARROW ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyHOME_PLATE:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_HOME_PLATE ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyCHEVRON:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_CHEVRON ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyLEFT_ARROW:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_LEFT_ARROW ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyRIGHT_ARROW_CALLOUT:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_RIGHT_ARROW_CALLOUT ) )
		{ LDEB(1); rval= -1;	}
	    break;
	case SHPtyLEFT_ARROW_CALLOUT:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_LEFT_ARROW_CALLOUT ) )
		{ LDEB(1); rval= -1;	}
	    break;
	case SHPtyUP_ARROW_CALLOUT:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_UP_ARROW_CALLOUT ) )
		{ LDEB(1); rval= -1;	}
	    break;
	case SHPtyDOWN_ARROW_CALLOUT:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_DOWN_ARROW_CALLOUT ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyLEFT_RIGHT_ARROW_CALLOUT:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_LEFT_RIGHT_ARROW_CALLOUT ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyUP_DOWN_ARROW_CALLOUT:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_UP_DOWN_ARROW_CALLOUT ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyQUAD_ARROW_CALLOUT:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_QUAD_ARROW_CALLOUT ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyLEFT_RIGHT_ARROW:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_LEFT_RIGHT_ARROW ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyUP_ARROW:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_UP_ARROW ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyDOWN_ARROW:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_DOWN_ARROW ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyUP_DOWN_ARROW:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_UP_DOWN_ARROW ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyQUAD_ARROW:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_QUAD_ARROW ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyLEFT_RIGHT_UP_ARROW:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_LEFT_RIGHT_UP_ARROW ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyLEFT_UP_ARROW:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_LEFT_UP_ARROW ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyBENT_UP_ARROW:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_BENT_UP_ARROW ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyPENTAGON:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_PENTAGON ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtySTAR:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_STAR ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtySEAL4:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_SEAL4 ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case 32:
	case SHPtyLINE:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_LINE ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyFLOW_CHART_MANUAL_INPUT:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_FLOW_CHART_MANUAL_INPUT ) )
		{ LDEB(1); rval= -1;	}
	    break;
	case SHPtyFLOW_CHART_OFF_PAGE_CONNECTOR:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_FLOW_CHART_OFF_PAGE_CONNECTOR ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyFLOW_CHART_PUNCHED_CARD:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_FLOW_CHART_PUNCHED_CARD ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyWEDGE_RECT_CALLOUT:
	    if  ( tedDrawShapePath( ds, dc, sdd, &drPixels, &SP_WEDGE_RECT_CALLOUT ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyFLOW_CHART_ONLINE_STORAGE:
	    if  ( tedDrawOnlineStorage( ds, dc, sdd, &drPixels ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyCAN:
	case SHPtyFLOW_CHART_MAGNETIC_DISK:
	    if  ( tedDrawMagneticDisk( ds, dc, sdd, &drPixels ) )
		{ LDEB(1); rval= -1;	}
	    break;

	case SHPtyRIGHT_BRACE:
	    {
	    const ShapePath *	sp= &SP_RECTANGLE;
	    int			np= sp->spVertexCount;
	    APP_POINT *		xp;

	    int			wide;
	    int			high;

	    xp= malloc( ( np+ 1 )* sizeof(APP_POINT) );
	    if  ( ! xp )
		{ LXDEB(np,xp); return -1;	}

	    tedDrawSetShapePath( xp, ds, sp, &drPixels );

	    wide= xp[2].x- xp[0].x;
	    high= xp[2].y- xp[0].y;

	    tedShapeGetLine( &line, &widthPixels, ds, dc, sdd );

	    appDrawDrawArc( add, xp[0].x- wide/ 2,
				    ( 8* xp[0].y+ 0* xp[2].y )/ 8,
				    wide+ 1, ( high+ 3 )/ 4,
				    64* 0, 64* 90 );

	    appDrawDrawLine( add,
		    xp[0].x+ wide/ 2, ( 7* xp[0].y+ 1* xp[2].y )/ 8,
		    xp[0].x+ wide/ 2, ( 5* xp[0].y+ 3* xp[2].y )/ 8 );

	    appDrawDrawArc( add, xp[2].x- wide/ 2,
				    ( 6* xp[0].y+ 2* xp[2].y )/ 8,
				    wide+ 1, ( high+ 3 )/ 4,
				    64* 180, 64* 90 );

	    appDrawDrawArc( add, xp[2].x- wide/ 2,
				    ( 4* xp[0].y+ 4* xp[2].y )/ 8,
				    wide+ 1, ( high+ 3 )/ 4,
				    64* 90, 64* 90 );

	    appDrawDrawLine( add,
		    xp[0].x+ wide/ 2, ( 3* xp[0].y+ 5* xp[2].y )/ 8,
		    xp[0].x+ wide/ 2, ( 1* xp[0].y+ 7* xp[2].y )/ 8 );

	    appDrawDrawArc( add, xp[0].x- wide/ 2,
				    ( 2* xp[0].y+ 6* xp[2].y )/ 8,
				    wide+ 1, ( high+ 3 )/ 4,
				    64* 270, 64* 90 );

	    free( xp );
	    }
	    break;

	case SHPtyFREEFORM_OR_NON_AUTOSHAPE:
	    if  ( ds->dsVertexCount > 1 )
		{
		const int	closed= 1;

		xp= tedDrawGetVertices( drTwips, ds, lc, sdd );
		if  ( ! xp )
		    { XDEB(xp); rval= -1; goto ready;	}

		tedShapeDrawPoints( ds, xp, ds->dsVertexCount- 1,
							closed, dc, sdd );
		}
	    break;

	case SHPtyBENT_ARROW:
	default:
	    LSDEB(ds->dsShapeType,docShapeTypeString(ds->dsShapeType));
	    break;
	}

    /* Done by docDrawDrawingShape()....
    if  ( ds->dsDocumentTree.eiRoot )
	{}
    */

  ready:
    if  ( xp )
	{ free( xp ); }

    return rval;
    }

