/************************************************************************/
/*									*/
/*  The Column page of the format tool.					*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<stddef.h>
#   include	<limits.h>

#   include	<geoString.h>
#   include	<appUnit.h>

#   include	"tedApp.h"
#   include	"tedFormatTool.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Refresh the 'Column' page of the format tool.			*/
/*									*/
/************************************************************************/

static void tedFormatToolRefreshColumnPage(	ColumnTool *	ct )
    {
    RowProperties *		rp= &(ct->ctPropertiesChosen);
    CellProperties *		cp;
    const TableRectangle *	tr= &(ct->ctTableRectangle);

    int				width;

    EditDocument *		ed;
    BufferDocument *		bd= tedFormatCurDoc( &ed, ct->ctApplication );

    if  ( ! bd )
	{ XDEB(bd); return;	}

    if  ( tr->trCol0 == tr->trCol1 )
	{
	appIntegerToTextWidget( ct->ctNumberText, tr->trCol0+ 1 );
	}
    else{
	char	scratch[30];
	sprintf( scratch, "%d - %d", tr->trCol0+ 1, tr->trCol1+ 1 );
	appStringToTextWidget( ct->ctNumberText, scratch );
	}

    cp= rp->rpCells;

    if  ( tr->trCol0 == 0 )
	{
	width= cp[tr->trCol0].cpRightBoundaryTwips- rp->rpLeftIndentTwips;
	}
    else{
	width= cp[tr->trCol0].cpRightBoundaryTwips-
				    cp[tr->trCol0- 1].cpRightBoundaryTwips;
	}

    appLengthToTextWidget( ct->ctWidthText, width, UNITtyPOINTS );

    tedBorderToolSetPropertiesByNumber( &(ct->ctLeftBorderTool), bd,
					cp[tr->trCol0].cpLeftBorderNumber );
    tedBorderToolSetPropertiesByNumber( &(ct->ctRightBorderTool), bd,
					cp[tr->trCol1].cpRightBorderNumber );
    }

void tedRefreshColumnTool(
				ColumnTool *			ct,
				int *				pEnabled,
				int *				pPref,
				InspectorSubject *		is,
				const DocumentSelection *	ds,
				const SelectionGeometry *	sg,
				const SelectionDescription *	sd )
    {
    const RowProperties *	rp;

    const BufferItem *		rowBi;
    TableRectangle *		tr= &(ct->ctTableRectangle);

    if  ( docGetTableRectangle( tr, ds ) )
	{
	docInitTableRectangle( tr );
	*pEnabled= 0; return;
	}

    rowBi= docGetRowItem( ds->dsHead.dpBi );
    if  ( ! rowBi )
	{ XDEB(rowBi); return;	}

    ct->ctBlockFrameWide= sg->sgBegin.pgBlockFrameX1-
						sg->sgBegin.pgBlockFrameX0;

    rp= &(rowBi->biRowProperties);

    if  ( docCopyRowProperties( &(ct->ctPropertiesChosen), rp,
				(const DocumentAttributeMap *)0 ) )
	{ LDEB(1); return;	}
    if  ( docCopyRowProperties( &(ct->ctPropertiesSet), rp,
				(const DocumentAttributeMap *)0 ) )
	{ LDEB(1); return;	}

    appGuiEnableWidget( is->isPrevButton,
				ct->ctTableRectangle.trCol0 > 0 );
    appGuiEnableWidget( is->isNextButton,
				ct->ctTableRectangle.trCol1 <
				ct->ctTableRectangle.trCol11 );

    tedFormatToolRefreshColumnPage( ct );

    appGuiEnableWidget( is->isDeleteButton, rp->rpCellCount > 1 );

    *pEnabled= ! sd->sdDocumentReadonly;
    return;
    }

/************************************************************************/
/*									*/
/*  Retrieve the width of the column(s) from the width widget.		*/
/*									*/
/*  1)  Determine original width. This is not really necessary, but the	*/
/*	widget routine expects a value.					*/
/*  2)  Retrieve width from GUI.					*/
/*  3)  Decide from what column we can steal space.			*/
/*  4)  Determine what column width is used by other columns.		*/
/*  5)  Determine how much we, including our victim need.		*/
/*  6)  Determine how much column width is availanable.			*/
/*  7)  If no space is avialable, not even if we rob a viction: fail.	*/
/*  8)  Rob victim if necessary.					*/
/*									*/
/************************************************************************/

static int tedFormatToolGetColumnWidth(	const ColumnTool *	ct,
					int *			pVictim,
					int *			pLeftForVictim,
					int *			pWidth )
    {
    const RowProperties *	rp= &(ct->ctPropertiesChosen);
    const TableRectangle *	tr= &(ct->ctTableRectangle);
    const CellProperties *	cp0= rp->rpCells+ tr->trCol0;

    int				width;
    int				changed;

    int				right;

    const int			minValue= 20* 5+ 2* rp->rpHalfGapWidthTwips;
    const int			adaptToMin= 0;
    const int			maxValue= INT_MAX;
    const int			adaptToMax= 0;

    int				victim= -1;
    int				usedByOthers;
    int				neededForUs;
    int				available;
    int				leftForVictim= INT_MAX;
    int				c0= tr->trCol0;
    int				c1= tr->trCol1;

    /*  1  */
    right= cp0->cpRightBoundaryTwips;
    if  ( tr->trCol0 == 0 )
	{ width= right- rp->rpLeftIndentTwips;		}
    else{ width= right- cp0[-1].cpRightBoundaryTwips;	}

    /*  2  */
    if  ( appGetLengthFromTextWidget( ct->ctWidthText,
				&width, &changed, UNITtyPOINTS,
				minValue, adaptToMin, maxValue, adaptToMax ) )
	{ return -1; }

    /*  3  */
    if  ( tr->trCol1 >= rp->rpCellCount- 1 )
	{
	if  ( tr->trCol0 > 0 )
	    { c0= victim= tr->trCol0- 1;	}
	}
    else{ c1= victim= tr->trCol1+ 1;		}

    /*  4  */
    usedByOthers= 0;
    if  ( c0 > 0 )
	{
	usedByOthers += rp->rpCells[c0-1].cpRightBoundaryTwips-
							rp->rpLeftIndentTwips;
	}
    if  ( c1 < rp->rpCellCount- 1 )
	{
	usedByOthers += rp->rpCells[rp->rpCellCount- 1].cpRightBoundaryTwips-
					rp->rpCells[c1].cpRightBoundaryTwips;
	}

    /*  5  */
    neededForUs= ( tr->trCol1- tr->trCol0+ 1 )* width;
    if  ( victim >= 0 )
	{ neededForUs += 20* 5+ 2* rp->rpHalfGapWidthTwips;	}

    if  ( tr->trCol0 == 0 )
	{ right= rp->rpLeftIndentTwips+ width;		}
    else{ right= cp0[-1].cpRightBoundaryTwips+ width;	}

    /*  6  */
    available= ct->ctBlockFrameWide-
			    rp->rpLeftIndentTwips+ rp->rpHalfGapWidthTwips;
    available -= usedByOthers;

    /*  7  */
    if  ( neededForUs > available )
	{
	int		availableWidth= available;

	if  ( victim >= 0 )
	    { availableWidth -= 20* 5+ 2* rp->rpHalfGapWidthTwips;	}

	availableWidth /= ( tr->trCol1- tr->trCol0+ 1 );

	appLengthToTextWidget( ct->ctWidthText, availableWidth, UNITtyPOINTS );
	appRefuseTextValue( ct->ctWidthText );
	return 1;
	}

    /*  8  */
    if  ( victim >= 0 )
	{
	int	lv;

	leftForVictim= available- ( tr->trCol1- tr->trCol0+ 1 )* width;

	if  ( victim == 0 )
	    { lv= rp->rpLeftIndentTwips;			}
	else{ lv= rp->rpCells[victim- 1].cpRightBoundaryTwips;	}

	if  ( rp->rpCells[victim].cpRightBoundaryTwips- lv <= leftForVictim )
	    { victim= -1; leftForVictim= INT_MAX;	}
	}

    *pWidth= width;
    *pVictim= victim;
    *pLeftForVictim= leftForVictim;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Set/Revert buttons have been pushed.				*/
/*									*/
/************************************************************************/

static APP_BUTTON_CALLBACK_H( tedTableChangeColumnPushed, w, voidct )
    {
    ColumnTool *		ct= (ColumnTool *)voidct;
    RowProperties *		rp= &(ct->ctPropertiesChosen);
    const TableRectangle *	tr= &(ct->ctTableRectangle);

    int				victim;
    int				leftForVictim;
    int				width;
    int				res;

    TableRectangle		trChange;

    BorderProperties		bp;

    PropertyMask		rpSetMask;
    PropertyMask		cpSetMask;
    PropertyMask		bpSetMask;

    /**/
    EditDocument *		ed;
    BufferDocument *		bd= tedFormatCurDoc( &ed, ct->ctApplication );
    DocumentProperties *	dp;

    if  ( ! bd )
	{ XDEB(bd); return;	}
    dp= &(bd->bdProperties);

    /**/

    utilPropMaskClear( &rpSetMask );
    utilPropMaskClear( &cpSetMask );

    docInitBorderProperties( &bp );

    /****/
    res= tedFormatToolGetColumnWidth( ct, &victim, &leftForVictim, &width );
    if  ( res != 0 )
	{ goto ready;	}
    PROPmaskADD( &cpSetMask, CLpropCELLX );

    /****/
    utilPropMaskClear( &bpSetMask );
    if  ( tedBorderToolGetProperties( &bp, &bpSetMask,
					    &(ct->ctLeftBorderTool), dp ) )
	{ goto ready;	}

    if  ( ! utilPropMaskIsEmpty( &bpSetMask ) )
	{
	docRowSetLeftBorderInCols( rp, tr->trCol0, tr->trCol0,
						&(bd->bdBorderPropertyList),
						&bpSetMask, &bp );
	PROPmaskADD( &cpSetMask, CLpropLEFT_BORDER );
	}

    /****/
    utilPropMaskClear( &bpSetMask );
    if  ( tedBorderToolGetProperties( &bp, &bpSetMask,
					    &(ct->ctRightBorderTool), dp ) )
	{ goto ready;	}

    if  ( ! utilPropMaskIsEmpty( &bpSetMask ) )
	{
	docRowSetRightBorderInCols( rp, tr->trCol1, tr->trCol1,
						&(bd->bdBorderPropertyList),
						&bpSetMask, &bp );
	PROPmaskADD( &cpSetMask, CLpropRIGHT_BORDER );
	}

    /****/
    docRowPropertiesSetWidth( rp, tr->trCol0, tr->trCol1, width,
						    victim, leftForVictim );

    /****/
    trChange= *tr;
    docExpandTableRectangleToWholeColumns( &trChange );

    if  ( victim >= 0 && trChange.trCol0 > victim )
	{ trChange.trCol0=  victim;	}
    if  ( victim >= 0 && trChange.trCol1 < victim )
	{ trChange.trCol1=  victim;	}

    tedAppSetTableProperties( ct->ctApplication, &trChange,
						&rpSetMask, &cpSetMask, rp );

  ready:
    docCleanBorderProperties( &bp );

    return;
    }

static APP_BUTTON_CALLBACK_H( tedFormatRevertColumnPushed, w, voidct )
    {
    ColumnTool *		ct= (ColumnTool *)voidct;

    docCopyRowProperties( &(ct->ctPropertiesChosen), &(ct->ctPropertiesSet),
					    (const DocumentAttributeMap *)0 );

    tedFormatToolRefreshColumnPage( ct );

    return;
    }

/************************************************************************/
/*									*/
/*  Check a number as typed by the user.				*/
/*									*/
/************************************************************************/

static APP_TXACTIVATE_CALLBACK_H( tedColumnWidthChanged, w, voidct )
    {
    ColumnTool *		ct= (ColumnTool *)voidct;
    int				victim;
    int				leftForVictim;
    int				width;

    if  ( ! tedFormatToolGetColumnWidth( ct, &victim, &leftForVictim, &width ) )
	{ appLengthToTextWidget( ct->ctWidthText, width, UNITtyPOINTS ); }
    }

/************************************************************************/
/*									*/
/*  Previous/Next Column/Row buttons have been pushed.			*/
/*									*/
/************************************************************************/

static APP_BUTTON_CALLBACK_H( tedColumnPreviousColumn, w, voidct )
    {
    ColumnTool *	ct= (ColumnTool *)voidct;
    EditApplication *	ea= ct->ctApplication;
    EditDocument *	ed= ea->eaCurrentDocument;

    TableRectangle	tr;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    tr= ct->ctTableRectangle;

    docExpandTableRectangleToWholeColumns( &tr );

    if  ( docShiftTableRectangleByColumns( &tr, -1 ) )
	{ return;	}

    tedAppSetTableSelection( ed, &tr );

    return;
    }

static APP_BUTTON_CALLBACK_H( tedTableSelectColumn, w, voidct )
    {
    ColumnTool *	ct= (ColumnTool *)voidct;
    EditApplication *	ea= ct->ctApplication;
    EditDocument *	ed= ea->eaCurrentDocument;

    TableRectangle	tr;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    tr= ct->ctTableRectangle;

    docExpandTableRectangleToWholeColumns( &tr );

    tedAppSetTableSelection( ed, &tr );

    return;
    }

static APP_BUTTON_CALLBACK_H( tedColumnNextColumn, w, voidct )
    {
    ColumnTool *	ct= (ColumnTool *)voidct;
    EditApplication *	ea= ct->ctApplication;
    EditDocument *	ed= ea->eaCurrentDocument;

    TableRectangle	tr;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    tr= ct->ctTableRectangle;

    docExpandTableRectangleToWholeColumns( &tr );

    if  ( docShiftTableRectangleByColumns( &tr, +1 ) )
	{ return;	}

    tedAppSetTableSelection( ed, &tr );

    return;
    }

/************************************************************************/
/*									*/
/*  Delete Column/Row/Table buttons have been pushed.			*/
/*									*/
/************************************************************************/

static APP_BUTTON_CALLBACK_H( tedTableDeleteColumn, w, voidct )
    {
    ColumnTool *		ct= (ColumnTool *)voidct;
    const TableRectangle *	tr= &(ct->ctTableRectangle);
    EditApplication *		ea= ct->ctApplication;
    EditDocument *		ed= ea->eaCurrentDocument;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    if  ( tedDeleteColumnsFromRows( ed,
		tr->trRow00, tr->trRow11, tr->trCol0, tr->trCol1 ) )
	{ LLDEB(tr->trRow00,tr->trRow11); return;	}

    return;
    }

/************************************************************************/
/*									*/
/*  Insert/Append Column/Row buttons have been pushed.			*/
/*									*/
/************************************************************************/

static APP_BUTTON_CALLBACK_H( tedTableInsertColumn, w, voidct )
    {
    ColumnTool *	ct= (ColumnTool *)voidct;
    EditApplication *	ea= ct->ctApplication;

    EditDocument *	ed= ea->eaCurrentDocument;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    tedInsertColumnInTable( ed );

    appDocumentChanged( ed, 1 );
    }

static APP_BUTTON_CALLBACK_H( tedTableAppendColumn, w, voidct )
    {
    ColumnTool *	ct= (ColumnTool *)voidct;
    EditApplication *	ea= ct->ctApplication;

    EditDocument *	ed= ea->eaCurrentDocument;

    if  ( ! ed )
	{ XDEB(ed); return;	}

    tedAppendColumnToTable( ed );

    appDocumentChanged( ed, 1 );
    }

/************************************************************************/
/*									*/
/*  Callback from the general color chooser: a color was chosen.	*/
/*									*/
/************************************************************************/

static void tedColumnToolGotColor(	void *			voidct,
					int 			which,
					const RGB8Color *	rgb8 )
    {
    ColumnTool *			ct= (ColumnTool *)voidct;

    switch( which )
	{
	case CLpropLEFT_BORDER:
	    tedBorderSetExplicitColorChoice( &(ct->ctLeftBorderTool), rgb8 );
	    break;

	case CLpropRIGHT_BORDER:
	    tedBorderSetExplicitColorChoice( &(ct->ctRightBorderTool), rgb8 );
	    break;

	default:
	    LDEB(which); return;
	}
    }

/************************************************************************/
/*									*/
/*  Make a column tool, I.E. the 'Column' page of the format tool.	*/
/*									*/
/************************************************************************/

void tedFormatFillColumnPage(	ColumnTool *			ct,
				const ColumnPageResources *	cpr,
				AppInspector *			ai,
				int				subjectPage,
				InspectorSubject *		is,
				APP_WIDGET			pageWidget,
				const InspectorSubjectResources * isr )
    {
    APP_WIDGET	widthLabel;
    APP_WIDGET	columnLabel;

    APP_WIDGET	row= (APP_WIDGET)0;

    const int	textColumns= 10;

    /**/
    ct->ctPageResources= cpr;

    is->isPrivate= ct;
    is->isGotColor= tedColumnToolGotColor;

    /**/
    appMakeLabelAndTextRow( &row, &columnLabel, &(ct->ctNumberText),
			    pageWidget, isr->isrSubjectName, textColumns, 0 );

    appMakeLabelAndTextRow( &row, &widthLabel, &(ct->ctWidthText),
			pageWidget, cpr->cprWidth, textColumns, 1 );

    appGuiSetGotValueCallbackForText( ct->ctWidthText,
					tedColumnWidthChanged, (void *)ct );

    tedMakeBorderTool( &(ct->ctLeftBorderTool), ai, pageWidget,
		cpr->cprLeftBorder, &(cpr->cprBorderToolResources),
		subjectPage, CLpropLEFT_BORDER );

    tedMakeBorderTool( &(ct->ctRightBorderTool), ai, pageWidget,
		cpr->cprRightBorder, &(cpr->cprBorderToolResources),
		subjectPage, CLpropRIGHT_BORDER );

    appInspectorMakeButtonRow( &row, pageWidget,
		    &(is->isPrevButton), &(is->isNextButton),
		    isr->isrPrevButtonText, isr->isrNextButtonText,
		    tedColumnPreviousColumn, tedColumnNextColumn, ct );

    appInspectorMakeButtonRow( &row, pageWidget,
		    &(is->isSelectButton), &(is->isDeleteButton),
		    isr->isrSelectButtonText, isr->isrDeleteButtonText,
		    tedTableSelectColumn, tedTableDeleteColumn, ct );

    appInspectorMakeButtonRow( &row, pageWidget,
		    &(is->isInsertButton), &(is->isAppendButton),
		    isr->isrInsertButtonText, isr->isrAppendButtonText,
		    tedTableInsertColumn, tedTableAppendColumn, ct );

    appInspectorMakeButtonRow( &row, pageWidget,
		    &(is->isRevertButton), &(is->isApplyButton),
		    isr->isrRevert, isr->isrApplyToSubject,
		    tedFormatRevertColumnPushed, tedTableChangeColumnPushed,
		    (void *)ct );

    return;
    }

void tedColumnToolFillChoosers(	ColumnTool *			ct,
					const ColumnPageResources *	cpr )
    {
    return;
    }

void tedFormatFinishColumnPage(		ColumnTool *			ct,
					const ColumnPageResources *	cpr )
    {
    tedFinishBorderTool( &(ct->ctLeftBorderTool) );
    tedFinishBorderTool( &(ct->ctRightBorderTool) );
    }

/************************************************************************/
/*									*/
/*  Clean a column tool							*/
/*									*/
/************************************************************************/

void tedInitColumnTool(	ColumnTool *	ct )
    {
    ct->ctApplication= (EditApplication *)0;
    ct->ctInspector= (AppInspector *)0;
    ct->ctPageResources= (const ColumnPageResources *)0;

    docInitTableRectangle( &(ct->ctTableRectangle) );

    ct->ctBlockFrameWide= 0;

    docInitRowProperties( &(ct->ctPropertiesSet) );
    docInitRowProperties( &(ct->ctPropertiesChosen) );

    ct->ctNumberText= (APP_WIDGET)0;
    ct->ctWidthText= (APP_WIDGET)0;

    tedInitBorderTool( &(ct->ctLeftBorderTool) );
    tedInitBorderTool( &(ct->ctRightBorderTool) );

    return;
    }

void tedCleanColumnTool(	ColumnTool *	ct )
    {
    docCleanRowProperties( &(ct->ctPropertiesSet) );
    docCleanRowProperties( &(ct->ctPropertiesChosen) );

    tedCleanBorderTool( &(ct->ctLeftBorderTool) );
    tedCleanBorderTool( &(ct->ctRightBorderTool) );

    return;
    }

/************************************************************************/
/*									*/
/*  Get column tool resources.						*/
/*									*/
/************************************************************************/

static AppConfigurableResource TED_TedColumnSubjectResourceTable[]=
    {
    APP_RESOURCE( "tableToolColumn",
		offsetof(InspectorSubjectResources,isrSubjectName),
		"Column" ),
    APP_RESOURCE( "tableToolChangeColumn",
		offsetof(InspectorSubjectResources,isrApplyToSubject),
		"Apply to Column" ),
    APP_RESOURCE( "tableToolRevert",
		offsetof(InspectorSubjectResources,isrRevert),
		"Revert" ),
    APP_RESOURCE( "tableToolNextColumn",
		offsetof(InspectorSubjectResources,isrNextButtonText),
		"Next Column" ),
    APP_RESOURCE( "tableToolPreviousColumn",
		offsetof(InspectorSubjectResources,isrPrevButtonText),
		"Previous Column" ),
    APP_RESOURCE( "tableToolSelectColumn",
		offsetof(InspectorSubjectResources,isrSelectButtonText),
		"Select Column" ),
    APP_RESOURCE( "tableToolDeleteColumn",
		offsetof(InspectorSubjectResources,isrDeleteButtonText),
		"Delete Column" ),
    APP_RESOURCE( "tableToolInsertColumn",
		offsetof(InspectorSubjectResources,isrInsertButtonText),
		"Insert Column" ),
    APP_RESOURCE( "tableToolAppendColumn",
		offsetof(InspectorSubjectResources,isrAppendButtonText),
		"Append Column" ),
    };

static AppConfigurableResource TED_TedColumnToolResourceTable[]=
    {
    APP_RESOURCE( "tableToolWidth",
		offsetof(ColumnPageResources,cprWidth),
		"Width" ),

    /**/
    APP_RESOURCE( "tableToolColumnLeftBorder",
		offsetof(ColumnPageResources,cprLeftBorder),
		"Left Border" ),
    APP_RESOURCE( "tableToolColumnRightBorder",
		offsetof(ColumnPageResources,cprRightBorder),
		"Right Border" ),

    /**/
    APP_RESOURCE( "formatToolColumnBorderWidth",
	    offsetof(ColumnPageResources, cprBorderToolResources.btrWidth),
	    "Width" ),
    APP_RESOURCE( "formatToolColumnBorderStyle",
	    offsetof(ColumnPageResources, cprBorderToolResources.btrStyle),
	    "Style" ),
    APP_RESOURCE( "formatToolColumnBorderColor",
	    offsetof(ColumnPageResources, cprBorderToolResources.btrColor),
	    "Color" ),
    APP_RESOURCE( "formatToolColumnBorderColorChooserAutomatic",
	    offsetof(ColumnPageResources,cprBorderToolResources.
			    btrColorChooserResources.ccrAutomaticColor),
	    "Automatic" ),
    APP_RESOURCE( "formatToolColumnBorderColorChooserMoreColors",
	    offsetof(ColumnPageResources,cprBorderToolResources.
			    btrColorChooserResources.ccrMoreColors),
	    "More Colors..." ),
    };

void tedFormatToolGetColumnResourceTable(EditApplication *		ea,
					ColumnPageResources *		cpr,
					InspectorSubjectResources *	isr )
    {
    static int	gotToolResources= 0;
    static int	gotSubjectResources= 0;

    if  ( ! gotToolResources )
	{
	appGuiGetResourceValues( &gotToolResources, ea, (void *)cpr,
				TED_TedColumnToolResourceTable,
				sizeof(TED_TedColumnToolResourceTable)/
				sizeof(AppConfigurableResource) );
	}

    if  ( ! gotSubjectResources )
	{
	appGuiGetResourceValues( &gotSubjectResources, ea, (void *)isr,
				TED_TedColumnSubjectResourceTable,
				sizeof(TED_TedColumnSubjectResourceTable)/
				sizeof(AppConfigurableResource) );
	}

    return;
    }
