/************************************************************************/
/*									*/
/*  Print images to PostScript.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<string.h>

#   include	<appImage.h>
#   include	<appWinMeta.h>
#   include	<sioMemory.h>
#   include	<sioHex.h>

#   include	"docDraw.h"
#   include	"docPsPrint.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Print a series of particules with the same attributes.		*/
/*									*/
/************************************************************************/

typedef int (*PLAY_METAFILE)(	PrintingState *			ps,
				SimpleInputStream *		sis,
				const PostScriptFontList *	psfl,
				int				mapMode,
				int				xWinExt,
				int				yWinExt,
				int				twipsWide,
				int				twipsHigh );

static int docPsPrintMetafile(	PrintingState *			ps,
				const PictureProperties *	pip,
				const MemoryBuffer *		mb,
				const PostScriptFontList *	psfl,
				int				x0,
				int				baseline )
    {
    int				scaleX= pip->pipScaleXUsed;
    int				scaleY= pip->pipScaleYUsed;
    int				xWinExt= pip->pip_xWinExt;
    int				yWinExt= pip->pip_yWinExt;
    int				twipsWide= pip->pipTwipsWide;
    int				twipsHigh= pip->pipTwipsHigh;

    SimpleInputStream *		sisMem;
    SimpleInputStream *		sisMeta;

    PostScriptTypeList		pstl;

    int				y0;

    PLAY_METAFILE		playMetafile;

    psInitPostScriptFaceList( &pstl );

    if  ( docPsListImageFonts( &pstl, pip, mb, psfl, "pf" ) )
	{ LDEB(1); return -1;	}

    switch( pip->pipType )
	{
	case DOCokPICTWMETAFILE:
	    playMetafile= appMetaPlayWmfPs;
	    break;

	case DOCokPICTEMFBLIP:
	    playMetafile= appMetaPlayEmfPs;
	    break;

	case DOCokMACPICT:
	    playMetafile= appMacPictPlayFilePs;
	    break;

	case DOCokPICTJPEGBLIP:
	case DOCokPICTPNGBLIP:
	default:
	    LDEB(pip->pipType); return 0;
	}

    sisMem= sioInMemoryOpen( mb );
    if  ( ! sisMem )
	{ XDEB(sisMem); return -1;	}

    sisMeta= sioInHexOpen( sisMem );
    if  ( ! sisMeta )
	{ XDEB(sisMem); sioInClose( sisMem ); return -1;	}

    y0= baseline- ( ( scaleY/100.0 )* twipsHigh );

    sioOutPrintf( ps->psSos, "100 dict begin\n" );

    psSelectFontProcedures( ps->psSos, &pstl, /*allFonts=*/ 1 );

    sioOutPrintf( ps->psSos, "gsave %d %d translate %%{IMG\n", x0, y0 );

    if  ( scaleX != 100 || scaleY != 100 )
	{
	sioOutPrintf( ps->psSos, "%f %f scale\n", scaleX/100.0, scaleY/100.0 );
	}

    sioOutPrintf( ps->psSos,
		    "0 0 bp %d 0 rl 0 %d rl %d 0 rl 0 %d rl closepath clip\n",
		    twipsWide, twipsHigh, -twipsWide, -twipsHigh );

    if  ( (*playMetafile)( ps, sisMeta, psfl,
				    pip->pipMapMode,
				    xWinExt, yWinExt, twipsWide, twipsHigh ) )
	{ LDEB(1);	}

    sioOutPrintf( ps->psSos, "grestore end %%}IMG\n" );

    sioInClose( sisMeta );
    sioInClose( sisMem );

    psCleanPostScriptFaceList( &pstl );

    ps->psLastPageMarked= ps->psPagesPrinted;

    return 0;
    }

static int psPrintIncludeEpsObject(	PrintingState *		ps,
					InsertedObject *	io,
					int			x0,
					int			baseLine )
    {
    int				rval= 0;
    SimpleInputStream *		sisMem;
    SimpleInputStream *		sisHex;

    DocumentRectangle		drTo;
    DocumentRectangle		drBBox;

    sisMem= sioInMemoryOpen( &(io->ioResultData) );
    if  ( ! sisMem )
	{ XDEB(sisMem); return -1;	}

    sisHex= sioInHexOpen( sisMem );
    if  ( ! sisHex )
	{ XDEB(sisHex); sioInClose( sisMem ); return -1;	}

    drBBox.drX0= 0;
    drBBox.drY0= 0;
    drBBox.drX1= io->ioTwipsWide;
    drBBox.drY1= io->ioTwipsHigh;

    drTo.drX0= x0;
    drTo.drY0= baseLine;
    drTo.drX1= drTo.drX0+ ( io->ioScaleXUsed/ 100.0 )* 20* io->ioTwipsWide;
    drTo.drY1= drTo.drY0- ( io->ioScaleYUsed/ 100.0 )* 20* io->ioTwipsHigh;

    psBeginEpsObject( ps->psSos, &drTo, &drBBox,
					(char *)io->ioObjectData.mbBytes );

    if  ( psIncludeEpsFile( ps->psSos, sisHex ) )
	{ LDEB(1); rval= -1;	}

    sioInClose( sisHex );
    sioInClose( sisMem );

    psEndEpsObject( ps->psSos );

    ps->psLastPageMarked= ps->psPagesPrinted;

    return rval;
    }

static void psPrintObjectBox(	DrawingContext *	dc,
				PrintingState *		ps,
				const InsertedObject *	io,
				int			x0,
				int			y0 )
    {
    int		high;
    int		wide;

    RGB8Color	rgb8;

    bmRGB8SolidBlack( &rgb8 );

    docDrawSetColorRgb( dc, (void *)ps, &rgb8 );

    high= ( io->ioScaleYUsed* io->ioTwipsHigh )/ 100;
    wide= ( io->ioScaleXUsed* io->ioTwipsWide )/ 100;

    sioOutPrintf( ps->psSos, "%d %d bp ", x0, y0- high );
    sioOutPrintf( ps->psSos, "%d 0 rl ", wide );
    sioOutPrintf( ps->psSos, "0 %d rl ", high );
    sioOutPrintf( ps->psSos, "%d 0 rl ", -wide );
    sioOutPrintf( ps->psSos, "closepath stroke\n" );

    ps->psLastPageMarked= ps->psPagesPrinted;
    return;
    }

/************************************************************************/
/*									*/
/*  Find the pixel rectangle inside the cropping margins.		*/
/*									*/
/************************************************************************/

static void docPsGetPictureSelection(
				DocumentRectangle *		dr,
				const PictureProperties *	pip,
				const BitmapDescription *	bd )
    {
    int		c1Twips;

    dr->drX0= 0;
    dr->drY0= 0;
    dr->drX1= bd->bdPixelsWide- 1;
    dr->drY1= bd->bdPixelsHigh- 1;

    if  ( pip->pipLeftCropTwips+ pip->pipRightCropTwips > 0 )
	{
	dr->drX0= ( bd->bdPixelsWide* pip->pipLeftCropTwips )/pip->pipTwipsWide;
	c1Twips= pip->pipTwipsWide- pip->pipRightCropTwips;
	dr->drX1= ( bd->bdPixelsWide* c1Twips )/ pip->pipTwipsWide;

	if  ( dr->drX0 < 0 )
	    { dr->drX0=  0;	}
	if  ( dr->drX1 > bd->bdPixelsWide- 1 )
	    { dr->drX1=  bd->bdPixelsWide- 1;	}
	}

    if  ( pip->pipTopCropTwips+ pip->pipBottomCropTwips > 0 )
	{
	dr->drY0= ( bd->bdPixelsHigh* pip->pipTopCropTwips )/pip->pipTwipsHigh;
	c1Twips= pip->pipTwipsHigh- pip->pipBottomCropTwips;
	dr->drY1= ( bd->bdPixelsHigh* c1Twips )/ pip->pipTwipsHigh;

	if  ( dr->drY0 < 0 )
	    { dr->drY0=  0;	}
	if  ( dr->drY1 > bd->bdPixelsHigh- 1 )
	    { dr->drY1=  bd->bdPixelsHigh- 1;	}
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Print a bitmap image included in the document.			*/
/*									*/
/************************************************************************/

static int docPsPrintBitmapObject( PrintingState *		ps,
				DrawingContext *		dc,
				int				x0,
				int				baseLine,
				const AppBitmapImage *		abi,
				const PictureProperties *	pip )
    {
    const BitmapDescription *	bd;

    double		scaleX= pip->pipScaleXUsed/ 100.0;
    double		scaleY= pip->pipScaleYUsed/ 100.0;

    int			imageWideTwips;
    int			imageHighTwips;

    DocumentRectangle	drSel;
    int			selWidePix;
    int			selHighPix;

    bd= &(abi->abiBitmap);

    bmImageSizeTwips( &imageWideTwips, &imageHighTwips, bd );

    if  ( imageWideTwips > 20 )
	{ scaleX= ( scaleX* pip->pipTwipsWide )/ imageWideTwips;	}
    if  ( imageHighTwips > 20 )
	{ scaleY= ( scaleY* pip->pipTwipsHigh )/ imageHighTwips;	}

    docPsGetPictureSelection( &drSel, pip, bd );
    selWidePix= drSel.drX1- drSel.drX0+ 1;
    selHighPix= drSel.drY1- drSel.drY0+ 1;

    if  ( bmPsPrintBitmap( ps->psSos, 1,
			    20.0* scaleX, -20.0* scaleY,
			    x0, baseLine, (const DocumentRectangle *)0,
			    ps->psUsePostScriptFilters,
			    ps->psUsePostScriptIndexedImages,
			    bd, abi->abiBuffer ) )
	{ LDEB(1); return -1; }

    ps->psLastPageMarked= ps->psPagesPrinted;
    return 0;
    }

static int docPsPrintShapeBitmap(
				int				kind,
				PrintingState *			ps,
				DrawingShape *			ds,
				const AffineTransform2D *	at )
    {
    const PictureProperties *	pip= &(ds->dsPictureProperties);
    int				x0= 0;
    int				y0= 0;

    if  ( ! ds->dsPrivate )
	{
	if  ( docGetBitmapForObjectData( kind,
				    &(ds->dsPrivate), &(ds->dsPictureData) ) )
	    { XDEB(ds->dsPrivate);	}
	}

    if  ( ds->dsPrivate )
	{
	const AppBitmapImage *		abi= (AppBitmapImage *)ds->dsPrivate;
	const BitmapDescription *	bd= &(abi->abiBitmap);
	AffineTransform2D		atLocal;

	DocumentRectangle		drSel;
	int				picWideTwips;
	int				picHighTwips;

	double				xs;
	double				ys;
	const int			onWhite= 0;

	/* Center image in frame */
	geoTranslationAffineTransform2D( &atLocal, -0.5, 0.5 );
	geoAffineTransform2DProduct( &atLocal, at, &atLocal );

	x0= AT2_X( 0, 0, &atLocal );
	y0= AT2_Y( 0, 0, &atLocal );

	picWideTwips= ( pip->pipScaleXUsed* pip->pipTwipsWide )/ 100;
	picHighTwips= ( pip->pipScaleYUsed* pip->pipTwipsHigh )/ 100;
	xs= picWideTwips;
	ys= -picHighTwips;

	docPsGetPictureSelection( &drSel, pip, bd );

	if  ( bmPsPrintBitmapImage( ps->psSos, 1, xs, ys, x0, y0, &drSel,
					    onWhite,
					    ps->psUsePostScriptFilters,
					    ps->psUsePostScriptIndexedImages,
					    bd, abi->abiBuffer ) )
	    { LDEB(1); return -1; }
	}

    ps->psLastPageMarked= ps->psPagesPrinted;
    return 0;
    }

int docPsPrintShapeImage(	PrintingState *			ps,
				DrawingContext *		dc,
				DrawingShape *			ds,
				const AffineTransform2D *	at,
				const PostScriptFontList *	psfl )
    {
    const PictureProperties *	pip= &(ds->dsPictureProperties);

    const int			x0= 0;
    const int			y0= 0;

    switch( pip->pipType )
	{
	case DOCokPICTWMETAFILE:
	case DOCokMACPICT:
	case DOCokPICTEMFBLIP:
	    if  ( docPsPrintMetafile( ps, pip, &(ds->dsPictureData), psfl,
								    x0, y0 ) )
		{ LDEB(1); break;	}

	    dc->dcCurrentTextAttributeSet= 0;
	    dc->dcCurrentColorSet= 0;
	    return 0;

	case DOCokPICTPNGBLIP:
	case DOCokPICTJPEGBLIP:
	    if  ( docPsPrintShapeBitmap(  pip->pipType, ps, ds, at ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    LDEB(pip->pipType); return 0;
	}

    return 0;
    }

static int docPsPrintJpegImage( PrintingState *			ps,
				int				x0,
				int				baseLine,
				InsertedObject *		io )
    {
    int				rval= 0;
    MemoryBuffer *		mb= &(io->ioObjectData);

    unsigned int		bitsPerComponent;
    unsigned int		componentCount;
    unsigned int		pixelsWide;
    unsigned int		pixelsHigh;

    const PictureProperties *	pip= &(io->ioPictureProperties);

    SimpleInputStream *	sisMem= (SimpleInputStream *)0;
    SimpleInputStream *	sisBitmap= (SimpleInputStream *)0;

    double		scaleX= pip->pipScaleXUsed/ 100.0;
    double		scaleY= pip->pipScaleYUsed/ 100.0;

    sisMem= sioInMemoryOpen( mb );
    if  ( ! sisMem )
	{ XDEB(sisMem); rval= -1; goto ready;	}

    sisBitmap= sioInHexOpen( sisMem );
    if  ( ! sisBitmap )
	{ XDEB(sisMem); rval= -1; goto ready;	}

    if  ( bmEpsTestJpegEmbeddable( &pixelsWide, &pixelsHigh,
			    &componentCount, &bitsPerComponent, sisBitmap ) )
	{ rval= 1; goto ready;	}

    sioInClose( sisBitmap ); sisBitmap= (SimpleInputStream *)0;
    sioInClose( sisMem ); sisMem= (SimpleInputStream *)0;

    sisMem= sioInMemoryOpen( mb );
    if  ( ! sisMem )
	{ XDEB(sisMem); rval= -1; goto ready;	}

    sisBitmap= sioInHexOpen( sisMem );
    if  ( ! sisBitmap )
	{ XDEB(sisMem); rval= -1; goto ready;	}

    if  ( bmPsPrintJpegImage( ps->psSos,
			pip->pipTwipsWide* scaleX, -pip->pipTwipsHigh* scaleY,
			componentCount, x0, baseLine,
			pixelsWide, pixelsHigh, bitsPerComponent, sisBitmap ) )
	{ LDEB(1); rval= -1; goto ready;	}

    ps->psLastPageMarked= ps->psPagesPrinted;

  ready:

    if  ( sisBitmap )
	{ sioInClose( sisBitmap ); }
    if  ( sisMem )
	{ sioInClose( sisMem ); }

    return rval;
    }

static int docPsPrintBitmapImage(	PrintingState *		ps,
					DrawingContext *	dc,
					int			xShift,
					int			yShift,
					InsertedObject *	io,
					const ParticuleData *	pd,
					int			baseLine )
    {
    if  ( ! io->ioPrivate )
	{
	if  ( docGetBitmapForObject( io ) )
	    { XDEB(io->ioPrivate);	}
	}

    if  ( io->ioPrivate )
	{
	AppBitmapImage *	abi= (AppBitmapImage *)io->ioPrivate;

	if  ( ps->psUsePostScriptFilters	&&
	      io->ioKind == DOCokPICTJPEGBLIP	)
	    {
	    if  ( ! docPsPrintJpegImage( ps,
			    pd->pdX0+ xShift, baseLine+ yShift, io ) )
		{
		ps->psLinkParticulesDone++;
		return 1;
		}
	    }

	if  ( docPsPrintBitmapObject( ps, dc,
				pd->pdX0+ xShift, baseLine+ yShift,
				abi, &(io->ioPictureProperties) ) )
	    { LDEB(1); return -1;	}

	ps->psLinkParticulesDone++;
	return 1;
	}

    return 0;
    }

int docPsPrintObject(		PrintingState *			ps,
				DrawingContext *		dc,
				int				xShift,
				int				yShift,
				InsertedObject *		io,
				const PostScriptFontList *	psfl,
				const ParticuleData *		pd,
				int				baseLine )
    {
    const PictureProperties *	pip= &(io->ioPictureProperties);

    switch( io->ioKind )
	{
	int		done;

	case DOCokPICTWMETAFILE:
	case DOCokPICTEMFBLIP:
	case DOCokMACPICT:

	    if  ( docPsPrintMetafile( ps, pip, &(io->ioObjectData), psfl,
				    pd->pdX0+ xShift, baseLine+ yShift ) )
		{ LDEB(1); break;	}

	    dc->dcCurrentTextAttributeSet= 0;
	    dc->dcCurrentColorSet= 0;
	    ps->psLinkParticulesDone++;
	    return 1;

	case DOCokPICTJPEGBLIP:
	case DOCokPICTPNGBLIP:

	    done= docPsPrintBitmapImage( ps, dc, xShift, yShift,
							io, pd, baseLine );
	    return done;
	    break;

	case DOCokOLEOBJECT:
	    if  ( io->ioResultKind == DOCokPICTWMETAFILE )
		{
		if  ( docPsPrintMetafile( ps, pip, &(io->ioResultData), psfl,
					pd->pdX0+ xShift, baseLine+ yShift ) )
		    { LDEB(1); break;	}

		dc->dcCurrentTextAttributeSet= 0;
		dc->dcCurrentColorSet= 0;
		ps->psLinkParticulesDone++;
		return 1;
		}

	    if  ( io->ioResultKind == DOCokPICTJPEGBLIP	||
	    	  io->ioResultKind == DOCokPICTPNGBLIP	)
		{
		done= docPsPrintBitmapImage( ps, dc, xShift, yShift,
							io, pd, baseLine );
		return done;
		}

	    break;

	case DOCokEPS_FILE:
	    if  ( psPrintIncludeEpsObject( ps, io,
			    pd->pdX0+ xShift, baseLine+ yShift ) )
		{ LDEB(1); break;	}

	    dc->dcCurrentTextAttributeSet= 0;
	    dc->dcCurrentColorSet= 0;
	    ps->psLinkParticulesDone++;
	    return 1;

	case DOCokDRAWING_SHAPE:
	    /*  Done in a separate loop from generic drawing code */
	    return 1;

	default:
	    LDEB(io->ioKind); return 0;
	}

    psPrintObjectBox( dc, ps, io, pd->pdX0+ xShift, baseLine+ yShift );
    ps->psLinkParticulesDone++;
    return 1;
    }


