/************************************************************************/
/*									*/
/*  Layout of a document.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>

#   include	<appImage.h>
#   include	"docLayout.h"
#   include	"docParticuleData.h"

#   include	<appDebugon.h>

#   define	SHOW_LINE_CHANGES	0

/************************************************************************/
/*									*/
/*  Layout successive lines of a paragraph.				*/
/*  Claim space above a particular line.				*/
/*									*/
/*  2)  Do not claim 'space before' at the top of the page.		*/
/*									*/
/************************************************************************/

static int docAboveLine(	int *				pStopCode,
				LayoutPosition *		lp,
				const BufferItem *		paraBi,
				int				part,
				const BlockFrame *		bf,
				const ParagraphFrame *		pf )
    {
    int			spaceAboveLineTwips= 0;

    if  ( part == 0 )
	{
	spaceAboveLineTwips= paraBi->biParaTopInset;

	/*  2  */
	if  ( lp->lpAtTopOfColumn )
	    { spaceAboveLineTwips -= paraBi->biParaSpaceBeforeTwips;	}
	}

    if  ( lp->lpPageYTwips+ spaceAboveLineTwips >= bf->bfContentRect.drY1 )
	{ *pStopCode= FORMATstopPAGE_FULL; return 1;	}

    if  ( lp->lpPageYTwips+ spaceAboveLineTwips >= pf->pfParaContentRect.drY1 )
	{ *pStopCode= FORMATstopFRAME_FULL; return 1;	}

    lp->lpPageYTwips += spaceAboveLineTwips;
    /* still at top */

    return 0;
    }

/************************************************************************/
/*									*/
/*  Layout successive lines of a paragraph.				*/
/*  Claim space below a particular line.				*/
/*									*/
/*  Also decide whether it will fit in the current formatting frame.	*/
/*									*/
/*  1)  Only use the space after paragraph property and the insets	*/
/*	of the table when the paragraph is in a table.			*/
/*  2)  Push the position for the next line down until below any shapes	*/
/*	on this line. Until we implement the various wrap modes, this	*/
/*	is the best we can do.						*/
/*									*/
/************************************************************************/

static int docBelowLine(	int *				pStopCode,
				LayoutPosition *		lp,
				const LayoutPosition *		lpLineTop,
				int				lineStride,
				const BufferItem *		paraBi,
				int				partFrom,
				int				partUpto,
				const BlockFrame *		bf,
				const ParagraphFrame *		pf,
				const NotesReservation *	nrLine )
    {
    int				spaceBelowLineTwips= 0;
    int				lineBottom;
    int				lineHeight;
    LayoutPosition		lpBelowLine;

    int				flowY1WithNotes;

    int				footnoteHeight;

    /*  1  */
    if  ( partUpto == paraBi->biParaParticuleCount			&&
	  paraBi->biParaTableNesting > 0				&&
	  paraBi->biNumberInParent ==
				    paraBi->biParent->biChildCount- 1	)
	{ spaceBelowLineTwips += paraBi->biParaBottomInset; }

    lpBelowLine= *lpLineTop;
    lpBelowLine.lpPageYTwips += lineStride;
					/********************************/
					/*  But add spacing to find	*/
					/*  position for next line	*/
					/********************************/

    lineBottom= lpBelowLine.lpPageYTwips+ spaceBelowLineTwips;
    lineHeight= lineBottom- lpLineTop->lpPageYTwips;

    footnoteHeight= nrLine->nrFtnsepHeight+ nrLine->nrFootnoteHeight;
    flowY1WithNotes= bf->bfFlowRect.drY1- footnoteHeight;

    if  ( lineBottom > flowY1WithNotes					&&
	  lineHeight < bf->bfContentRect.drY1- bf->bfContentRect.drY0	)
	{ *pStopCode= FORMATstopPAGE_FULL; return 1;	}

    if  ( lineBottom > pf->pfParaContentRect.drY1				&&
	  lineHeight < pf->pfParaContentRect.drY1- pf->pfParaContentRect.drY0	)
	{ *pStopCode= FORMATstopFRAME_FULL; return 1;	}

    *lp= lpBelowLine;
    lp->lpAtTopOfColumn= 0;

    if  ( partUpto == paraBi->biParaParticuleCount )
	{
	spaceBelowLineTwips= paraBi->biParaBottomInset;

	lp->lpPageYTwips += spaceBelowLineTwips;
	lp->lpAtTopOfColumn= 0;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Calculations on line inserts.					*/
/*									*/
/************************************************************************/

static int docPlaceShape(	const BufferItem *	paraBi,
				const LayoutPosition *	lpLineTop,
				InsertedObject *	io,
				const ParagraphFrame *	pf,
				const BlockFrame *	bf,
				int			xChar )
    {
    DrawingShape *		ds= io->ioDrawingShape;
    int				x1;
    LayoutPosition		lpBelowShape;

    if  ( ! ds )
	{ XDEB(ds); return -1;	}

    docShapePageRectangle( &(io->ioY0Position), &lpBelowShape,
				&(io->ioX0Twips), &x1,
				ds, paraBi, lpLineTop, pf, bf, xChar );

    docDrawingShapeInvalidateTextLayout( ds );

    return 0;
    }

static int docPlaceShapeY(	const BufferItem *	paraBi,
				const LayoutPosition *	lpLineTop,
				InsertedObject *	io,
				const BlockFrame *	bf )
    {
    DrawingShape *		ds= io->ioDrawingShape;
    LayoutPosition		lpBelowShape;

    docShapePageY( &(io->ioY0Position), &lpBelowShape,
						ds, paraBi, lpLineTop, bf );

    docDrawingShapeInvalidateTextLayout( ds );

    return 0;
    }

static int docPlaceLineInserts(	BufferDocument *	bd,
				const BufferItem *	paraBi,
				const TextLine *	tl,
				const ParticuleData *	pd,
				const ParagraphFrame *	pf,
				const BlockFrame *	bf,
				const LayoutPosition *	lp )
    {
    int				i;
    const TextParticule *	tp;

    tp= paraBi->biParaParticules+ tl->tlFirstParticule;
    for ( i= 0; i < tl->tlParticuleCount; tp++, i++ )
	{
	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    docSetPageOfField( &(bd->bdFieldList),
					tp->tpObjectNumber,  lp->lpPage );
	    }

	if  ( tp->tpKind == DOCkindOBJECT )
	    {
	    InsertedObject *	io;

	    io= docGetObject( bd, tp->tpObjectNumber );
	    if  ( ! io )
		{ LPDEB(tp->tpObjectNumber,io);	}

	    if  ( io && io->ioKind == DOCokDRAWING_SHAPE )
		{
		if  ( docPlaceShape( paraBi, &(tl->tlTopPosition),
						    io, pf, bf, pd->pdX0 ) )
		    { LDEB(1);	}
		}
	    }
	}

    return 0;
    }

static int docPlaceLineInsertsY(BufferDocument *	bd,
				const BufferItem *	paraBi,
				const TextLine *	tl,
				const BlockFrame *	bf,
				const LayoutPosition *	lp )
    {
    int				i;
    const TextParticule *	tp;

    tp= paraBi->biParaParticules+ tl->tlFirstParticule;
    for ( i= 0; i < tl->tlParticuleCount; tp++, i++ )
	{
	if  ( tp->tpKind == DOCkindFIELDSTART )
	    {
	    docSetPageOfField( &(bd->bdFieldList),
					tp->tpObjectNumber,  lp->lpPage );
	    }

	if  ( tp->tpKind == DOCkindOBJECT )
	    {
	    InsertedObject *	io;

	    io= docGetObject( bd, tp->tpObjectNumber );
	    if  ( ! io )
		{ LPDEB(tp->tpObjectNumber,io);	}

	    if  ( io && io->ioKind == DOCokDRAWING_SHAPE )
		{
		if  ( docPlaceShapeY( paraBi, &(tl->tlTopPosition), io, bf ) )
		    { LDEB(1);	}
		}
	    }
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Layout successive lines of a paragraph.				*/
/*									*/
/*  1)  Cope with the output of sgmls: Ignore enormous space before's	*/
/*	in footers.							*/
/*									*/
/************************************************************************/

static int docLayout_Line(	int *				pStopCode,
				TextLine *			resTl,
				NotesReservation *		pNrLine,
				const BlockFrame *		bf,
				BufferItem *			paraBi,
				int				redoLineLayout,
				int				part,
				ParticuleData *			pd,
				const LayoutJob *		lj,
				const ParagraphFrame *		pf,
				const LayoutPosition *		lpTop,
				LayoutPosition *		lpBottom )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    BufferDocument *		bd= lc->lcDocument;

    int				accepted;
    int				res;

    TextParticule *		tp= paraBi->biParaParticules+ part;

    TextLine			workTl;
    TextLine *			tl= &(workTl);

    LayoutPosition		lpHere;
    const SectionProperties *	sp;
    const DocumentGeometry *	dg;

    NotesReservation		nrLine;

    lpHere= *lpTop;
    /* Should not this refer to lj->ljBodySectBi? */
    sp= &(paraBi->biParent->biParent->biParent)->biSectProperties;
    dg= &(sp->spDocumentGeometry);

    if  ( redoLineLayout )
	{
	docInitTextLine( tl );
	tl->tlStroff= tp->tpStroff;
	}
    else{
	workTl= *resTl;
	}
    tl->tlFirstParticule= part;

    docInitNotesReservation( &nrLine );

    /*  1  */
    res= docAboveLine( pStopCode, &lpHere, paraBi, part, bf, pf );
    if  ( res < 0 )
	{ LDEB(res); return -1;	}
    if  ( res > 0 )
	{ return 0; }

    tl->tlTopPosition= lpHere;
    if  ( redoLineLayout )
	{
	accepted= docLayoutLineBox( bd, tl, paraBi, part,
					lc->lcPostScriptFontList, pd, pf );
	}
    else{
	accepted= tl->tlParticuleCount;
	}

    if  ( accepted < 1 )
	{ LDEB(accepted); return -1;	}

    if  ( docLayoutCollectParaFootnoteHeight( &nrLine,
		    tl->tlTopPosition.lpPage, tl->tlTopPosition.lpColumn,
		    bd, lj->ljBodySectBi, paraBi,
		    tl->tlFirstParticule, tl->tlFirstParticule+ accepted ) )
	{ LDEB(1); return -1;	}

    res= docBelowLine( pStopCode, &lpHere,
				    &(tl->tlTopPosition), tl->tlLineStride,
				    paraBi, part, part+ accepted,
				    bf, pf, &nrLine );
    if  ( res < 0 )
	{ LDEB(res); return -1;	}
    if  ( res > 0 )
	{ return 0; }

    if  ( redoLineLayout )
	{
	tl->tlStrlen=
	    tp[accepted-1].tpStroff+ tp[accepted-1].tpStrlen- tp->tpStroff;
	tl->tlParticuleCount= accepted;
	tl->tlFlowWidthTwips= pf->pfParaContentRect.drX1- pf->pfParaContentRect.drX0;

	if  ( lj->ljLayoutScreenLine					&&
	      (*lj->ljLayoutScreenLine)( tl, paraBi, lc, pf, pd ) < 0	)
	    { LDEB(accepted); return -1;	}

	if  ( docPlaceLineInserts( bd, paraBi, tl, pd, pf, bf, &lpHere ) )
	    { LDEB(accepted); }
	}
    else{
	if  ( docPlaceLineInsertsY( bd, paraBi, tl, bf, &lpHere ) )
	    { LDEB(accepted); }
	}

    *lpBottom= lpHere;
    *pNrLine= nrLine;
    *resTl= *tl;

    return accepted;
    }

/************************************************************************/
/*									*/
/*  Add the rectangle of the old and the new locations of a line to the	*/
/*  rectangle that must be redrawn.					*/
/*									*/
/************************************************************************/

static void docLayoutAddLineToExpose(
				DocumentRectangle *		drChanged,
				const LayoutJob *		lj,
				const ParagraphFrame *		pf,
				const TextLine *		boxLine,
				const TextLine *		tlLine )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    DocumentRectangle		drBox;
    double			xfac;

    LayoutPosition		lpTop;
    LayoutPosition		lpBottom;

    xfac= lc->lcAdd->addMagnifiedPixelsPerTwip;

    lpTop= boxLine->tlTopPosition;
    lpBottom= lpTop;
    lpBottom.lpPageYTwips += boxLine->tlLineStride;

    docGetPixelRectForPos( &drBox, lc,
			    pf->pfRedrawX0Twips, pf->pfRedrawX1Twips,
			    &lpTop, &lpBottom );

    geoUnionRectangle( drChanged, drChanged, &drBox );

    if  ( tlLine )
	{
	DocumentRectangle	drTl;

	lpTop= tlLine->tlTopPosition;
	lpBottom= lpTop;
	lpBottom.lpPageYTwips += tlLine->tlLineStride;

	docGetPixelRectForPos( &drTl, lc,
			    pf->pfRedrawX0Twips, pf->pfRedrawX1Twips,
			    &lpTop, &lpBottom );

	if  ( drTl.drX0 != drBox.drX0	||
	      drTl.drX1 != drBox.drX1	||
	      drTl.drY0 != drBox.drY0	||
	      drTl.drY1 != drBox.drY1	)
	    {
	    geoUnionRectangle( drChanged, drChanged, &drTl );
	    geoUnionRectangle( drChanged, drChanged, &drBox );

#	    if SHOW_LINE_CHANGES
	    appDebug( "EXPOSE [%4d..%4d x %4d..%4d]\n",
				    drChanged->drX0, drChanged->drX1,
				    drChanged->drY0, drChanged->drY1 );
#	    endif
	    }
	}
    else{
	geoUnionRectangle( drChanged, drChanged, &drBox );

#	if SHOW_LINE_CHANGES
	appDebug( "EXPOSE [%4d..%4d x %4d..%4d]\n",
				    drChanged->drX0, drChanged->drX1,
				    drChanged->drY0, drChanged->drY1 );
#	endif
	}
    }

/************************************************************************/
/*									*/
/*  Layout a series of lines in a paragraph.				*/
/*									*/
/*  1)  Make sure that we have scratch space for the layout routines.	*/
/*  2)  As long as there are any particules to be placed/formatted	*/
/*  3)  Can the current line be reused?					*/
/*  4)  If so just place it at a new position.				*/
/*  5)  Otherwise recalculate layout.					*/
/*  6)  If the line does not fit on this page (column) stop.		*/
/*  7)  Cause the line to be redrawn if either it is reformatted, or it	*/
/*	is moved.							*/
/*  8)  Subtract the space needed for the footnotes in this line from	*/
/*	the space left on this page. (column)				*/
/*  9)  Insert into administration.					*/
/*  10) If the line ends in a page break, make sure nothing will fit on	*/
/*	this page (in this column) anymore.				*/
/*  11) Truncate the number of lines when the paragraph is completely	*/
/*	formatted.							*/
/*									*/
/************************************************************************/

int docLayoutParaLines(		int *				pStopCode,
				const ParagraphFrame *		pf,
				LayoutPosition *		lpHere,
				int *				pLine,
				BlockFrame *			bf,
				const LayoutJob *		lj,
				BufferItem *			paraBi,
				int				part )
    {
    int				stopCode= FORMATstopREADY;

    int				line= (*pLine);
    int				done= 0;

    ParticuleData *		pd;

    LayoutPosition		lp= (*lpHere);

    /*  1  */
    if  ( docPsClaimParticuleData( paraBi->biParaParticuleCount, &pd ) )
	{ LDEB(paraBi->biParaParticuleCount); return -1;	}

    /*  2  */
    while( part < paraBi->biParaParticuleCount )
	{
	int			accepted;
	TextLine		boxLine;

	int			placeOldLine= 0;
	TextLine *		tlLine= (TextLine *)0;

	NotesReservation	nrLine;

	DocumentRectangle *	drChanged= lj->ljChangedRectanglePixels;

	int			flowWidth;

	docInitNotesReservation( &nrLine );

	flowWidth= pf->pfParaContentRect.drX1- pf->pfParaContentRect.drX0;

	docInitTextLine( &boxLine );

	/*  3  */
	if  ( line < paraBi->biParaLineCount )
	    {
	    const TextParticule *	tp= paraBi->biParaParticules+ part;

	    tlLine= paraBi->biParaLines+ line;
	    placeOldLine= 1;
	    boxLine= *tlLine;

	    if  ( tlLine->tlFirstParticule+ tlLine->tlParticuleCount >
					paraBi->biParaParticuleCount	||
		  tlLine->tlStroff != tp->tpStroff			||
		  tlLine->tlFlowWidthTwips != flowWidth			)
		{ placeOldLine= 0;	}
	    }

	/*  4, 5  */
	accepted= docLayout_Line( &stopCode, &boxLine, &nrLine, bf,
			paraBi, ! placeOldLine, part, pd, lj, pf, &lp, &lp );

	if  ( accepted < 0 )
	    { LDEB(accepted); return -1;	}

	/*  6  */
	if  ( accepted == 0 )
	    { break;	}

	/*  7  */
	if  ( drChanged	)
	    {
	    docLayoutAddLineToExpose( drChanged, lj, pf, &boxLine, tlLine );
	    }

	/*  8  */
	docLayoutReserveNoteHeight( bf, &nrLine );

	/*  9  */
	if  ( line >= paraBi->biParaLineCount )
	    {
	    tlLine= docInsertTextLine( paraBi, -1 );
	    if  ( ! tlLine )
		{ XDEB(tlLine); return -1;		}
	    }
	else{
	    tlLine= paraBi->biParaLines+ line;
	    }

	*tlLine= boxLine;

	part += accepted; done += accepted; line++;

	/*  10  */
	if  ( boxLine.tlFlags & TLflagPAGEBREAK )
	    { stopCode= FORMATstopPAGE_FULL; break; }
	}

    *pLine= line;
    *lpHere= lp;
    *pStopCode= stopCode;

    /*  11  */
    if  ( part >= paraBi->biParaParticuleCount	&&
	  paraBi->biParaLineCount > line	)
	{ paraBi->biParaLineCount=  line; }

    return done;
    }

