/************************************************************************/
/*									*/
/*  Layout of a document. Do the layout of the particules inside the	*/
/*  line.								*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>

#   include	<psFontMetrics.h>
#   include	<appMatchFont.h>
#   include	"docLayout.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Get font information.						*/
/*									*/
/************************************************************************/

const AfmFontInfo * docLayoutGetAfi(
			TextAttribute *			ta,
			unsigned char *			pLineFlags,
			BufferDocument *		bd,
			const PostScriptFontList *	psfl,
			int				textAttrNumber )
    {
    DocumentProperties *	dp= &(bd->bdProperties);
    DocumentFontList *		dfl= &(dp->dpFontList);

    const AfmFontInfo *		afi;

    utilGetTextAttributeByNumber( ta, &(bd->bdTextAttributeList),
							    textAttrNumber );

    afi= appGetFontInfoForAttribute( ta, dfl, psfl );
    if  ( ! afi )
	{ LDEB(textAttrNumber); return (AfmFontInfo *)0; }

    if  ( docBorderNumberIsBorder( &(bd->bdBorderPropertyList),
							ta->taBorderNumber ) )
	{ *pLineFlags |= TLflagBORDER;	}

    if  ( docShadingNumberIsShading( &(bd->bdItemShadingList),
							ta->taShadingNumber ) )
	{ *pLineFlags |= TLflagSHADING;	}

    return afi;
    }

/************************************************************************/
/*									*/
/*  Get descent/ascent for the font of a special particule.		*/
/*									*/
/************************************************************************/

int docLayoutFontAscDesc(	BufferDocument *		bd,
				const BufferItem *		paraBi,
				DocumentRectangle *		drAscDesc,
				int *				pBorder,
				const PostScriptFontList *	psfl,
				const TextParticule *		tp )
    {
    int				rval;
    DocumentRectangle		drBBox;

    const AfmFontInfo *		afi;
    TextAttribute		ta;
    BorderProperties		bp;

    int				sizeTwips;
    const int			vswap= 1;

    unsigned char		lineFlags= 0;

    afi= docLayoutGetAfi( &ta, &lineFlags, bd, psfl, tp->tpTextAttrNr );
    if  ( ! afi )
	{ XDEB(afi); return -1;	}

    sizeTwips= 10* ta.taFontSizeHalfPoints;

    rval= psFontBBox( &drBBox, drAscDesc, sizeTwips, vswap, afi );

    if  ( pBorder && ( lineFlags & TLflagBORDER ) )
	{
	int		thick= 0;
	int		space= 0;

	docGetBorderPropertiesByNumber( &bp,
			&(bd->bdBorderPropertyList), ta.taBorderNumber );

	thick= docBorderThick( &space, &bp );

	*pBorder= thick+ space;
	}

    return rval;
    }

/************************************************************************/
/*									*/
/*  Calculate the layout of the next stretch of text upto the point	*/
/*  where it can be folded.						*/
/*									*/
/************************************************************************/

static int docLayoutSegmentedStringExtents(
				DocumentRectangle *	drRet,
				const char *		printString,
				const int *		segments,
				int			segmentCount,
				int			sizeTwips,
				const AfmFontInfo *	afi )
    {
    const char *		s= printString;
    int				seg;
    int				scapsSize= ( 8* sizeTwips )/ 10;

    int				x= 0;
    int				wide;

    DocumentRectangle		dr;
    int				done= 0;

    const int			withKerning= 0;
    const int			vswap= 1;

    for ( seg= 0; seg < segmentCount; seg++ )
	{
	if  ( segments[2* seg+ 0] > 0 )
	    {
	    wide= psCalculateStringExtents( &dr, s, segments[2* seg+ 0],
					sizeTwips, withKerning, vswap, afi );

	    if  ( done )
		{
		drRet->drX1= x+ dr.drX1;

		includeRectangleY( drRet, &dr );
		}
	    else{ *drRet= dr;	}

	    s += segments[2* seg+ 0];
	    x += wide;
	    done= 1;
	    }

	if  ( segments[2* seg+ 1] > 0 )
	    {
	    wide= psCalculateStringExtents( &dr, s, segments[2* seg+ 1],
					scapsSize, withKerning, vswap, afi );

	    if  ( done )
		{
		drRet->drX1= x+ dr.drX1;

		includeRectangleY( drRet, &dr );
		}
	    else{ *drRet= dr;	}

	    s += segments[2* seg+ 1];
	    x += wide;
	    done= 1;
	    }
	}

    return x;
    }

int docLayoutStringExtents(	int *			pWidth,
				int *			pDecWidth,
				DocumentRectangle *	dr,
				const TextAttribute *	ta,
				int			sizeTwips,
				const BufferDocument *	bd,
				const AfmFontInfo *	afi,
				const char *		printString,
				int			len )
    {
    char *			upperString= (char *)0;
    int *			segments= (int *)0;
    int				segmentCount= 0;

    int				rval= 0;
    const int			withKerning= 0;
    const int			vswap= 1;

    int				dec;
    int				d;

    int				width;
    int				fontHigh;
    int				decWidth;

    DocumentRectangle		drFontBBox;
    DocumentRectangle		drFontAscDesc;

    if  ( len > 0 && ( ta->taSmallCaps || ta->taCapitals ) )
	{
	if  ( docMakeCapsString( &upperString, &segments, &segmentCount,
						    ta, printString, len ) )
	    { LDEB(len); return -1;	}

	printString= upperString;
	}

    if  ( len > 0 && ta->taSmallCaps && ! ta->taCapitals )
	{
	width= docLayoutSegmentedStringExtents( dr, printString,
				    segments, segmentCount,
				    sizeTwips, afi );
	if  ( width < 0 )
	    { LLDEB(ta->taFontNumber,width);	}
	}
    else{
	width= psCalculateStringExtents( dr, printString, len,
					sizeTwips, withKerning, vswap, afi );
	if  ( width < 0 )
	    { LLDEB(ta->taFontNumber,width); }
	if  ( width < 0 )
	    { SLDEB(printString,width);	}
	}

    (void)psFontBBox( &drFontBBox, &drFontAscDesc, sizeTwips, vswap, afi );
    dr->drY0= drFontAscDesc.drY0;
    dr->drY1= drFontAscDesc.drY1;

    /*LINEDISTANCE: scale the position of the baseline based on the bbox */
    fontHigh= drFontBBox.drY1- drFontBBox.drY0;
    dr->drY0= ( drFontBBox.drY0* sizeTwips )/ fontHigh;
    dr->drY1= ( drFontBBox.drY1* sizeTwips )/ fontHigh;

    dec= -1;
    for ( d= 0; d < len; d++ )
	{
	if  ( printString[d] == '.' || printString[d] == ','	)
	    { dec= d;	}
	}

    if  ( dec >= 0 )
	{
	DocumentRectangle		drDec;

	decWidth= psCalculateStringExtents( &drDec, printString, dec,
					sizeTwips, withKerning, vswap, afi );
	if  ( decWidth < 0 )
	    { LDEB(decWidth);	}
	}
    else{ decWidth= width;	}

    *pWidth= width;
    *pDecWidth= decWidth;

    if  ( upperString )
	{ free( upperString );	}
    if  ( segments )
	{ free( segments );	}

    return rval;
    }

/************************************************************************/

int docLayoutInlineObject(	const BufferDocument *		bd,
				const TextParticule *		tp,
				InsertedObject *		io,
				DocumentRectangle *		drWord,
				int *				pWordBorder,
				int *				pX1,
				int				x0 )
    {
    const int			accepted= 1;

    TextAttribute		ta;
    BorderProperties		bp;
    int				thick= 0;
    int				space= 0;

    int				width;

    if  ( io->ioKind == DOCokDRAWING_SHAPE )
	{
	DrawingShape *		ds= io->ioDrawingShape;
	const ShapeProperties *	sp= &(ds->dsShapeProperties);

	if  ( sp->spWrapStyle == SHPswsNONE	||
	      sp->spWrapStyle == SHPswsTHROUGH	)
	    {
	    drWord->drX0= x0;
	    drWord->drX1= x0;
	    drWord->drY0= 0;
	    drWord->drY1= 0;

	    *pWordBorder= 0;

	    *pX1= x0;
	    return accepted;
	    }
	}

    width= ( io->ioScaleXUsed* io->ioTwipsWide )/ 100.0;

    utilGetTextAttributeByNumber( &ta, &(bd->bdTextAttributeList),
							    tp->tpTextAttrNr );
    docGetBorderPropertiesByNumber( &bp,
			    &(bd->bdBorderPropertyList), ta.taBorderNumber );

    if  ( DOCisBORDER( &bp ) )
	{ thick= docBorderThick( &space, &bp );	}

    drWord->drX0= x0;
    drWord->drX1= x0+ width;
    drWord->drY0= -( io->ioScaleYUsed* io->ioTwipsHigh )/ 100.0;
    drWord->drY1= 0;

    *pWordBorder= thick+ space;

    *pX1= x0+ width;
    return accepted;
    }

/************************************************************************/
/*									*/
/*  Delimit a string that can be drawn at once.				*/
/*									*/
/*  Delimit a stretch of text that can be drawn using one string	*/
/*  drawing instruction. For smallcaps, the routine is still usefull	*/
/*  but the stretch will not be drawn in one instruction.		*/
/*									*/
/************************************************************************/

int docLayoutDelimitSpan(		int *			pLen,
					int *			pX1Twips,
					int			x0Twips,
					TextAttribute *		ta,
					const BufferDocument *	bd,
					const BufferItem *	paraBi,
					int			part,
					int			past,
					int			separate )
    {
    int				count= past- part;
    int				done;
    int				upto;
    int				len;
    int				x1Twips= x0Twips;

    const TextParticule *	tp= paraBi->biParaParticules+ part;

    x1Twips += tp[0].tpTwipsWide;

    for ( done= 1; done < count; done++ )
	{
	if  ( tp[done].tpTextAttrNr != tp->tpTextAttrNr	||
	      tp[done].tpKind != DOCkindSPAN		)
	    { break;	}

	if  ( separate )
	    { break;	}

	x1Twips += tp[done].tpTwipsWide;
	}

    upto= tp[done-1].tpStroff+ tp[done-1].tpStrlen;
    upto= docParaPastLastNonBlank( paraBi, tp->tpStroff, upto );
    len= upto- tp->tpStroff;

    if  ( ta )
	{
	utilGetTextAttributeByNumber( ta, &(bd->bdTextAttributeList),
						tp->tpTextAttrNr );
	}

    if  ( part+ done < paraBi->biParaParticuleCount- 1		&&
	  tp[done].tpKind == DOCkindOPT_HYPH			&&
	  tp[done].tpTwipsWide > 0				)
	{
	len += tp[done].tpStrlen;
	x1Twips += tp[done].tpTwipsWide;
	done++;
	}

    *pLen= len;
    *pX1Twips= x1Twips;

    return done;
    }

