/************************************************************************/
/*									*/
/*  Geometry calculations about external items.				*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<limits.h>
#   include	<math.h>

#   include	"docLayout.h"
#   include	"docPageGrid.h"
#   include	"docEvalField.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  Determine the box around a header or a footer.			*/
/*									*/
/************************************************************************/

static int docNoteSeparatorBox(	DocumentRectangle *		dr,
				int				noteItKind,
				int				noteSepKind,
				int				page,
				int				column,
				const LayoutContext *		lc )
    {
    DocumentTree *		eiBody;
    DocumentTree *		eiNoteSep;
    int				y0Twips;
    DocumentNote *		dnFirstNote;
    BufferItem *		bodySectBi;
    DocumentField *		dfNote;

    dfNote= docGetFirstNoteInColumn( &dnFirstNote, lc->lcDocument, 
						page, column, noteItKind );

    if  ( ! dfNote )
	{ XDEB(dfNote); return -1;	}

    if  ( docGetRootOfSelectionScope( &eiBody, &bodySectBi, lc->lcDocument,
						&(dfNote->dfSelectionScope) ) )
	{ LDEB(1); return -1;	}

    if  ( docNoteSeparatorRectangle( dr, &eiNoteSep, &y0Twips,
			    bodySectBi, dnFirstNote, noteSepKind, lc ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

int docGetExternalItemBox(	DocumentRectangle *		dr,
				const BufferItem *		bodySectBi,
				const DocumentTree *		ei,
				int				justUsed,
				int				page,
				int				column,
				const LayoutContext *		lc )
    {
    BufferItem *		externSectBi= ei->eiRoot;

    BlockFrame			bf;
    DocumentRectangle		drTwips;
    DocumentRectangle		drBox;

    if  ( ! externSectBi )
	{ XDEB(externSectBi); return -1;	}

    docLayoutInitBlockFrame( &bf );

    switch( externSectBi->biInExternalItem )
	{
	case DOCinFIRST_HEADER:
	case DOCinLEFT_HEADER:
	case DOCinRIGHT_HEADER:
	case DOCinFIRST_FOOTER:
	case DOCinLEFT_FOOTER:
	case DOCinRIGHT_FOOTER:

	    docBlockFrameTwips( &bf, externSectBi, bodySectBi,
						lc->lcDocument, page, column );

	    drTwips= bf.bfContentRect;

	    if  ( justUsed )
		{
		drTwips.drY0= ei->eiY0UsedTwips;
		drTwips.drY1= ei->eiY1UsedTwips;
		}
	    else{
		drTwips.drY0= ei->eiY0ReservedTwips;
		drTwips.drY1= ei->eiY1ReservedTwips;
		}

	    docGetPixelRect( &drBox, lc, &drTwips, page );

	    *dr= drBox; return 0;

	case DOCinFOOTNOTE:
	case DOCinENDNOTE:

	    {
	    DocumentPosition	dpTop;
	    DocumentPosition	dpBot;

	    int			lineTop;
	    int			lineBot;

	    int			partTop;
	    int			partBot;

	    const TextLine *	tlTop;
	    const TextLine *	tlBot;

	    LayoutPosition	lpTop;
	    LayoutPosition	lpBot;

	    docBlockFrameTwips( &bf, ei->eiRoot, bodySectBi, lc->lcDocument, page, column );

	    if  ( docGetFirstInColumnForItem( &dpTop, &lineTop, &partTop,
						ei->eiRoot, page, column ) )
		{ LLDEB(page,column); return -1;	}

	    if  ( docGetLastInColumnForItem( &dpBot, &lineBot, &partBot,
						ei->eiRoot, page, column ) )
		{ LDEB(page); return -1;	}

	    tlTop= dpTop.dpBi->biParaLines+ lineTop;
	    tlBot= dpBot.dpBi->biParaLines+ lineBot;

	    lpTop= tlTop->tlTopPosition;
	    lpBot= tlBot->tlTopPosition;
	    lpBot.lpPageYTwips += tlBot->tlLineStride;

	    docGetPixelRectForPos( &drBox, lc,
			     bf.bfContentRect.drX0, bf.bfContentRect.drX1,
			     &lpTop, &lpBot );
	    }

	    *dr= drBox; return 0;

	case DOCinFTNSEP:
	case DOCinFTNSEPC:
	case DOCinFTNCN:

	    if  ( docNoteSeparatorBox( &drBox,
			    DOCinFOOTNOTE, externSectBi->biInExternalItem,
			    page, column, lc ) )
		{ LLDEB(page,column); return -1;	}

	    *dr= drBox; return 0;

	case DOCinAFTNSEP:
	case DOCinAFTNSEPC:
	case DOCinAFTNCN:

	    if  ( docNoteSeparatorBox( &drBox,
			    DOCinENDNOTE, externSectBi->biInExternalItem,
			    page, column, lc ) )
		{ LLDEB(page,column); return -1;	}

	    *dr= drBox; return 0;

	default:
	    LDEB(externSectBi->biInExternalItem);
	    return -1;
	}
    }

/************************************************************************/
/*									*/
/*  Do a preliminary layout of an external item.			*/
/*  Inside the external item, geometry is correct. Some items are	*/
/*  however used in different positions and here we just calculate the	*/
/*  layout in order to use the size of the item in geometry		*/
/*  calculations about the document as a whole.				*/
/*									*/
/*  1)  Remove the bottom of the page master frame. This is the routine	*/
/*	that calculates it for future use.				*/
/*									*/
/************************************************************************/

int docExternalItemPrelayout(		DocumentTree *		ei,
					const BufferItem *	bodySectBi,
					LayoutJob *		lj )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    const DocumentGeometry *	dgRef= &(bodySectBi->biSectDocumentGeometry);
    LayoutJob			extItLj;
    LayoutPosition		extItLp;

    int				high;
    int				y1;

    BlockFrame			bf;

    docLayoutInitBlockFrame( &bf );

    ei->eiPageFormattedFor= -1;
    ei->eiColumnFormattedFor= -1;

    if  ( ! ei->eiRoot )
	{ return 0;	}

    docDelimitTables( ei->eiRoot );

    ei->eiRoot->biSectDocumentGeometry.dgPageWideTwips=
						dgRef->dgPageWideTwips;
    ei->eiRoot->biSectDocumentGeometry.dgLeftMarginTwips=
						dgRef->dgLeftMarginTwips;
    ei->eiRoot->biSectDocumentGeometry.dgRightMarginTwips=
						dgRef->dgRightMarginTwips;

    extItLp.lpPage= ei->eiRoot->biTopPosition.lpPage;
    extItLp.lpColumn= 0;
    extItLp.lpPageYTwips= dgRef->dgHeaderPositionTwips;
    extItLp.lpAtTopOfColumn= 1; /* not really */

    extItLj= *lj;
    extItLj.ljChangedRectanglePixels= (DocumentRectangle *)0;

    extItLj.ljBodySectBi= bodySectBi;
    extItLj.ljChangedItem= lj->ljChangedItem;

    docBlockFrameTwips( &bf, ei->eiRoot, extItLj.ljBodySectBi, lc->lcDocument,
					extItLp.lpPage, extItLp.lpColumn );
    bf.bfContentRect.drY0= extItLp.lpPageYTwips;
    bf.bfFlowRect.drY0= bf.bfContentRect.drY0;

    /*  1  */
    bf.bfContentRect.drY1= dgRef->dgPageHighTwips;
    bf.bfFlowRect.drY1= bf.bfContentRect.drY1;

    if  ( docLayoutItemImplementation( &extItLp, &extItLp,
						ei->eiRoot, &bf, &extItLj ) )
	{ LDEB(1); return -1;	}

    y1= ei->eiRoot->biBelowPosition.lpPageYTwips;

    switch( ei->eiRoot->biInExternalItem )
	{
	case DOCinFIRST_HEADER:
	case DOCinLEFT_HEADER:
	case DOCinRIGHT_HEADER:

	    ei->eiY0UsedTwips= dgRef->dgHeaderPositionTwips;
	    ei->eiY1UsedTwips= y1;
	    ei->eiY0ReservedTwips= ei->eiY0UsedTwips;
	    if  ( y1 > dgRef->dgTopMarginTwips )
		{ ei->eiY1ReservedTwips= y1;				}
	    else{ ei->eiY1ReservedTwips= dgRef->dgTopMarginTwips;	}
	    break;

	case DOCinFIRST_FOOTER:
	case DOCinLEFT_FOOTER:
	case DOCinRIGHT_FOOTER:

	    high= y1- dgRef->dgHeaderPositionTwips;
	    ei->eiY1UsedTwips= dgRef->dgPageHighTwips-
					    dgRef->dgFooterPositionTwips;
	    ei->eiY0UsedTwips= ei->eiY1UsedTwips- high;

	    ei->eiY1ReservedTwips= ei->eiY1UsedTwips;

	    if  ( ei->eiY0UsedTwips >
			dgRef->dgPageHighTwips- dgRef->dgBottomMarginTwips )
		{
		ei->eiY0ReservedTwips= dgRef->dgPageHighTwips-
						dgRef->dgBottomMarginTwips;
		}
	    else{
		ei->eiY0ReservedTwips= ei->eiY0UsedTwips;
		}
	    break;

	case DOCinFOOTNOTE:
	case DOCinENDNOTE:

	    /*  temporarily: will be placed later on */
	    ei->eiY0UsedTwips= dgRef->dgHeaderPositionTwips;
	    ei->eiY1UsedTwips= y1;
	    break;

	case DOCinFTNSEP:
	case DOCinFTNSEPC:
	case DOCinFTNCN:
	case DOCinAFTNSEP:
	case DOCinAFTNSEPC:
	case DOCinAFTNCN:

	    /*  temporarily */
	    ei->eiY0UsedTwips= dgRef->dgHeaderPositionTwips;
	    ei->eiY1UsedTwips= y1;
	    break;

	default:
	    LDEB(ei->eiRoot->biInExternalItem); return -1;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Do the preliminary layout of the headers and footers of a section	*/
/*									*/
/************************************************************************/

int docSectHeaderFooterPrelayout(	BufferItem *	sectBi,
					LayoutJob *	lj )
    {
    const DocumentGeometry *	dgSect= &(sectBi->biSectDocumentGeometry);


    sectBi->biSectFirstPageHeader.eiY0ReservedTwips=
    sectBi->biSectLeftPageHeader.eiY0ReservedTwips=
    sectBi->biSectRightPageHeader.eiY0ReservedTwips=
					dgSect->dgHeaderPositionTwips;

    sectBi->biSectFirstPageHeader.eiY1ReservedTwips=
    sectBi->biSectLeftPageHeader.eiY1ReservedTwips=
    sectBi->biSectRightPageHeader.eiY1ReservedTwips=
					dgSect->dgTopMarginTwips;

    if  ( docExternalItemPrelayout( &(sectBi->biSectFirstPageHeader),
								sectBi, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docExternalItemPrelayout( &(sectBi->biSectLeftPageHeader),
								sectBi, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docExternalItemPrelayout( &(sectBi->biSectRightPageHeader),
								sectBi, lj ) )
	{ LDEB(1); return -1;	}

    /**/

    sectBi->biSectFirstPageFooter.eiY0ReservedTwips=
    sectBi->biSectLeftPageFooter.eiY0ReservedTwips=
    sectBi->biSectRightPageFooter.eiY0ReservedTwips=
		dgSect->dgPageHighTwips- dgSect->dgBottomMarginTwips;

    sectBi->biSectFirstPageFooter.eiY1ReservedTwips=
    sectBi->biSectLeftPageFooter.eiY1ReservedTwips=
    sectBi->biSectRightPageFooter.eiY1ReservedTwips=
		dgSect->dgPageHighTwips- dgSect->dgFooterPositionTwips;

    if  ( docExternalItemPrelayout( &(sectBi->biSectFirstPageFooter),
								sectBi, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docExternalItemPrelayout( &(sectBi->biSectLeftPageFooter),
								sectBi, lj ) )
	{ LDEB(1); return -1;	}

    if  ( docExternalItemPrelayout( &(sectBi->biSectRightPageFooter),
								sectBi, lj ) )
	{ LDEB(1); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Reset page dependent layout administration of headers/footers etc	*/
/*  fo force a subsequent recalculation.				*/
/*									*/
/************************************************************************/

void docResetExternalItemLayout(	BufferDocument *	bd )
    {
    int		i;

    bd->bdEiFtnsep.eiPageFormattedFor= -1;
    bd->bdEiFtnsep.eiColumnFormattedFor= -1;
    bd->bdEiFtnsepc.eiPageFormattedFor= -1;
    bd->bdEiFtnsepc.eiColumnFormattedFor= -1;
    bd->bdEiFtncn.eiPageFormattedFor= -1;
    bd->bdEiFtncn.eiColumnFormattedFor= -1;

    bd->bdEiAftnsep.eiPageFormattedFor= -1;
    bd->bdEiAftnsep.eiColumnFormattedFor= -1;
    bd->bdEiAftnsepc.eiPageFormattedFor= -1;
    bd->bdEiAftnsepc.eiColumnFormattedFor= -1;
    bd->bdEiAftncn.eiPageFormattedFor= -1;
    bd->bdEiAftncn.eiColumnFormattedFor= -1;

    for ( i= 0; i < bd->bdBody.eiRoot->biChildCount; i++ )
	{
	BufferItem *	sectBi=  bd->bdBody.eiRoot->biChildren[i];

	sectBi->biSectFirstPageHeader.eiPageFormattedFor= -1;
	sectBi->biSectFirstPageHeader.eiColumnFormattedFor= -1;
	sectBi->biSectLeftPageHeader.eiPageFormattedFor= -1;
	sectBi->biSectLeftPageHeader.eiColumnFormattedFor= -1;
	sectBi->biSectRightPageHeader.eiPageFormattedFor= -1;
	sectBi->biSectRightPageHeader.eiColumnFormattedFor= -1;

	sectBi->biSectFirstPageFooter.eiPageFormattedFor= -1;
	sectBi->biSectFirstPageFooter.eiColumnFormattedFor= -1;
	sectBi->biSectLeftPageFooter.eiPageFormattedFor= -1;
	sectBi->biSectLeftPageFooter.eiColumnFormattedFor= -1;
	sectBi->biSectRightPageFooter.eiPageFormattedFor= -1;
	sectBi->biSectRightPageFooter.eiColumnFormattedFor= -1;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Verify that the root of a selection is formatted for the current	*/
/*  page, if not format it.						*/
/*									*/
/************************************************************************/

static int docGetY0ForSelectedNoteSeparator(
					int *			pY0Twips,
					BufferItem **		pBodySectBi,
					const LayoutContext *	lc,
					const DocumentTree *	selRootEi,
					int			noteItKind,
					int			sepItKind )
    {
    DocumentNote *		dnFirstNote;
    DocumentTree *		eiBody;
    DocumentTree *		eiNoteSep;
    int				y0Twips;

    DocumentRectangle		drExtern;

    DocumentField *		dfNote;
    BufferItem *		bodySectBi;

    dfNote= docGetFirstNoteInColumn( &dnFirstNote, lc->lcDocument, 
					    selRootEi->eiPageSelectedUpon,
					    selRootEi->eiColumnSelectedIn,
					    noteItKind );
    if  ( ! dfNote )
	{
	LLDEB(selRootEi->eiPageSelectedUpon,selRootEi->eiColumnSelectedIn);
	return -1;
	}

    if  ( docGetRootOfSelectionScope( &eiBody, &bodySectBi, lc->lcDocument,
						&(dfNote->dfSelectionScope) ) )
	{ LDEB(1); return -1;	}

    if  ( docNoteSeparatorRectangle( &drExtern, &eiNoteSep,
		&y0Twips, bodySectBi, dnFirstNote, sepItKind, lc ) )
	{ LDEB(1); return -1;	}

    *pY0Twips= y0Twips;
    *pBodySectBi= bodySectBi;
    return 0;
    }

int docCheckPageOfSelectedExtItem(
				int *			pChanged,
				BufferItem **		pBodySectBi,
				DocumentRectangle *	drChanged,
				DocumentTree *		selRootEi,
				const LayoutContext *	lc,
				INIT_LAYOUT_EXTERNAL	initLayoutExternal )
    {
    int				y0Twips;
    BufferItem *		selRootBodySectBi= (BufferItem *)0;
    const SelectionScope *	selRootScope;

    if  ( ! selRootEi->eiRoot )
	{ XDEB(selRootEi->eiRoot); return -1;	}

    selRootScope= &(selRootEi->eiRoot->biSectSelectionScope);

    switch( selRootEi->eiRoot->biInExternalItem )
	{
	case DOCinBODY:
	    LDEB(selRootEi->eiRoot->biInExternalItem);
	    return -1;

	case DOCinFIRST_HEADER:
	case DOCinLEFT_HEADER:
	case DOCinRIGHT_HEADER:

	case DOCinFIRST_FOOTER:
	case DOCinLEFT_FOOTER:
	case DOCinRIGHT_FOOTER:

	    selRootBodySectBi= docGetBodySectBiOfScope( selRootScope, lc->lcDocument );

	    y0Twips= selRootEi->eiY0UsedTwips;
	    break;

	case DOCinFOOTNOTE:
	case DOCinENDNOTE:

	    selRootBodySectBi= docGetBodySectBiOfScope( selRootScope, lc->lcDocument );

	    *pBodySectBi= selRootBodySectBi;
	    *pChanged= 0;
	    return 0;

	case DOCinFTNSEP:

	    if  ( selRootEi->eiPageSelectedUpon < 0 )
		{ LDEB(selRootEi->eiPageSelectedUpon); return -1;	}

	    if  ( docGetY0ForSelectedNoteSeparator( &y0Twips,
						&selRootBodySectBi,
						lc, selRootEi,
						DOCinFOOTNOTE, DOCinFTNSEP ) )
		{ LDEB(1); return -1;	}

	    break;

	case DOCinFTNSEPC:
	case DOCinFTNCN:
	    LDEB(selRootEi->eiRoot->biInExternalItem);
	    return -1;

	case DOCinAFTNSEP:

	    if  ( selRootEi->eiPageSelectedUpon < 0 )
		{ LDEB(selRootEi->eiPageSelectedUpon); return -1;	}

	    if  ( docGetY0ForSelectedNoteSeparator( &y0Twips,
						&selRootBodySectBi,
						lc, selRootEi,
						DOCinENDNOTE, DOCinAFTNSEP ) )
		{ LDEB(1); return -1;	}

	    break;

	case DOCinAFTNSEPC:
	case DOCinAFTNCN:
	    LDEB(selRootEi->eiRoot->biInExternalItem);
	    return -1;

	default:
	    LDEB(selRootEi->eiRoot->biInExternalItem);
	    return -1;
	}

    if  ( selRootEi->eiPageSelectedUpon < 0 )
	{ LDEB(selRootEi->eiPageSelectedUpon); return -1;	}

    if  ( selRootEi->eiPageFormattedFor == selRootEi->eiPageSelectedUpon &&
	  selRootEi->eiColumnFormattedFor == selRootEi->eiColumnSelectedIn )
	{
	*pBodySectBi= selRootBodySectBi;
	*pChanged= 0;
	return 0;
	}

    if  ( docLayoutExternalItem( selRootEi, drChanged,
				    selRootEi->eiPageSelectedUpon,
				    selRootEi->eiColumnSelectedIn,
				    y0Twips, selRootBodySectBi, lc,
				    initLayoutExternal ) )
	{ LDEB(selRootEi->eiPageSelectedUpon); return -1; }

    *pBodySectBi= selRootBodySectBi;
    *pChanged= 1; return 0;
    }

/************************************************************************/
/*									*/
/*  Calculate the layout of a page header.				*/
/*									*/
/************************************************************************/

int docLayoutExternalItem(	DocumentTree *		ei,
				DocumentRectangle *	drChanged,
				int			page,
				int			column,
				int			y0Twips,
				const BufferItem *	bodySectBi,
				const LayoutContext *	lc,
				INIT_LAYOUT_EXTERNAL	initLayoutExternal )
    {
    int		rval= 0;

    if  ( page != ei->eiPageFormattedFor	||
	  column != ei->eiColumnFormattedFor	)
	{
	RecalculateFields		rf;
	LayoutJob			lj;
	BlockFrame			bf;

	LayoutPosition			lpHere;

	docLayoutInitBlockFrame( &bf );
	docInitRecalculateFields( &rf );
	docInitLayoutJob( &lj );

	rf.rfBd= lc->lcDocument;
	rf.rfCloseObject= lc->lcCloseObject;
	rf.rfUpdateFlags=
		FIELDdoDOC_FORMATTED|FIELDdoDOC_COMPLETE|FIELDdoPAGE_NUMBER;
	rf.rfFieldsUpdated= 0;


	if  ( docRecalculateTextLevelFieldsInExternalItem( &rf, ei,
							    bodySectBi, page ) )
	    { LDEB(page); return -1;	}

	lpHere.lpPage= page;
	lpHere.lpColumn= column;
	lpHere.lpPageYTwips= y0Twips;
	lpHere.lpAtTopOfColumn= 1; /* not really */

	lj.ljBodySectBi= bodySectBi;

	lj.ljChangedRectanglePixels= drChanged;
	lj.ljContext= *lc;
	lj.ljChangedItem= lc->lcDocument->bdBody.eiRoot;

	if  ( initLayoutExternal 				&&
	      (*initLayoutExternal)( &lj, ei, page, column )	)
	    { LDEB(1); rval= -1; goto ready;	}

	if  ( docLayoutGetInitialFrame( &bf, &lj, &lpHere, ei->eiRoot ) )
	    { LDEB(1); return -1;	}

	if  ( docLayoutItemImplementation( &lpHere, &lpHere,
						    ei->eiRoot, &bf, &lj ) )
	    { LDEB(1); rval= -1; goto ready; }

	ei->eiPageFormattedFor= page;
	ei->eiColumnFormattedFor= column;
	ei->eiY0UsedTwips= y0Twips;
	ei->eiY1UsedTwips= lpHere.lpPageYTwips;

      ready:
	docCleanLayoutJob( &lj );
	}

    return rval;
    }

