/************************************************************************/
/*									*/
/*  Layout of a document.						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>
#   include	<stdlib.h>
#   include	<limits.h>

#   include	<appImage.h>
#   include	"docLayout.h"
#   include	"docPixels.h"

#   include	<appDebugon.h>

/************************************************************************/
/*									*/
/*  layout a paragraph or a cell: Both are shortcuts to			*/
/*  docLayoutParagraphs(): trkeep and widow/orphan control makes the	*/
/*  formatting of paragraphs dependent on eachother.			*/
/*									*/
/************************************************************************/

static int docLayoutParaItem(	LayoutPosition *	lpBelow,
				const LayoutPosition *	lpTop,
				BufferItem *		paraBi,
				BlockFrame *		bf,
				LayoutJob *		lj )
    {
    int				rval= 0;
    ParagraphLayoutJob		plj;
    const int			line= 0;
    const int			part= 0;

    docInitParagraphLayoutJob( &plj );

    docBeginParagraphLayoutProgress( &plj,
			paraBi->biNumberInParent, line, part,
			paraBi->biNumberInParent+ 1,
			lpTop );

    if  ( docLayoutParagraphs( paraBi->biParent, bf, lj, &plj ) )
	{ LDEB(1); rval= -1; goto ready;	}

    if  ( paraBi->biParaLineCount < 1 )
	{
	LDEB(paraBi->biParaLineCount); docListItem(0,paraBi,0);
	rval= -1; goto ready;
	}

    *lpBelow= plj.pljPos.plpPos;

  ready:

    docCleanParagraphLayoutJob( &plj );

    return rval;
    }

int docLayoutCellItem(	LayoutPosition *	lpBelow,
			const LayoutPosition *	lpTop,
			BufferItem *		cellBi,
			int			from,
			BlockFrame *		bf,
			LayoutJob *		lj )
    {
    int				rval= 0;
    ParagraphLayoutJob		plj;
    const int			line= 0;
    const int			part= 0;

    docInitParagraphLayoutJob( &plj );

    docBeginParagraphLayoutProgress( &plj,
				from, line, part, cellBi->biChildCount,
				lpTop );

    if  ( docLayoutParagraphs( cellBi, bf, lj, &plj ) )
	{ LDEB(1); rval= -1; goto ready;	}

    *lpBelow= plj.pljPos.plpPos;

  ready:
    docCleanParagraphLayoutJob( &plj );

    return rval;
    }

/************************************************************************/
/*									*/
/*  Redo the layout of a section because one of its external items	*/
/*  changed size.							*/
/*									*/
/************************************************************************/

static void docItemLayoutStartPosition(	LayoutPosition *	lp,
					const BufferItem *	bi )
    {
    if  ( bi->biNumberInParent == 0 )
	{
	if  ( bi->biParent )
	    { *lp= bi->biParent->biTopPosition;	}
	else{
	    if  ( bi->biInExternalItem == DOCinBODY )
		{ docInitLayoutPosition( lp );		}
	    else{ *lp= bi->biTopPosition;		}
	    }
	}
    else{
	const BufferItem *	prevBi;

	prevBi= bi->biParent->biChildren[bi->biNumberInParent- 1];

	*lp= prevBi->biBelowPosition;
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Do the layout of the document root item.				*/
/*									*/
/************************************************************************/

static int docLayoutDocItem(		LayoutPosition *	lpBelow,
					const LayoutPosition *	lpTop,
					BufferItem *		docBi,
					BlockFrame *		bf,
					LayoutJob *		lj )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    BufferDocument *		bd= lc->lcDocument;
    const DocumentProperties *	dp= &(bd->bdProperties);
    const NotesProperties *	npEndnotes= &(dp->dpEndnoteProperties);

    LayoutPosition		lpHere= *lpTop;

    int				i;

    if  ( docBi->biChildCount > 0 )
	{
	lj->ljBodySectBi= docBi->biChildren[0];

	if  ( docExternalItemPrelayout( &(bd->bdEiFtnsep),
						    lj->ljBodySectBi, lj ) )
	    { LDEB(1); return -1;	}
	if  ( docExternalItemPrelayout( &(bd->bdEiFtnsepc),
						    lj->ljBodySectBi, lj ) )
	    { LDEB(1); return -1;	}
	if  ( docExternalItemPrelayout( &(bd->bdEiFtncn),
						    lj->ljBodySectBi, lj ) )
	    { LDEB(1); return -1;	}

	if  ( docExternalItemPrelayout( &(bd->bdEiAftnsep),
						    lj->ljBodySectBi, lj ) )
	    { LDEB(1); return -1;	}
	if  ( docExternalItemPrelayout( &(bd->bdEiAftnsepc),
						    lj->ljBodySectBi, lj ) )
	    { LDEB(1); return -1;	}
	if  ( docExternalItemPrelayout( &(bd->bdEiAftncn),
						    lj->ljBodySectBi, lj ) )
	    { LDEB(1); return -1;	}
	}

    for ( i= 0; i < docBi->biChildCount; i++ )
	{
	lj->ljBodySectBi= docBi->biChildren[i];

	if  ( docLayoutItemImplementation( &lpHere, &lpHere,
						docBi->biChildren[i], bf, lj ) )
	    { LDEB(1); return -1;	}
	}

    if  ( docBi->biInExternalItem == DOCinBODY		&&
	  npEndnotes->npPosition == FTN_POS_DOC_END	)
	{
	if  ( docLayoutEndnotesForDocument( &lpHere, &lpHere, bf, lj ) )
	    { LDEB(1); return -1;	}
	}

    *lpBelow= lpHere;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Start/Finish the layout of items: Is done around the item specific	*/
/*  layout procedures.							*/
/*									*/
/************************************************************************/

void docLayoutStartItemLayout(	BufferItem *		bi,
				const LayoutJob *	lj,
				const LayoutPosition *	lpHere )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;
    int				y0;
    int				y1;

    y0= BI_TOP_PIXELS( lc->lcAdd, bi );
    y1= BI_BELOW_PIXELS( lc->lcAdd, bi )- 1;

    if  ( drChanged				&&
	  drChanged->drY0 > y0	)
	{ drChanged->drY0=  y0;	}

    bi->biTopPosition= *lpHere;

    if  ( drChanged )
	{
	y0= BI_TOP_PIXELS( lc->lcAdd, bi );

	if  ( drChanged->drY0 > y0 )
	    { drChanged->drY0=  y0;			}
	if  ( drChanged->drY1 < y1 )
	    { drChanged->drY1=  y1;	}
	}

    return;
    }

void docLayoutFinishItemLayout(	BufferItem *		bi,
				const LayoutJob *	lj,
				const LayoutPosition *	lpHere )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    DocumentRectangle *		drChanged= lj->ljChangedRectanglePixels;
    int				y1;

    bi->biBelowPosition= *lpHere;

    y1= BI_BELOW_PIXELS( lc->lcAdd, bi )- 1;

    if  ( drChanged		&&
	  drChanged->drY1 < y1	)
	{ drChanged->drY1=  y1;	}

    return;
    }

/************************************************************************/
/*									*/
/*  Do the layout of a document item.					*/
/*									*/
/*  This is the main entry poin of the formatter.			*/
/*									*/
/*  1)  While balancing section columns.. Do not reformat items that	*/
/*	are irrelevant.							*/
/*									*/
/************************************************************************/

int docLayoutItemImplementation(	LayoutPosition *	lpBelow,
					const LayoutPosition *	lpTop,
					BufferItem *		bi,
					BlockFrame *		bf,
					LayoutJob *		lj )
    {
    const LayoutContext *	lc= &(lj->ljContext);
    int				i;

    LayoutPosition		lpHere= *lpTop;

    /*  1  */
    if  ( lj->ljBalancePage >= 0				&&
	  bi->biBelowPosition.lpPage < lj->ljBalancePage	)
	{ *lpBelow= bi->biBelowPosition; return 0;	}

    docLayoutStartItemLayout( bi, lj, &lpHere );

    switch( bi->biLevel )
	{
	case DOClevBODY:

	    if  ( docLayoutDocItem( &lpHere, &lpHere, bi, bf, lj ) )
		{ LDEB(1); return -1;	}
	    break;

	rowAsGroup:
	sectAsGroup:
	    for ( i= 0; i < bi->biChildCount; i++ )
		{
		if  ( docLayoutItemImplementation( &lpHere, &lpHere,
						bi->biChildren[i], bf, lj ) )
		    { LDEB(1); return -1;	}
		}
	    if  ( bi->biChildCount > 0 )
		{ bi->biTopPosition= bi->biChildren[0]->biTopPosition;	}
	    break;

	case DOClevCELL:
	    if  ( docLayoutCellItem( &lpHere, &lpHere, bi, 0, bf, lj ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevSECT:
	    if  ( ! bi->biParent )
		{ goto sectAsGroup;	}

	    if  ( docLayoutSectItem( &lpHere, &lpHere, bi, bf, lj ) )
		{ LDEB(1); return -1;	}
	    break;

	case DOClevROW:
	    {
	    int		stopCode= FORMATstopREADY;
	    int		stayInThisColumn= 0;

	    if  ( ! docIsRowItem( bi ) )
		{
		if  ( bi->biNumberInParent > 0 )
		    { docLayoutCalculateAfterRowTopInset( bi, lc->lcDocument ); }

		goto rowAsGroup;
		}

	    if  ( docLayoutRowItem( &stopCode, &lpHere, &lpHere,
					    bi, bf, stayInThisColumn, lj ) )
		{ LDEB(1); return -1;	}
	    }
	    break;

	case DOClevPARA:
	    if  ( docLayoutParaItem( &lpHere, &lpHere, bi, bf, lj ) )
		{ LDEB(1); return -1;	}

	    break;

	default:
	    LDEB(bi->biLevel); return -1;
	}

    docLayoutFinishItemLayout( bi, lj, &lpHere );

    *lpBelow= lpHere;
    return 0;
    }

/************************************************************************/
/*									*/
/*  Recalculate the layout for a block in the document hierarchy.	*/
/*									*/
/*  1)  If we do not have to recalculate the layout of the document as	*/
/*	a whole..							*/
/*  2)  Calculate the frame in which the text is to be laid out.	*/
/*  3)  If the preceding paragraph ends on the same page where this	*/
/*	nodes begins, reserve space for the footnotes upto the		*/
/*	beginning of this block and subtract the height from the buttom	*/
/*	of the frame.							*/
/*  4)  Perform the actual layout operation.				*/
/*  5)  Adjust the positions of the parent nodes and any children of	*/
/*	the parent below this node.					*/
/*									*/
/************************************************************************/

int docLayoutItemAndParents(	BufferItem *		bi,
				LayoutJob *		lj )
    {
    BlockFrame			bf;
    LayoutPosition		lpHere;
    const LayoutContext *	lc= &(lj->ljContext);
    const BufferItem *		bodyBi= lc->lcDocument->bdBody.eiRoot;
    const BufferItem *		sectBi= docGetSectItem( bi );

    docItemLayoutStartPosition( &lpHere, bi );

    if  ( bi->biInExternalItem == DOCinBODY )
	{
	if  ( ! sectBi )
	    { sectBi= bodyBi->biChildren[0];			}

	lj->ljBodySectBi= sectBi;
	}
    else{
	const BufferItem *		bodySectBi;

	if  ( ! sectBi )
	    { XDEB(sectBi); bodySectBi= bodyBi->biChildren[0];	}
	else{
	    const SelectionScope *	ss= &(sectBi->biSectSelectionScope);

	    bodySectBi= bodyBi->biChildren[ss->ssOwnerSectNr];
	    }

	lj->ljBodySectBi= bodySectBi;
	}

    docLayoutInitBlockFrame( &bf );

    /*  1  */
    if  ( bi->biLevel != DOClevBODY				&&
	  docLayoutGetInitialFrame( &bf, lj, &lpHere, bi )	)
	{ LDEB(1); return -1;	}

    /*  4  */
    if  ( docLayoutItemImplementation( &lpHere, &lpHere, bi, &bf, lj ) )
	{ LDEB(1); return -1;	}

    /*  5  */
    if  ( docAdjustParentLayout( &lpHere, bi, &bf, lj ) )
	{ LDEB(1); return -1;	}

    return 0;
    }
