%{
(* SwiftSurf
 * Sebastien Ailleret *)

open Unix

open Types

(* log file *)
let logfile = ref ""

(* port on which the server is listening *)
let port = ref 8080

(* from where can we connect *)
let connect_from = ref None

(* Is there a following proxy *)
let proxy = ref None
let proxy_auth = ref None
let noProxy = ref []

(* what are the headers we don't want *)
let forbidden_req_headers = ref ["*roxy*"]
let forbidden_ans_headers = ref []
let add_req_headers = ref ""
let add_ans_headers = ref ""

(* list of the different profiles *)
let prof_list = ref []
let default_prof = {
  auth_pass = "";
  forbidden_urls = [];
  allowed_urls = [], "";
  methods = [];
  canCONNECT = true;
  allCONNECT = false;
  req_1 = false;
  req_2 = false;
  req_in = false;
  req_out = false;
  ans_1 = false;
  ans_2 = false;
  ans_in = false;
  ans_out = false }

(* list of known mime types *)
let mime_types = ref
    [ ".html", "text/html";
      ".htm", "text/html";
      ".pdf", "application/pdf";
      ".ps", "application/postscript";
      ".wav", "audio/x-wav";
      ".jpg", "image/jpeg";
      ".jpeg", "image/jpeg";
      ".gif", "image/gif";
      ".png", "image/png";
      ".txt", "test/plain" ]

(* get mime type *)
let get_mime file =
  let rec aux = function
    | [] -> ""
    | (ext, mt)::l ->
        if Filename.check_suffix file ext
        then Printf.sprintf "Content-Type : %s\r\n" mt
        else aux l in
  aux !mime_types

(* find the location of a file *)
let real_file f =
  if Sys.file_exists f
  then f
  else Filename.concat !Types.conf_file_dir f

(* make the http ans from a file name *)
let make_ans = function
  | "default" | "green" ->
      make_ans "200 OK" "image/gif" Colors.green
  | "transparent" -> make_ans "200 OK" "image/gif" Colors.transparent
  | "white" -> make_ans "200 OK" "image/gif" Colors.white
  | "black" -> make_ans "200 OK" "image/gif" Colors.black
  | "gray" -> make_ans "200 OK" "image/gif" Colors.gray
  | "orange" -> make_ans "200 OK" "image/gif" Colors.orange
  | "blue" -> make_ans "200 OK" "image/gif" Colors.blue
  | "red" -> make_ans "200 OK" "image/gif" Colors.red
  | "yellow" -> make_ans "200 OK" "image/gif" Colors.yellow
  | file ->
      let content = Utils.read_file (real_file file) in
      Printf.sprintf
        "HTTP/1.0 200 OK\r\nServer: swiftsurf\r\nAccept-Ranges: bytes\r\nConnection: close\r\n%sContent-Length: %d\r\n\r\n%s"
        (get_mime file)
        (String.length content)
        content
%}

%token LOGFILE
%token PORT
%token CONNECTFROM
%token PROFILE
%token ENDPROFILE
%token TIMEOUT
%token STATS
%token DEBUG
%token METHODS
%token NOCONNECT
%token ALLCONNECT
%token REQUEST
%token ANSWER
%token MAXCONNS
%token NEXTPROXY
%token NOPROXY
%token FORBIDDENURL
%token FORBIDDENFILE
%token ALLOWEDURL
%token FORBIDDENREQHEADER
%token FORBIDDENANSHEADER
%token ADDREQHEADER
%token ADDANSHEADER
%token MIMETYPE
%token USERAGENTOVERRIDE
%token END

%token <string> STRING
%token <float> FLOAT
%token <int> INT

%token EOF

%start conf_file
%start string_list
%type < int * Str.regexp option * Unix.sockaddr option * string option * string list * string list * string list * string * string * Types.profile list * string ref > conf_file
%type < string list > string_list

%%

conf_file:
| def_list
    { let pl =
        if !prof_list = []
        then [default_prof]
        else !prof_list in
      !port, !connect_from, !proxy, !proxy_auth, !noProxy,
      !forbidden_req_headers, !forbidden_ans_headers,
      !add_req_headers, !add_ans_headers, pl, logfile }

def_list:
| /* nothing */
    { () }
| def def_list
    { () }

def:
| LOGFILE STRING
    { logfile := $2 }
| PORT INT
    { port := $2 }
| CONNECTFROM STRING
    { connect_from := Some (Str.regexp $2) }
| PROFILE STRING prof_def_list ENDPROFILE
    {
     let prof = {
       auth_pass = $2;
       forbidden_urls = [];
       allowed_urls = [], "";
       methods = [];
       canCONNECT = true;
       allCONNECT = false;
       req_1 = false;
       req_2 = false;
       req_in = false;
       req_out = false;
       ans_1 = false;
       ans_2 = false;
       ans_in = false;
       ans_out = false } in
     List.iter (fun x -> x prof) $3;
     prof_list := prof :: !prof_list }
| TIMEOUT INT
    { Types.timeout := float_of_int $2;
      if !Types.timeout=0. then Types.timeout := 99999. }
| TIMEOUT FLOAT
    { Types.timeout := $2;
      if !Types.timeout=0. then Types.timeout := 99999. }
| STATS INT
    { Types.stats := $2 }
| DEBUG INT
    { Types.debug := $2 }
| MAXCONNS INT
    { Types.max_conns := $2 }
| NEXTPROXY STRING INT auth
    {
     let p_host = $2 in
     let p_port = $3 in
     let entry = gethostbyname p_host in
     proxy := Some (ADDR_INET (entry.h_addr_list.(0), p_port));
     proxy_auth := $4; }
| NOPROXY string_list
    { noProxy := $2 }
| FORBIDDENREQHEADER string_list
    { forbidden_req_headers := !forbidden_req_headers @ $2 }
| FORBIDDENANSHEADER string_list
    { forbidden_ans_headers := !forbidden_ans_headers @ $2 }
| ADDREQHEADER STRING
    { add_req_headers :=
        Printf.sprintf "%s%s\r\n"
          !add_req_headers $2 }
| ADDANSHEADER STRING
    { add_ans_headers :=
        Printf.sprintf "%s%s\r\n"
          !add_ans_headers $2 }
| USERAGENTOVERRIDE STRING
    { forbidden_req_headers := "user-agent: *" :: !forbidden_req_headers;
      add_req_headers :=
        Printf.sprintf "%sUser-Agent: %s\r\n"
          !add_req_headers $2 }
| MIMETYPE STRING STRING
    { mime_types := ($2, $3) :: !mime_types }
| prof_def
    { $1 default_prof }

auth:
|  /* nothing */
    { None }
| STRING
    { Some (Printf.sprintf "proxy-authorization: Basic %s\r\n"
              (Utils.encode64 $1)) }

string_list:
| /* nothing */
    { [] }
| END
    { [] }
| STRING string_list
    { $1 :: $2 }

string_list_end:
| END
    { [] }
| STRING string_list
    { $1 :: $2 }

prof_def_list:
| /* nothing */
    { [] }
| prof_def prof_def_list
    { $1 :: $2 }

prof_def:
| METHODS string_list_end
    { fun x -> x.methods <- $2 }
| NOCONNECT
    { fun x -> x.canCONNECT <- false }
| ALLCONNECT
    { fun x -> x.allCONNECT <- true }
| REQUEST INT
    { let req = $2 in
    (fun x ->
      x.req_1 <- ((req land 1) = 1);
      x.req_2 <- ((req land 2) = 2);
      x.req_in <- ((req land 4) = 4);
      x.req_out <- ((req land 8) = 8)) }
| ANSWER INT
    { let ans = $2 in
    (fun x ->
      x.ans_1 <- ((ans land 1) = 1);
      x.ans_2 <- ((ans land 2) = 2);
      x.ans_in <- ((ans land 4) = 4);
      x.ans_out <- ((ans land 8) = 8)) }
| FORBIDDENFILE STRING repl
    { let file = real_file $2 in
      fun x ->
        x.forbidden_urls <- (Utils.read_string_list file, make_ans $3)
          :: x.forbidden_urls }
| FORBIDDENURL string_list_end repl
    { fun x -> x.forbidden_urls <- ($2, make_ans $3) :: x.forbidden_urls }
| ALLOWEDURL string_list_end repl
    { fun x -> x.allowed_urls <- ($2, make_ans $3) }

repl:
| /* nothing */
    { "default" }
| STRING
    { $1 }
