(* SwiftSurf
 * Sebastien Ailleret *)

open Unix
open Pervasives

open Types
open Utils

exception Auth_failed

(* port on which the server is listening *)
let port = ref 8080

(* from where can we connect *)
let connect_from = ref None

(* Is there a following proxy *)
let proxy = ref None
let noProxy = ref []

(* what are the headers we don't want (and which do we want to add) *)
let forbidden_req_headers = ref []
let add_req_headers = ref ""
let forbidden_ans_headers = ref []
let add_ans_headers = ref ""

(* list of the different profiles *)
let prof_list = ref []

(*******************************)
(* read the configuration file *)
let init conf_file daemon =
  (* read the conf file *)
  let conf_file =
    if conf_file <> ""
    then conf_file
    else
      (let home_file = Filename.concat (Sys.getenv "HOME") ".swiftsurf" in
      if Sys.file_exists home_file
      then home_file
      else if Sys.file_exists "/etc/swiftsurf.conf"
      then "/etc/swiftsurf.conf"
      else if Sys.file_exists "swiftsurf.conf"
      then "swiftsurf.conf"
      else "/dev/null") in
  Types.conf_file_dir := Filename.dirname conf_file;
  (* hack for using the parser in the parser *)
  Utils.pars := Obj.magic Confpars.string_list;
  Utils.lex := Obj.magic Conflex.token;
  (* Open file *)
  let chan_in = open_in conf_file in
  let lexbuf = Lexing.from_channel chan_in in
  let p, cf, prox, pa, nop, frh, fah, arh, aah, pl, lf =
    try
      Confpars.conf_file Conflex.token lexbuf
    with e ->
      Printf.fprintf stderr
        "Erreur : %d--%d, %s\n"
        (Lexing.lexeme_start lexbuf)
        (Lexing.lexeme_end lexbuf)
        (Printexc.to_string e);
      flush stderr;
      exit 1 in
  close_in chan_in;
  port := p;
  connect_from := cf;
  proxy :=
    (match prox with
    | None -> None
    | Some p -> Some (p, pa));
  noProxy := nop;
  forbidden_req_headers := frh;
  forbidden_ans_headers := fah;
  add_req_headers := arh;
  add_ans_headers := aah;
  prof_list := pl;
  (* redirect output and fork if necessary *)
  if (daemon && !lf = "")
  then
    (lf := "swiftsurf.log";
     Printf.printf "writing log in swiftsurf.log");
  if (!lf <> "")
  then
    (if (fork ()) <> 0 then exit 1;
     let fds = Unix.openfile (!lf)
         [Unix.O_WRONLY; Unix.O_TRUNC; O_CREAT] 420 in
     Unix.dup2 fds Unix.stdout;
     Unix.close fds)

(*****************************************)
(* functions for managing configurations *)

(* give the profile for this authentication *)
let get_profile auth =
  let rec aux = function
    | [] -> raise Auth_failed
    | a::l ->
        if a.auth_pass = auth
        then a
        else aux l in
  aux !prof_list

(* Does this site need a proxy *)
let need_proxy site =
  let rec need_aux = function
    | [] ->
        !proxy
    | pat::l ->
        if match_pattern site pat
        then None
        else need_aux l
  in
  need_aux !noProxy

(* see if a string matches a pattern in the list *)
let rec match_list str = function
  | [] -> false
  | pat::l ->
      match_pattern str pat || match_list str l

(* test if an url is allowed or not *)
exception Reject of string

let ok_url url prof =
  try
    let verif_forb () =
      List.iter
        (fun (l, s) ->
          if match_list url l
          then raise (Reject s))
        prof.forbidden_urls;
      None in
    let allw, allw_ans = prof.allowed_urls in
    if allw = []
    then verif_forb ()
    else
      if not (match_list url allw)
      then Some allw_ans
      else verif_forb ()
  with Reject reason -> Some reason

(* test if this line of header is allowed *)
let ok_req_header header =
  not (match_list header !forbidden_req_headers)

let ok_ans_header header =
  not (match_list header !forbidden_ans_headers)
