/*
 * $Id: test-transfer.c,v 1.1.2.2 2004/10/14 21:29:22 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <glib.h>
#include "st-transfer.h"
#include "st-transfer-api.h"

typedef enum
{
  MODE_NORMAL,
  MODE_BINARY,
  MODE_LINE
} Mode;

static void
line_cb (const char *line, gpointer data)
{
  g_print("line: %s\n", line);
}

int
main (int argc, char **argv)
{
  int c;
  gboolean no_newline = FALSE;
  char *mode_str = NULL;
  const char *url = NULL;
  Mode mode;
  STTransferSession *session;
  GError *err = NULL;

  g_log_set_fatal_mask(G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL);

  while ((c = getopt(argc, argv, "m:n")) != -1)
    switch (c)
      {
      case 'm':
	mode_str = g_strdup(optarg);
	break;
      case 'n':
	no_newline = TRUE;
	break;
      case '?':
	exit(1);
      default:
	g_assert_not_reached();
      }

  argc -= optind;
  argv += optind;

  if (argc == 1)
    url = argv[0];
  else
    g_critical("wrong number of arguments");

  if (mode_str)
    {
      if (! strcmp(mode_str, "normal"))
	mode = MODE_NORMAL;
      else if (! strcmp(mode_str, "binary"))
	mode = MODE_BINARY;
      else if (! strcmp(mode_str, "line"))
	mode = MODE_LINE;
      else
	g_critical("unknown mode \"%s\"", mode_str);
    }
  else
    g_critical("-m argument is mandatory");

  st_transfer_init();

  session = st_transfer_session_new();
  switch (mode)
    {
    case MODE_NORMAL:
      {
	char *body;

	if (st_transfer_session_get(session,
				    url,
				    0,
				    NULL,
				    &body,
				    &err))
	  {
	    g_print("%s", body);
	    g_free(body);
	  }
	else
	  {
	    g_warning("unable to transfer: %s", err->message);
	    g_error_free(err);
	  }
      }
      break;

    case MODE_BINARY:
      {
	guint8 *body;
	unsigned int body_len;

	if (st_transfer_session_get_binary(session,
					   url,
					   0,
					   NULL,
					   &body,
					   &body_len,
					   &err))
	  {
	    write(fileno(stdout), body, body_len);
	    g_free(body);
	  }
	else
	  {
	    g_warning("unable to transfer: %s", err->message);
	    g_error_free(err);
	  }
      }
      break;

    case MODE_LINE:
      if (! st_transfer_session_get_by_line(session,
					    url,
					    no_newline ? 0 : ST_TRANSFER_PASS_NEWLINE,
					    NULL,
					    NULL,
					    line_cb,
					    NULL,
					    &err))
	{
	  g_warning("unable to transfer: %s", err->message);
	  g_error_free(err);
	}
      break;
    }
  st_transfer_session_free(session);

  return 0;
}
