/*
 * stegtunnel.c
 *
 * Common functions for both stegclient and stegserver. A lot of the
 * file-transfer mode code resides here.
 *
 * Copyright (c) 2003 Todd MacDermid <tmacd@synacklabs.net> 
 *
 */

#include <netinet/in.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "hamming.h"
#include "sha1.h"
#include "stegtunnel.h"

extern int errno;

const char *verstr= "%s: v0.4 [(c)2003, Todd MacDermid]\n";

/* version: Prints the version */

void 
version(char *prog_name) 
{
  printf(verstr, prog_name);
}

/* stegt_ctx_init allocates and fills in some default values for the
 * stegt_ctx object-zilla that holds most of the configuration and state
 * for the app
 */

struct stegt_ctx *
stegt_ctx_init(int mode)
{
  struct stegt_ctx *new_ctx;
  
  new_ctx = (struct stegt_ctx *)calloc(1, sizeof(struct stegt_ctx));
  if(new_ctx == NULL)
    return(NULL);
  
  if((new_ctx->rand_handle = rand_open()) == NULL) {
    fprintf(stderr, "Error opening random number handle\n");
    free(new_ctx);
    return(NULL);
  }

  new_ctx->mode = mode;
  new_ctx->verbose = 0;

  new_ctx->sessions = hcreate(8);
  new_ctx->hosts = hcreate(8);
  new_ctx->files = hcreate(8);
  new_ctx->repeat = 0;
  new_ctx->file_server = 0;
  new_ctx->file_write_permit = 0;
  new_ctx->nat = 0;
  return(new_ctx);
}

struct stegt_file *
stegt_file_wcreate(void) 
{
  struct stegt_file *new_file;
  
  if((new_file = calloc(1, sizeof(struct stegt_file))) == NULL)
    return(NULL);

  gettimeofday(&(new_file->last_active), NULL);
  new_file->file_loc = 0;
  new_file->last_good_seq = 0;
  new_file->need_resynch = 0;
  new_file->pot_resynch[0] = 0;
  new_file->pot_resynch[1] = 0;
  new_file->phase = 0;
  new_file->name_loc = 0;
  new_file->check_loc = 0;
  new_file->hamming_loc = 0;
  return(new_file);
}

int
stegt_gen_sum(struct stegt_file *file_ctx, uint8_t *sha_digest) 
{
  SHA1Context sha;
  uint8_t sha_inbuf[SHA_INBUF_SZ];
  int num_read;
  int i;

  lseek(file_ctx->fd, 0, SEEK_SET);
  SHA1Reset(&sha);
  while((num_read = read(file_ctx->fd, sha_inbuf, SHA_INBUF_SZ)) > 0) {
    SHA1Input(&sha, sha_inbuf, num_read);
  }
  if(num_read < 0)
    return(-1);;
  SHA1Result(&sha, sha_digest);

  /*  printf("stegt_gen_sum: ");
  for (i = 0; i < 20; i++) {
    printf("%02x", sha_digest[i]);
  }
  printf("\n"); */

  return(0);
}


int
stegt_file_wopen(struct stegt_file *file_ctx) 
{

  if(strchr(file_ctx->name_buf, '/') != NULL)
    return(-1);

  file_ctx->fd = open(file_ctx->name_buf, O_RDWR|O_CREAT|O_EXCL);
  if(file_ctx->fd < 0) {
    return(-1);
  }
  return(0);
}

struct stegt_file *
stegt_file_ropen(char *name) 
{
  struct stegt_file *new_file;
  struct stat stat_buf;
  uint8_t sha_digest[SHA1HashSize];
  
  if((new_file = calloc(1, sizeof(struct stegt_file))) == NULL)
    return(NULL);

  if((new_file->fd = open(name, O_RDONLY)) < 0)
    goto fail;

  gettimeofday(&(new_file->last_active), NULL);
  fstat(new_file->fd, &stat_buf);
  new_file->size = stat_buf.st_size;
  if(stegt_gen_sum(new_file, sha_digest) < 0)
    goto fail;
  memcpy(new_file->checksum, sha_digest, STEGT_CHECKSUM_BYTES);
  
  lseek(new_file->fd, 0, SEEK_SET);
  new_file->file_loc = 0;
  new_file->last_good_seq = 0;
  new_file->need_resynch = 0;
  new_file->phase = 0;
  new_file->name_loc = 0;
  new_file->check_loc = 0;
  strncpy(new_file->name_buf, name, STEGT_MAXNAME-1);
  if((stegt_file_read(new_file)) < 0) 
    goto fail;
  
  return(new_file);

 fail:
  free(new_file);
  return(NULL);

}

int
stegt_check_sum(struct stegt_file *file_ctx)
{
  uint8_t sha_digest[SHA1HashSize];
  int i;

  if(stegt_gen_sum(file_ctx, sha_digest) < 0) {
    fprintf(stderr, "Error generating checksum\n");
    return(-1);
  }

  for(i = 0; i < STEGT_CHECKSUM_BYTES; i++) {
    if(sha_digest[i] != file_ctx->checksum[i]) {
      return(-1);
    }
  }

  return(0);
}

/* 
int
stegt_file_synch(struct stegt_file *file_ctx, uint32_t loc)
{
  
  lseek(file_ctx->fd, loc, SEEK_SET);
  stegt_file_read(file_ctx);

*/

int 
stegt_file_read(struct stegt_file *file_ctx) 
{
  uint8_t buffer[STEGT_MAXBLOCK];
  uint8_t *buf_ptr;
  int bytes_left;
  int num_read;
  int ret_val;
  int i;

  bytes_left = STEGT_MAXBLOCK;

  buf_ptr = buffer;

  for(i = 0; i < STEGT_MAXBLOCK; i++) {
    buffer[i] = 0;
  }

  while(((num_read = read(file_ctx->fd, buf_ptr, bytes_left)) > 0) && 
	 (bytes_left > 0)) {
    bytes_left -= num_read;
    buf_ptr += num_read;
  }
  if(num_read < 0) {
    return(-1);
  }

  file_ctx->file_loc += (STEGT_MAXBLOCK - bytes_left);

  ret_val = hamming_encode(sizeof(uint16_t), STEGT_K, 
			   buffer, file_ctx->hamming_encoded);
  if(ret_val < 0) {
    return(-1);
  }

  file_ctx->hamming_loc = 0;

  if(bytes_left == STEGT_MAXBLOCK) {
    return(0);
  } else {
    return(1);
  }  
}

int 
stegt_file_write(struct stegt_file *file_ctx) 
{
  uint8_t buffer[STEGT_MAXBLOCK];
  uint8_t *buf_ptr;
  int bytes_left;
  int num_written;
  int ret_val;
  int file_left;
  int lastblock;
  int i;
  
  file_left = file_ctx->size - file_ctx->file_loc;

  if(file_left < STEGT_MAXBLOCK) {
    bytes_left = file_left;
    lastblock = 1;
  } else {
    bytes_left = STEGT_MAXBLOCK;
    lastblock = 0;
  }

  ret_val = hamming_decode(sizeof(uint16_t), STEGT_K,
			   file_ctx->hamming_encoded, buffer);
  buf_ptr = buffer;

  if(ret_val >= 0) {
    file_ctx->file_loc += bytes_left;
    while(bytes_left > 0) {
      num_written = write(file_ctx->fd, buf_ptr, bytes_left);
      bytes_left -= num_written;
      buf_ptr += num_written;
    }
    if(lastblock) {
      file_ctx->phase = STEGT_CHECK_SUM; 
    }    
  }

  for(i = 0; i < STEGT_HAMMING_BLOCK; i++) {
    file_ctx->hamming_encoded[i] = 0;
  }

  file_ctx->hamming_loc = 0;

  return(ret_val);
}

int
stegt_file_output(struct stegt_file *file_ctx, uint8_t *out_buf, int width)
{
  int i;
  int j;
  uint8_t *size_ptr;
  uint32_t tmp_size;
  int ret_val;

  i = 0;
  
  if(file_ctx->phase == STEGT_SENDING_NAME) {
    while((file_ctx->phase == STEGT_SENDING_NAME) && (i < width)) {
      if(file_ctx->name_loc > STEGT_MAXNAME)
	return(-1);
      if(file_ctx->name_buf[file_ctx->name_loc] == 0) {
	out_buf[i] = 0;
	file_ctx->phase = STEGT_SENDING_SIZE;
	/*	printf("size transition\n"); */
      } else {
	out_buf[i] = file_ctx->name_buf[file_ctx->name_loc];
	file_ctx->name_loc++;
      }
      i++;
    }
  }
  
  if(file_ctx->phase == STEGT_SENDING_SIZE) {
    tmp_size = htonl(file_ctx->size);
    size_ptr = (uint8_t *)(&tmp_size);
    while((file_ctx->phase == STEGT_SENDING_SIZE) && (i < width)) {
      out_buf[i] = size_ptr[file_ctx->size_loc];
      file_ctx->size_loc++;
      if(file_ctx->size_loc > STEGT_SIZE_BYTES) {
	file_ctx->phase = STEGT_SENDING_CHECKSUM;
	/*	printf("sum transition\n"); */
      }
      i++;
    }
  }
  
  if(file_ctx->phase == STEGT_SENDING_CHECKSUM) {
    while((file_ctx->phase == STEGT_SENDING_CHECKSUM) && (i < width)) {
      out_buf[i] = file_ctx->checksum[file_ctx->check_loc];
      file_ctx->check_loc++;
      if(file_ctx->check_loc > STEGT_CHECKSUM_BYTES) {
	/*	printf("Checksum sent: ");
	for(j = 0; j < STEGT_CHECKSUM_BYTES; j++) {
	  printf("%02x", file_ctx->checksum[j]);
	}
	printf("\n");
	printf("data transition\n"); */
	file_ctx->phase = STEGT_SENDING_DATA;
      }
      i++;
    }
  }
  
  if(file_ctx->phase == STEGT_SENDING_DATA) {
    while((file_ctx->phase == STEGT_SENDING_DATA) && (i < width)) {
      out_buf[i] = file_ctx->hamming_encoded[file_ctx->hamming_loc];
      file_ctx->hamming_loc++;
      if(file_ctx->hamming_loc >= STEGT_HAMMING_BLOCK) {
	ret_val = stegt_file_read(file_ctx);
	if(ret_val == 0) {
	  file_ctx->phase = STEGT_FILE_DONE;
	  return(0);
	} else if(ret_val < 0) {
	  return(-1);
	} 
      }
      i++;
    }
  }
  
  return(0);
}

void
stegt_file_close(struct stegt_file *file_ctx)
{
  if(file_ctx->fd != -1) {
    fprintf(stderr, "File received, checksum good\n");
    close(file_ctx->fd);
    file_ctx->fd = -1;
  }
}

void
stegt_file_bad(struct stegt_file *file_ctx)
{
  if(file_ctx->fd != -1) {
    fprintf(stderr, "Checksum on file failed\n");
    close(file_ctx->fd);
    file_ctx->fd = -1;
  }
}


int 
stegt_file_input(struct stegt_file *file_ctx, struct stegt_ctx *st_ctx, 
		 uint8_t *in_buf, int width)
{
  int i;
  int j;
  
  i = 0;

  if(file_ctx->phase == STEGT_SENDING_NAME) {
    while((file_ctx->phase == STEGT_SENDING_NAME) && (i < width)) {
      if(file_ctx->name_loc > STEGT_MAXNAME)
	return(-1);
      if(in_buf[i] == 0) {
	/*	printf("size transition\n"); */
	file_ctx->phase = STEGT_SENDING_SIZE;
      } else {
	file_ctx->name_buf[file_ctx->name_loc] = in_buf[i];
	file_ctx->name_loc++;
      }
      i++;
    }
  }
  
  if(file_ctx->phase == STEGT_SENDING_SIZE) {
    while((file_ctx->phase == STEGT_SENDING_SIZE) && (i < width)) {
      file_ctx->size_buf[file_ctx->size_loc] = in_buf[i];
      file_ctx->size_loc++;
      if(file_ctx->size_loc > STEGT_SIZE_BYTES) {
	file_ctx->size = ntohl(*(uint32_t *)file_ctx->size_buf);
	if(st_ctx->verbose) {
	  fprintf(stderr, "Inbound file name: %s - file size %d bytes\n",
		  file_ctx->name_buf, file_ctx->size);
	}
	file_ctx->phase = STEGT_SENDING_CHECKSUM;
	/*	printf("sum transition\n"); */
      }
      i++;
    }
  }
  if(file_ctx->phase == STEGT_SENDING_CHECKSUM) {
    while((file_ctx->phase == STEGT_SENDING_CHECKSUM) && (i < width)) {
      file_ctx->checksum[file_ctx->check_loc] = in_buf[i];
      file_ctx->check_loc++;
      if(file_ctx->check_loc > STEGT_CHECKSUM_BYTES) {
	/*	printf("Checksum received: ");
	for(j = 0; j < STEGT_CHECKSUM_BYTES; j++) {
	  printf("%02x", file_ctx->checksum[j]);
	}
	printf("\n");
	printf("data transition\n"); */

	file_ctx->phase = STEGT_SENDING_DATA;
	if(stegt_file_wopen(file_ctx) < 0) {
	  return(-1);
	}
      }
      i++;
    }
  }
  if(file_ctx->phase == STEGT_SENDING_DATA) {
    while((file_ctx->phase == STEGT_SENDING_DATA) && (i < width)) {
      file_ctx->hamming_encoded[file_ctx->hamming_loc] = in_buf[i];
      file_ctx->hamming_loc++;
      if(file_ctx->hamming_loc >= STEGT_HAMMING_BLOCK) {
	if(stegt_file_write(file_ctx) < 0) { 
	  return(1);	  
	}
	if(file_ctx->size - file_ctx->file_loc == 0) {
	  file_ctx->phase = STEGT_CHECK_SUM;
	}
      }
      i++;
    }
  }
  if(file_ctx->phase == STEGT_CHECK_SUM) {
    if(stegt_check_sum(file_ctx) == 0) {
      stegt_file_close(file_ctx);
    } else {
      stegt_file_bad(file_ctx);
    }
    file_ctx->phase = STEGT_FILE_DONE;
  } 

  return(0);
}
