/*
 * stegclient.c 
 *
 * Main code/starting point for stegclient
 *
 * Copyright (c) 2003 Todd MacDermid <tmacd@synacklabs.net> 
 *
 */

#include <netinet/in.h>

#include <dnet.h>
#include <fcntl.h>
#include <pcap.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <packetp.h>

#include "crypto.h"
#include "session.h"
#include "sha1.h"
#include "stegtunnel.h"


/* usage: prints the usage string and then exits */

void 
usage(char *prog_name) 
{
  printf("usage: %s [-hnvSV] [-p <proxyIp> -r <remoteIp>]\n", prog_name);
  exit(0);
}

/*
 * inbound: this function handles packets destined back to us. It is passed
 * to and called from the packet purgatory library. It will update the packet 
 * and the st_ctx appropriately. Since stegclient is using a simplified packet
 * purgatory session, it does not need to handle writing the packet out.
 */ 

int 
inbound(struct packetp_ctx *pp_ctx, uint8_t *packet, void *void_ctx) 
{
  struct stegt_ctx *st_ctx;
  struct stegt_file *file_ctx;
  struct stegt_session *session;
  struct ip_hdr *ip_header;
  struct tcp_hdr *tcp_header;
  struct addr verbose_addr;
  char verbose_buf[80];
  uint32_t ack_num;
  uint32_t tmp_seq;
  uint32_t tmp_seqoff;
  uint32_t tmp_loc;
  uint8_t extracted[4];

  st_ctx = void_ctx;

  ip_header = (struct ip_hdr *)packet;

  if(st_ctx->verbose) {
    memcpy(&(verbose_addr.addr_ip), &(ip_header->ip_src), IP_ADDR_LEN);
    verbose_addr.addr_type = ADDR_TYPE_IP;
    verbose_addr.addr_bits = 32;
    addr_ntop(&verbose_addr, verbose_buf, 80);
  }

  if (ip_header->ip_p != IP_PROTO_TCP) {
    if(st_ctx->verbose) {
      fprintf(stderr, 
	      "Received non-TCP packet from %s, continuing...\n", 
	      verbose_buf);
    }
    return(0);
  }

  tcp_header = (struct tcp_hdr *) (packet + (ip_header->ip_hl * 4));
  if((session = stegt_session_find(packet, st_ctx, IN, pp_ctx)) == NULL) {
    if(st_ctx->verbose)
      fprintf(stderr, 
	      "inbound packet with unknown session from %s, continuing...\n", 
	      verbose_buf);
    return(0); /* Ignore it */
  } else {
    if(session->keyed) {
      ipid_decrypt(packet, st_ctx, extracted);
      if((file_ctx = session->cur_file) != NULL) {
	if(file_ctx->need_resynch) {
	  if(ntohl(tcp_header->th_seq) > file_ctx->synch_seq) {
	    file_ctx->need_resynch = 0;
	  }
	} 
	if(!(file_ctx->need_resynch)) {
	  if((stegt_file_input(file_ctx, st_ctx, extracted, 2)) == 1) {
	    if(st_ctx->verbose) {
	      fprintf(stderr, 
		      "Error detected in hamming block. Resynching...\n");
	    }
	    file_ctx->need_resynch = 1;
	    file_ctx->hamming_loc = 0;
	    file_ctx->synch_seq = ntohl(tcp_header->th_seq) + 
	      ((ntohs(ip_header->ip_len) - IP_HDR_LEN - TCP_HDR_LEN) * 3);
	    file_ctx->pot_resynch[0] = 0;
	    file_ctx->pot_resynch[1] = 0;
	    file_ctx->pot_resynch[0] = 
	      (((file_ctx->file_loc % 64) >> 1) & 0x1F);
	    tmp_seqoff = file_ctx->synch_seq % 16384;
	    file_ctx->pot_resynch[0] |= ((tmp_seqoff >> 11) & 0xE0);
	    file_ctx->pot_resynch[1] = ((tmp_seqoff >> 3) & 0xFF);
	  }
	}
      } else {
	if(extracted[0] != 0) fprintf(stdout, "%c", extracted[0]);
	if(extracted[1] != 0) fprintf(stdout, "%c", extracted[1]);
      }
    }
    ack_num = htonl(ntohl(tcp_header->th_ack) + session->seq_offset);
    memcpy(&(tcp_header->th_ack), &(ack_num), 4);
    
    if(tcp_header->th_flags & TH_SYN) {
      sequence_decrypt(packet, st_ctx, extracted);
      if((extracted[0] == 0) && (extracted[1] == 0) &&
	 (extracted[2] == 0) && (extracted[3] == 0)) {
	session->keyed = 1;
	if(st_ctx->verbose) {
	  fprintf(stderr, "%s: New session keyed\n", verbose_buf);
	}
      } else if ((extracted[0] == 0) && (extracted[1] == 0) &&
		 (extracted[2] == 0) && (extracted[3] == 1)) {
	if(st_ctx->verbose) {
	  printf("New file inbound from %s\n", verbose_buf);
	}
	if((file_ctx = stegt_file_find(packet, st_ctx, IN)) == NULL) {
	  if((file_ctx = stegt_file_wcreate()) == NULL) {
	    fprintf(stderr, "inbound: Error creating new file\n");
	    session->keyed = 0;
	    return(-1);
	  }
	  stegt_file_add(st_ctx, session, file_ctx);
	  session->keyed = 1;
	} else {
	  if(file_ctx->active_session == NULL) {
	    session->keyed = 1;
	    file_ctx->active_session = session;
	    if(file_ctx->need_resynch) {
	      file_ctx->need_resynch = 0;
	    }
	    session->cur_file = file_ctx;
	  } else {
	    session->keyed = 0;
	  }
	}
      }
      if(tcp_header->th_flags & TH_ACK) {
	session->tcp_state = TCP_SYNACK_RECEIVED;
      }
    } else if (tcp_header->th_flags & TH_RST) {
      stegt_session_del(st_ctx, pp_ctx);
      if(st_ctx->verbose)
	fprintf(stderr, "%s: session closed (reset)\n", verbose_buf);
    } else if (tcp_header->th_flags & TH_FIN) {
      if(session->tcp_state == TCP_ESTABLISHED)
	session->tcp_state = TCP_FIN_RECEIVED;
      else if (session->tcp_state == TCP_FIN_SENT)
	if(tcp_header->th_flags & TH_ACK) 
	  session->tcp_state = TCP_FINACK_RECEIVED;
	else
	  session->tcp_state = TCP_FINACK_SENT;
      else
	session->tcp_state = TCP_FIN_RECEIVED;
    } else if (tcp_header->th_flags & TH_ACK) {
      if(session->tcp_state == TCP_SYNACK_SENT)
	session->tcp_state = TCP_ESTABLISHED;
      else if (session->tcp_state == TCP_FINACK_SENT){
	stegt_session_del(st_ctx, pp_ctx);
	if(st_ctx->verbose)
	  fprintf(stderr, "%s: session closed (regular close)\n", verbose_buf);
      }
    } 
  }
  return(0);
}

/*
 * outbound: this function handles packets destined outward. It is passed to 
 * and called from packet purgatory. It will update the packet and 
 * the st_ctx appropriately. Since stegclient is using a simplified packet
 * purgatory session, it does not need to handle writing the packet out.
 */ 

int 
outbound(struct packetp_ctx *pp_ctx, uint8_t *packet, void *void_ctx) 
{
  struct stegt_ctx *st_ctx;
  struct stegt_session *session;
  struct stegt_file *file_ctx;
  struct ip_hdr *ip_header;
  struct tcp_hdr *tcp_header;
  struct addr verbose_addr;
  char verbose_buf[80];
  uint32_t seq_num;
  uint8_t packet_hash[16];
  uint8_t cipher_stream[4];
  uint8_t readbuf[2];
  int i;

  st_ctx = void_ctx;

  ip_header = (struct ip_hdr *)packet;

  if(st_ctx->verbose) {
    if(pp_ctx->wedge_type == PP_FAKEIP) {
      memcpy(&(verbose_addr.addr_ip), &(pp_ctx->target_ip.addr_ip), 
	     IP_ADDR_LEN);
    } else {
      memcpy(&(verbose_addr.addr_ip), &(ip_header->ip_dst), IP_ADDR_LEN);
    }
    verbose_addr.addr_type = ADDR_TYPE_IP;
    verbose_addr.addr_bits = 32;
    addr_ntop(&verbose_addr, verbose_buf, 80);
  }

  if (ip_header->ip_p != IP_PROTO_TCP) {
    if(st_ctx->verbose)
      fprintf(stderr, 
	      "Received non-TCP packet destined to %s, continuing...\n", 
	      verbose_buf);
    return(0);
  }  
  
  tcp_header = (struct tcp_hdr *) (packet + (ip_header->ip_hl * 4));
  if(pp_ctx->wedge_type == PP_FAKEIP) {
    memcpy(&(ip_header->ip_dst), &(pp_ctx->target_ip.addr_ip), IP_ADDR_LEN);
    memcpy(&(ip_header->ip_src), &(pp_ctx->proxy_ip.addr_ip), IP_ADDR_LEN);
  }

  if((session = stegt_session_find(packet, st_ctx, OUT, pp_ctx)) == NULL) {
    if((tcp_header->th_flags & TH_SYN) && (!(tcp_header->th_flags & TH_ACK))) {
      if((session = stegt_session_add(pp_ctx, st_ctx, packet)) == NULL) {
	fprintf(stderr, "Error creating new session to %s\n", verbose_buf);
	return(-1);
      } else {
	if(st_ctx->verbose)
	  fprintf(stderr, "Created new session to %s\n", verbose_buf);
      }
      sha_seq_stream(packet, st_ctx, cipher_stream);
      memcpy(&seq_num, cipher_stream, 4);
      session->seq_offset = ntohl(tcp_header->th_seq) - ntohl(seq_num);
      memcpy(&(tcp_header->th_seq), &seq_num, 4);
      }
  } else {
    seq_num = htonl(ntohl(tcp_header->th_seq) - session->seq_offset);
    memcpy(&(tcp_header->th_seq), &(seq_num), 4);
    if(session->keyed) {
      sha_ipid_stream(packet, st_ctx, cipher_stream);
      if((file_ctx = (session->cur_file)) != NULL) {
	if(file_ctx->need_resynch) {
	  for(i = 0; i < 2; i++) {
	    cipher_stream[i] = cipher_stream[i] ^ file_ctx->pot_resynch[i];
	  }
	}
      } else {
	if(read(0, readbuf, 2) > 0) {
	  for(i = 0; i < 2; i++) {
	    cipher_stream[i] = cipher_stream[i] ^ readbuf[i];
	  }
	}
      }
      memcpy(&(ip_header->ip_id), cipher_stream, 2);
      
      if(tcp_header->th_flags & TH_SYN) {
	session->tcp_state = TCP_SYN_SENT;
      } else if (tcp_header->th_flags & TH_RST) {

	/* Hackish hack, ugly workaround. Why does this reset get generated? */

	if(session->tcp_state != TCP_SYNACK_RECEIVED) {
	  stegt_session_del(st_ctx, pp_ctx);
	  if(st_ctx->verbose)
	    fprintf(stderr, "%s: session closed (reset)\n", verbose_buf);
	}

      } else if (tcp_header->th_flags & TH_FIN) {
	if(session->tcp_state == TCP_ESTABLISHED)
	  session->tcp_state = TCP_FIN_SENT;
	else if(session->tcp_state == TCP_FIN_RECEIVED) {
	  session->tcp_state = TCP_FINACK_SENT;
	}
      } else if (tcp_header->th_flags & TH_ACK) {
	if((session->tcp_state == TCP_SYNACK_SENT) || 
	   (session->tcp_state == TCP_SYNACK_RECEIVED))
	  session->tcp_state = TCP_ESTABLISHED;
	else if((session->tcp_state == TCP_FINACK_SENT) ||
		(session->tcp_state == TCP_FINACK_RECEIVED)) {
	  stegt_session_del(st_ctx, pp_ctx);
	  if(st_ctx->verbose)
	    fprintf(stderr, 
		    "%s: session closed (regular close)\n", 
		    verbose_buf);
	}
      }
    }
  }
  return(0);
}

/* 
 * main: main loop. Interpret command line options, initialize data 
 * structures, then call packetp_start, which only returns if something's
 * gone wrong.
 */

int main(int argc, char **argv) {
  struct packetp_ctx *pp_ctx;
  struct stegt_ctx *st_ctx;
  int c;
  int got_proxy;
  int got_target;
  int i;

  if(!(getuid() == 0)) {
    fprintf(stderr, "Gotta be root to run %s\n", argv[0]);
    return(-1);
  }

  /*
   * ARGUMENTS:
   *
   * -p proxy IP: used to set up the tunnel through a nonexistent 
   * IP address on the local subnet.
   *
   * -r remote IP: the final destination of connections made to proxy IP.
   *
   * -n NAT mode: Weakens the nonce in order to traverse NAT
   *
   * -h: Print help message
   *
   * -V: Verbose mode
   *
   * -v: Print version
   */

  if((pp_ctx = packetp_init()) == NULL) {
    fprintf(stderr, "packetp_init failed\n");
    return(-1);
  }

  if((st_ctx = stegt_ctx_init(SEQ_AND_IPID)) == NULL) {
    fprintf(stderr, "stegt_ctx_init failed\n");
    return(-1);
  }

  got_proxy = 0;
  got_target = 0;

  while((c=getopt(argc, argv, "p:r:hnvV")) != -1) {
    switch(c) {
    case 'p':
      if ((addr_pton(optarg, &(pp_ctx->proxy_ip))) < 0) {
        fprintf(stderr, "Cannot resolve proxy IP: %s\n", optarg);
	return(-1);
      }
      /*     fprintf(stderr, "stegclient proxy:  %s\n", 
	     addr_ntoa(&(pp_ctx->proxy_ip))); */
      got_proxy = 1;
      break;

    case 'r':
      if ((addr_pton(optarg, &(pp_ctx->target_ip))) < 0) {
	fprintf(stderr, "Cannot resolve target IP: %s\n", optarg);
	return(-1);
      }

      got_target = 1;
      break;
      
    case 'h':
      usage(argv[0]);
      break;
      
    case 'n':
      st_ctx->nat = 1;
      break;

    case 'V':
      st_ctx->verbose=1;
      break;

    case 'v':
      version(argv[0]);
      break;
    }
  }

  if(got_proxy) {
    if(!(got_target)) {
      fprintf(stderr, 
	      "Must specify remote IP address when specifying a proxy IP\n");
      usage(argv[0]);
    }
    if(pp_ctx->target_ip.addr_bits < IP_ADDR_BITS) {
      fprintf(stderr, 
	      "Remote IP address cannot be CIDR block if proxy mode used\n"); 
      usage(argv[0]);
    }
    pp_ctx->wedge_type = PP_FAKEIP;

  }
  
  if(get_passphrase(st_ctx) < 0) {
    fprintf(stderr, "get_passphrase failed\n");
    return(-1);
  }

  if (fcntl(0, F_SETFL, O_NONBLOCK) < 0) {
    fprintf(stderr, "fcntl error\n");
    return(-1);
  } 

  packetp_start(pp_ctx, inbound, outbound, st_ctx);
  return(0);
}
