/*
 * routestate.c
 *
 * Functions involved in mangling the routing table, and restoring it
 * to "normalcy." Also contains functions to determine where the old
 * routing table used to point to, so that ethernet frames can be written
 * out those interfaces.
 *
 * Copyright (c) 2003 Todd MacDermid <tmacd@synacklabs.net>
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "packetp.h"

/* packetp_route_delete - used to free all memory associated with a a linked 
 * list of route_node structures. Cannot fail. (Well, even if it does, 
 * you did your best already).
 */

void packetp_route_delete(struct packetp_ctx *ctx) {
  struct route_node *walk_node, *free_node;

  walk_node = ctx->route_head;
  while (walk_node != NULL) {
    free_node = walk_node;
    walk_node = walk_node->next;
    eth_close(free_node->eth_handle);
    free(free_node);
  }
}

/* packetp_route_restore - This will (hopefully) return the routing table to
 * the state it was in when you first took a snapshot of it using
 * packetp_route_save(). 0 on success, -1 on failure (Usually because you're
 * not root or somesuch).
 */

int packetp_route_restore(struct packetp_ctx *ctx) {
  struct route_node *walk_node;
  struct route_entry *entry;

  if (ctx == NULL) {
    fprintf(stderr, "packetp_route_restore: was passed a NULL pointer\n");
    return(-1);
  }

  packetp_route_clear(ctx);

  walk_node = ctx->route_head;
  while(walk_node != NULL) {
    entry = &(walk_node->route);
    if((route_add(ctx->route_handle, entry)) < 0) {
      fprintf(stderr, "packetp_route_restore: route_add failed\n");
      return(-1);
    }
    walk_node = walk_node->next;
  }
  return(0);
}

/* loop_wiperoute - Called by the route_loop() in packetp_route_clear(). 
 * Deletes the route given from the routing table.
 */

int loop_wiperoute(const struct route_entry *entry, void *handle_arg) {
  route_t *route_handle;

  route_handle = (route_t *)handle_arg;
  return(route_delete(route_handle, entry));
}

/* packetp_route_clear - Wipes all entries from the routing table. Make sure
 * you've saved the state before you call this, unless you're just trying
 * to make it difficult to use the machine. 0 on success, -1 on failure
 */

int packetp_route_clear(struct packetp_ctx *ctx) {
  int ret;

  ret = route_loop(ctx->route_handle, loop_wiperoute, ctx->route_handle);
  return(ret);
}
  
/* loop_saveroute is called by the route_loop() in packetp_route_save(), 
 * building up the linked list of route table entries.
 */

int loop_saveroute(const struct route_entry *entry, void *ctx_arg) {
  struct route_node *new_node;
  struct packetp_ctx *ctx;
  struct route_entry local_entry;

  /* A word about the next memcpy. entry must be const in this function,
   * as libdnet requires, but we use the route_gw member as an argument
   * to intf_get_dst. This won't change it, but the compiler spits
   * out a warning anyways. This shuts the warning up.
   */

  memcpy(&local_entry, entry, sizeof(struct route_entry)); 

  ctx = (struct packetp_ctx *)ctx_arg;

  if ((new_node = (struct route_node *)
       calloc(1, sizeof(struct route_node))) == NULL) {
    fprintf(stderr, "loop_saveroute: Could not allocate memory\n");
    return(-1);
  }

  new_node->interface.intf_len = sizeof(struct intf_entry);
  if((intf_get_dst(ctx->intf_handle, &(new_node->interface), 
		   &(local_entry.route_gw))) < 0) {
    fprintf(stderr, "loop_saveroute: Error in intf_get_dst\n");
    free(new_node);
    return(-1);
  }

  if((new_node->eth_handle = eth_open(new_node->interface.intf_name)) == NULL) {
    fprintf(stderr, "loop_saveroute: eth_open failed to open interface %s\n", new_node->interface.intf_name);
    free(new_node);
    return(-1);
  }

  memcpy(&(new_node->route), entry, sizeof(struct route_entry));

  new_node->next = ctx->route_head;
  ctx->route_head = new_node;
  return(0);
}
  
/* packetp_route_save creates a linked list of routing entries within the 
 * given context. Returns -1 on failure. This function used to do quite a 
 * bit more, but has been rather eviscerated by the recent rewrite into a 
 * thin wrapper.
 */

int packetp_route_save(struct packetp_ctx *ctx) {
  return(route_loop(ctx->route_handle, loop_saveroute, ctx));
}

/* packetp_route_find will give you the node with info best matching the 
 * destination, provided a context showing what the routing table once 
 * looked like before you started messing with it, you cruel route-table 
 * mangler, you. Dest must point at a valid IP address. Returns NULL on 
 * failure.
 */

struct route_node *
packetp_route_find(struct packetp_ctx *ctx, struct addr *dest) 
{
  struct route_node *walk_node;
  struct route_node *best = NULL;
  int best_bits = -1;
  
  if ((ctx == NULL) || (dest == NULL)) {
    fprintf(stderr, "packetp_route_find: NULL pointer passed as argument\n");
    return(NULL);
  }

  if (dest->addr_type != ADDR_TYPE_IP) {
    fprintf(stderr, "packetp_route_find: Non-IP address passed as destination\n");
    return(NULL);
  }

  walk_node = ctx->route_head;
  while (walk_node != NULL) {
    if (addr_netcmp(&(walk_node->route.route_dst), dest) == 0) { 
      if (walk_node->route.route_dst.addr_bits > best_bits) {
	best = walk_node;
	best_bits = walk_node->route.route_dst.addr_bits;
      }
    } 
    walk_node = walk_node->next;
  }
  if(best_bits > -1) {
    return(best);
  } else {
    fprintf(stderr, "packetp_route_find: No matching route found\n");
    return(NULL);
  }
}

/* packetp_route_print will print out the routing entries contained in the
 * context passed to it. Useful for debugging purposes
 */

void packetp_route_print(struct packetp_ctx *ctx) {
  struct route_node *walk_node;

  if(ctx != NULL) {

  fprintf(stderr,"Saved route table:\n");
  
    walk_node = ctx->route_head;
    
    while(walk_node != NULL) {
      printf("dst: %s gw: %s intf: %s\n", addr_ntoa(&(walk_node->route.route_dst)),
	     addr_ntoa(&(walk_node->route.route_gw)), walk_node->interface.intf_name);
      walk_node = walk_node->next;
    }
  }		   
}
