/*
 * packetp.c 
 *
 * This code is responsible for initializing the packet redirection through
 * silent IP or loopback, and routing packets to the appropriate logic 
 * for mangling
 *
 * Copyright (c) 2003 Todd MacDermid <tmacd@synacklabs.net> 
 *
 */

#ifdef POSIX_SELECT
#include <sys/select.h>
#else
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#endif
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <dnet.h>
#include <pcap.h>

#include "packetp.h"

#define MAX_FILTER_SIZE 1024
#define INTFSIZE 1024

int loop_continue = 1;
extern int errno;

/* end_loop is used as a signal handler, and attempts to terminate the
 * process cleanly
 */

void end_loop(int throwaway) {
  loop_continue = 0;
}

/* arpd_dies is the handler for when the arping child goes away. Right
 * now we just exit. In the future we may try to restart it.
 */

void arpd_dies(int throwaway) {
  loop_continue = 0;
}

int packetp_process(struct packetp_ctx *ctx, packet_handler inbound, 
		   packet_handler outbound, uint8_t *packet, void *state,
		   int direction);

/* zero_low_net_bits will copy one struct addr into another, but make sure 
 * that any bits that are not included in the netmask are zeroed out. This
 * will make libpcap happier when it gets them as part of a filter.
 * (mini-rant: Why can't libpcap do this for itself?)
 */

int 
packetp_zero_low_net_bits(struct addr *dest, const struct addr *src) 
{
  int i;
  uint8_t bytemask;
  
  if((dest == NULL) || (src == NULL)) {
    fprintf(stderr, "zero_low_net_bits: NULL struct addr pointer passed\n");
    return(-1);
  }
  dest->addr_type = src->addr_type;
  dest->addr_bits = src->addr_bits;
  for(i = 0; ((i+1)*8) <= src->addr_bits; i ++) 
    dest->addr_data8[i] = src->addr_data8[i];
  
  if((src->addr_bits % 8) != 0) {
    bytemask = (~0 << (8 - (src->addr_bits % 8)));
    dest->addr_data8[i] = (src->addr_data8[i] & bytemask);
    i++;
  }
  for(; i < 16; i++) {
    dest->addr_data8[i] = 0;
  }
  return(0);
}


/* set_local_routes is called by intf_loop, and it sets a route for each
 * local IP range out the loopback interface. This is so local IP addresses
 * will still be routed out the loopback.
 */

int set_local_routes(const struct intf_entry *entry, void *ctx_arg) {
  struct packetp_ctx *ctx;
  struct route_entry temp_route;
  int retval;

  if(entry->intf_type == INTF_TYPE_LOOPBACK) {
    return(0);
  }

  ctx = ctx_arg;
  addr_pton("127.0.0.1", &(temp_route.route_gw));
  packetp_zero_low_net_bits(&(temp_route.route_dst), &(entry->intf_addr));
  if ((retval = route_add(ctx->route_handle, &temp_route)) < 0) {
    fprintf(stderr, "set_local_routes: Could not add route: dest:%s gw: %s %d\n",
	    addr_ntoa(&(temp_route.route_dst)), addr_ntoa(&(temp_route.route_gw)), errno);
    return(-1);
  }
  return(0);
}

/* set_routes updates the routing table to point the route to the target
 * system(s) to an interface we can sniff on  (loopback), and won't really 
 * make it to the target host. The existing route table information has 
 * already been saved by packetp_route_save().
 */

int set_routes (struct packetp_ctx *ctx) {
  struct route_entry temp_route;

  if(ctx->target_ip.addr_ip == IP_ADDR_ANY) {
    if (packetp_route_clear(ctx) < 0) {
      fprintf(stderr, "set_routes: Error wiping existing route table\n");
      return(-1);
    }
  }

  addr_pton("127.0.0.1", &(temp_route.route_gw));
  memcpy(&(temp_route.route_dst), &(ctx->target_ip), sizeof(struct addr));
  if (route_add(ctx->route_handle, &temp_route) < 0) {
    fprintf(stderr, "set_routes: Could not add route: dest:%s gw: %s\n",
	    addr_ntoa(&(temp_route.route_dst)), addr_ntoa(&(temp_route.route_gw)));
    packetp_route_restore(ctx);
    return(-1);
  }

  /* If we're trying to reroute packets to IP_ADDR_ANY, then we need
   * to set interface-specific routes, or else packets to local
   * subnets will just go out their local interfaces.
   */

  if(ctx->target_ip.addr_ip == IP_ADDR_ANY) {
    if((intf_loop(ctx->intf_handle, set_local_routes, ctx)) < 0) {
      fprintf(stderr, "set_routes: Could not add local routes\n");
      packetp_route_restore(ctx);
      return(-1);
    }
  }
  
  return(0);
}

/* packetp_init allocates and fills in some values for a packetp_ctx 
 * structure. By default, it creates a firewall based tunnel, to all 
 * IP addresses, simplified. This is the probably the most commonly used
 * tunnel type. If a system doesn't support firewalls, or the program only 
 * wants a single IP address tampered with, or doesn't want simplified mode
 * (if the input->output mapping of packets won't be 1 to 1), edit 
 * these parameters _before_ calling packetp_start. It will not open any
 * handles, as those will be dependent on tunnel type, which we're not
 * set on quite yet.
 */

struct packetp_ctx *packetp_init(void) {
  struct packetp_ctx *new_ctx;
  
  if((new_ctx = 
      (struct packetp_ctx *)calloc(1, sizeof(struct packetp_ctx))) == NULL) {
    fprintf(stderr, "packetp_init: calloc() failed\n");
    return(NULL);
  }

  new_ctx->wedge_type = PP_LOOPFW;
  new_ctx->simplified = 1;
  new_ctx->verbose = 0;
  new_ctx->pcap_head = NULL;
  new_ctx->route_head = NULL;
  new_ctx->fw_head = NULL;
  new_ctx->ip_handle = NULL;
  new_ctx->fw_handle = NULL;
  new_ctx->arp_handle = NULL;
  new_ctx->route_handle = NULL;
  new_ctx->intf_handle = NULL;

  new_ctx->arp_table = hcreate(8);
  pthread_mutex_init(&(new_ctx->arptab_mutex), NULL);

  new_ctx->target_ip.addr_ip = IP_ADDR_ANY;
  new_ctx->target_ip.addr_type = ADDR_TYPE_IP;
  new_ctx->target_ip.addr_bits = 0;

  return(new_ctx);
}

/* 
 * packetp_set_target resets the IP addresses that will have packets captured
 * on their way to. If the wedge_type is PP_FAKEIP, this must be a /32
 * (That is, a host and not a net).
 *
 * There's no reason why a user can't do this directly, but it means they
 * don't have to poke directly at the context guts themselves.
 */

int
packetp_set_target(struct packetp_ctx *ctx, struct addr *target)
{
  
  memcpy(&(ctx->target_ip), target, sizeof(struct addr));
  return(0);
}

/*
 * packetp_set_type sets the wedge_type parameter in the packetp_ctx.
 * No reason not to do this directly, other than not forcing people to
 * look directly in the context guts.
 */

int
packetp_set_type(struct packetp_ctx *ctx, int type)
{

  ctx->wedge_type = type;
  return(0);
}

/*
 * packetp_set_proxy sets the proxy IP address. This address must be a
 * host, not a network. This has no effect whatsoever unless the
 * wedge_type is set to PP_FAKEIP.
 *
 * Again, no reason not to set it directly yourself, but shielding the
 * guts.
 */

int
packetp_set_proxy(struct packetp_ctx *ctx, struct addr *proxy)
{

  if(proxy->addr_bits != IP_ADDR_BITS)
    return(-1);
  memcpy(&(ctx->proxy_ip), proxy, sizeof(struct addr));
  return(0);
}

/*
 * packetp_set_simplified sets the simplified parameter in the context.
 * No reason people can't do this their own selves, but the call is 
 * provided. Makes the man pages pretty ;)
 */

int
packetp_set_simplified(struct packetp_ctx *ctx, int simplified)
{

  ctx->simplified = simplified;
  return(0);
}

/*
 * packetp_set_verbose sets the verbosity level of packet purgatory.
 * Another context-shielder.
 */

int
packetp_set_verbose(struct packetp_ctx *ctx, int verbose)
{

  ctx->verbose = verbose;
  return(0);
}

/* handle_open opens the various handles supplied by dnet  that will be 
 * used by our tunnel. Returns 0 on success, -1 on failure.
 */

int handle_open(struct packetp_ctx *ctx) {

  if((ctx->ip_handle = ip_open()) == NULL) {
    fprintf(stderr, "handle_open: ip_open() failed\n");
    return(-1);
  }
  if((ctx->intf_handle = intf_open()) == NULL) {
    fprintf(stderr, "handle_open: intf_open failed\n"); 
    goto fail1;
  }

  if (ctx->wedge_type == PP_FAKEIP) {
    
  } else if (ctx->wedge_type == PP_LOOPFW) {
    if((ctx->fw_handle = fw_open()) == NULL) {
      fprintf(stderr, "handle_open: fw_open failed\n"); 
      goto fail2;
    }
    if((ctx->arp_handle = arp_open()) == NULL) {
      fprintf(stderr, "handle_open: arp_open failed\n"); 
      goto fail3;
    }
    if((ctx->route_handle = route_open()) == NULL) {
      fprintf(stderr, "handle_open: route_open failed\n"); 
      goto fail4;
    }
  } else {
    fprintf(stderr, "handle_open: Unknown tunnel type\n");
    return(-1);
  }
  return(0);
  
 fail4:
  arp_close(ctx->arp_handle);
 fail3:
  fw_close(ctx->fw_handle);
 fail2:
  intf_close(ctx->intf_handle);
 fail1:
  ip_close(ctx->ip_handle);
  return(-1);
}

/* handle_close winds down all the handles we opened with handle_open()
 */

void handle_close(struct packetp_ctx *ctx) {
  
  if(ctx->ip_handle != NULL) ip_close(ctx->ip_handle);
  if(ctx->fw_handle != NULL) fw_close(ctx->fw_handle);
  if(ctx->arp_handle != NULL) arp_close(ctx->arp_handle);
  if(ctx->route_handle != NULL) route_close(ctx->route_handle);
  if(ctx->intf_handle != NULL) intf_close(ctx->intf_handle);
}

/*
 * packetp_frame_send tacks on an ethernet header and lets it rip out for the
 * simplified settings crowd. Uses all info from the ctx it
 * takes in. If it can't find a MAC addr destination, it will split
 * off a new thread to baby that frame along until it either times out
 * or finds a MAC. (Don't want to block on one frame forever, things to do,
 * people to see, packets to mangle).
 */

int packetp_frame_send(struct packetp_ctx *ctx, uint8_t *packet) {
  uint8_t frame[ETH_LEN_MAX];
  uint8_t *detachframe;
  struct thread_arg *targ;
  struct eth_hdr *eth_header;
  struct ip_hdr *ip_header;
  struct addr ip_out;
  struct arp_entry dest_arp;
  struct route_node *route_out;
  pthread_t thread_id;
  pthread_attr_t thread_attr;
  int i;
  
  eth_header = (struct eth_hdr *)frame;
  ip_header = (struct ip_hdr *)packet;

  ip_out.addr_type = ADDR_TYPE_IP;
  ip_out.addr_bits = IP_ADDR_BITS;
  memcpy(&(ip_out.addr_ip), &(ip_header->ip_dst), IP_ADDR_LEN);

  if((route_out = packetp_route_find(ctx, &ip_out)) == NULL) {
    fprintf(stderr, "packetp_frame_send: packetp_route_find failed\n");
    return(-1);
  }

  if(memcmp(&(ip_header->ip_dst), 
	    &(route_out->interface.intf_addr.addr_ip),
	    IP_ADDR_LEN) == 0) 
    return(0);

  memcpy(&(eth_header->eth_src), 
	 &(route_out->interface.intf_link_addr.addr_eth),
	 ETH_ADDR_LEN);
  eth_header->eth_type = htons(ETH_TYPE_IP);

  ip_checksum(ip_header, ntohs(ip_header->ip_len));
  memcpy(((uint8_t *)frame) + ETH_HDR_LEN, 
	 ip_header, ntohs(ip_header->ip_len));
  
  if(addr_netcmp(&(route_out->interface.intf_addr) ,&ip_out) == 0) {
    memcpy(&(dest_arp.arp_pa), &ip_out, sizeof(struct addr));
  } else {
    memcpy(&(dest_arp.arp_pa), &(route_out->route.route_gw), 
	   sizeof(struct addr));
  }
  if(packetp_get_mac_nonblock(ctx, &dest_arp) == 0) { 
    memcpy(&(eth_header->eth_dst), 
	   &(dest_arp.arp_ha.addr_eth), 
	   ETH_ADDR_LEN);
    eth_send(route_out->eth_handle, frame, 
	     ntohs(ip_header->ip_len)+ETH_HDR_LEN);

    /*    printf("eth_sending: ");
    for (i = 0; i < (ntohs(ip_header->ip_len)+ETH_HDR_LEN); i++) {
      printf("%02x", *(frame+i));
      if(i%4 == 1)
	printf(" ");
    }
    printf("\n"); */
    

  } else { 
    /* 
     * No arp entry currently, we need to stick this packet into a separate
     * thread so we don't block waiting for an arp response
     */
    
    if((detachframe = calloc(1, ETH_LEN_MAX)) == NULL) {
      fprintf(stderr, "packetp_frame_send: failed allocating memory\n");
      return(-1);
    }

    if((targ = (struct thread_arg *)
       calloc(1, sizeof(struct thread_arg))) == NULL) {
      fprintf(stderr, "packetp_frame_send: failed allocating memory\n");
      free(detachframe);
      return(-1);
    }

    memcpy(detachframe, frame, ntohs(ip_header->ip_len)+ETH_HDR_LEN);

    /*    printf("delayed_sending: ");
    for (i = 0; i < (ntohs(ip_header->ip_len)+ETH_HDR_LEN); i++) {
      printf("%02x", *(frame+i));
      if(i%4 == 1)
	printf(" ");
    }
    printf("\n"); */

    targ->ctx = ctx;
    targ->frame = detachframe;
    targ->rnode = route_out;
    memcpy(&(targ->next_hop), &(dest_arp.arp_pa), sizeof(struct addr));

    pthread_attr_init(&thread_attr);
    pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);
    pthread_create(&thread_id, &thread_attr, packetp_arp_wait, targ);
    pthread_attr_destroy(&thread_attr);
  }
  return(0);
}

/* packetp_start is supposed to be a relatively infinite loop, and will only
 * exit upon some error, or if the child that was forked to arp for the
 * silent IP address goes away. The inbound and outbound handlers should
 * handle all application logic from here on out.
 */

int packetp_start(struct packetp_ctx *ctx, packet_handler inbound, 
		 packet_handler outbound, void *state) {
  
  uint8_t *packet_buf;
  struct pcap_pkthdr packet_head;
  struct eth_hdr *eth_header;
  struct ip_hdr *ip_header;
  int retval;
  struct pcap_node *walk_pcap_node;
  fd_set readfds;
  int max_fd = 0;
  int i;

  if((handle_open(ctx)) < 0) {
    fprintf(stderr, "packetp_start: handle_open() failed\n");
    return(EOPEN);
  }

  if (packetp_route_save(ctx) < 0) {
    fprintf(stderr, "packetp_start: packetp_route_save() failed\n");
    handle_close(ctx);
    return(EROUTE);
  }

  if((packetp_pcap_init(ctx)) < 0) {
    fprintf(stderr, "packetp_start: packetp_pcap_init() failed\n");
    handle_close(ctx);
    return(EPCAP);
  }

  if (ctx->wedge_type == PP_FAKEIP) {
    if((packetp_fork_arpd(&(ctx->proxy_ip), arpd_dies, NULL)) < 0) {
      fprintf(stderr, "packetp_start: packetp_fork_arpd() failed\n");
      handle_close(ctx);
      return(EARP);
    }
    
    while(loop_continue) {
      packet_buf = (uint8_t *)pcap_next(ctx->pcap_head->pcap_handle, &packet_head);
      if(packet_buf != NULL) {
	packet_buf += ETH_HDR_LEN;
	ip_header = (struct ip_hdr *)packet_buf;
	if (ip_header->ip_src == ctx->my_ip.addr_ip) {
	  packetp_process(ctx, inbound, outbound, packet_buf, state, OUT);
	} else {
	  packetp_process(ctx, inbound, outbound, packet_buf, state, IN);
	}
      }
    }
    fprintf(stderr, "packetp_start: child arp daemon died\n");
    handle_close(ctx);
    return(EARPDIED);
  } else if (ctx->wedge_type == PP_LOOPFW) {

    if((packetp_fw_init(ctx) < 0)) {
      fprintf(stderr, "packetp_start: Error setting firewall rules (Do you have firewall capability? Is the firewall running?)\n");
      packetp_pcap_end(ctx);
      handle_close(ctx);
      return(EFIREWALL);
    }
    
    if((set_routes(ctx) < 0)) {
      fprintf(stderr, "packetp_start: Error setting new route\n");
      packetp_fw_end(ctx);
      packetp_pcap_end(ctx);
      handle_close(ctx);
      return(EROUTE);
    }
    
    /* OK, we're set on routes, firewalls, everything. Time to start
     * sniffing. We'll select() across all of our pcap file descriptors
     * repeatedly, and process the packets as they come in.
     */
 
    walk_pcap_node = ctx->pcap_head;
    while(walk_pcap_node != NULL) {
      if (walk_pcap_node->pcap_fd >= max_fd) {
	max_fd = walk_pcap_node->pcap_fd + 1;
      }
      walk_pcap_node = walk_pcap_node->next;
    }

    signal(SIGINT, end_loop);
    signal(SIGTERM, end_loop);

    while(loop_continue) {
      FD_ZERO(&readfds);
      walk_pcap_node = ctx->pcap_head;
      while(walk_pcap_node != NULL) {
	FD_SET(walk_pcap_node->pcap_fd, &readfds);
	walk_pcap_node = walk_pcap_node->next;
      }

      retval = select(max_fd, &readfds, NULL, NULL, NULL);
      if(retval <= 0) {
	packetp_fw_end(ctx);
	packetp_route_restore(ctx);
	packetp_pcap_end(ctx);
	handle_close(ctx);
	return(ESELECT);
      }

      walk_pcap_node = ctx->pcap_head;
      while(walk_pcap_node != NULL) {
	if(FD_ISSET(walk_pcap_node->pcap_fd, &readfds)) { 
	  packet_buf = (uint8_t *)pcap_next(walk_pcap_node->pcap_handle, &packet_head);
	  if(packet_buf != NULL) {
	    eth_header = (struct eth_hdr *)packet_buf;
	    if(htons(eth_header->eth_type) == ETH_TYPE_IP) {
	      if(walk_pcap_node->interface.intf_type == INTF_TYPE_LOOPBACK) {
		packet_buf += ETH_HDR_LEN;
		packetp_process(ctx, inbound, outbound, 
				packet_buf, state, OUT);
	      } else {
		packet_buf += ETH_HDR_LEN;
		packetp_process(ctx, inbound, outbound, packet_buf, state, IN);
	      }
	    } else if (htons(eth_header->eth_type) == ETH_TYPE_ARP) {
	      packetp_arp_update(ctx, packet_buf);
	    }
	  }
	}
	walk_pcap_node = walk_pcap_node->next;
      }
    }
    packetp_fw_end(ctx);
    packetp_route_restore(ctx);
    packetp_pcap_end(ctx);
    handle_close(ctx);
  } else {
    fprintf(stderr, "Unknown tunnel type\n");
    return(EUNKNOWN);
  }
  return(0);
}

/* packetp_process will call the appropriate inbound or outbound handler, and
 * it also will reinject the packet after modification, if the simplified 
 * option is set. 
 */

int packetp_process(struct packetp_ctx *ctx, packet_handler inbound, 
		   packet_handler outbound, uint8_t *packet, void *state,
		   int direction) {

  struct ip_hdr *ip_header;
  int sendbytes, sentbytes;
  int retval;
  int i;

  /*  for (i = 0; i < 40; i++) {
    printf("%02x", *(packet+i));
    if(i%4 == 3)
      printf("\n");
  }
  printf("\n");
  
  printf(" ------------ \n");
  */

  if (direction == OUT) {
    retval = outbound(ctx, packet, state);
  } else if (direction == IN) {
    retval = inbound(ctx, packet, state);
  } else {
    fprintf(stderr, "packetp_process: invalid direction\n");
    return(-1);
  }

  if((ctx->simplified) && (retval == 0)) {
    ip_header = (struct ip_hdr *)packet;
    if (ctx->wedge_type == PP_FAKEIP) {
      /*       fprintf(stderr, "stegclient target process:  %s\n", 
	  addr_ntoa(&(ctx->target_ip))); 
       fprintf(stderr, "stegclient my process:  %s\n", 
	  addr_ntoa(&(ctx->my_ip))); 
      */
      if (direction == OUT) {
	memcpy(&(ip_header->ip_dst), &(ctx->target_ip.addr_ip), IP_ADDR_LEN);
      } else {
	memcpy(&(ip_header->ip_dst), &(ctx->my_ip.addr_ip), IP_ADDR_LEN);
      }
      memcpy(&(ip_header->ip_src), &(ctx->proxy_ip.addr_ip), IP_ADDR_LEN);
      sendbytes = ntohs(ip_header->ip_len);
      ip_checksum(packet, sendbytes);
      sentbytes = ip_send(ctx->ip_handle, packet, sendbytes);
      if (sendbytes != sentbytes) {
	fprintf(stderr, "packetp_process: Attempted to send %d bytes, actually sent %d bytes\n", sendbytes, sentbytes);
      }
      /*  printf("sent %d bytes\n", sentbytes);
      for (i = 0; i < sentbytes; i++) {
	printf("%02x", *(packet+i));
	if(i%4 == 3)
	  printf("\n");
      }
      printf("\n");
      */
    } else if (ctx->wedge_type == PP_LOOPFW) {
      if(direction == OUT) { 
	packetp_frame_send(ctx, packet);
      } else {
	/*	packetp_loopback_send(ctx, packet);
		}*/

      	sendbytes = ntohs(ip_header->ip_len);
	ip_checksum(packet, sendbytes);
	sentbytes = ip_send(ctx->ip_handle, packet, sendbytes);

	if (sendbytes != sentbytes) {
	  fprintf(stderr, "packetp_process: Attempted to send %d bytes, actually sent %d bytes\n", sendbytes, sentbytes);
	} 
      }  
    } else {
      fprintf(stderr, "packetp_process: Unknown tunnel type\n");
      return(-1);
    }
  }
  return(0);
}
