/*
 * crypto.c 
 *
 * Code having to do with hashing and stream cipher use for
 * stegclient and stegserver. 
 *
 * Copyright (c) 2003 Todd MacDermid <tmacd@synacklabs.net> 
 *
 */

#include <dnet.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>

#include "crypto.h"
#include "rand.h"
#include "sha1.h"
#include "stegtunnel.h"


/* get_passphrase reads in and hashes the passphrase that will be used
 * as the shared secret between client and server. Note that the 
 * signal-catching and terminal-attribute-setting bits are swiped from
 * Richard Stevens' Advanced Programming in the Unix Environment, 
 * a most excellent book.
 */

int 
get_passphrase(struct stegt_ctx *st_ctx) 
{
  char passphrase[PASSPHRASE_LEN + 1];
  SHA1Context sha1_ctx;
  FILE *fp;
  struct termios term;
  struct termios saved_term;
  sigset_t sig;
  sigset_t saved_sig;
  int passlen;
  int c;
  char *ptr;

  if( (fp = fopen(ctermid(NULL), "r+")) == NULL) {
    fprintf(stderr, "get_passphrase: failure to open terminal device\n");
    return(-1);
  }
  setbuf(fp, NULL);

  sigemptyset(&sig);
  sigaddset(&sig, SIGINT);
  sigaddset(&sig, SIGTSTP);
  sigprocmask(SIG_BLOCK, &sig, &saved_sig);

  tcgetattr(fileno(fp), &saved_term);
  term = saved_term;
  term.c_lflag &= ~(ECHO | ECHOE | ECHOK | ECHONL);
  tcsetattr(fileno(fp), TCSAFLUSH, &term);

  fputs("Enter passphrase:", fp);
  
  ptr = passphrase;
  while((c = getc(fp)) != EOF && c != '\n') {
    if(ptr < &passphrase[PASSPHRASE_LEN])
      *ptr++ = c;
  }

  *ptr = 0;
  putc('\n', fp);
  
  tcsetattr(fileno(fp), TCSAFLUSH, &saved_term);
  sigprocmask(SIG_SETMASK, &saved_sig, NULL);
  fclose(fp);

  passlen = strlen(passphrase);

  SHA1Reset(&sha1_ctx);
  SHA1Input(&sha1_ctx, passphrase, passlen);
  SHA1Result(&sha1_ctx, st_ctx->pw_hash);
  for(c = 0; c < passlen; c++) {
    passphrase[c] = 0;
  }
  return(0);
}

int 
hash_ipid(uint8_t *packet, struct stegt_ctx *st_ctx, uint8_t *hash)
{
  SHA1Context sha1_ctx;
  struct ip_hdr *ip_header;
  struct tcp_hdr *tcp_header;
  uint8_t *tmp_pkt;
  
  ip_header = (struct ip_hdr *)packet;
  tcp_header = (struct tcp_hdr *) (packet + (ip_header->ip_hl * 4));
  
  if ((tmp_pkt = (uint8_t *)calloc(1, ntohs(ip_header->ip_len))) == NULL) {
    fprintf(stderr, "ip_id_decrypt: calloc failed\n");
    return(-1);
  }
  
  memcpy(tmp_pkt, packet, ntohs(ip_header->ip_len));
  ip_header = (struct ip_hdr *)tmp_pkt;
  tcp_header = (struct tcp_hdr *) (tmp_pkt + (ip_header->ip_hl * 4));
  
  ip_header->ip_ttl = 0;
  ip_header->ip_sum = 0;
  ip_header->ip_id = 0;
  tcp_header->th_sum = 0;
  if(st_ctx->nat) {
    ip_header->ip_src = 0;
    ip_header->ip_dst = 0;
    tcp_header->th_sport = 0;
    tcp_header->th_dport = 0;
  } 
  
  SHA1Reset(&sha1_ctx);
  SHA1Input(&sha1_ctx, st_ctx->pw_hash, SHA1HashSize);
  SHA1Input(&sha1_ctx, tmp_pkt, ntohs(ip_header->ip_len));
  SHA1Result(&sha1_ctx, hash);
  
  return(0);
}

/* 
 * ip_id_decrypt takes in a packet and a passphrase hash, determines the
 * SHA-1 hash for that combo, and decrypts the IPID using that stream.
 */

int
ipid_decrypt(uint8_t *packet, struct stegt_ctx *st_ctx, uint8_t *extracted) 
{
  struct ip_hdr *ip_header;
  uint8_t packet_hash[SHA1HashSize];
  uint8_t *ip_id;
  int i;

  if(hash_ipid(packet, st_ctx, packet_hash) != 0) 
    return(-1);

  ip_header = (struct ip_hdr *)packet;
  ip_id = (uint8_t *)&(ip_header->ip_id);
  
  for(i = 0; i < 2; i++) {
    extracted[i] = ip_id[i] ^ packet_hash[i];
  }
  
  return(0);
}

int
hash_sequence(uint8_t *packet, struct stegt_ctx *st_ctx, uint8_t *hash)
{
  SHA1Context sha1_ctx;
  struct ip_hdr *ip_header;
  struct tcp_hdr *tcp_header;
  uint8_t *tmp_pkt;
  int i;

  ip_header = (struct ip_hdr *)packet;
  tcp_header = (struct tcp_hdr *) (packet + (ip_header->ip_hl * 4));

  if ((tmp_pkt = (uint8_t *)calloc(1, ntohs(ip_header->ip_len))) == NULL) {
    fprintf(stderr, "sequence_decrypt: calloc failed\n");
    return(-1);
  }

  memcpy(tmp_pkt, packet, ntohs(ip_header->ip_len));
  ip_header = (struct ip_hdr *)tmp_pkt;
  tcp_header = (struct tcp_hdr *) (tmp_pkt + (ip_header->ip_hl * 4));

  ip_header->ip_ttl = 0;
  ip_header->ip_sum = 0;
  tcp_header->th_sum = 0;
  tcp_header->th_seq = 0;
  tcp_header->th_ack = 0;
  if(st_ctx->nat) {
    ip_header->ip_src = 0;
    ip_header->ip_dst = 0;
    tcp_header->th_sport = 0;
    tcp_header->th_dport = 0;
  }
  
  SHA1Reset(&sha1_ctx);
  SHA1Input(&sha1_ctx, st_ctx->pw_hash, SHA1HashSize);
  SHA1Input(&sha1_ctx, tmp_pkt, ntohs(ip_header->ip_len));
  SHA1Result(&sha1_ctx, hash);
  
  return(0);
}


/*
 * sequence_decrypt takes in a packet and a passphrase, and fills in
 * 4 bytes un-streamciphered from the sequence number.
 */

int
sequence_decrypt(uint8_t *packet, struct stegt_ctx *st_ctx, 
		 uint8_t *extracted) 
{
  struct ip_hdr *ip_header;
  struct tcp_hdr *tcp_header;
  uint8_t packet_hash[SHA1HashSize];
  uint8_t *sequence;
  int i;

  if(hash_sequence(packet, st_ctx, packet_hash) != 0)
    return(-1);

  ip_header = (struct ip_hdr *)packet;
  tcp_header = (struct tcp_hdr *) (packet + (ip_header->ip_hl * 4));

  sequence = (uint8_t *)&(tcp_header->th_seq);

  for(i = 0; i < 4; i++) {
    extracted[i] = sequence[i] ^ packet_hash[i];
  }
  
  return(0);
} 

int
sha_seq_stream(uint8_t *packet, struct stegt_ctx *st_ctx, 
	       uint8_t *cipher_stream)
{
  uint8_t packet_hash[SHA1HashSize];
  int i;

  if(hash_sequence(packet, st_ctx, packet_hash) != 0)
     return(-1);
     
  for(i = 0; i < 4; i++) 
    cipher_stream[i] = packet_hash[i];

  return(0);
}

int
sha_ipid_stream(uint8_t *packet, struct stegt_ctx *st_ctx, 
		uint8_t *cipher_stream)
{
  uint8_t packet_hash[SHA1HashSize];
  int i;

  if(hash_ipid(packet, st_ctx, packet_hash) != 0)
     return(-1);
     
  for(i = 0; i < 2; i++) 
    cipher_stream[i] = packet_hash[i];

  return(0);
}
