/* This source is under the Licence GPL version 2            */
/* It is a scanner for a ASE (3dsmax) file                   */
/* Written with the help of david lecorfec & julien hognon   */
/*        This is a part of Space Racer                      */


%{
/* need this for the call to atof() below */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

char *YYText();

%}





%option outfile="sp_loadase.cpp"
%option stack
%option prefix="ASE"
%option noyywrap


blank       [ \t]+
command     \*[A-Z0-9][A-Z0-9_]*
digit       [0-9]
integer     [-+]?{digit}+
index       {blank}{integer}
context     {command}{index}?{blank}\{
real        [-+]?{digit}+.?{digit}*
comment     [A-Za-z0-9_:,.]*
vertex      {real}{blank}{real}{blank}{real}
id_vtx    {integer}{blank}{vertex}
b_integer_b {blank}{integer}{blank}
mesh_face {integer}:{blank}A:{b_integer_b}B:{b_integer_b}C:{b_integer_b}AB:{b_integer_b}BC:{b_integer_b}CA:{b_integer_b}


%x ASCIIEXPORT

%x SCENE

%x MATERIAL_LIST
%x MATERIAL_COUNT
%x MATERIAL
%x MATERIAL_DIFFUSE
%x MATERIAL_DIFFUSE_COEFF
%x MATERIAL_AMBIENT_COEFF
%x MATERIAL_SPECULAR
%x MATERIAL_SHINE
%x MATERIAL_SHINESTRENGTH
%x MATERIAL_TRANSPARENCY
%x MATERIAL_REF
%x MATERIAL_INDEX

%x GEOMOBJECT
%x MESH
%x MESH_NUMVERTEX
%x MESH_NUMFACES
%x MESH_VERTEX_LIST
%x MESH_VERTEX
%x MESH_FACE_LIST
%x MESH_FACE
%x MESH_NORMALS
%x MESH_FACENORMAL

%x CAMERAOBJECT
%x CAMERA_NODE_TM
%x TM_ROW0
%x TM_ROW1
%x TM_ROW2
%x TM_ROW3
%x CAMERA_SETTINGS
%x CAMERA_TDIST

%x LIGHTOBJECT
%x LIGHT_NODE_TM
%x TM_POS
%x LIGHT_SETTINGS
%x LIGHT_COLOR

%x UNKNOWN_CONTEXT







%%
<INITIAL>{command} {
             if(!strcmp(YYText(), "*3DSMAX_ASCIIEXPORT")) {
               yy_push_state(ASCIIEXPORT);
             }
           }




<INITIAL>{context} { if(!strcmp(YYText(), "*SCENE {")) {
                       yy_push_state(SCENE);
                     }
                     else if(!strcmp(YYText(), "*MATERIAL_LIST {")) {
                       yy_push_state(MATERIAL_LIST);
                     }
                     else if(!strcmp(YYText(), "*GEOMOBJECT {")) {
                       //  definition of a new object
                       // do something if necessary
                       yy_push_state(GEOMOBJECT);
                     }
                     else if(!strcmp(YYText(), "*CAMERAOBJECT {")) {
                       // definition of a new camera
                       // do something if necessary
                       yy_push_state(CAMERAOBJECT);
                     }
                     else if(!strcmp(YYText(), "*LIGHTOBJECT {")) {
                       // definition of a new light
                       // do something if necessary
                       yy_push_state(LIGHTOBJECT);
                     }
                     else {
                       yy_push_state(UNKNOWN_CONTEXT);
                     }
                   }



<ASCIIEXPORT>{integer} { if(atoi(YYText()) == 200) {
                           printf("Version 200 of ASE found\n");
                           yy_pop_state();
                         }
                         else {
                           printf("Version of Ase unknown\n");
                           yyterminate();
                         }
                       }


<SCENE>{context} { yy_push_state(UNKNOWN_CONTEXT); }
<SCENE>\}   { yy_pop_state(); }
<SCENE>.|\n   /* do nothing */


<MATERIAL_LIST>{command} { if(!strcmp(YYText(), "*MATERIAL_COUNT")) {
                            yy_push_state(MATERIAL_COUNT);
                           }
                         }

<MATERIAL_LIST>{context} { if(!strncmp(YYText(), "*MATERIAL", 9)) {
                           // definition of a new material
                             yy_push_state(MATERIAL);
                           }
                           else {
                            yy_push_state(UNKNOWN_CONTEXT);
                           }
                         }
<MATERIAL_LIST>\} { yy_pop_state(); }
<MATERIAL_LIST>.|\n

<MATERIAL_COUNT>{integer} { // reads the number of material
                            int nb;
                            sscanf(YYText(), "%d", &nb);
                            yy_pop_state();
                          }

<MATERIAL>{command} {
                      if (!strcmp(YYText(), "*MATERIAL_DIFFUSE")) {
                       yy_push_state(MATERIAL_DIFFUSE);
                      }
                      else if (!strcmp(YYText(), "*MATERIAL_SPECULAR")) {
                       yy_push_state(MATERIAL_SPECULAR);
                      }
                      else if (!strcmp(YYText(), "*MATERIAL_SHINE")) {
                       yy_push_state(MATERIAL_SHINE);
                      }
                      else if (!strcmp(YYText(), "*MATERIAL_SHINESTRENGTH")) {
                       yy_push_state(MATERIAL_SHINESTRENGTH);
                      }
                      else if (!strcmp(YYText(), "*MATERIAL_TRANSPARENCY")) {
                       yy_push_state(MATERIAL_TRANSPARENCY);
                      }
                    }
<MATERIAL>{context} { yy_push_state(UNKNOWN_CONTEXT); }
<MATERIAL>\} { yy_pop_state(); }
<MATERIAL>.|\n

<MATERIAL_DIFFUSE>{vertex} { // reads the diffusion color
                             double r,g,b;
                             sscanf(YYText(), "%lf %lf %lf", &r, &g, &b);
                             yy_pop_state();
                           }


<MATERIAL_SPECULAR>{vertex} { // reads the specular coeff
                              double r,g,b;
                              sscanf(YYText(), "%lf %lf %lf", &r, &g, &b);
                              yy_pop_state();
                            }
<MATERIAL_SHINE>{real} { // reads the shine coeff
                         double ks;
                         sscanf(YYText(), "%lf", &ks);
                         yy_pop_state();
                       }
<MATERIAL_SHINESTRENGTH>{real} { //reads the shinestrength coeff
                                 double ksh;
                                 sscanf(YYText(), "%lf", &ksh);
                                 yy_pop_state();
                               }

<MATERIAL_TRANSPARENCY>{real} { // reads the transparency coeff
                                double kt;
                                sscanf(YYText(), "%lf", &kt);
                                yy_pop_state();
                              }


<GEOMOBJECT>{command} { if(!strcmp(YYText(), "*MATERIAL_REF")) {
                          yy_push_state(MATERIAL_REF);
                        }
                      }
<GEOMOBJECT>{context} { if(!strcmp(YYText(), "*MESH {")) {
                         yy_push_state(MESH);
                        }
                        else {
                         yy_push_state(UNKNOWN_CONTEXT);
                        }
                      }
<GEOMOBJECT>\} {  // end of the object
                  yy_pop_state();
               }
<GEOMOBJECT>.|\n

<MATERIAL_REF>{integer} { // reads the material assigned to the object
                          int nb;
                          sscanf(YYText(), "%d", &nb);
                          yy_pop_state();
                        }

<MESH>{command} {
                  if(!strcmp(YYText(), "*MESH_NUMVERTEX")) {
                    yy_push_state(MESH_NUMVERTEX);
                  }
                  else if(!strcmp(YYText(), "*MESH_NUMFACES")) {
                    yy_push_state(MESH_NUMFACES);
                  }
                }
<MESH>{context} { if(!strcmp(YYText(), "*MESH_VERTEX_LIST {")) {
                    yy_push_state(MESH_VERTEX_LIST);
                  }
                  else if(!strcmp(YYText(), "*MESH_FACE_LIST {")) {
                    yy_push_state(MESH_FACE_LIST);
                  }
                  else if(!strcmp(YYText(), "*MESH_NORMALS {")) {
                    yy_push_state(MESH_NORMALS);
                  }
                  else {
                    yy_push_state(UNKNOWN_CONTEXT);
                  }
                }
<MESH>\} { yy_pop_state(); }
<MESH>.|\n

<MESH_NUMVERTEX>{integer} { // reads the number of vertex of the object
                            int nb;
                            sscanf(YYText(), "%d", &nb);
                            yy_pop_state();
                          }
<MESH_NUMFACES>{integer}  { // reads the number of faces of the object
                            int nb;
                            sscanf(YYText(), "%d", &nb);
                            yy_pop_state();
                          }

<MESH_VERTEX_LIST>{command} { if(!strcmp(YYText(), "*MESH_VERTEX")) {
                               yy_push_state(MESH_VERTEX);
                              }
                            }
<MESH_VERTEX_LIST>{context} { yy_push_state(UNKNOWN_CONTEXT); }
<MESH_VERTEX_LIST>\} { yy_pop_state(); }
<MESH_VERTEX_LIST>.|\n

<MESH_VERTEX>{id_vtx} { // reads a vertex
                        int i;
                        double x, y, z;
                        sscanf(YYText(), "%d %lf %lf %lf", &i, &x, &y, &z);

                        yy_pop_state();
                      }

<MESH_FACE_LIST>{command} { if(!strcmp(YYText(), "*MESH_FACE")) {
                              yy_push_state(MESH_FACE);
                            }
                          }
<MESH_FACE_LIST>{context} { yy_push_state(UNKNOWN_CONTEXT); }
<MESH_FACE_LIST>\} { yy_pop_state(); }
<MESH_FACE_LIST>.|\n

<MESH_FACE>{mesh_face} { // reads a face
                         int i;
                         unsigned int a, b, c;
                         sscanf(YYText(), "%d: A: %u B: %u C: %u", &i, &a, &b,
&c);
                         yy_pop_state();
                       }

<MESH_NORMALS>{command} { if(!strcmp(YYText(), "*MESH_FACENORMAL")) {
                            yy_push_state(MESH_FACENORMAL);
                          }
                        }
<MESH_NORMALS>{context} { yy_push_state(UNKNOWN_CONTEXT); }
<MESH_NORMALS>\} { yy_pop_state(); }
<MESH_NORMALS>.|\n

<MESH_FACENORMAL>{id_vtx} { // reads the vector normal of the face
                            int i;
                            double a, b, c;
                            sscanf(YYText(), "%d %lf %lf %lf", &i, &a, &b, &c);
                            yy_pop_state();
                          }

<CAMERAOBJECT>{context} { if(!strcmp(YYText(), "*NODE_TM {")) {
                            yy_push_state(CAMERA_NODE_TM);
                          }
                          else if(!strcmp(YYText(), "*CAMERA_SETTINGS {")) {
                            yy_push_state(CAMERA_SETTINGS);
                          }
                          else {
                            yy_push_state(UNKNOWN_CONTEXT);
                          }
                        }
<CAMERAOBJECT>\} {
                   yy_pop_state();
                 }
<CAMERAOBJECT>.|\n

<CAMERA_NODE_TM>{command} { if(!strcmp(YYText(), "*TM_ROW0")) {
                              yy_push_state(TM_ROW0);
                            }
                            else if(!strcmp(YYText(), "*TM_ROW1")) {
                              yy_push_state(TM_ROW1);
                            }
                            else if(!strcmp(YYText(), "*TM_ROW2")) {
                              yy_push_state(TM_ROW2);
                            }
                            else if(!strcmp(YYText(), "*TM_ROW3")) {
                              yy_push_state(TM_ROW3);
                            }
                          }
<CAMERA_NODE_TM>{context} { yy_push_state(UNKNOWN_CONTEXT); }
<CAMERA_NODE_TM>\} { yy_pop_state(); }
<CAMERA_NODE_TM>.|\n

<TM_ROW0>{vertex} { // reads the first row of the transformation matrix
                    double x, y, z;
                    sscanf(YYText(), "%lf %lf %lf", &x, &y, &z);
                    yy_pop_state();
                  }
<TM_ROW1>{vertex} { // reads the second row of the transformation matrix
                    double x, y, z;
                    sscanf(YYText(), "%lf %lf %lf", &x, &y, &z);
                    yy_pop_state();
                  }
<TM_ROW2>{vertex} { // reads the third row of the transformation matrix
                    double x, y, z;
                    sscanf(YYText(), "%lf %lf %lf", &x, &y, &z);
                    yy_pop_state();
                  }
<TM_ROW3>{vertex} { // reads the last row of the transformation matrix
                    double x, y, z;
                    sscanf(YYText(), "%lf %lf %lf", &x, &y, &z);
                    yy_pop_state();
                  }

<CAMERA_SETTINGS>{command} { if(!strcmp(YYText(), "*CAMERA_TDIST")) {
                               yy_push_state(CAMERA_TDIST);
                             }
                           }
<CAMERA_SETTINGS>{context} { yy_push_state(UNKNOWN_CONTEXT); }
<CAMERA_SETTINGS>\} { yy_pop_state(); }
<CAMERA_SETTINGS>.|\n

<CAMERA_TDIST>{real} { // reads the camera focal
                       double d;
                       sscanf(YYText(), "%lf", &d);
                       yy_pop_state();
                     }

<LIGHTOBJECT>{context} { if(!strcmp(YYText(), "*NODE_TM {")) {
                            yy_push_state(LIGHT_NODE_TM);
                          }
                          else if(!strcmp(YYText(), "*LIGHT_SETTINGS {")) {
                            yy_push_state(LIGHT_SETTINGS);
                          }
                          else {
                            yy_push_state(UNKNOWN_CONTEXT);
                          }
                        }
<LIGHTOBJECT>\} {
                  yy_pop_state();
                }
<LIGHTOBJECT>.|\n

<LIGHT_SETTINGS>{command} { if(!strcmp(YYText(), "*LIGHT_COLOR")) {
                              yy_push_state(LIGHT_COLOR);
                            }
                          }
<LIGHT_SETTINGS>{context} { yy_push_state(UNKNOWN_CONTEXT); }
<LIGHT_SETTINGS>\} { yy_pop_state(); }
<LIGHT_SETTINGS>.|\n

<LIGHT_COLOR>{vertex} { // reads the color of the light
                        double r, g, b;
                        sscanf(YYText(), "%lf %lf %lf", &r, &g, &b);
                        yy_pop_state();
                      }

<LIGHT_NODE_TM>{command} { if(!strcmp(YYText(), "*TM_POS")) {
                             yy_push_state(TM_POS);
                           }
                         }
<LIGHT_NODE_TM>{context} { yy_push_state(UNKNOWN_CONTEXT); }
<LIGHT_NODE_TM>\} { yy_pop_state(); }
<LIGHT_NODE_TM>.|\n

<TM_POS>{vertex} { double x, y, z;
                   sscanf(YYText(), "%lf %lf %lf", &x, &y, &z);
                   yy_pop_state();
                 }





<UNKNOWN_CONTEXT>{context} { yy_push_state(UNKNOWN_CONTEXT); }
<UNKNOWN_CONTEXT>\} { yy_pop_state(); }
<UNKNOWN_CONTEXT>.|\n

<*>.|\n
%%


void Parse_ASE(char *filename)
{
     yyin=fopen(filename, "r" );
     yylex();
}

char *YYText()
{
   return yytext;
}

int main()
{
    Parse_ASE("/home/sracer/space-racer/data/sracer.ase");
    return 0;
}
