#include <math.h>
#include "sp_matrice.h"
#include "../util/sp_debug.h"

/* We adopt the OpenGL convention: */
/* a0  a4  a8  a12  */
/* a1  a5  a9  a13  */
/* a2  a6 a10  a14  */
/* a3  a7 a11  a15  */


// *** The constructors ***

// La classe Sp_Math est un tableau de 16 float
Sp_Mat4::Sp_Mat4()
{
  tab=new float[16];
  tab[0]=1; tab[1]=0; tab[2]=0; tab[3]=0;
  tab[4]=0; tab[5]=1; tab[6]=0; tab[7]=0;
  tab[8]=0; tab[9]=0; tab[10]=1;tab[11]=0;
  tab[12]=0;tab[13]=0;tab[14]=0;tab[15]=1;
}

Sp_Mat4::Sp_Mat4(float *t)
{
  int i;
  tab=new float[16];
  for(i=0;i<16;i++)
    {
      tab[i]=t[i];
    }
}

Sp_Mat4::Sp_Mat4(const Sp_Vecteur &i,const Sp_Vecteur &j,const Sp_Vecteur &k,const Sp_Point &O)
{
    tab=new float[16];
    tab[0]=i.x;  tab[4]=j.x;  tab[8]=k.x;  tab[12]=O.x;
    tab[1]=i.y;  tab[5]=j.y;  tab[9]=k.y;  tab[13]=O.y;
    tab[2]=i.z;  tab[6]=j.z;  tab[10]=k.z; tab[14]=O.z;
    tab[3]=0;    tab[7]=0;    tab[11]=0;   tab[15]=1;
}

Sp_Mat4::Sp_Mat4(const Sp_Mat4 &M)
{
    tab=new float[16];
    tab[0]=M.tab[0];  tab[4]=M.tab[4];  tab[8]=M.tab[8];     tab[12]=M.tab[12];
    tab[1]=M.tab[1];  tab[5]=M.tab[5];  tab[9]=M.tab[9];     tab[13]=M.tab[13];
    tab[2]=M.tab[2];  tab[6]=M.tab[6];  tab[10]=M.tab[10];   tab[14]=M.tab[14];
    tab[3]=M.tab[3];  tab[7]=M.tab[7];  tab[11]=M.tab[11];   tab[15]=M.tab[15];
}



Sp_Mat4::~Sp_Mat4()
{
        delete[] tab;
}


Sp_Mat4 &Sp_Mat4::operator=(const Sp_Mat4 &t)
{
  tab[0]=t.tab[0];  tab[1]=t.tab[1];  tab[2]=t.tab[2];  tab[3]=t.tab[3];
  tab[4]=t.tab[4];  tab[5]=t.tab[5];  tab[6]=t.tab[6];  tab[7]=t.tab[7];
  tab[8]=t.tab[8];  tab[9]=t.tab[9];  tab[10]=t.tab[10];tab[11]=t.tab[11];
  tab[12]=t.tab[12];tab[13]=t.tab[13];tab[14]=t.tab[14];tab[15]=t.tab[15];
  return *this;
}

void Sp_Mat4::Id()
{
  tab[0]=1; tab[1]=0; tab[2]=0; tab[3]=0;
  tab[4]=0; tab[5]=1; tab[6]=0; tab[7]=0;
  tab[8]=0; tab[9]=0; tab[10]=1;tab[11]=0;
  tab[12]=0;tab[13]=0;tab[14]=0;tab[15]=1;
}



void Sp_Mat4::Set(float *t)
{
  int i;
  for(i=0;i<16;i++)
    {
      tab[i]=t[i];
    }
}

void Sp_Mat4::Set(const Sp_Vecteur &i,const Sp_Vecteur &j,const Sp_Vecteur &k,const Sp_Point &O)
{
    tab[0]=i.x; tab[4]=j.x; tab[8]=k.x;  tab[12]=O.x;
    tab[1]=i.y; tab[5]=j.y; tab[9]=k.y;  tab[13]=O.y;
    tab[2]=i.z; tab[6]=j.z; tab[10]=k.z; tab[14]=O.z;
    tab[3]=0;   tab[7]=0;   tab[11]=0;   tab[15]=1;
}



void Sp_Mat4::Get(float *t)
{
  int i;
  for(i=0;i<16;i++)
   {
     t[i]=tab[i];
   }
}

void Sp_Mat4::Trans(float x,float y,float z)
{
  tab[0]=1;  tab[4]=0;   tab[8]=0;   tab[12]=x;
  tab[1]=0;  tab[5]=1;   tab[9]=0;   tab[13]=y;
  tab[2]=0;  tab[6]=0;   tab[10]=1;  tab[14]=z;
  tab[3]=0;  tab[7]=0;   tab[11]=0;  tab[15]=1;
}




// matrice de rotation autour de l axe X de l angle r
void	Sp_Mat4::SetX(float r)
{
  tab[0]=1;  tab[4]=0;      tab[8]=0;       tab[12]=0;
  tab[1]=0;  tab[5]=cos(r); tab[9]=-sin(r); tab[13]=0;
  tab[2]=0;  tab[6]=sin(r); tab[10]=cos(r); tab[14]=0;
  tab[3]=0;  tab[7]=0;      tab[11]=0;      tab[15]=1;
}

void	Sp_Mat4::SetY(float s)
{
  tab[0]=cos(s); tab[4]=0;  tab[8]=-sin(s);  tab[12]=0;
  tab[1]=0;      tab[5]=1;  tab[9]=0;        tab[13]=0;
  tab[2]=sin(s); tab[6]=0;  tab[10]=cos(s);  tab[14]=0;
  tab[3]=0;      tab[7]=0;  tab[11]=0;       tab[15]=1;
}

void	Sp_Mat4::SetZ(float t)
{
  tab[0]=cos(t);  tab[4]=-sin(t);  tab[8]=0;  tab[12]=0;
  tab[1]=sin(t);  tab[5]=cos(t);   tab[9]=0;  tab[13]=0;
  tab[2]=0;       tab[6]=0;        tab[10]=1; tab[14]=0;
  tab[3]=0;       tab[7]=0;        tab[11]=0; tab[15]=1;
}


Sp_Mat4 Sp_Mat4::operator*(const Sp_Mat4 &b) const
{
  Sp_Mat4 tmp;

  tmp.tab[0]=tab[0]*b.tab[0] + tab[4]*b.tab[1] + tab[8]*b.tab[2] + tab[12]*b.tab[3];
  tmp.tab[1]=tab[1]*b.tab[0] + tab[5]*b.tab[1] + tab[9]*b.tab[2] + tab[13]*b.tab[3];
  tmp.tab[2]=tab[2]*b.tab[0] + tab[6]*b.tab[1] + tab[10]*b.tab[2]+ tab[14]*b.tab[3];
  tmp.tab[3]=tab[3]*b.tab[0] + tab[7]*b.tab[1] + tab[11]*b.tab[2]+ tab[15]*b.tab[3];
  
  tmp.tab[4]=tab[0]*b.tab[4] + tab[4]*b.tab[5] + tab[8]*b.tab[6]+  tab[12]*b.tab[7];
  tmp.tab[5]=tab[1]*b.tab[4] + tab[5]*b.tab[5] + tab[9]*b.tab[6]+  tab[13]*b.tab[7];
  tmp.tab[6]=tab[2]*b.tab[4] + tab[6]*b.tab[5] + tab[10]*b.tab[6]+ tab[14]*b.tab[7];
  tmp.tab[7]=tab[3]*b.tab[4] + tab[7]*b.tab[5] + tab[11]*b.tab[6]+ tab[15]*b.tab[7];

  tmp.tab[8]=tab[0]*b.tab[8] + tab[4]*b.tab[9] + tab[8]*b.tab[10]+ tab[12]*b.tab[11];
  tmp.tab[9]=tab[1]*b.tab[8] + tab[5]*b.tab[9] + tab[9]*b.tab[10]+ tab[13]*b.tab[11];
  tmp.tab[10]=tab[2]*b.tab[8]+ tab[6]*b.tab[9] + tab[10]*b.tab[10]+tab[14]*b.tab[11];
  tmp.tab[11]=tab[3]*b.tab[8]+ tab[7]*b.tab[9] + tab[11]*b.tab[10]+tab[15]*b.tab[11];

  tmp.tab[12]=tab[0]*b.tab[12]+tab[4]*b.tab[13]+ tab[8]*b.tab[14]+ tab[12]*b.tab[15];
  tmp.tab[13]=tab[1]*b.tab[12]+tab[5]*b.tab[13]+ tab[9]*b.tab[14]+ tab[13]*b.tab[15];
  tmp.tab[14]=tab[2]*b.tab[12]+tab[6]*b.tab[13]+ tab[10]*b.tab[14]+tab[14]*b.tab[15];
  tmp.tab[15]=tab[3]*b.tab[12]+tab[7]*b.tab[13]+ tab[11]*b.tab[14]+tab[15]*b.tab[15];
  
  return tmp;
}


Sp_Vecteur Sp_Mat4::operator*(const Sp_Vecteur &v)const
{
    Sp_Vecteur result;
    result.x=tab[0]*v.x+tab[4]*v.y+tab[8]*v.z;
    result.y=tab[1]*v.x+tab[5]*v.y+tab[9]*v.z;
    result.z=tab[2]*v.x+tab[6]*v.y+tab[10]*v.z;
    return result;
}

Sp_Point Sp_Mat4::operator*(const Sp_Point &p) const
{
    Sp_Point result;
    result.x=tab[0]*p.x+tab[4]*p.y+tab[8]*p.z+tab[12];
    result.y=tab[1]*p.x+tab[5]*p.y+tab[9]*p.z+tab[13];
    result.z=tab[2]*p.x+tab[6]*p.y+tab[10]*p.z+tab[14];
    result.w=tab[3]*p.x+tab[7]*p.y+tab[11]*p.z+tab[15];
    return result;
}


Sp_Mat4 Sp_Mat4::Inverse() const
{
  Sp_Mat4 tmp;
  // on prend la transpose de la mat 3x3
  tmp.tab[0]=tab[0]; tmp.tab[4]=tab[1]; tmp.tab[8]=tab[2];
  tmp.tab[1]=tab[4]; tmp.tab[5]=tab[5]; tmp.tab[9]=tab[6];
  tmp.tab[2]=tab[8]; tmp.tab[6]=tab[9]; tmp.tab[10]=tab[10];
  tmp.tab[3]=0; tmp.tab[7]=0; tmp.tab[11]=0;
  
  
  tmp.tab[12]= -(tab[0]*tab[12]+tab[1]*tab[13]+tab[2]*tab[14])/tab[15];
  tmp.tab[13]= -(tab[4]*tab[12]+tab[5]*tab[13]+tab[6]*tab[14])/tab[15];
  tmp.tab[14]= -(tab[8]*tab[12]+tab[9]*tab[13]+tab[10]*tab[14])/tab[15];
  

  tmp.tab[15]=1/tab[15];
  return tmp;
}

void Sp_Mat4::Print() const
{
  int i;
  printf("une Sp_Mat4:\n");
  for(i=0;i<4;i+=1)
    {
      printf("a%02d:%03f ",i,tab[i]);
      printf("a%02d:%03f ",i+4,tab[i+4]);
      printf("a%02d:%03f ",i+8,tab[i+8]);
      printf("a%02d:%03f\n",i+12,tab[i+12]);
    }
}

/* *************     Quaternion Code  ************* */

Sp_Quat::Sp_Quat()
{}

Sp_Quat::Sp_Quat(float r1,float x1,float y1,float z1)
{
    r=r1;
    x=x1;
    y=y1;
    z=z1;
}

Sp_Quat::Sp_Quat(const Sp_Axis &axe,const float angle)
{
    float thetha=angle/2;
    Sp_Vecteur v= axe.getDirection();
    float tmp= -sin(thetha);
    r=cos(thetha);
    x=tmp*v.x;
    y=tmp*v.y;
    z=tmp*v.z;
}

void Sp_Quat::Null()
{
    r=x=y=z=0;
}

void Sp_Quat::Id()
{
    r=1;
    x=y=z=0;
}

void Sp_Quat::Norme()
{
    float d=ComputeNorme();
    SP_ERROR(d==0,"it is not a good idea to norme a null quaternion\n");
    r/=d;
    x/=d;
    y/=d;
    z/=d;
}

float Sp_Quat::ComputeNorme() const
{
    return sqrt(r*r+x*x+y*y+z+z);
}


void Sp_Quat::ComposedBy(Sp_Axis &,float &angle) const
{
  printf("not yet implemented\n");
}

Sp_Mat4 Sp_Quat::Mat4() const
{
    Sp_Mat4 res;

    float twxx = 2*x*x;
    float twxy = 2*x*y;
    float twxz = 2*x*z;
    
    float twrx = 2*r*x;
    float twry = 2*r*y;
    float twrz = 2*r*z;

    float twyy = 2*y*y;
    float twyz = 2*y*z;
    float twzz = 2*z*z;

    res.tab[0]=1-twyy+twzz;
    res.tab[1]=twxy-twrz;
    res.tab[2]=twxz+twry;
    res.tab[3]=0;

    res.tab[4]=twxy+twrz;
    res.tab[5]=1-(twxx+twzz);
    res.tab[6]=twyz-twrx;
    res.tab[7]=0;

    res.tab[8]=twxz-twry;
    res.tab[9]=twyz+twrx;
    res.tab[10]=1-(twxx+twyy);
    res.tab[11]=0;
    
    res.tab[12]=0; res.tab[13]=0; res.tab[14]=0; res.tab[15]=1;
    return res;
}



Sp_Quat Sp_Quat::Inverse() const
{
    Sp_Quat result;
    float d=ComputeNorme();
    SP_ERROR(d==0,"it is not a good idea to inverse a null quaternion\n");
    result.r=r/d;
    result.x=-(x/d);
    result.y=-(y/d);
    result.z=-(z/d);
    return result;
}

Sp_Quat Sp_Quat::operator*(const Sp_Quat &b) const
{
    Sp_Quat result;
    result.r = r * b.r - ( x * b.x + y * b.y + z * b.z) ;
    result.x = y * b.z -  z * b.y + r * b.x + b.r * x ;
    result.y = z * b.x -  x * b.z + r * b.y + b.r * y ;
    result.z = x * b.y -  y * b.x + r * b.z + b.r * z ;
    return result;
}


void Sp_Quat::Print()
{
    printf("Quaternion r:%f x:%f y:%f z:%f\n",r,x,y,z);
    
}
