/*
 * Copyright (c) 2006 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/queue.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <ctype.h>
#include <netdb.h>
#include <err.h>
#include <syslog.h>
#include <regex.h>

#include "pathnames.h"
#include "smtp-vilter.h"
#include "vilter-regex.h"

#ifdef __linux__
#include <sys/param.h>
#include "strlfunc.h"
#else
#include <sys/syslimits.h>
#endif

struct pat_head hdr_pats;
struct pat_head bdy_pats;

int
vilter_scan(SMFICTX *ctx, char *fn, size_t fnlen, char *chroot, char *reason,
    size_t rlen)
{
	int inhdr = 1;
	FILE *fp;
	struct pattern *pat;
	struct pat_head *head = &hdr_pats;
	char *where = "header";
	char line[1024];

	if (SLIST_EMPTY(&hdr_pats) && SLIST_EMPTY(&bdy_pats))
		return SCAN_OK;

	if ((fp = fopen(fn, "r")) == NULL) {
		syslog(LOG_ERR, "regex: can't open file to scan");
		return SCAN_ERROR;
	}

	while (fgets(line, sizeof(line), fp) != NULL) {
		if (inhdr && line[0] == '\r') {
			inhdr = 0;
			head = &bdy_pats;
			where = "body";
			continue;
		}

		SLIST_FOREACH(pat, head, pats) {
			if (regexec(&pat->preg, line, 0, NULL, 0) == 0) {
				strlcpy(reason, where, rlen);
				strlcat(reason, " pattern: ", rlen);
				strlcat(reason, pat->pat, rlen);
				if (verbose)
					warnx("regex: %s pattern %s matches "
					    "'%s'", where, pat->pat, line);
				fclose(fp);
				return SCAN_UNWANTED;
			}
		}
	}

	fclose(fp);
	
	return SCAN_OK;
}

char *
vilter_name(void)
{
	return "Regular Expression Filter (regex)";
}

int
vilter_type(void)
{
	return BE_SCAN_UNWANTED;
}

void
vilter_exit(void)
{
	struct pattern *p;

	if (verbose)
		warnx("regex: vilter_exit()");

	while (!SLIST_EMPTY(&hdr_pats)) {
		p = SLIST_FIRST(&hdr_pats);
		SLIST_REMOVE_HEAD(&hdr_pats, pats);
		regfree(&p->preg);
		free(p->pat);
		free(p);
	}

	while (!SLIST_EMPTY(&bdy_pats)) {
		p = SLIST_FIRST(&bdy_pats);
		SLIST_REMOVE_HEAD(&bdy_pats, pats);
		regfree(&p->preg);
		free(p->pat);
		free(p);
	}
}
