/*
 * Copyright (c) 2004, 2005, 2006 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/queue.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <ctype.h>
#include <netdb.h>
#include <err.h>
#include <syslog.h>
#include <fnmatch.h>

#include "pathnames.h"
#include "smtp-vilter.h"
#include "vilter-attachment.h"

#ifdef __linux__
#include <sys/param.h>
#include "strlfunc.h"
#else
#include <sys/syslimits.h>
#endif

extern char *tmpdir;

char *attachment_notification;

SLIST_HEAD(parthead, part);
SLIST_HEAD(argshead, arg);

struct pat_head fname_pats;
struct pat_head ctype_pats;

struct part {
	char	*start;
	char	*end;
	int	 skip;
	SLIST_ENTRY(part) parts;
};

struct arg {
	char	*name;
	char	*value;
	int	 name_len;
	int	 value_len;
	SLIST_ENTRY(arg) args;
};

#ifdef FUTURE
char *base64_chars = 
             "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
             "abcdefghijklmnopqrstuvwxyz"
             "0123456789+/";

static char
find_base64(char c)
{
	int r;

	for (r = 0; base64_chars[r] != '\0' && base64_chars[r] != c; r++)
		;	
	return (char)r; 
}

static int
is_base64(unsigned char c) {
	return (isalnum(c) || (c == '+') || (c == '/'));
}
#endif /* FUTURE */

static char *
sol(char *p, char *eom)
{
	/* Find a CR */

	while (p < eom && *p != '\r')
		++p;

	/* Skip a LF, if present */

	if (p < eom && *p == '\r') {
		++p;

		if (p < eom && *p == '\n')
			++p;
	}

	return (p == eom) ? NULL : p++;
}

static size_t
mstrlen(const char *p, const char *eom)
{
	const char *q;

	q = p;
	while (q < eom && *q != '\r')
		++q;

	return q - p;
}

static char *
analyze_header(char *hdr, char *eom, int *name_len, struct argshead *arguments)
{
	char *value;
	int value_len;

	char *p;
	struct arg *a, *la;
	char *eov;

	la = NULL;

	/* warnx("analyze_header()"); */

	for (p = hdr, *name_len = 0; p < eom && *p != '\r' && *p != ':';		    ++*name_len, ++p)
		;

	if (*p == '\r' || p == eom)
		return ++p;	/* XXX */

	for (value = ++p; value < eom && (*value == ' ' || *value == '\t');
	    ++value)
		;
	
	p = value;
	value_len = -1;
	do {
		++value_len;

		while (p < eom && *p != '\r') {
			++value_len;
			++p;
		}
		if (p < eom && *p == '\r') {
			p++;	/* Skip LF */
			++value_len;
		}
		if (p < eom)
			p++;	/* Position at first char of next line */

	} while ((p < eom) && (*p == ' ' || *p == '\t'));

	/* Now split the value part */
	p = value;
	eov = p + value_len;

	do {

		/* Skip whitespace */
		while (p < eov && (*p == ' ' || *p == '\t'))
			++p;

		a = malloc(sizeof(struct arg));

		if (a == NULL) {
			syslog(LOG_ERR, "attachment: unable to allocate memory "
			    "for header");
			/* XXX if this happens we crash */
		}
		bzero(a, sizeof(struct arg));

		a->name = p;

		while (p < eov && *p != '=' && *p != ';' && *p != '\n' &&
		    *p != '\r') {
			++p;
			++a->name_len;
		}

		if (p < eov && *p == '=') {
			++p;

			if (p < eov && *p == '"')
				++p;

			a->value = p;

			while (p < eov && *p != '"' && *p != ';' && *p != '\r'
			    && *p != '\n') {
				++p;
				++a->value_len;
			}
		}		
		if (p < eov && *p == ';')
			++p;
		if (p < eov)
			++p;

		if (la == NULL)
			SLIST_INSERT_HEAD(arguments, a, args);
		else
			SLIST_INSERT_AFTER(la, a, args);

		la = a;
	} while (p < eov);

	if (*eov == '\r')
		++eov;

	return eov == eom ? NULL : eov + 1;	/* Evt. 1 weglassen */
}


static int
analyze_part(struct part *p, char *eom, char *pattern, size_t patlen)
{
	int inhdr;
	char *m;
	char *body;
	char *hdr;
	int name_len;
	struct argshead arguments;
	struct arg *a;
	char *filename, *ctype;
	struct pattern *pat;
	int skip;
	int base64;
#ifdef FUTURE
	char encoded[3], decoded[3];
#endif

	inhdr = 1;
	skip = base64 = 0;

	/* warnx("analyze_part()"); */

	m = p->start;
	do {

		if (mstrlen(m, p->end) == 0) {
			/* warnx("eoh"); */
			inhdr = 0;

			body = sol(m, eom);
#ifdef FUTURE			
			if (base64) {
				/* Need at least 3 bytes */
				if (body < (eom - 3)) {
					warnx("attachment: first two bytes of "
					    "message body are %c %c", *body,
					    *(body + 1));
					encoded[0] = find_base64(*body);
					encoded[1] = find_base64(*(body + 1));
					encoded[2] = find_base64(*(body + 2));

					decoded[0] = (encoded[0] << 2) +
					    ((encoded[1] & 0x30) >> 4);
					decoded[1] = ((encoded[1] & 0xf) << 4) +
					    ((encoded[2] & 0x3c) >> 2);
					decoded[2] = ((encoded[2] & 0x3) << 6) +
					    encoded[3];

					warnx("%c%c%c = %02x %02x", body[0],
					    body[1], body[2], decoded[0],
					    decoded[1]);

					if (decoded[0] == 0xff && decoded[1] ==
					    0xd8) {
						warnx("attachment: found a jpeg"						    " image");
						p->skip = skip = 1;
						strlcpy(pattern, "JPEG IMAGE",
						    patlen);
					}
				}
			}
#endif
		} else {

			SLIST_INIT(&arguments);
			hdr = m;

			m = analyze_header(hdr, eom, &name_len, &arguments);
			/* mprint(hdr, name_len, eom); */

			if (name_len >= strlen("Content-Type") &&
			    !strncmp(hdr, "Content-Type",
			    strlen("Content-Type"))) {
				a = SLIST_FIRST(&arguments);

				if (a->name_len > 0) {
					ctype = malloc(a->name_len + 1);

					if (ctype == NULL)
						syslog(LOG_ERR,"attachment: "
						    "unable to allocate memory "
						    "for content-type");
					strncpy(ctype, a->name, a->name_len);
					ctype[a->name_len] = 0;
					/* warnx("Content-Type: %s", ctype); */

					SLIST_FOREACH(pat, &ctype_pats, pats) {
						if (!fnmatch(ctype, pat->pat, pat->flags)) {
							p->skip = skip = 1;
							strlcpy(pattern,
							    "content-type "
							    "pattern: ",
							    patlen);
							strlcat(pattern,
							    pat->pat, patlen);
							if (verbose)
								warnx("attachment: content-type pattern %s matches content type %s", pat->pat, ctype);
						}
					}
					free(ctype);
				} 
			} else if (name_len >=
			    strlen("Content-Transfer-Encoding") &&
			    !strncmp(hdr, "Content-Transfer-Encoding",
			    strlen("Content-Transfer-Encoding"))) {
				SLIST_FOREACH(a, &arguments, args) {
										
					if (a->name_len >= strlen("base64") &&
					    !strncmp(a->name, "base64",
					    strlen("base64")))
						base64 = 1;

				}
			}
			SLIST_FOREACH(a, &arguments, args) {

				if (a->name_len >= strlen("filename") &&
				    !strncmp(a->name, "filename",
				    strlen("filename")) && a->value != NULL) {
					filename = malloc(a->value_len + 1);

					if (filename == NULL)
						syslog(LOG_ERR, "attachment: "
						    "unable to allocate memory "
						    "for filename");
					strncpy(filename, a->value,
					    a->value_len);
					filename[a->value_len] = 0;
					SLIST_FOREACH(pat, &fname_pats, pats) {
						if (!fnmatch(filename, pat->pat, pat->flags)) {
							p->skip = skip = 1;
							strlcpy(pattern,
							    "filname pattern: ",
							    patlen);
							strlcat(pattern,
							    pat->pat, patlen);
							if (verbose)
								warnx("attachment: filename pattern %s matches filename %s", pat->pat, filename);
						}
					}
					free(filename);
				} 
			}

			while (!SLIST_EMPTY(&arguments)) {
				a = SLIST_FIRST(&arguments);
				SLIST_REMOVE_HEAD(&arguments, args);
				free(a);
			}
		}		
	} while (m != NULL && inhdr);

	return skip;	
}

int
vilter_scan(SMFICTX *ctx, char *fn, size_t fnlen, char *chroot, char *reason,
    size_t rlen)
{
	char *msg;
	int fd;
	struct stat statbuf;
	struct part *p;
	struct part *lp;
	struct parthead parts;
	int lines;
	char *m, *q;
	char *som;	/* Start of msg, point to first byte of mmapped file */
	char *eom;	/* End of msg, points to last byte of mmapped file */
	char *body;
	char *boundary;
	int boundary_len;
	long size;
	int inhdr;
	int skip;
	char newfn[PATH_MAX];
	int newfd;
	FILE *fp;
	char buf[1024];
	int nread;
	int retval;

	p = lp = NULL;
	boundary = NULL;
	lines = skip = boundary_len = 0;
	inhdr = 1;
	retval = SCAN_OK;

	if (SLIST_EMPTY(&fname_pats) && SLIST_EMPTY(&ctype_pats))
		return SCAN_OK;

	if (stat(fn, &statbuf)) {
		syslog(LOG_ERR, "attachment: can't stat file to scan (%s %s)",
		    chroot, fn);
		return SCAN_ERROR;
	}	

	if ((fd = open(fn, O_RDONLY, 0)) == -1) {
		syslog(LOG_ERR, "attachment: can't open file to scan");
		return SCAN_ERROR;
	}

	if ((msg = mmap(0, statbuf.st_size, PROT_READ, MAP_PRIVATE | MAP_FILE,
	    fd, (off_t)0L)) == MAP_FAILED) {
		syslog(LOG_ERR, "attachment: can't mmap file");
		close(fd);
		return SCAN_ERROR;
	}

	SLIST_INIT(&parts);

	size = (long)statbuf.st_size;
	som = msg;
	eom = som + size - 1;	/* eom points to the last byte */

	m = som;

	do {
		if (inhdr == 1) {
			if (strncmp(m, "Content-Type: ",
			    strlen("Content-Type: ")) == 0) {
				for (q = m; q < eom && isspace((int)*q) == 0;
				    q++)
					;
				while (q < eom && isspace((int)*q))
					q++;

				if (strncmp(q, "multipart",
				    strlen("multipart")) == 0) {
					
					boundary = q;
					while (boundary < (eom - strlen("boundary=")) && strncmp(boundary, "boundary=", strlen("boundary=")))
						boundary++;
					while (boundary < eom && *boundary !=
					    '=')
						++boundary;
					if (boundary < eom)
						++boundary;

					if (boundary < eom && *boundary == '"')
						++boundary;

					/* See RFC-2046 for characters allowed in a boundary */

					for (q = boundary, boundary_len = 0; q < eom && (
						isalpha((int)*q) || isdigit((int)*q) ||
						*q == '\'' || *q == '(' || *q == ')' ||
						*q == '+' || *q == '_' || *q == ',' ||
						*q == '-' || *q == '.' || *q == '/' ||
						*q == ':' || *q == '=' || *q == '?'); q++, boundary_len++)
						;
				}
			}
		} else if (boundary != NULL && mstrlen(m, eom) >= boundary_len + 2) {
			if (strncmp(m, "--", 2))
				goto line;

			if (strncmp(m + 2, boundary, boundary_len))
				goto line;

			/*
			 * p->end points to the last byte of the part, so
			 * to write the complete part, one has to write
			 * p->end - p->start + 1 bytes.
			 */
			if (p != NULL) {
				p->end = m - 1;
				lp = p;
			}

			if ((mstrlen(m, eom) >= boundary_len + 4) &&
			    !strncmp(m + 2 + boundary_len, "--", 2)) {
				/* end of message */ ;
			} else {
				p = malloc(sizeof(struct part));
				if (p == NULL) {
					syslog(LOG_ERR, "attachment: memory allocation error, not all message parts will be written");
					goto line;
				}
				bzero(p, sizeof(struct part));
				if ((p->start = sol(m, eom)) != NULL) {
					if (lp == NULL)
						SLIST_INSERT_HEAD(&parts, p, parts);
					else
						SLIST_INSERT_AFTER(lp, p, parts);
				} else {
					free(p);
					p = NULL;
				}
			}
		}
line:
		++lines;

		if (mstrlen(m, eom) == 0 && inhdr == 1) {
			inhdr = 0;
			body = sol(m, eom);
		}

		m = sol(m, eom);
	} while (m != NULL);

	if (p != NULL && p->end == 0)
		p->end = eom;

	SLIST_FOREACH(p, &parts, parts)
		skip += analyze_part(p, eom, reason, rlen);

	if (verbose)
		warnx("attachment: will skip %d attachments", skip);

	if (skip <= 0)
		goto endproc;

	strlcpy(newfn, tmpdir, sizeof(newfn));
	strlcat(newfn, "/vilter-attachment.XXXXXXXXXX", sizeof(newfn));
	if ((newfd = mkstemp(newfn)) == -1) {
		syslog(LOG_ERR, "unable to create temp file");
		retval = SCAN_ERROR;
		goto endproc;
	}

	write(newfd, "This is a multi-part message in MIME format.\r\n\r\n",
	    strlen("This is a multi-part message in MIME format.\r\n\r\n"));
	write(newfd, "--", 2);
	write(newfd, boundary, boundary_len);
	write(newfd, "\r\n", 2);
	write(newfd, "Content-Type: text/plain\r\n\r\n",
	    strlen("Content-Type: text/plain\r\n\r\n"));
		
	if (attachment_notification != NULL
	    && (fp = fopen(attachment_notification, "r")) != NULL) {
		while ((nread = fread(buf, 1, sizeof(buf), fp)))
			write(newfd, buf, nread);
		fclose(fp);
	} else
		write(newfd, "Vaporized an unwelcome attachment"
		    "\r\n\r\n",
		    strlen("Vaporized an unwelcome attachment"
		    "\r\n\r\n"));

	SLIST_FOREACH(p, &parts, parts) {
		if (p->skip == 0 && (p->end - p->start) > 0) {
			write(newfd, "--", 2);
			write(newfd, boundary, boundary_len);
			write(newfd, "\r\n", 2);
			write(newfd, p->start, p->end - p->start + 1);
		} 
	}
	write(newfd, "--", 2);
	write(newfd, boundary, boundary_len);
	write(newfd, "--\r\n", 4);
	close(newfd);
	strlcpy(fn, newfn, fnlen);

endproc:
	if (munmap(msg, statbuf.st_size)) {
		syslog(LOG_ERR, "attachment: can't munmap file");
		/* memory leak */
		return SCAN_ERROR;
	}

	close(fd);
	
	while (!SLIST_EMPTY(&parts)) {
		p = SLIST_FIRST(&parts);
		SLIST_REMOVE_HEAD(&parts, parts);
		free(p);
	}

	return SCAN_OK;
}

char *
vilter_name(void)
{
	return "Attachment Filter (attachment)";
}

int
vilter_type(void)
{
	return BE_SCAN_UNWANTED;
}

void
vilter_exit(void)
{
	struct pattern *p;

	if (verbose)
		warnx("attachment: vilter_exit()");

	while (!SLIST_EMPTY(&fname_pats)) {
		p = SLIST_FIRST(&fname_pats);
		SLIST_REMOVE_HEAD(&fname_pats, pats);
		free(p->pat);
		free(p);
	}

	while (!SLIST_EMPTY(&ctype_pats)) {
		p = SLIST_FIRST(&ctype_pats);
		SLIST_REMOVE_HEAD(&ctype_pats, pats);
		free(p->pat);
		free(p);
	}

	free(attachment_notification);
}
