%{
/*
 * Copyright (c) 2005 - 2013 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/queue.h>

#include <err.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <fnmatch.h>

#include "smtp-vilter.h"
#include "pathnames.h"
#include "vilter-attachment.h"

extern FILE	*attachmentin;
extern int	 attachmentlineno;
extern char	*attachmenttext;

extern int	verbose;

extern struct pat_head fname_pats;
extern struct pat_head ctype_pats;

extern char	*attachment_notification;

static struct pattern *p;
static int	cflags;

static int	 attachmenterrcnt;
static char	*attachmentcfgfile;

int	attachmenterror(const char *, ...);
int	attachmentparse(void);
int	attachmentlex(void);

static struct pattern *
compile_pattern(char *value, int cflags)
{
	struct pattern *p;

	p = malloc(sizeof(struct pattern));
	if (p == NULL)
		errx(1, "attachment: out of memory");
	bzero(p, sizeof(struct pattern));
	p->pat = strdup(value);
	if (p->pat == NULL)
		errx(1, "attachment: out of memory");
	p->flags = cflags;
	return p;
}

%}

%union {
	long number;
	char *string;
}

%token	CASE_SENSITIVE UNWANTED_FILENAME UNWANTED_CONTENT NOTIFICATION
%token	TRUE FALSE
%token	<string>	NUMBER
%token	<string>	TEXT

%%
statement	: /* empty */
		| statement '\n'
		| statement case_sensitive '\n'
		| statement unwanted_filename '\n'
		| statement unwanted_content '\n'
		| statement notification '\n'
		;

case_sensitive		: CASE_SENSITIVE '=' TRUE		{
				cflags &= ~FNM_CASEFOLD;
			}
			| CASE_SENSITIVE '=' FALSE		{
				cflags |= FNM_CASEFOLD;
			}
			;

unwanted_filename	: UNWANTED_FILENAME '=' TEXT		{
				p = compile_pattern($3, cflags);
				SLIST_INSERT_HEAD(&fname_pats, p, pats);
				if (verbose)
					warnx("attachment: adding unwanted "
					    "filename pattern %s", $3);

				free($3);
			}
			;

unwanted_content	: UNWANTED_CONTENT '=' TEXT		{
				p = compile_pattern($3, cflags);
				SLIST_INSERT_HEAD(&ctype_pats, p, pats);
				if (verbose)
					warnx("attachment: adding unwanted "
					    "content-type pattern %s", $3);

				free($3);
			}
			;

notification		: NOTIFICATION '=' TEXT			{
				attachment_notification = $3;
			}
			;

%%

int
vilter_init(char *cfgfile)
{
	cflags = FNM_NOESCAPE | FNM_PATHNAME | FNM_PERIOD | FNM_LEADING_DIR
	    | FNM_CASEFOLD;
	attachment_notification = NULL;

	if (verbose)
		warnx("attachment: vilter_init()");

	SLIST_INIT(&fname_pats);
	SLIST_INIT(&ctype_pats);

	if (cfgfile == NULL)
		cfgfile = ATTACHMENT_CONF;

	attachmentlineno = 1;
	attachmentcfgfile = cfgfile;
	attachmenterrcnt = 0;
	if ((attachmentin = fopen(cfgfile, "r")) != NULL) {
		if (verbose)
			warnx("attachment: using configuration from file %s",
			    cfgfile);

		while (!feof(attachmentin))
			attachmentparse();
		fclose(attachmentin);
	} else if (verbose)
		warnx("attachment: configuration file %s for attachment backend"
		    " not found, using default values", cfgfile);

	if (attachmenterrcnt)
		errx(1, "configuration file contains errors, terminating");
	if (verbose)
		warnx("attachment: vilter_init() return");
	return 0;
}

int
attachmenterror(const char *fmt, ...)
{
	va_list		 ap;
	char		*nfmt;

	++attachmenterrcnt;
	va_start(ap, fmt);
	if (asprintf(&nfmt, "%s, line %d: %s near '%s'", attachmentcfgfile,
	    attachmentlineno, fmt, attachmenttext) == -1)
		errx(1, "attachment: asprintf failed");
	va_end(ap);
	fprintf(stderr, "%s\n", nfmt);
	free(nfmt);
	return 0;
}
