/* sdig.c - the Switch Digger main file

   Copyright (C) 2000  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h> 
#include <string.h> 
#include <unistd.h> 
#include <arpa/inet.h> 
#include <netinet/in.h> 
#include <sys/socket.h>

#include "sdig.h"
#include "common.h"
#include "snmpget.h"
#include "version.h"

#define CONFIGFILE "/etc/sdig.conf"

	stype	*firstsw = NULL;
	rtype	*firstrt = NULL;
	pdtype	*firstpd = NULL;
	litype	*firstli = NULL;

	char	*wins = NULL, *nbname = NULL, *nmblookup = NULL, 
		*mactable = NULL;

	int	debuglevel = 0, verbose = 0, fastmode = 0;

/* debug levels:
 *
 * 1 - basic entry messages
 * 2 - bit more detailed findings in the scan functions
 * 5 - messy stuff involving snmpget 
 * 6 - ugly text pointers
 */

void debug(int level, char *format, ...)
{
	va_list	args;

	if (debuglevel < level)
		return;

	va_start(args, format);
	vprintf(format, args);
	va_end(args);
}

static char *findmac(char *ip, rtype *rtr)
{
	char	query[256], *ret;
	int	ifnum;

	debug(1, "\n\nfindmac: [%s] [%s] [%s]\n", ip, rtr->ip, rtr->pw);

	/* find the router's internal interface number */

	snprintf(query, sizeof(query), 
		"ip.ipAddrTable.ipAddrEntry.ipAdEntIfIndex.%s", rtr->ip);

	ifnum = snmpget_int(rtr->ip, rtr->pw, query);

	if (ifnum == -1)
		return NULL;

	debug(5, "ifnum is [%d]\n", ifnum);

	/* now look it up in the net to media table relative to the ifnum */

	/* if digging the router itself, use a different OID */

	if (!strcmp(ip, rtr->ip))
		snprintf(query, sizeof(query), 
			"interfaces.ifTable.ifEntry.ifPhysAddress.%d",
			ifnum);
	else
		snprintf(query, sizeof(query), 
		"ip.ipNetToMediaTable.ipNetToMediaEntry.ipNetToMediaPhysAddress.%d.%s",
		ifnum, ip);

	ret = snmpget_mac(rtr->ip, rtr->pw, query);

	return ret;
}

static int findport(unsigned char *mac, stype *sw)
{
	char	query[64];

	if (sw->ip == NULL) {
		printf("No switch defined for that network\n");
		exit(1);
	}

	/* build the OID for the mapping of MAC addresses to port numbers */

	snprintf(query, sizeof(query), "17.4.3.1.2.%u.%u.%u.%u.%u.%u",
		mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

	debug(5, "findport: query is [%s]\n", query);

	return snmpget_int(sw->ip, sw->pw, query);
}

/* ROUTER <netblock> <ip> <pw> <"desc"> */
static void addrouter(char *block, char *ip, char *pw, char *desc)
{
	rtype	*tmp, *last;
	char	*addr, *mask;

	mask = strchr (block, '/');

	if (!mask)
		return;

	*mask++ = '\0';
	addr = block;

	tmp = last = firstrt;

	while (tmp != NULL) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = malloc (sizeof(rtype));
	tmp->addr = ntohl(inet_addr (addr));
	tmp->pw = strdup(pw);
	tmp->ip = strdup(ip);
	tmp->desc = strdup(desc);
	tmp->next = NULL;

	if (strstr(mask, ".") == NULL) { /* must be a /nn CIDR type block */
		if (atoi(mask) != 32)
			tmp->mask = ((unsigned int) ((1 << atoi(mask)) - 1) <<
				(32 - atoi(mask)));
		else
			tmp->mask = 0xffffffff; /* avoid overflow from 2^32 */
	}
	else
		tmp->mask = ntohl(inet_addr(mask));

	if (last != NULL)
		last->next = tmp;
	else
		firstrt = tmp;
}

/* SWITCH <netblock> <ip> <community> ["<desc>"] */
static void addswitch(char *block, char *ip, char *pw, char *desc)
{
	stype	*tmp, *last;
	char	*addr, *mask;

	mask = strchr(block, '/');

	if (!mask)
		return;

	*mask++ = '\0';
	addr = block;

	tmp = last = firstsw;

	while (tmp != NULL) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = malloc(sizeof(stype));
	tmp->addr = ntohl(inet_addr(addr));
	tmp->pw = strdup(pw);
	tmp->ip = strdup(ip);
	tmp->desc = strdup(desc);
	tmp->firstlink = NULL;
	tmp->next = NULL;

	if (strstr(mask, ".") == NULL) {  /* must be a /nn CIDR type block */
		if (atoi(mask) != 32)
			tmp->mask = ((unsigned int) ((1 << atoi(mask)) - 1) <<
				(32 - atoi(mask)));
		else
			tmp->mask = 0xffffffff; /* avoid overflow from 2^32 */
	}
	else
		tmp->mask = ntohl(inet_addr(mask));

	if (last != NULL)
		last->next = tmp;
	else
		firstsw = tmp;
}

/* LINKINFO <ip> <port> "<desc>" */
static void addli(char *ip, char *port, char *desc)
{
	litype	*tmp, *last;

	tmp = last = firstli;

	while (tmp) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = malloc(sizeof(litype));
	tmp->ip = strdup(ip);
	tmp->port = strtol(port, (char **) NULL, 10);
	tmp->desc = strdup(desc);

	if (last)
		last->next = tmp;
	else
		firstli = tmp;
}

/* PORTDESC <ip> <port> "<desc>" */
static void addpd(char *ip, char *port, char *desc)
{
	pdtype	*last, *tmp;

	tmp = last = firstpd;

	while (tmp) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = malloc(sizeof(pdtype));
	tmp->ip = strdup(ip);
	tmp->port = strtol(port, (char **) NULL, 10);
	tmp->desc = strdup(desc);

	if (last)
		last->next = tmp;
	else
		firstpd = tmp;
}

static void loadconfig(char *fn)
{
	FILE	*conf;
	char	buf[256], *arg[5];
	int	ln, i;

	if (fn)
		conf = fopen(fn, "r");
	else
		conf = fopen(CONFIGFILE, "r");

	if (!conf) {
		if (fn)
			fprintf(stderr, "fopen %s: %s\n", fn, strerror(errno));
		else
			fprintf(stderr, "fopen %s: %s\n", CONFIGFILE, 
				strerror(errno));
		exit(1);
	}

	ln = 0;
	while (fgets(buf, sizeof(buf), conf)) {
		buf[strlen(buf) - 1] = '\0';
		ln++;

		i = parseconf("sdig.conf", ln, buf, arg, 5);

		if (i == 0)
			continue;

		if (!strcmp(arg[0], "ROUTER"))
			addrouter(arg[1], arg[2], arg[3], arg[4]);
		if (!strcmp(arg[0], "SWITCH"))
			addswitch(arg[1], arg[2], arg[3], arg[4]);
		if (!strcmp(arg[0], "LINKINFO"))
			addli(arg[1], arg[2], arg[3]);
		if (!strcmp(arg[0], "PORTDESC"))
			addpd(arg[1], arg[2], arg[3]);
		if (!strcmp(arg[0], "WINS"))
			wins = strdup(arg[1]);
		if (!strcmp(arg[0], "NBNAME"))
			nbname = strdup(arg[1]);
		if (!strcmp(arg[0], "NMBLOOKUP"))
			nmblookup = strdup(arg[1]);
		if (!strcmp(arg[0], "MACTABLE"))
			mactable = strdup(arg[1]);
	}

	fclose(conf);
}

char *getlink(char *ip, long port)
{
	litype	*tmp;

	tmp = firstli;

	while (tmp) {
		if ((!strcmp(tmp->ip, ip)) && (tmp->port == port))
			return tmp->desc;

		tmp = tmp->next;
	}

	return NULL;
}

static char *getdesc(char *ip, long port)
{
	pdtype	*tmp;

	tmp = firstpd;

	while (tmp) {
		if ((!strcmp(tmp->ip, ip)) && (tmp->port == port))
			return tmp->desc;

		tmp = tmp->next;
	}

	return NULL;
}

static char *macmfr(unsigned char *inmac)
{
	FILE	*macdb;
	char	buf[256], *tmp, macfind[16];
	int	i;

	macdb = fopen(mactable, "r");
	if (!macdb)
		return "MAC table file not available";

	/* rewrite the MAC address into something that'll match the table */

	snprintf(macfind, sizeof(macfind), "%02x %02x %02x", 
		inmac[0], inmac[1], inmac[2]);

	while (fgets(buf, sizeof(buf), macdb)) {
		buf[strlen(buf) - 1] = '\0';

		if (!strncasecmp(buf, macfind, 8)) {
			tmp = strdup(&buf[9]);
			for (i = strlen(tmp) - 1; i >= 0; i--) {
				if (!isspace(tmp[i])) {
					tmp[i+1] = '\0';
					return tmp;
				}
			}
			return tmp;
		}
	}

	fclose(macdb);
	return "Not available";
}

static void netbiosreverse(char *ip)
{
	char	exec[256], buf[256], *arg[9];
	FILE	*nb;
	int	ret;

	/* XXX: nbname is evil, and we should be using nmblookup */

	if (!nbname)
		return;

	snprintf(exec, sizeof(exec), "%s %s 2>/dev/null", nbname, ip);

	debug(5, "popen: %s\n", exec);
	nb = popen(exec, "r");

	bzero(buf, sizeof(buf));
	fgets(buf, sizeof(buf), nb);
	buf[strlen(buf) - 1] = '\0';
	debug(5, "read [%s]\n", buf);
	pclose(nb);

	if (strncmp(buf, "NetBIOS name of", 15) != 0)
		return;

	/* NetBIOS name of <IP> is <NAME> [(in workgroup <WG>]) */

	ret = parseconf(NULL, 0, buf, arg, 9);

	if (ret == 0)
		return;

	/* NULL return means nbname failed */
	if (!arg[5])
		return;

	printf(" Hostname: %s (NetBIOS query)\n", arg[5]);

	if (arg[8]) {
		arg[8][strlen(arg[8])-1] = '\0';	/* lose trailing ) */
		printf("Workgroup: %s\n", arg[8]);
	}

	return;
}

static void help(char *prog)
{
	printf("SNMP-based router and switch probe for locating client systems.\n\n");
	printf("usage: %s [-d] [-f <config>] [-m <MAC>] [-v] [-F] (<IP> | <hostname>)\n", prog);
	printf("\n");
	printf("  -d		- increase debug level\n");
	printf("  -F		- fast mode - no DNS/NetBIOS reverse lookups\n");	
	printf("  -f <config>	- use alternate config <config>, default %s\n", CONFIGFILE);
	printf("  -m <MAC>	- force MAC <MAC>, xx:xx:xx:xx:xx:xx format\n");
	printf("  -v		- be verbose\n");
	printf("  <IP>		- IP address to find\n");
	printf("  <hostname>	- DNS/WINS hostname to find\n");

	exit(1);
}

static char *wins_resolve(char *host)
{
	char	exec[256], buf[256];
	FILE	*wq;

	if (!wins) {
		fprintf(stderr, "WINS not defined in config file!\n");
		return NULL;
	}

	if (!nmblookup) {
		fprintf(stderr, "NMBLOOKUP not defined in config file!\n");
		return NULL;
	}

	snprintf(exec, sizeof(exec), "%s -U %s -R %s | tail -1 | cut -f 1 -d \" \"",
		nmblookup, wins, host);

	debug(5, "popen: %s\n", exec);
	wq = popen(exec, "r");

	fgets(buf, sizeof(buf), wq);
	pclose(wq);

	buf[strlen(buf) - 1] = '\0';
	debug(5, "read [%s]\n", buf);
	if (!strcmp(buf, "name_query")) {
		fprintf(stderr, "WINS lookup failed\n");
		exit(1);
	}

	printf("  Address: %s (WINS)\n", buf);

	return(strdup(buf));
}

static char *dns_resolve(char *host)
{
	struct	hostent	*dns;
	struct	in_addr	addr;

	if ((dns = gethostbyname(host)) == (struct hostent *) NULL)
		return NULL;

	memcpy(&addr, dns->h_addr, dns->h_length);

	printf("  Address: %s (DNS)\n", inet_ntoa(addr));

	return(strdup(inet_ntoa(addr)));
}

static void do_ifdescr(stype *sw, long port)
{
	char	query[256], *ifdescr;
	long	ifnum;

	/* first get the switch's ifnum for the port */
	
	snprintf(query, sizeof(query), "17.1.4.1.2.%ld", port);
	ifnum = snmpget_int(sw->ip, sw->pw, query);

	if (ifnum == -1)
		return;

	snprintf(query, sizeof(query), "interfaces.ifTable.ifEntry.ifDescr.%ld",
		ifnum);

	ifdescr = snmpget_str(sw->ip, sw->pw, query);

	if (ifdescr) {
		printf(" (%s)", ifdescr);
		free(ifdescr);
	}
}

static void printport(stype *sw, long port)
{
	char	*ds, *li;

	/* don't print if it's a switch-switch link unless in verbose mode */

	li = getlink(sw->ip, port);

	if ((li) && (!verbose))
		return;

	printf("   Switch: %s - %s\n", sw->desc, sw->ip);


	printf("     Port: %ld", port);
	do_ifdescr(sw, port);
	printf("\n");
	
	if (li)
		printf("     Link: %s\n", li);

	ds = getdesc(sw->ip, port);
	if (ds)
		printf("     Info: %s\n", ds);

	printf("\n");
}

static int isip(char *buf)
{
	int	i;

	for (i = 0; i < strlen(buf); i++)
		if ((!isdigit(buf[i])) && (buf[i] != '.'))
			return 0;

	return 1;
}

static void dnsreverse(char *ip)
{
	struct	hostent	*dns;
	struct	in_addr	addr;

	inet_aton(ip, &addr);

	dns = gethostbyaddr((char *)&addr, sizeof(struct in_addr), AF_INET);

	if (dns)
		printf(" Hostname: %s (DNS)\n", dns->h_name);
}

static stype *find_switch(char *ipaddr, stype *last)
{
	stype	*tmp;
	int	addrchk, swchk;

	if (last)
		tmp = last->next;
	else
		tmp = firstsw;

	while (tmp) {
		addrchk = ntohl(inet_addr(ipaddr)) & tmp->mask;
		swchk = tmp->addr & tmp->mask;

		if (swchk == addrchk)
			return tmp;

		tmp = tmp->next;
	}

	return NULL;
}

/* ask the switch about where the MAC address is */
static void switchscan(char *ipaddr, unsigned char *macaddr)
{
	stype	*sw;
	long	port;

	printf("\n");

	debug(1, "switchscan: seeking %s\n", ipaddr);

	sw = find_switch(ipaddr, NULL);

	while (sw) {
		debug(2, "switchscan: matched %s\n", sw->ip);

		port = findport(macaddr, sw);

		if (port != -1)
			printport(sw, port);

		sw = find_switch(ipaddr, sw);
	}

	exit(0);
}

static rtype *find_router(char *ipaddr, rtype *last)
{
	rtype	*tmp;
	int	addrchk, rtchk;

	if (last)
		tmp = last->next;
	else
		tmp = firstrt;

	while (tmp) {
		addrchk = ntohl(inet_addr(ipaddr)) & tmp->mask;
		rtchk = tmp->addr & tmp->mask;

		if (rtchk == addrchk)
			return tmp;

		tmp = tmp->next;
	}

	return NULL;
}

/* make the octet string into something nicer for humans */
static void printmac(unsigned char *mac)
{
	int	i;

	for (i = 0; i < 5; i++)
		printf("%02x:", mac[i]);

	printf("%02x", mac[5]);
}

/* walk the list of routers checking for the IP address */
static void routerscan(char *ipaddr)
{
	unsigned char	*macaddr;
	rtype	*rtr;

	/* spew out some additional info about the IP address */
	if (fastmode == 0) {
		dnsreverse(ipaddr);
		netbiosreverse(ipaddr);
	}

	printf("\n");

	debug(1, "routerscan: seeking %s\n", ipaddr);

	/* XXX: ping code for waking up sleeping/inactive hosts */

	/* find the first one that covers this network */
	rtr = find_router(ipaddr, NULL);

	while (rtr) {
		debug(2, "routerscan: matched %s\n", rtr->ip);

		/* try to find the target IP address on this router */
		macaddr = findmac(ipaddr, rtr);

		if (macaddr) {
			printf("   Router: %s - %s\n", rtr->desc, rtr->ip);

			printf("      MAC: ");
			printmac(macaddr);
			printf(" (%s)\n", macmfr(macaddr));

			switchscan(ipaddr, macaddr);
		}

		rtr = find_router(ipaddr, rtr);
	}

	fprintf(stderr, "Error: no routers found for %s\n", ipaddr);
	exit(1);
}	

/* turn <name> into an IP address and pass it to the router scanner */
static void resolvename(char *name)
{
	char	*ipaddr;

	/* first try DNS */
	ipaddr = dns_resolve(name);

	if (ipaddr)
		routerscan(ipaddr);

	/* now try WINS */
	ipaddr = wins_resolve(name);

	if (ipaddr)
		routerscan(ipaddr);

	fprintf(stderr, "Can't resolve %s with DNS or WINS!\n", name);
	exit(1);
}

/* see if the specified mac address is sane */
static void checkmac(char *buf)
{
	int	i;

	for (i = 0; i < strlen(buf); i++)
		if ((!isxdigit(buf[i])) && (buf[i] != ':')) {
			fprintf(stderr, "Invalid MAC address specified: %s\n", buf);
			fprintf(stderr, "Valid characters are hex digits and :\n");
			exit(1);
		}
}

int main(int argc, char **argv)
{
	char	*prog, *query, *conf = NULL, *mac = NULL;
	int	i;

	printf("Switch Digger %s - http://www.exploits.org/sdig/\n\n",
		VERSION);

	prog = argv[0];

	while ((i = getopt(argc, argv, "+dhf:m:vF")) != EOF) {
		switch (i) {
			case 'd':
				debuglevel++;
				break;

			case 'f':
				conf = optarg;
				break;

			case 'h':
				help(prog);
				break;

			case 'm':
				mac = optarg;
				break;

			case 'v':
				verbose++;
				break;

			case 'F':
				fastmode = 1;
				break;
				
			default:
				help(prog);
		}
	}

	argc -= optind;
	argv += optind;

	if (argc < 1)
		help(prog);

	query = argv[0];

	loadconfig(conf);

	/* split off to resolve things based on what kind of input we got */

	/* hostname (DNS or WINS) given */
	if (!isip(query)) {
		printf("    Query: %s\n", query);
		resolvename(query);

		/* NOTREACHED */
	}

	/* MAC address specified, along with target network */
	if ((mac) && (isip(query))) {
		printf("    Query: %s in network %s\n", 
			mac, query);

		checkmac(mac);
		switchscan(query, mac);

		/* NOTREACHED */
	}

	/* just an IP address given */
	if (isip(query)) {
		printf("    Query: %s\n", query);
		routerscan(query);

		/* NOTREACHED */
	}

	/* unknown! */
	fprintf(stderr, "Error: unknown query type!\n");
	exit(1);
}
