/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdarg.h>
#include <signal.h>
#include <glib.h>
#include <gtk/gtk.h>
#include "sg_project_statusbar.h"
#include "sg_main.h"

#define TEXT_TIMEOUT     3.0 /* in seconds */
#define PROGRESS_TIMEOUT 0.1 /* in seconds */

static gboolean text_clear (gpointer data);

static GtkWidget *statusbar;
static GtkWidget *progressbar;

static gint context_text_id;
static gint context_progress_id;

static GTimer *progress_timer = NULL;


GtkWidget*
sg_project_statusbar_new(GtkWidget *app, GtkWidget *vbox)
{
 /* app is used by Gnome, vbox used by Gtk
  * this construction is not so nice, but I
  * can't come up with anything better. -Rob- */

 GtkWidget *appbar;

#ifdef __WITH_GNOME
 appbar = gnome_appbar_new(TRUE, TRUE, FALSE);
 gnome_app_set_statusbar(GNOME_APP(app), appbar);
#else
 GtkAdjustment *adj;

 /* build the status bar */
 statusbar = gtk_statusbar_new();
 context_text_id = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar), "text");
 context_progress_id = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar), "progress");
 gtk_widget_show(statusbar);

 /* build the progress bar */
 adj = (GtkAdjustment*)gtk_adjustment_new (0, 0, 1, 0, 0, 0);
 progressbar = gtk_progress_bar_new_with_adjustment(adj);
 gtk_progress_set_show_text(GTK_PROGRESS(progressbar), FALSE);
 gtk_widget_show(progressbar);
 progress_timer = g_timer_new();

 /* group status and progress bar into appbar */
 appbar = gtk_table_new(1, 4, TRUE);
 gtk_table_attach_defaults(GTK_TABLE(appbar), statusbar, 0, 3, 0, 1);
 gtk_table_attach_defaults(GTK_TABLE(appbar), progressbar, 3, 4, 0, 1);
 gtk_widget_show(appbar);

 gtk_box_pack_start(GTK_BOX(vbox), appbar, FALSE, TRUE, 0);
#endif
  
 return appbar;
}


/* call this function to initiate the statusbar:
 *   sg_project_statusbar_set(char *text, SGstatusbarMode mode);
 *
 * to stop the activity progress, call this function with NULL pointer:
 *   sg_project_statusbar_set(NULL);
 */
void
sg_project_statusbar_set(gchar *text, ...)
{
 if (gui_mode != SG_GUI_ICONLIST) return;

 if (text)
    {SGstatusbarMode mode;
     va_list ap;

     va_start(ap, text);
     mode = va_arg(ap, SGstatusbarMode);
     va_end(ap);

     if (mode == SG_STATUSBAR_TEXTONLY)
        { /* show only text for some time */
#ifdef __WITH_GNOME
         gnome_app_flash(GNOME_APP(main_window), text);
#else
         static guint timeout_id = 0;

         /* clear (possibly) previous text-only textfield */
	 text_clear(&timeout_id);

         gtk_statusbar_push(GTK_STATUSBAR(statusbar), context_text_id, text);
         timeout_id = g_timeout_add(TEXT_TIMEOUT*1000, text_clear, &timeout_id);
#endif
        }
     else if (mode == SG_STATUSBAR_ACTIVITY)
        { /* show text and progress as activity */
#ifdef __WITH_GNOME
  #ifdef WITH_WARNINGS
  #warning Prepare progressbar activity in Gnome - How?
  #endif
#else
	 gtk_statusbar_push(GTK_STATUSBAR(statusbar), context_progress_id, text);
         gtk_progress_set_activity_mode(GTK_PROGRESS(progressbar), TRUE);
#endif
        }
    }
 else
    { /* stop progress activity when text is NULL pointer */
#ifdef __WITH_GNOME
  #ifdef WITH_WARNINGS
  #warning Stop progressbar activity in Gnome - How?
  #endif
#else
     gtk_statusbar_pop(GTK_STATUSBAR(statusbar), context_progress_id);
     gtk_progress_set_activity_mode(GTK_PROGRESS(progressbar), FALSE);
     gtk_progress_set_value(GTK_PROGRESS(progressbar), 0.0);
#endif
    }

 /* update GUI */
 while (g_main_iteration(FALSE)) ;
}

void
sg_project_statusbar_update(void)
{
 if (gui_mode != SG_GUI_ICONLIST) return;

#ifdef __WITH_GNOME
#else
 if (progress_timer && g_timer_elapsed(progress_timer, NULL) > PROGRESS_TIMEOUT)
    {static guint value = 0;
    
     if (++value > 10) value = 0; /* value: 0..10 */
     gtk_progress_set_value(GTK_PROGRESS(progressbar), value/10.0);
     gtk_progress_set_activity_mode(GTK_PROGRESS(progressbar), TRUE);

     gtk_widget_draw(progressbar, NULL);
     gtk_widget_draw(statusbar, NULL);
     g_timer_reset(progress_timer);
    }
#endif
}

static gboolean
text_clear(gpointer data)
{
 static volatile sig_atomic_t locked = FALSE;
 guint *timeout_id = (guint*)data;

 if (locked) return FALSE;
 locked = TRUE;
 
 if (*timeout_id)
    {g_source_remove(*timeout_id);
     *timeout_id = 0;
     gtk_statusbar_pop(GTK_STATUSBAR(statusbar), context_text_id);
    }
 
 locked = FALSE;
 return FALSE;
}
