/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <math.h>
#include <stdio.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <Python.h>
#include <scigraphica/sg.h>
#include <scigraphica/dialogs/sg_misc_dialogs.h>
#include <scigraphica/sg_config.h>
#include "sg_logo_dialogs.h"
#include "sg_main.h"
#include "sg_project.h"
#include "sg_project_config.h"
#include "sg_project_statusbar.h"
#include "sg_project_rescue.h"
#include "pixmaps/sg.xpm"

/* default GUI-mode is ICONLIST */
SGguiMode gui_mode = SG_GUI_ICONLIST;

GtkWidget *main_window;
GtkWidget *gui_notebook;

GtkWidget *iconlist1;
GtkWidget *iconlist2;

SGapplication *app;

gint sg_canvas_label_width = 60;
gint sg_canvas_label_pos = 0;

extern gchar *last_project_filename;
extern gchar *last_project_path;

extern void	sg_project_connect_signals(SGapplication *app);
static GtkWidget *build_gui (gchar *my_name);
static void gui_init (gchar *app_name);
static void scan_commandline_args (int *argc, char ***argv);
static gboolean open_worksheet		(GtkWidget *widget,
         				 GtkIconListItem *item,
         				 GdkEvent *event,
         				 gpointer data);
static gboolean open_plot		(GtkWidget *widget,
         				 GtkIconListItem *item,
         				 GdkEvent *event,
         				 gpointer data);
#ifdef __WITH_GNOME
static gint main_win_restore_session (void);
static gint main_win_save_session (void);
static gint save_my_session (void);
static gint save_session (GnomeClient *client,
                          gint phase,
                          GnomeSaveStyle save_style,
                          gint is_shutdown,
                          GnomeInteractStyle interact_style,
                          gint is_fast,
                          gpointer client_data);
#endif


int main(int argc, char **argv, char **envp)
{ 
  scan_commandline_args(&argc, &argv);

#ifdef __WITH_GNOME
  gnome_init (PACKAGE, VERSION_INFO, argc, argv);
#else
  gtk_init(&argc,&argv);
#endif

  sg_presentation_dialog();

  sg_init("scigraphica");
  sg_project_config_init();
  app = SG_APPLICATION(sg_application_new());
  gui_init(argv[0]);
  sg_project_connect_signals(app);
  sg_project_new(TRUE);
  sg_project_rescue_init();
  gtk_main();

  return(0);
}

gint
project_quit_warning(GtkWidget *widget)
{
  if(sg_application_changed(app)){
    gchar label[255];
    SGpluginFile *plugin = NULL;
    gchar *path = NULL;
    GObject *object = G_OBJECT(app);
    g_snprintf(label, 255, _("Save project \"%s\" before leaving?"), last_project_filename);
    switch( sg_accept_dialog(label, 0) ) {
      case SG_BUTTON_YES:
        plugin = sg_plugin_file_get("xml","project",SG_PLUGIN_FILE_SAVE);
        if(!plugin)
          plugin = sg_plugin_file_get("xml","project",SG_PLUGIN_FILE_EXPORT);
        if(!plugin)
          plugin = sg_plugin_file_get("xml","project",SG_PLUGIN_FILE_SAVE_AS);

        if(!plugin){
          g_warning("Failed to save file\n");
          return FALSE;
        }
                                                                         
        path = g_strconcat(app->path,G_DIR_SEPARATOR_S,last_project_filename, NULL);
        plugin->action(plugin, path, NULL, &object, NULL);
        sg_application_set_changed(app, FALSE);
        g_free(path);
        break; 
      case SG_BUTTON_NO:
        break;
      case SG_BUTTON_CANCEL:
      case SG_BUTTON_CLOSE:
        return TRUE;
      default:
        break;
    }
  }
  return FALSE;
}

gint
main_quit (GtkWidget *widget)
{
  sg_project_quit();
  return FALSE;
}

static GtkWidget *
build_gui(gchar *my_name)
{
 GtkWidget *top_level;
 GtkWidget *vbox1;
 GtkWidget *hbox1;
 GtkWidget *label1;
 GtkWidget *label2;
 GtkWidget *toolbar;
 GtkWidget *menu_bar;
 GtkWidget *scrollw1;
 GtkWidget *scrollw2;
 GtkWidget *appbar;

 /* Build GUI */

#ifdef __WITH_GNOME
 GnomeClient *client;

 top_level = gnome_app_new (PACKAGE, "SciGraphica");
 client = gnome_master_client ();
 gtk_signal_connect (GTK_OBJECT (client), "save_yourself",
                     GTK_SIGNAL_FUNC (save_session), my_name);
#else
 top_level = gtk_window_new(GTK_WINDOW_TOPLEVEL);
 gtk_window_set_title(GTK_WINDOW(top_level), "SciGraphica");
 gtk_container_border_width(GTK_CONTAINER(top_level),0);
 gtk_widget_set_usize(top_level, 700, 300);
#endif /* WITH_GNOME */

 gtk_widget_realize(top_level);
 icon_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                                gdk_colormap_get_system(),
                                                &icon_mask, NULL, sg_xpm);
                          
 gdk_window_set_icon(top_level->window, NULL, icon_pixmap, icon_mask);            
 gtk_signal_connect (GTK_OBJECT (top_level), "destroy",
		     GTK_SIGNAL_FUNC (main_quit), NULL);

 gtk_signal_connect(GTK_OBJECT(top_level), "delete_event",
                    (GtkSignalFunc)project_quit_warning, NULL);

 vbox1 = gtk_vbox_new(FALSE,0);
 
#ifdef __WITH_GNOME
 gnome_app_set_contents(GNOME_APP(top_level),vbox1);
#else
 gtk_container_add(GTK_CONTAINER(top_level),vbox1);
#endif 

 menu_bar = sg_project_menu_new(app);
 gtk_box_pack_start (GTK_BOX(vbox1), menu_bar, FALSE, FALSE, 0);
 gtk_widget_show_all (GTK_WIDGET(menu_bar));

 toolbar = sg_project_toolbar_new(app);
 gtk_box_pack_start (GTK_BOX(vbox1), toolbar, FALSE, FALSE, 0);
 gtk_widget_show_all (GTK_WIDGET(toolbar));

 sg_project_popups_build();

 hbox1 = gtk_hbox_new(FALSE,0);
 gtk_container_set_border_width(GTK_CONTAINER(hbox1), 5);
 gtk_box_set_spacing(GTK_BOX(hbox1), 5);
 gtk_box_pack_start(GTK_BOX(vbox1), hbox1, TRUE, TRUE, 0);
 gtk_widget_show(hbox1);

 gui_notebook = gtk_notebook_new();
 gtk_box_pack_start(GTK_BOX(hbox1),gui_notebook, TRUE, TRUE,0);
 gtk_widget_show(gui_notebook);

 scrollw1 = gtk_scrolled_window_new(NULL, NULL);
 gtk_container_border_width(GTK_CONTAINER(scrollw1),0);
 gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollw1),
				GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
 label1 = gtk_label_new("Worksheets");
 gtk_notebook_append_page(GTK_NOTEBOOK(gui_notebook), scrollw1, label1);
 gtk_widget_show(scrollw1);

 scrollw2 = gtk_scrolled_window_new(NULL, NULL);
 gtk_container_border_width(GTK_CONTAINER(scrollw2),0);
 gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollw2),
				GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
 label2 = gtk_label_new("Plots");
 gtk_notebook_append_page(GTK_NOTEBOOK(gui_notebook), scrollw2, label2);
 gtk_widget_show(scrollw2);

 sg_canvas_label_width = MIN(sg_canvas_label_width, 140);

 iconlist1 = gtk_icon_list_new(48, sg_canvas_label_pos+1);
 gtk_icon_list_set_text_space(GTK_ICON_LIST(iconlist1), sg_canvas_label_width);
 gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollw1),iconlist1);
 gtk_signal_connect(GTK_OBJECT(iconlist1),"select_icon",
                    (GtkSignalFunc) open_worksheet, NULL);
 iconlist2 = gtk_icon_list_new(48, sg_canvas_label_pos+1);
 gtk_icon_list_set_text_space(GTK_ICON_LIST(iconlist2), sg_canvas_label_width);
 gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollw2),iconlist2);
 gtk_signal_connect(GTK_OBJECT(iconlist2),"select_icon",
                    (GtkSignalFunc) open_plot, NULL);
 gtk_widget_show(iconlist1);
 gtk_widget_show(iconlist2);

 appbar = sg_project_statusbar_new(top_level, vbox1);

 gtk_widget_show(vbox1);

 return(top_level);
}

static void
scan_commandline_args(int *argc, char ***argv)
{
 gint i;
 gui_mode = SG_GUI_ICONLIST;

 if (!argc || !argv) return;

 for (i = 1; i < *argc; i++)
 {
  if (!strcmp ("--version",(*argv)[i]) || !strcmp ("-v",(*argv)[i]))
  {/* print to stdout the build/version information and then exit */
   printf("%s\n", VERSION_INFO);
   exit(0);
  }
  else if (!strncmp("--plot",(*argv)[i],6) || !strncmp("-p",(*argv)[i],2))
  {/* start SciGraphica in plot-mode */
   gchar *plot_name = NULL;
   gui_mode = SG_GUI_PLOT;
   plot_name = strchr((*argv)[i],'=');
   if (plot_name) plot_name++;
   (*argv)[i] = NULL;
  }
  else if (!strncmp ("--worksheet",(*argv)[i],11) || !strncmp ("-w",(*argv)[i],2))
  {/* start SciGraphica in worksheet-mode */
   gchar *worksheet_name = NULL;
   gui_mode = SG_GUI_WORKSHEET;
   worksheet_name = strchr((*argv)[i],'=');
   if (worksheet_name) worksheet_name++;
   (*argv)[i] = NULL;
  }
 }

 /* remove arguments from argumentlist that are detected above */
 for (i = 1; i < *argc; )
   if ((*argv)[i] == NULL)
     {gint k;
      for (k = i + 1; k < *argc; k++) (*argv)[k-1] = (*argv)[k];
      (*argc)--;
     }
   else i++;
}


static void
gui_init(gchar *app_name)
{
  if(gui_mode == SG_GUI_ICONLIST){
     main_window = build_gui(app_name);
     sg_project_popups_build();
  }

#ifdef __WITH_GNOME
  main_win_restore_session();
#endif
  sg_project_rescue_init();

  if(gui_mode == SG_GUI_ICONLIST) gtk_widget_show (main_window);
}
 

#ifdef __WITH_GNOME
static gint 
main_win_restore_session(void)
{
  gchar id[80];
  gint width,height;

  if(gui_mode != SG_GUI_ICONLIST) return TRUE;

  g_snprintf(id,80,"/" PACKAGE"/main_window/width=400");
  width = gnome_config_get_int(id);

  g_snprintf(id,80,"/" PACKAGE"/main_window/height=400");
  height = gnome_config_get_int(id);

  gtk_window_set_default_size(GTK_WINDOW(main_window),width,height);

  return TRUE;
}


static gint 
main_win_save_session(void)
{ 
  gchar id[80];

  if (gui_mode != SG_GUI_ICONLIST) return TRUE;

  g_snprintf(id,80,"/" PACKAGE"/main_window/width");
  gnome_config_set_int(id,GTK_WIDGET(main_window)->allocation.width);

  g_snprintf(id,80,"/" PACKAGE"/main_window/height");
  gnome_config_set_int(id,GTK_WIDGET(main_window)->allocation.height);

  return TRUE;
}


static gint 
save_my_session(void)
{
 main_win_save_session();
 gnome_config_sync();
 return TRUE;
}

static gint
save_session (GnomeClient *client, gint phase, GnomeSaveStyle save_style,
              gint is_shutdown, GnomeInteractStyle interact_style,
              gint is_fast, gpointer client_data)
{
  gchar** argv;
  guint argc;
  save_my_session();

  return TRUE;
}

#endif /* WITH_GNOME */


#ifdef WITH_BONOBO
/*
 *  produce a brand new bonobo_scigraphica
 */
#define OAFIID_SCIGRAPHICA_FACTORY \
 "OAFIID:bonobo_scigraphica_factory:46a396b7-7e6d-4781-9163-d90338cbd51b"
/* We don't need the next #define, do we? -Rob-
 * #define OAFIID_SCIGRAPHICA \
 * "OAFIID:bonobo_scigraphica:3d2d8d14-3994-4e5b-9e26-bf175d5064d2"
 */
static BonoboObject*
bonobo_scigraphica_factory  (BonoboGenericFactory *factory, void *closure)
{
        BonoboControl* control;

        gui_init("SciGraphica");

        /* create a BonoboControl from a widget */
        control = bonobo_control_new (main_window);

        return BONOBO_OBJECT (control);
}

static void
bonobo_scigraphica_factory_init (void)
{
        /* perform initialization once */
        static BonoboGenericFactory *scigraphica_factory = NULL;

        if (scigraphica_factory) return;

        /* register ourselves (using OAF) */
        scigraphica_factory =
          bonobo_generic_factory_new (OAFIID_SCIGRAPHICA_FACTORY,
                                      bonobo_scigraphica_factory, NULL);

        /* registration worked? */
        if (!scigraphica_factory)
                g_error ( _("Registration of Bonobo SciGraphica factory failed") );
}
#endif /* WITH_BONOBO */

static gboolean
open_worksheet(GtkWidget *widget,
               GtkIconListItem *item, GdkEvent *event,
               gpointer data)
{
  GtkIconList *iconlist = GTK_ICON_LIST(widget);
  SGworksheetWindow *window;
  GdkModifierType mods;
  gboolean changed = sg_application_changed(app);

  gdk_window_get_pointer(widget->window, NULL, NULL, &mods);

  window = SG_WORKSHEET_WINDOW(gtk_icon_list_get_link(item));

  if(!event) return TRUE;
  if((mods & GDK_BUTTON1_MASK) && event->type == GDK_2BUTTON_PRESS)
  {
    gtk_widget_show(GTK_WIDGET(window));
    gdk_window_raise(GTK_WIDGET(window)->window);
  }

  if(mods & GDK_BUTTON3_MASK)
                sg_worksheet_popup_show(item,(GdkEventButton *)event);

  sg_application_set_active_worksheet(app, window->worksheet);

  return TRUE;

}

static gboolean
open_plot(GtkWidget *widget,
          GtkIconListItem *item,
          GdkEvent *event,
          gpointer data)
{
  GtkIconList *iconlist = GTK_ICON_LIST(widget);
  SGplotWindow *plot_window;
  GdkModifierType mods;
  gboolean changed = sg_application_changed(app);

  gdk_window_get_pointer(widget->window, NULL, NULL, &mods);

  plot_window = SG_PLOT_WINDOW(gtk_icon_list_get_link(item));

  if(!event) return TRUE;
  if((mods & GDK_BUTTON1_MASK) && event->type == GDK_2BUTTON_PRESS){
    gtk_widget_show(GTK_WIDGET(plot_window));
    gdk_window_raise(GTK_WIDGET(plot_window)->window);
  }

  if(mods & GDK_BUTTON3_MASK)
                sg_plot_popup_show(item,(GdkEventButton *)event);

  sg_application_set_active_plot(app, plot_window->plot);

  return TRUE;
}
