/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>
#include <libxml/parser.h>
#include <scigraphica/sg.h>
#include <scigraphica/dialogs/sg_misc_dialogs.h>
#include <libxml/xmlreader.h>
#include "pixmaps/sg_small.xpm"

#ifdef WITH_WARNINGS
#warning Temporary Hack to avoid the numerous type mismatches
#warning between CHAR and gchar. All CHAR have been replaced by
#warning XCHAR, which equals gchar in the next typedef. - Rob -
#endif
typedef gchar XCHAR;

enum{
  SG_LAYER_2D,
  SG_LAYER_3D,
  SG_LAYER_POLAR,
};

enum{
      GTK_PLOT_CANVAS_NONE,
      GTK_PLOT_CANVAS_PLOT,
      GTK_PLOT_CANVAS_AXIS,
      GTK_PLOT_CANVAS_LEGENDS,
      GTK_PLOT_CANVAS_TITLE,
      GTK_PLOT_CANVAS_TEXT,
      GTK_PLOT_CANVAS_DATA,
      GTK_PLOT_CANVAS_LINE,
      GTK_PLOT_CANVAS_RECTANGLE,
      GTK_PLOT_CANVAS_ELLIPSE,
      GTK_PLOT_CANVAS_PIXMAP,
      GTK_PLOT_CANVAS_MARKER,
      GTK_PLOT_CANVAS_POLYGON,
      GTK_PLOT_CANVAS_CUSTOM,
};

typedef enum {
	PARSE_WORKSHEET,
	PARSE_PLOT,
	PARSE_PROJECT,
} SGparseObject;

typedef enum {
	PARSER_IN_WORKSHEET,
	PARSER_IN_PLOT,
	PARSER_IN_PAGE,
	PARSER_IN_LAYER,
	PARSER_IN_FRAME,
	PARSER_IN_CORNER,
	PARSER_IN_AXIS,
	PARSER_IN_TITLE,
	PARSER_IN_VMAJOR,
	PARSER_IN_VMINOR,
	PARSER_IN_HMAJOR,
	PARSER_IN_HMINOR,
	PARSER_IN_MAJOR,
	PARSER_IN_MINOR,
	PARSER_IN_X0,
	PARSER_IN_Y0,
	PARSER_IN_LEGENDS,
	PARSER_IN_DATA,
	PARSER_IN_XLINE,
	PARSER_IN_YLINE,
	PARSER_IN_ZLINE,
	PARSER_IN_POINTS,
        PARSER_IN_CHILD,
        PARSER_IN_PIXMAP,
        PARSER_IN_BG_PIXMAP,
        PARSER_IN_COLOR,
        PARSER_UNKNOWN,
} SGparserState;

static gchar *constructor_names[] =
{
  "scatter",
  "vbars",
  "hbars",
  "vboxes",
  "hboxes",
  "bubbles",
  "vectors",
  "color",
  "density",
  "contour",
  "contour",
  "surface"
};

typedef struct _SGparseState SGparseState;
struct _SGparseState {
  XCHAR *name;
  GString *content;

  SGparseObject object;
  SGapplication *app;

  SGparserState state;
  SGparserState prev_state;

  gint num_layers;

  gint last_worksheet;
  gint last_matrix;
  gint last_plot;
  gint last_dataset;
  gint last_function;
  gint last_expression;

  gboolean in_text;

  SGworksheet *worksheet;
  gint row, col;
  
  guint layer_type;
  guint child_type;

  SGplot *plot;
  SGlayer *layer;
  SGdataset *dataset;
  GtkPlotData *dataset_child;

  gboolean visible;
  GtkPlotText text;
  GtkPlotCanvasChild child;
  GtkPlotLine line;

  GtkPlotCanvasLine arrow;
  GtkPlotCanvasRectangle rectangle;
  GtkPlotCanvasEllipse ellipse;
  GdkWindow *window;
  GdkImage *image;
  gint px, py;
  gint pwidth, pheight;
  gchar color_str[5];
  gint img_char;
  GdkGC *gc;

  guint axis_pos;
  GtkPlotAxis *axis;
  gdouble tick_value;

  gint id;
  gint type;
  gboolean show_labels;
  guint connector;
  gboolean show_gradient;
  gboolean fill_area;
  gchar *legend;
  gboolean legends_visible;
  gboolean legends_precision;
  gchar *data_name;

  gint nx;
  gint ny;
  GdkColor surface_fg;
  GdkColor surface_bg;
  gboolean surface_transparent;
  gboolean height_gradient;
  gboolean use_amplitud;

  GdkColor grid_fg;
  GdkColor grid_bg;
  gboolean grid_visible;
  gboolean mesh_visible;
  gboolean contour_visible;
  gboolean contour_project;

  gulong *pixels;
  gint ncolors;
};

static gint powers[4] = {1, 16, 256, 4096};

static void sgWarning				(SGparseState *state, 
						 const char *msg, ...);
static void sgError				(SGparseState *state, 
						 const char *msg, ...);
static void sgFatalError			(SGparseState *state, 
						 const char *msg, ...);
static void sgStartDocument			(SGparseState *state);
static void sgEndDocument			(SGparseState *state);
static void sgCharacters			(SGparseState *state, 
						 const XCHAR *chars,
                                                 int len); 
static void sgStartElement			(SGparseState *state, 
						 const XCHAR *name, 
						 const XCHAR **attrs);
static void sgEndElement			(SGparseState *state, 
						 const XCHAR *name); 

static const XCHAR *get_real_name		(const XCHAR *name);

static gboolean default_open			(const gchar *file, 
						 SGapplication* app, 
						 SGworksheet *worksheet, 
						 SGplot *plot);

static xmlSAXHandler sgSAXParser = {
  (internalSubsetSAXFunc)        NULL, /* internal Subset */
  (isStandaloneSAXFunc)          NULL, /* isStandalone */
  (hasInternalSubsetSAXFunc)     NULL, /* hasInternalSubset */
  (hasExternalSubsetSAXFunc)     NULL, /* hasExternalSubset */
  (resolveEntitySAXFunc)         NULL, /* resolveEntity */
  (getEntitySAXFunc)             NULL, /* getEntity */
  (entityDeclSAXFunc)            NULL, /* entityDecl */
  (notationDeclSAXFunc)          NULL, /* notationDecl */
  (attributeDeclSAXFunc)         NULL, /* attributeDecl */
  (elementDeclSAXFunc)           NULL, /* elementDecl */
  (unparsedEntityDeclSAXFunc)    NULL, /* unparseEntityDecl */
  (setDocumentLocatorSAXFunc)    NULL, /* setDocumentLocator */
  (startDocumentSAXFunc)         sgStartDocument, /* startDocument */
  (endDocumentSAXFunc)           sgEndDocument, /* endDocument */
  (startElementSAXFunc)          sgStartElement, /* startElement */
  (endElementSAXFunc)            sgEndElement, /* endElement */
  (referenceSAXFunc)             NULL, /* reference */
  (charactersSAXFunc)            sgCharacters, /*characters */
  (ignorableWhitespaceSAXFunc)   NULL, /* ignorableWhitespace */
  (processingInstructionSAXFunc) NULL, /* processingInstruction */
  (commentSAXFunc)               NULL, /* comment */
  (warningSAXFunc)               sgWarning, /* warning */
  (errorSAXFunc)                 sgError, /* error */
  (fatalErrorSAXFunc)            sgFatalError, /* fatalError */
  (getParameterEntitySAXFunc)    NULL, /* parameterEntity */
  (cdataBlockSAXFunc)            NULL /* blockSAX */
};

void
project_sax_open_init(SGplugin *_plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;
  SGpluginFile *plugin = SG_PLUGIN_FILE(_plugin);
                                                                                
  plugin->mode = SG_PLUGIN_FILE_OPEN;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".sg");
                                                                                
  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

SGdataset *
get_dataset_by_id(SGapplication *app, gint id)
{
  GList *list = app->datasets->list;
  while(list){
    SGlistChild *child = (SGlistChild *)list->data;
    SGdataset *dataset = SG_DATASET(child->object);
    if(child->id-1 == id || dataset->id == id) return dataset;
    list = list->next;
  }
  return NULL;
}


SGpluginStyle *
get_constructor(gint type, guint layer)
{
  gchar name[200];

  if(layer == SG_LAYER_2D)
    g_snprintf(name,200,"%s_2d",constructor_names[type]);
  else if(layer == SG_LAYER_3D)
    g_snprintf(name,200,"%s_3d",constructor_names[type]);
  else if(layer == SG_LAYER_POLAR)
    g_snprintf(name,200,"%s_polar",constructor_names[type]);

  if(type == 9)
    g_snprintf(name,200,"contour_2d");

  return(sg_plugin_style_get(name));
}
static void  
sgWarning(SGparseState *state, const char *msg, ...)
{
  va_list args;

  va_start(args, msg);
  g_logv("XML", G_LOG_LEVEL_WARNING, msg, args);
  va_end(args);
}

static void  
sgError(SGparseState *state, const char *msg, ...)
{
  va_list args;

  va_start(args, msg);
  g_logv("XML", G_LOG_LEVEL_CRITICAL, msg, args);
  va_end(args);
}

static void  
sgFatalError(SGparseState *state, const char *msg, ...)
{
  va_list args;

  va_start(args, msg);
  g_logv("XML", G_LOG_LEVEL_ERROR, msg, args);
  va_end(args);
}

static void
sgStartDocument(SGparseState *state)
{
  state->name = NULL;
  state->content = g_string_sized_new(128);

  state->state = PARSER_UNKNOWN;
  state->prev_state = PARSER_UNKNOWN;
}

static void
sgEndDocument(SGparseState *state)
{
  GList *list;

  if(state->text.text)
     g_free(state->text.text);

  if(state->text.font)
   g_free(state->text.font);

  if(state->name)
   g_free(state->name);

  if(state->content)
   g_string_free(state->content, TRUE);

  if(state->pixels)
   g_free(state->pixels);

  gdk_gc_unref(state->gc);
  gdk_window_unref(state->window);

  list = state->app->worksheets->list;
  while (list && list->data)
  {   
      SGlistChild *child = (SGlistChild *)list->data;
      SGworksheet *worksheet = SG_WORKSHEET(child->object);
      sg_worksheet_update_exp_all(SG_WORKSHEET(worksheet));
      list = list->next;
  }
  list = state->app->plots->list;
  while(list){
    SGlistChild *child = (SGlistChild *)list->data;
    SGplot *plot = SG_PLOT(child->object);

    sg_plot_refresh_datasets(plot);
    gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
    gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot));
    list = list->next;
  }

}

static void
sgCharacters(SGparseState *state, const XCHAR *chars, int len)
{
  gint i, n;

  if(state->state == PARSER_IN_PIXMAP || state->state == PARSER_IN_BG_PIXMAP){
     GdkColor color; 
     gint index;

     for(i = 0; i < len; i++){

       if((chars[i] >= '0' && chars[i] <= '9')||(chars[i] >= 'A' && chars[i] <= 'F')){
         state->color_str[state->img_char++] = chars[i];

         if(state->img_char == 4){
           gchar *s;

           state->img_char = 0;
       
           s = state->color_str;
           index = 0;
           for(n = 0; n < 4; n++){
             gint value;
             if(*s <= '9') 
               value = *s - '0';
             else
               value = *s - 'A' + 10;
         
             index += powers[3 - n] * value; 
             s++;
           } 

           color.pixel = state->pixels[index];
           gdk_image_put_pixel(state->image, state->px, state->py, color.pixel);

           state->px++;
           if(state->px == state->pwidth){
              state->py++;
              state->px = 0;
           }
         }
       }
     }
  }
  else
  {
    for(i = 0; i < len; i++){
      g_string_append_c(state->content, chars[i]);    
    }
  }
}

static void
sgStartElement(SGparseState *state, const XCHAR *real_name, const XCHAR **attrs)
{
  SGplot *plot = state->plot;
  GtkPlot *real_plot = NULL;
  GtkPlotCanvas *canvas = NULL;
  GtkSheet *sheet = NULL;
  const XCHAR **aux = attrs;
  const XCHAR *child = NULL, *value = NULL;
  const XCHAR *name;

  name = get_real_name(real_name);
/*
  fprintf(stderr,"START ELEMENT %s\n", name);
*/
  if(state->plot) {
    canvas = GTK_PLOT_CANVAS(state->plot);
  }

  if(state->worksheet) {
    sheet = GTK_SHEET(state->worksheet);
  }

  state->name = g_strdup(real_name);
  state->content = g_string_truncate(state->content, 0);

  if(state->layer) real_plot = GTK_PLOT_CANVAS_PLOT(state->layer)->plot;

  if(strcmp(name, "Environment") == 0){
     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "LastWorksheet") == 0) 
                               state->last_worksheet = atoi(value);
        if(strcmp(child, "LastMatrix") == 0) 
                               state->last_matrix = atoi(value);
        if(strcmp(child, "LastPlot") == 0) 
                               state->last_plot = atoi(value);
        if(strcmp(child, "LastDataset") == 0) 
                               state->last_dataset = atoi(value);
        if(strcmp(child, "LastFunction") == 0) 
                               state->last_function = atoi(value);
        if(strcmp(child, "LastExpression") == 0) 
                               state->last_expression = atoi(value);
     }
  }

  if(strcmp(name, "Worksheet") == 0){
     state->prev_state = state->state;
     state->state = PARSER_IN_WORKSHEET;

     state->row = 0;
     state->col = 0;

     state->plot = NULL;
     state->layer = NULL;

     if(state->object == PARSE_PROJECT){
         state->worksheet = sg_worksheet_new("",20,5);
	 sg_application_add_worksheet(state->app, state->worksheet);
         sheet = GTK_SHEET(state->worksheet);
     }
  }

  if(strcmp(name, "Matrix") == 0){
     state->prev_state = state->state;
     state->state = PARSER_IN_WORKSHEET;

     state->row = 0;
     state->col = 0;

     state->plot = NULL;
     state->layer = NULL;

     if(state->object == PARSE_PROJECT){
         state->worksheet = SG_WORKSHEET(sg_matrix_new("",10,10));
	 sg_application_add_matrix(state->app, state->worksheet);
         sheet = GTK_SHEET(state->worksheet);
     }
  }

  if(strcmp(name, "Range") == 0 && state->state == PARSER_IN_WORKSHEET){
     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Xmin") == 0) SG_MATRIX(state->worksheet)->xmin = atof(value);
        if(strcmp(child, "Xmax") == 0) SG_MATRIX(state->worksheet)->xmax = atof(value);
        if(strcmp(child, "Ymin") == 0) SG_MATRIX(state->worksheet)->ymin = atof(value);
        if(strcmp(child, "Ymax") == 0) SG_MATRIX(state->worksheet)->ymax = atof(value);
     }
  }

  if(strcmp(name, "MatrixFormat") == 0 && state->state == PARSER_IN_WORKSHEET){
     SGcolumntype type = SG_TYPE_NUMBER;
     SGcolumnformat format = SG_FORMAT_DECIMAL;
     SGcolumninternal internal = SG_INTERNAL_DOUBLE;
     gint precision = 3;
     gchar *exp = NULL;

     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Exp") == 0) exp = (gchar *)value;
        if(strcmp(child, "Internal") == 0) internal = (SGcolumninternal)atoi(value);
        if(strcmp(child, "Format") == 0) format = (SGcolumnformat)atoi(value);
        if(strcmp(child, "Precision") == 0 ||
           strcmp(child, "Presicion") == 0) precision = atoi(value);
     }

     sg_matrix_set_exp(SG_MATRIX(state->worksheet), exp);
     SG_MATRIX(state->worksheet)->format.type = type;
     SG_MATRIX(state->worksheet)->format.format = format;
     SG_MATRIX(state->worksheet)->format.internal = internal;
     SG_MATRIX(state->worksheet)->format.precision = precision;
  }

  if(state->state == PARSER_IN_WORKSHEET && strcmp(name, "Cell") == 0){
  }

  if(strcmp(name, "Plot") == 0){
     state->prev_state = state->state;
     state->state = PARSER_IN_PLOT;
     state->in_text = FALSE;

     state->worksheet = NULL;
     if(state->object == PARSE_PROJECT){
         state->plot = sg_plot_new("");
	 sg_application_add_plot(state->app, state->plot);
         canvas = GTK_PLOT_CANVAS(state->plot);
     }
  }

  if(strcmp(name, "Function") == 0){
     const gchar *data_name = NULL;
     const gchar *exp = NULL;

     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Name") == 0) data_name = value;
        if(strcmp(child, "Exp") == 0) exp = value;
     }

     if(exp && data_name){
         SGdataset *dataset;
         SGpluginIterator *plugin;
         GtkArg args[1];

         args[0].type = GTK_TYPE_STRING;
         args[0].name = "SGdatasetFunction::exp";
         GTK_VALUE_STRING(args[0]) = (gchar *)exp;

         plugin = sg_plugin_iterator_get("sg_dataset_function");
         dataset = plugin->construct(plugin);
         sg_dataset_set_name(dataset, (gchar *)data_name);
         g_object_set(G_OBJECT(dataset), args[0].name, exp, NULL);

     }
       
  }

  if(strcmp(name, "Text") == 0){
     state->in_text = TRUE;
  }

  if(strcmp(name, "Child") == 0){
     state->state = PARSER_IN_CHILD;
     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "X1") == 0) state->child.rx1 = atof(value);
        if(strcmp(child, "Y1") == 0) state->child.ry1 = atof(value);
        if(strcmp(child, "X2") == 0) state->child.rx2 = atof(value);
        if(strcmp(child, "Y2") == 0) state->child.ry2 = atof(value);
        if(strcmp(child, "Type") == 0) state->child_type = atoi(value);
        if(strcmp(child, "Flags") == 0) state->child.flags = (GtkPlotCanvasFlag)atoi(value);
     }
  }

  if(strcmp(name, "XPMColor") == 0){
     state->state = PARSER_IN_COLOR;
  }


  if(strcmp(name, "Rectangle") == 0){
     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Fill") == 0) state->rectangle.filled = atoi(value);
        if(strcmp(child, "Border") == 0) state->rectangle.border = (GtkPlotBorderStyle)atoi(value);
        if(strcmp(child, "Shadow") == 0) state->rectangle.shadow_width = atoi(value);
        if(strcmp(child, "R") == 0) state->rectangle.bg.red = atoi(value);
        if(strcmp(child, "G") == 0) state->rectangle.bg.green = atoi(value);
        if(strcmp(child, "B") == 0) state->rectangle.bg.blue = atoi(value);
     }
  }

  if(strcmp(name, "Ellipse") == 0){
     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Fill") == 0) state->ellipse.filled = atoi(value);
        if(strcmp(child, "R") == 0) state->ellipse.bg.red = atoi(value);
        if(strcmp(child, "G") == 0) state->ellipse.bg.green = atoi(value);
        if(strcmp(child, "B") == 0) state->ellipse.bg.blue = atoi(value);
     }
  }

  if(strcmp(name, "Pixels") == 0){
     if(state->state == PARSER_IN_CHILD)
       state->state = PARSER_IN_PIXMAP;
     else
       state->state = PARSER_IN_BG_PIXMAP;
  }

  if(strcmp(name, "Pixmap") == 0){
     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Width") == 0) state->pwidth = atoi(value);
        if(strcmp(child, "Height") == 0) state->pheight = atoi(value);
     }
     state->px = state->py = 0;
     state->img_char = 0;
     state->image = gdk_image_new(GDK_IMAGE_FASTEST, 
                                  gdk_visual_get_system(),
                                  state->pwidth, state->pheight);
  }
 

  if(strcmp(name, "Arrow") == 0){
     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Mask") == 0) state->arrow.arrow_mask = (GtkPlotCanvasArrow)atoi(value);
        if(strcmp(child, "Length") == 0) state->arrow.arrow_length = atoi(value);
        if(strcmp(child, "Width") == 0) state->arrow.arrow_width = atoi(value);
        if(strcmp(child, "Style") == 0) state->arrow.arrow_style = (GtkPlotSymbolStyle)atoi(value);
     }
  }


  if(state->state != PARSER_IN_LAYER && strcmp(name, "Geometry") == 0){
     gint width = 400;
     gint height = 350;

     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Width") == 0) width = atoi(value); 
        if(strcmp(child, "Height") == 0) height = atoi(value); 
     }

     if(state->state == PARSER_IN_PLOT){
/* FIXME
       state->plot->width = width; 
       state->plot->height = height; 
       if(state->plot->is_mapped)
             gtk_window_set_default_size(GTK_WINDOW(plot), width, height);
*/
     }

     if(state->state == PARSER_IN_WORKSHEET){
/* FIXME
       state->worksheet->width = width;
       state->worksheet->height = height;
       if(state->worksheet->is_mapped)
            gtk_window_set_default_size(GTK_WINDOW(state->worksheet), width, height);
*/
     }

  }

  if(state->state == PARSER_IN_WORKSHEET && strcmp(name, "Column") == 0){
     gint ncol = 0;
     gint width = 80;
     const gchar *title = NULL;
     SGcolumntype type = SG_TYPE_NONE;
     SGcolumnformat format = SG_FORMAT_DECIMAL;
     SGcolumninternal internal = SG_INTERNAL_DOUBLE;
     gint precision = 3;

     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Width") == 0) width = atoi(value);
        if(strcmp(child, "No") == 0) ncol = atoi(value);
        if(strcmp(child, "Title") == 0) title = value;
        if(strcmp(child, "Type") == 0) type = (SGcolumntype)atoi(value);
        if(strcmp(child, "Internal") == 0) internal = (SGcolumninternal)atoi(value);
        if(strcmp(child, "Format") == 0) format = (SGcolumnformat)atoi(value);
        if(strcmp(child, "Precision") == 0 ||
           strcmp(child, "Presicion") == 0) precision = atoi(value);
     }
     gtk_sheet_set_column_width(sheet, ncol, width);
     gtk_sheet_set_column_title(sheet, ncol, title);
     gtk_sheet_column_button_add_label(sheet, ncol, title);

     if(type == SG_TYPE_NONE){ /* before 0.7 */
       type = (SGcolumntype)format;
       format = SG_FORMAT_DECIMAL;
     } 

     sg_worksheet_column_set_format(state->worksheet, ncol, type, format, internal, precision);
     state->col = ncol;
  }

  if(state->state == PARSER_IN_WORKSHEET && strcmp(name, "Cell") == 0){
     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Row") == 0) state->row = atoi(value);
        if(strcmp(child, "Col") == 0) state->col = atoi(value);
     }
  }


  if(strcmp(name, "Page") == 0){
     gint width = GTK_PLOT_LETTER_W;
     gint height = GTK_PLOT_LETTER_H;
     gint temp;
     gint orientation = GTK_PLOT_PORTRAIT;
     gint page_size = GTK_PLOT_LETTER;
     gdouble scale = .65;

     state->state = PARSER_IN_PAGE;
     state->in_text = FALSE;

     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Width") == 0) width = atoi(value); 
        if(strcmp(child, "Height") == 0) height = atoi(value);
        if(strcmp(child, "Orientation") == 0) orientation = atoi(value); 
        if(strcmp(child, "Size") == 0) page_size = atoi(value); 
        if(strcmp(child, "Scale") == 0) scale = atof(value); 
     }
     sg_plot_rescale(plot, scale);

     if (orientation==GTK_PLOT_LANDSCAPE){
         temp=width;
         width=height;
         height=temp;
     }
     sg_plot_set_size(plot, page_size, width, height, orientation);
  }

  if(strcmp(name, "Layer") == 0){
     state->prev_state = state->state;
     state->state = PARSER_IN_LAYER;
     state->in_text = FALSE;
     state->layer_type = SG_LAYER_2D;

     state->num_layers++;

     while(aux && *aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Type") == 0) state->layer_type = atoi(value); 
     }
  }

  if(state->layer && strcmp(name, "Autosymbols") == 0){
     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Type") == 0) state->layer->symbol = atoi(value); 
        if(strcmp(child, "Style") == 0) state->layer->symbol_style = atoi(value); 
        if(strcmp(child, "Line") == 0) state->layer->line_style = atoi(value); 
        if(strcmp(child, "Connector") == 0) state->layer->connector = atoi(value); 
     }
  }

  if(state->state == PARSER_IN_LAYER && strcmp(name, "Geometry") == 0){
     SGpluginLayer *plugin_layer = NULL;
     gdouble x = .175;
     gdouble y = .15;
     gdouble width = .65;
     gdouble height = .45;
     GtkPlotScale xscale = GTK_PLOT_SCALE_LINEAR;
     GtkPlotScale yscale = GTK_PLOT_SCALE_LINEAR;
     gboolean reflect_x = FALSE;
     gboolean reflect_y = FALSE;

     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "X") == 0) x = atof(value); 
        if(strcmp(child, "Y") == 0) y = atof(value); 
        if(strcmp(child, "Width") == 0) width = atof(value); 
        if(strcmp(child, "Height") == 0) height = atof(value); 
        if(strcmp(child, "Xscale") == 0) xscale = (GtkPlotScale)atoi(value); 
        if(strcmp(child, "Yscale") == 0) yscale = (GtkPlotScale)atoi(value); 
        if(strcmp(child, "ReflectX") == 0) reflect_x = atoi(value); 
        if(strcmp(child, "ReflectY") == 0) reflect_y = atoi(value); 
     }

/*   This is to fix the bug in gtkplotcanvas in gtkextra-0.99.6 */
     switch(state->layer_type){
       case SG_LAYER_2D:
         plugin_layer = sg_plugin_layer_get("layer_2d");
         break;
       case SG_LAYER_POLAR:
         plugin_layer = sg_plugin_layer_get("layer_polar");
         break;
       case SG_LAYER_3D:
         plugin_layer = sg_plugin_layer_get("layer_3d");
         break;
     }
     state->layer = sg_layer_new(plugin_layer, width, height);
     sg_plot_add_layer(plot, state->layer, x, y);
     real_plot = GTK_PLOT_CANVAS_PLOT(state->layer)->plot;

     gtk_plot_set_xscale(real_plot, xscale);
     gtk_plot_set_yscale(real_plot, yscale);
     gtk_plot_reflect_x(real_plot, reflect_x);
     gtk_plot_reflect_y(real_plot, reflect_y);
  }

  if(state->state == PARSER_IN_LAYER){
    if(strcmp(name, "Frame") == 0){
      gdouble xfactor = 1.0, yfactor = 1.0, zfactor = 1.0;
      gint titles_offset = 6;

      state->state = PARSER_IN_FRAME;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Titles") == 0) titles_offset = atoi(value); 
        if(strcmp(child, "Xfactor") == 0) xfactor = atof(value); 
        if(strcmp(child, "Yfactor") == 0) yfactor = atof(value); 
        if(strcmp(child, "Zfactor") == 0) zfactor = atof(value); 
      }

      if(GTK_IS_PLOT3D(real_plot)){
        GTK_PLOT3D(real_plot)->titles_offset = titles_offset;
        gtk_plot3d_set_xfactor(GTK_PLOT3D(real_plot), xfactor);
        gtk_plot3d_set_yfactor(GTK_PLOT3D(real_plot), yfactor);
        gtk_plot3d_set_xfactor(GTK_PLOT3D(real_plot), xfactor);
      }
    }

    if(strcmp(name, "Corner") == 0){
      gint visible = FALSE;

      state->state = PARSER_IN_CORNER;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Visible") == 0) visible = atoi(value); 
      }

      if(GTK_IS_PLOT3D(real_plot))
        gtk_plot3d_corner_set_visible(GTK_PLOT3D(real_plot), visible);
    }

    if(strcmp(name, "Plane") == 0){
      GtkPlotPlane position = GTK_PLOT_PLANE_XY;
      gint visible = TRUE;
      GdkColor color;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Position") == 0) position = (GtkPlotPlane)atoi(value); 
        if(strcmp(child, "Visible") == 0) visible = atoi(value); 
        if(strcmp(child, "R") == 0) color.red = atoi(value); 
        if(strcmp(child, "G") == 0) color.green = atoi(value); 
        if(strcmp(child, "B") == 0) color.blue = atoi(value); 
      }
      gdk_color_alloc(gdk_colormap_get_system(), &color);

      if(GTK_IS_PLOT3D(real_plot)){
        gtk_plot3d_plane_set_color(GTK_PLOT3D(real_plot), position, &color);
        gtk_plot3d_plane_set_visible(GTK_PLOT3D(real_plot), position, visible);
      }
    }

    if(strcmp(name, "Range") == 0){
      gdouble xmin = 0.0;
      gdouble xmax = 1.0;
      gdouble ymin = 0.0;
      gdouble ymax = 1.0;
      gdouble zmin = 0.0;
      gdouble zmax = 1.0;
      gdouble rotation = 0.0;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Xmin") == 0) xmin = atof(value); 
        if(strcmp(child, "Xmax") == 0) xmax = atof(value); 
        if(strcmp(child, "Ymin") == 0) ymin = atof(value); 
        if(strcmp(child, "Ymax") == 0) ymax = atof(value); 
        if(strcmp(child, "Zmin") == 0) zmin = atof(value); 
        if(strcmp(child, "Zmax") == 0) zmax = atof(value); 
        if(strcmp(child, "Rotation") == 0) rotation = atof(value); 
      }

      if(GTK_IS_PLOT_POLAR(real_plot))
        gtk_plot_polar_rotate(GTK_PLOT_POLAR(real_plot), rotation);

      if(GTK_IS_PLOT3D(real_plot)){
        gtk_plot3d_set_xrange(GTK_PLOT3D(real_plot), xmin, xmax);
        gtk_plot3d_set_yrange(GTK_PLOT3D(real_plot), ymin, ymax);
        gtk_plot3d_set_zrange(GTK_PLOT3D(real_plot), zmin, zmax);
      }
      else 
        gtk_plot_set_range(real_plot, xmin, xmax, ymin, ymax);

    }


    if(strcmp(name, "Vmajor") == 0){ 

      state->state = PARSER_IN_VMAJOR;
      state->in_text = FALSE;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Visible") == 0) real_plot->bottom->show_major_grid = atoi(value); 
      }

    }

    if(strcmp(name, "Vminor") == 0){ 

      state->state = PARSER_IN_VMINOR;
      state->in_text = FALSE;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Visible") == 0) real_plot->bottom->show_minor_grid = atoi(value); 
      }

    }
    if(strcmp(name, "Hmajor") == 0){ 

      state->state = PARSER_IN_HMAJOR;
      state->in_text = FALSE;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Visible") == 0) real_plot->left->show_major_grid = atoi(value); 
      }

    }
    if(strcmp(name, "Hminor") == 0){ 

      state->state = PARSER_IN_HMINOR;
      state->in_text = FALSE;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Visible") == 0) real_plot->left->show_minor_grid = atoi(value); 
      }
    }

    if(strcmp(name, "X0") == 0){
      gint visible = FALSE;

      state->state = PARSER_IN_X0;
      state->in_text = FALSE;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Visible") == 0) visible = atoi(value);
      }
      gtk_plot_x0_set_visible(real_plot, visible);
    }

    if(strcmp(name, "Y0") == 0){
      gint visible = FALSE;

      state->state = PARSER_IN_Y0;
      state->in_text = FALSE;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Visible") == 0) visible = atoi(value);
      }
      gtk_plot_y0_set_visible(real_plot, visible);
    }

    if(strcmp(name, "Xticks") == 0 || strcmp(name, "Yticks") == 0){
      gdouble step = .1;
      gint nminor = 1;
      gboolean limits = FALSE;
      gdouble begin = 0.;
      gdouble end = 0.;
      GtkPlotOrientation orientation = GTK_PLOT_AXIS_X;

      if(strcmp(name, "Xticks") == 0)
        orientation = (GtkPlotOrientation)GTK_ORIENTATION_HORIZONTAL;
      else
        orientation = (GtkPlotOrientation)GTK_ORIENTATION_VERTICAL;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Step") == 0) step = atof(value); 
        if(strcmp(child, "Begin") == 0) begin = atof(value); 
        if(strcmp(child, "End") == 0) end = atof(value); 
        if(strcmp(child, "Nminor") == 0) nminor = atoi(value); 
        if(strcmp(child, "Limits") == 0) limits = atoi(value); 
      }

      gtk_plot_set_ticks(real_plot, orientation, step, nminor);
      if(limits)
         gtk_plot_set_ticks_limits(real_plot, orientation, begin, end);
      else
         gtk_plot_unset_ticks_limits(real_plot, orientation); 
    }
  }

  if(state->state == PARSER_IN_LAYER && strcmp(name, "Side") == 0){
     GtkPlotSide side = 0;
     gint major_mask = 0, minor_mask = 0, label_mask = 0;
     gint title_visible = TRUE;

     state->in_text = FALSE;

     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Position") == 0) side = (GtkPlotSide)atoi(value); 
        if(strcmp(child, "Major") == 0) major_mask = atoi(value); 
        if(strcmp(child, "Minor") == 0) minor_mask = atoi(value);
        if(strcmp(child, "Labels") == 0) label_mask = atoi(value); 
        if(strcmp(child, "Title") == 0) title_visible = atoi(value); 
     }

     if(GTK_IS_PLOT3D(real_plot)){
       GtkPlotAxis *the_side;
       the_side = gtk_plot3d_get_side(GTK_PLOT3D(real_plot), side);

       the_side->major_mask = major_mask;
       the_side->minor_mask = minor_mask;
       the_side->label_mask = label_mask;
       the_side->title_visible = title_visible;
     }
  }
 
  if(state->state == PARSER_IN_LAYER && strcmp(name, "Axis") == 0){
     gdouble align = 0.;
     gboolean visible = TRUE;
     GtkPlotScale scale = GTK_PLOT_SCALE_LINEAR;
     GtkPlotAxis *the_axis = NULL;
     guint axis_pos = 0;

     state->state = PARSER_IN_AXIS;
     state->in_text = FALSE;
     state->tick_value = 1.0;

     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Position") == 0){
           if(strcmp(value, "Top") == 0) axis_pos = GTK_PLOT_AXIS_TOP; 
           if(strcmp(value, "Bottom") == 0) axis_pos = GTK_PLOT_AXIS_BOTTOM; 
           if(strcmp(value, "Left") == 0) axis_pos = GTK_PLOT_AXIS_LEFT; 
           if(strcmp(value, "Right") == 0) axis_pos = GTK_PLOT_AXIS_RIGHT; 
        }
        if(strcmp(child, "Visible") == 0) visible = atoi(value);
        if(strcmp(child, "Scale") == 0) scale = (GtkPlotScale)atoi(value);
        if(strcmp(child, "Align") == 0) align = atof(value);
     }
  
     state->axis_pos = axis_pos; 
     state->axis = the_axis = gtk_plot_get_axis(real_plot, axis_pos); 
     the_axis->ticks.scale = scale;
     gtk_plot_axis_set_visible(state->axis, visible); 
     switch(state->axis_pos){
       case GTK_PLOT_AXIS_TOP:
           real_plot->top_align = align;
           break;
       case GTK_PLOT_AXIS_BOTTOM:
           real_plot->bottom_align = align;
           break;
       case GTK_PLOT_AXIS_LEFT:
           real_plot->left_align = align;
           break;
       case GTK_PLOT_AXIS_RIGHT:
           real_plot->right_align = align;
           break;
     }
  }

  if(state->state == PARSER_IN_AXIS){
    GtkPlotAxis *the_axis;

    the_axis = state->axis;

    if(strcmp(name, "Ticks") == 0){
      gint major_mask = 0;
      gint minor_mask = 0;
      gint length = 0;
      gfloat width = 0.0;
      gdouble step = .1;
      gint nminor = 1;
      gboolean limits = FALSE;
      gdouble begin = 0.;
      gdouble end = 0.;
      GtkPlotOrientation orientation;

      state->in_text = FALSE;

      while(*aux){
         child = *aux++;
         value = *aux++;
         if(strcmp(child, "Major") == 0) major_mask = atoi(value); 
         if(strcmp(child, "Minor") == 0) minor_mask = atoi(value); 
         if(strcmp(child, "Length") == 0) length = atoi(value); 
         if(strcmp(child, "Width") == 0) width = atof(value); 
         if(strcmp(child, "Step") == 0) step = atof(value); 
         if(strcmp(child, "Begin") == 0) begin = atof(value); 
         if(strcmp(child, "End") == 0) end = atof(value); 
         if(strcmp(child, "Nminor") == 0) nminor = atoi(value); 
         if(strcmp(child, "Limits") == 0) limits = atoi(value); 
      }

      if(GTK_IS_PLOT3D(real_plot) && state->axis_pos == GTK_PLOT_AXIS_RIGHT) return;

      orientation = the_axis->orientation;

      if(GTK_IS_PLOT3D(real_plot)){
        gtk_plot3d_set_ticks(GTK_PLOT3D(real_plot), orientation, step, nminor);
        gtk_plot3d_set_ticks_length(GTK_PLOT3D(real_plot), orientation, length);
        gtk_plot3d_set_ticks_width(GTK_PLOT3D(real_plot), orientation, width);
      } else {
        gtk_plot_axis_show_ticks(state->axis, major_mask, minor_mask);
        gtk_plot_set_ticks(real_plot, orientation, step, nminor);
        gtk_plot_axis_set_ticks_length(state->axis, length);
        gtk_plot_axis_set_ticks_width(state->axis, width);
      }

      if(limits)
         gtk_plot_set_ticks_limits(real_plot, the_axis->orientation, begin, end);
      else
         gtk_plot_unset_ticks_limits(real_plot, the_axis->orientation); 
    }

    if(strcmp(name, "Break") == 0){
      gint apply_break = 0;
      gint break_scale = 0;
      gint break_nminor = 0;
      gdouble break_step = .1;
      gdouble break_min = 0.;
      gdouble break_max = 0.1;
      gdouble break_position = 0.5;

      state->in_text = FALSE;

      while(*aux){
         child = *aux++;
         value = *aux++;
         if(strcmp(child, "Apply") == 0) apply_break = atoi(value); 
         if(strcmp(child, "Scale") == 0) break_scale = atoi(value); 
         if(strcmp(child, "Nminor") == 0) break_nminor = atoi(value); 
         if(strcmp(child, "Step") == 0) break_step = atof(value); 
         if(strcmp(child, "Min") == 0) break_min = atof(value); 
         if(strcmp(child, "Max") == 0) break_max = atof(value); 
         if(strcmp(child, "Position") == 0) break_position = atof(value); 
      }

      gtk_plot_set_break(real_plot, the_axis->orientation,
                              break_min, break_max, break_step, break_nminor,
                              break_scale, break_position);

      if(!apply_break)
        gtk_plot_remove_break(real_plot, the_axis->orientation);
    }

    if(strcmp(name, "Title") == 0){
      gboolean visible = FALSE;

      state->state = PARSER_IN_TITLE;
      state->in_text = FALSE;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Visible") == 0) visible = atoi(value);
      }
      if(visible)
        gtk_plot_axis_show_title(state->axis);
      else
        gtk_plot_axis_hide_title(state->axis);
    }
   
    if(strcmp(name, "Labels") == 0){
      gint mask = 0;
      gint precision = 2;
      gint offset = 10;
      gint style = 0;
      gchar *prefix = NULL;
      gchar *suffix = NULL;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Mask") == 0) mask = atoi(value);
        if(strcmp(child, "Precision") == 0 ||
           strcmp(child, "Presicion") == 0) precision = atoi(value);
        if(strcmp(child, "Offset") == 0) offset = atoi(value);
        if(strcmp(child, "Style") == 0) style = atoi(value);
        if(strcmp(child, "Prefix") == 0) prefix = g_strdup(value);
        if(strcmp(child, "Suffix") == 0) suffix = g_strdup(value);
      }
      
      the_axis->labels_offset = offset;
      gtk_plot_axis_show_labels(state->axis, mask);
      gtk_plot_axis_set_labels_style(state->axis, style, precision);
      if(prefix){
        gtk_plot_axis_set_labels_prefix(state->axis, prefix);
        g_free(prefix);
      }
      if(suffix){
        gtk_plot_axis_set_labels_suffix(state->axis, suffix);
        g_free(suffix);
      }    
    }

    if(strcmp(name, "Minor") == 0){ 
      state->state = PARSER_IN_MINOR;
      state->in_text = FALSE;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Visible") == 0) the_axis->show_minor_grid = atoi(value); 
      }
    }

    if(strcmp(name, "Major") == 0){ 
      state->state = PARSER_IN_MAJOR;
      state->in_text = FALSE;

      while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Visible") == 0) the_axis->show_major_grid = atoi(value); 
      }
    }

  }

  if(state->state == PARSER_IN_AXIS && strcmp(name, "Ticklabels") == 0){
    gchar *worksheet_name = NULL;
    gint values_column = -1;
    gint labels_column = -1;

    while(aux && *aux){
      child = *aux++;
      value = *aux++;
      if(strcmp(child, "Worksheet") == 0) worksheet_name = g_strdup(value);
      if(strcmp(child, "Labels") == 0) labels_column = atoi(value);
      if(strcmp(child, "Values") == 0) values_column = atoi(value);
    }

    if(worksheet_name){
      SGworksheet *labels_worksheet;

      labels_worksheet = SG_WORKSHEET(sg_list_get(state->app->worksheets, worksheet_name));       

/* TODO
      if(labels_worksheet){ 
        sg_layer_set_tick_labels(state->layer, state->axis, labels_worksheet, 
                                 values_column, labels_column); 
      }
*/

      g_free(worksheet_name);
    }
  }
 
  if(state->state == PARSER_IN_LAYER && strcmp(name, "Legends") == 0){
     gdouble x = 0.6;
     gdouble y = 0.6;
     gint width = 10;
     gint height = 10;
     gboolean visible = TRUE;

     state->state = PARSER_IN_LEGENDS;
     state->in_text = FALSE;

     while(*aux){
        child = *aux++;
        value = *aux++;
        if(strcmp(child, "Visible") == 0) visible = atoi(value);
        if(strcmp(child, "X") == 0) x = atof(value);
        if(strcmp(child, "Y") == 0) y = atof(value);
        if(strcmp(child, "Width") == 0) width = atoi(value);
        if(strcmp(child, "Height") == 0) height = atoi(value);
     }

     if(visible)
       gtk_plot_show_legends(real_plot);  
     else
       gtk_plot_hide_legends(real_plot);  
     gtk_plot_legends_move(real_plot, x, y); 
  } 
  
  if(strcmp(name, "Border") == 0){
    GtkPlotBorderStyle style = GTK_PLOT_BORDER_NONE;
    gint width = 0;
    gint shadow = 3;
    gint space = 2;

    while(*aux){
       child = *aux++;
       value = *aux++;
       if(strcmp(child, "Style") == 0) style = (GtkPlotBorderStyle)atoi(value); 
       if(strcmp(child, "Space") == 0) space = atoi(value); 
       if(strcmp(child, "Line") == 0) width = atoi(value); 
       if(strcmp(child, "Shadow") == 0) shadow = atoi(value); 
    }

    if(state->in_text){
        state->text.border = style; 
        state->text.border_space = space; 
        state->text.border_width = width;
        state->text.shadow_width = shadow; 
    } else {
        gtk_plot_set_legends_border(real_plot, style, shadow);
        real_plot->legends_border_width = width;
    }
  }

  if(strcmp(name, "Dataset") == 0){
    SGdataset *real_data = NULL;
    gboolean show_labels = TRUE;
    gboolean show_gradient = FALSE;
    gboolean fill_area = FALSE;
    gint id = -1;
    GtkPlotConnector connector = GTK_PLOT_CONNECT_NONE;
    gint style = 0;
    const gchar *data_name = NULL;
    const gchar *exp = NULL;

    state->prev_state = state->state;
    state->state = PARSER_IN_DATA;
    state->in_text = FALSE;
    state->visible = TRUE;

    while(aux && *aux){
       child = *aux++;
       value = *aux++;
       if(strcmp(child, "ID") == 0) id = atoi(value); 
       if(strcmp(child, "Type") == 0) style = (atoi(value) - 2);/* before 0.7.2 */
       if(strcmp(child, "Style") == 0) style = atoi(value); 
       if(strcmp(child, "Visible") == 0) state->visible = atoi(value); 
       if(strcmp(child, "Labels") == 0) show_labels = atoi(value); 
       if(strcmp(child, "Connector") == 0) connector = (GtkPlotConnector)atoi(value); 
       if(strcmp(child, "Fill") == 0) fill_area = atoi(value); 
       if(strcmp(child, "Gradient") == 0) show_gradient = atoi(value); 
       if(strcmp(child, "Name") == 0) data_name = value; 
       if(strcmp(child, "Function") == 0) exp = value;
    }

    state->id = id;
    state->type = style;
    state->show_labels = show_labels;
    state->connector = connector;
    state->fill_area = fill_area;
    state->show_gradient = show_gradient;
    if(state->data_name) g_free(state->data_name);
    state->data_name = NULL;
    if(data_name) state->data_name = g_strdup(data_name);

/* FUNCTIONS */
    if(exp){
      SGpluginIterator *plugin;
      plugin = SG_PLUGIN_ITERATOR(sg_plugin_iterator_get("sg_dataset_function"));
      if(id >= 0){ /* after 0.7.2 */
        real_data = get_dataset_by_id(state->app, id);
        if(real_data) {
          state->dataset_child = sg_dataset_new_child(real_data);
          sg_dataset_add_child(real_data, state->dataset_child);
        } else {
          GtkArg args[1];

          args[0].type = GTK_TYPE_STRING;
          args[0].name = "SGdatasetFunction::exp";
          GTK_VALUE_STRING(args[0]) = (gchar *)exp;

          real_data = state->dataset = plugin->construct(plugin);
          sg_application_add_dataset(state->app, state->dataset);
          g_object_set(G_OBJECT(state->dataset), args[0].name, exp, NULL);
        }
        state->dataset->id = id; 
      }else{ /* before 0.7.2 */
        real_data = SG_DATASET(sg_list_get(state->app->datasets,name));
        if(real_data) {
          state->dataset_child = sg_dataset_new_child(real_data);
          sg_dataset_add_child(real_data, state->dataset_child);
        } else {
          GtkArg args[1];

          args[0].type = GTK_TYPE_STRING;
          args[0].name = "SGdatasetFunction::exp";
          GTK_VALUE_STRING(args[0]) = (gchar *)exp;

          real_data = state->dataset = plugin->construct(plugin);
          sg_application_add_dataset(state->app, state->dataset);
          g_object_set(G_OBJECT(state->dataset), args[0].name, exp, NULL);
        }
      }

      gtk_plot_data_show_labels(state->dataset_child, show_labels);
      gtk_plot_data_gradient_set_visible(state->dataset_child, show_gradient);
      gtk_plot_data_fill_area(state->dataset_child, fill_area);
      gtk_plot_data_set_connector(state->dataset_child, connector);

      if(state->data_name && strlen(state->data_name) > 0){
        sg_dataset_set_name(state->dataset, (gchar *)state->data_name);
      }
    }
  }  

  if(state->state == PARSER_IN_DATA){
    if(strcmp(name, "Surface") == 0){
      state->surface_transparent = FALSE;
      state->height_gradient= FALSE;
      state->use_amplitud = FALSE;

      state->surface_fg.red = 0; 
      state->surface_fg.green = 0; 
      state->surface_fg.blue = 65535; 
      state->surface_bg.red = 0; 
      state->surface_bg.green = 0; 
      state->surface_bg.blue = 0; 

      while(*aux){
         child = *aux++;
         value = *aux++;
         if(strcmp(child, "Transparent") == 0) state->surface_transparent = atoi(value); 
         if(strcmp(child, "Gradient") == 0) state->height_gradient = atoi(value); 
         if(strcmp(child, "UseHeight") == 0) state->height_gradient = atoi(value); 
         if(strcmp(child, "UseAmplitud") == 0) state->use_amplitud = atoi(value); 
         if(strcmp(child, "FGR") == 0) state->surface_fg.red = atoi(value); 
         if(strcmp(child, "FGG") == 0) state->surface_fg.green = atoi(value); 
         if(strcmp(child, "FGB") == 0) state->surface_fg.blue = atoi(value); 
         if(strcmp(child, "BGR") == 0) state->surface_bg.red = atoi(value); 
         if(strcmp(child, "BGG") == 0) state->surface_bg.green = atoi(value); 
         if(strcmp(child, "BGB") == 0) state->surface_bg.blue = atoi(value); 
      }
      gdk_color_alloc(gdk_colormap_get_system(), &state->surface_fg);
      gdk_color_alloc(gdk_colormap_get_system(), &state->surface_bg);

    }

    if(strcmp(name, "Grid") == 0){
      state->grid_visible = TRUE;

      while(*aux){
         child = *aux++;
         value = *aux++;
         if(strcmp(child, "Visible") == 0) state->grid_visible = atoi(value); 
         if(strcmp(child, "FGR") == 0) state->grid_fg.red = atoi(value); 
         if(strcmp(child, "FGG") == 0) state->grid_fg.green = atoi(value); 
         if(strcmp(child, "FGB") == 0) state->grid_fg.blue = atoi(value); 
         if(strcmp(child, "BGR") == 0) state->grid_bg.red = atoi(value); 
         if(strcmp(child, "BGG") == 0) state->grid_bg.green = atoi(value); 
         if(strcmp(child, "BGB") == 0) state->grid_bg.blue = atoi(value); 
      }
      gdk_color_alloc(gdk_colormap_get_system(), &state->grid_fg);
      gdk_color_alloc(gdk_colormap_get_system(), &state->grid_bg);
    }
    if(strcmp(name, "Mesh") == 0){
      state->mesh_visible = FALSE;

      while(*aux){
         child = *aux++;
         value = *aux++;
         if(strcmp(child, "Visible") == 0) state->mesh_visible = atoi(value); 
      }
    }
    if(strcmp(name, "Contour") == 0){
      state->contour_visible = TRUE;
      state->contour_project = TRUE;

      while(*aux){
         child = *aux++;
         value = *aux++;
         if(strcmp(child, "Visible") == 0) state->contour_visible = atoi(value);
         if(strcmp(child, "Project") == 0) state->contour_project = atoi(value);
      }

    }
    if(strcmp(name, "Legend") == 0){
      state->legends_visible = TRUE;
      state->legends_precision = 3;

      while(*aux){
         child = *aux++;
         value = *aux++;
         if(strcmp(child, "Visible") == 0) state->legends_visible = atoi(value);
         if(strcmp(child, "Precision") == 0) state->legends_precision = atoi(value); 
      }

    }

    if(strcmp(name, "Datasheet") == 0){
      gchar *data_name = NULL;
      gint col_index[10];
      gint i;
      for(i = 0; i < 10; i++) col_index[i] = -1;
      SGworksheet *datasheet = NULL;

      while(*aux){
         child = *aux++;
         value = *aux++;
         if(strcmp(child, "Name") == 0) data_name = g_strdup(value); 
         if(strcmp(child, "X") == 0) col_index[0] = atoi(value); 
         if(strcmp(child, "Y") == 0) col_index[1] = atoi(value); 
         if(strcmp(child, "Z") == 0) col_index[2] = atoi(value); 
         if(strcmp(child, "A") == 0) col_index[3] = atoi(value); 
         if(strcmp(child, "DX") == 0) col_index[4] = atoi(value); 
         if(strcmp(child, "DY") == 0) col_index[5] = atoi(value); 
         if(strcmp(child, "DZ") == 0) col_index[6] = atoi(value); 
         if(strcmp(child, "DA") == 0) col_index[7] = atoi(value); 
         if(strcmp(child, "Labels") == 0) col_index[8] = atoi(value); 
      }
  
      datasheet = SG_WORKSHEET(sg_list_get(state->app->worksheets, data_name));
      if(datasheet){
        SGpluginIterator *plugin = NULL;
        SGpluginStyle *style_plugin = NULL;
        SGdataset *real_data = NULL;
        gboolean new_child = FALSE;

        if(GTK_IS_SG_MATRIX(datasheet) || col_index[2] != -1)
          style_plugin = get_constructor(state->type, SG_LAYER_3D);
        else
          style_plugin = get_constructor(state->type, SG_LAYER_2D);

        if(GTK_IS_SG_MATRIX(datasheet))
          plugin = sg_plugin_iterator_get("sg_dataset_matrix");
        else
          plugin = sg_plugin_iterator_get("sg_dataset_worksheet");

        if(state->id >= 0){ /* after 0.7.2 */
          real_data = get_dataset_by_id(state->app,state->id);
          if(real_data){
            new_child = TRUE;
            state->dataset_child = sg_dataset_new_child(real_data);
            sg_dataset_add_child(real_data, state->dataset_child);
          } else {
            real_data = state->dataset = sg_plugin_iterator_construct(plugin, style_plugin);
            sg_application_add_dataset(state->app, state->dataset);
            state->dataset_child = sg_dataset_new_child(state->dataset);
          }
          state->dataset->id = state->id; 
        }else{ /* before 0.7.2 */
          real_data = SG_DATASET(sg_list_get(state->app->datasets,name));
          if(real_data){
            state->dataset_child = sg_dataset_new_child(real_data);
            sg_dataset_add_child(real_data, state->dataset_child);
            new_child = TRUE;
          } else {
            real_data = state->dataset = sg_plugin_iterator_construct(plugin, style_plugin);
            sg_application_add_dataset(state->app, state->dataset);
            state->dataset_child = sg_dataset_new_child(real_data);
            sg_dataset_add_child(real_data, state->dataset_child);
            new_child = TRUE;
          }
        }

        if(!new_child){
          if(GTK_IS_SG_MATRIX(datasheet)){
            GtkArg args[1];
  
            args[0].type = GTK_TYPE_POINTER;
            args[0].name = "SGdatasetMatrix::matrix";
            GTK_VALUE_POINTER(args[0]) = datasheet;
            g_object_set(G_OBJECT(real_data), args[0].name, datasheet, NULL);
          }else{
            gint i;
            GList *list = style_plugin->arrays->arrays;
            gchar *dim_name[] = {"x","y","z","a","dx","dy","dz","da","labels"};
            GtkArg args[1];
            args[0].type = GTK_TYPE_POINTER;
    	    GTK_VALUE_POINTER(args[0]) = datasheet;
            args[0].name = "SGdatasetWorksheet::worksheet";
            g_object_set(G_OBJECT(real_data), args[0].name, datasheet, NULL);
  
            args[0].type = GTK_TYPE_INT;
            for(i = 0; i < 9; i++){
              gint n = 0;
              list = style_plugin->arrays->arrays;
              while(list){
                GtkPlotArray *array = GTK_PLOT_ARRAY(list->data);
                if(strcmp(array->name, dim_name[i]) == 0){
                  gchar col_name[200];
                  g_snprintf(col_name,200,"SGdatasetWorksheet::col_%02d",n);
                  args[0].name = col_name;
                  GTK_VALUE_INT(args[0]) = col_index[i];
                  g_object_set(G_OBJECT(real_data), args[0].name, col_index[i], NULL);
                }
                n++;
                list = list->next;
              }
            }
          }
  
          gtk_plot_data_show_labels(state->dataset_child, state->show_labels);
          gtk_plot_data_gradient_set_visible(state->dataset_child, state->show_gradient);
          gtk_plot_data_fill_area(state->dataset_child, state->fill_area);
          gtk_plot_data_set_connector(state->dataset_child, state->connector);
        }

        if(state->data_name && strlen(state->data_name) > 0){
          sg_dataset_set_name(real_data, state->data_name);
        }
        sg_dataset_refresh(real_data);
      }
    }

    if(strcmp(name, "Expressionsdata") == 0){
      SGpluginIterator *plugin = NULL;
      SGpluginStyle *style_plugin = NULL;
      SGdataset *real_data = NULL;
      const gchar *data_name = NULL;
      GtkArg args[9];
      gchar *p_exp[9];
      gint i;
    
      args[0].name = "SGdatasetPython::exp_x";
      args[1].name = "SGdatasetPython::exp_y";
      args[2].name = "SGdatasetPython::exp_z";
      args[3].name = "SGdatasetPython::exp_a";
      args[4].name = "SGdatasetPython::exp_x";
      args[5].name = "SGdatasetPython::exp_y";
      args[6].name = "SGdatasetPython::exp_z";
      args[7].name = "SGdatasetPython::exp_a";
      args[8].name = "SGdatasetPython::exp_labels";
 
      for(i = 0; i < 9; i++) {
        args[i].type = GTK_TYPE_STRING;
        p_exp[i] = NULL;
      }
 
      while(*aux){
         child = *aux++;
         value = *aux++;
         if(strcmp(child, "Name") == 0) data_name = value; 
         if(strcmp(child, "X") == 0) p_exp[0] = g_strdup((gchar *)value); 
         if(strcmp(child, "Y") == 0) p_exp[1] = g_strdup((gchar *)value);
         if(strcmp(child, "Z") == 0) p_exp[2] = g_strdup((gchar *)value); 
         if(strcmp(child, "A") == 0) p_exp[3] = g_strdup((gchar *)value);
         if(strcmp(child, "DX") == 0) p_exp[4] = g_strdup((gchar *)value);
         if(strcmp(child, "DY") == 0) p_exp[5] = g_strdup((gchar *)value);
         if(strcmp(child, "DZ") == 0) p_exp[6] = g_strdup((gchar *)value);
         if(strcmp(child, "DA") == 0) p_exp[7] = g_strdup((gchar *)value);
         if(strcmp(child, "Labels") == 0) p_exp[8] = g_strdup(value);
      }

      for(i = 0; i < 9; i++) GTK_VALUE_STRING(args[i]) = p_exp[i];

      if(p_exp[2])
        style_plugin = get_constructor(state->type, SG_LAYER_3D);
      else
        style_plugin = get_constructor(state->type, SG_LAYER_2D);

      plugin = sg_plugin_iterator_get("sg_dataset_python");

      if(state->id >= 0){ /* after 0.7.2 */
        real_data = get_dataset_by_id(state->app, state->id);
        if(real_data)
          state->dataset_child = sg_dataset_new_child(real_data);
        else {
          state->dataset = sg_plugin_iterator_construct(plugin, style_plugin);
          sg_application_add_dataset(state->app, state->dataset);
          state->dataset_child = sg_dataset_new_child(state->dataset);
        }
        state->dataset->id = state->id; 
      }else{ /* before 0.7.2 */
        real_data = SG_DATASET(sg_list_get(state->app->datasets,name));
        if(real_data)
          state->dataset_child = sg_dataset_new_child(real_data);
        else {
          state->dataset = sg_plugin_iterator_construct(plugin, style_plugin);
          sg_application_add_dataset(state->app, state->dataset);
          state->dataset_child = sg_dataset_new_child(state->dataset);
        }
      }

      for(i = 0 ; i < 9; i++) 
        g_object_set(G_OBJECT(state->dataset), args[i].name, p_exp[i], NULL);
      for(i = 0 ; i < 9; i++) g_free(p_exp[i]);

      gtk_plot_data_show_labels(state->dataset_child, state->show_labels);
      gtk_plot_data_gradient_set_visible(state->dataset_child, state->show_gradient);
      gtk_plot_data_fill_area(state->dataset_child, state->fill_area);
      gtk_plot_data_set_connector(state->dataset_child, state->connector);

      if(state->data_name && strlen(state->data_name) > 0){
        sg_dataset_set_name(state->dataset, state->data_name);
      }
    }

    if(strcmp(name, "Xline") == 0){

      state->state = PARSER_IN_XLINE;
      state->in_text = FALSE;

    }

    if(strcmp(name, "Yline") == 0){

      state->state = PARSER_IN_YLINE;
      state->in_text = FALSE;

    }

    if(strcmp(name, "Zline") == 0){

      state->state = PARSER_IN_ZLINE;
      state->in_text = FALSE;

    }

    if(strcmp(name, "Gradient") == 0){
      GdkColor color_min, color_max;
      GdkColor color_lt_min = state->dataset_child->color_lt_min;
      GdkColor color_gt_max = state->dataset_child->color_gt_max;
      gdouble min = 0, max = 0;
      gint nlevels = 0, nsublevels = 0, mask = 0;
      gboolean custom = FALSE;
      gboolean visible = state->dataset_child->show_gradient;

      while(*aux){
         child = *aux++;
         value = *aux++;
         if(strcmp(child, "Custom") == 0) custom = atoi(value); 
         if(strcmp(child, "Visible") == 0) visible = atoi(value); 
         if(strcmp(child, "Mask") == 0) mask = atoi(value); 
         if(strcmp(child, "Min") == 0) min = atof(value); 
         if(strcmp(child, "Max") == 0) max = atof(value); 
         if(strcmp(child, "Nlevels") == 0) nlevels = atoi(value); 
         if(strcmp(child, "Nsublevels") == 0) nsublevels = atoi(value); 
         if(strcmp(child, "Rmin") == 0) color_min.red = atoi(value); 
         if(strcmp(child, "Gmin") == 0) color_min.green = atoi(value); 
         if(strcmp(child, "Bmin") == 0) color_min.blue = atoi(value); 
         if(strcmp(child, "Rmax") == 0) color_max.red = atoi(value); 
         if(strcmp(child, "Gmax") == 0) color_max.green = atoi(value); 
         if(strcmp(child, "Bmax") == 0) color_max.blue = atoi(value); 
         if(strcmp(child, "Rltmin") == 0) color_lt_min.red = atoi(value); 
         if(strcmp(child, "Gltmin") == 0) color_lt_min.green = atoi(value); 
         if(strcmp(child, "Bltmin") == 0) color_lt_min.blue = atoi(value); 
         if(strcmp(child, "Rgtmax") == 0) color_gt_max.red = atoi(value); 
         if(strcmp(child, "Ggtmax") == 0) color_gt_max.green = atoi(value); 
         if(strcmp(child, "Bgtmax") == 0) color_gt_max.blue = atoi(value); 
      }
      gdk_color_alloc(gdk_colormap_get_system(), &color_min);
      gdk_color_alloc(gdk_colormap_get_system(), &color_max);
      gdk_color_alloc(gdk_colormap_get_system(), &color_lt_min);
      gdk_color_alloc(gdk_colormap_get_system(), &color_gt_max);

      state->dataset_child->gradient_mask = mask;
      gtk_plot_data_gradient_use_custom_colors(state->dataset_child, custom); 
      gtk_plot_data_set_gradient_colors(state->dataset_child, 
                                        &color_min, &color_max);
      gtk_plot_data_set_gradient_outer_colors(state->dataset_child, 
                                        &color_lt_min, &color_gt_max);
      gtk_plot_data_set_gradient(state->dataset_child, min, max, nlevels, nsublevels);
      gtk_plot_data_gradient_set_visible(state->dataset_child, visible);

    }

    if(strcmp(name, "GradientColor") == 0){
      GdkColor color;
      gint nlevel = 0;

      while(*aux){
         child = *aux++;
         value = *aux++;
         if(strcmp(child, "Level") == 0) nlevel = atoi(value); 
         if(strcmp(child, "R") == 0) color.red = atoi(value); 
         if(strcmp(child, "G") == 0) color.green = atoi(value); 
         if(strcmp(child, "B") == 0) color.blue = atoi(value); 
      }
      gdk_color_alloc(gdk_colormap_get_system(), &color);
      gtk_plot_data_set_gradient_nth_color(state->dataset_child, nlevel, &color);
    }
  }

  if(state->state == PARSER_IN_DATA && strcmp(name, "Points") == 0){
    state->nx = state->ny = 0;

    state->state = PARSER_IN_POINTS;
    state->in_text = FALSE;

    while(*aux){
       child = *aux++;
       value = *aux++;

       if(strcmp(child, "NX") == 0) state->nx = atoi(value); 
       if(strcmp(child, "NY") == 0) state->ny = atoi(value); 
    }
  }

/* FIXME 
  if(state->object == PARSE_PLOT && state->state == PARSER_IN_POINTS && strcmp(name, "Point") == 0){
    gdouble x = 0.;
    gdouble y = 0.;
    gdouble z = 0.;
    gdouble a = 0.;
    gdouble dx = 0.;
    gdouble dy = 0.;
    gdouble dz = 0.;
    gdouble da = 0.;
 
    while(*aux){
       child = *aux++;
       value = *aux++;

       if(strcmp(child, "X") == 0) x = atof(value); 
       if(strcmp(child, "Y") == 0) y = atof(value); 
       if(strcmp(child, "Z") == 0) z = atof(value); 
       if(strcmp(child, "A") == 0) a = atof(value); 
       if(strcmp(child, "DX") == 0) dx = atof(value); 
       if(strcmp(child, "DY") == 0) dy = atof(value); 
       if(strcmp(child, "DZ") == 0) dz = atof(value); 
       if(strcmp(child, "DA") == 0) da = atof(value); 
    }
    sg_dataset_add_point(state->dataset, &x, &y, &z, &a, &dx, &dy, &dz, &da, NULL);
  }
*/

  if(state->state == PARSER_IN_DATA && strcmp(name, "Xerrbars") == 0){
    gboolean visible = FALSE;
    gint width = 0;
    gint caps = 0;

    while(*aux){
       child = *aux++;
       value = *aux++;

       if(strcmp(child, "Visible") == 0) visible = atoi(value); 
       if(strcmp(child, "Width") == 0) width = atoi(value); 
       if(strcmp(child, "Caps") == 0) caps = atoi(value); 
    }

    if(visible)
       gtk_plot_data_show_xerrbars(state->dataset_child);
    state->dataset_child->xerrbar_width = width;
    state->dataset_child->xerrbar_caps = caps;
  }

  if(state->state == PARSER_IN_DATA && strcmp(name, "Yerrbars") == 0){
    gboolean visible = FALSE;
    gint width = 0;
    gint caps = 0;

    while(*aux){
       child = *aux++;
       value = *aux++;

       if(strcmp(child, "Visible") == 0) visible = atoi(value); 
       if(strcmp(child, "Width") == 0) width = atoi(value); 
       if(strcmp(child, "Caps") == 0) caps = atoi(value); 
    }

    if(visible)
       gtk_plot_data_show_yerrbars(state->dataset_child);
    state->dataset_child->yerrbar_width = width;
    state->dataset_child->yerrbar_caps = caps;
  }

  if(state->state == PARSER_IN_DATA && strcmp(name, "Zerrbars") == 0){
    gboolean visible = FALSE;
    gint width = 0;
    gint caps = 0;

    while(*aux){
       child = *aux++;
       value = *aux++;

       if(strcmp(child, "Visible") == 0) visible = atoi(value); 
       if(strcmp(child, "Width") == 0) width = atoi(value); 
       if(strcmp(child, "Caps") == 0) caps = atoi(value); 
    }

    if(visible)
       gtk_plot_data_show_zerrbars(state->dataset_child);
    state->dataset_child->zerrbar_width = width;
    state->dataset_child->zerrbar_caps = caps;
  }


  if(state->state == PARSER_IN_DATA && strcmp(name, "Datalabels") == 0){
    gboolean visible = FALSE;
    gint offset = 6;

    state->in_text = FALSE;

    while(*aux){
       child = *aux++;
       value = *aux++;

       if(strcmp(child, "Visible") == 0) visible = atoi(value); 
       if(strcmp(child, "Offset") == 0) offset = atoi(value); 
    }

    gtk_plot_data_show_labels(state->dataset_child, visible);
    state->dataset_child->labels_offset = offset;
  }

/*
  if(state->state == PARSER_IN_DATA && strcmp(name, "Label") == 0){
    const gchar *text = NULL;
    gint npoint = 0;
 
    while(*aux){
       child = *aux++;
       value = *aux++;

       if(strcmp(child, "Text") == 0) text = value; 
       if(strcmp(child, "Point") == 0) npoint = atoi(value); 
    }

    if(state->dataset_child->labels && 
       npoint < state->dataset_child->num_points && text)
       state->dataset_child->labels[npoint] = g_strdup(text);
    }
  }
*/

/***********************************************************
 * LINES
 ***********************************************************/
  if(strcmp(name, "Line") == 0){
    GtkPlotAxis *the_axis;
    GtkPlotLineStyle style = GTK_PLOT_LINE_NONE;
    GdkCapStyle cap_style = 0;
    GdkJoinStyle join_style = 0;
    gfloat width = 0.0;
    GdkColor color;

    while(*aux){
       child = *aux++;
       value = *aux++;

       if(strcmp(child, "Style") == 0) style = (GtkPlotLineStyle)atoi(value); 
       if(strcmp(child, "Width") == 0) width = atof(value); 
       if(strcmp(child, "R") == 0) color.red = atoi(value); 
       if(strcmp(child, "G") == 0) color.green = atoi(value); 
       if(strcmp(child, "B") == 0) color.blue = atoi(value); 
    }
    gdk_color_alloc(gdk_colormap_get_system(), &color);

    switch(state->state){
        case PARSER_IN_AXIS:
          gtk_plot_axis_set_attributes(state->axis, width, &color);
          
	  break;
        case PARSER_IN_FRAME:
          if(GTK_IS_PLOT3D(real_plot))
            gtk_plot3d_frame_set_attributes(GTK_PLOT3D(real_plot), style, width, &color);
          break;
        case PARSER_IN_CORNER:
          if(GTK_IS_PLOT3D(real_plot))
            gtk_plot3d_corner_set_attributes(GTK_PLOT3D(real_plot), style, width, &color);
          break;
        case PARSER_IN_VMAJOR:
          gtk_plot_major_vgrid_set_attributes(real_plot, style, width, &color);
	  break;
        case PARSER_IN_VMINOR:
          gtk_plot_minor_vgrid_set_attributes(real_plot, style, width, &color);
	  break;
        case PARSER_IN_HMAJOR:
          gtk_plot_major_hgrid_set_attributes(real_plot, style, width, &color);
	  break;
        case PARSER_IN_HMINOR:
          gtk_plot_minor_hgrid_set_attributes(real_plot, style, width, &color);
	  break;
        case PARSER_IN_MAJOR:
          the_axis = state->axis;
          the_axis->major_grid.line_style = style;
          the_axis->major_grid.line_width = width;
          the_axis->major_grid.color = color;
	  break;
        case PARSER_IN_MINOR:
          the_axis = state->axis;
          the_axis->minor_grid.line_style = style;
          the_axis->minor_grid.line_width = width;
          the_axis->minor_grid.color = color;
	  break;
        case PARSER_IN_X0:
          gtk_plot_x0line_set_attributes(real_plot, style, width, &color);
	  break;
        case PARSER_IN_Y0:
          gtk_plot_y0line_set_attributes(real_plot, style, width, &color);
	  break;
        case PARSER_IN_DATA:
          gtk_plot_data_set_line_attributes(state->dataset_child,
                                            style, cap_style, join_style, 
                                            width, &color);
	  break;
        case PARSER_IN_XLINE:
          gtk_plot_data_set_x_attributes(state->dataset_child,
                                         style, cap_style, join_style, 
                                         width, &color);
	  break;
        case PARSER_IN_YLINE:
          gtk_plot_data_set_y_attributes(state->dataset_child,
                                         style, cap_style, join_style, 
                                         width, &color);
	  break;
        case PARSER_IN_ZLINE:
          gtk_plot_data_set_z_attributes(state->dataset_child,
                                         style, cap_style, join_style, 
                                         width, &color);
	  break;
        case PARSER_IN_CHILD:
          state->line.line_style = style; 
          state->line.line_width = width; 
          state->line.color = color; 
	  break;
        default:
	  break;
    }

  }

/***********************************************************
 * SYMBOLS 
 ***********************************************************/
  if(strcmp(name, "Symbol") == 0){
    GtkPlotSymbolType type = GTK_PLOT_SYMBOL_NONE;
    GtkPlotSymbolStyle style = GTK_PLOT_SYMBOL_EMPTY;
    gint size = 0;
    gfloat width = 0.0;
    gint arrow_width = 0;
    gint arrow_length = 12;
    GtkPlotSymbolStyle arrow_style = GTK_PLOT_SYMBOL_EMPTY;
    gint arrow_centered = TRUE;
    gdouble bar_width = 0.05;
    gdouble scale = 1.0;
    GdkColor color, border;
 
    while(*aux){
       child = *aux++;
       value = *aux++;

       if(strcmp(child, "Type") == 0) type = (GtkPlotSymbolType)atoi(value); 
       if(strcmp(child, "Style") == 0) style = (GtkPlotSymbolStyle)atoi(value); 
       if(strcmp(child, "Size") == 0) size = atoi(value); 
       if(strcmp(child, "Width") == 0) width = atof(value); 
       if(strcmp(child, "ArrowStyle") == 0) arrow_style = (GtkPlotSymbolStyle)atoi(value); 
       if(strcmp(child, "ArrowLength") == 0) arrow_length = atoi(value); 
       if(strcmp(child, "ArrowWidth") == 0) arrow_width = atoi(value); 
       if(strcmp(child, "Center") == 0) arrow_centered = atoi(value);
       if(strcmp(child, "BarWidth") == 0) bar_width = atof(value); 
       if(strcmp(child, "Scale") == 0) scale = atof(value); 
       if(strcmp(child, "R") == 0) border.red = color.red = atoi(value); 
       if(strcmp(child, "G") == 0) border.green = color.green = atoi(value); 
       if(strcmp(child, "B") == 0) border.blue = color.blue = atoi(value); 
       if(strcmp(child, "BorderR") == 0) border.red = atoi(value); 
       if(strcmp(child, "BorderG") == 0) border.green = atoi(value); 
       if(strcmp(child, "BorderB") == 0) border.blue = atoi(value); 
    }
    gdk_color_alloc(gdk_colormap_get_system(), &color);
    gdk_color_alloc(gdk_colormap_get_system(), &border);

    gtk_plot_data_set_symbol(state->dataset_child,
                             type, style, size, width, &color, &border);
    gtk_plot_data_set_a_scale(state->dataset_child, scale);

    if(GTK_IS_PLOT_FLUX(state->dataset_child)){
       GTK_PLOT_FLUX(state->dataset_child)->arrow_style = arrow_style;
       GTK_PLOT_FLUX(state->dataset_child)->arrow_length = arrow_length;
       GTK_PLOT_FLUX(state->dataset_child)->arrow_width = arrow_width;
       GTK_PLOT_FLUX(state->dataset_child)->centered = arrow_centered;
    }
    if(GTK_IS_PLOT_BAR(state->dataset_child))
         GTK_PLOT_BAR(state->dataset_child)->width = bar_width;
  }
/***********************************************************
 * TEXT 
 ***********************************************************/
  if(strcmp(name, "Text") == 0){
    gdouble x = 0.;
    gdouble y = 0.;
    gint angle = 0;
    GtkJustification justification = GTK_JUSTIFY_FILL;
    gboolean transparent = TRUE;

    state->in_text = TRUE;
 
    while(*aux){
       child = *aux++;
       value = *aux++;

       if(strcmp(child, "X") == 0) x = atof(value); 
       if(strcmp(child, "Y") == 0) y = atof(value); 
       if(strcmp(child, "Angle") == 0) angle = atoi(value); 
       if(strcmp(child, "Justification") == 0) justification = (GtkJustification)atoi(value); 
       if(strcmp(child, "Transparent") == 0) transparent = atoi(value); 
    }

    state->text.x = x;
    state->text.y = y;
    state->text.angle = angle;
    state->text.justification = justification;
    state->text.transparent = transparent;
  }

  if(strcmp(name, "Foreground") == 0){
    GdkColor color;
 
    while(*aux){
       child = *aux++;
       value = *aux++;

       if(strcmp(child, "R") == 0) color.red = atoi(value); 
       if(strcmp(child, "G") == 0) color.green = atoi(value); 
       if(strcmp(child, "B") == 0) color.blue = atoi(value); 
    }
    gdk_color_alloc(gdk_colormap_get_system(), &color);
    if(state->in_text) state->text.fg = color;
  }

  if(strcmp(name, "Background") == 0){
    GdkColor color;
    gint transparent = TRUE;
 
    while(*aux){
       child = *aux++;
       value = *aux++;

       if(strcmp(child, "R") == 0) color.red = atoi(value); 
       if(strcmp(child, "G") == 0) color.green = atoi(value); 
       if(strcmp(child, "B") == 0) color.blue = atoi(value); 
       if(strcmp(child, "Transparent") == 0) transparent = atoi(value); 
    }
    gdk_color_alloc(gdk_colormap_get_system(), &color);

    if(state->in_text) 
       state->text.bg = color;
    else
      switch(state->state){
       case PARSER_IN_PAGE:
       case PARSER_IN_PLOT:
          GTK_PLOT_CANVAS(state->plot)->background = color;
          gtk_plot_canvas_set_transparent(GTK_PLOT_CANVAS(plot), transparent);
          break;
       case PARSER_IN_LAYER:
          GTK_PLOT_CANVAS_PLOT(state->layer)->plot->background = color;
          gtk_plot_set_transparent(GTK_PLOT_CANVAS_PLOT(state->layer)->plot, transparent);
          break;
       default:
          break;
      }
  }
 
  if(strcmp(name, "Font") == 0){
    const gchar *font_name = NULL; 
    gint height = 12;

    while(*aux){
       child = *aux++;
       value = *aux++;

       if(strcmp(child, "Name") == 0) font_name = value; 
       if(strcmp(child, "Height") == 0) height = atoi(value); 
    }

    if(state->text.font)
      g_free(state->text.font);

    state->text.font = g_strdup(font_name);
    state->text.height = height;
  }
}

static void
sgEndElement(SGparseState *state, const XCHAR *real_name)
{
  SGworksheet *worksheet = state->worksheet;
  GtkSheet *sheet = NULL;
  GdkPixmap *pixmap = NULL;
  const XCHAR *name;

  name = get_real_name(real_name);

  if(worksheet) sheet = GTK_SHEET(worksheet);
  
  if(strcmp(name, "Worksheet") == 0 || strcmp(name, "Matrix") == 0){
     SGworksheet *worksheet;
     gint nrows, ncols;

     state->prev_state = state->state;
     state->state = PARSER_UNKNOWN;
     state->layer = NULL;

     worksheet = state->worksheet;
     ncols = gtk_sheet_get_columns_count(GTK_SHEET(worksheet));
     nrows = gtk_sheet_get_rows_count(GTK_SHEET(worksheet));

  }

  if(strcmp(name, "MaxRow") == 0)
     sg_worksheet_add_rows(state->worksheet,
                           atoi(state->content->str)-sheet->maxrow);

  if(strcmp(name, "MaxCol") == 0)
     sg_worksheet_add_columns(state->worksheet,
                              atoi(state->content->str)-sheet->maxcol);

  if(state->state == PARSER_IN_WORKSHEET && strcmp(name, "Name") == 0){
     sg_worksheet_rename(state->worksheet, state->content->str);
     sg_application_rename_worksheet(state->app, state->worksheet);
  }

  if(strcmp(name, "Begin") == 0)
     sg_worksheet_set_begin(state->worksheet, atoi(state->content->str));

  if(strcmp(name, "End") == 0)
    sg_worksheet_set_end(state->worksheet, atoi(state->content->str));

  if(strcmp(name, "Content") == 0)
     sg_worksheet_cell_set(state->worksheet, state->row, state->col,
                           state->content->str, FALSE, FALSE);
  if(strcmp(name, "Formula") == 0)
      sg_worksheet_cell_set(state->worksheet, state->row, state->col,
                                 state->content->str, TRUE,FALSE);

  if(strcmp(name, "ColFormula") == 0 && strlen(state->content->str)>0)
  {
       sg_worksheet_column_set_exp(state->worksheet, state->col,
                                   g_strdup(state->content->str));
  }

  if(state->state == PARSER_IN_PLOT && strcmp(name, "Name") == 0){
     sg_plot_rename(state->plot, state->content->str); 
     sg_application_rename_plot(state->app, state->plot);
  }

  /* This is to fix a bug: tag Page closed in the wrong place */
  if(state->state == PARSER_IN_PAGE && strcmp(name, "Name") == 0){
     sg_plot_rename(state->plot, state->content->str); 
     sg_application_rename_plot(state->app, state->plot);
     state->state = PARSER_IN_PLOT;
  }

  if(strcmp(name, "Plot") == 0){
     state->prev_state = state->state;
     state->state = PARSER_UNKNOWN;
     state->num_layers = 0;
  }

  if(strcmp(name, "Page") == 0){
     state->state = PARSER_IN_PLOT;
  }

  if(strcmp(name, "Child") == 0){
     GtkPlotCanvas *canvas;
     GtkPlotCanvasChild *child = NULL;

     state->state = PARSER_IN_PLOT;

     canvas = GTK_PLOT_CANVAS(state->plot);
     
     switch(state->child_type){
       case GTK_PLOT_CANVAS_TEXT: 
         return;
       case GTK_PLOT_CANVAS_LINE: 
         child = gtk_plot_canvas_line_new(
                                          state->line.line_style,
                                          state->line.line_width,
                                          &state->line.color,
                                          state->arrow.arrow_mask);
         gtk_plot_canvas_put_child(canvas, child,
                                   state->child.rx1, state->child.ry1,
                                   state->child.rx2, state->child.ry2);
         break;
       case GTK_PLOT_CANVAS_RECTANGLE: 
         child = gtk_plot_canvas_rectangle_new(
                                          state->line.line_style,
                                          state->line.line_width,
                                          &state->line.color,
                                          &state->rectangle.bg,
                                          state->rectangle.border,
                                          state->rectangle.filled);
         gtk_plot_canvas_put_child(canvas, child,
                                   state->child.rx1, state->child.ry1,
                                   state->child.rx2, state->child.ry2);
         break;
       case GTK_PLOT_CANVAS_ELLIPSE: 
         child = gtk_plot_canvas_ellipse_new(
                                          state->line.line_style,
                                          state->line.line_width,
                                          &state->line.color,
                                          &state->ellipse.bg,
                                          state->ellipse.filled);
         gtk_plot_canvas_put_child(canvas, child,
                                   state->child.rx1, state->child.ry1,
                                   state->child.rx2, state->child.ry2);
         break;
       case GTK_PLOT_CANVAS_PIXMAP: 
         pixmap = gdk_pixmap_new(state->window, state->pwidth, state->pheight, -1); 
         gdk_draw_image(pixmap, state->gc, state->image, 0, 0, 0, 0,
                        state->pwidth, state->pheight);

         child = gtk_plot_canvas_pixmap_new(pixmap, NULL);
         gtk_plot_canvas_put_child(canvas, child,
                                   state->child.rx1, state->child.ry1,
                                   state->child.rx2, state->child.ry2);
         gdk_image_destroy(state->image);
         state->image = NULL;
         state->img_char = 0;
         break;
       default:
         break;
     }

  }

  if(strcmp(name, "Layer") == 0){
     state->prev_state = state->state;
     state->state = PARSER_IN_PLOT;
  }

  if(strcmp(name, "Pixels") == 0){
     if(state->state == PARSER_IN_PIXMAP)
        state->state = PARSER_IN_CHILD;
     if(state->state == PARSER_IN_BG_PIXMAP)
        state->state = PARSER_IN_LAYER;
  }
     
  if(strcmp(name, "BGpixmap") == 0){
     pixmap = gdk_pixmap_new(state->window, state->pwidth, state->pheight, -1); 
     gdk_draw_image(pixmap, state->gc, state->image, 0, 0, 0, 0,
                    state->pwidth, state->pheight);

     gtk_plot_set_background_pixmap(GTK_PLOT_CANVAS_PLOT(state->layer)->plot, pixmap);
     gdk_image_destroy(state->image);
     state->image = NULL;
     state->img_char = 0;
  }
 
  if(strcmp(name, "Vmajor") == 0){
     state->state = PARSER_IN_LAYER;
  }

  if(strcmp(name, "Vminor") == 0){
     state->state = PARSER_IN_LAYER;
  }

  if(strcmp(name, "Hmajor") == 0){
     state->state = PARSER_IN_LAYER;
  }

  if(strcmp(name, "Hminor") == 0){
     state->state = PARSER_IN_LAYER;
  }

  if(strcmp(name, "X0") == 0){
     state->state = PARSER_IN_LAYER;
  }

  if(strcmp(name, "Y0") == 0){
     state->state = PARSER_IN_LAYER;
  }

  if(strcmp(name, "Axis") == 0){
     state->state = PARSER_IN_LAYER;
  }

  if(strcmp(name, "Ticks") == 0){
     state->state = PARSER_IN_AXIS;
  }

  if(strcmp(name, "Title") == 0){
     state->state = PARSER_IN_AXIS;
  }

  if(strcmp(name, "Labels") == 0){
     state->state = PARSER_IN_AXIS;
  }

  if(strcmp(name, "Major") == 0){
     state->state = PARSER_IN_AXIS;
  }

  if(strcmp(name, "Minor") == 0){
     state->state = PARSER_IN_AXIS;
  }

  if(strcmp(name, "Legends") == 0){
     state->state = PARSER_IN_LAYER;
  }

  if(strcmp(name, "Frame") == 0){
     state->state = PARSER_IN_LAYER;
  }

  if(strcmp(name, "Corner") == 0){
     state->state = PARSER_IN_LAYER;
  }

  if(strcmp(name, "Side") == 0){
     state->state = PARSER_IN_LAYER;
  }

  if(strcmp(name, "Legend") == 0 && state->state == PARSER_IN_DATA){
     if(state->legend) g_free(state->legend);
     state->legend = g_strdup(state->content->str);
  }

  if(strcmp(name, "String") == 0 && state->in_text){
     state->text.text = g_strdup(state->content->str);
  }

  if(strcmp(name, "Pixmap") == 0){
     state->state = PARSER_IN_CHILD;
     if(state->pixels) g_free(state->pixels);
     state->pixels = g_new0(gulong, 1);
     state->ncolors = 0;
  }

  if(strcmp(name, "XPMColor") == 0){
     GdkColor color;
     gchar color_str[17];

     g_snprintf(color_str, 17, "#%s", state->content->str);

     gdk_color_parse(color_str, &color);
/*
     printf("%s %d %d %d\n",color_str,color.red,color.green,color.blue);
*/
     gdk_color_alloc(gdk_colormap_get_system(), &color);

     state->pixels = (gulong *)g_realloc(state->pixels, (state->ncolors+1)*sizeof(gulong)); 
     state->pixels[state->ncolors] = color.pixel;

/*
     printf("%d %d\n",state->ncolors,color.pixel);
*/
     state->ncolors++;
  }


  if(strcmp(name, "Text") == 0){
     GtkPlotCanvasChild *child = NULL;

     switch(state->state){
       case PARSER_IN_PLOT:
       case PARSER_IN_CHILD:
         child = gtk_plot_canvas_text_new(
				  state->text.font,
				  state->text.height,
				  state->text.angle,
				  &state->text.fg,
				  &state->text.bg,
				  state->text.transparent,
				  state->text.justification,
				  state->text.text);
         gtk_plot_text_set_border(&(GTK_PLOT_CANVAS_TEXT(child)->text),
				  state->text.border,
				  state->text.border_space,
				  state->text.border_width,
				  state->text.shadow_width);
         gtk_plot_canvas_put_child(GTK_PLOT_CANVAS(state->plot), child,
				   state->text.x, state->text.y, 0., 0.);
         break;
       case PARSER_IN_LEGENDS:
         gtk_plot_legends_set_attributes(GTK_PLOT_CANVAS_PLOT(state->layer)->plot,
				         state->text.font,
				         state->text.height,
				         &state->text.fg,
				         &state->text.bg);
         break;
       case PARSER_IN_AXIS:
         gtk_plot_axis_set_labels_attributes(state->axis,
			 	             state->text.font,
				             state->text.height,
				             state->text.angle,
				             &state->text.fg,
				             &state->text.bg,
                                             state->text.transparent,
                                             state->text.justification);

         break;
       case PARSER_IN_TITLE:
         gtk_plot_axis_move_title(state->axis,
	                          state->text.angle,
                                  state->text.x, state->text.y);
         gtk_plot_axis_title_set_attributes(state->axis,
				            state->text.font,
				            state->text.height,
				            state->text.angle,
				            &state->text.fg,
				            &state->text.bg,
                                            state->text.transparent,
                                            state->text.justification);
         if(state->text.text && strlen(state->text.text) > 0){
           gtk_plot_axis_set_title(state->axis,
                                   state->text.text);
         }
         break;
       case PARSER_IN_DATA:
         gtk_plot_data_labels_set_attributes(state->dataset_child,
				             state->text.font,
				             state->text.height,
				             state->text.angle,
				             &state->text.fg,
				             &state->text.bg);
         break;
       default:
         break;
     }

     state->in_text = FALSE;

     if(state->text.text)
        g_free(state->text.text);
     state->text.text = NULL;
     if(state->text.font)
        g_free(state->text.font);
     state->text.font = NULL;
     return;
  }

  if(strcmp(name, "Dataset") == 0){
     if(!state->legends_visible)
       gtk_plot_data_hide_legend(state->dataset_child);
     state->dataset_child->legends_precision = state->legends_precision;
     gtk_plot_data_set_legend(state->dataset_child, state->legend); 
     if(state->legend) g_free(state->legend);
     state->legend = NULL;

     if(GTK_IS_PLOT_SURFACE(state->dataset_child)){
       GtkPlotSurface *surface = GTK_PLOT_SURFACE(state->dataset_child);
       surface->color = state->surface_fg;
       surface->shadow = state->surface_bg;
       surface->transparent = state->surface_transparent;
       surface->use_height_gradient = state->height_gradient;
       surface->use_amplitud = state->use_amplitud;
       surface->grid_foreground = state->grid_fg;
       surface->grid_background = state->grid_bg;
       surface->show_grid = state->grid_visible;
       surface->show_mesh = state->mesh_visible;
       surface->nx = state->nx;
       surface->ny = state->ny;
     }
     if(GTK_IS_PLOT_CSURFACE(state->dataset_child)){
       GtkPlotCSurface *surface = GTK_PLOT_CSURFACE(state->dataset_child);
       surface->lines_visible = state->contour_visible;
       surface->projection = state->contour_project;
     }
     if(state->prev_state == PARSER_IN_LAYER){
       sg_layer_add_dataset_child(state->layer, state->dataset_child);
       if(!state->visible) gtk_widget_hide(GTK_WIDGET(state->dataset_child));
       state->prev_state = state->state;
       state->state = PARSER_IN_LAYER;
     }else{
       state->state = state->prev_state;
       gtk_widget_destroy(GTK_WIDGET(state->dataset_child));
     }

  }

  if(strcmp(name, "Points") == 0){
     state->state = PARSER_IN_DATA;
  }

  if(strcmp(name, "Datalabels") == 0){
     state->state = PARSER_IN_DATA;
  }

  if(strcmp(name, "Xline") == 0){
     state->state = PARSER_IN_DATA;
  }

  if(strcmp(name, "Yline") == 0){
     state->state = PARSER_IN_DATA;
  }

  if(strcmp(name, "Zline") == 0){
     state->state = PARSER_IN_DATA;
  }

  g_free(state->name);
  state->name = NULL;
}

gboolean  
project_sax_open(SGpluginFile *plugin,
                    const gchar *filename,
                    FILE *file,
                    GObject **object, gpointer data)
{
  SGapplication *app = SG_APPLICATION(*object);
  return (default_open(filename, app, NULL, NULL));
}

static gboolean  
default_open(const gchar *file, SGapplication* app, SGworksheet *worksheet, SGplot *plot)
{
  SGparseState state;
  GdkWindowAttr attributes;
  gint attributes_mask;
  xmlTextReaderPtr reader;
  gint i = 0;
  gchar *doc_version = NULL;
  GObject *object = G_OBJECT(app);

  /* Check for Doc version first */
                                                                         
  reader = xmlNewTextReaderFilename(file);
  if(!reader) return FALSE;
  for(i = 1; i < 20; i++){
    gint ret_val;
    ret_val = xmlTextReaderRead(reader);
    if(!ret_val) break;
                                                                         
    if(xmlTextReaderNodeType(reader) == 1){
      xmlChar *name = xmlTextReaderName(reader);
      if(strcmp(name,"sg:Doc") == 0){
                                                                         
        while(xmlTextReaderMoveToNextAttribute(reader)){
          xmlChar *child = xmlTextReaderName(reader);
          xmlChar *value = xmlTextReaderValue(reader);
                                                                         
          if(strcmp(child, "version") == 0) doc_version = g_strdup(value);
                                                                         
          xmlFree(child);
          xmlFree(value);
        }
      }
      xmlFree(name);
    }
  }
  xmlFreeTextReader(reader);

  /* Read document */
                                                                         
  if(doc_version != NULL){
    SGpluginFile *xml_plugin;

    g_free(doc_version);
    xml_plugin = sg_plugin_file_get("xml", "project", SG_PLUGIN_FILE_OPEN);
    if(xml_plugin)
      return (sg_plugin_file_action(xml_plugin, file, NULL, &object, NULL));
    return TRUE;
  } 

  state.worksheet = worksheet;
  state.plot = plot;
  state.app = app;
  state.layer = NULL;
  state.dataset = NULL;
  state.image = NULL;

  state.text.font = NULL;
  state.text.text = NULL;
  state.text.height = 12;

  state.in_text = FALSE;
  state.last_worksheet = -1;
  state.last_matrix = 0;
  state.last_plot = -1;
  state.last_dataset = -1;
  state.last_function = -1;
  state.last_expression = -1;

  state.color_str[4] = '\0';

  state.data_name = NULL;
  state.legend = NULL;

  state.num_layers = 0;

  state.object = PARSE_PROJECT;
  if(worksheet) state.object = PARSE_WORKSHEET;
  if(plot) state.object = PARSE_PLOT;

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.title = NULL;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gdk_visual_get_system ();
  attributes.colormap = gdk_colormap_get_system ();
  attributes.event_mask = 0;
  attributes_mask = GDK_WA_VISUAL | GDK_WA_COLORMAP;
  
  state.window = gdk_window_new (NULL, &attributes, attributes_mask);

  state.gc = gdk_gc_new(state.window);

  state.ncolors = 0;
  state.pixels = g_new0(gulong, 1);

  xmlSubstituteEntitiesDefault(TRUE);

  if( xmlSAXUserParseFile(&sgSAXParser, &state, file) ) {
       /* has sgEndDocument() freed all the necessary pointers ? */
       sg_message_dialog("Document not well formed!", 1);
       return FALSE;
  }

  if(state.data_name) g_free(state.data_name);
  if(state.legend) g_free(state.legend);

  return TRUE;
}

static const XCHAR *
get_real_name(const XCHAR *name)
{
  const XCHAR *real_name = name;

  while(real_name && real_name != '\0'){
    if(*real_name++ == ':') return real_name;
  }

  return name;
} 

