/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <string.h>
#include <gtkextra/gtkextra.h>
#include <scigraphica/sg.h>
#include "sg_project.h"
#include "pixmaps/sg_small.xpm"
#include <libxml/xmlreader.h>

typedef struct
{
  SGpluginFile *plugin;
  SGapplication *app;
  gchar *last_node;
  gchar *doc_version;
  FILE *file;
} parser_state;

static gboolean default_open     		(SGpluginFile *plugin, 
		    				 const gchar *filename, 
		    				 FILE *file,
                    				 GObject **object, 
						 gpointer data);

void
project_xml_open_init(SGplugin *_plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;
  SGpluginFile *plugin = SG_PLUGIN_FILE(_plugin);

  plugin->mode = SG_PLUGIN_FILE_OPEN;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".sg");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

void
project_xml_import_init(SGplugin *_plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;
  SGpluginFile *plugin = SG_PLUGIN_FILE(_plugin);

  plugin->mode = SG_PLUGIN_FILE_IMPORT;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".sg");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

static void
process_node(xmlTextReaderPtr reader, parser_state *state)
{
  xmlChar *name;

  name = xmlTextReaderName(reader);

/*
  printf("Node %d %d %s %d\n",xmlTextReaderDepth(reader), xmlTextReaderNodeType(reader), name, xmlTextReaderIsEmptyElement(reader));
*/

  /* Start Element */

  if(xmlTextReaderNodeType(reader) == 1){
    if(strcmp(name, "sgw:Matrix") == 0){
      SGpluginFile *plugin = sg_plugin_file_get("xml", "SGmatrix", SG_PLUGIN_FILE_OPEN);
      GObject *matrix = NULL;
      gchar *name;
      
      plugin->action(plugin, NULL, state->file, &matrix, reader);
      name = g_strdup(SG_WORKSHEET(matrix)->name);
      sg_application_add_matrix(state->app, SG_WORKSHEET(matrix));
      sg_worksheet_rename(SG_WORKSHEET(matrix),name);
      sg_application_rename_worksheet(state->app, SG_WORKSHEET(matrix));
      g_free(name);
    }
    if(strcmp(name, "sgw:Worksheet") == 0){
      SGpluginFile *plugin = sg_plugin_file_get("xml", "SGworksheet", SG_PLUGIN_FILE_OPEN);
      GObject *worksheet = NULL;
      gchar *name;
      
      plugin->action(plugin, NULL, state->file, &worksheet, reader);
      name = g_strdup(SG_WORKSHEET(worksheet)->name);
      sg_application_add_worksheet(state->app, SG_WORKSHEET(worksheet));
      sg_worksheet_rename(SG_WORKSHEET(worksheet),name);
      sg_application_rename_worksheet(state->app, SG_WORKSHEET(worksheet));
      g_free(name);
    }
    if(strcmp(name, "sgp:Plot") == 0){
      SGpluginFile *plugin = sg_plugin_file_get("xml", "SGplot", SG_PLUGIN_FILE_OPEN);
      GObject *plot = NULL;
      gchar *name;

      plugin->action(plugin, NULL, state->file, &plot, reader);
      name = g_strdup(SG_PLOT(plot)->name);

      /* check for existing --linked-- datasets, before adding plot to app */
      if(state->plugin->mode == SG_PLUGIN_FILE_OPEN || state->plugin->mode == SG_PLUGIN_FILE_IMPORT){
        if(SG_PLOT(plot)->datasets != state->app->datasets){
          GList *plot_list = SG_PLOT(plot)->datasets->list;
          while(plot_list){
            SGlistChild *plot_child = (SGlistChild *)plot_list->data;
            GList *app_list = state->app->datasets->list;
            gboolean found = FALSE;
            while(app_list){
              SGlistChild *app_child = (SGlistChild *)app_list->data;
              if(SG_DATASET(app_child->object)->id == SG_DATASET(plot_child->object)->id && strcmp(app_child->name, plot_child->name) == 0){
                SGdataset *app_dataset = SG_DATASET(app_child->object);
                SGdataset *plot_dataset = SG_DATASET(plot_child->object);
                GList *children;
                gboolean empty = app_dataset->ref_count == 0;

                found = TRUE;
  
                app_dataset->children = g_list_concat(app_dataset->children, plot_dataset->children);
                app_dataset->ref_count++;
                plot_dataset->children = NULL;
                
                if(empty){
  	 	  if(app_dataset->arrays)
                    g_object_unref(G_OBJECT(app_dataset->arrays));
                  app_dataset->arrays = plot_dataset->arrays;
                  g_object_ref(G_OBJECT(plot_dataset->arrays));
                } else {
  		  if(plot_dataset->arrays)
                    g_object_unref(G_OBJECT(plot_dataset->arrays));
                  plot_dataset->arrays = app_dataset->arrays;
                  g_object_ref(G_OBJECT(plot_dataset->arrays));
                }
                children = app_dataset->children;
                while(children){
                  GtkPlotData *data_child = GTK_PLOT_DATA(children->data);
                  data_child->link = app_dataset;
                  children = children->next;
                }
                break;
              } 
              app_list = app_list->next;
            }        
            if(!found){
              sg_application_add_dataset(state->app, SG_DATASET(plot_child->object));
              plot_child->object = NULL;
            }
            plot_list = plot_list->next;
          }
        }
      }

      sg_application_add_plot(state->app, SG_PLOT(plot));
      sg_plot_rename(SG_PLOT(plot),name);
      sg_application_rename_plot(state->app, SG_PLOT(plot));
      g_free(name);
    }
    if(strcmp(name, "sgp:Dataset") == 0){
      SGpluginFile *plugin = sg_plugin_file_get("xml", "sg_dataset", SG_PLUGIN_FILE_OPEN);
      GObject *dataset = NULL;
      plugin->action(plugin, NULL, state->file, &dataset, reader);
      if(dataset) {
        sg_application_add_dataset(state->app, SG_DATASET(dataset));
      }
    }

    if(state->last_node) g_free(state->last_node);
    state->last_node = g_strdup(name);
  }

  /* Text node */

  if(state->last_node && xmlTextReaderNodeType(reader) == 3){
    xmlChar *value = xmlTextReaderValue(reader);
    if(!value) return;

    if(strcmp(state->last_node, "sg:DocVersion") == 0){
      state->doc_version = g_strdup(value);
    }

    xmlFree(value);
    if(state->last_node) g_free(state->last_node);
    state->last_node = g_strdup(name);
  }

  xmlFree(name);
}

gboolean
project_xml_open   (SGpluginFile *plugin, 
		    const gchar *filename, 
		    FILE *file,
                    GObject **object, gpointer data)
{  
  sg_project_close();
  sg_project_new(FALSE);
  return (default_open(plugin, filename, file, object, data));
}

gboolean
project_xml_import   (SGpluginFile *plugin, 
		      const gchar *filename, 
		      FILE *file,
                      GObject **object, gpointer data)
{  
  return (default_open(plugin, filename, file, object, data));
}

static gboolean
default_open       (SGpluginFile *plugin, 
		    const gchar *filename, 
		    FILE *file,
                    GObject **object, gpointer data)
{
  xmlTextReaderPtr reader;
  gint ret_val = 0;
  gint i = 0;
  parser_state *state;

  state = g_new0(parser_state, 1);
  state->last_node = NULL;
  state->app = SG_APPLICATION(*object);
  state->file = file;
  state->plugin = plugin;

  /* Check for Doc version first */

  reader = xmlNewTextReaderFilename(filename);
  if(!reader) return FALSE;
  for(i = 1; i < 20; i++){
    ret_val = xmlTextReaderRead(reader);
    if(!ret_val) break;

    if(xmlTextReaderNodeType(reader) == 1){
      xmlChar *name = xmlTextReaderName(reader);
      if(strcmp(name,"sg:Doc") == 0){

        while(xmlTextReaderMoveToNextAttribute(reader)){
          xmlChar *child = xmlTextReaderName(reader);
          xmlChar *value = xmlTextReaderValue(reader);

          if(strcmp(child, "version") == 0) state->doc_version = g_strdup(value);

          xmlFree(child);
          xmlFree(value);
        }
      }
      xmlFree(name);
    }
  }
  xmlFreeTextReader(reader);

  /* Read document */

  if(state->doc_version == NULL){
    SGpluginFile *sax_plugin;
    sax_plugin = sg_plugin_file_get("sax", "project", SG_PLUGIN_FILE_OPEN);
    if(sax_plugin)
      return (sg_plugin_file_action(sax_plugin, filename, file, object, data)); 
    ret_val = 1;
  } else {
    reader = xmlNewTextReaderFilename(filename);
    if(!reader) return FALSE;
    while(ret_val == 1){
      process_node(reader, state);
      ret_val = xmlTextReaderRead(reader);
    }
    xmlFreeTextReader(reader);
  }

  if(state->last_node) g_free(state->last_node);
  if(state->doc_version) g_free(state->doc_version);
  g_free(state);

  if(ret_val != 0) return FALSE;

  return TRUE;
}

