/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <scigraphica/sg.h>
#include <scigraphica/sg_application.h>
#include <scigraphica/dialogs/sg_misc_dialogs.h>
#include "sg_project_statusbar.h"
#include "sg_project_autosave.h"
#include "sg_project_rescue.h"
#include "sg_main.h"


gboolean sg_project_changed_since_last_autosave = FALSE;

extern gchar* last_project_path;
extern gchar* last_project_filename;

static gchar autosave_filename[250] = "";
gint sg_autosave_interval = 5;
gint sg_compress_level = 6;

static gint autosave(gpointer data);
 

/* set new autosave timeout when interval (in minutes) > 0
 * otherwise disable timeout */ 
void sg_project_autosave_set( gint autosave_interval )
{
 static gint *timeout_tag = NULL;

 if ( autosave_interval > 0 )
    {
     if ( timeout_tag )
        gtk_timeout_remove( *timeout_tag ); /* remove previous timeout */
     else
        timeout_tag = g_new(gint, 1);

     /* set new timeout; gtk_timeout_add wants milliseconds */
     *timeout_tag = gtk_timeout_add(autosave_interval * 60 * 1000,
                                    (GtkFunction)autosave, NULL );
    }
 else if ( timeout_tag )
    {/* remove autosave timeout and filename */
     gtk_timeout_remove(*timeout_tag);
     g_free(timeout_tag);
     timeout_tag = NULL;
     
     if ( *autosave_filename ) remove(autosave_filename);
     *autosave_filename = '\0';
    }
}

/* autosave the current project if necessary
 * return TRUE/FALSE: continues/terminates subsequent autosave timeouts */
static
gint autosave( gpointer data )
{
 static gboolean autosave_locked = FALSE;
 SGpluginFile *plugin;

 plugin = sg_plugin_file_get("xml","Project",SG_PLUGIN_FILE_SAVE);
 if(!plugin) return FALSE;

 /* locked prevents autosave invokations within autosave */
 if ( autosave_locked ) return TRUE;
 autosave_locked = TRUE;

 if ( sg_application_changed(app) == FALSE )
    {
     /* only remove previous autosave file; no save needed, nothing has changed */
     if (*autosave_filename) remove(autosave_filename);
     *autosave_filename = '\0';
    }
 else if ( sg_project_changed_since_last_autosave )
    {
     /* autosave the current project */
     if ( *autosave_filename ) remove(autosave_filename);
     g_snprintf(autosave_filename, 250, "%s/" AUTOSAVE_PREFIX "%s", last_project_path,
                                                                    last_project_filename);
     if ( plugin->action (plugin, autosave_filename, NULL, NULL, NULL) )
        sg_project_changed_since_last_autosave = FALSE;
     else
        sg_project_statusbar_set( _("autosave failed!") , SG_STATUSBAR_TEXTONLY);
    }

 /* unlock */
 autosave_locked = FALSE;

 return TRUE;
}


/* Check for a newer rescued or autosaved version of the file. Pop up a dialog that asks
 * whether to open the newer rescued/autosaved file instead. This situation can occur when
 * SciGraphica is forcefully terminated, leaving rescue and/or autosave files behind. */
gchar*
sg_project_autosave_check(gchar *filename)
{
 gboolean autosave_exist = FALSE,
          rescue_exist = FALSE;

 gchar autosave_fn[250], rescue_fn[250];
 struct stat file_sb, autosave_sb, rescue_sb;
 
 /* quit when stat on filename fails */
 if ( stat(filename, &file_sb) == -1 ) return filename;

 /* stat the autosave file and check if it is more recent */
 g_snprintf(autosave_fn, 250, "%s/%s%s", AUTOSAVE_PREFIX, g_dirname(filename), g_basename(filename));
 if ( stat(autosave_fn, &autosave_sb) != -1 && autosave_sb.st_mtime > file_sb.st_mtime )
    autosave_exist = TRUE;

 /* stat the rescue file and check whether it is more recent */
 g_snprintf(rescue_fn, 250, "%s/%s%s", RESCUE_PREFIX, g_dirname(filename), g_basename(filename));
 if ( stat(rescue_fn, &rescue_sb) != -1 )
    {if ( autosave_exist && rescue_sb.st_mtime > autosave_sb.st_mtime )
        rescue_exist = TRUE;
     else if ( rescue_sb.st_mtime > file_sb.st_mtime )
        rescue_exist = TRUE;
    }

 if ( rescue_exist && sg_accept_dialog( _("Found rescued file that is more recent.\nOpen that instead?"), 1) == SG_BUTTON_YES )
    g_snprintf(filename, 1+strlen(rescue_fn), "%s", rescue_fn);
 else if ( autosave_exist && sg_accept_dialog( _("Found autosaved file that is more recent.\nOpen that instead?"), 1) == SG_BUTTON_YES )
    g_snprintf(filename, 1+strlen(autosave_fn), "%s", autosave_fn);
 
 return filename;
}
