/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_main.h"
#include "sg_project.h"
#include "pixmaps/sheet_icon2.xpm"
#include "pixmaps/plot_icon2.xpm"
#include "pixmaps/matrix_icon.xpm"
#include <scigraphica/dialogs/sg_misc_dialogs.h>

gboolean project_name_changed = FALSE;
gchar *last_project_filename = NULL;
gchar *last_project_path = NULL;

gulong handler1, handler2;

extern gboolean sg_show_plot_toolbar;
extern gboolean sg_show_plot_menubar;
extern gboolean sg_show_toolbar_labels;

extern gint project_quit_warning(GtkWidget *widget);

GtkIconListItem *active_icon;

static gboolean rename_worksheet	(GtkIconList *iconlist, 
					 GtkIconListItem *item);
static gboolean rename_plot		(GtkIconList *iconlist, 
					 GtkIconListItem *item);
static void	sg_project_changed	(GtkObject *object);
static void 	sg_project_new_worksheet(SGapplication *app, 
					 SGworksheet *worksheet);
static void 	sg_project_new_matrix	(SGapplication *app, 
					 SGworksheet *worksheet);
static void 	sg_project_new_plot	(SGapplication *app, 
					 SGplot *plot);
static void 	sg_project_remove_worksheet(SGapplication *app, 
					 SGworksheet *worksheet);
static void 	sg_project_remove_matrix(SGapplication *app, 
					 SGworksheet *worksheet);
static void 	sg_project_remove_plot	(SGapplication *app, 
					 SGplot *plot);
static void 	sg_project_rename_worksheet(SGapplication *app, 
					 SGworksheet *worksheet);
static void 	sg_project_rename_matrix(SGapplication *app, 
					 SGworksheet *worksheet);
static void 	sg_project_rename_plot	(SGapplication *app, 
					 SGplot *plot);
static void 	sg_project_activate_worksheet(SGapplication *app, 
					 SGworksheet *worksheet);
static void 	sg_project_activate_plot(SGapplication *app, 
					 SGplot *plot);
static gint     delete_event            (GtkWidget *window);

static gboolean project_plugins_initialized = FALSE;

void
sg_project_plugins_init()
{
  if(project_plugins_initialized) return;
  project_plugins_initialized = TRUE;
}

void
sg_project_new(gboolean default_worksheet)
{
  active_icon = NULL;

  sg_application_set_changed(app, FALSE);
  sg_project_set_filename(_("untitled.sg") );
  sg_project_set_path(g_get_current_dir());
  sg_project_set_title();

  if((default_worksheet && gui_mode == SG_GUI_ICONLIST) || gui_mode == SG_GUI_WORKSHEET){
     SGworksheet *worksheet;
     gchar name[200];
     worksheet = sg_worksheet_new(_("Data01"),20,5);
     sg_application_add_worksheet(app, worksheet);
     if (gui_mode == SG_GUI_WORKSHEET) gtk_widget_show(GTK_WIDGET(worksheet));
  }
                                                                                
  if(gui_mode == SG_GUI_PLOT){
     SGplot *plot;
     SGpluginLayer *layer = sg_plugin_layer_get("layer_2d");
     plot = sg_plot_new_with_layer(layer,_("Plot01"));
     sg_application_add_plot(app, plot);
     gtk_widget_show(GTK_WIDGET(plot));
  }
                                                                                
  if (gui_mode == SG_GUI_ICONLIST){
     handler2 = gtk_signal_connect(GTK_OBJECT(iconlist2), "deactivate_icon",
                        (GtkSignalFunc)rename_plot, iconlist2);

     handler1 = gtk_signal_connect(GTK_OBJECT(iconlist1), "deactivate_icon",
                        (GtkSignalFunc)rename_worksheet, iconlist1);
  }
  sg_application_set_changed(app, FALSE);
}

void
sg_project_set_title()
{
  gchar window_title[1000];
  gchar icon_title[1000];

  if (gui_mode != SG_GUI_ICONLIST || !main_window || !GTK_IS_WIDGET(main_window) || !GTK_WIDGET(main_window)->window) return;

  if (sg_application_changed(app)){
     g_snprintf(window_title, 1000, _("SciGraphica: %s (changed)"), last_project_filename);
     g_snprintf(icon_title, 1000, _("%s (changed)"), last_project_filename);
  } else {
     g_snprintf(window_title, 1000, "SciGraphica: %s", last_project_filename);
     g_snprintf(icon_title, 1000, "%s", last_project_filename);
  }

  gtk_window_set_title(GTK_WINDOW(main_window), window_title);
  gdk_window_set_icon_name(GTK_WIDGET(main_window)->window, icon_title);
}

void
sg_project_set_path(const gchar *path)
{
  if(last_project_path) g_free(last_project_path);
  last_project_path = NULL;
  if(path) last_project_path = g_strdup(path);
  sg_application_set_path(app, path);
}

void
sg_project_set_filename(const gchar *filename)
{
  if(last_project_filename) g_free(last_project_filename);
  last_project_filename = NULL;
  if(filename) 
    last_project_filename = g_strdup(filename);
  else
    last_project_filename = g_strdup( _("untitled.sg") );
  sg_application_set_filename(app, filename);
  sg_project_set_title();
}

static void
sg_project_changed(GtkObject *object)
{
  sg_project_set_title();
  sg_project_changed_since_last_autosave = sg_application_changed(app);
}

static void 
sg_project_new_worksheet(SGapplication *app, SGworksheet *worksheet)
{
 GtkWidget *toolbar = NULL;

 if (gui_mode == SG_GUI_ICONLIST)
 {
  GtkIconList *iconlist = GTK_ICON_LIST(iconlist1);
  GtkIconListItem *icon;
  SGworksheetWindow *window;

  window = sg_worksheet_window_new(worksheet);
  icon = gtk_icon_list_add_from_data(iconlist,
                                     sheet_icon2_xpm,
                                     worksheet->name,
                                     window);
  gtk_object_set_data(GTK_OBJECT(worksheet), "icon", icon);
  toolbar = gtk_object_get_data(GTK_OBJECT(window),"menubar");
  if(toolbar) gtk_toolbar_set_style(GTK_TOOLBAR(toolbar), sg_show_toolbar_labels?GTK_TOOLBAR_BOTH:GTK_TOOLBAR_ICONS);

  gtk_icon_list_set_active_icon(iconlist, icon);

  gtk_entry_select_region(GTK_ENTRY(icon->entry), 0, strlen(worksheet->name));
  sg_entry_cancel_space(GTK_ENTRY(icon->entry));

  gtk_notebook_set_page(GTK_NOTEBOOK(gui_notebook), 0);

  gtk_signal_connect(GTK_OBJECT(window),"delete_event",
                     (GtkSignalFunc) delete_event, NULL);
 }
 if(gui_mode == SG_GUI_WORKSHEET){
  SGworksheetWindow *window;

  window = sg_worksheet_window_new(worksheet);
  toolbar = gtk_object_get_data(GTK_OBJECT(window),"menubar");
  if(toolbar) gtk_toolbar_set_style(GTK_TOOLBAR(toolbar), sg_show_toolbar_labels?GTK_TOOLBAR_BOTH:GTK_TOOLBAR_ICONS);
  gtk_signal_connect(GTK_OBJECT(window),"delete_event",
                     (GtkSignalFunc) delete_event, NULL);
  gtk_widget_show(GTK_WIDGET(window));
 }

}

static void
sg_project_new_matrix(SGapplication *app, SGworksheet *worksheet)
{
 GtkWidget *toolbar = NULL;
 if (gui_mode == SG_GUI_ICONLIST)
 {
  GtkIconList *iconlist = GTK_ICON_LIST(iconlist1);
  GtkIconListItem *icon;
  SGworksheetWindow *window;

  window = sg_worksheet_window_new(worksheet);
  toolbar = gtk_object_get_data(GTK_OBJECT(window),"menubar");
  if(toolbar) gtk_toolbar_set_style(GTK_TOOLBAR(toolbar), sg_show_toolbar_labels?GTK_TOOLBAR_BOTH:GTK_TOOLBAR_ICONS);
  icon = gtk_icon_list_add_from_data(iconlist,
                                     matrix_icon_xpm,
                                     worksheet->name,
                                     window);
  gtk_object_set_data(GTK_OBJECT(worksheet), "icon", icon);

  gtk_icon_list_set_active_icon(iconlist, icon);

  gtk_entry_select_region(GTK_ENTRY(icon->entry), 0, strlen(worksheet->name));
  sg_entry_cancel_space(GTK_ENTRY(icon->entry));

  gtk_notebook_set_page(GTK_NOTEBOOK(gui_notebook), 0);

  gtk_signal_connect(GTK_OBJECT(window),"delete_event",
                     (GtkSignalFunc) delete_event, NULL);
 }
 if(gui_mode == SG_GUI_WORKSHEET){
  SGworksheetWindow *window;

  window = sg_worksheet_window_new(worksheet);
  toolbar = gtk_object_get_data(GTK_OBJECT(window),"menubar");
  if(toolbar) gtk_toolbar_set_style(GTK_TOOLBAR(toolbar), sg_show_toolbar_labels?GTK_TOOLBAR_BOTH:GTK_TOOLBAR_ICONS);
  gtk_signal_connect(GTK_OBJECT(window),"delete_event",
                     (GtkSignalFunc) delete_event, NULL);
  gtk_widget_show(GTK_WIDGET(window));
 }
}

static void
sg_project_new_plot(SGapplication *app, SGplot *plot)
{
 GtkWidget *toolbar;
 if (gui_mode == SG_GUI_ICONLIST)
 {
  GtkIconList *iconlist = GTK_ICON_LIST(iconlist2);
  GtkIconListItem *icon;
  SGplotWindow *plot_window;

  plot_window = sg_plot_window_new(plot);

  toolbar = gtk_object_get_data(GTK_OBJECT(plot_window),"menubar");
  if(toolbar) gtk_toolbar_set_style(GTK_TOOLBAR(toolbar), sg_show_toolbar_labels?GTK_TOOLBAR_BOTH:GTK_TOOLBAR_ICONS);

  sg_plot_window_set_toolbar_visible(SG_PLOT_WINDOW(plot_window),
                                     sg_show_plot_toolbar);
  sg_plot_window_set_menubar_visible(SG_PLOT_WINDOW(plot_window),
                                     sg_show_plot_menubar);

  icon = gtk_icon_list_add_from_data(iconlist, plot_icon2_xpm, plot->name, plot_window);
  gtk_object_set_data(GTK_OBJECT(plot), "icon", icon);

  gtk_icon_list_set_active_icon(iconlist, icon);
  gtk_entry_select_region(GTK_ENTRY(icon->entry), 0, strlen(plot->name));
  sg_entry_cancel_space(GTK_ENTRY(icon->entry));

  gtk_notebook_set_page(GTK_NOTEBOOK(gui_notebook), 1);

  gtk_signal_connect(GTK_OBJECT(plot_window),"delete_event",
                     (GtkSignalFunc) delete_event, NULL);
 }
 if(gui_mode == SG_GUI_PLOT){
  SGplotWindow *plot_window;

  plot_window = sg_plot_window_new(plot);

  toolbar = gtk_object_get_data(GTK_OBJECT(plot_window),"menubar");
  if(toolbar) gtk_toolbar_set_style(GTK_TOOLBAR(toolbar), sg_show_toolbar_labels?GTK_TOOLBAR_BOTH:GTK_TOOLBAR_ICONS);

  sg_plot_window_set_toolbar_visible(SG_PLOT_WINDOW(plot_window),
                                     sg_show_plot_toolbar);
  sg_plot_window_set_menubar_visible(SG_PLOT_WINDOW(plot_window),
                                     sg_show_plot_menubar);

  gtk_signal_connect(GTK_OBJECT(plot_window),"delete_event",
                     (GtkSignalFunc) delete_event, NULL);
  gtk_widget_show(GTK_WIDGET(plot_window));
 }
}

static void
sg_project_remove_worksheet(SGapplication *app, SGworksheet *worksheet)
{
  if(gui_mode == SG_GUI_ICONLIST){
    GtkIconListItem *icon = (GtkIconListItem *)gtk_object_get_data(GTK_OBJECT(worksheet), "icon");
    SGworksheetWindow *window = SG_WORKSHEET_WINDOW(icon->link);

    gtk_widget_ref(GTK_WIDGET(worksheet));
    gtk_widget_hide(GTK_WIDGET(window));
    if(GTK_IS_WIDGET(window)) gtk_widget_destroy(GTK_WIDGET(window));

    gtk_icon_list_remove(GTK_ICON_LIST(iconlist1), icon);
    gtk_icon_list_update(GTK_ICON_LIST(iconlist1));
  }

  if(gui_mode == SG_GUI_WORKSHEET && !app->worksheets->list) {
    sg_project_quit();
  }
}

static void
sg_project_remove_matrix(SGapplication *app, SGworksheet *worksheet)
{
  if(gui_mode == SG_GUI_ICONLIST){
    GtkIconListItem *icon = (GtkIconListItem *)gtk_object_get_data(GTK_OBJECT(worksheet), "icon");
    SGworksheetWindow *window = SG_WORKSHEET_WINDOW(icon->link);

    gtk_widget_ref(GTK_WIDGET(worksheet));
    gtk_widget_hide(GTK_WIDGET(window));
    if(GTK_IS_WIDGET(window)) gtk_widget_destroy(GTK_WIDGET(window));

    gtk_icon_list_remove(GTK_ICON_LIST(iconlist1), icon);
    gtk_icon_list_update(GTK_ICON_LIST(iconlist1));
  }

  if(gui_mode == SG_GUI_WORKSHEET && !app->worksheets->list) {
    sg_project_quit();
  }
}

static void
sg_project_remove_plot(SGapplication *app, SGplot *plot)
{
  if(gui_mode == SG_GUI_ICONLIST){
    GtkIconListItem *icon = (GtkIconListItem *)gtk_object_get_data(GTK_OBJECT(plot), "icon");
    SGplotWindow *plot_window = SG_PLOT_WINDOW(icon->link);

    gtk_widget_ref(GTK_WIDGET(plot));
    gtk_widget_hide(GTK_WIDGET(plot_window));
    if(GTK_IS_WIDGET(plot_window)) gtk_widget_destroy(GTK_WIDGET(plot_window));

    gtk_icon_list_remove(GTK_ICON_LIST(iconlist2), icon);
    gtk_icon_list_update(GTK_ICON_LIST(iconlist2));
  }

  if(gui_mode == SG_GUI_PLOT && !app->plots->list) {
    sg_project_quit();
  }
}

void
sg_project_close()
{
  GList *list;

  if(gui_mode == SG_GUI_ICONLIST){
    gtk_signal_disconnect(GTK_OBJECT(iconlist1), handler1);
    gtk_signal_disconnect(GTK_OBJECT(iconlist2), handler2);
  }

  list = app->plots->list;
  while(list){
    SGlistChild *child = (SGlistChild *)list->data;
    sg_application_remove_plot(app, SG_PLOT(child->object));
    list = app->plots->list;
  }  
  list = app->datasets->list;
  while(list){
    SGlistChild *child = (SGlistChild *)list->data;
    sg_application_remove_dataset(app, SG_DATASET(child->object));
    list = app->datasets->list;
  }  
  list = app->worksheets->list;
  while(list){
    SGlistChild *child = (SGlistChild *)list->data;
    sg_application_remove_worksheet(app, SG_WORKSHEET(child->object));
    list = app->worksheets->list;
  }  
  GTK_ICON_LIST(iconlist1)->active_icon = NULL;
  GTK_ICON_LIST(iconlist2)->active_icon = NULL;
/*
  sg_list_clear(app->plots);
  sg_list_clear(app->datasets);
  sg_list_clear(app->worksheets);
  gtk_icon_list_clear(GTK_ICON_LIST(iconlist1));
  gtk_icon_list_clear(GTK_ICON_LIST(iconlist2));
  app->num_worksheets = app->num_plots = 0;
  app->last_worksheet = app->last_matrix = app->last_plot = app->last_dataset = 0;

  if(gui_mode == SG_GUI_ICONLIST){
    gtk_icon_list_set_active_icon(GTK_ICON_LIST(iconlist1), NULL);
    gtk_icon_list_set_active_icon(GTK_ICON_LIST(iconlist2), NULL);
  }
*/
}

void
sg_project_quit()
{
#ifdef __WITH_GNOME
  save_my_session();
#endif

  sg_project_close();

  if(last_project_filename) g_free(last_project_filename);
  if(last_project_path) g_free(last_project_path);

  sg_project_autosave_set(0);
  gtk_object_unref(GTK_OBJECT(app)); 
  sg_quit();
  gtk_main_quit();
}

static void
sg_project_rename_worksheet(SGapplication *app, SGworksheet *worksheet)
{
  gchar *name = g_strdup(worksheet->name);
  gboolean free_name = FALSE;
  gboolean accept = FALSE;
  GList *list = NULL;
  guint n = 0;

  while(!name || strlen(name) == 0){
      if(free_name) g_free(name);
      free_name = FALSE;
      name = sg_text_dialog( _("The worksheet must have a name.\nPlease enter a name:"), _("New worksheet"), 1);
      free_name = TRUE;
  }

  list = app->worksheets->list;
  while(!accept){
    gboolean new_name = FALSE;
    while(list){
     SGworksheet *data;
     SGlistChild *child = (SGlistChild *)list->data;
                                                                                
     data = SG_WORKSHEET(child->object);
                                                                                
     if(data != worksheet && strcmp(data->name, name) == 0){
        if(free_name) g_free(name);
        name = NULL;
        free_name = FALSE;
        while(!name || strlen(name) == 0){
           name = sg_text_dialog( _("Another worksheet has the same name.\nPlease enter a new name for this worksheet:"), data->name, 1);
        }
        free_name = TRUE;
        new_name = TRUE;
        break;
     }
     new_name = FALSE;
     list = list->next;
    }
    if(!new_name) accept = TRUE;
  }
                                                                                
  for(n = 0; n < strlen(name); n++){
    if(name[n] < 'a' || name[n] > 'z')
     if(name[n] < 'A' || name[n] > 'Z')
      if(name[n] < '0' || name[n] > '9')
        if(name[n] != '.' && name[n] != '_' && name[n] != '-' && name[n] != '+'
&& name[n] != '='){
          name[n] = '_';
/*
          sg_message_dialog("The name contains invalid characters", 0);
          return FALSE;
*/
        }
  }

  g_free(worksheet->name);
  worksheet->name = g_strdup(name);

  if(gui_mode == SG_GUI_ICONLIST){
    GtkIconListItem *icon = (GtkIconListItem *)gtk_object_get_data(GTK_OBJECT(worksheet), "icon");
    
    if(icon){
      GtkWindow *window = NULL;
      gchar window_title[500];
      window = GTK_WINDOW(icon->link);
      g_snprintf(window_title, 500, "SciGraphica: %s", worksheet->name);
      gtk_window_set_title(GTK_WINDOW(window), window_title);
      
      gtk_entry_set_text(GTK_ENTRY(icon->entry), name);

      if (icon->entry_label){
        g_free(icon->entry_label);
        icon->entry_label = NULL;
      }
 
      icon->entry_label = g_strdup(name);

      if (icon->entry_label){
        g_free(icon->label);
        icon->label = NULL;
      }

      icon->label = g_strdup(name);
    }
  }
  g_free(name);
}

static void
sg_project_rename_matrix(SGapplication *app, SGworksheet *worksheet)
{
  gchar *name = g_strdup(worksheet->name);
  gboolean free_name = FALSE;
  gboolean accept = FALSE;
  GList *list = NULL;
  guint n = 0;

  while(!name || strlen(name) == 0){
      if(free_name) g_free(name);
      free_name = FALSE;
      name = sg_text_dialog( _("The matrix must have a name.\nPlease enter a name:"), _("New matrix"), 1);
      free_name = TRUE;
  }

  list = app->worksheets->list;
  while(!accept){
    gboolean new_name = FALSE;
    while(list){
     SGworksheet *data;
     SGlistChild *child = (SGlistChild *)list->data;
                                                                                
     data = SG_WORKSHEET(child->object);
                                                                                
     if(data != worksheet && strcmp(data->name, name) == 0){
        if(free_name) g_free(name);
        name = NULL;
        free_name = FALSE;
        while(!name || strlen(name) == 0){
           name = sg_text_dialog( _("Another worksheet has the same name.\nPlease enter a new name for this worksheet:"), data->name, 1);
        }
        free_name = TRUE;
        new_name = TRUE;
        break;
     }
     new_name = FALSE;
     list = list->next;
    }
    if(!new_name) accept = TRUE;
  }
                                                                                
  for(n = 0; n < strlen(name); n++){
    if(name[n] < 'a' || name[n] > 'z')
     if(name[n] < 'A' || name[n] > 'Z')
      if(name[n] < '0' || name[n] > '9')
        if(name[n] != '.' && name[n] != '_' && name[n] != '-' && name[n] != '+'
&& name[n] != '='){
          name[n] = '_';
/*
          sg_message_dialog("The name contains invalid characters", 0);
          return FALSE;
*/
        }
  }

  g_free(worksheet->name);
  worksheet->name = g_strdup(name);

  if(gui_mode == SG_GUI_ICONLIST){
    GtkIconListItem *icon = (GtkIconListItem *)gtk_object_get_data(GTK_OBJECT(worksheet), "icon");

    if(icon){
      GtkWindow *window = NULL;
      gchar window_title[500];
      window = GTK_WINDOW(icon->link);
      g_snprintf(window_title, 500, "SciGraphica: %s", worksheet->name);
      gtk_window_set_title(GTK_WINDOW(window), window_title);
      gtk_entry_set_text(GTK_ENTRY(icon->entry), name);

      if (icon->entry_label){
        g_free(icon->entry_label);
        icon->entry_label = NULL;
      }
   
      icon->entry_label = g_strdup(name);
  
      if (icon->entry_label){
        g_free(icon->label);
        icon->label = NULL;
      }
    }

    icon->label = g_strdup(name);
  }
  g_free(name);
}

static void
sg_project_rename_plot(SGapplication *app, SGplot *plot)
{
  gchar *name = g_strdup(plot->name);
  gboolean free_name = FALSE;
  gboolean accept = FALSE;
  GList *list = NULL;
  guint n = 0;

  while(!name || strlen(name) == 0){
      if(free_name) g_free(name);
      free_name = FALSE;
      name = sg_text_dialog( _("The plot must have a name.\nPlease enter a name:"), _("New plot"), 1);
      free_name = TRUE;
  }

  list = app->plots->list;
  while(!accept){
    gboolean new_name = FALSE;
    while(list){
     SGplot *data;
     SGlistChild *child = (SGlistChild *)list->data;
                                                                                
     data = SG_PLOT(child->object);
                                                                                
     if(data != plot && strcmp(data->name, name) == 0){
        if(free_name) g_free(name);
        name = NULL;
        free_name = FALSE;
        while(!name || strlen(name) == 0){
           name = sg_text_dialog( _("Another plot has the same name.\nPlease enter a new name for this plot:"), data->name, 1);
        }
        free_name = TRUE;
        new_name = TRUE;
        break;
     }
     new_name = FALSE;
     list = list->next;
    }
    if(!new_name) accept = TRUE;
  }
                                                                                
  for(n = 0; n < strlen(name); n++){
    if(name[n] < 'a' || name[n] > 'z')
     if(name[n] < 'A' || name[n] > 'Z')
      if(name[n] < '0' || name[n] > '9')
        if(name[n] != '.' && name[n] != '_' && name[n] != '-' && name[n] != '+'
&& name[n] != '='){
          name[n] = '_';
/*
          sg_message_dialog("The name contains invalid characters", 0);
          return FALSE;
*/
        }
  }

  g_free(plot->name);
  plot->name = g_strdup(name);

  if(gui_mode == SG_GUI_ICONLIST){
    GtkIconListItem *icon = (GtkIconListItem *)gtk_object_get_data(GTK_OBJECT(plot), "icon");

    if(icon){
      GtkWindow *window = NULL;
      gchar window_title[500];
      window = GTK_WINDOW(icon->link);
      g_snprintf(window_title, 500, "SciGraphica: %s", plot->name);
      gtk_window_set_title(GTK_WINDOW(window), window_title);

      gtk_entry_set_text(GTK_ENTRY(icon->entry), name);

      if (icon->label){
        g_free(icon->label);
        icon->label = NULL;
      }
	 
      icon->label = g_strdup(name);

      if(icon->entry_label){
        g_free(icon->entry_label);
        icon->entry_label = NULL;
      }

      icon->entry_label = g_strdup(name);
    }
  }
  g_free(name);
}

static gboolean
rename_worksheet(GtkIconList *iconlist, GtkIconListItem *item)
{
  const gchar *name;
  SGworksheetWindow *window;

  if (gui_mode != SG_GUI_ICONLIST) return TRUE;

  window = SG_WORKSHEET_WINDOW(gtk_icon_list_get_link(item));
  name = gtk_entry_get_text(GTK_ENTRY(item->entry));

  if(strcmp(name,window->worksheet->name) == 0) return TRUE;
  sg_worksheet_rename(window->worksheet, name);
  sg_application_rename_worksheet(app, window->worksheet);
  return TRUE;
}

static gboolean
rename_plot(GtkIconList *iconlist, GtkIconListItem *item)
{
  const gchar *name;
  SGplotWindow *plot_window;
  SGplot *plot;

  if (gui_mode != SG_GUI_ICONLIST) return TRUE;

  plot_window = SG_PLOT_WINDOW(gtk_icon_list_get_link(item));
  plot = plot_window->plot;
  name = gtk_entry_get_text(GTK_ENTRY(item->entry));

  if(strcmp(name,plot->name) == 0) return TRUE;
  sg_plot_rename(plot, name);
  sg_application_rename_plot(app, plot);
  return TRUE;
}

static gint
delete_event(GtkWidget *widget)
{
  if(GTK_IS_SG_PLOT_WINDOW(widget)){
    SGplotWindow *plot_window = SG_PLOT_WINDOW(widget);

    gdk_window_get_position(widget->window,
                            &plot_window->x,
                            &plot_window->y);
    gdk_window_get_size(widget->window,
                        &plot_window->width,
                        &plot_window->height);
    if(gui_mode == SG_GUI_PLOT){
      project_quit_warning(NULL);
      sg_project_quit();
      return TRUE;
    }

  } else if(GTK_IS_SG_WORKSHEET_WINDOW(widget)){
    SGworksheetWindow *window = SG_WORKSHEET_WINDOW(widget);

    gdk_window_get_position(widget->window,
                            &window->x,
                            &window->y);
    gdk_window_get_size(widget->window,
                        &window->width,
                        &window->height);

    if(gui_mode == SG_GUI_WORKSHEET){
      project_quit_warning(NULL);
      sg_project_quit();
      return TRUE;
    }
  }

  gtk_widget_hide(widget);

  return TRUE;
}

static void
sg_project_activate_plot(SGapplication *app, SGplot *plot)
{
  if(plot == app->active_plot) return;
  if(gui_mode == SG_GUI_ICONLIST){
    gchar msg[80];
    g_snprintf(msg,80, _("Active plot is %s"), plot->name);
    sg_project_statusbar_set(msg, SG_STATUSBAR_TEXTONLY);
  }
}

static void
sg_project_activate_worksheet(SGapplication *app, SGworksheet *worksheet)
{
  if(worksheet == app->active_worksheet) return;
  if(gui_mode == SG_GUI_ICONLIST){
    gchar msg[80];
    g_snprintf(msg,80, _("Active worksheet is %s"), worksheet->name);
    sg_project_statusbar_set(msg, SG_STATUSBAR_TEXTONLY);
  }
}

void
sg_project_connect_signals(SGapplication *app)
{
  gtk_signal_connect(GTK_OBJECT(app), "changed",
		     GTK_SIGNAL_FUNC(sg_project_changed), NULL);
  gtk_signal_connect(GTK_OBJECT(app), "add_worksheet",
		     GTK_SIGNAL_FUNC(sg_project_new_worksheet), NULL);
  gtk_signal_connect(GTK_OBJECT(app), "add_matrix",
		     GTK_SIGNAL_FUNC(sg_project_new_matrix), NULL);
  gtk_signal_connect(GTK_OBJECT(app), "add_plot",
		     GTK_SIGNAL_FUNC(sg_project_new_plot), NULL);
  gtk_signal_connect(GTK_OBJECT(app), "remove_worksheet",
		     GTK_SIGNAL_FUNC(sg_project_remove_worksheet), NULL);
  gtk_signal_connect(GTK_OBJECT(app), "remove_matrix",
		     GTK_SIGNAL_FUNC(sg_project_remove_matrix), NULL);
  gtk_signal_connect(GTK_OBJECT(app), "remove_plot",
		     GTK_SIGNAL_FUNC(sg_project_remove_plot), NULL);
  gtk_signal_connect(GTK_OBJECT(app), "rename_worksheet",
		     GTK_SIGNAL_FUNC(sg_project_rename_worksheet), NULL);
  gtk_signal_connect(GTK_OBJECT(app), "rename_matrix",
		     GTK_SIGNAL_FUNC(sg_project_rename_matrix), NULL);
  gtk_signal_connect(GTK_OBJECT(app), "rename_plot",
		     GTK_SIGNAL_FUNC(sg_project_rename_plot), NULL);
  gtk_signal_connect(GTK_OBJECT(app), "activate_worksheet",
		     GTK_SIGNAL_FUNC(sg_project_activate_worksheet), NULL);
  gtk_signal_connect(GTK_OBJECT(app), "activate_plot",
		     GTK_SIGNAL_FUNC(sg_project_activate_plot), NULL);

}
