/*
 * nasd_edrfs_wipedir.c
 *
 * Read a NASD object, parse it as a directory
 *
 * Authors: Jim Zelenka, Nat Lanza
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>
#include <nasd/nasd_getopt.h>
#include <nasd/nasd_pdrive.h>
#include <nasd/nasd_pdrive_client.h>
#include <nasd/nasd_pdrive_client_kpdev.h>
#include <nasd/nasd_security.h>
#include <nasd/nasd_edrfs_types.h>
#include <nasd/nasd_edrfs_dir.h>
#include <nasd/nasd_edrfs_internal.h>
#include <nasd/nasd_mem.h>

char *progname;

int                        binding_type;
int                        binding_args_len;
void                      *binding_args;
char                      *binding_port = NASD_PDRIVE_PORT;
nasd_drive_param_kpdev_t   kpdev_args;
int                        nondefault_binding = 0;

nasd_sec_keyring_t         keys;
nasd_error_string_t error_text;

char *
typestr(
  nasd_edrfs_dirdata_t  *dd)
{
  static char str[128];

  switch(NASD_EDRFS_DIRD_TYPEOF(dd)) {
    case NASD_EDRFS_DIRD_TYPE_BOGUS: return("?");
    case NASD_EDRFS_DIRD_TYPE_EXNAM: return("n");
    case NASD_EDRFS_DIRD_TYPE_DATA:  return("r");
    case NASD_EDRFS_DIRD_TYPE_LINK:  return("l");
    case NASD_EDRFS_DIRD_TYPE_DIR:   return("d");
    case NASD_EDRFS_DIRD_TYPE_CHR:   return("c");
    case NASD_EDRFS_DIRD_TYPE_BLK:   return("b");
    case NASD_EDRFS_DIRD_TYPE_FIFO:  return("f");
    case NASD_EDRFS_DIRD_TYPE_SOCK:  return("s");
    default:
      sprintf(str, "0x%02" NASD_64x_FMT , NASD_EDRFS_DIRD_TYPEOF(dd)&0xff);
  }
  return(str);
}

void
usage()
{
  int i;
  fprintf(stderr, "USAGE: %s [options] servername partnum nasdid master_password\n", progname);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "  -t don't print NASD IDs\n");
  fprintf(stderr, "  -k use kernel device\n");
  fprintf(stderr, "  -s security_level\n" );
  for(i = 0; i <= NASD_MAX_SECURITY_LEVEL; i++) {
    fprintf(stderr, "     %d %s\n", i, nasd_sec_level_string(i));
  }
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
  fprintf(stderr, "  -T use DCE-TCP\n");
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */
  fflush(stderr);
  exit(1);
}

int
main(
  int     argc,
  char  **argv)
{
  int p, i, j, k, used, free, npages, mapbyte, mapent, len, seg_len, slot;
  char *master_password, *server_name, name[NASD_EDRFS_MAX_NAME_LEN], c;
  nasd_p_smpl_op_dr_args_t wr_args;
  nasd_p_fastwrite_dr_res_t wr_res;
  nasd_edrfs_attributes_t   *edrfsattr;
  nasd_p_getattr_dr_args_t   args;
  nasd_p_getattr_dr_res_t    res;

  nasd_edrfs_dirpage_t       page;
  char                       block[NASD_EDRFS_DIRPAGE_SIZE];

  nasd_identifier_t          parentnid;

  nasd_rpc_status_t          status;
  nasd_attribute_t          *attr;
  nasd_edrfs_dirdata_t      *dd;
  nasd_drive_handle_t        h;
  nasd_cookie_t              cookie;
  nasd_security_param_t      sp;
  nasd_timespec_t            tm;
  nasd_status_t              rc;
  int                        sec_level=0;
  nasd_uint16                protection;
  int                        printnasd = 1;

  progname = argv[0];

  binding_type = NASD_BIND_DEFAULT;
  binding_args = NULL;
  binding_args_len = 0;

  while (nasd_getopt(argc, argv, "kTt", &c)) {
    switch(c) {
    case 't':
      printnasd = 0;
      break;
    case 'k':
      if (nondefault_binding)
        usage();
      nondefault_binding = 1;
      binding_type = NASD_BIND_KPDEV_DEFAULT;
      binding_args = &kpdev_args;
      binding_args_len = sizeof(kpdev_args);
      strcpy(kpdev_args.devname, "/dev/nasdkp0");
      break;
    case 's':
      if (sscanf(nasd_optarg, "%d", &sec_level) != 1) {
        usage();
      }
      break;
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
    case 'T':
      if (nondefault_binding)
        usage();
      nondefault_binding = 1;
      binding_type = NASD_BIND_DCE_DIRECT_TCP;
      break;
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */
    default:
      fprintf(stderr, "Unknown option '%c'\n", c);
      usage();
    }
  }

  if (nasd_optind >= argc)
    usage();
  server_name = argv[nasd_optind];
  nasd_optind++;

  bzero((char *)&args, sizeof(args));

  if (nasd_optind >= argc)
    usage();
  if (sscanf(argv[nasd_optind], "%d", &p) != 1)
    usage();
  nasd_optind++;
  args.in_partnum = p;

  if (nasd_optind >= argc)
    usage();
  rc = nasd_str_to_nasd_id(argv[nasd_optind], &args.in_identifier);
  nasd_optind++;
  if (rc)
    usage();

  if (nasd_optind >= argc)
    usage();
  master_password=argv[nasd_optind];
  nasd_optind++;

  if (nasd_optind >= argc)
    usage();
  rc = nasd_str_to_nasd_id(argv[nasd_optind], &parentnid);
  nasd_optind++;
  if (rc)
    usage();

  if (nasd_optind < argc)
    usage();

  rc = nasd_sec_seclevel_to_protection(sec_level, &protection);
  if(rc) {
    fprintf(stderr, "ERROR: invalid security level %d\n", sec_level);
    usage();
  }

  rc = nasd_cl_p_init();
  if (rc) {
    fprintf(stderr, "ERROR: cannot init client library, rc=0x%x (%s)\n",
      rc, nasd_error_string(rc));
    exit(1);
  }

  rc = nasd_bind_to_drive(server_name, binding_port,
    binding_type, binding_args, binding_args_len, &h);
  if (rc) {
    fprintf(stderr, "ERROR: cannot bind to server %s\n", server_name);
    fflush(stderr);
    exit(1);
  }

  /* Fill in capability and security param */
  nasd_sec_password_to_keys(master_password, args.in_partnum, &keys);

  nasd_drive_handle_get_time(h, &tm);
  tm.ts_sec += (60*60);
  nasd_sec_build_capability(args.in_partnum,
    args.in_identifier,
    NASD_ACCESS_RIGHTS_GETATTR | NASD_ACCESS_RIGHTS_READ,
    0, tm.ts_sec, protection, NASD_BLACK_CAPABILITY,
    0, 0, 7, keys.black_key, &cookie);

  sp.type = NASD_BLACK_CAPABILITY; 
  sp.partnum = args.in_partnum;
  sp.actual_protection = protection;

  /*
   * Read NASD attributes
   */

  nasd_cl_p_getattr_dr(h, cookie.key, &sp, &cookie.capability,
                       &args, &res, &status);

  if (res.nasd_status || status) {
    fprintf(stderr, "ERROR: nasd_status=0x%x (%s) status=0x%x (%s)\n",
      res.nasd_status, nasd_error_string(res.nasd_status),
      status, nasd_cl_error_string(h, status, error_text));
    exit(1);
  }

  attr = &res.out_attribute;
  edrfsattr = (nasd_edrfs_attributes_t *)attr->fs_specific;
  if (edrfsattr->type != NASD_EDRFS_TYPE_DIR) {
    fprintf(stderr, "Unexpected type 0x%02x\n", edrfsattr->type);
    exit(1);
  }

  printf("type 0x%02x mode 0x%04x nlink %u uid %u gid %u\n",
    edrfsattr->type, edrfsattr->mode, edrfsattr->nlink,
    edrfsattr->uid, edrfsattr->gid);

  nasd_edrfs_dir_init_page(&page);
  nasd_edrfs_dir_add_entry(&page, "..", parentnid,
                           NASD_EDRFS_DIRD_TYPE_DIR);
  nasd_edrfs_dir_add_entry(&page,  ".", args.in_identifier,
                           NASD_EDRFS_DIRD_TYPE_DIR);
  nasd_edrfs_dir_marshall_page(&page, (void *) block);

  /*
   * Write the contents
   */

  wr_args.in_partnum = args.in_partnum;
  wr_args.in_identifier = args.in_identifier;
  wr_args.in_offset = 0;
  wr_args.in_len = NASD_EDRFS_DIRPAGE_SIZE;

  sp.type = cookie.capability.type;
  sp.partnum = args.in_partnum;
  sp.actual_protection = cookie.capability.min_protection;
  nasd_cl_p_write_simple_dr(h, cookie.key, &sp, &cookie.capability,
                            &wr_args, block, &wr_res, &status);


  rc = nasd_unbind_drive(&h);
  if (rc) {
    fprintf(stderr, "ERROR: got 0x%x (%s) unbinding drive\n",
      rc, nasd_error_string(rc));
    exit(1);
  }
  nasd_cl_p_shutdown();

  exit(0);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
