/*
 * nasd_edrfs_dumpdir.c
 *
 * Read a NASD object, parse it as a directory
 *
 * Authors: Jim Zelenka, Nat Lanza
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>
#include <nasd/nasd_getopt.h>
#include <nasd/nasd_pdrive.h>
#include <nasd/nasd_pdrive_client.h>
#include <nasd/nasd_pdrive_client_kpdev.h>
#include <nasd/nasd_security.h>
#include <nasd/nasd_edrfs_types.h>
#include <nasd/nasd_edrfs_dir.h>
#include <nasd/nasd_edrfs_internal.h>
#include <nasd/nasd_mem.h>

char *progname;

int                        binding_type;
int                        binding_args_len;
void                      *binding_args;
char                      *binding_port = NASD_PDRIVE_PORT;
nasd_drive_param_kpdev_t   kpdev_args;
int                        nondefault_binding = 0;

nasd_sec_keyring_t         keys;
nasd_error_string_t error_text;

char *
typestr(
  nasd_edrfs_dirdata_t  *dd)
{
  static char str[128];

  switch(NASD_EDRFS_DIRD_TYPEOF(dd)) {
    case NASD_EDRFS_DIRD_TYPE_BOGUS: return("?");
    case NASD_EDRFS_DIRD_TYPE_EXNAM: return("n");
    case NASD_EDRFS_DIRD_TYPE_DATA:  return("r");
    case NASD_EDRFS_DIRD_TYPE_LINK:  return("l");
    case NASD_EDRFS_DIRD_TYPE_DIR:   return("d");
    case NASD_EDRFS_DIRD_TYPE_CHR:   return("c");
    case NASD_EDRFS_DIRD_TYPE_BLK:   return("b");
    case NASD_EDRFS_DIRD_TYPE_FIFO:  return("f");
    case NASD_EDRFS_DIRD_TYPE_SOCK:  return("s");
    default:
      sprintf(str, "0x%02" NASD_64x_FMT , NASD_EDRFS_DIRD_TYPEOF(dd)&0xff);
  }
  return(str);
}

void
usage()
{
  int i;
  fprintf(stderr, "USAGE: %s [options] servername partnum nasdid master_password\n", progname);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "  -t don't print NASD IDs\n");
  fprintf(stderr, "  -k use kernel device\n");
  fprintf(stderr, "  -s security_level\n" );
  for(i = 0; i <= NASD_MAX_SECURITY_LEVEL; i++) {
    fprintf(stderr, "     %d %s\n", i, nasd_sec_level_string(i));
  }
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
  fprintf(stderr, "  -T use DCE-TCP\n");
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */
  fflush(stderr);
  exit(1);
}

int
main(
  int     argc,
  char  **argv)
{
  int p, i, j, k, used, free, npages, mapbyte, mapent, len, seg_len, slot;
  char *master_password, *server_name, name[NASD_EDRFS_MAX_NAME_LEN], c;
  nasd_p_smpl_op_dr_args_t   rd_args;
  nasd_p_fastread_dr_res_t   rd_res;
  nasd_edrfs_attributes_t   *edrfsattr;
  nasd_p_getattr_dr_args_t   args;
  nasd_p_getattr_dr_res_t    res;

  nasd_edrfs_dirpage_t      *pages;
  void                      *blocks;
  char                      *cur_block;

  nasd_rpc_status_t          status;
  nasd_attribute_t          *attr;
  nasd_edrfs_dirdata_t      *dd;
  nasd_drive_handle_t        h;
  nasd_cookie_t              cookie;
  nasd_security_param_t      sp;
  nasd_timespec_t            tm;
  nasd_status_t              rc;
  int                        sec_level=0;
  nasd_uint16                protection;
  int                        printnasd = 1;

  progname = argv[0];

  binding_type = NASD_BIND_DEFAULT;
  binding_args = NULL;
  binding_args_len = 0;

  while (nasd_getopt(argc, argv, "kTt", &c)) {
    switch(c) {
    case 't':
      printnasd = 0;
      break;
    case 'k':
      if (nondefault_binding)
        usage();
      nondefault_binding = 1;
      binding_type = NASD_BIND_KPDEV_DEFAULT;
      binding_args = &kpdev_args;
      binding_args_len = sizeof(kpdev_args);
      strcpy(kpdev_args.devname, "/dev/nasdkp0");
      break;
    case 's':
      if (sscanf(nasd_optarg, "%d", &sec_level) != 1) {
        usage();
      }
      break;
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
    case 'T':
      if (nondefault_binding)
        usage();
      nondefault_binding = 1;
      binding_type = NASD_BIND_DCE_DIRECT_TCP;
      break;
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */
    default:
      fprintf(stderr, "Unknown option '%c'\n", c);
      usage();
    }
  }

  if (nasd_optind >= argc)
    usage();
  server_name = argv[nasd_optind];
  nasd_optind++;

  bzero((char *)&args, sizeof(args));

  if (nasd_optind >= argc)
    usage();
  if (sscanf(argv[nasd_optind], "%d", &p) != 1)
    usage();
  nasd_optind++;
  args.in_partnum = p;

  if (nasd_optind >= argc)
    usage();
  rc = nasd_str_to_nasd_id(argv[nasd_optind], &args.in_identifier);
  nasd_optind++;
  if (rc)
    usage();

  if (nasd_optind >= argc)
    usage();
  master_password=argv[nasd_optind];
  nasd_optind++;

  if (nasd_optind < argc)
    usage();

  rc = nasd_sec_seclevel_to_protection(sec_level, &protection);
  if(rc) {
    fprintf(stderr, "ERROR: invalid security level %d\n", sec_level);
    usage();
  }

  rc = nasd_cl_p_init();
  if (rc) {
    fprintf(stderr, "ERROR: cannot init client library, rc=0x%x (%s)\n",
      rc, nasd_error_string(rc));
    exit(1);
  }

  rc = nasd_bind_to_drive(server_name, binding_port,
    binding_type, binding_args, binding_args_len, &h);
  if (rc) {
    fprintf(stderr, "ERROR: cannot bind to server %s\n", server_name);
    fflush(stderr);
    exit(1);
  }

  /* Fill in capability and security param */
  nasd_sec_password_to_keys(master_password, args.in_partnum, &keys);

  nasd_drive_handle_get_time(h, &tm);
  tm.ts_sec += (60*60);
  nasd_sec_build_capability(args.in_partnum,
    args.in_identifier,
    NASD_ACCESS_RIGHTS_GETATTR | NASD_ACCESS_RIGHTS_READ,
    0, tm.ts_sec, protection, NASD_BLACK_CAPABILITY,
    0, 0, 7, keys.black_key, &cookie);

  sp.type = NASD_BLACK_CAPABILITY; 
  sp.partnum = args.in_partnum;
  sp.actual_protection = protection;

  /*
   * Read NASD attributes
   */

  nasd_cl_p_getattr_dr(h, cookie.key, &sp, &cookie.capability,
                       &args, &res, &status);

  if (res.nasd_status || status) {
    fprintf(stderr, "ERROR: nasd_status=0x%x (%s) status=0x%x (%s)\n",
      res.nasd_status, nasd_error_string(res.nasd_status),
      status, nasd_cl_error_string(h, status, error_text));
    exit(1);
  }

  attr = &res.out_attribute;
  edrfsattr = (nasd_edrfs_attributes_t *)attr->fs_specific;
  if (edrfsattr->type != NASD_EDRFS_TYPE_DIR) {
    fprintf(stderr, "Unexpected type 0x%02x\n", edrfsattr->type);
    exit(1);
  }

  printf("type 0x%02x mode 0x%04x nlink %u uid %u gid %u\n",
    edrfsattr->type, edrfsattr->mode, edrfsattr->nlink,
    edrfsattr->uid, edrfsattr->gid);

  /*
   * Read in the contents
   */

  if (attr->object_len % NASD_EDRFS_DIRPAGE_SIZE) {
    fprintf(stderr, "WARNING: object len %" NASD_64u_FMT
      " not a multiple of %d\n", attr->object_len, NASD_EDRFS_DIRPAGE_SIZE);
  }
  npages = attr->object_len / NASD_EDRFS_DIRPAGE_SIZE;
  if (npages < 1) {
    fprintf(stderr, "ERROR: %d pages\n", npages);
    exit(1);
  }

  NASD_Malloc(blocks, attr->object_len, (void *));
  NASD_Malloc(pages, npages * sizeof(nasd_edrfs_dirpage_t), (nasd_edrfs_dirpage_t *));
  if ((blocks == NULL) || (pages == NULL)) {
    fprintf(stderr, "ERROR: could not allocate %" NASD_64u_FMT
      " bytes for directory\n", attr->object_len);
    exit(1);
  }

  rd_args.in_partnum = args.in_partnum;
  rd_args.in_identifier = args.in_identifier;

  rd_args.in_offset = 0;
  while(rd_args.in_offset < attr->object_len) {
    rd_args.in_len = (nasd_len_t)attr->object_len
      - (nasd_len_t)rd_args.in_offset;

    nasd_cl_p_read_simple_dr(h, cookie.key, &sp, &cookie.capability,
                             &rd_args, blocks, &rd_res, &status);

    if (status || rd_res.nasd_status) {
      fprintf(stderr, "ERROR: nasd_status=0x%x (%s) status=0x%x (%s)\n",
        rd_res.nasd_status, nasd_error_string(rd_res.nasd_status),
        status, nasd_cl_error_string(h, status, error_text));
      exit(1);
    }

    rd_args.in_offset += (nasd_offset_t)rd_res.out_datalen;
  }

  /*
   * Dump contents
   */
  printf("Usage report:\n");
  used = 0;
  cur_block = blocks;
  for(i = 0; i < npages; i++) {

    nasd_edrfs_dir_parse_page(cur_block, &pages[i]);

    for(j = 0; j < sizeof(pages[i].header.usemap); j++) {
      for(k = 0; k < 8; k++) {
        if (pages[i].header.usemap[j]&(1<<k)) {used++;}
      }
    }
    
    free = pages[i].header.freecnt;

    if ((used + free) == 127) {
      printf("   page %2d  used %3d  free %3d\n",
             i, used, pages[i].header.freecnt);
    }
    else {
      printf(" ! page %2d  used %3d  free %3d  (sum %d)\n",
             i, used, free, used+free);
    }

    cur_block += NASD_EDRFS_DIRPAGE_SIZE;
  }

  printf("Names:\n");
  for (i = 0; i < npages; i++) {
    for (j = 0; j < NASD_EDRFS_DIRPAGE_NUM_SLOTS; j++) {
      if (!NASD_EDRFS_DIRMAP_TEST_SLOT(pages[i].header.usemap, j)) { continue; }
      dd = &pages[i].data_slots[j];

      if (dd->name_len >= NASD_EDRFS_MAX_NAME_LEN) {
        fprintf(stderr, "WARNING: slot %d name len %d\n", j, dd->name_len);
      }

      nasd_edrfs_dir_extract_name(&pages[i], j, name);
      len = NASD_EDRFS_DIRD_EXLEN(dd);
      seg_len = NASD_EDRFS_DIRD_SEGLEN(dd);

      if (dd->name[seg_len] != '\0') {
        fprintf(stderr, "WARNING: slot %d name not null-terminated\n", j);
      }

      if (printnasd) {
        printf(" page %2d  slot %3d  len %2d seg %2d type %s extype 0x%02x  0x%" NASD_ID_FMT " %s\n",
               i, j, len, NASD_EDRFS_DIRD_SEGLEN(dd), typestr(dd), 
               NASD_EDRFS_DIRD_EXTYPEOF(dd), dd->nasdid, dd->name);
      } else {
        printf(" page %2d  slot %3d  len %2d seg %2d type %s extype 0x%02x  %s\n",
               i, j, len, NASD_EDRFS_DIRD_SEGLEN(dd), typestr(dd),
               NASD_EDRFS_DIRD_EXTYPEOF(dd), dd->name);
      }
    }
  }

  /*
   * Clean up and exit
   */

  rc = nasd_unbind_drive(&h);
  if (rc) {
    fprintf(stderr, "ERROR: got 0x%x (%s) unbinding drive\n",
      rc, nasd_error_string(rc));
    exit(1);
  }
  nasd_cl_p_shutdown();

  exit(0);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
