
#include <stdio.h>
#include <time.h>
#include <stdlib.h>

#define StartingMessage \
    printf("Polclock-1.11, October 1997 by Masaki Kawase (g93088@isc.chubu.ac.jp)\n") ; \
    printf("\n") ;

#define MAX_OBJECTS           61
#define MAX_CHARS             11
#define MAX_CHILDREN           8
#define MAX_TOPS_OF_OBJECT    40
#define MAX_TOPS_OF_POLYGON    6
#define MAX_POLYGONS          20

#define MODE_MIN               1
#define MODE_SEC               0

#define SIZE_LITTLE            0
#define SIZE_SMALL             1
#define SIZE_MIDDLE            2
#define SIZE_LARGE             3
#define SIZE_HUGE              4

#define CLOCK_REVOLVE         72
#define NUM_REVOLVE           24
#define SEG_COLORING           8
#define SEG_REVOLVE           36
#define COLON_COLORING         2
#define COLON_REVOLVE         36
#define MOTION_COUNT          40
#define TRANS_RATE             0.86
#define SEG_DEPTH              4.0

#define BG_COLOR                C_SkyBlue
#define FG_COLOR                CD_Gold
#define RG_COLOR                CD_Iron
#define MONO_COLOR              C_White

#define RESIZE_MODE             RESIZE_SHORT
#define CLOCK_SIZE              SIZE_MIDDLE

#define MSLEEP_MOVE           10
#define MSLEEP_EVENT         100

#define CL_TYPE                 RGB

#define NM_RED                 5
#define NM_GREEN               6
#define NM_BLUE                4
#define NM_COLORS             32

#define NM_LIGHT_BALLS        20
#define LIGHT_BALL_SIZE        2.0

#define WIN_NULL          -99999

/*
#define RESIZE_MODE             RESIZE_DEFAULT
#define RESIZE_MODE             RESIZE_WIDTH
#define RESIZE_MODE             RESIZE_HEIGHT
#define RESIZE_MODE             RESIZE_SHORT
#define RESIZE_MODE             RESIZE_LONG
*/

/*
#define NO_SELECT
*/

#include "polsys.c"

typedef struct
{
    int         clock_mode, rs_mode, size, bg, fg, rg, mono,
                cl_type, nm_red, nm_green, nm_blue, nm_colors, nm_balls ;
    double      seg_depth, trans_rate, ball_size ;
    Position    wp ;
    W_Size      ws ;
    Color       bg_color, mono_color ;
    Color_Data  fg_color, rg_color ;
    char        *disp, *geom, *title ;
    unsigned long   msleep_move, msleep_event ;
} Option ;

typedef struct
{
    int     num1, num10 ;
} Num ;

typedef struct
{
    int     n[6] ;
} NumData ;

typedef struct
{
    int     n[6][7] ;
} SegData ;

typedef struct
{
    int     obj_no, revolve_flag, coloring_flag ;
    double  color_rate ;
} Obj ;

typedef struct
{
    int     none_class, seg_u_class, seg_v_class, dot_class, num_class,
            colon_class, clock_class, order_class, eye_obj, star_class ;
    int     motion_event1, motion_event3, button_event, event_loop_flag,
            old_clock, move_flag, init_flag ;
    Obj     order_obj, clock_obj, num_obj[6], colon_obj[2][2], seg_obj[6][7] ;
    double  distance ;
    NumData num_data, pre_num ;
    SegData seg_data, pre_seg ;
    double  mouse_x, mouse_y, m1_x, m1_y, m3_x, m3_y ;
    TP_Data tp[3] ;
    time_t  t ;
    struct  tm  *t_area ;
    Num     number[2] ;
    Option  opt ;
} PClock ;

void u_sleep(unsigned long usecs) ;
void CommandOption(char argc, char *argv[]) ;
void PolInitialize(char *argv[]) ;
void OptionToColor_Data(Color_Data *cd, int i, char *argv[]) ;
void Usage(void) ;
void Initialize(void) ;
void CreateTP_Data(void) ;
void CreatePolData(TP_Data *tp) ;
void CreateClasses(void) ;
void CreateInstances(void) ;
void InitializeData(void) ;
Num  GetNum1_10(int nm) ;
void ClockProcedure(void) ;
void ControlObjects(void) ;
void ControlOrder(void) ;
void ControlDistance(void) ;
void ControlClock(void) ;
void ControlNum(void) ;
void ControlSeg(void) ;
void ControlColon(void) ;
void ControlObj(Obj *objs, int nm_revolve, int nm_coloring) ;

PClock  pc ;
int     seg_data[11][7] = {{1,1,1,0,1,1,1}, {0,0,1,0,0,1,0},
			   {1,0,1,1,1,0,1}, {1,0,1,1,0,1,1},
			   {0,1,1,1,0,1,0}, {1,1,0,1,0,1,1},
			   {1,1,0,1,1,1,1}, {1,1,1,0,0,1,0},
			   {1,1,1,1,1,1,1}, {1,1,1,1,0,1,1},
			   {0,0,0,0,0,0,0}} ;

W_Size  w_size[2][5] = {{{96,32},{192,64},{384,128},{600,200},{960,320}},
			{{90,36},{180,72},{360,144},{540,216},{900,360}}} ;

char *wSize[2][5] = {{"96x32", "192x64", "384x128", "600x200", "960x320"},
		     {"90x36", "180x72", "360x144", "540x216", "900x320"}} ;

S_Size  s_size[2] = {{1.5, 0.5}, {1.125, 0.45}} ;

main(int argc, char *argv[])
{
    int     win_no = 0, lgt_no = 0, i, j, old_min ;

    StartingMessage ;

    CommandOption(argc, argv) ;
    PolInitialize(argv) ;
    Initialize() ;

    while(True)
    {
        int count = 0 ;

	ControlObjects() ;
	DisplayObjects(pc.eye_obj, lgt_no, win_no) ;

	do
	{
	    pc.button_event = False ;
	    pc.event_loop_flag = False ;
	    if (count) EventOfWindowConfig(OFF, ON) ;
	    else       EventOfWindowConfig(OFF, OFF) ;
	    if (XCheckMaskEvent(X.display, Button1MotionMask, &X.event))
	    {
	        PressureEvents() ;
		pc.motion_event1 = MOTION_COUNT ;
		pc.mouse_x = pc.m1_x = (double)(X.event.xmotion.x) /
		                       (double)XW.size.width - 0.5 ;
		pc.mouse_y = pc.m1_y = (double)(X.event.xmotion.y) /
		                       (double)XW.size.height - 0.5 ;
	    }
	    while (XCheckMaskEvent(X.display, -1, &X.event))
	    {
	        switch(X.event.type)
		{
		  case MotionNotify:
		    PressureEvents() ;
		    pc.motion_event3 = MOTION_COUNT ;
		    pc.mouse_x = pc.m3_x = (double)(X.event.xmotion.x) /
		                           (double)XW.size.width - 0.5 ;
		    pc.mouse_y = pc.m3_y = (double)(X.event.xmotion.y) /
		                           (double)XW.size.height - 0.5 ;
		    break ;
		  case ButtonPress:
		    if (X.event.xbutton.button == Button2)
		    pc.button_event = True ;
		    break ;
		}
	    }
	    pc.t = time(NULL) ;
	    pc.t_area = localtime(&pc.t) ;

	    if (pc.t_area->tm_sec == pc.old_clock && !pc.motion_event1 &&
		!pc.motion_event3 && !pc.move_flag && !pc.button_event)
	        pc.event_loop_flag = True ;

	    if (count > 2) u_sleep(pc.opt.msleep_event * 1000) ;
	    else           u_sleep(pc.opt.msleep_move  * 1000) ;
	    count ++ ;

	} while (pc.event_loop_flag) ;
	pc.init_flag = False ;
    }
    CloseWindow(win_no) ;
    DestroyData() ;
}

#ifdef NO_SELECT  /* NO_SELECT */
void u_sleep(unsigned long usecs)
{
    usleep(usecs) ;
    return ;
}

#else  /* ! NO_SELECT */
void u_sleep(unsigned long usecs)
{
    struct timeval tv ;
    tv.tv_sec  = usecs / 1000000L ;
    tv.tv_usec = usecs % 1000000L ;
    select (0, 0, 0, 0, &tv);
    return ;
}
#endif  /* ! NO_SELECT */


/* ץ */
void CommandOption(char argc, char *argv[])
{
    int         i, n ;
    Color       cl ;
    Color_Data  cd ;

    pc.opt.clock_mode   = MODE_MIN ;
    pc.opt.size         = CLOCK_SIZE ;
    pc.opt.bg_color     = BG_COLOR ;
    pc.opt.fg_color     = FG_COLOR ;
    pc.opt.rg_color     = RG_COLOR ;
    pc.opt.mono_color   = MONO_COLOR ;
    pc.opt.bg           = 0 ;
    pc.opt.fg           = 0 ;
    pc.opt.rg           = 0 ;
    pc.opt.mono         = 0 ;
    pc.opt.msleep_move  = MSLEEP_MOVE ;
    pc.opt.msleep_event = MSLEEP_EVENT ;
    pc.opt.disp         = pc.opt.geom = NULL ;
    pc.opt.title        = "polclock" ;
    pc.opt.rs_mode      = RESIZE_MODE ;
    pc.opt.cl_type      = RGB ;
    pc.opt.nm_red       = NM_RED ;
    pc.opt.nm_green     = NM_GREEN ;
    pc.opt.nm_blue      = NM_BLUE ;
    pc.opt.nm_colors    = NM_COLORS ;
    pc.opt.seg_depth    = SEG_DEPTH ;
    pc.opt.trans_rate   = TRANS_RATE ;
    pc.opt.nm_balls     = NM_LIGHT_BALLS ;
    pc.opt.ball_size    = LIGHT_BALL_SIZE ;

    for (i = 1 ; i < argc ; i ++)
    {
        n = 0 ;

        if (strcmp(argv[i], "-little") == 0)      pc.opt.size = SIZE_LITTLE ;
        else if (strcmp(argv[i], "-small") == 0)  pc.opt.size = SIZE_SMALL ;
        else if (strcmp(argv[i], "-middle") == 0) pc.opt.size = SIZE_MIDDLE ;
        else if (strcmp(argv[i], "-large") == 0)  pc.opt.size = SIZE_LARGE ;
        else if (strcmp(argv[i], "-huge") == 0)   pc.opt.size = SIZE_HUGE ;

        else if (strcmp(argv[i], "-sec") == 0)   pc.opt.clock_mode = MODE_SEC ;
        else if (strcmp(argv[i], "-min") == 0)   pc.opt.clock_mode = MODE_MIN ;

        else if (strcmp(argv[i], "-color") == 0) pc.opt.cl_type = RGB ;
	else if (strcmp(argv[i], "-mono") == 0 && i + 1 < argc)
	{
	    pc.opt.cl_type = SCALE ;
	    pc.opt.mono = i ++ ;
	}

	else if (strncmp(argv[i], "-display", 5) == 0 && i + 1 < argc)
	    pc.opt.disp = argv[++ i] ;
	else if (strncmp(argv[i], "-geometry", 5) == 0 && i + 1 < argc)
	    pc.opt.geom = argv[++ i] ;
	else if (strncmp(argv[i], "-title", 5) == 0 && i + 1 < argc)
	    pc.opt.title = argv[++ i] ;
	else if (strcmp(argv[i], "-notitle") == 0)
	    pc.opt.title = NULL ;

	else if (strcmp(argv[i], "-bg") == 0 && i + 1 < argc)
	    pc.opt.bg = i ++ ;
	else if (strcmp(argv[i], "-fg") == 0 && i + 2 < argc)
	{
	    pc.opt.fg = i ;
	    i += 2 ;
	}
	else if (strcmp(argv[i], "-rg") == 0 && i + 2 < argc)
	{
	    pc.opt.rg = i ;
	    i += 2 ;
	}

	else if (strcmp(argv[i], "-mi") == 0 && i + 1 < argc)
	    pc.opt.msleep_move = atoi(argv[++ i]) ;
	else if (strcmp(argv[i], "-ei") == 0 && i + 1 < argc)
	    pc.opt.msleep_event = atoi(argv[++ i]) ;

	else if (strcmp(argv[i], "-rs") == 0 && i + 1 < argc)
	{
	    if (strcmp(argv[++ i], "short") == 0)
	        pc.opt.rs_mode = RESIZE_SHORT ;
	    else if (strcmp(argv[i], "width") == 0)
	        pc.opt.rs_mode = RESIZE_WIDTH ;
	    else if (strcmp(argv[i], "height") == 0)
	        pc.opt.rs_mode = RESIZE_HEIGHT ;
	    else if (strcmp(argv[i], "long") == 0)
	        pc.opt.rs_mode = RESIZE_LONG ;
	    else if (strcmp(argv[i], "scaling") == 0)
	        pc.opt.rs_mode = RESIZE_DEFAULT ;
	}

	else if (strcmp(argv[i], "-R") == 0 && i + 1 < argc)
	    pc.opt.nm_red = atoi(argv[++ i]) ;
	else if (strcmp(argv[i], "-G") == 0 && i + 1 < argc)
	    pc.opt.nm_green = atoi(argv[++ i]) ;
	else if (strcmp(argv[i], "-B") == 0 && i + 1 < argc)
	    pc.opt.nm_blue = atoi(argv[++ i]) ;
	else if (strcmp(argv[i], "-CL") == 0 && i + 1 < argc)
	    pc.opt.nm_colors = atoi(argv[++ i]) ;

	else if (strncmp(argv[i], "-segdepth", 5) == 0 && i + 1 < argc)
	    pc.opt.seg_depth = atof(argv[++ i]) ;
	else if (strncmp(argv[i], "-trate", 5) == 0 && i + 1 < argc)
	{
	    if ((pc.opt.trans_rate = atof(argv[++ i])) <= 0.0 ||
		pc.opt.trans_rate > 1.0) pc.opt.trans_rate = TRANS_RATE ;
	}

	else Usage() ;
    }
    return ;
}

void PolInitialize(char *argv[])
{
    int     geom_flag ;
    Color   cl ;
    S_Size  ss ;
    W_Size  ws ;

    if (CreateData(pc.opt.disp) == ERROR_CODE)
    {
        fprintf(stderr, "CreateData error\n") ;
        exit(1) ;
    }

    if (pc.opt.bg)
    {
        Color   cl ;
        cl = GetColorByName(argv[pc.opt.bg + 1]) ;
	if (cl.r >= 0.0)
	    pc.opt.bg_color = cl ;
    }
    if (pc.opt.mono)
    {
        Color   cl ;
        cl = GetColorByName(argv[pc.opt.mono + 1]) ;
	if (cl.r >= 0.0)
	    pc.opt.mono_color = cl ;
    }

    if (pc.opt.fg) OptionToColor_Data(&pc.opt.fg_color, pc.opt.fg, argv) ;
    if (pc.opt.rg) OptionToColor_Data(&pc.opt.rg_color, pc.opt.rg, argv) ;

    ws = w_size[pc.opt.clock_mode][pc.opt.size] ;
    ss = s_size[pc.opt.clock_mode] ;

    SetWindowStatus(0, ws.width,ws.height, pc.opt.cl_type,
		    pc.opt.mono_color.r, pc.opt.mono_color.g,
		    pc.opt.mono_color.b, 2.0, ss.width,ss.height,
		    Button1MotionMask | Button3MotionMask |ButtonPressMask,
		    pc.opt.rs_mode) ;

    if (pc.opt.cl_type == RGB)
        SetNumberOfRGB(0, pc.opt.nm_red, pc.opt.nm_green, pc.opt.nm_blue) ;
    else
        SetNumberOfRGB(0, pc.opt.nm_colors, 1, 1) ;
    SetWindowSizeHintsByGeometry(0, pc.opt.geom,
				 wSize[pc.opt.clock_mode][pc.opt.size]) ;

    SetupWindow(0, pc.opt.title) ;
    SetLightStatus(0, C_White, pc.opt.bg_color, GetPoint(5,4,0), 10000) ;
    return ;
}

void OptionToColor_Data(Color_Data *cd, int i, char *argv[])
{
    int     strct ;
    Color   cl ;

    cl = GetColorByName(argv[++ i]) ;
    if (strncmp(argv[++ i], "mat", 3) == 0) strct = MAT ;
    else if (strncmp(argv[i], "plastic", 3) == 0) strct = PLASTIC ;
    else if (strncmp(argv[i], "metal", 3) == 0) strct = METAL ;
    else strct = MAT ;
    if (cl.r >= 0.0) *cd = GetColor_DataByColor(cl, strct) ;
    return ;
}

void Usage(void)
{
    fprintf(stderr, "Usage: polclock [options]\n") ;
    fprintf(stderr, "       Options  [-little | -small |") ;
    fprintf(stderr, " -middle | -large | -huge]\n") ;
    fprintf(stderr, "                  clock size (default middle)\n") ;
    fprintf(stderr, "                [-min | -sec]\n") ;
    fprintf(stderr, "                  clock mode (default min)\n") ;
    fprintf(stderr, "                [-color | -mono <color>]\n") ;
    fprintf(stderr, "                  color type (default color)\n") ;
    fprintf(stderr, "                [-bg <color>]\n") ;
    fprintf(stderr, "                  background color\n") ;
    fprintf(stderr, "                [-fg <color> {mat | plastic | metal}]\n") ;
    fprintf(stderr, "                  foreground color\n") ;
    fprintf(stderr, "                [-rg <color> {mat | plastic | metal}]\n") ;
    fprintf(stderr, "                  reverse color\n") ;
    fprintf(stderr, "                [-display <display>]\n") ;
    fprintf(stderr, "                [-geometry <geometry>]\n") ;
    fprintf(stderr, "                [-title <title>]\n") ;
    fprintf(stderr, "                [-notitle]\n") ;
    fprintf(stderr, "                [-mi msec]\n") ;
    fprintf(stderr, "                  moveloop interval (default 10)\n") ;
    fprintf(stderr, "                [-ei msec]\n") ;
    fprintf(stderr, "                  eventloop interval (default 100)\n") ;
    fprintf(stderr, "                [-rs {short | long | width | height | scaling}]\n") ;
    fprintf(stderr, "                  resize mode (default short)\n") ;

    fprintf(stderr, "                [-{R | G | B} ncolors]\n") ;
    fprintf(stderr, "                  only use PseudoColor\n") ;
    fprintf(stderr, "                  number of RGB colorcells (default R:5, G:6, B:4)\n") ;
    fprintf(stderr, "                [-CL ncolors]\n") ;
    fprintf(stderr, "                  only for PseudoColor && -mono\n") ;
    fprintf(stderr, "                  number of colorcells (default 32)\n") ;
    fprintf(stderr, "                  -R -G -B & -CL >= 2\n") ;
    fprintf(stderr, "                [-segdepth depth]\n") ;
    fprintf(stderr, "                  depth of seg (default 4.0)\n") ;
    fprintf(stderr, "                [-trate rate]\n") ;
    fprintf(stderr, "                  trans rate of seg\n") ;
    fprintf(stderr, "                  0.0 < rate <= 1.0 (default 0.86)\n") ;

    exit(1) ;
}

void Initialize(void)
{
    CreateTP_Data() ;
    CreateClasses() ;
    CreateInstances() ;
    InitializeData() ;

    return ;
}

void CreateTP_Data(void)
{
    double  d ;
    if ((d = -pc.opt.seg_depth) == 0.0) d = -0.0000001 ;

    pc.tp[0].display_switch = ON ;
    pc.tp[0].nm_polygons = 7 ;
    pc.tp[0].nm_tops = 8 ;
    pc.tp[0].color2 = pc.opt.bg_color ;

    pc.tp[0].tops[0] = GetPoint(   5, 0, 20) ;
    pc.tp[0].tops[1] = GetPoint(  -2, 0, 13) ;
    pc.tp[0].tops[2] = GetPoint(  -9, 0,-16) ;
    pc.tp[0].tops[3] = GetPoint(  -5, 0,-20) ;
    pc.tp[0].tops[4] = GetPoint(   2, 0,-13) ;
    pc.tp[0].tops[5] = GetPoint(   9, 0, 16) ;
    pc.tp[0].tops[6] = GetPoint( 2.5, d, 10) ;
    pc.tp[0].tops[7] = GetPoint(-2.5, d,-10) ;

    CreatePolData(&pc.tp[0]) ;


    pc.tp[1].display_switch = ON ;
    pc.tp[1].nm_polygons = 7 ;
    pc.tp[1].nm_tops = 8 ;
    pc.tp[1].color2 = pc.opt.bg_color ;

    pc.tp[1].tops[0] = GetPoint(-20, 0, -2) ;
    pc.tp[1].tops[1] = GetPoint(-17, 0, -5) ;
    pc.tp[1].tops[2] = GetPoint( 13, 0, -5) ;
    pc.tp[1].tops[3] = GetPoint( 20, 0,  2) ;
    pc.tp[1].tops[4] = GetPoint( 17, 0,  5) ;
    pc.tp[1].tops[5] = GetPoint(-13, 0,  5) ;
    pc.tp[1].tops[6] = GetPoint(-10, d,  0) ;
    pc.tp[1].tops[7] = GetPoint( 10, d,  0) ;

    CreatePolData(&pc.tp[1]) ;


    pc.tp[2].display_switch = ON ;
    pc.tp[2].nm_polygons = 5 ;
    pc.tp[2].nm_tops = 5 ;
    pc.tp[2].color2 = pc.opt.bg_color ;

    pc.tp[2].tops[0] = GetPoint( -3, 0,  8) ;
    pc.tp[2].tops[1] = GetPoint( -7, 0, -8) ;
    pc.tp[2].tops[2] = GetPoint(  3, 0, -8) ;
    pc.tp[2].tops[3] = GetPoint(  7, 0,  8) ;
    pc.tp[2].tops[4] = GetPoint(  0, d * 0.75,  0) ;

    pc.tp[2].polygons[0].polygon_switch = ON ;
    pc.tp[2].polygons[0].color_data = pc.opt.rg_color ;
    pc.tp[2].polygons[0].nm_tops = 3 ;
    pc.tp[2].polygons[0].top_no[0] = 0 ;
    pc.tp[2].polygons[0].top_no[1] = 1 ;
    pc.tp[2].polygons[0].top_no[2] = 4 ;

    pc.tp[2].polygons[1].polygon_switch = ON ;
    pc.tp[2].polygons[1].color_data = pc.opt.rg_color ;
    pc.tp[2].polygons[1].nm_tops = 3 ;
    pc.tp[2].polygons[1].top_no[0] = 1 ;
    pc.tp[2].polygons[1].top_no[1] = 2 ;
    pc.tp[2].polygons[1].top_no[2] = 4 ;

    pc.tp[2].polygons[2].polygon_switch = ON ;
    pc.tp[2].polygons[2].color_data = pc.opt.rg_color ;
    pc.tp[2].polygons[2].nm_tops = 3 ;
    pc.tp[2].polygons[2].top_no[0] = 2 ;
    pc.tp[2].polygons[2].top_no[1] = 3 ;
    pc.tp[2].polygons[2].top_no[2] = 4 ;

    pc.tp[2].polygons[3].polygon_switch = ON ;
    pc.tp[2].polygons[3].color_data = pc.opt.rg_color ;
    pc.tp[2].polygons[3].nm_tops = 3 ;
    pc.tp[2].polygons[3].top_no[0] = 3 ;
    pc.tp[2].polygons[3].top_no[1] = 0 ;
    pc.tp[2].polygons[3].top_no[2] = 4 ;

    pc.tp[2].polygons[4].polygon_switch = ON ;
    pc.tp[2].polygons[4].color_data = pc.opt.fg_color ;
    pc.tp[2].polygons[4].nm_tops = 4 ;
    pc.tp[2].polygons[4].top_no[0] = 3 ;
    pc.tp[2].polygons[4].top_no[1] = 2 ;
    pc.tp[2].polygons[4].top_no[2] = 1 ;
    pc.tp[2].polygons[4].top_no[3] = 0 ;

    return ;
}

void CreatePolData(TP_Data *tp)
{
#define T_P     tp->polygons

    T_P[0].polygon_switch = ON ;
    T_P[0].color_data = pc.opt.fg_color ;
    T_P[0].nm_tops = 3 ;
    T_P[0].top_no[0] = 6 ;
    T_P[0].top_no[1] = 5 ;
    T_P[0].top_no[2] = 0 ;

    T_P[1].polygon_switch = ON ;
    T_P[1].color_data = pc.opt.fg_color ;
    T_P[1].nm_tops = 3 ;
    T_P[1].top_no[0] = 6 ;
    T_P[1].top_no[1] = 0 ;
    T_P[1].top_no[2] = 1 ;

    T_P[2].polygon_switch = ON ;
    T_P[2].color_data = pc.opt.fg_color ;
    T_P[2].nm_tops = 3 ;
    T_P[2].top_no[0] = 7 ;
    T_P[2].top_no[1] = 2 ;
    T_P[2].top_no[2] = 3 ;

    T_P[3].polygon_switch = ON ;
    T_P[3].color_data = pc.opt.fg_color ;
    T_P[3].nm_tops = 3 ;
    T_P[3].top_no[0] = 7 ;
    T_P[3].top_no[1] = 3 ;
    T_P[3].top_no[2] = 4 ;

    T_P[4].polygon_switch = ON ;
    T_P[4].color_data = pc.opt.fg_color ;
    T_P[4].nm_tops = 4 ;
    T_P[4].top_no[0] = 6 ;
    T_P[4].top_no[1] = 1 ;
    T_P[4].top_no[2] = 2 ;
    T_P[4].top_no[3] = 7 ;

    T_P[5].polygon_switch = ON ;
    T_P[5].color_data = pc.opt.fg_color ;
    T_P[5].nm_tops = 4 ;
    T_P[5].top_no[0] = 7 ;
    T_P[5].top_no[1] = 4 ;
    T_P[5].top_no[2] = 5 ;
    T_P[5].top_no[3] = 6 ;

    T_P[6].polygon_switch = ON ;
    T_P[6].color_data = pc.opt.rg_color ;
    T_P[6].nm_tops = 6 ;
    T_P[6].top_no[0] = 5 ;
    T_P[6].top_no[1] = 4 ;
    T_P[6].top_no[2] = 3 ;
    T_P[6].top_no[3] = 2 ;
    T_P[6].top_no[4] = 1 ;
    T_P[6].top_no[5] = 0 ;

    return ;

#undef  T_P
}

/* 饹 */
void CreateClasses(void)
{
    int     id[8], i ;
    Locate  l[20] ;
    Color   cl ;
    double  r ;

    pc.none_class  = CreateClassOfNone() ;
    pc.seg_v_class = CreateClassByTP_Data(&(pc.tp[0])) ;
    pc.seg_u_class = CreateClassByTP_Data(&(pc.tp[1])) ;
    pc.dot_class   = CreateClassByTP_Data(&(pc.tp[2])) ;

    l[0] = GetLocate( 10, 0, 42,  0,0,0, 1,1,1) ;
    l[1] = GetLocate(-16, 0, 19,  0,0,0, 1,1,1) ;
    l[2] = GetLocate( 26, 0, 23,  0,0,0, 1,1,1) ;
    l[3] = GetLocate(  0, 0,  0,  0,0,0, 1,1,1) ;
    l[4] = GetLocate(-26, 0,-23,  0,0,0, 1,1,1) ;
    l[5] = GetLocate( 16, 0,-19,  0,0,0, 1,1,1) ;
    l[6] = GetLocate(-10, 0,-42,  0,0,0, 1,1,1) ;
    id[0] = id[3] = id[6] =         pc.seg_u_class ;
    id[1] = id[2] = id[4] = id[5] = pc.seg_v_class ;
    pc.num_class = CreateClassHaveChildren(NONE, 7, id, l, INDEPENDANCE) ;

    l[0] = GetLocate( 5, 0, 20,  0,0,0, 1,1,1) ;
    l[1] = GetLocate(-5, 0,-20,  0,0,0, 1,1,1) ;
    id[0] = id[1] = pc.dot_class ;
    pc.colon_class = CreateClassHaveChildren(NONE,2, id, l, INDEPENDANCE) ;

    l[0] = GetLocate( 165, 0,-8,  0,0,0, 0.75,0.75,0.75) ;
    l[1] = GetLocate( 117, 0,-8,  0,0,0, 0.75,0.75,0.75) ;
    l[2] = GetLocate(  40, 0, 0,  0,0,0, 1,1,1) ;
    l[3] = GetLocate( -24, 0, 0,  0,0,0, 1,1,1) ;
    l[4] = GetLocate(-112, 0, 0,  0,0,0, 1,1,1) ;
    l[5] = GetLocate(-176, 0, 0,  0,0,0, 1,1,1) ;
    l[6] = GetLocate(  84, 0,-8,  0,0,0, 0.75,0.75,0.75) ;
    l[7] = GetLocate( -68, 0, 0,  0,0,0, 1,1,1) ;
    id[0] = id[1] = id[2] = id[3] = id[4] = id[5] = pc.num_class ;
    id[6] = id[7] = pc.colon_class ;
    if (pc.opt.clock_mode == MODE_MIN)
        for (i = 0 ; i < 8 ; i ++) l[i].point.x += 68.0 ;
    pc.clock_class =
      CreateClassHaveChildren(NONE, 8, id, l, INDEPENDANCE) ;

    r = 2.0 ;
    cl = C_Cyan ;

    for (i = 0 ; i < 20 ; i ++)
      l[i] = GetLocate(Rand(300) - 150, Rand(300) - 150, Rand(300) -150,
		       0,0,0, 1,1,1) ;
    pc.star_class = CreateClassOfGatheringCircles(1, &cl, &r, 20, l) ;


    id[0] = pc.clock_class ; id[1] = pc.star_class ;
    l[0] = l[1] = L_Normal ;

    pc.order_class =
      CreateClassHaveChildren(NONE, 2, id, l, INDEPENDANCE) ;

    return ;
}

/* 󥹥󥹺 */
void CreateInstances(void)
{
    int     ids[8], s_ids[7], d_ids[2], n, s ;

    pc.order_obj.obj_no =
      CreateObject(pc.order_class, NONE,
		   GetLocate(0,0,600, 0,80,10, 1,1,1), 0,0,0) ;
    pc.eye_obj = CreateObject(pc.none_class, NONE, L_Normal, 0,0,0) ;

    pc.clock_obj.obj_no = GetInstanceID(pc.order_obj.obj_no, pc.clock_class) ;

    GetInstanceIDsOfChildren(pc.clock_obj.obj_no, 0, ids) ;

    for (n = 0 ; n < 6 ; n ++)
    {
	pc.num_obj[n].obj_no = ids[n] ;
        GetInstanceIDsOfChildren(ids[n], 0, s_ids) ;
	if (pc.opt.clock_mode == MODE_MIN && n < 2)
	{
	    DestroyObject(pc.num_obj[n].obj_no) ;
	    continue ;
	}
	for (s = 0 ; s < 7 ; s ++)
	    pc.seg_obj[n][s].obj_no = s_ids[s] ;
    }
    for (n = 0 ; n < 2 ; n ++)
    {
	GetInstanceIDsOfChildren(ids[6 + n], 0, d_ids) ;
	pc.colon_obj[n][0].obj_no = d_ids[0] ;
	pc.colon_obj[n][1].obj_no = d_ids[1] ;
    }

/*
    {
	GetInstanceIDsOfChildren(ids[6], 0, d_ids) ;
	pc.colon_obj[0][0].obj_no = d_ids[0] ;
	pc.colon_obj[0][1].obj_no = d_ids[1] ;
	GetInstanceIDsOfChildren(ids[7], 0, d_ids) ;
	pc.colon_obj[1][0].obj_no = d_ids[0] ;
	pc.colon_obj[1][1].obj_no = d_ids[1] ;
    }
*/

    if (pc.opt.clock_mode == MODE_MIN) DestroyObject(ids[6]) ;

    return ;
}

void InitializeData(void)
{
    int     n, s ;

    pc.init_flag = True ;
    pc.motion_event1 = pc.motion_event3 = False ;
    pc.distance = 2.0 ;

    for (n = 0 ; n < 6 ; n ++)
    {
        for (s = 0 ; s < 7 ; s ++)
	{
	    pc.seg_data.n[n][s] = 0 ;
	    SetColorRate(pc.seg_obj[n][s].obj_no, 1.0) ;
	}
    }
    return ;
}

Num  GetNum1_10(int nm)
{
    Num     nume ;
    nume.num1  = nm % 10 ;
    nume.num10 = nm / 10 ;
    return nume ;
}

void ClockProcedure(void)
{
    int     n, s ;
    Num     nm ;

    pc.pre_num = pc.num_data ;
    pc.pre_seg = pc.seg_data ;

    pc.t = time(NULL) ;
    pc.t_area = localtime(&pc.t) ;

    pc.old_clock = pc.t_area->tm_sec ;

    nm = GetNum1_10(pc.t_area->tm_sec) ;
    pc.num_data.n[0] = nm.num1 ;
    pc.num_data.n[1] = nm.num10 ;

    nm = GetNum1_10(pc.t_area->tm_min) ;
    pc.num_data.n[2] = nm.num1 ;
    pc.num_data.n[3] = nm.num10 ;

    if (pc.t_area->tm_hour > 12)
        nm = GetNum1_10(pc.t_area->tm_hour - 12) ;
    else
        nm = GetNum1_10(pc.t_area->tm_hour) ;
    pc.num_data.n[4] = nm.num1 ;
    if (nm.num10 == 0) pc.num_data.n[5] = 10 ;
    else               pc.num_data.n[5] = nm.num10 ;

    for (n = 0 ; n < 6 ; n ++)
    {
        if (pc.num_data.n[n] == pc.pre_num.n[n] &&
	    !pc.init_flag) continue ;

        if (pc.opt.clock_mode == MODE_SEC)
	{
	    if (pc.num_data.n[n] == 0 && !pc.init_flag)
	        pc.num_obj[n].revolve_flag = True ;
	    if (n == 0 || n == 2)
	    {
	        pc.colon_obj[n / 2][0].coloring_flag = True ;
		pc.colon_obj[n / 2][0].color_rate = pc.opt.trans_rate ;
		pc.colon_obj[n / 2][1].coloring_flag = True ;
		pc.colon_obj[n / 2][1].color_rate = pc.opt.trans_rate ;
	    }
	}
	else
	{
	    if (n == 0)
	    {
	        pc.colon_obj[1][0].coloring_flag = True ;
		pc.colon_obj[1][0].color_rate =
		  pc.opt.trans_rate * (pc.num_data.n[0] & 1) ;
	        pc.colon_obj[1][1].coloring_flag = True ;
	        pc.colon_obj[1][1].color_rate =
		  pc.opt.trans_rate * (pc.num_data.n[0] & 1) ;
	    }
	    if (n < 2) continue ;
	    pc.num_obj[n].revolve_flag = True ;
	}
	for (s = 0 ; s < 7 ; s ++)
	{
	    if ((pc.seg_data.n[n][s] = seg_data[pc.num_data.n[n]][s]) ==
		pc.pre_seg.n[n][s] && !pc.init_flag) continue ;
	    pc.seg_obj[n][s].coloring_flag = True ;
	    pc.seg_obj[n][s].color_rate =
	      (double)(1 - pc.seg_data.n[n][s]) * pc.opt.trans_rate ;
	}
    }
    return ;
}

void ControlObjects(void)
{
    ClockProcedure() ;
    pc.move_flag = False ;
    ControlOrder() ;
    ControlDistance() ;
    ControlClock() ;
    ControlNum() ;
    ControlSeg() ;
    ControlColon() ;
    return ;
}

void ControlOrder(void)
{
    Angle   a, o ;

    if (!pc.motion_event1) return ;
    pc.motion_event1 -- ;

    a = GetAngle(0.0, pc.m1_y * 360.0 + 90.0, -pc.m1_x * 360.0) ;
    o = GetAngleOnParent(pc.order_obj.obj_no) ;

    DivideMovedAngle(&a, o, GetAngle(0.1, 0.1, 0.1)) ;
    LimitMovedAngle(&a, o, GetAngle(10.0, 10.0, 10.0)) ;

    SetAngleOnParent(pc.order_obj.obj_no, a) ;
    return ;
}

void ControlDistance(void)
{
    double  d, sd ;
    Point   p, o ;

    if (!pc.motion_event3) return ;
    pc.motion_event3 -- ;

    d = data->x_objs.win[0].scr_distance ;
    sd = Pow(2.0, pc.m3_x * 6.0) ;

    if (sd <=  0.2) sd =  0.1 ;
    if (sd >= 10.0) sd = 10.0 ;
    d += (sd - d) * 0.1 ;
    SetScreenDistance(0, d) ;

    p = GetPointOnParent(pc.order_obj.obj_no) ;
    p.z /= pc.distance / 2.0 ;
    o = GetPoint(0.0, 0.0, -pc.m3_y * 1200.0 + 600.0);
    p.z = (p.z + (o.z - p.z) * 0.02) * d / 2.0 ;
    SetPointOnParent(pc.order_obj.obj_no, p) ;

    pc.distance = d ;
    return ;
}

void ControlClock(void)
{
    ControlObj(&pc.clock_obj, CLOCK_REVOLVE, 0) ;
    if ((!pc.clock_obj.revolve_flag) && (!pc.motion_event1) &&
	Rand(2000.0) < 1.0) pc.clock_obj.revolve_flag = pc.move_flag = True ;
    if (pc.button_event)
    {
        int     n, s ;
	switch (pc.opt.clock_mode)
	{
	  case MODE_SEC:
	    for (n = 0 ; n < 6 ; n ++)
	        for (s = 0 ; s < 7 ; s ++)
		    pc.seg_obj[n][s].revolve_flag = True ;
	    for (n = 0 ; n < 2 ; n ++)
	        for (s = 0 ; s < 2 ; s ++)
		    pc.colon_obj[n][s].revolve_flag = True ;
	    break ;
	  case MODE_MIN:
	    for (n = 2 ; n < 6 ; n ++)
	        for (s = 0 ; s < 7 ; s ++)
		    pc.seg_obj[n][s].revolve_flag = True ;
	    for (s = 0 ; s < 2 ; s ++)
	        pc.colon_obj[1][s].revolve_flag = True ;
	    break ;
	}
    }
    return ;
}

void ControlNum(void)
{
    int     n ;
    if (pc.opt.clock_mode == MODE_SEC)
    {
        for (n = 0 ; n < 6 ; n ++)
	    ControlObj(&pc.num_obj[n], NUM_REVOLVE, 0) ;
    }
    else
    {
        for (n = 2 ; n < 6 ; n ++)
	    ControlObj(&pc.num_obj[n], NUM_REVOLVE * 2, 0) ;
    }
    return ;
}

void ControlSeg(void)
{
    int     n, s ;
    if (pc.opt.clock_mode == MODE_SEC)
    {
        for (n = 0 ; n < 6 ; n ++)
	    for (s = 0 ; s < 7 ; s ++)
	        ControlObj(&pc.seg_obj[n][s], SEG_REVOLVE, SEG_COLORING) ;
    }
    else
    {
        for (n = 2 ; n < 6 ; n ++)
	    for (s = 0 ; s < 7 ; s ++)
	        ControlObj(&pc.seg_obj[n][s], SEG_REVOLVE, SEG_COLORING * 6) ;
    }
    return ;
}

void ControlColon(void)
{
    int     c, d ;

    if (pc.opt.clock_mode == MODE_SEC)
    {
        for (c = 0 ; c < 2 ; c ++)
	{
	    for (d = 0 ; d < 2 ; d ++)
	    {
	        ControlObj(&pc.colon_obj[c][d], COLON_REVOLVE,COLON_COLORING) ;
		if (GetColorRate(pc.colon_obj[c][d].obj_no) ==
		    pc.opt.trans_rate)
		{
		    pc.colon_obj[c][d].coloring_flag = pc.move_flag = True ;
		    pc.colon_obj[c][d].color_rate = 0.0 ;
		}
	    }
	}
    }
    else for (d = 0 ; d < 2 ; d ++)
        ControlObj(&pc.colon_obj[1][d], COLON_REVOLVE, COLON_COLORING / 2) ;

    return ;
}

void ControlObj(Obj *objs, int nm_revolve, int nm_coloring)
{
    double  r_step, c_step, color_rate ;
    Angle   angle ;

    if (!(objs->revolve_flag || objs->coloring_flag)) return ;
    if (nm_revolve)
    {
        r_step = 360.0 / (double)nm_revolve ;
        if (objs->revolve_flag)
	{
	    Revolve(objs->obj_no, GetAngle(r_step, 0, 0)) ;
	    angle = GetAngleOnParent(objs->obj_no) ;
	    if (Abs(angle.rl) < r_step / 2.0)
	    {
	        objs->revolve_flag = False ;
		angle.rl = 0.0 ;
		SetAngleOnParent(objs->obj_no, angle) ;
	    }
	}
    }
    if (nm_coloring)
    {
        c_step = pc.opt.trans_rate / (double)nm_coloring ;
        color_rate = GetColorRate(objs->obj_no) ;
	if (Abs(objs->color_rate - color_rate) > c_step)
	{
	    color_rate += Sgn(objs->color_rate - color_rate) * c_step ;
	    SetColorRate(objs->obj_no, color_rate) ;
	}
	else
	{
	    SetColorRate(objs->obj_no, objs->color_rate) ;
	    objs->coloring_flag = False ;
	}
    }
    pc.move_flag |= objs->revolve_flag | objs->coloring_flag ;
    return ;
}
