# Copyright (c) 1997 Sun Microsystems, Inc.
# All rights reserved.
# 
# Permission is hereby granted, without written agreement and without
# license or royalty fees, to use, copy, modify, and distribute this
# software and its documentation for any purpose, provided that the
# above copyright notice and the following two paragraphs appear in
# all copies of this software.
# 
# IN NO EVENT SHALL SUN MICROSYSTEMS, INC. BE LIABLE TO ANY PARTY FOR
# DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
# OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF SUN
# MICROSYSTEMS, INC. HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# 
# SUN MICROSYSTEMS, INC. SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THE SOFTWARE PROVIDED
# HEREUNDER IS ON AN "AS IS" BASIS, AND SUN MICROSYSTEMS, INC. HAS NO
# OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR
# MODIFICATIONS.

package SyncCM::pilot;

use SyncCM::timeops;
use Carp;
use strict;
use PDA::Pilot;
use POSIX;
use sigtrap;

my (%NeutralRepeat);
my (%PilotRepeat);
my (%UnitConvert);
my ($DEFAULTS);

%UnitConvert =
    (
     60           => "minutes",
     60 * 60      => "hours",
     24 * 60 * 60 => "days",
     "minutes"    => 60,
     "hours"      => 60 * 60,
     "days"       => 24 * 60 * 60,
     );

%NeutralRepeat =
    ( 
     "None" => ["none", 0],
     "Daily" => ["daily", "n"],
     "Weekly" => ["weekly", "n"],
     "MonthlyByDay" => ["monthly-day", "n"],
     "MonthlyByDate" => ["monthly-date", "n"],
     "Yearly" => ["yearly", "n"],
     );

%PilotRepeat =
    (
     "none" => "None",
     "daily" => "Daily",
     "weekly" => "Weekly",
     "monthly-day" => "MonthlyByDay",
     "monthly-date" => "MonthlyByDate",
     "yearly" => "Yearly",
     );

sub cleanup
{
    my ($pilot_dbhandle) = @_;
    $pilot_dbhandle->purge();
    $pilot_dbhandle->resetFlags();
}

sub setup
{
    ($DEFAULTS) = @_;
}

sub loadChangedAppointments
{
    my ($pilot_dbhandle) = @_;
    my ($db);
    my ($id, $record);
    my ($i);
    
    SyncCM::status("Loading Pilot Datebook [fast sync]", 0);
    while (1)
    {
	SyncCM::checkCancel();

	$record = $pilot_dbhandle->getNextModRecord();

	if (!defined($record))
	{
	    PilotMgr::checkErrNotFound($pilot_dbhandle);
	    last;
	}

	$id = $record->{"id"};

	if ($record->{"deleted"} || $record->{"archived"})
	{
	    $db->{$id} = "DELETED";
	}
	else
	{
	    eval
	    {
		$db->{$id} = &pilotToNeutral($record, $id);
	    };
	    if ($@)
	    {
		my ($summary, $date);

		$@ =~ s/X#X.*//;
		$summary = $record->{"description"};
		$date = 
		    localtime(Time::Local::timelocal(@{$record->{"begin"}}));

		PilotMgr::msg("Pilot appointment $summary / $date " .
			      "is unsupported\n$@");
	    }
	}
    }
    SyncCM::status("Loading Pilot Datebook [fast sync]", 100);

    return $db;
}

sub loadAppointments
{
    my ($pilot_dbhandle) = @_;
    my ($db);
    my ($record, $i, $id);
    my ($max);
    my $errstr;

    $i = 0;
    $max = $pilot_dbhandle->getRecords();

    if ($max == 0)
    {
	return undef;
    }

    while (1)
    {
	unless ($i % &fake_ceil($max / 20))
	{
	    SyncCM::status("Loading Pilot Datebook [full sync]",
			   &fake_ceil(100 * $i / $max));
	    SyncCM::checkCancel();
	}

	$record = $pilot_dbhandle->getRecord($i);

	$i++;

	if (!defined($record))
	{
	    PilotMgr::checkErrNotFound($pilot_dbhandle);
	    last;
	}

	next if ($record->{"deleted"});
	next if ($record->{"archived"});
	next if ($record->{"busy"}); # This should never happen

	$id = $record->{"id"};

	eval
	{
	    $db->{$id} = &pilotToNeutral($record, $id);
	};
	if ($@)
	{
	    my ($summary, $date);

	    $@ =~ s/X#X.*//;
	    $summary = $record->{"description"};
	    $date = "";
	    eval {
	      $date = 
		localtime(Time::Local::timelocal(@{$record->{"begin"}}));
	    };

	    PilotMgr::msg("Pilot appointment $summary / $date " .
			  "is unsupported\n$@");
	}
    }

    SyncCM::status("Loading Pilot Datebook [full sync]", 100);

    return $db;
}

sub pilotToNeutral
{
    my ($pi_appt, $id) = @_;
    my ($appt, $i, $tm);

    $appt->{"repeat_forever"} = 0;
    $appt->{"pilot_id"} = $id;

    $appt->{"timeless"} = $pi_appt->{"event"};
    $appt->{"begin"} = Time::Local::timelocal(@{$pi_appt->{"begin"}});

    if ($appt->{"timeless"})
    {
	# The appointment *should* already be at the beginning of the
	# day, but just in case...
	#
	$appt->{"begin"} = SyncCM::timeops::startOfDay($appt->{"begin"});
    }

    if (defined($pi_appt->{"end"}))
    {
	$appt->{"end"} = Time::Local::timelocal(@{$pi_appt->{"end"}});

	# Sanity check: end before start?
	#
	if ($appt->{"end"} < $appt->{"begin"})
	{
	    croak("Appointment ends before start date\n" .
		  "Start time: " . 
		  localtime($appt->{"begin"}) . "\n" .
		  "  End time: " . 
		  localtime($appt->{"end"}) . "X#X");
	}

    }

    if (exists($pi_appt->{"repeat"}))
    {
	if ($pi_appt->{"repeat"}->{"type"} eq "Weekly")
	{
	    # Pilot 'days' bits appear to be reversed
	    #
	    my ($bits) = &bitify(reverse @{$pi_appt->{"repeat"}->{"days"}});

	    $appt->{"repeat"} = ["weekly", "n", $bits];
	}
	elsif ($pi_appt->{"repeat"}->{"type"} eq "MonthlyByDay")
	{
	    $appt->{"repeat"} = ["monthly-day", 
			       "n", 
			       int($pi_appt->{"repeat"}->{"day"} / 7),
			       1 << ($pi_appt->{"repeat"}->{"day"} % 7)];
	}
	else
	{
	    $appt->{"repeat"} = 
		[@{$NeutralRepeat{$pi_appt->{"repeat"}->{"type"}}}];
	}
	
	if ($appt->{"repeat"}[1] =~ /n/)
	{
	    $appt->{"repeat"}[1] =~ s/n/$pi_appt->{"repeat"}->{"frequency"}/;
	    $appt->{"repeat"}[1] = eval $appt->{"repeat"}[1];
	}

	if ($pi_appt->{"repeat"}->{"type"} ne "None")
	{
	    if (exists $pi_appt->{"repeat"}->{"end"})
	    {
		my (@tmp);

		# Calculate the offset from the beginning of the day to
		# the exact tick where the repeat would end.
		#
		@tmp = @{$pi_appt->{"repeat"}->{"end"}};
		@tmp[0..2] = (localtime($appt->{"begin"}))[0..2];

		$appt->{"repeat_end"} = Time::Local::timelocal(@tmp);

		# Sanity check: repeat_end before start?
		#
		if ($appt->{"repeat_end"} < $appt->{"begin"})
		{
		    croak("Repeating appointment ends before start date\n" .
			  "     Start time: " . 
			  localtime($appt->{"begin"}) . "\n" .
			  "Repeat end time: " . 
			  localtime($appt->{"repeat_end"}) . "X#X");
		}
	    }
	    else
	    {
		$appt->{"repeat_forever"} = 1;
	    }
	}

	if (exists($pi_appt->{"exceptions"}))
	{
	    my ($offset);
	    my ($e);

	    foreach $e (@{$pi_appt->{"exceptions"}})
	    {
		# Pilot exceptions are at the beginning of the day.
		# Copy the hours:minutes:secs from the 'begin' field.
		#
		@$e[0..2] = @{$pi_appt->{"begin"}}[0..2];
		push(@{$appt->{"exceptions"}}, Time::Local::timelocal(@$e));
	    }
	}
    }
    else
    {
	$appt->{"repeat"} = [@{$NeutralRepeat{"None"}}];
	$appt->{"repeat_forever"} = 0;
    }

    chomp($appt->{"description"} = $pi_appt->{"description"});
    if ($pi_appt->{"note"})
    {
	chomp($appt->{"note"} = $pi_appt->{"note"});
    }

    if ($DEFAULTS->{"Alarm"}->{"on"})
    {
	if (exists($pi_appt->{"alarm"}))
	{
	    my ($units);
	    
	    if ($pi_appt->{"alarm"}->{"units"} =~ /\D/)
	    {
		$units = $UnitConvert{$pi_appt->{"alarm"}->{"units"}};
	    }
	    else
	    {
		$units = $pi_appt->{"alarm"}->{"units"};
	    }
	    
	    $appt->{"alarm"} = ($pi_appt->{"alarm"}->{"advance"} * $units);
	}
    }


    # Annoyingly, it's possible for the start date to be
    # different from the actual date of the first occurence.
    # This can only happen on the weekly repeating appointments
    # and then Calendar Manager gets confused, because it 
    # has no way of keeping track of a start date that preceeds
    # the first appointment, so it changes the 'begin' field
    # to something else (causing the change to ripple back to
    # the pilot).  The easiest way to fix this is to have the
    # Pilot identify the first actual appointment and use that
    # as the begin date.
    #
    if ($appt->{"repeat"}[0] eq "weekly")
    {
	my ($tick) = $appt->{"begin"};

	while (1)
	{
	    if (!$appt->{"repeat_forever"} &&
		exists($appt->{"repeat_end"}) &&
		$tick > $appt->{"repeat_end"})
	    {
		# Hmm.  We passed the end and yet we did not find a legal
		# day to start on.  This is bad.  Silently ignore it.
		# Calendar Manager will fix it and its changes will get
		# propagated back.
		last;
	    }

	    if ($appt->{"repeat"}[2] & 1 << (localtime($tick))[6] &&
		(!defined($appt->{"exceptions"}) ||
		 !grep(/^$tick$/, @{$appt->{"exceptions"}})))
	    {
		# Legal day.  
		#
		if (defined($appt->{"end"}))
		{
		    $appt->{"end"} = $tick + $appt->{"end"} - $appt->{"begin"};
		}
		$appt->{"begin"} = $tick;
		last;
	    }
	    else
	    {
		# Illegal day, check the next day.
		#
		$tick = &SyncCM::timeops::next_ndays($tick, 1);
	    }
	}
    }

    if ($DEFAULTS->{"Privacy"}->{"on"} && $pi_appt->{"secret"})
    {
	$appt->{"private"} = 1;
    }
    else
    {
	$appt->{"private"} = 0;
    }
    

    SyncCM::debug("Pilot converting %s to %s", $pi_appt, $appt);

    return $appt;
}

sub neutralToPilot
{
    my ($pilot_dbhandle, $appt) = @_;
    my ($pi_appt, $i, $tm);

    SyncCM::debug("Pilot converting %s ", $appt);

    $pi_appt = $pilot_dbhandle->newRecord();
    $pi_appt->{"begin"} = [localtime($appt->{"begin"})];
    $pi_appt->{"event"} = $appt->{"timeless"};

    $pi_appt->{"end"} = [localtime($appt->{"end"})]
	if ($appt->{"end"});

    $pi_appt->{"repeat"}->{"type"} = $PilotRepeat{$appt->{"repeat"}[0]};
    $pi_appt->{"repeat"}->{"frequency"} = $appt->{"repeat"}[1];
    $pi_appt->{"repeat"}->{"weekstart"} = 0; # XXX: Do something with this
    $pi_appt->{"category"} = 0;

    if (scalar(@{$appt->{"repeat"}}) > 2)
    {
	if ($pi_appt->{"repeat"}->{"type"} eq "Weekly")
	{
	    my (@days) = split(//, unpack("b*", 
					  pack("c", $appt->{"repeat"}[2])));
	    pop @days;

	    @{$pi_appt->{"repeat"}->{"days"}} = @days;
	}
	elsif ($pi_appt->{"repeat"}->{"type"} eq "MonthlyByDay")
	{
	    $pi_appt->{"repeat"}->{"day"} = $appt->{"repeat"}[2] * 7 + 
		&bitToIndex($appt->{"repeat"}[3]);
	}
    }
    elsif ($appt->{"repeat"}[0] eq "weekly")
    {
	my (@days) = split(//, unpack("b*", 
				      pack("c", # tm_wday
					   1 << $pi_appt->{"begin"}->[6])));
	pop @days;

	@{$pi_appt->{"repeat"}->{"days"}} = @days;
    }


    if (!$appt->{"repeat_forever"})
    {
	$pi_appt->{"repeat"}->{"end"} = [localtime($appt->{"repeat_end"})]
	    if (defined($appt->{"repeat_end"}));
    }

    if (defined $appt->{"exceptions"} && @{$appt->{"exceptions"}})
    {
	my ($tick, @e);

	@{$pi_appt->{"exceptions"}} = ();
	foreach $tick (@{$appt->{"exceptions"}})
	{
	    next unless defined($tick);

	    # Pilot exceptions are at the beginning of the day.
	    # zero out the hours:minutes:secs
	    #
	    @e = localtime($tick);
	    @e[0..2] = (0, 0, 0);
	    push(@{$pi_appt->{"exceptions"}}, [@e]);
	}
    }
	
    $pi_appt->{"description"} = $appt->{"description"};
    if (defined($appt->{"note"}))
    {
	$pi_appt->{"note"} = $appt->{"note"};
    }

    if ($DEFAULTS->{"Alarm"}->{"on"})
    {
	if (defined($appt->{"alarm"}))
	{
	    my ($key, $units);
	    my (@conv) = (
			  24 * 60 * 60,
			  60 * 60,
			  60,
			  );
	    
	    if ($appt->{"alarm"} == 0)
	    {
		$pi_appt->{"alarm"}->{"advance"} = 0;
		$pi_appt->{"alarm"}->{"units"} = 60; # arbitrary
	    }
	    else
	    {
		# XXX: What if this falls through?
		#
		foreach $units (@conv)
		{
		    if ($appt->{"alarm"} % $units == 0)
		    {
			$pi_appt->{"alarm"}->{"advance"} = 
			    $appt->{"alarm"} / $units;
			$pi_appt->{"alarm"}->{"units"} = $units;
			last;
		    }
		}
	    }

	    if ($pi_appt->{"alarm"}->{"advance"} > 99)
	    {
		PilotMgr::msg("Pilot cannot handle alarm count > 99");
		return undef;
	    }
	}
    }
    else
    {
	$pi_appt->{"alarm"} = undef;
    }

    if ($DEFAULTS->{"Privacy"}->{"on"})
    {
	if (defined($appt->{"private"}) && $appt->{"private"})
	{
	    $pi_appt->{"secret"} = 1;
	}
    }

    SyncCM::debug("to %s", $pi_appt);

    return $pi_appt;
}

sub createAppt
{
    my ($pilot_dbhandle, $appt) = @_;

    return &writeAppt($pilot_dbhandle, 0, $appt);
}

sub deleteAppt
{
    my ($pilot_dbhandle, $appt) = @_;
    my ($ret);

    $ret = $pilot_dbhandle->deleteRecord($appt->{"pilot_id"});

    return ($ret < 0);
}

sub changeAppt
{
    my ($pilot_dbhandle, $appt, $orig_id) = @_;
    my ($id);

    $id = &writeAppt($pilot_dbhandle, $orig_id, $appt);

    if ($id > 0 && $orig_id != $id)
    {
        # Change resulted in the creation of a new record, so
        # delete the old one.
        #
        &deleteAppt($orig_id);
    }

    return $id;
}

sub writeAppt
{
    my ($pilot_dbhandle, $id, $appt) = @_;
    my ($pi_appt);
    my ($ret);
    
    $pi_appt = &neutralToPilot($pilot_dbhandle, $appt);

    return undef
	if (!defined($pi_appt));

    $pi_appt->{"id"} = $id;

    $id = $pilot_dbhandle->setRecord($pi_appt);

    if ($id < 0)
    {
	PilotMgr::msg("createAppt(): setRecord failed with error $id");
	return undef;
    }
        
    return($id);
}

sub bitcount
{
    unpack("%B*", pack("i", shift));
}

# Turn an array of 1/0's into number
#
# Replace these routines with pack and unpack someday.
#
sub bitify
{
    my (@arr) = @_;
    my ($result);

    $result = 0;
    while (@arr)
    {
	my ($val) = shift @arr;
	$result <<= 1;
	$result += ($val != 0);
    }

    return $result;
}

sub bitToIndex
{
    my ($val) = @_;
    my ($ret) = 0;

    while ($val / 2 >= 1)
    {
	$val /= 2;
	$ret++;
    }

    return $ret;
}

# POSIX's ceil is hosed on some systems
#
sub fake_ceil
{
    my ($val) = int($_[0]);

    return 1 if ($val == 0);
    return $val;
}

1;
