/*
   Copyright (c) 2000 MicroSource

   This source file and all accompanying files are the exclusive property
   of the author Paul Carapetis and MicroSource, and no modifications are
   to be performed on said source without expressed permission from the
   author or an authorised representative of MicroSource, unless the
   purpose of such modifications is to effect a successful compile on your
   platform.

   This source is only to be distributed in its complete form and only with
   all of the following files included:

        INSTALL
        README
        ibmchars.uu
        license.doc
        newargs.txt
        pcps.man
        register.doc

        makefile
        define.h
        pcps.c
        pcps.h
        struct.h
        unix.c
        unix.h
        valargs.h

   The utility "PCPS" is a SHAREWARE product and you, as the user, agree
   that the source will be removed from your filesystem once the utility
   has been successfully compiled and is found to be functional.  Any
   changes to the source that were required for a successful compile must
   be returned to the author for inclusion in the main source control.

   The utility may be trialed for the period of time mentioned in the file
   license.doc, but must either be registered or destroyed within said
   trial time.  The cost and method of registration is detailed in the file
   license.doc.

Author:  Paul Carapetis  Internet: Paul_Carapetis@iname.com
Company: MicroSource     Snail:    10 Denise Court
                                   Narre Warren
                                   Australia  3805

*/

/*****************************************************************************/
/* PCPS     - A program to input a normal text file, convert it to           */
/*            PostScript and write it to an output file.                     */
/*****************************************************************************/


/*****************************************************************************/
/*                              HISTORY                                      */
/*****************************************************************************/
/*
7.80       Mar 1994   Not converting file path for non-gaudy banner. Fix
                       for last line with no lf. Cut lines were still
                       decrementing YLine. Remove savetty & resetty.  Move
                       reset of FFJustDone to end of while in ReadFileF()
                       Add duplex support. Add executive, legal, and 3
                       envelope paper sizes.  Do not init CURSES if -qt.
                       Add enhaced paper tray support.  Fix EndPage bug
                       when swapping margin in LandScape. Modify for 2 col
                       in both portrait and landscape. Add decimal font
                       sizes. Add configurable line spacing.  Add some
                       differing date formats.

7.90       Aug 1994   Only close OutputFP if opened (NULL pointer in family
                       mode). Clean up some compiler warnings on diff unix
                       platforms.  Fix font size prob when using cfg or adm
                       files.  Add danish fix. Add 2 char args and redesign
                       ProcessArgs().  Support gaudy on 1st page only.
                       Support for tumble in duplex. Fix "-$" in page
                       ranges.  Support for <cr> no <lf>.  Separate PC
                       specific chars to ease posting unix source. Handle
                       ^Z in config files.  Add font size to -id.  Add
                       alternate file name "-fn" and add ##id processing to
                       the file comment and alternate filename. Added mail
                       mode.

7.91       Aug 1994   Fixed ydec in flushline after FF.  Fixed blank line
                       in ConvertChar.

7.92       Aug 1994   UnFixed ydec in flushline after FF :-)

8.00       Dec 1994   Fixed GetOrigin for RS6K. Fixed THeight when -g1 and
                      multiple files. Improve mail header detection algorithm.
                      Added line shading. Made YLine and YDecrement long to
                      avoid overflows in calculations.  Add box page
                      support. Introduce pipe to 'more' for Unix. Make
                      Courier-Bold header font for IBMPCFont. Support
                      flushing/no flush of pages b/w files in duplex.
                      Modified man mode to use widthshow for compatibility
                      on all printers.

8.10       Aug 1995   Enhanced mail headers to handle wrapped lines.  Added
                      2up and 4up support, and made more DSC compliant.
                      Added full ISO-8859-1 support.

8.20       Aug 1996   Fixed bug in defining PCPSTimes-Bold for watermark.
                      Added support for A3,A2,A1,A0 paper sizes. Added
                      enhanced modes for C, Pascal, Script/Perl, 4GL.

8.30       Aug 1997   Ported to Microsoft Visual C++ 4.0 32 bit for Windows
                      95/NT to support lfn & speed it up.  Added support for
                      environment variable PRINTER.  Fixed bug in double
                      PS conversion of FNamePS in non-gaudy banner.
                      Improved enhanced print algorithm to handle more
                      complicated combinations of characters.

8.31       Dec 1997   Fixed bug where under NT the source directory was not
                      being found.

8.32       Jan 1999   On some UNIX platforms if ^C is entered whilst piping
                      to more, session left with no character echo -
                      rectified.  Also updated SOLARIS print command.
                      Duplex mode now set to either true or false, no
                      assumptions made.  Fixed bug with -bx -smo and -ro.

8.40       Apr 2000   Added -nz argument, added fortran mode, added dividing
                      lines between pages in -upx mode,
                      Fixed bugs: Output screen looks OK at >80 cols, Pages
                      now flushed correctly using -up, adjusted 2up scaling
                      factor for USLetter so boxes print correctly.
						

*/

/*  Some general info:

    All global variables contain upper case characters (E.g. ArgString) and
    all local variables are lower case.  If a variable is prefixed with b,
    it is boolean.  Anything all upper case is a defined value or a macro.

    DEFINES:  For family mode, define MSC.  For any unix mode,
              define UNIX plus others as specified in the Makefile.  For
              MSVC++ 4.0 (Windows 95/98/NT), define MSVC32.

              For debugging, define FUNCS to have function names printed
              out as they execute, define DEBUG1 for some further debug
              information especially in the area of memory allocation.

*/

#ifdef NEXT
#define NULL 0
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "define.h"
#include "struct.h"
#include "pcps.h"
#include "valargs.h"

#ifdef UNIX
#include "unix.h"
#else
#include "pc.h"
#endif

/*****************************************************************************/
/*                       Global Variable Declaration                         */
/*****************************************************************************/

BOOL bBackText;                 /* print grey text in background */
BOOL bBannerOn;                 /* if banner required */
BOOL bBoxPage;                  /* surround printable area in a box */
BOOL bControlD;                 /* encompass print in ^D chars */
BOOL bDanishFix;                /* code oslash and Oslash to 155 and 157 */
BOOL bDeleteInput;              /* Delete input file if no errors on output */
BOOL bDuplex;                   /* duplex printing */
BOOL bExtendErr;                /* download ext err handler to printer */
BOOL bFFJustSent;               /* just at the start of a new page */
BOOL bFlushPage;                /* flush pages between files in duplex mode */
BOOL bFNSpecified;              /* An alternate file name has been specified */
BOOL bGaudy1;                   /* print gaudy header on first page only */
BOOL bGaudyMode;                /* print gaudy header */
BOOL bIBMPCFont;                /* IBM ext char supported */
BOOL bISO8859;                  /* ISO-Latin-1 on non-IBMPCFont char sets */
BOOL bLandscape;                /* landscape print */
BOOL bLineLimitSpecified;       /* user specified linelength */
BOOL bLineNumbers;              /* Output line numbers */
BOOL bEnh;                      /* in bold font at present */
BOOL bEnhNLine;                 /* enahnced print next line regardless */
BOOL bManMargin;                /* user has specified at least 1 marg */
BOOL bManualFeed;               /* Manual sheet feed for whole doc */
BOOL bManualFeedPage1;          /* Manual feed for first page only */
BOOL bManualFontSize;           /* user selected font size */
BOOL bMemSort;                  /* inverse pge & enough mem for sort */
BOOL bMemSortAllow;             /* mem sort not manually disabled */
BOOL bNewRev;                   /* use new temp file sort algorithm */
BOOL bNoDisplay;                /* fprintf's will not work */
BOOL bNoSig;                    /* do not print signature (TM) */
BOOL bPrint2nd;                 /* print every other page */
BOOL bPrintCZ;                  /* print ^Z as right arrow */
BOOL bPrintInv;                 /* if reverse order required */
BOOL bPrintName;                /* print user@host in bottom corner */
BOOL bPrintPage;                /* print current page */
BOOL bPrologDone;               /* prolog already sent to output */
BOOL bPSTiming;                 /* printer send timing info */
BOOL bQuiet;                    /* don't print junk */
BOOL bRanging;                  /* page ranges specified */
BOOL bShade;                    /* shade every second line */
BOOL bSkipToEnd;                /* rest of file is not to be printed */
BOOL bSpaceTabs;                /* replace tabs with spaces */
BOOL bStartPageIssued;          /* so we know whether to EndPage or not */
BOOL bSwap;                     /* swap l/r for double sided print */
BOOL bSwapThisPage;             /* doing margin swapping on current page*/
BOOL bTray;                     /* true if tray selection required */
BOOL bTrayName;                 /* true if selction is by name */
BOOL bTumble;                   /* true if tumble with duplex is required */
BOOL bTwoCol;                   /* landscape with 2 columns */
BOOL bUpLines;                  /* Use dividing lines in -up mode? */
BOOL bWasteRest;                /* waste rest of line (cut) */
BOOL bWrapLine;                 /* WRAP not cut */
BOOL bWrapping;                 /* a line wrap is in progress */

char *MiscBuf;                  /* General usage buffer for transient use -
                                   use only locally as many functions use it */
char *StreamBuf;                  /* Orig read line from input file */
char *TextBuf;                  /* Buffer for final PostScript output line */
char *TextPtr;                  /* pointer into TextBuf */

char ActEndSeq[3];              /* current End sequence for enhanced print */
char AltFileName[SMLBUF];       /* string to print instead of file name */
char ArgString[SMLBUF];         /* command line arg string */
char Argum[SMLBUF];             /* common area for processing arguments */
char BackText[BACKTEXTCHARS*2]; /* text to print in background */
char Banner[200];               /* stores Banner string */
char DateFormat;                /* for varying date formats */
char DefAction='\0';            /* def action for files already in ps */
char DirPath[MAX_PATH];         /* to store the dir path for MyFindFirst */
char FileComment[SMLBUF];       /* comment in gaudy header */
char FileDateString[11];        /* file date string */
char FileTimeString[9];         /* file time string */
char FName[MAX_PATH];           /* input file name  */
char FNamePS[MAX_PATH*2];       /* input file name ready for PS (\\ etc) */
char Font[3][30];               /* for storing font names */
char FontCode[6];               /* Requested font code */
char FontPrefix[3][5];          /* prefix for all fonts */
char IDString[SMLBUF];          /* to store the user@host */
char LineNumberPad[1];          /* default char to pad linenumbers */
char LineNumberWidthStr[4];     /* How many columns of line numbers? */
char LineStr[4];                /* either ")B\n" or ")W\n" for PS output line */
char ModeDesc[8][11] = {        /* description of various modes */
  {"Normal"},
  {"Man Page"},
  {"Mail Msg"},
  {"C/C++"},
  {"4GL"},
  {"Shell/PERL"},
  {"Pascal"},
  {"Fortran"},
};
char NoSig[] = {(char)0x4e,
        (char)0x5c,
        (char)0x48,
        (char)0x5e,
				(char)0x44,
        (char)0x0c,
				'\0'}; /* encrypt nosig arg */
char OName[MAX_PATH];           /* output file name */
char Origin[MAX_PATH];          /* Origin path of PCPS.EXE (argv[0]) */
char PageString[6];             /* page */
char PaperTray1[20];            /* the name of the first paper tray */
char PaperTray2[20];            /* the name of the second paper tray */
char PatchString[6];            /* target string for prolog patches */
char PPFName[MAX_PATH]={"$"};   /* user specified printer prep file */
                                /* init as non-null to enable ppf */
char ReEncArray[20];            /* string to point to correct reenc array */
char ReEncFunc[20];             /* string to point to correct reenc function */
char SwapText[6];               /* "true" or "false" */
char TMString[160];             /* string for TM */
char VerString[30];             /* string to store version (PCPS V##) */


char Host[TMSZ] = "???";        /* Machine name for TM */
char UserID[TMSZ] = "???";      /* Userid or machine name for TM in corner */

FILE *FileOut;                  /* temporary file pointer */
FILE *InputFP;                  /* input file pointer */
FILE *OutputFP;                 /* output file pointer */
FILE *Display;                  /* where to send the fancy info */

unsigned ArgNum;                /* the argument we are up to */
unsigned BackFontSize;          /* size of the font for bkgrd grey printing */
unsigned BannerGap;             /* space that banner takes up */
unsigned BottomLine;            /* y coord of bottom of printable area */
unsigned Column;                /* column count */
unsigned Copies;                /* how many copies of each page are required */
unsigned EnhPrintFont;          /* what font style for enhanced print */
unsigned EvenOrOdd;             /* if bPrint2nd set, this indicates whether */
                                /* odd or even pages should be printed */
unsigned FSecs, FMins, FHour;   /* for the file date and time */
unsigned FDay, FMth, FYear;
unsigned FieldCount;            /* count of "@" fields in prolog */
unsigned FontSizeBy10;          /* font size multiplied by 10 */
unsigned HighPage;              /* highest page in range specification */
unsigned IDFontSizeBy10;        /* font size for user@host in corner */
unsigned Key = 0x3d;            /* key for decrypting nosig */
unsigned LineNoOnPage;          /* line number */
unsigned LineNumberWidth;       /* How many columns of line numbers? */
unsigned LinesPerPage;          /* configurable lines per page */
unsigned LinePos;               /* char position on line */
unsigned LineSpace;             /* as a percentage of font size */
unsigned MaxLineLength;         /* calculated max line length */
unsigned MaxPageBytes;          /* size in bytes of biggest page */
unsigned Mode;                  /* current "mode" (-mo#) */
unsigned NewMarg;               /* to find new margin */
unsigned NewLeft = 0;           /* manual margin for Left */
unsigned NewRight = 0;          /* manual margin for Right */
unsigned NewTop = 0;            /* manual margin for Top */
unsigned NewBot = 0;            /* manual margin for Bot */
unsigned PageCount;             /* a page counter */
unsigned PageNo_Doc;            /* current document page number which is the */
                                /* same as PageNo_PS in 1 up mode but != when */
                                /* in -2 mode */
unsigned PageNo_PS;             /* current PostScript page number */
unsigned PagesLeft;             /* a countdown for reversing pages */
unsigned PagesPrinted;          /* the total pages that the printer will output */
unsigned PagesUp;               /* 1up, 2up or 4up */
unsigned Rows;                  /* number of screen rows currently displayed */
unsigned Shade;                 /* the shade factor for line shading 0-1 */
unsigned ShadeLines;            /* Shade every x lines */
unsigned SkipStartSeq;          /* # chars of start to skip to find end */
unsigned Stdin_Tty;             /* non zero if stdin is attached to a tty */
unsigned Stdout_Tty;            /* non zero if stdout is attached to a tty */
unsigned SwapEO;                /* swap l/r margins for dbl side prnt? */
unsigned TabSpace;              /* how many spaces to replace tabs */
unsigned Tray;                  /* which paper tray to use */
unsigned Tray1;                 /* number of first paper tray */
unsigned Tray2;                 /* number of second paper tray */
unsigned TraySwPage;            /* when to switch to other tray */
unsigned TrunCount;             /* # of lines truncated */
unsigned WrapCount;             /* # of lines wrapped */

long CurrentLineNumber;         /* for whole doc - for printing line numbers */
long GHeight, GWidth;           /* for the gaudy function */
long Height;                    /* the printable height */
long MemReq;                    /* how much memory is required */
long MemUsed;                   /* for memory utilisation reporting */
long THeight;                   /* Height at which text starts */
long THeightMod;                /* Theight and YLine modifier marg swap */
long Width;                     /* printable Width */
long XLine;                     /* x co-ord on page */
long YDecrement;                /* co-ord displacement of line feed */
long YLine;                     /* y co-ord on page */

time_t Elapsedtime;             /* for current time */

unsigned long TotalFileBytes;   /* for counting byte offsets in temp file */
                                /* this is a count of non-prolog bytes */

struct tm *Timeptr;             /* for curr time when using stdin */

file_dtl FndFile;               /* destination structure for information */

EnhChar *ActEnh;                /* pointer to table of enh mode chars */

FILE *Pipe;                     /* used when opening pipes */

/*****************************************************************************/
/*                        Function Prototypes                                */
/*                                                                           */
/*            All functions after main are in alphabetical order             */
/*                                                                           */
/*****************************************************************************/
void  main(int,char *[]);

void  AddPage(void);            /* add start page pointer in linked list */
void  AddPageOffset(void);      /* add start page offset in linked list */
void  BadArg(char []);          /* show bad argument, show help and exit */
void  BorWLine(char *);         /* output buffer in ()B or ()W line */
void  CheckEndLine(void);       /* check for wrap/cut point */
BOOL  CheckFile(FILE *, char *);/* check for ps & whether to process */
void  CheckMailHeader(BOOL);    /* check if current line is a mail header */
void  ClearRange(void);         /* clear any current ranges */
void  CompressRange(void);      /* rationalise the range link list */
void  ConvertChar(char *);      /* convert special characters for printing */
void  CopyID(char *);           /* look for %%id and replace with ID string */
void  DoEnhancedPrint(BOOL);    /* enhanced comments etc */
void  DoPage(void);             /* new page or column */
void  DoProlog(void);           /* output the required prolog */
void  DoStartup(unsigned);      /* show banner and check for help */
void  EndPageF(void);            /* end of page sequence */
void  FileLoop(unsigned,char *[]);  /* main file loop */
FILE * FindFile(char *);        /* search for pcps.ext in current & orig dir */
int   FindEndEnh(char *, unsigned); /* find end enh sequence in string */
int   FindStartEnh(char *);     /* find start enh sequence in string */
void  FlushLine(BOOL);          /* Output current line wrap or not */
void  GetArgs(unsigned,char *[]); /* parse all of the arguments */
void  GetCurrentDate(void);     /* load the fndfile struct for stdin */
int   GetLastNum(char *);       /* find last num in string and return as int */
extern void  GetName(void);     /* get userid or mach name for TM in corner */
unsigned GetNum(char [],unsigned); /* get the number following the argument */
extern void  GetOrigin(char *); /* find the origin directory of PCPS.EXE */
void  GetReady(void);           /* setup various parameters */
void  GetSortMemory(long);      /* allocate memory for sorting pages */
void  InsertBanner(void);       /* Write banner to intermmediate output file */
void  InsertLineNumbers(void);  /* insert line numbers in line */
void  InsertRange(unsigned,unsigned); /* insert a page range into linked list */
BOOL  IsNum(char);              /* return true if number */
void  ListFonts(void);          /* list available fonts */
void  LoadFonts(void);          /* load the specified fonts if not default */
void  MakeBanner(void);         /* build the Banner string */
void  MemReverse(void);         /* reverse page order in memory if memsort */
void  MyExit(unsigned,unsigned);/* exit and show message */
void  ParseConfig(FILE *);      /* read file and parse contents as arguments */
int   ParseRanges(char *);      /* parse the range argument */
int   ParseShade(char *);       /* parse -sh arg */
unsigned ParseSize(char []);    /* return size*10 from string */
void  ProcessArg(char *);       /* process an argument set flags or whatever */
void  ProcessFile(char *);      /* processor for each file */
void  ReadFileF(void);           /* read file line by line and process */
void  ReversePages(void);       /* reverse the order of the pages if req */
void  RevPagesN(void);          /* New reverse algorithm */
void  RevPagesO(void);          /* Old reverse algorithm */
void  SetDefaults(void);        /* set up global variable defaults */
void  ShowScreen(void);         /* show them what they're getting */
extern void  ShowUsage(void);   /* show how to use */
void  SLine(char *);            /* output buffer in ()S line */
void  StartPageF(void);          /* start of page sequence */
void  StreamFile(FILE *);       /* send file to output unprocessed */
void  StringOut(char *);        /* output string direct */
void  TermLine(void);           /* terminate long line - wrap or cut */

#ifdef UNIX

extern void  BadSys(int);          /* routine called when bad system call */
extern void  BadPipe(int);         /* routine called for bad pipe */
extern BOOL  GetStat(char *);      /* get file stats for unix */
extern void  Interrupt(int);       /* routine called when int or break detected */
extern char *strlwr(char *);       /* convert string to lower case */
extern char *strupr(char *);       /* convert string to upper case */

#else

extern unsigned   DoMore(void);    /* show [more] and get key */
extern  BOOL MyFindFirst(char *);  /* findfirst for both archs */
extern  BOOL MyFindNext(void);     /* findnext for both archs */
extern void  SpinProp(void);       /* make a fancy prop to show its working */
# ifdef MSVC32
extern void  Title(char *);        /* Title current window */
# endif

#endif

/*****************************************************************************/
/*  Function main()                                                          */
/*                                                                           */
/*****************************************************************************/
void  main (int argc,char *argv[] )
{
  DoStartup(argc);                /* set up banner and look for help */
  GetOrigin(argv[0]);             /* find out where pcps came from */
  SetDefaults();                  /* set up global variable defaults */
  GetArgs(argc,argv);             /* parse command line */
  GetReady();                     /* various parameters eg. max line length */
  if (!(bQuiet || bNoDisplay))    /* if they want to know... */
    ShowScreen();                 /* show them what they're getting */
  FileLoop(argc,argv);            /* go do what we're being paid for! */
  MyExit(0,NOHELP);
}

/*****************************************************************************/
/*  Function AddPage()                                                       */
/*                                                                           */
/*    Add another page pointer to linked list and point to previous one      */
/*    For printing inverse with memory sorting (-i)                          */
/*****************************************************************************/
void AddPage(void)
{
#if defined (UNIX) || defined (MSVC32)
  static pptrm *pptrm_new;
  if (( pptrm_new = (pptrm *)malloc(sizeof(pptrm)) ) == NULL)
#else
  static pptrm huge *pptrm_new;
  if (( pptrm_new = (pptrm huge *)malloc(sizeof(pptrm)) ) == NULL)
#endif
  
  {
    SHOW(Display
         ,"%sCouldn't get memory at page %u for linked list.\n"
          "%sTry printing in sections with -v or use the -if argument.\n"
         ,VERS
         ,PagesPrinted-1
         ,VERS
        );
    MyExit(1,NOHELP);
  }
  pptrm_new->pageptr = BlockPoint;
  if (PPtrM_curr)
  {
    pptrm_new->prevpageptr = PPtrM_curr;
  }
  else
  {
    pptrm_new->prevpageptr = NULL;
  }
  PPtrM_curr = pptrm_new;
}

/*****************************************************************************/
/*  Function AddPageOffset()                                                 */
/*                                                                           */
/*    Add another page offset to linked list and point to previous one       */
/*    For printing inverse with fast file sort (-if)                         */
/*****************************************************************************/
void AddPageOffset(void)
{
  static unsigned pagebytes;
#if defined (UNIX) || defined (MSVC32)
  static pptrf *pptrf_new;

  if ((  pptrf_new = (pptrf *)malloc(sizeof(pptrf)) ) == NULL)
#else
  static pptrf huge *pptrf_new;

  if ((  pptrf_new = (pptrf huge *)malloc(sizeof(pptrf)) ) == NULL)
#endif
  {
    SHOW(Display
         ,"%sCouldn't get memory at page %u for linked list.\n"
          "%sTry printing in sections with -v or use the -io argument.\n"
         ,VERS
         ,PagesPrinted-1
         ,VERS
        );
    MyExit(1,NOHELP);
  }
  pptrf_new->offset = TotalFileBytes;
  if (PPtrF_curr)
  {
    pptrf_new->prevpageptr = PPtrF_curr;
    pagebytes = (unsigned)(TotalFileBytes - PPtrF_curr->offset);
  }
  else
  {
    pptrf_new->prevpageptr = NULL;
    pagebytes = 0;
  }
  PPtrF_curr = pptrf_new;
  if (pagebytes > MaxPageBytes) MaxPageBytes = pagebytes;
#ifdef DEBUG1
  if (PagesPrinted > 1)
    fprintf(Display
            ,"Page %u: Bytes = %u  MaxPageBytes = %u\n"
            ,PagesPrinted-1
            ,pagebytes
            ,MaxPageBytes
           );
#endif
}

/*****************************************************************************/
/*  Function BadArg()                                                        */
/*                                                                           */
/*    Show bad argument then Show how to use                                 */
/*****************************************************************************/
void BadArg(char argum[])
{
  SHOW(Display,"%sInvalid argument %s \"%s\"\n",VerString,ArrowR,argum);
  MyExit(1,HELP);
}

/*****************************************************************************/
/*  Function BorWLine()                                                      */
/*                                                                           */
/*    Output a buffer in a ()B line for normal text, ()W line for man mode   */
/*    Convert characters before processing.                                  */
/*****************************************************************************/
void BorWLine(char *buf)
{
  static char locbuf[10];

  ConvertChar(buf);
  if (*buf)
  {
    if (bPrintPage)
    {
      sprintf(locbuf,"%ld (",YLine);
      StringOut(locbuf);
      StringOut(buf);
      StringOut(LineStr);
    }
  }
}

/*****************************************************************************/
/*  Function CheckEndLine()                                                  */
/*                                                                           */
/*    Check to see if another character will get to the wrap/cut point       */
/*    and call TermLine() if so.  USed by several switches in ReadFileF()     */
/*****************************************************************************/
void CheckEndLine(void)
{
  if (++LinePos > MaxLineLength)     /* cut/wrap point */
  {
    TermLine();
  }
}

/*****************************************************************************/
/*  Function CheckFile()                                                     */
/*                                                                           */
/*    Check to see if input file is PostScript.  If so, ask if file should   */
/*    be sent unprocessed, processed or just skipped.                        */
/*****************************************************************************/
BOOL CheckFile(FILE *fp, char *name)
{

  static char ch,buf[5];   /* for reading the first 4 characters of the file */
  static char *cp;

  fread(buf,1,4,fp);
  rewind(fp);
  if ( (cp = strchr(buf,'%')) != NULL)
  {
    if (*(++cp) == '!')
    {  /* if we made it to here, we have found the characters %! in the first
          four bytes of the file and it can be assumed that this is a ps file */
      SHOW(Display
           ,"\aFile \'%s\' is already PostScript.\n"
           ,name
          );
      for (;;)
      {
        if (DefAction)
        {
           ch = DefAction;
        }
        else
        {
          SHOW(Display
               ,"(S)kip, (O)utput unchanged, or (P)rocess? (S, O or P): "
              );
          ch = (char)getchar();
        }
        switch (ch)
        {
          case 'S':
          case 's': SHOW(Display,"Skipping...\n");
                    return(FALSE);       /* tells processfile to skip */

          case 'P':
          case 'p': SHOW(Display,"Processing...\n");
                    return(TRUE);        /* tells processfile to process */

          case 'O':
          case 'o': SHOW(Display,"Output unchanged...");
                    StreamFile(fp);      /* send file to output */
                    SHOW(Display,"Done.\n");
                    return(FALSE);       /* do nothing further */
        } /* end switch */
      } /* end for */
    } /* end if */
  } /* end if */

  return(TRUE);               /* if we got to here, not PostScript */
}
                  
/*****************************************************************************/
/*  Function CheckMailHeader()                                               */
/*                                                                           */
/*    Test to see if current line is a mail header and take appropriate      */
/*    action.  If already in bold do nothing, else switch to bold.  If not   */
/*    header and we are in header bold mode, switch back to normal font.     */
/*****************************************************************************/
void CheckMailHeader(BOOL bwrap)
{
  register char *text_ptr;
  BOOL bheader;


  if ( LineNoOnPage > 1 || PageNo_Doc > 1)
  {
    bheader = FALSE;              /* start by saying it's not a header */

    /* only do this loop if we have to.  If the previous line automatically
       means that this line is bold, then skip the loop, just check for
       continuance of bold onto the next line
    */
    if (bEnhNLine == FALSE)
    {

      /* check for mail header */

      /* A mail header may be made up of several words delimited by hyphens,
         each word beginning with a capital letter and the final word
         terminated by a colon and a space. Lets start looping through each
         word.
      */


      for ( text_ptr = TextBuf; *text_ptr != '\0'; text_ptr++)
      {
        /* This is the start of a word in what may be a header line */

        if ( *text_ptr < 'A' || *text_ptr > 'Z' )
        {
          break;  /* The first letter of the word is not capitalised, so
                  ** this is not a mail header
                  */
        }

        /* we're at the start of a capitalised word.  The next sequence
        ** of chars must all be lower case terminated by either a hyphen
        ** or a colon
        */

        text_ptr++; /* move on to the next char */

        while ( *text_ptr >= 'a' && *text_ptr <= 'z' )
        {
          /* keep going until we find a non-lower case alpha */
          text_ptr++;
        }

        if (*text_ptr == ':')
        {
          /* we have reached a colon - now check for a space following */

          if (*(text_ptr + 1) == ' ')
          {
            /* This line obeys the rules so it is probably a mail header line */

            bheader = TRUE;   /*****Header line*****/
          }
          break;              /* we've finished one way or another */
        }

        /* so its not a colon, the only other legal char is a hyphen */

        if (*text_ptr == '-')
        {
          /* end of this word, so go to the top of the loop and process
          ** the next
          */
          continue;
        }
        else
        {
          /* not a legal character, so break out of the loop */
          break;
        }
      }
    } /* end check for mail header loop */
    else
    { /* bEnhNLine == TRUE*/
      bheader = TRUE;  /* if we have been told to bold this line, it must
                          be a continuation of a mail header line */
    }

    /* So now we know if this line is a mail header line.  Now we need to
       account for some headers that span multiple lines - these will be
       terminated by a comma.
    */
    bEnhNLine = FALSE;
    if (bheader)
    {
      if (TextBuf[strlen(TextBuf)-1] == ',')
      {
        bEnhNLine = TRUE;
      }
      else
      {
        /* We also need to account for the lines starting with "Received:" as
           these tend to span two lines.  If this is one of those, set a flag
           to say keep the next line bold
        */

        if (!strncmp(TextBuf,"Received: ",10))
        {
          bEnhNLine = TRUE;
        }
      }
    }
  }
  else
  {     /* this is line 1 page 1, so we will print it bold anyway, as SMTP
           mail has the first line with no colon */
    bheader = TRUE;
  }

  /* now take any action necessary */

  switch (bheader)
  {
    case FALSE: if (bEnh)    /* only turn it off if it is currently on */
                {
                  StringOut("0 F\n");
                  bEnh = FALSE;
                }
                break;

    case TRUE:  if (!bEnh)   /* only turn it on if it is currently off */
                {
                  StringOut("1 F\n");
                  bEnh = TRUE;
                }
                /* if we are wrapping this line and it is a header line, we
                ** need to carry the wrapping forward.
                */
                if (bwrap)
                {
                  bEnhNLine = TRUE;
                }
                break;
  }
}

/*****************************************************************************/
/*  Function ClearRange()                                                    */
/*                                                                           */
/*    Clear out any current page range list                                  */
/*****************************************************************************/
void ClearRange(void)
{
  Range = Range_1;
  while (Range)
  {
    NRange = Range->NextRange;
    HFREE(Range);
    Range = NRange;
  }
  Range_1 = NULL;
}

/*****************************************************************************/
/*  Function CompressRange()                                                 */
/*                                                                           */
/*    Compress and rationalise the page range linked list                    */
/*****************************************************************************/
void CompressRange(void)
{
  Range = Range_1;
  NRange = Range->NextRange;

  while (NRange)
  {
    if (Range->EndPage < (NRange->StartPage - 1))
    {
      /* this Range doesn't overlap or encompass the next so go to next
         element in list */
      Range = NRange;
      NRange = NRange->NextRange;
    }
    else
    {
      /* curr range extends into next or encompasses */
      /* we will be dumping NRange so reset pointer and set DRange */
      Range->NextRange = NRange->NextRange;
      DRange = NRange;
      if (Range->EndPage <= NRange->EndPage)
      {
        /* next range goes further */
        Range->EndPage = NRange->EndPage;
      }
      NRange = NRange->NextRange;
      HFREE(DRange);
    }
  }
}

/*****************************************************************************/
/*  Function ConvertChar()                                                   */
/*                                                                           */
/*    Convert the characters so they will print correctly.  If >0x7f then    */
/*    convert the character to octal unless we are not using IBMPCFont and   */
/*    the character is a line graphic, in which case we convert it to a      */
/*    close printable character.                                             */
/*****************************************************************************/
void ConvertChar(char *inpbuf)
{

  register unsigned j,k;
  register char *fromptr, *toptr;
  static char octbuf[5];
  static char workbuf[1024];              /* for char conversion */


  strcpy(workbuf,inpbuf);

  if (!(*workbuf))
  {                  /* nothing to print */
    *inpbuf = '\0';
    return;
  }

  fromptr = workbuf;
  toptr = inpbuf;
  for ( j=0; j<strlen(workbuf); j++, fromptr++, toptr++ )
  {
    if ( (*fromptr == '(') || (*fromptr == ')') || (*fromptr == '\\') )
    { /* precede these chars with '\' */
      *toptr++ = '\\';     /* print backslash */
      *toptr = *fromptr;    /* then char */
    }
    else
    {
      if ( (unsigned char)*fromptr >= SPACE     &&
           (unsigned char)*fromptr <= HIGHCHAR
         )
      {
        *toptr = *fromptr;                /* normal char */
      } 
      else                                /* extended chars */
      {
        if (
             (!bIBMPCFont) && (!bISO8859) &&
             ((*fromptr >= (char)0xb3) && (*fromptr <= (char)0xda))
           )  /* if not ibm font and is line graphic */
        {                              /* are ibm line grahics so substitute */
          switch (*fromptr)               /* printable chars */
          {
            case (char)0xb3:                /* Vertical lines to | */
            case (char)0xba: *toptr = '|';
                             break;
            case (char)0xc4: *toptr = '-';   /* Horizontal lines to dash or = */
                             break;
            case (char)0xcd: *toptr = '=';
                             break;
            default:         *toptr = '+';   /* Rest of line graphics */
          }/* endsw */
        }  /* endibm */
        else
        { /* either we are using IBMPCFont, ISO8859 encoding or its not line
             graphic so dump it down as octal */
          sprintf(octbuf,"\\%3.3o",(unsigned char)*fromptr);/* conv to octal */
          for (k=0; k < 4; k++,toptr++)
            *toptr = octbuf[k];
          toptr--;
        }
      }    /* endext */
    }      /* end non ()\ */
  }        /* end of processing this line of text */
  *toptr = '\0';           /* terminate the string */
}

/*****************************************************************************/
/*  Function CopyID()                                                        */
/*                                                                           */
/*    Look for %%id in string and replace this with ID string                */
/*****************************************************************************/
void CopyID(char *buf)
{
  register int a;
  char *ptr;
  static char tmpbuf[SMLBUF];

  if ( (ptr = strstr(buf,"##id")) != NULL)
  {
    a = (int)ptr - (int)buf;
    strncpy(tmpbuf,buf,a);
    tmpbuf[a] = '\0';
    strcat(tmpbuf,IDString);
    ptr += 4;
    strcat(tmpbuf,ptr);
    strcpy(buf,tmpbuf);
  }
}

/*****************************************************************************/
/*  Function DoEnhancedPrint()                                               */
/*                                                                           */
/*    Look for start and end sequences of enhanced print characters          */
/*****************************************************************************/
void DoEnhancedPrint(BOOL bwrap)
{
  static BOOL bblinedone;
  static int  start, end;

  /* OK, so here is where I start making excuses about using goto's...
     This is the only place in all of the source you'll find it, but it
     just suits this routine.  I can't help it... I need help!  Someone
     save me!  ARRGGGHHHHHHH!!!!!!
  */

  bblinedone = FALSE;
  TextPtr = TextBuf;

  if (bEnh)  /* currently in enhanced mode */
  {
    if (bLineNumbers && !bWrapping)
    /* Do not print line numbers enhanced */
    {
      StringOut("0 F\n");       /* reset enhanced font */
      strncpy(MiscBuf,TextBuf,LineNumberWidth+1); /* copy line nos & space */
      MiscBuf[LineNumberWidth+1]=0; /* terminate string */
      OUTLINE(MiscBuf);     /* print it */
      bblinedone = TRUE;    /* remeber that we have already done co-ords */
      TextPtr = TextBuf + LineNumberWidth+1;  /* set ptr past line nos */
      StringOut("1 F\n");       /* set back to enhanced font */
    }

    end = FindEndEnh(TextPtr,0);
    if (end == -1)
    {                /* if no end seq, print out rest of line & exit */
      OUTLINE(TextPtr);
      return;
    }
    /* if we get to here, then an end seq was found */
    strncpy(MiscBuf,TextPtr,end);  /* copy up to & inc end seq */
    MiscBuf[end]=0; /* terminate string */
    TextPtr += end;
    OUTLINE(MiscBuf);
    goto EndEnh;   /* do what we do at end of enhanced */
  }
  else   /* Not currently in enhanced mode */
  {
    start = FindStartEnh(TextPtr);

    if (start == -1)
    {
      OUTLINE(TextPtr); /* no start seq, so just  process as normal */
      return;
    }

    strncpy(MiscBuf,TextPtr,start);  /* copy up to start seq */
    MiscBuf[start]=0;     /* terminate string */
    OUTLINE(MiscBuf);     /* and print with co-ords */
    TextPtr += start;     /* repos ptr up to start seq. */
StartEnh:
    bEnh = TRUE;          /* turn on enhanced mode */
    StringOut("1 F\n");   /* turn on enhanced font */

    if (*ActEndSeq) /* multi line spanning start enhance seq */
    {
      end = FindEndEnh(TextPtr,SkipStartSeq);
      if (end == -1)
      {                /* if no end seq, print out rest of line & exit */
        OUTLINE(TextPtr);
        return;
      }
      else
      {                /* there is a end seq */
        strncpy(MiscBuf,TextPtr,end);
        MiscBuf[end]=0; /* terminate string */
        OUTLINE(MiscBuf);        /* print up to and including end seq */
        TextPtr += end;        /* repos ptr past end seq */
EndEnh:
        bEnh = FALSE;          /* reset enhanced mode */
        StringOut("0 F\n");     /* reset enhanced font */

        start = FindStartEnh(TextPtr);  /* any more start seq on this line? */

        if (start == -1)
        {       /* no further start seq, so just dump the rest of the line */
          OUTLINE(TextPtr);
          return;
        }
        else
        {       /* another start seq */
          strncpy(MiscBuf,TextPtr,start);
          MiscBuf[start]=0;   /* terminate string */
          OUTLINE(MiscBuf);   /* print out up to start seq */
          TextPtr += start;
          goto StartEnh;
        }
      }
    }
    else  /* just enhanced to end of line */
    {
      OUTLINE(TextPtr);
      if (!bwrap)
      {  /* we are wrapping so not yet end of line */
        bEnh = FALSE;       /* reset enhanced at end of line */
        StringOut("0 F\n"); /* reset enhanced font */
      }
      return;
    }
  }
}


/*****************************************************************************/
/*  Function DoPage()                                                        */
/*                                                                           */
/*    Perform what is necessary for a new page or shift of column and make   */
/*    decisions whether to print page or not                                 */
/*****************************************************************************/
void DoPage(void)
{

#ifdef FUNCS
fprintf(Display,"->DoPage\n");
#endif

  PageNo_Doc++;                  /* another page within document */
  if (bRanging && PageNo_PS > HighPage)
  {                              /* if we are beyond the end of print range */
    bSkipToEnd = TRUE;           /* set the flag to skip the rest of the file */
    return;
  }
  LineNoOnPage = 0;                        /* incremented to 1 in FlushLine() */

  if ((bTwoCol) && (Column == 1) && PageNo_Doc > 1)/* if 2 cols and have fill */
  {                                                /* 1st col and this is not */
                                                   /* the initial page */
    Column = 2;                            /* we set up x for col 2     */

    /* start of right col |LM|width|RM|LM|width|RM| */
    /*                                           */
    /*   left edge of paper    start col 2        right edge */

    XLine = ((long)PInfo.Left * 2) + Width + (long)PInfo.Right;

    /* to segment doc pages within PostScript pages */
    if (bPrintPage)
    { /* only output this if we are printing the current page */
      sprintf(MiscBuf
              ,"%s %u %u\n/LeftMarg %d def\n"
              ,SP
              ,PageNo_Doc
              ,PageNo_PS
              ,XLine
             );
      StringOut(MiscBuf);
      if (bBoxPage && !bGaudyMode)
      { /* box the page - built-in to gaudy, so if not gaudy we need to do it */
        StringOut("BoxPage\n");
      }
      if (bShade)
      {
        StringOut("ShadePage\n");
      }
    }
  }
  else  /* new PostScript page */
  {
    Column = 1;             /* Else we have just filled Column 2   */
    PageNo_PS++;            /* and increment the page no */

    if ((bSwap)  && (PageNo_PS%2 == SwapEO))
    {
      bSwapThisPage = TRUE;   /* easier than re-using above line several times */
      strcpy(SwapText,"true");
    }
    else
    {
      bSwapThisPage = FALSE;  /* as above */
      strcpy(SwapText,"false");
    }

    if (bSwapThisPage)        /* swap page */
    {
      if (!bLandscape)          /* portrait */
      {
        XLine = PInfo.Right;    /* swap l/r margins */
        BottomLine = PInfo.Bot; /* do not swap top/bot */
        THeightMod = 0;         /* no modifier to Text Height */
      }
      else                      /* landscape */
      {
        XLine = PInfo.Left;     /* do not swap l/r margins */
        BottomLine = PInfo.Top; /* swap top/bot margins */
        /* if margin swapping in landscape mode */
        /* we need to swap top & bot by adding  */
        /* top and subtracting bottom margins   */
        THeightMod = (long)PInfo.Top - (long)PInfo.Bot; 
      }
    }
    else                      /* no swapping */
    {
      XLine = PInfo.Left;     /* so we reset XLine and issue newpage */
      BottomLine = PInfo.Bot; /* put bot/top back to normal */
      THeightMod = 0;         /* no modifier to Text Height */
    }

    if (!bRanging && !bPrint2nd)
    {                       /* printing all pages so just do */
      if (bStartPageIssued)
        EndPageF();          /* end page only after 1st page started */
      StartPageF();
    }
    else                    /* print only every second page */
    {
      if ( (bPrintPage == TRUE) && /* the prev page was printed */
           (bStartPageIssued) )       /* & we have printed a page */
      {
        EndPageF();            /* so terminate it correctly */
      }
      /* now determine whether we print the next page */
      if (bRanging)
      {                       /* check that page is within spec range */
        Range = Range_1;
        bPrintPage = FALSE;   /* will end up true if in spec range */
        while (Range)
        {
          if ( (PageNo_PS >= Range->StartPage) &&
               (PageNo_PS <= Range->EndPage)
             )                /* it is within spec range */
          {
            bPrintPage = TRUE;
            break;            /* break out of the while loop */
          }
          Range = Range->NextRange; /* not in this one so check next range */
        } /* end while */
      } /* end Ranging */
      else
      { /* no ranging specified so print all pages */
        bPrintPage = TRUE;
      }
      /* now bPrintPage shows print status not considering odd/even
         lets check out that now
      */
      if (bPrintPage)
      {
        if ((bPrint2nd) && (PageNo_PS%2 != EvenOrOdd) )
        {
          bPrintPage = FALSE;  /* does not match the odd/even criteria */
        }
      }
      if (bPrintPage)
      {
        StartPageF();
      }
      else
      {
        if (!bQuiet)
        {
          SHOW(Display,"\rSkipping   Page %-5d ",PageNo_PS);
        }
      }
    }
  }

  YLine = THeight + THeightMod;   /* reset to top of text area */

}


/*****************************************************************************/
/*  Function DoProlog()                                                      */
/*                                                                           */
/*    Open the final Output file and output necessary prn prep and prolog    */
/*****************************************************************************/
void DoProlog(void)
{

  unsigned numchar;
  char *prnprepextent = {"ppf"}; /* extent of printer preparation file */
  FILE *PRN;
  register int i;


#ifdef FUNCS
fprintf(Display,"->DoProlog\n");
#endif

  /* open final output file for write */
  /* this will delete the file if it  */
  /* already exists                   */

  if (OutputFP != stdout)  /* if output is not to stdout, it is to a file */
  {                 /* or, if unix, a printer possibly */

#ifdef UNIX
    if (*Printer)     /* if we are piping to a printer */
    {
# if defined (HP_UX) || defined (SOLARIS24)
      sprintf(PrnCmd,"%s%s -s",PipeCmd,Printer);
# else
      sprintf(PrnCmd,"%s%s -Jpcpsv%s_output",PipeCmd,Printer,VERS);
# endif
      OutputFP = (FILE *)popen(PrnCmd,"w");
      Pipe = OutputFP;    /* so we remember to close in MyExit() */
    }
    else            /* we are outputting to a file */
#endif

    if ( (OutputFP = fopen(OName,"wb")) == NULL )
    {
      SHOW(Display
           ,"%sCan not open %s %s for output\n"
           ,VerString
           ,ArrowR
           ,OName
          );
      MyExit(1,NOHELP);
    }
    FileOut = OutputFP;     /* until we open the temporary file if needed */
  }

  while (*PPFName)          /* if not disabled */
  {                         /* go find the printer preparation file */
    if (PPFName[0] != '$')
    {
      /* open user defined ppf file */
      if ( (PRN = fopen(PPFName,"rb")) == NULL)
      {
        /* couldn't find user file so tell them then exit */
        SHOW(Display
             ,"%sPrinter Preparation File %s not found.\n"
             ,VerString
             ,PPFName
            );
        MyExit(1,HELP);
      }
    }
    else
    {
      /* go looking for default */
      if ( (PRN = FindFile(prnprepextent)) == NULL) break;
    }

    for (;;)
    {
      numchar = fread(MiscBuf,sizeof(char),sizeof(MiscBuf),PRN); /* read */
      fwrite(MiscBuf,1,numchar,OutputFP);                        /* write */
      if (numchar < sizeof(MiscBuf)) break;     /* means end of read file */
    }
    fclose(PRN);
    break;
  }

  /* download extended error handler if required */
  if (bExtendErr)
  {
    for(i=0;Error_Handler[i];i++)
    {
      fwrite(Error_Handler[i],1,strlen(Error_Handler[i]),OutputFP);
    }
  }

  /* Copy POSTCRIPT PROLOG to final output File */

  /* send a control-D if required */
  if (bControlD) fprintf(OutputFP,"%c\n",CTLD);

  /* first the variable part */
  fprintf(OutputFP
          ,"%%!PS-Adobe-1.0\n%%%%Creator: %s[%s@%s]\n%%%%CreationDate: %s"
           "%%%%Title: PCPS Document\n%%%%EndComments\nsave /PCPSJob exch def\n"
          ,VerString
          ,UserID
          ,Host
          ,ctime(&Elapsedtime)
         );
  if (bPSTiming)
  { /* Output timing routine */
    fwrite(PSTiming,1,strlen(PSTiming),OutputFP);
  }

  /* then the fixed part */
  for(i=0;Prolog[i];i++)
  {
    fwrite(Prolog[i],1,strlen(Prolog[i]),OutputFP);
  }

  /* now the correct function and array for re-encoding */
  if (bISO8859 && !bIBMPCFont)
  {
    for(i=0;ReEncISO[i];i++)
    {
      fwrite(ReEncISO[i],1,strlen(ReEncISO[i]),OutputFP);
    }
  }
  else
  {
    fwrite(ReEncEuro,1,strlen(ReEncEuro),OutputFP);
  }

  if (bGaudyMode)  /* if we need the gaudy banner */
  {                /* define the gaudy functions  */
    fwrite(Gaudy,1,strlen(Gaudy),OutputFP);
  }

  fprintf(OutputFP,"end\nStartPCPSDoc\n");

  if (bIBMPCFont)
  {
    FontPrefix[NORMAL][0] = '\0';  /* no prefix as full font is defined */
    for (i=0; IBMPCFont[i]; i++)/* write the chunks of IBMPCFont - MSC */
    {                   /* requires strings to be less than 2048 chars */
      fwrite(IBMPCFont[i],1,strlen(IBMPCFont[i]),OutputFP);
    }
  }

  if (bPSTiming)
    fprintf(OutputFP,"(End of Prolog) prtime\n");

  fprintf(OutputFP,"%%%%EndProlog\n");

  /* now do setup for document */
  fprintf(OutputFP,"%%%%BeginSetup\n");

  for (i=0;i<3;i++)
  {
    if (strcmp(Font[i],IBMF))
    {
      strcpy(FontPrefix[i],"PCPS"); /* prefix name with PCPS to show re-encode */
      fprintf(OutputFP
              ,"%s /PCPS%s /%s %s\n"
              ,ReEncArray
              ,Font[i]
              ,Font[i]
              ,ReEncFunc
             );
    }
  }

  if (bGaudyMode)
  {            /* this font is required for the gaudy banner */

    fprintf(OutputFP
            ,"%s /PCPSTimes-Roman /Times-Roman %s\n"
            ,ReEncArray
            ,ReEncFunc
           );
  }

  if (Copies > 1) fprintf(OutputFP,"/#copies %d def\n",Copies);

  fprintf(OutputFP
          ,"/TMStr (%s) def\n%s DM\n/DanishFix %s def\n/PrintCZ %s def\n"
          ,TMString
          ,bDuplex ? (bTumble ? "true true" : "false true") : "false false"
          ,bDanishFix ? "true" : "false"
          ,bPrintCZ ? "true" : "false"
         );

  if (bBackText)
  {
    /* approx 11" to work with (792 points).  Multiply this by some arbitrary
    ** number to give 1200 which, when Divided by the number of characters to
    ** print, gives the font size that will fit in  :-)
    */
    BackFontSize = 1200 / strlen(BackText);
    if (BackFontSize > 200) BackFontSize = 200; /* limit font size to 200 pt */
    /* Multiply the size by 20 as we now do it after the scale */
    BackFontSize *= 20;
    ConvertChar(BackText);                 /* make sure it's good to print */
    fprintf(OutputFP
            ,"/BackText (%s) def /BackSize %u def\n"
             "%s /PCPSTimes-Bold /Times-Bold %s\n"
            ,BackText
            ,BackFontSize
            ,ReEncArray
            ,ReEncFunc
           );
  }

  fprintf(OutputFP,"/LandMode ");
  if (bLandscape)
  {
    fprintf(OutputFP,"true def\n/Landscape{90 rotate 0 -%u translate}def\n"
                    ,PInfo.PaperH);
  }
  else
  {
    fprintf(OutputFP,"false def\n");
  }

  if (bShade)
  {
    fprintf(OutputFP,"/Shade .%.2u def\n/NLine %u def\n",Shade,ShadeLines);
  }

  /* 2up and 4up stuff */

  /* for 4 up, do not use 2 up scale */
  if (PagesUp == 4) strcpy(PInfo.ScaleUp,ALL_4UP);

  fprintf(OutputFP
          ,"/PgsUp %d def\n/CurPg 1 def\n/Scale %s def\n/PgHeight %ld def\n"
           "/PgWidth %ld def\n/UpLines %s def\n"
          ,PagesUp
          ,PInfo.ScaleUp
          ,PInfo.PaperH
          ,PInfo.PaperW
          ,(bUpLines?"true":"false")
         );

  fprintf(OutputFP
          ,"/TMFSize %u def\n"
           "3 TMFSize /Helvetica\n"
           "2 200 /%s%s\n"
           "1 %u /%s%s\n"
           "0 %u /%s%s\n"
           "4 SetUpFonts\n"
          ,(IDFontSizeBy10 * 2)
          ,FontPrefix[HEADER]
          ,Font[HEADER]
          ,(FontSizeBy10 * 2)
          ,FontPrefix[ENHANCED]
          ,Font[ENHANCED]
          ,(FontSizeBy10 * 2)
          ,FontPrefix[NORMAL]
          ,Font[NORMAL]
         );

  /* send out the gaudy stuff */
  if (bBannerOn && bGaudyMode)
  {
    fprintf(OutputFP
            ,"%s %s GaudyInit\n"
            ,bTwoCol ? "true" : "false"
            ,bBoxPage ? "true" : "false"
           );
  }

  fprintf(OutputFP,"/Flush %s def\n",bFlushPage?"true":"false");

  if (bPSTiming)
  {
    fprintf(OutputFP,"(End of Setup) prtime\n");
  }

  fprintf(OutputFP,"%%%%EndSetup\n");

  bPrologDone = TRUE;                               /* so only done once */
}

/*****************************************************************************/
/*  Function DoStartup()                                                     */
/*                                                                           */
/*    Check how many rows on screen, set up Verstring and look for help      */
/*    If required, get the userid and host name for the TM marking and get   */
/*    the current time.                                                      */
/*****************************************************************************/
void DoStartup(unsigned no_args)
{

#ifdef __TURBOC__

  char far *pc;

  pc = MK_FP(0x0040,0x0084); /* pointer to memory location with # rows - 1 */
  Rows = (*pc)+1;

#else

# ifdef MSC

  VIOMODEINFO vmode;

  vmode.cb = sizeof(vmode);
  VioGetMode(&vmode,0);      /* for family mode */
  Rows = vmode.row;          /* the number of rows on screen */

# else
#  ifdef MSVC32

  CONSOLE_SCREEN_BUFFER_INFO bufinf;
  
  if (GetConsoleScreenBufferInfo
      (GetStdHandle(STD_OUTPUT_HANDLE),&bufinf) != TRUE)
  {
	  Rows = 25;	/* hey, if we can't get an answer, assume 25 lines */
  }
  else
  {
	  Rows = bufinf.dwSize.Y;
  }

#  else /* UNIX */

  (void) signal(SIGINT, Interrupt);  /* catch interrupt */
  (void) signal(SIGQUIT, Interrupt); /* catch quit */
  (void) signal(SIGSYS, BadSys);     /* catch bad system calls */
  (void) signal(SIGPIPE, BadPipe);   /* catch bad pipe call Eg. popen */

#  endif

# endif

#endif

#ifdef FUNCS
fprintf(Display,"->DoStartup\n");
#endif

  if (
     ((TextBuf        = malloc(1024)) == NULL)                  ||
     ((MiscBuf        = malloc(1024)) == NULL)                  ||
     ((StreamBuf      = malloc(4096)) == NULL)
     )
  {  /* not enough memory */
    SHOW(Display,"%sNot enough free memory available!\n",VerString);
    MyExit(1,NOHELP);
  }
  TextPtr = TextBuf;    /* initialise the pointer to the output buffer */

  sprintf(VerString,"PCPS V%s:",VERS); /* Used in error outputs */

  Stdin_Tty = isatty(fileno(stdin));
  Stdout_Tty = isatty(fileno(stdout));

  if ( (no_args == 1) && Stdin_Tty) /* no args and no stdin */
  {
    ShowUsage();
  }

  if (!Stdout_Tty)
  {                     /* output piped to other program */
    Display = stderr;   /* where to display the messages */
    OutputFP = stdout;  /* where to send the PostScript */
    strcpy(OName,"stdout");
#ifdef DEBUG1
    fprintf(stderr,"Output piped to stdout...\n");
#endif
  }
  else
  {
    Display = stdout;
  }

  time(&Elapsedtime);     /* get current time */
}

/*****************************************************************************/
/*  Function EndPageF()                                                       */
/*                                                                           */
/*    Performs and end of page sequence                                      */
/*****************************************************************************/
void EndPageF(void)
{

#ifdef FUNCS
  fprintf(Display,"->EndPageF\n");
#endif
#ifdef DEBUG1
  if (bMemSort) fprintf(Display,"BlockPoint = %p\n",BlockPoint);
#endif

  bStartPageIssued = FALSE;   /* page is now ended */
  StringOut("EndPage\n");

  if ( (bManualFeedPage1 && PageNo_PS == 2) || /* man feed off after 1st pg */
       (bManualFeed)                       /* or man feed through whole doc */
     )
  {
    StringOut("ME\n");
  }
  if (bPrintInv && !bMemSort && !bNewRev)
  {                             /* only set flag if using temporary file */
    strcpy(MiscBuf,EPAGE);      /* and old sorting method */
    StringOut(MiscBuf);
  }
}

/*****************************************************************************/
/*  Function FileLoop()                                                      */
/*                                                                           */
/*    This is where we  open files and commence the processing of the text   */
/*****************************************************************************/
void FileLoop(unsigned ArgTotal,char *Arguments[])
{

  static char *fn;

#ifdef FUNCS
fprintf(Display,"->FileLoop\n");
#endif

  /* We have the arguments (if any) and now 'ArgNum++' is pointing
     to the first input file   */

  if (ArgNum == ArgTotal) /* no input file spec so must be stdin */
  {
    GetCurrentDate();
    ProcessFile(0);
  }
  else
  {
    while (ArgNum < ArgTotal) /* Process the input files one at a time */
    {
      fn = Arguments[ArgNum++];
#ifndef UNIX
      if (MyFindFirst(fn))
      {
        do
        {
          ProcessFile(FndFile.fname);
        } while (MyFindNext());
      }
#else
      if (GetStat(fn))
      {
        ProcessFile(FndFile.fname);
      }
#endif
      else
      {
        SHOW(Display,"%sNo file matching \"%s\"\n",VerString,fn);
        continue;  /* Can't find this input file so try the next */
      }
    }
  }

  if (bPrologDone)   /* if we did any processing */
  {
    if (bPSTiming)
      fprintf(OutputFP,"(End of PCPS job) prtime\n");

    fprintf(OutputFP,"%%%%Trailer\nEndPCPSDoc\nPCPSJob restore\n%%%%EOF\n");

    /* send a control-D if required */
    if (bControlD) fputc(CTLD,OutputFP);
    fclose(OutputFP);
  }

  SHOW(Display,"\n");

}

/*****************************************************************************/
/*  Function FindFile()                                                      */
/*                                                                           */
/*    Look for PCPS.fileextent - first check env vars, then current dir then */
/*    finally the origin directory of PCPS                                   */
/*****************************************************************************/
FILE * FindFile(char *fileextent)
{
  FILE *FP1;
  char *cp,
       filename[13],
       filepath[MAX_PATH],
       fext[4];
  int  len;

#ifdef FUNCS
fprintf(Display,"->FindFile\n");
#endif

  strlwr(strcpy(fext,fileextent));
  sprintf(filename,"pcps.%s",fext);          /* file name we are looking for */
  sprintf(filepath,"PCPS%s",fext);           /* env var we are looking for */
  strupr(filepath);
  if ((cp = getenv(filepath)) != NULL)                    /* get env variable */
  {                                                     /* env variable found */
    strcpy(filepath,cp);
    len = strlen(filepath);
    if (filepath[len-1] != PD)
    {
      filepath[len++] = PD;
      filepath[len] = '\0';
    }
    strcat(filepath,filename);
  }
  else  /* no environment variable */
  {
    *filepath = '\0';   /* reset the filepath so fopen fails */
  }
  if (
      !(*filepath)                            /* if no env variable */
      ||                                              /* or */
      ((FP1 = fopen(filepath,"rb")) == NULL)  /* open failed */
     )
  {
    sprintf(filepath,filename);
    if ((FP1 = fopen(filepath,"rb")) == NULL)      /* try current directory */
    {                                                   /* not there so ... */
      strcpy(filepath,Origin);                      /* check the origin dir */
      strcat(filepath,filename);                    /* tack on desired file */
      FP1 = fopen(filepath,"rb");                 /* origin dir of PCPS.EXE */
    }
  }

#ifdef DEBUG1
  if (FP1) fprintf(Display,
                   "File Found: Ext \"%s\" at %s\n",
                   fext,filepath);
#endif
  return(FP1);                            /* will return file pointer or NULL */
}

/*****************************************************************************/
/*  Function FindEndEnh()                                                    */
/*                                                                           */
/*    Find the end sequence for enhanced print in string, returning the      */
/*    offset from the end of the string.                                   */
/*****************************************************************************/
int FindEndEnh(char *textptr, unsigned jump)
{
  static char *loc;
  static int offset;

  textptr += jump;                  /* skip over start chars */
  loc = strstr(textptr,ActEndSeq);
  if (loc != NULL)
  {
    offset = (int)(loc - textptr);  /* offset to start of end seq */
    offset += jump;                 /* plus the chars we jumped at start */
    offset += strlen(ActEndSeq);  /* plus length of end seq = end of end seq */
  }
  else
  {
    offset = -1;
  }
  return(offset);
}

/*****************************************************************************/
/*  Function FindStartEnh()                                                  */
/*                                                                           */
/*    Find a start sequence for enhanced print in string, copy end sequence  */
/*    into ActEndSeq, and return count from start of string to start seq.    */
/*****************************************************************************/
int FindStartEnh(char *textptr)
{
  static EnhChar *act;
  static char *loc1, *loc2;
  static int offset;

  offset = -1;
  loc2 = NULL;
  for (act = ActEnh; act->StartSeq[0]; act++)
  {
    loc1 = strstr(textptr,act->StartSeq);
    if (loc1 != NULL)
    {
      if (loc2)
      {
        if (loc2 < loc1) continue;
      }
      strcpy(ActEndSeq,act->EndSeq);
      SkipStartSeq = strlen(act->StartSeq);
      offset = (int)(loc1 - textptr);
      loc2 = loc1;
    }
  }
  
  /* For Fortran, only check for comment character in column 1 and 2
     1 is used in .f files
     2 is used in .L files
     If it was not found in one of those columns ignore.
  */
  if (Mode == M_FORT)
  {
     if (offset != 0 && offset != 1)
     {
        offset = -1;
     }
  }

  return(offset);
}

/*****************************************************************************/
/*  Function FlushLine()                                                     */
/*                                                                           */
/*    flush current line inserting line #'s start and end line sequence      */
/*****************************************************************************/
void FlushLine(BOOL bwrap)
{
  if (!bWasteRest)  /* if we're not throwing away the rest of this line */
  {
    if ((YLine <= (long)BottomLine) || (LineNoOnPage >= LinesPerPage))
    { /* at end of page or Column */
      DoPage();
      if (bSkipToEnd) return;
      bFFJustSent = TRUE;
    }

    LineNoOnPage++;              /* increment line number on this page */

    *TextPtr = '\0';

    if (Mode == M_MAIL)
    {
      CheckMailHeader(bwrap);
    }

    if (bLineNumbers)
    {
      InsertLineNumbers();
    }

    if (Mode > 2)
    {
      DoEnhancedPrint(bwrap);
    }
    else
    {
      BorWLine(TextBuf);
    }
    YLine -= YDecrement; /* reposition line pointer */
  }

  TextPtr = TextBuf;   /* reset pointer back to start of line */
  LinePos = bLineNumbers ? LineNumberWidth + 2 : 0;
  if ((bWrapping = bwrap) == FALSE)
  {
    bWasteRest = FALSE;   /* reset cut indicator */
  }
}

/*****************************************************************************/
/*  Function GetArgs()                                                       */
/*                                                                           */
/*    Get args from adm file, then cfg file then parse command line.         */
/*****************************************************************************/
void GetArgs(unsigned number,char *arguments[])
{
  char *configextent = {"cfg"},
       filepath[MAX_PATH],
       *prn;

  FILE *cfg;

#ifdef FUNCS
fprintf(Display,"->GetArgs\n");
#endif

  /* look for an admin config file in the origin directory */
  if (*Origin)   /* only look if we were able to find the origin path */
  {
    strcpy(filepath,Origin);
    strcat(filepath,"pcps.adm");
    if ( (cfg = fopen(filepath,"rb")) != NULL)
    {  /* there is one present */
#ifdef DEBUG1
fprintf(stderr,"Admin config file found at %s.\n",filepath);
#endif
      ParseConfig(cfg);
    }
  }

  /* Now look for PRINTER environment variable */
  if ( (prn = getenv("PRINTER")) != NULL)
  {
#ifdef UNIX
    strcpy(Printer,prn);
#else
    strcpy(OName,prn);
#endif
  }

  /* look for a config file */

  if ( (cfg = FindFile(configextent)) != NULL)
  {
    /* OK so we found the file, now lets get the arguments */
    ParseConfig(cfg);
  }

  ArgNum = 1;
  while (ArgNum <= (number-1))
  {
    strcpy(Argum,arguments[ArgNum]); /* Fetch argument */
    if (Argum[0] == '-') /* if it isn't preceded by "-" then not arg */
    {
      ProcessArg(Argum);
    }
    else break;
    ArgNum++;
  }
  if (ArgNum == number) /* if we have used up all args, then no input spec */
  {
    if (Stdin_Tty)
    {
      SHOW(Display,
              "%sNo files specified and no stdin available.\n",
              VerString);
      MyExit(1,HELP);
    }
    else
    {
      /* set up some defaults for stdin */
      bMemSortAllow = FALSE;
      bDeleteInput = FALSE;
    }
  }

}

/*****************************************************************************/
/*  Function GetCurrentDate()                                                */
/*                                                                           */
/*    Get the current date and time to plug into the fndfile struct - this   */
/*     is for stdin processing.                                              */
/*****************************************************************************/
void GetCurrentDate(void)
{
  Timeptr = localtime(&Elapsedtime);
  FndFile.fsecs = Timeptr->tm_sec;
  FndFile.fmins = Timeptr->tm_min;
  FndFile.fhour = Timeptr->tm_hour;
  FndFile.fday  = Timeptr->tm_mday;
  FndFile.fmonth = Timeptr->tm_mon + 1;
  FndFile.fyear = Timeptr->tm_year + 1900;
}

/*****************************************************************************/
/*  Function GetLastNum                                                      */
/*                                                                           */
/*    Find the last number in a string and return it as an integer           */
/*****************************************************************************/
int GetLastNum(char *given)
{
  given = strrchr(given,' ') + 1;     /* pos at last space + 1 */
  return(atoi(given));                /* convert to int and return */
}

/*****************************************************************************/
/*  Function GetNum()                                                        */
/*                                                                           */
/*    take the characters offset bytes into argument and return it           */
/*    converted to int                                                       */
/*****************************************************************************/
unsigned GetNum(char argument[], unsigned offset)
{
  register unsigned a,b;
  static char scratchstr[12];

#ifdef FUNCS
fprintf(Display,"->GetNum\n");
#endif

  for (a = 0, b= offset; b < strlen(argument) ; a++, b++)
  {
    scratchstr[a] = argument[b];  /* copy string */
  }
  scratchstr[a] = '\0';        /* terminate the string */
  return(atoi(scratchstr));    /* return number */
}

/*****************************************************************************/
/*  Function GetReady()                                                      */
/*                                                                           */
/*    Calculate some useful parameters such as max line length and banners,  */
/*    get the userid and host name, and expand any ##id fields.              */
/*****************************************************************************/
void GetReady(void)
{
  unsigned long height;

#ifdef FUNCS
fprintf(Display,"->GetReady\n");
#endif

  LoadFonts();    /* load the correct font names into Font[] array */

  /* OK, PaperH and PaperW are currently set for Portrait - if Landscape
     we must swap them around */
  if (bLandscape)
  {
    height = PInfo.PaperH;
    PInfo.PaperH = PInfo.PaperW;
    PInfo.PaperW = height;
  }

  /* lets set margins and calculate the maxline length */

  if (NewLeft) PInfo.Left = NewLeft;
  else         PInfo.Left = bLandscape ? PInfo.LeftLM : PInfo.LeftPM;
  if (NewRight) PInfo.Right = NewRight;
  else         PInfo.Right = bLandscape ? PInfo.RightLM : PInfo.RightPM;
  if (NewTop) PInfo.Top = NewTop;
  else         PInfo.Top = bLandscape ? PInfo.TopLM : PInfo.TopPM;
  if (NewBot) PInfo.Bot = NewBot;
  else         PInfo.Bot = bLandscape ? PInfo.BotLM : PInfo.BotPM;

  GWidth = PInfo.PaperW - (PInfo.Left + PInfo.Right);  /* for gaudy banner */
  GHeight = PInfo.PaperH - PInfo.Top;                  /* for gaudy banner */
  Height = GHeight - PInfo.Bot;                        /* print height */
  if (bTwoCol)
  {
    /* the printable width of each column */
    Width = (PInfo.PaperW / 2) - (PInfo.Left + PInfo.Right);
  }
  else
  {
    Width = GWidth;
  }

  if (!bGaudyMode)          /* GHeight points to top of banner, so for non */
  {                         /* gaudy banner, set to bottom left of banner  */
    GHeight -= FontSizeBy10*2; /* so that top of letters are below top margin */
  }
  /* lets calculate the bannergap which is the space that the banner
     uses plus a gap after it before the text
  */
  if (!bBannerOn)
  {
    BannerGap = 0;  /* no gap for no banner */
  }
  else
  {
    BannerGap = FontSizeBy10*24/10 + (bGaudyMode ? 720 : 80);
  }

  THeight = GHeight - BannerGap;

  if (!bLineLimitSpecified)  /* calculate ll if not already specified */
  {
    /* OK, so this is how we calculate the maximum line length:  we are
       using units of 1/1440 inch ( points (1/72inch) divided by 20).
       If a font size is 10 point, it means that each character (assuming
       non-proportional font courier) is 10 points high (including inter-
       -line gap) and 6 points wide.  So the width of each character in our
       units as stated above is FontSizeBy10 * .6 * 20 /10, or
       FontSizeBy10 * 12 / 10.
       So divide the printable width by this value to get the maxlinelength.
    */

    /* chars before wrap/trunc */
    MaxLineLength = (unsigned)Width/(FontSizeBy10 * 12 / 10);
  }

  /* FontSizeBy10   LineSpace
     ------------ x --------- x 20(working in 20th of points) =
         10           100
  */
  YDecrement = (long)FontSizeBy10 * (long)LineSpace / 50;

  GetName();

  sprintf(IDString,"%s@%s",UserID,Host);

  *TMString = '\0';
  if (bPrintName)
  {
    sprintf(TMString,"[ %s ]",IDString);
  }
  if (!bNoSig)
  {
    strcat(TMString,VerString);
    strcat(TMString,ANNOY);
  }

  /* now expand any ##id fields in text from -ga, -g1, -fn and/or -wm */

  if (*FileComment)
  {
    CopyID(FileComment);
  }

  if (bFNSpecified)
  {
    CopyID(AltFileName);
  }

  if (bBackText)
  {
    CopyID(BackText);
  }

  /* to call the correct PS print routine on each line of output */
  sprintf(LineStr,")%c\n", (Mode==M_MAN) ? 'W' : 'B');

  /* set up correct re-encode array string */
  if (bISO8859 && !bIBMPCFont)
  {
    strcpy(ReEncArray,ISOARRAY);
    strcpy(ReEncFunc,ISOFUNC);
  }
  else
  {
    strcpy(ReEncArray,EUROARRAY);
    strcpy(ReEncFunc,EUROFUNC);
  }


}

/*****************************************************************************/
/*  Function GetSortMemory()                                                 */
/*                                                                           */
/*    Allocate some memory for sorting the pages if enough free              */
/*****************************************************************************/
void GetSortMemory(long SourceSize)
{

  if (SourceSize > 5000)
    MemReq = SourceSize * 2; /* good size?? */
  else
    MemReq = 10000;  /* if input file is small and Banner is on, the */
                     /* Banner may be larger than the file itself so */
                     /* allocate a minimum of 10000 bytes            */
  if(bMemSortAllow)  /* if the user has not disabled */
  {                  /* allocate the required memory */
#ifdef __TURBOC__
    if( (Block = farmalloc(MemReq)) != NULL)
#else
# ifdef MSC
    if( (Block = (char huge *)halloc(MemReq,1)) != NULL)
# else /* UNIX */
    if( (Block = (char *)malloc(MemReq)) != NULL)
# endif
#endif
    {
#ifdef DEBUG1
      fprintf(Display,"Memsort true %lu bytes allocated\n",MemReq);
#endif
      bMemSort = TRUE;  /* set memsort flag */
      BlockPoint = Block;  /* set up Block pointer for mem write */
      BlockTop = Block + MemReq;
    }
  }
}

/*****************************************************************************/
/*  Function GetTrays()                                                      */
/*                                                                           */
/*    Parse the -h argument to decide what paper trays the user wants        */
/*****************************************************************************/
BOOL GetTrays(char *Arg)
{
  char *sv,*ev;
  char arg[MAX_PATH];
  unsigned i,j;
  int len;


  if (!Arg) return(ERROR);   /* -h must have at least 1 parameter */

  TraySwPage = 0;   /* initialise */
  strcpy(arg,Arg);  /* so that I can hack without worry */

  for (
        sv = arg, ev = arg, i=0;
        sv != NULL && ev != NULL && i < 3;
        i++
      )
  {
    if ( (ev = strchr(sv,',')) != NULL )
      *ev = '\0';  /* terminate the parameter */

    if ( (len = strlen(sv)) > 2)
      return (ERROR);  /* only 2 char parameters allowed */

    if (
        ( (len == 1) && (i == 0) ) /* 1st param 1 char */
        ||                              /* OR */
        ( (i) && !bTrayName )       /* 1st param was 1 char */
       )
    {

      /* numbered trays */

      bTrayName = FALSE;   /* bin not specifically named */
      if (i != 1)
      {
        switch((char)*sv)
        {
          case '0':
          case '1':
          case '2': if (i)                 /* if 1st tray already specified */
                    {
                      Tray2 = GetNum(sv,0);
                      sprintf(PaperTray2,"%d",Tray2);  /* for reporting */
                    }
                    else                   /* else first tray */
                    {
                      Tray1 = GetNum(sv,0);
                      sprintf(PaperTray1,"%d",Tray1);  /* for reporting */
                    }
                    break;

          default:  return(ERROR);
        }
      }
      else
      {
        if ( (TraySwPage = GetNum(sv,0)) < 1) return(ERROR);
      }
    }
    else
    {

      /* named trays */

      bTrayName = TRUE;   /* bin specifically named */
      if (i != 1)
      {
        for (j=0; j <= NUMPAPERSIZES; j++)
        {
          if (j == NUMPAPERSIZES) return(ERROR);  /* didn't find match */
          if (!strcmp(sv,PaperStats[j].shortname))
          {
            if (i)
            {
              strcpy(PaperTray2,PaperStats[j].trayname);
            }
            else
            {
              strcpy(PaperTray1,PaperStats[j].trayname);
            }
            break;
          }
        }
      }
      else
      {
        if ( (TraySwPage = GetNum(sv,0)) < 1) return(ERROR);
      }
    }
    if (ev) sv = ++ev;   /* go to next param */
  }
  return(0);
}


/*****************************************************************************/
/*  Function InsertBanner()                                                  */
/*                                                                           */
/*    Write the banner to the intermediate output file                       */
/*****************************************************************************/
void InsertBanner(void)
{

  static long ypos;
#ifdef FUNCS
fprintf(Display,"->InsertBanner\n");
#endif

  ypos = GHeight; /* top line for Banner landscape */
  if (bSwapThisPage && bLandscape)
  {                                   /* if margin swapping in landscape mode */
    ypos += (PInfo.Top - PInfo.Bot); /* we need to swap top & bot by adding  */
  }                                   /* top and subtracting bottom margins   */

  if (bGaudyMode)  /* assemble last of gaudy header (page number) */
  {
    sprintf(MiscBuf
            ,"%s%u) %s %d Gaudy\n"
            ,Banner
            ,PageNo_PS
            ,SwapText
            ,ypos
           );
    StringOut(MiscBuf);
  }
  else             /* Assemble non-gaudy header */
  {
    StringOut("2 F\n");
    sprintf(MiscBuf,"%s%u",Banner,PageNo_PS);
    YLine = ypos;
    BorWLine(MiscBuf);
  }
}

/*****************************************************************************/
/*  Function InsertLineNumbers()                                             */
/*                                                                           */
/*    Insert line numbers into the string to be printed (before wrapping)    */
/*****************************************************************************/
void InsertLineNumbers()
{
  register i;
  static char str[15];
  static int len;

  if (bWrapping)   /* if this is the second or nth part of a wrapped line */
  {
    for (i=0; i <= (int)LineNumberWidth; i++)
    {
      MiscBuf[i] = ' ';
    }
    MiscBuf[i] = '\0';
    strcat(MiscBuf,TextBuf);
  }
  else
  {
    CurrentLineNumber++;   /* another line! */
    sprintf(str,"%u",CurrentLineNumber);
    if ( (len = strlen(str)) > (int)LineNumberWidth)
    {
      LineNumberWidth = len;
    }

    for (i = LineNumberWidth - len; len >= 0; len--)
    {
      str[len+i] = str[len];
    }
    while(i)
    {
      i--;
      str[i] = *LineNumberPad;
    }
    sprintf(MiscBuf,"%s %s",str,TextBuf);
  }
  strcpy(TextBuf,MiscBuf);
}

/*****************************************************************************/
/*  Function InsertRange()                                                   */
/*                                                                           */
/*    Take a page range and insert it into the current page ranges           */
/*****************************************************************************/
void InsertRange(unsigned start, unsigned end)
{
#if defined (UNIX) || defined (MSVC32)
  if ( (NewRange = (p_range *)malloc(sizeof(p_range))) == NULL)
#else
  if ( (NewRange = (p_range huge *)malloc(sizeof(p_range))) == NULL)
#endif
  {
    SHOW(Display
         ,"%sCouldn't get memory for linked list.\n"
          "%sYou must be very short of memory - free some up and try again.\n"
         ,VERS
         ,VERS
        );
    MyExit(1,NOHELP);
  }
  NewRange->StartPage = start;             /* set it up */
  NewRange->EndPage = end;
  NewRange->NextRange = NULL;
  if (Range_1)
  { /* now decide where to put it */
    Range = Range_1;
    PRange = NULL;
    while (Range)
    {
      if (start < Range->StartPage)
      {
        NewRange->NextRange = Range;
        if (!PRange)
        {
          Range_1 = NewRange;         /* if prev=NULL, then point
                                         Range_1 = new */
        }
        else
        {
          PRange->NextRange = NewRange; /* set prev to point at new */
        }
        break;                          /* we have inserted it, so quit out */
      }
      PRange = Range;
      Range = Range->NextRange;
    }
    if (!NewRange->NextRange)
    {
      /* if this is still NULL, then this new range slots in at the end of
         the list */
      PRange->NextRange = NewRange;
    }
  }
  else
  {
    Range_1 = NewRange;
  }
}

/*****************************************************************************/
/*  Function IsNum()                                                         */
/*    Return true if char is between 0 and 9.                                */
/*****************************************************************************/
BOOL IsNum(char ch)
{
  if (ch > 0x2f && ch < 0x3a) return TRUE;
  else return FALSE;
}

/*****************************************************************************/
/*  Function LoadFonts()                                                     */
/*                                                                           */
/*    Load font names into Font[] structure, checking for valid names.       */
/*****************************************************************************/
void LoadFonts(void)
{
  register unsigned i;
  unsigned key;
  BOOL bfontfound;

#ifdef FUNCS
fprintf(Display,"->LoadFonts\n");
#endif

  /* lets check out the requested font code */
  if (strlen(strupr(FontCode)) > 0)
                            /* if not >0 then only font size was specified */
  {                         /* so skip finding font and stick with default */
    if ( (Mode == M_MAN) && strncmp(FontCode,"C",1))
    {
      SHOW(Display,
              "\n%sOnly font Courier[Bold] is supported in man page mode\n",
              VerString);

      MyExit(1,HELP);
    }
    bfontfound = FALSE;
    for (i=0; i < NUMBFONTS; i++)
    {
      if (strcmp(FontCode,FontList[i].FontIndex) == 0)
      {
        bfontfound = TRUE;
        strcpy(Font[NORMAL],FontList[i].FontName);
        key = FontList[i].HeadFontNum;
        strcpy(Font[HEADER],FontList[key].FontName);
        key = FontList[i].EnhIndex[EnhPrintFont];
        strcpy(Font[ENHANCED],FontList[key].FontName);
        break;
      }
    }
    if (!bfontfound)
    {
      SHOW(Display
           ,"\n%sFontcode \"%s\" is not valid!\n\n"
            "Refer to the PCPS manual <=|\n"
            "         *OR*              |==> to see valid codes!\n"
            "     Type \"pcps -ls\"     <=|\n"
           ,VerString
           ,FontCode
          );
      MyExit(1,NOHELP);
    }
    /* set flag according to font */
    bIBMPCFont = strcmp(Font[NORMAL],IBMF)?FALSE:TRUE;
  }
}

/*****************************************************************************/
/*  Function MakeBanner()                                                    */
/*                                                                           */
/*     Assemble the Banner if enabled.                                       */
/*****************************************************************************/
void  MakeBanner(void)
{
#ifdef FUNCS
fprintf(Display,"->MakeBanner\n");
#endif

  switch(DateFormat)
  {
    case DEFDATE: sprintf(FileDateString   /* default */
                          ,"%u/%.2u/%.2u"
                          ,FndFile.fyear
                          ,FndFile.fmonth
                          ,FndFile.fday
                         );
                  break;
    case USDATE:  sprintf(FileDateString   /* US */
                          ,"%.2u/%.2u/%.2u"
                          ,FndFile.fmonth
                          ,FndFile.fday
                          ,FndFile.fyear % 100
                         );
                  break;
    case EURDATE: sprintf(FileDateString   /* European */
                          ,"%.2u/%.2u/%.2u"
                          ,FndFile.fday
                          ,FndFile.fmonth
                          ,FndFile.fyear % 100
                         );
                  break;
  }
  sprintf(FileTimeString
          ,"%0.02d:%0.02d:%0.02d"
          ,FndFile.fhour
          ,FndFile.fmins
          ,FndFile.fsecs
         );

  if (bGaudyMode)
  {
    sprintf(Banner
            ,"(%s)(%s)[(%s)(%s)]("
            ,FileComment
            ,(bFNSpecified ? AltFileName : FNamePS)
            ,FileDateString
            ,FileTimeString
           );
  }
  else
  {
    if (bFNSpecified)
    {
      sprintf(Banner,"%s       Page: ",AltFileName);
    }
    else
    {
      sprintf(Banner
              ,"%s               %s   %s   Page: "
              ,FName
              ,FileDateString
              ,FileTimeString
             );
    }
  }
}

/*****************************************************************************/
/*  Function MemReverse()                                                    */
/*                                                                           */
/*     Reverses (Inverses) the order of pages in the final output file       */
/*      if enabled.  Takes input from the memory buffer Block.               */
/*****************************************************************************/
void  MemReverse(void)
{
  register unsigned page;
  unsigned countbytes;
#if defined (UNIX) || defined (MSVC32)
  pptrm *pptrm_old;
#else
  pptrm huge *pptrm_old;
#endif

#ifdef FUNCS
fprintf(Display,"->MemReverse\n");
#endif

  AddPage();     /* to tag the end of the buffer */
  page = PagesPrinted;
  while (PPtrM_curr->prevpageptr)
  {
    if (!bQuiet)
    {
      SHOW(Display,"\rReversing  Page %-5d ",page--);
    }
#ifndef UNIX
    SpinProp();
#endif
    countbytes = (unsigned)
                 (PPtrM_curr->pageptr - PPtrM_curr->prevpageptr->pageptr);

    fwrite((char *)PPtrM_curr->prevpageptr->pageptr
           ,1
           ,countbytes
           ,OutputFP);
    pptrm_old = PPtrM_curr;
    PPtrM_curr = PPtrM_curr->prevpageptr;
    free((char *)pptrm_old);
  }
  free((char *)PPtrM_curr);
  SHOW(Display,"%u Pages ",PageNo_PS);
}

/*****************************************************************************/
/*  Function MyExit()                                                        */
/*                                                                           */
/*    Exit and optionally show how to get help                               */
/*****************************************************************************/
void MyExit(unsigned level,unsigned flag)
{
  if (flag)
  {
    SHOW(Display,"\n%sEnter \"pcps -he\" for help.\n",VerString);
  }
#ifdef UNIX
  if (Pipe) pclose(Pipe);
#endif
  exit(level);
}

/*****************************************************************************/
/*  Function ParseConfig()                                                   */
/*                                                                           */
/*    Read the file cfg and parse the contents as arguments to pcps          */
/*****************************************************************************/
void ParseConfig(FILE *cfg)
{
  char configstring[MAX_PATH];
  register unsigned i,j;
  BOOL bquote_on=FALSE;
  int line=0;

  /*
     This routine has been modified to handle quotes (\") to be
     compatible with the normal command line processing.  If a quote is
     found, all characters until the next quote are considered part of
     the argument.  This can handle either "-arg 1 2 3" or -arg" 1 2 3"
  */
  for (;;)
  {
    /* read a line */
    if (fgets(configstring,sizeof(configstring),cfg) == NULL)
    {
      break;
    }
    line++;
    for (i=0; i < strlen(configstring); i++)
    {
      switch (configstring[i])
      {
        case CTLZ:                    /* ignore DOS EOF and */
        case CR:                      /* ignore all white-space */
        case LF:
        case ' ':
        case '\t':  break;

        case '\"':  bquote_on = TRUE; /* quote before '-' */
                    break;

        case '-':   for (j=0;;j++)    /* argument found */
                    {
                      Argum[j] = configstring[i]; /* copy argument to Argum */
                      if (!configstring[i++]) break;
                      if (configstring[i] == '\"')
                      {
                        if (bquote_on)    /* end of argument */
                        {
                          bquote_on = FALSE;
                          break;          /* process arg */
                        }
                        else              /* begin quote argument */
                        {
                          bquote_on = TRUE;
                          i++;            /* throw away quote */
                          continue;       /* go to next char */
                        }
                      }
                      else
                      {
                        if (
                            (configstring[i] < '!') &&
                            (bquote_on == FALSE)
                           )
                        {            /* break out if unquoted whitespace */
                          break;
                        }
                      }
                    }
                    if (bquote_on)
                    {
                      SHOW(Display,
                              "\nUnbalanced quotes in config file at line"
                              " %d.\n\n-> %s\n",
                              line,
                              configstring);
                      MyExit(1,HELP);
                    }
                    Argum[++j] = '\0';
                    ProcessArg(Argum);                      /* and process the argument */
                    break;

        default:    SHOW(Display,
                            "\nSyntax error in config file on line %d,"
                            " column %d.\n\n-> %s",
                            line,
                            i+1,
                            configstring);
                    for (j=0; j < i+3; j++)
                    {
                      fputc(' ',Display);
                    }
                    SHOW(Display,"^\n");
                    MyExit(1,HELP);
      }
    }                        /* now go back and get any more on this line */
  }
  fclose(cfg);
}

/*****************************************************************************/
/*  Function ParseRanges()                                                   */
/*                                                                           */
/*    Pass the arg specifying page print ranges                              */
/*****************************************************************************/
int ParseRanges(char *spt)
{
  unsigned spage, epage, number;
  register b;
  char scratch[20];

  HighPage = 0;               /* used to find the highest page */
  spage = epage = 0;
  for (;;)
  {
    switch (*spt)
    {
      case '\0':
      case COMMA:                      /* we should have something to process */
        if (!spage) return(1);         /* nothing to process */
        if (!epage) epage = spage;     /* range is only 1 page */
        InsertRange(spage,epage);      /* plug into linked list */
        spage=epage=0;                 /* reset these */
        break;

      case ELAN:                       /* just delimiting a range */
        if (!spage) spage = 1;         /* -# means 1-# */
        break;

      case DOLLAR:                     /* last page - only valid as epage */
        if (!spage) return(1);
        else epage = HighPage = 0xffff;
        break;

      default:                         /* lets assume that its a number */
        for (b=0; *spt; spt++, b++)
        {
          if (isdigit(*spt))
            scratch[b] = *spt;         /* grab number */
          else
            break;
        }
        scratch[b] = '\0';             /* terminate string */
        spt--;                         /* this will be inc'd back at top */
        if (b > 0)                     /* there was a number there */
        {
          number = GetNum(scratch,0);
          if (number > HighPage)
            HighPage = number;         /* store high page */
          if (!spage)
            spage = number;
          else
            if ( (epage = number) < spage)
              return(1);
        }
        else return(1);               /* wasn't legal character */
    } /* end switch */
    if (!*spt) break;                  /* exit when at end of arg */
    spt++;                             /* inc the pointer */
  } /* end for */
  return(0);
}

/*****************************************************************************/
/*  Function ParseShade()                                                    */
/*                                                                           */
/*    Parse the -sh argument for shade ot line settings                      */
/*****************************************************************************/
int ParseShade(char *argum)
{
  char buf[5];
  register int i;

  for (;*argum;)
  {
    switch (*argum)
    {
      case 's':
      case 'S': for (i=0;;i++)
                {
                  if (IsNum(*(++argum)))
                  {
                    buf[i] = *argum;
                  }
                  else
                  {
                    break;
                  }
                }
                buf[i] = '\0';
                if ( (Shade = GetNum(buf,0)) == 0 ||
                     (Shade > 99)
                   )
                {
                  return(-1);
                }
                break;

      case 'l':
      case 'L': for (i=0;;i++)
                {
                  if (IsNum(*(++argum)))
                  {
                    buf[i] = *argum;
                  }
                  else
                  {
                    break;
                  }
                }
                buf[i] = '\0';
                if ( (ShadeLines = GetNum(buf,0)) == 0 ||
                     (ShadeLines > 5)
                   )
                {
                  return(-1);
                }
                break;

      default:  return(-1);
    }
  }
  return(0);
}

/*****************************************************************************/
/*  Function ParseSize()                                                     */
/*                                                                           */
/*    Take a string representing a number (####.#) and return the number*10  */
/*    as an unsigned int.                                                    */
/*****************************************************************************/
unsigned ParseSize(char *strnum)
{
  static unsigned int sizeby10;
  static char size[12];
  register int i=0;

  while (IsNum(*strnum))
  {
    size[i++] = *strnum++;
  }
  size[i] = '\0';
  if ( (sizeby10 = atoi(size) * 10) == 0) return(0);
  if (*strnum++ == '.')
  {            /* the size has a decimal point */
    i=0;       /* so get one decimal number */

    if (IsNum(*strnum))
    {                  /* and next char is number */
      size[i++] = *strnum;
    }
    size[i] = '\0';
    sizeby10 += atoi(size);
  }
  return(sizeby10);
}

/*****************************************************************************/
/*  Function ProcessArg()                                                    */
/*                                                                           */
/*    Take an argument and process it.                                       */
/*****************************************************************************/
void ProcessArg(char *argum)
{
  register unsigned index;
  register unsigned a,b;

#ifdef FUNCS
fprintf(Display,"->ProcessArg\n");
#endif

  /* first we need to check that it is not 1 char argument E.g. + or
  ** "-P" if this is for Unix
  */

  if (argum[1] == '+')
  {
    for (a=0; a<strlen(NoSig);a++)
    {
      NoSig[a] ^= (char)Key;
    }
    if (!strcmp(strlwr(argum+2),NoSig))
    {
      bNoSig = TRUE;
    }
    else
    {
      BadArg(argum);
    }
  }

#ifdef UNIX
  else if (argum[1] == 'P')
  { /* specify alternate printer to pipe to */
    if (argum[2] != '-')
    {
      if (OutputFP != stdout)
      {
        strcpy(OName,PIPESTR);
        for (a = 0, b= 2; b <= strlen(argum); a++, b++)
          Printer[a] = argum[b];   /* copy output filename to OName[] */
        Printer[++a] = '\0';
      }
    }
    else
    {
      OutputFP = (FILE *)NULL;
      strcpy(OName,PIPESTR);
      strcpy(Printer,DEFPRINT);
    }
  }
#endif

  else
  {

    /* this loop validates the 2 char argument against the list of valid
    ** arguments in array ArgTable[][].  If index i is pointing to a valid
    ** argument at the end of the loop, the argument was valid, else
    ** report an error and exit.
    */


    for (index=0; ArgTable[index][0] != '\0'; index++)
    {
      if ((argum[1] | 0x20) == ArgTable[index][0])    /* first char */
      {
        if ((argum[2] | 0x20) == ArgTable[index][1])  /* 2nd char */
        {
          break; /* it matches, exit with index pointing to entry */
        }
      }
    }
    if (ArgTable[index][0] == '\0') BadArg(argum);

    if (argum[3] != '-')
    {                       /* Not switching arg off */
      switch(index)
      {

        case eHELP:
              /* Show usage - all other args ignored */
              ShowUsage();
              break;

        case eLIST:
              /* list available fonts - all other args ignored */
              ListFonts();
              break;

        case eALTPAGE:
              /* print alternate pages */
              bPrint2nd = TRUE;
              switch (argum[3])
              {
                case 'E':
                case 'e': EvenOrOdd = EVENPAGES;
                          break;
                case 'O':
                case 'o': EvenOrOdd = ODDPAGES;
                          break;
                default:  BadArg(argum);
              }
              break;

        case eBOXPAGE:
              /* surround the printable area in a box */
              bBoxPage = TRUE;
              break;

        case eCTLD:
              /* turn on ^D mode */
              bControlD = TRUE;
              break;

        case eCOPIES:
              /* specify how many copies are required */
              if (argum[3])
              {
                if ( (Copies = GetNum(argum,3)) < 1) BadArg(argum);
              }
              else Copies = 1; /* for ZAT :-) */
              break;

        case eCTLZ:
              /* do not print ^Z */
              bPrintCZ = FALSE;
              break;

        case eDEFACT:
              /* default action for ps file */
              switch (argum[3])
              {
                case 'S': /* skip */
                case 's':
                case 'O': /* output unchanged */
                case 'o':
                case 'P': /* process */
                case 'p': DefAction = argum[3];
                          break;

                default:  BadArg(argum);
              }
              break;

        case eDENMARK:
              /* Danish Fix */
              bDanishFix = TRUE;
              bIBMPCFont = TRUE;
              strcpy(Font[NORMAL],IBMF); /* must be IBMPCFont for danish fix */
              strcpy(Font[HEADER],IBMF);
              break;

        case eDELFILE:
              /* delete input files after processing */
              bDeleteInput = TRUE;
              break;

        case eDUPLEX:
              /* print in duplex mode */
              bDuplex = TRUE;
              if ( (argum[3] | 0x20) == 't') bTumble = TRUE;
              break;

        case eDATEFORM:
              /* date format in banner */
              switch (argum[3])
              {
                case '1':
                case '2':
                case '3': DateFormat = (char)argum[3]-(char)0x30;
                          break;

                default:  BadArg(argum);
              }
              break;

        case eERRHAND:
              /* download extended error handler to printer */
              bExtendErr = TRUE;
              break;

        case eENCODING:
              /* specify which char encoding, ISO or traditional PCPS euro */
              switch (argum[3])
              {
                case 'i':
                case 'I': bISO8859 = TRUE;
                          break;

                case 'e':
                case 'E': bISO8859 = FALSE;
                          break;

                default:  BadArg(argum);
              }
              break;

        case eENHSTYLE:
              /* specify enhanced print style */
              switch (argum[3])
              {
                case 'b':
                case 'B': EnhPrintFont=BOLD;
                          break;

                case 'i':
                case 'I': EnhPrintFont=ITALIC;
                          break;

                case 'c':
                case 'C': EnhPrintFont=BOLDITALIC;
                          break;

                default:  BadArg(argum);
              }
              break;

        case eFILENAME:
              /* replace the filename with another string in the banner */
              bFNSpecified = TRUE;
              if (argum[3])    /* Is there a comment? */
              {
                for (a = 0, b= 3; b < strlen(argum); a++, b++)
                  if (argum[b] != '\"')
                    AltFileName[a] = argum[b];  /* copy comment to string */
                AltFileName[a] = '\0';
                ConvertChar(AltFileName);
              }
              else BadArg(argum);
              break;

        case eFONT:
              /* specify font and/or size */
              a=0; /* FontCode */
              b=3; /* argum */
              while ( IsNum(argum[b]) == FALSE && argum[b])
              {
                FontCode[a++] = argum[b++]; /* if not number, must be font */
                FontCode[a] = '\0';     /* Font name ended */
              }
              /* now get size, first non decimal */
              if (argum[b])
              {
                bManualFontSize = TRUE;
                if ( (FontSizeBy10 = ParseSize(&argum[b])) == 0)
                  BadArg(argum);
              }
              break;

        case eGAUDY1:
              /* print gaudy only on first page */
              bGaudy1 = TRUE;

        case eGAUDY:
              /* print a gaudy header at the top of each page */
              bGaudyMode = TRUE;
              bBannerOn = TRUE;         /* cancels any previous -nb */
              if (argum[3])    /* Is there a comment? */
              {
                for (a = 0, b= 3; b < strlen(argum); a++, b++)
                  if (argum[b] != '\"')
                    FileComment[a] = argum[b];  /* copy comment to string */
                FileComment[a] = '\0';
                ConvertChar(FileComment);
              }
              else *FileComment = '\0'; /* no comment */
              break;

        case eIDENT:
              /* identify user@host in corner */
              bPrintName = TRUE;
              if (argum[3])
              {
                if ( (IDFontSizeBy10 = ParseSize(&argum[3])) < 30)
                  BadArg(argum);    /* font size must be at least 3 */
              }
              break;

        case eINVERSE:
              /* output pages in inverse order (last page first) */
              bPrintInv = TRUE;
              if (argum[3])
              {
                switch (argum[3])
                {
                  case 'F':
                  case 'f': bMemSortAllow = FALSE;
                            bNewRev       = TRUE;
                            break;
                  case 'O':
                  case 'o': bMemSortAllow = FALSE;
                            bNewRev       = FALSE;
                            break;
                  default:  BadArg(argum);
                }
              }
              else
              {
                bMemSortAllow = TRUE;      /* default use mem if possible */
                bNewRev = TRUE;            /* if not, use new method */
              }
              break;

        case eLINESPC:
              /* custom line space as a percentage of font size */
              if ( (LineSpace = GetNum(argum,3)) < 1) BadArg(argum);
              break;

        case eLINELEN:
              /* set to linelength to 32k or user specification */
              bLineLimitSpecified = TRUE;
              if (argum[3])    /* # of chars specified?? */
              {
                if ( (MaxLineLength = GetNum(argum,3)) < 1) BadArg(argum);
              }
              else
              {
                MaxLineLength = (unsigned) 0xffff;
              }
              break;

        case eLINENUM:
              /* print line numbers - may be followed by # of digits */
              bLineNumbers = TRUE;
              if (argum[3])   /* # of digits specified ?? */
              {
                if ( (LineNumberWidth = GetNum(argum,3)) < 1)
                  BadArg(argum);
                if (LineNumberWidth > 6)
                {
                  SHOW(Display
                       ,"%u digit line numbers!!  Don't be "
                        "unrealistic!!  Set to 6 digits...\n"
                       ,LineNumberWidth
                      );
                  LineNumberWidth = 6;
                }
              }
              else
              {
                LineNumberWidth = 4;
              }
              break;

        case eLINEPG:
              /* no of lines per page */
              if ( (LinesPerPage = GetNum(argum,3)) < 1 )
                BadArg(argum);
              if (!LinesPerPage) BadArg(argum);
              break;

        case eMARGIN:
              /* Edges - modify margins */
              if (!bManMargin)  /* if curr default, set flag  */
              {                 /* and initialise new margins */
                bManMargin = TRUE;
                NewLeft = NewRight = NewTop = NewBot = 0;
              }
              /* convert points to 1/1440" */
              NewMarg = GetNum(argum,4) * 20;
              switch(argum[3])
              {
                case 'L':
                case 'l': NewLeft = NewMarg;
                          break;
                case 'R':
                case 'r': NewRight = NewMarg;
                          break;
                case 'T':
                case 't': NewTop = NewMarg;
                          break;
                case 'B':
                case 'b': NewBot = NewMarg;
                          break;
                default : BadArg(argum);
              }
              break;

        case eMANFEED:
              /* put printer into manual sheet feed */
              switch (argum[3])
              {
                case 0  : bManualFeed = TRUE;
                          break;
                case '1': bManualFeedPage1 = TRUE;
                          break;
                default : BadArg(argum);
              }
              break;

        case eMODE:
              /* mode */

              Mode = argum[3] - 0x30; /* coverts '1' to 1 */

              switch (Mode)
              {
                case M_MAN: /* Man Page mode */
                          LinesPerPage = 66; /* true for all man pages */
                          strcpy(FontCode,"C");    /* required for man */
                          break;
                case M_MAIL: /* Mail mode */
                          strcpy(FontCode,"C");    /* default for mail */
                          break;
                case M_CPLUS: /* C/C++ comment mode */
                          ActEnh = C_Char;
                          break;
                case M_4GL: /* 4GL comment mode */
                          ActEnh = GL_Char;
                          break;
                case M_SHELL: /* Shell/PERL comment mode */
                          ActEnh = S_Char;
                          break;
                case M_PASC: /* Pascal comment mode */
                          ActEnh = P_Char;
                          break;
                case M_FORT: /* Fortran comment mode */
                          ActEnh = F_Char;
                          break;
                default : BadArg(argum);
              }
              break;

        case eNOBANN:
              /* turn off banner - inhibits gaudy banner also */
              bBannerOn = FALSE;
              break;

        case eNOFLUSH:
              /* Do Not Flush pages between files in duplex mode */
              bFlushPage = FALSE;
              break;

        case eNOZEROS:
              /* Replace leading 0's in linenumbers with spaces */
              *LineNumberPad = ' ';
              break;

        case eOUTFILE:
              /* output to specified file */
              if (OutputFP != stdout)
              {
#ifdef UNIX
                Printer[0] = '\0';   /* cancel print command */
#endif
                for (a = 0, b= 3; b <= strlen(argum); a++, b++)
                  OName[a] = argum[b];   /* copy output filename to OName[] */
                OName[++a] = '\0';
                }
              break;

        case ePRNPREP:
              /* specify printer preparation file */
              for (a=0, b=3; b <= strlen(argum); a++, b++)
                PPFName[a] = argum[b]; /* copy PPF to PPFName */
              PPFName[++a] = '\0';
              break;

        case ePGRANGE:
              /* page range(s) - break down ranges #,#-#,-#,#-$ */
              bRanging = TRUE;
              if (ParseRanges(&argum[3])) BadArg(argum);
              CompressRange();     /* now rationalise the range */
              break;

        case ePSTIME:
              /* download timing routines to printer */
              bPSTiming = TRUE;
              break;

        case eQUIET:
              /* don't print the fancy stuff */
              switch (argum[3])
              {
                case 'T':
                case 't': bNoDisplay = TRUE;

                case 0  : bQuiet = TRUE;
                          break;

                default:  BadArg(argum);
              }
              break;

        case eROTATE:
              /* rotate to landscape mode */
              bLandscape = TRUE;
              break;

        case e2COL:
              /* two column mode */
              bTwoCol = TRUE;
              if ( (argum[3] | (char)0x20) == 'r') /* -2cr */
              {
                bLandscape = TRUE;
              }
              if (!bManualFontSize) /* ie. no font size so far specified */
              {                             /* via -f argument */
                FontSizeBy10 = 70;  /* default font size for two Column */
              }
              break;

        case eSHADE:
              /* Shade every second line */
              bShade = TRUE;
              Shade = 96;       /* default shading factor */
              ShadeLines = 1;   /* default shade every other line */
              if (argum[3])
              {
                if (ParseShade(&argum[3])) BadArg(argum);
              }
              break;

        case eSWAPMAR:
              /* swap margins for double sided printing */
              bSwap = TRUE;
              switch (argum[3])
              {
                case 'O':
                case 'o': SwapEO = ODDPAGES;
                          break;
                case 'E':
                case 'e': SwapEO = EVENPAGES;
                          break;
                default:  BadArg(argum);
              }
              break;

        case ePSIZE:
              /* paper size */
              for (a = 0; a < NUMPAPERSIZES; a++)
              {
                if (!strcmp(&argum[3],PaperStats[a].shortname))
                { /* correct code in shortname */
                  PInfo.PaperH = PaperStats[a].height;
                  PInfo.PaperW = PaperStats[a].width;
                  strcpy(PInfo.PaperName,PaperStats[a].name);
                  strcpy(PInfo.ScaleUp,PaperStats[a].scale); /* assume 2up */
                  a = 0;  /* to flag match found */
                  break;
                }
              }
              if (a) BadArg(argum); /* no match found so bad code */
              break;

        case eTABSTOP:
              /* space out tabs - non intelligent at this stage */
              bSpaceTabs = TRUE;
              if (argum[3])    /* # of spaces specified?? */
              {
                if ( (TabSpace = GetNum(argum,3)) < 1) BadArg(argum);
              }
              else
                TabSpace = 8;
              break;

        case eTRUNC:
              /* truncate lines at maxlinelength */
              bWrapLine = FALSE;
              break;

        case eTRAY:
              /* Hopper - set whether top or bottom tray is required */
              /* or help */
              bTray = TRUE;
              if (GetTrays(&argum[3])) BadArg(argum);
              break;

        case ePGUP:
              /* 2 or 4 pages per physical page */
              switch (argum[3])
              {
                case '2': PagesUp = 2;
                          break;

                case '4': PagesUp = 4;
                          break;

                default:  BadArg(argum);
              }
              /* Do they want lines? */
              switch (argum[4])
              {
                case 'L': 
                case 'l': bUpLines = TRUE;
                          break;
              }
              break;

        case eWATERMK:
              /* background comment */
              if (argum[3])  /* comment included */
              {
                bBackText = TRUE;
                for (a = 0, b = 3; a < BACKTEXTCHARS; a++, b++)
                {
                  if (argum[b]) BackText[a] = argum[b];
                  else break;
                }
                BackText[a] = '\0';
              }
              else
              {
                BadArg(argum);
              }
              break;
      }
    }
    else   /* switching arg off */
    {
      switch(index)
      {
        case eHELP:
              /* Show usage - all other args ignored */
              ShowUsage();
              break;

        case eLIST:
              /* list available fonts - all other args ignored */
              ListFonts();
              break;

        case eALTPAGE:
              /* do not print alternate pages */
              bPrint2nd = FALSE;
              break;

        case eBOXPAGE:
              /* do not surround the printable area in a box */
              bBoxPage = FALSE;
              break;

        case eCTLD:
              /* turn off ^D mode */
              bControlD = FALSE;
              break;

        case eCOPIES:
              /* default 1 copy */
              Copies = 1;
              break;

        case eCTLZ:
              /* print ^Z */
              bPrintCZ = TRUE;
              break;

        case eDEFACT:
              /* take no default action on PS file */
              DefAction = '\0';
              break;

        case eDENMARK:
              /* turn off danish fix */
              bDanishFix = FALSE;
              break;

        case eDELFILE:
              /* Do not delete input files */
              bDeleteInput = FALSE;
              break;

        case eDUPLEX:
              /* do not use duplex */
              bDuplex = bTumble = FALSE;
              break;

        case eDATEFORM:
              /* use default date format */
              DateFormat = 1;
              break;

        case eERRHAND:
              /* do not send extended error handler */
              bExtendErr = FALSE;
              break;

        case eENCODING:
              /* traditional PCPS euro encoding */
              bISO8859 = FALSE;
              break;

        case eENHSTYLE:
              /* set default enhanced print style */
              EnhPrintFont=BOLD;
              break;

        case eFILENAME:
              /* do not replace the filename with another string in the banner */
              bFNSpecified = FALSE;
              break;

        case eFONT:
              /* use default font and size */
              bManualFontSize = FALSE;
              bIBMPCFont = TRUE;
              strcpy(FontCode,"IBM");            /* default font */
              FontSizeBy10 = bTwoCol ? 70 : 100;
              break;

        case eGAUDY1:
        case eGAUDY:
              /* turn off all gaudy banners */
              bGaudyMode = bGaudy1 = FALSE;
              break;

        case eIDENT:
              /* do not identify user */
              bPrintName = FALSE;
              IDFontSizeBy10 = 40;
              break;

        case eINVERSE:
              /* do not print pages in inverse order */
              bPrintInv = FALSE;
              break;

        case eLINESPC:
              /* turn off custom line space */
              LineSpace = 105;
              break;

        case eLINELEN:
              /* use default line length */
              bLineLimitSpecified = FALSE;
              break;

        case eLINENUM:
              /* do not number lines */
              bLineNumbers = FALSE;
              break;

        case eLINEPG:
              /* no preset lines/page */
              LinesPerPage = 0xffff;
              break;

        case eMARGIN:
              /* Margins - use default margins */
              bManMargin = FALSE;
              NewLeft = NewRight = NewTop = NewBot = 0;
              break;

        case eMANFEED:
              /* No manual sheet feed */
              bManualFeedPage1 = bManualFeed = FALSE;
              break;

        case eMODE:
              /* turn off any set mode */
              Mode = 0;
              strcpy(FontCode,"IBM");
              LinesPerPage = 0xffff;
              break;

        case eNOBANN:
              /* banner on/off */
              bBannerOn = TRUE;
              break;

        case eNOFLUSH:
              /* Flush pages between files in duplex mode */
              bFlushPage = TRUE;
              break;

        case eNOZEROS:
              /* Use leading 0's in linenumbers */
              *LineNumberPad = '0';
              break;

        case eOUTFILE:
              /* use default output file/device */
              OutputFP = (FILE *)NULL;
#ifdef UNIX
              strcpy(OName,PIPESTR);
              strcpy(Printer,DEFPRINT);
#else
              strcpy(OName,"LPT2");   /* restore original default */
#endif
              break;

        case ePRNPREP:
              /* disable all PPF files */
              PPFName[0] = '\0';
              break;

        case ePGRANGE:
              /* turn off page ranging */
              bRanging = FALSE;
              ClearRange();                 /* clear any current ranges */
              break;

        case ePSTIME:
              /* do not send ps timing info */
              bPSTiming = FALSE;
              break;

        case eQUIET:
              /* make it noisy */
              bQuiet = bNoDisplay = FALSE;
              break;

        case eROTATE:
              /* no rotate */
              bLandscape = FALSE;
              break;

        case e2COL:
              /* turn off 2 col */
              bTwoCol = FALSE;
              if (!bManualFontSize) /* ie. no font size so far specified */
              {                             /* via -f argument */
                FontSizeBy10 = 100; /* restore these back to default */
              }
              break;

        case eSHADE:
              /* Turn off Shade every second line */
              bShade = FALSE;
              break;

        case eSWAPMAR:
              /* do not swap margins */
              bSwap = FALSE;
              break;

        case ePSIZE:
              /* reset back to default (A4 is first element in struct) */
              PInfo.PaperH = PaperStats[0].height;
              PInfo.PaperW = PaperStats[0].width;
              strcpy(PInfo.PaperName,PaperStats[0].name);
              strcpy(PInfo.ScaleUp,PaperStats[0].scale); /* assume 2up */
              break;

        case eTABSTOP:
              /* do not space out tabs */
              bSpaceTabs = FALSE;
              break;

        case eTRUNC:
              /* do not truncate lines - wrap instead */
              bWrapLine = TRUE;
              break;

        case eTRAY:
              /* don't play with trays */
              bTray = FALSE;
              break;

        case ePGUP:
              /* 1 page per physical page */
              PagesUp = 1;
              bUpLines = FALSE;
              break;

        case eWATERMK:
              /* no background watermark */
              bBackText = FALSE;
              break;
      }
    }
  }
}

/*****************************************************************************/
/*  Function ProcessFile()                                                   */
/*                                                                           */
/*    This is the main process executed on each file specified.              */
/*****************************************************************************/
void ProcessFile(char *fn)
{
 
#ifdef MSVC32
  Title(fn?fn:"stdin");
#endif

  /* if we got to here, then at least one input file is valid so fire out
     the prolog */

  if (!bPrologDone) DoProlog();           /* output the required prolog */

  /* first reset all per-file flags */

  CurrentLineNumber = 0;        /* haven't gotten to first line yet */
  TotalFileBytes = 0;           /* none written yet */
  MaxPageBytes = 0;             /* not known yet */
  bMemSort = FALSE;             /* default - inv sort done on disk */
  Column = 1;                   /* Reset to Column 1 for each new file */
  XLine = PInfo.Left;           /* Reset LH margin for each new file */
  PageNo_PS = PageNo_Doc = PagesPrinted = 0;      /* inc to 1 in DoPage() */
  if (bGaudy1)  /* set these to on so that the gaudy banner is printed
                   on the  1st page - they are reset in StartPageF()
                */
  {
    bGaudyMode = TRUE;
    bBannerOn = TRUE;
  }

#if defined (UNIX) || defined (MSVC32)
  PPtrM_curr = (pptrm *)NULL; /* reset linked list for memory sort */
  PPtrF_curr = (pptrf *)NULL; /* reset linked list for fast file sort */
#else
  PPtrM_curr = (pptrm huge *)NULL; /* reset linked list for memory sort */
  PPtrF_curr = (pptrf huge *)NULL; /* reset linked list for fast file sort */
#endif

  /* So now lets open the input file */
  if (fn)
  {
    strcpy(FName,fn);         /* input filename */
    strcpy(FNamePS,fn);       /* input filename in PS */
    ConvertChar(FNamePS);     /* make it printable in PS */
    if ( (InputFP = fopen(fn,"rb")) == NULL )
    {
      SHOW(Display,"%sCannot open %s for input\n",VerString,fn);
      return;  /* Can't open this input file so try the next */
    }

    if (CheckFile(InputFP,fn) == FALSE)   /* check whether to process file */
    {
      if (!bQuiet)
      {
        SHOW(Display,"%s\n",Line);
      }
      return;
    }
  }
  else
  {
    InputFP = stdin;
    strcpy(FNamePS,"stdin");
    strcpy(FName,"stdin");         /* input filename */
  }

  if (bPrintInv)
  {
#ifdef DEBUG1
    fprintf(Display,"File size is %lu\n",FndFile.fsize);
#endif
    GetSortMemory(FndFile.fsize);
  }

  if (bBannerOn)
      MakeBanner();

  if (!bQuiet)
  {
    /* Tell user what's happening */
    SHOW(Display,"Input filename %s %s ",ArrowR,fn?fn:FNamePS);
    if (!bPrintInv)   /* only tell them about sort if sort is used */
    {
      SHOW(Display,"\n");
    }
    else
    {
      if (bMemSort)
      {
        SHOW(Display,"%s Memory Sort\n",ArrowL);
      }
      else if (bNewRev)
      {
        SHOW(Display,"%s Fast File Sort\n",ArrowL);
      }
      else
      {
        SHOW(Display,"%s Slow File Sort\n",ArrowL);
      }
    }
  }

  if ((bPrintInv) && (!bMemSort))
  {
    /* Write the pages into a temporary file for reversing output order */
    FileOut = fopen(TEMPFILE,"wb");
  }
  else
  {
    /* no inv - just write straight to output file */
    FileOut = OutputFP; 
  }

  DoPage();     /* start first page */
  WrapCount = TrunCount = 0;
  ReadFileF();

  /* Do not close the file if it is stdin.
  */
  if (InputFP != stdin)
  {
		fclose(InputFP);
  }

  
  if (bStartPageIssued)
  {                     /* if we have started a page, end it */
    EndPageF();
  }

  if (bPrintInv)
  {

  /* At this point we have written:
  **
  ** - The prolog to the output file (OutputFP)
  ** - The converted ps pages to the temporary file (TEMPFILE)
  **    or to memory.
  ** Now lets reverse the order of the pages and write them into the output
  ** file and then delete the temporary file if necessary.
  */

    if (!bPrintPage) PageNo_PS--;  /* if we didn't print the last page
                                      dec the total pages by one */
    if (bMemSort)
    {
#ifdef DEBUG1
      /* show utilisation */
        MemUsed = BlockPoint - Block;
        fprintf(Display
                ,"\nMemSort Buffer %d%% Utilised\n"
                ,(100 * MemUsed / MemReq)
               );
#endif
      MemReverse();
    }
    else
      ReversePages();
  }

  StringOut("EndDoc\n"); /* to clean up before the next file */

  if (!bQuiet)
  {
    if (bPrint2nd || bRanging)
    {
      SHOW(Display,"processed - %u pages printed.\n",PagesPrinted);
    }
    else
    {
      SHOW(Display,"Done!\n");
    }
    if (WrapCount > 0)
    {
      SHOW(Display
           ,"%u Line(s) wrapped that were >%u characters...\n"
           ,WrapCount
           ,MaxLineLength
          );
    }
    else
    {
      if (TrunCount > 0)
      {
        SHOW(Display
             ,"%u Line(s) truncated after %u characters...\n"
             ,TrunCount
             ,MaxLineLength
            );
      }
    }
  }

  if ( bDeleteInput )
  {
    if ( unlink(fn) )
    {
      SHOW(Display,"Could not delete %s\n",fn);
    }
    else
    {
      SHOW(Display,"%s deleted\n",fn);
    }
  }

  if (!bQuiet)
  {
    SHOW(Display,"%s\n",Line);
  }
  else
  {
    SHOW(Display,"%s - %u pages\n",fn,PageNo_PS);
  }

  if (bGaudy1)
  { /* reset the text height back to underneath the banner for the next file */
    THeight = GHeight - BannerGap;
  }

  if (bMemSort)
  {
    HFREE(Block);  /* free up the sort memory buffer */

#ifdef DEBUG1
    fprintf(Display,"Freeing %lu bytes of allocated memory...\n",MemReq);
#endif

  }
}

/*****************************************************************************/
/*  Function ReadFileF()                                                      */
/*                                                                           */
/*    Read the file character by character and process it.                   */
/*****************************************************************************/
void ReadFileF(void)
{
  static int ch;


#ifdef FUNCS
fprintf(Display,"->ReadFileF\n");
#endif


  for (;;)
  {
    if ( (ch = getc(InputFP)) == EOF)
    {
      if (TextPtr != TextBuf) FlushLine(FALSE); /* if the last line has no lf
                                                   sequence */
      break;
    }

    switch ((char)ch)
    {
      case LF:    FlushLine(FALSE);                    /* lf - end of line */
                  break;

      case CR:    FlushLine(FALSE);                    /* end of line */
                  if ( (ch = getc(InputFP)) != LF)
                  {
                    /* not crlf so allow overwrite - must account for
                    ** line feed (YDecrement) and line count (LineNoOnPage)
                    ** which are adjusted in FlushLine()
                    */
                    YLine += YDecrement;
                    LineNoOnPage--;
                    /* Now put the 2nd char back into the stream for
                    ** processing as per normal
                    */
                    ungetc(ch,InputFP);
                  }
                  break;

                    
      case FF:    FlushLine(FALSE);
                  if (!bFFJustSent)
                  {
                    DoPage();                          /* ff - end of page */
                  }
                  break;

      case '\t':  if (!bWasteRest)
                  {
                    if (bSpaceTabs)                    /* tabs */
                    {
                      do
                      {
                        CheckEndLine();
                        *TextPtr++ = ' ';
                      } while (LinePos % TabSpace);
                    }
                    else          /* not spacing tabs, just send tab character */
                    {
                      CheckEndLine();
                      *TextPtr++ = (char)ch;
                    }
                  }
                  break;

      case '\b':  if (!bWasteRest)
                  {
                    if (Mode == M_MAN)
                    {
                      LinePos--;
                    }
                    else
                    {
                      CheckEndLine();
                    }
                    *TextPtr++ = (char)ch;
                  }
                  break;

      default:    if (!bWasteRest)
                  {
                    CheckEndLine();
                    *TextPtr++ = (char)ch;
                  }
                  break;
    }
    if (bSkipToEnd) break;     /* end of printing so exit rest of file */
    bFFJustSent = FALSE;                 /* done its job */
  }
}

/*****************************************************************************/
/*  Function ReversePages()                                                  */
/*                                                                           */
/*    Reverse the order of the pages either by the new method or old         */
/*****************************************************************************/
void ReversePages(void)
{

#ifdef FUNCS
fprintf(Display,"->ReversePages\n");
#endif

  fclose(FileOut);
  if( (FileOut = fopen(TEMPFILE,"rb") )  == NULL)
  {
    SHOW(Display
         ,"%sCan not find the temporary sort file %s %s\n"
         ,VerString
         ,ArrowR
         ,FileOut
        );
    MyExit(1,NOHELP);
  }

  if (bNewRev)
  {
    RevPagesN();
  }
  else
  {
    RevPagesO();
  }

  fclose(FileOut);
  unlink(TEMPFILE);
  SHOW(Display,"%u Pages ",PageNo_PS);
}

/*****************************************************************************/
/*  Function RevPagesN()                                                     */
/*                                                                           */
/*     Reverses (Inverses) the order of pages in the final output file       */
/*      if enabled.  New fast method using linked list to remember byte      */
/*      offset within temporary file of start of each page.                  */
/*****************************************************************************/
void  RevPagesN(void)
{

  register unsigned page;
  unsigned countbytes;
  char *revptr;
#if defined (UNIX) || defined (MSVC32)
  pptrf *pptrf_old;
#else
  pptrf huge *pptrf_old;
#endif

#ifdef FUNCS
fprintf(Display,"->RevPagesN\n");
#endif

  page = PagesPrinted;
  AddPageOffset();                   /* to find out enf of file offset */
  if ( (revptr = malloc(MaxPageBytes)) == NULL)
  {
    SHOW(Display
         ,"%sCouldn't get memory for fast file sort.\n"
          "%sTry using the slow file sort (\"-io\" argument).\n"
         ,VERS
         ,VERS
        );
    MyExit(1,NOHELP);
  }

  while (PPtrF_curr->prevpageptr)
  {
    if (!bQuiet)
    {
      SHOW(Display,"\rReversing  Page %-5d ",page--);
    }
#ifndef UNIX
    SpinProp();
#endif
    countbytes = (unsigned)
                 (PPtrF_curr->offset -
                  PPtrF_curr->prevpageptr->offset); /* total bytes in page */
    fseek(FileOut                                  /* stream */
          ,PPtrF_curr->prevpageptr->offset         /* offset of start page */
          ,SEEK_SET                                /* from beginning of file */
         );
    fread(revptr
          ,1
          ,(size_t)countbytes
          ,FileOut
         );
    fwrite(revptr
           ,1
           ,(size_t)countbytes
           ,OutputFP
          );
    pptrf_old = PPtrF_curr;
    PPtrF_curr = PPtrF_curr->prevpageptr;
    free((char *)pptrf_old);
  }
  free((char *)PPtrF_curr);
}

/*****************************************************************************/
/*  Function RevPagesO()                                                     */
/*                                                                           */
/*     Reverses (Inverses) the order of pages in the final output file       */
/*      if enabled.  This is the old slow method but doesn't require any     */
/*      additional memory, so kept it just in case.                          */
/*****************************************************************************/
void  RevPagesO(void)
{

#ifdef FUNCS
fprintf(Display,"->RevPagesO\n");
#endif

  PagesLeft = PagesPrinted;
  while (PagesLeft > 0) {
    fgets(MiscBuf,512,FileOut);
    if(!strncmp(MiscBuf,SPAGE,4))
    {
      if( (PageCount = GetLastNum(MiscBuf)) == PagesLeft)
      {
        fgets(MiscBuf,512,FileOut);
        do
        {
          fputs(MiscBuf,OutputFP);
#ifndef UNIX
          SpinProp();
#endif
          fgets(MiscBuf,512,FileOut);
        } while (strncmp(MiscBuf,EPAGE,4) != (unsigned)NULL);
        fseek(FileOut,0L,SEEK_SET);
        if (!bQuiet)
        {
          SHOW(Display,"\rReversing  Page %-5d ",PagesLeft);
        }
        PagesLeft--;
      }
    }
  }
}
/*****************************************************************************/
/*  Function SetDefaults()                                                   */
/*                                                                           */
/*    Set up global variable defaults and check for config file              */
/*****************************************************************************/
void SetDefaults(void)
{
#ifdef FUNCS
fprintf(Display,"->SetDefaults\n");
#endif

  bFNSpecified = FALSE;              /* just print file name */
  bBackText = FALSE;                 /* not printing bg text */
  bBannerOn = TRUE;                  /* Banner on */
  bBoxPage = FALSE;                  /* do not box the printable area */
  bControlD = FALSE;                 /* do not send ^d */
  bDeleteInput = FALSE;              /* leave input file alone */
  bDuplex = FALSE;                   /* non duplex printing */
  bExtendErr = FALSE;                /* no download of ext err */
  bFlushPage = TRUE;                 /* flush pages between files in duplex */
  bGaudy1 = FALSE;                   /* not gaudy for first page only */
  bGaudyMode = FALSE;                /* no gaudy header */
  bIBMPCFont = TRUE;                 /* IBM ext char support */
  bISO8859 = TRUE;                   /* ISO-8859-1 re-encoding enabled */
  bLandscape = FALSE;                /* portrait mode */
  bLineLimitSpecified = FALSE;       /* no user-specified length */
  bLineNumbers = FALSE;              /* no line numbers */
  bEnh = FALSE;                 /* Not bolding a mail header yet */
  bEnhNLine = FALSE;         /* don't unconditionally bold next line */
  bManualFeed = FALSE;               /* continuous paper feed */
  bManualFeedPage1 = FALSE;          /* continuous paper feed */
  bManualFontSize = FALSE;           /* font size automatic */
  bManMargin = FALSE;                /* margins */
  bMemSortAllow = TRUE;              /* memory sort allowed */
  bNoDisplay = FALSE;                /* fprintf's will work */
  bNoSig = FALSE;                    /* print signature (TM) */
  bPrintCZ = TRUE;                   /* print ^Z char */
  bPrintInv = FALSE;                 /* inverse pages off */
  bPrintName = FALSE;                /* do not print user@host */
  bPrintPage = TRUE;                 /* print the first page */
  bPrint2nd = FALSE;                 /* print every page */
  bPrologDone = FALSE;               /* prolog not yet sent */
  bPSTiming = FALSE;                 /* no PostScript timing info */
  bQuiet = FALSE;                    /* quiet mode off */
  bRanging = FALSE;                  /* print whole document */
  bShade = FALSE;                    /* do not shade evry second line */
  bSkipToEnd = FALSE;                /* do not skip to EOF */
  bSpaceTabs = TRUE;                 /* space out tabs to 8 spaces */
  bStartPageIssued = FALSE;          /* Hvae not yet issued StartPage */
  bSwap = FALSE;                     /* no margin swapping */
  bTray = FALSE;                     /* don't mess with trays */
  bTumble = FALSE;                   /* no tumble */
  bTwoCol = FALSE;                   /* one Column */
  bUpLines = FALSE;                  /* no dividing lines in -up mode */
  bWasteRest = FALSE;                /* print whole line */
  bWrapLine = TRUE;                  /* Line wrap */
  Column = 1;                        /* Start in Column 1 */
  Copies = 1;                        /* 1 copy of each page */
  DateFormat = 1;                    /* yyyy/mm/dd as default date format */
  DefAction = '\0';                  /* no default action for ps */
  EnhPrintFont=BOLD;                 /* default enhaced print style */
  FontSizeBy10 = 100;                /* font size for the body */
  IDFontSizeBy10 = 40;               /* size of id in corner */
  *LineNumberPad = '0';               /* show leading zeros in line numbers */
  LinesPerPage = 0xffff;             /* whatever will fit on page */
  LineSpace = 105;                   /* line feed as perc of font size */
  Mode = M_NONE;                     /* no mode to start */
  PagesUp = 1;                       /* 1 page per page */
  TabSpace = 8;
  strcpy(FontCode,"IBM");            /* default font */
  if (OutputFP != stdout)
  {                                  /* if not stdout */
#ifdef UNIX
    strcpy(OName,PIPESTR);
    strcpy(Printer,DEFPRINT);
#else
    strcpy(OName,"LPT2");            /* output to LPT2 */
#endif
  }
}

/*****************************************************************************/
/*  Function ShowScreen()                                                    */
/*                                                                           */
/*    Show the people what they are getting.                                 */
/*****************************************************************************/
void ShowScreen(void)
{
#ifdef FUNCS
fprintf(Display,"->ShowScreen\n");
#endif


  fprintf(Display,Flash,VERS);
  fprintf(Display,"%s%s%s\n",FirstPart,TopJoin,LastPartT);
  fprintf(Display
          ," Page Size/Orientation/Mode.\t%s %s, %s%s%s.\n"
          ,Bar
          ,PInfo.PaperName
          ,bLandscape ? "Landscape" : "Portrait"
          ,bDuplex ? ", Duplex" : ""
          ,bTumble ? "/Tumble" : ""
         );

  fprintf(Display
          ," Page Style.................\t%s %c Column, %d-Up.\n"
          ,Bar
          ,bTwoCol ? '2' : '1'
          ,PagesUp
         );

  fprintf(Display
          ," Page Order/Feed/Boxing.....\t%s %s, %s, %sBoxed.\n"
          ,Bar
          ,bPrintInv ? "Reverse" : "Normal"
          ,(bManualFeed || bManualFeedPage1) ? "Manual\a" : "Continuous"
          ,bBoxPage ? "" : "Non-"
         );

  fprintf(Display," Banner style...............\t%s ",Bar);
  if (bBannerOn)
  {
    if (bGaudyMode)
    {
      fprintf(Display,"!!GAUDY!!%s\n",bGaudy1 ? " on the first page only." : "");
    }
    else
    {
      fprintf(Display,"Normal text.\n");
    }
  }
  else
  {
    fprintf(Display,"No banner.\n");
  }

  fprintf(Display
          ," Font details...............\t%s %s, %u.%u point, %s mode.\n"
          ,Bar
          ,Font[NORMAL]
          ,FontSizeBy10/10
          ,FontSizeBy10%10
          ,ModeDesc[Mode]
         );

  fprintf(Display
          ," Line print mode >%u chars..\t%s %s%s.\n"
          ,bLineNumbers ? (MaxLineLength - (LineNumberWidth+1)) : MaxLineLength
          ,Bar
          ,bWrapLine ? "Wrapped" : "Truncated"
          ,bShade ? ", with background shading" : ""
         );

  if (Copies > 1)
  {
    fprintf(Display," Number of copies...........\t%s %d.\n",Bar,Copies);
  }

  if (bSpaceTabs)
  {
    fprintf(Display
            ," Tabstops every.............\t%s %u columns.\n"
            ,Bar
            ,TabSpace
           );
  }

  if (bBackText)
  {
    fprintf(Display
            ," Background text............\t%s \"%s\"\n"
            ,Bar
            ,BackText
           );
  }

  if (bLineNumbers)
  {
    fprintf(Display
            ," Lines will be numbered to..\t%s %u Digits.\n"
            ,Bar
            ,LineNumberWidth
           );
  }

  if (bTray)
  {
    fprintf(Display
            ," Paper Tray(s)..............\t%s Start: %s\n"
            ,Bar
            ,PaperTray1
           );
    if (TraySwPage)
    {
      fprintf(Display
              ," Paper Tray(s)..............\t%s  Then: %s after %d page(s).\n"
              ,Bar
              ,PaperTray2
              ,TraySwPage
             );
    }
  }

  if (bManMargin)
  {
    fprintf(Display
            ," Paper Margins..............\t%s L = %u  R = %u  T = %u  "
             "B = %u  (points)\n"
            ,Bar
            ,PInfo.Left/20,PInfo.Right/20
            ,PInfo.Top/20,PInfo.Bot/20
           );
  }

  if (LinesPerPage != 0xffff)
  {
    fprintf(Display," Number of lines per page...\t%s %u.\n",Bar,LinesPerPage);
  }

  if (bPrint2nd)
  {
    fprintf(Display
            ," Print every other page mode\t%s %s Pages.\n"
            ,Bar
            ,EvenOrOdd ? "Odd" : "Even"
           );
  }

  if (bSwap)
  {
    fprintf(Display
            ," Swap margins on............\t%s %s pages.\n"
            ,Bar
            ,SwapEO ? "Odd" : "Even"
           );
  }

  if (bRanging)
  {
    Range = Range_1;
    while (Range)
    {
      fprintf(Display
              ," Print page ranges..........\t%s %5u - %-5u\t%s\n"
              ,Bar
              ,Range->StartPage
              ,Range->EndPage
              ,Range->NextRange ? "AND" : ""
             );
      Range = Range->NextRange;
    }
  }

#ifdef UNIX
  fprintf(Display
          ," Output PostScript to.......\t%s %s%s\n"
          ,Bar
          ,OName
          ,Printer
         );
#else
  fprintf(Display," Output PostScript filename.\t%s %s\n",Bar,OName);
#endif

  fprintf(Display,"%s%s%s\n",FirstPart,BotJoin,LastPartB);

}

/*****************************************************************************/
/*  Function SLine()                                                         */
/*                                                                           */
/*    Output a buffer in a ()S line - ie. no positioning                     */
/*****************************************************************************/
void SLine(char *buf)
{

  ConvertChar(buf);
  if (*buf)
  {
    if (bPrintPage)
    {
      StringOut("(");
      StringOut(buf);
      StringOut(")S\n");
    }
  }
}

/*****************************************************************************/
/*  Function StartPageF()                                                     */
/*                                                                           */
/*    Performs a start of page sequence                                      */
/*****************************************************************************/
void StartPageF(void)
{

#ifdef FUNCS
fprintf(Display,"->StartPageF\n");
#endif

  if (!bQuiet)
  {
    SHOW(Display,"\rProcessing Page %-5d ",PageNo_PS);
  }

  PagesPrinted++;
  bStartPageIssued = TRUE;

  /* if we are only printing gaudy on 1st page, reset these flags */
  if (bGaudy1 && (PageNo_PS > 1))
  {
    bGaudyMode = FALSE;
    bBannerOn = FALSE;
    THeight = GHeight;   /* we can now print where the banner was */
  }

  if (bPrintInv)     /* add flag to make reversal easy */
  {
    if (bMemSort)
    {                                       /* set ptr for reversing */
      AddPage();
    }
    else
    {
      if (bNewRev)
      {
        AddPageOffset(); /* remember byte offset within file of start of page */
      }
      else
      {
        sprintf(MiscBuf,"%s %d\n",SPAGE,PagesPrinted);/* set flag for reversing */
        StringOut(MiscBuf);
      }
    }
  }

  if (bPSTiming)
  {
    sprintf(MiscBuf,"(Start Page %u of %s) prtime\n",PageNo_PS,FNamePS);
    StringOut(MiscBuf);
  }

  sprintf(MiscBuf,"%s %u %u\n",SP,PageNo_Doc,PageNo_PS);
  StringOut(MiscBuf);

  if ( (PageNo_PS == 1 && bManualFeedPage1) ||
       (bManualFeed)
     )
  {
    StringOut("MB\n");
  }

  if (bTray)                   /* if we are playing with trays */
  {
    if (
        (!TraySwPage)                /* if we are only using one tray */
        ||                           /* ** OR ** */
        (PagesPrinted <= TraySwPage) /* or we are still on first tray */
       )
    {
      if (bTrayName)
      {
        sprintf(MiscBuf,"%s\n",PaperTray1);  /* named */
      }
      else
      {
        sprintf(MiscBuf,"%d SPT\n",Tray1);   /* numbered */
      }
    }
    else
    {                          /* we are onto the 2nd tray */
      if (bTrayName)
      {
        sprintf(MiscBuf,"%s\n",PaperTray2);  /* named */
      }
      else
      {
        sprintf(MiscBuf,"%d SPT\n",Tray2);   /* numbered */
      }
    }
    StringOut(MiscBuf);
  }

  sprintf(MiscBuf
          ,"/LeftMarg %lu def\n/YDec %ld def\n/THeight %lu def\n"
           "/BotLine %u def\n/TWidth %lu def\n/PWidth %lu def\n"
          ,XLine
          ,YDecrement
          ,THeight + THeightMod
          ,BottomLine
          ,Width
          ,GWidth
         );
  StringOut(MiscBuf);

  StringOut("StartPage\n");

  if (bBoxPage && !bGaudyMode)
  { /* box the page - built-in to gaudy, so if not gaudy we need to do it */
    StringOut("BoxPage\n");
  }

  if (bShade)
  { /* shade every second line */
    StringOut("ShadePage\n");
  }

  if (bBackText)
  {
    StringOut("WM\n");
  }

  if (bBannerOn)
  {
    InsertBanner();
  }

  if (*TMString)        /* if something to print, make it so */
  {
    StringOut("TM\n");
  }

  StringOut(bEnh?"1 F\n":"0 F\n");
}

/*****************************************************************************/
/*  Function StreamFile()                                                    */
/*                                                                           */
/*    Send file to output file unprocessed and close input file.             */
/*****************************************************************************/
void StreamFile(FILE *fp)
{
  int count;

  do
  {
    count = fread(StreamBuf,1,sizeof(StreamBuf),fp);
    fwrite(StreamBuf,1,count,OutputFP);
  } while (count == sizeof(StreamBuf));

  fclose(fp);
}

/*****************************************************************************/
/*  Function StringOut()                                                     */
/*                                                                           */
/*    Output string to current output file.                                  */
/*****************************************************************************/
void StringOut(char *str)
{
  static unsigned len;

  len = strlen(str);

#ifndef UNIX
  SpinProp();
#endif

  TotalFileBytes += len;  /* only used for -if */

  if (!bMemSort)
  {
    fwrite(str,1,len,FileOut);
  }
  else
  {
    if ( (BlockPoint + len) >= BlockTop) /* check that we have not gone beyond */
                                         /* the end of allocated memory        */
    {
      SHOW(stderr
           ,"\n%s*****Program Critical Memory Error*****\n"
            "%sPlease report this error to Paul Carapetis...\n"
            "%sThis is a program error and NOT a problem with"
            " your machine...\n%s(Try -if or -io options in the "
            "meantime)"
           ,VerString
           ,VerString
           ,VerString
           ,VerString
          );
#ifdef DEBUG1
      fprintf(Display
              ,"\nCrit Mem: PageNo_PS = %u\tPageNo_Doc = %u\tLineNoOnPage = %u\n"
              ,PageNo_PS
              ,PageNo_Doc
              ,LineNoOnPage
             );
#endif
      MyExit(1,NOHELP);   /* exit errorlevel 1 */
    }
#if defined (UNIX) || defined (MSVC32)
    memcpy((char *)BlockPoint,str,len);
#else
    memcpy((char far *)BlockPoint,str,len);
#endif
    BlockPoint += len;
  }
}

/*****************************************************************************/
/*  Function TermLine()                                                      */
/*                                                                           */
/*    Called when line exceeds max width - Cut or wrap line                  */
/*****************************************************************************/
void TermLine(void)
{
  FlushLine(TRUE);           /* signify that we are wrapping */
  if (!bWrapLine)
  {
    if (bPrintPage) TrunCount++;
    bWasteRest = TRUE;
  }
  else
  {
    if (bPrintPage) WrapCount++;
  }
}

