#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <string.h>

#include <sys/types.h>
#include <fcntl.h>

#include "filebrowser.h"
#include "fileicon.h"

#include "mainwindow.h"
#include "mainwindowui.h"
#include "glade_support.h"

#include "prefdlg.h"

#include "editbookmarks.h"

#include "settings.h"

#define GLADE_HOOKUP_OBJECT(component,widget,name) \
  g_object_set_data_full (G_OBJECT (component), name, \
    gtk_widget_ref (widget), (GDestroyNotify) gtk_widget_unref)


GQuark mainWindowDataId = 0;
static int windowCount = 0;
static GList* allWindows = NULL;

typedef struct{
  GtkWidget* mainWindow;
}MainWindowData;

void errorMessage(GtkWindow* parent, const char* message )
{
  GtkWidget* dlg = gtk_message_dialog_new(parent, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, message);
  gtk_dialog_run( GTK_DIALOG(dlg) );
  gtk_widget_destroy( dlg );
}

static GtkWidget* create_bookmark_menu ( GtkWidget* mainWindow );
static GtkWidget* create_bookmark_menu_item ( GtkWidget* mainWindow,
                                              const char* path,
                                              GdkPixbuf* icon );

static void close_window( GtkWidget* mainWindow )
{
  gtk_widget_destroy( mainWindow );
}

gboolean
on_mainWindow_delete_event             (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  close_window( widget );
  return FALSE;
}

static void
on_close_notebook_page( GtkButton* btn, FileBrowserData* data )
{
  GtkWidget* notebook = lookup_widget( data->mainWindow, "folderNoteBook" );

  if ( gtk_notebook_get_n_pages (GTK_NOTEBOOK(notebook)) <= 1 )
  {
    close_window( data->mainWindow );
    return;
  }

  gint idx = gtk_notebook_page_num ( GTK_NOTEBOOK(notebook),
                                     data->folderPane);
  gtk_notebook_remove_page( GTK_NOTEBOOK(notebook), idx );
  if( gtk_notebook_get_n_pages( GTK_NOTEBOOK(notebook) ) == 1 )
    gtk_notebook_set_show_tabs( GTK_NOTEBOOK(notebook), FALSE );
}

void add_new_tab( GtkWidget* mainWindow,
                  const char* folder_path,
                  gboolean open_dir_tree )
{
  FileBrowserData* data = file_browser_new ( mainWindow,
                                             appSettings.viewMode,
                                             open_dir_tree,
                                             TRUE );
  data->show_hidden_files = appSettings.showHiddenFiles;
  data->show_dir_tree = open_dir_tree;
  data->sort_order = appSettings.sortOrder;
  data->sort_type = appSettings.sortType;

  GtkNotebook* notebook = GTK_NOTEBOOK( lookup_widget(
                                        mainWindow, "folderNoteBook") );

  const char* dir_name = strrchr( folder_path, '/' );
  gint idx;

  if( dir_name[1] )
    ++dir_name;

  gtk_label_set_text( GTK_LABEL( data->tab_text ), dir_name );

  idx =   gtk_notebook_append_page( notebook, data->folderPane,
                                  data->tab_label );
  gtk_notebook_set_current_page ( notebook, idx );

  if( gtk_notebook_get_n_pages ( notebook ) > 1 )
    gtk_notebook_set_show_tabs( notebook, TRUE );
  else
    gtk_notebook_set_show_tabs( notebook, FALSE );

  g_signal_connect( G_OBJECT(data->close_btn), "clicked",
                    G_CALLBACK( on_close_notebook_page ), data );
  file_browser_chdir( data->folderPane, folder_path, TRUE );

  gtk_widget_grab_focus( GTK_WIDGET(data->folderView) );
}

/* Callback called by update callback of FolderContent */
static void
on_update_folder_content               ( FolderContent* content,
                                         gpointer data )
{
  GList* l;
  GtkWidget* file_browser;
  for( l = allWindows; l; l = l->next )
  {
    file_browser = get_current_file_browser( (GtkWidget*)l->data );
    file_browser_update_mime_icons( file_browser );
  }
}

void
on_mainWindow_realize                  (GtkWidget       *widget,
                                        gpointer         user_data)
{
  MainWindowData* data = g_new0(MainWindowData, 1);
  GtkWidget* mainVBox = lookup_widget(widget, "mainVBox");
  GtkWidget* folderNoteBook;
  GtkToolButton* btn;
  GtkCheckMenuItem* menu;
  GtkWidget* bookmark;
  GtkWidget* bookmark_menu;

  mainWindowDataId = g_quark_from_static_string("mainWindowData");
  g_object_set_qdata_full( G_OBJECT(widget), mainWindowDataId, data, g_free );

  gtk_window_set_icon( GTK_WINDOW(widget), get_folder_icon32() );

  if( 0 == windowCount ){
    folder_content_set_update_callback( on_update_folder_content, NULL );
  }

  allWindows = g_list_prepend( allWindows, widget );
  ++windowCount;

  folderNoteBook = gtk_notebook_new ();
  gtk_notebook_set_scrollable ( GTK_NOTEBOOK(folderNoteBook), TRUE );
  gtk_widget_show (folderNoteBook);
  gtk_box_pack_start (GTK_BOX (mainVBox), folderNoteBook, TRUE, TRUE, 0);

  g_signal_connect_swapped ((gpointer) folderNoteBook, "switch_page",
                            G_CALLBACK (on_folderNoteBook_switch_page),
                            GTK_OBJECT (widget));

  GLADE_HOOKUP_OBJECT (widget, folderNoteBook, "folderNoteBook");

  menu = GTK_CHECK_MENU_ITEM( lookup_widget( widget, "open_directory_tree" ) );
  btn = GTK_TOGGLE_TOOL_BUTTON( lookup_widget( widget,
                                               "dirTreeBtn" ) );

  if( appSettings.showDirTree ){
    gtk_check_menu_item_set_active( menu, TRUE );
    gtk_check_menu_item_toggled ( menu );
  }
  menu = GTK_CHECK_MENU_ITEM( lookup_widget( widget,
                                             "show_hidden" ) );
  gtk_check_menu_item_set_active ( menu,
                                   appSettings.showHiddenFiles );

  if( appSettings.viewMode == FBVM_ICON_VIEW ) {
    menu = GTK_CHECK_MENU_ITEM( lookup_widget( widget,
                                        "view_as_icons" ) );
  }
  else if(appSettings.viewMode == FBVM_LIST_VIEW){
    menu = GTK_CHECK_MENU_ITEM( lookup_widget( widget,
                                "view_as_list" ) );
  }
  gtk_check_menu_item_set_active ( menu,
                                   TRUE );

  switch( appSettings.sortOrder )
  {
    case FB_SORT_BY_NAME:
      menu = GTK_CHECK_MENU_ITEM( lookup_widget( widget,
                                  "sort_by_name" ) );
      break;
    case FB_SORT_BY_SIZE:
      menu = GTK_CHECK_MENU_ITEM( lookup_widget( widget,
                                  "sort_by_size" ) );
      break;
    case FB_SORT_BY_TIME:
      menu = GTK_CHECK_MENU_ITEM( lookup_widget( widget,
                                  "sort_by_time" ) );
      break;
  }
  gtk_check_menu_item_set_active ( menu,
                                   TRUE );

  if( appSettings.sortType == GTK_SORT_ASCENDING ){
    menu = GTK_CHECK_MENU_ITEM( lookup_widget( widget,
                                "sort_ascending" ) );
  }
  else{
    menu = GTK_CHECK_MENU_ITEM( lookup_widget( widget,
                                "sort_descending" ) );
  }
  gtk_check_menu_item_set_active ( menu,
                                   TRUE );

  bookmark = lookup_widget( widget, "bookmark" );
  bookmark_menu = create_bookmark_menu( widget );
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (bookmark), bookmark_menu);
}


void
on_mainWindow_destroy                  (GtkObject       *object,
                                        gpointer         user_data)
{
  --windowCount;
  allWindows = g_list_remove( allWindows, object );
  if( 0 ==   windowCount )
    gtk_main_quit();
}


GtkWidget* get_current_file_browser ( GtkWidget* mainWindow )
{
  GtkNotebook* notebook = GTK_NOTEBOOK(
                        lookup_widget(mainWindow, "folderNoteBook") );
  gint idx = gtk_notebook_get_current_page( notebook );
  return gtk_notebook_get_nth_page( notebook, idx );
}

void
on_backBtn_clicked                     (GtkWidget          *mainWindow,
                                        GtkToolButton   *toolbutton)
{
  GtkWidget* file_browser = get_current_file_browser( mainWindow );
  if( file_browser_can_back( file_browser ) )
    file_browser_go_back( file_browser );
}


void
on_forwardBtn_clicked                  (GtkWidget          *mainWindow,
                                        GtkToolButton   *toolbutton)
{
  GtkWidget* file_browser = get_current_file_browser( mainWindow );
  if( file_browser_can_forward( file_browser ) )
    file_browser_go_forward( file_browser );
}


void
on_upBtn_clicked                       (GtkWidget          *mainWindow,
                                        GtkToolButton   *toolbutton)
{
  char *parent_dir;
  GtkWidget* file_browser = get_current_file_browser( mainWindow );

  parent_dir = g_path_get_dirname( file_browser_get_cwd( file_browser ) );
  file_browser_chdir( file_browser, parent_dir, TRUE );
  g_free( parent_dir );
}


void
on_homeBtn_clicked                     (GtkWidget          *mainWindow,
                                        GtkToolButton   *toolbutton)
{
  GtkWidget* file_browser = get_current_file_browser( mainWindow );
  file_browser_chdir( file_browser, g_get_home_dir(), TRUE );
}


void
on_addressBar_activate                 (GtkWidget        *mainWindow,
                                        GtkEntry         *entry)
{
  GtkWidget* file_browser = get_current_file_browser( mainWindow );
 FileBrowserData* data = file_browser_get_data( file_browser );
  file_browser_chdir( file_browser, gtk_entry_get_text( entry ), TRUE );
 gtk_widget_grab_focus( GTK_WIDGET( data->folderView ) );
}


void
on_refreshBtn_clicked                  (GtkToolButton   *toolbutton,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( toolbutton );
  GtkWidget* file_browser = get_current_file_browser( mainWindow );
  file_browser_refresh( file_browser );
}


void
on_quit_activate                       (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  close_window( GTK_WIDGET(menuitem) );
}

static void create_new_file( GtkWidget* mainWindow,
                             gboolean create_folder )
{
  GtkWidget* file_browser = get_current_file_browser( mainWindow );
  gchar* ufull_path;
  gchar* full_path;
  gchar* ufile_name;
  GtkLabel* prompt;
  int result;
  GtkWidget* dlg;
  if( create_folder ) {
    dlg = input_dialog_new( _("New Folder"),
                            _("Input a name for the new folder:"),
                            _("New Folder"),
                            GTK_WINDOW(mainWindow));
  }
  else  {
    dlg = input_dialog_new( _("New File"),
                            _("Input a name for the new file:"),
                            _("New File"),
                            GTK_WINDOW(mainWindow));
  }

  while( gtk_dialog_run( GTK_DIALOG(dlg)) == GTK_RESPONSE_OK )
  {
    ufile_name = input_dialog_get_text( dlg );
    ufull_path = g_build_filename(
                              file_browser_get_cwd(file_browser),
                              ufile_name, NULL );
    full_path = g_filename_from_utf8( ufull_path, -1,
                                      NULL, NULL, NULL );
    g_free( ufull_path );
    g_free( ufile_name );
    if( g_file_test( full_path, G_FILE_TEST_EXISTS ) )
    {
      prompt = GTK_LABEL( input_dialog_get_label( dlg ) );
      gtk_label_set_text( prompt,
                _("The file name you specified has existed.\n"
                  "Please input a new one:") );
      g_free( full_path );
      continue;
    }
    if( create_folder ) {
      result = g_mkdir( full_path, S_IRWXU|S_IRUSR|S_IRGRP );
    }
    else  {
      result = g_open( full_path, O_CREAT, S_IWUSR|S_IRUSR|S_IRGRP|S_IROTH );
      if( result != -1 )
        close( result );
    }
    g_free( full_path );
    break;
  }
  gtk_widget_destroy( dlg );

  if( result == -1 )
    errorMessage( GTK_WINDOW(mainWindow),
                  "The new file cannot be created!" );

}

void
on_new_folder_activate                 (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET(menuitem);
  create_new_file( mainWindow, TRUE );
}


void
on_new_text_file_activate              (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET(menuitem);
  create_new_file( mainWindow, FALSE );
}


void
on_preference_activate                 (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWindow* mainWindow = GTK_WINDOW( menuitem );
  show_preference_dialog( mainWindow );
}


void
on_about_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* dlg = create_aboutDialog();
  gtk_window_set_transient_for( GTK_WINDOW(dlg), GTK_WINDOW(menuitem)  );
  gtk_dialog_run( GTK_DIALOG(dlg) );
  gtk_widget_destroy( dlg );
}


gboolean
on_backBtn_popup_menu                  (GtkWidget       *widget,
                                        gpointer         user_data)
{
  GtkWidget* file_browser = get_current_file_browser( widget );

  return FALSE;
}


gboolean
on_forwardBtn_popup_menu               (GtkWidget       *widget,
                                        gpointer         user_data)
{
  GtkWidget* file_browser = get_current_file_browser( widget );

  return FALSE;
}



void
on_new_window_activate                 (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
/*  GtkWidget* mainWindow = GTK_WIDGET( menuitem );  */
  GtkWidget* new_win = create_mainWindow();
  gtk_widget_show( new_win );
  add_new_tab( new_win, g_get_home_dir(), appSettings.showDirTree );
}


void
on_new_tab_activate                    (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( menuitem );
  add_new_tab( mainWindow, g_get_home_dir(), appSettings.showDirTree);
}

void update_mainWindow_ui( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  GtkWidget* mainWindow = data->mainWindow;
  GtkEntry* addressBar = GTK_ENTRY( lookup_widget( mainWindow,
                                                   "addressBar") );
  GtkStatusbar* statusBar = GTK_STATUSBAR( lookup_widget( mainWindow,
                                                   "statusBar") );
  GtkNotebook* notebook = GTK_NOTEBOOK( lookup_widget( mainWindow,
                                                   "folderNoteBook") );
  GtkWidget* backBtn = lookup_widget(mainWindow, "backBtn");
  GtkWidget* forwardBtn = lookup_widget(mainWindow, "forwardBtn");

  GtkLabel* label;
  GtkWidget* hbox;
  gchar *msg;
  const char* path = file_browser_get_cwd(file_browser);
  if( !path )
    return;

  gtk_entry_set_text( addressBar, path );
  msg = g_strdup_printf( _("%d Files"), data->fileCount );
  gtk_statusbar_push( statusBar, 0, msg );
  g_free( msg );

  gtk_window_set_title( GTK_WINDOW(data->mainWindow), path );

  gtk_label_set_text( GTK_LABEL(data->tab_text), data->dir_name );

  gtk_widget_set_sensitive( backBtn,
                  file_browser_can_back( file_browser ));
  gtk_widget_set_sensitive( forwardBtn,
                  file_browser_can_forward( file_browser ));
}

static gboolean delayed_focus( GtkWidget* widget )
{
  gtk_widget_grab_focus( widget );
  return FALSE;
}

void
on_folderNoteBook_switch_page          (GtkNotebook     *notebook,
                                        GtkNotebookPage *page,
                                        guint            page_num,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( notebook );
  notebook = GTK_NOTEBOOK(user_data);
  GtkWidget* file_browser = gtk_notebook_get_nth_page( notebook, page_num );
  FileBrowserData* data = file_browser_get_data( file_browser );
  update_mainWindow_ui( file_browser );
  file_browser_update_mime_icons( file_browser );
  g_idle_add( (GSourceFunc)delayed_focus, data->folderView );
}


void
on_cut_activate                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET(menuitem);
  GtkWidget* file_browser;
  file_browser = get_current_file_browser( mainWindow );
  file_browser_cut( file_browser );
}


void
on_copy_activate                       (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET(menuitem);
  GtkWidget* file_browser;
  file_browser = get_current_file_browser( mainWindow );
  file_browser_copy( file_browser );
}


void
on_paste_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET(menuitem);
  GtkWidget* file_browser;
  file_browser = get_current_file_browser( mainWindow );
  file_browser_paste( file_browser );
}


void
on_delete_activate                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET(menuitem);
  GtkWidget* file_browser;
  file_browser = get_current_file_browser( mainWindow );
  file_browser_delete( file_browser );
}


void
on_select_all_activate                 (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( menuitem );
  GtkWidget* file_browser = get_current_file_browser( mainWindow );
  file_browser_select_all( file_browser );
}

void
on_edit_bookmark_activate            (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWindow* mainWindow = GTK_WINDOW(user_data);
  GtkWidget* bookmark;
  GtkWidget* bookmark_menu;
  GList* l;
  if( edit_bookmarks( mainWindow ) )
  {
    /* Re-create all bookmark menus */
    for( l = allWindows; l; l = l->next )
    {
      bookmark = lookup_widget( GTK_WIDGET(l->data), "bookmark" );
      bookmark_menu = create_bookmark_menu( GTK_WIDGET(l->data) );
      gtk_menu_item_set_submenu (GTK_MENU_ITEM (bookmark), bookmark_menu);
    }
  }
}


void
on_add_to_bookmark_activate            (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GList* l;
  GtkWidget* mainWindow = GTK_WIDGET(user_data);
  GtkWidget* bookmark;
  GtkWidget* bookmark_menu;
  GtkWidget* file_browser = get_current_file_browser(mainWindow);
  const char* path = file_browser_get_cwd( file_browser );

  if( ! g_list_find_custom( appSettings.bookmarks,
                            path,
                            (GCompareFunc)strcmp ) )
  {
    appSettings.bookmarks = g_list_append( appSettings.bookmarks,
                                           path );
    for( l = allWindows; l; l = l->next )
    {
      /*
      * FIXME: Reload all bookmark menus
      * This is very inefficient, and should be rewritten in the future.
      */
      bookmark = lookup_widget( GTK_WIDGET(l->data), "bookmark" );

      bookmark_menu = create_bookmark_menu( GTK_WIDGET(l->data) );
      gtk_menu_item_set_submenu (GTK_MENU_ITEM (bookmark), bookmark_menu);
    }
  }
}


void
on_invert_selection_activate           (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( menuitem );
  GtkWidget* file_browser = get_current_file_browser( mainWindow );
  file_browser_invert_selection( file_browser );
}


void
on_close_tab_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( menuitem );
  GtkNotebook* notebook = GTK_NOTEBOOK(lookup_widget( mainWindow, "folderNoteBook" ));
  GtkWidget* file_browser = get_current_file_browser( mainWindow );

  if ( gtk_notebook_get_n_pages (notebook) <= 1 )
  {
    close_window( mainWindow );
    return;
  }
  gint idx = gtk_notebook_page_num ( GTK_NOTEBOOK(notebook),
                                     file_browser);
  gtk_notebook_remove_page( notebook, idx );
  if( gtk_notebook_get_n_pages (notebook) == 1 )
    gtk_notebook_set_show_tabs( notebook, FALSE );
}


void
on_rename_activate                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET(menuitem);
  GtkWidget* file_browser = get_current_file_browser(mainWindow);
  file_browser_rename_selected_file( file_browser );
}


void
on_open_directory_tree_activate        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( menuitem );
  GtkCheckMenuItem* check = GTK_CHECK_MENU_ITEM(user_data);
  GtkWidget* file_browser;
  GtkNotebook* nb = lookup_widget(mainWindow, "folderNoteBook");
  GtkToggleToolButton* btn = GTK_TOGGLE_TOOL_BUTTON(lookup_widget(mainWindow,
                                                             "dirTreeBtn") );
  int i;
  int n = gtk_notebook_get_n_pages( nb );

  appSettings.showDirTree = gtk_check_menu_item_get_active( check );
  g_signal_handlers_block_matched ( btn, G_SIGNAL_MATCH_FUNC,
                                    0, 0, NULL, on_dirTreeBtn_toggled, NULL );
  gtk_toggle_tool_button_set_active( btn, appSettings.showDirTree);
  g_signal_handlers_unblock_matched ( btn, G_SIGNAL_MATCH_FUNC,
                                    0, 0, NULL, on_dirTreeBtn_toggled, NULL );

  for( i = 0; i < n; ++i )  {
    file_browser = gtk_notebook_get_nth_page( nb, i );
    file_browser_show_dir_tree( file_browser,
                                appSettings.showDirTree );
  }
}


void
on_show_hidden_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( menuitem );
  menuitem = GTK_MENU_ITEM( user_data );
  GtkWidget* file_browser = get_current_file_browser(mainWindow);

  appSettings.showHiddenFiles = gtk_check_menu_item_get_active(menuitem);
  if( file_browser )
    file_browser_show_hidden_files( file_browser,
                                    appSettings.showHiddenFiles );
}


void
on_sort_by_name_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( menuitem );
  GtkWidget* file_browser = get_current_file_browser(mainWindow);
  appSettings.sortOrder = FB_SORT_BY_NAME;
  if( file_browser )
    file_browser_sort_by_name( file_browser );
}


void
on_sort_by_size_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( menuitem );
  GtkWidget* file_browser = get_current_file_browser(mainWindow);
  appSettings.sortOrder = FB_SORT_BY_SIZE;
  if( file_browser )
    file_browser_sort_by_size( file_browser );
}


void
on_sort_by_time_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( menuitem );
  GtkWidget* file_browser = get_current_file_browser(mainWindow);
  appSettings.sortOrder = FB_SORT_BY_TIME;
  if( file_browser )
    file_browser_sort_by_time( file_browser );
}


void
on_sort_ascending_activate             (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( menuitem );
  GtkWidget* file_browser = get_current_file_browser(mainWindow);
  appSettings.sortType = GTK_SORT_ASCENDING;
  if( file_browser )
    file_browser_sort_ascending(file_browser);
}


void
on_sort_descending_activate            (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( menuitem );
  GtkWidget* file_browser = get_current_file_browser(mainWindow);
  appSettings.sortType = GTK_SORT_DESCENDING;
  if( file_browser )
    file_browser_sort_descending(file_browser);
}


void
on_view_as_icons_activate              (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( menuitem );
  GtkWidget* file_browser = get_current_file_browser(mainWindow);
  appSettings.viewMode = FBVM_ICON_VIEW;
  if( file_browser )
    file_browser_view_as_icons( file_browser );
}


void
on_view_as_list_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( menuitem );
  GtkWidget* file_browser = get_current_file_browser(mainWindow);
  appSettings.viewMode = FBVM_LIST_VIEW;
  if( file_browser )
    file_browser_view_as_list( file_browser );
}


void
on_dirTreeBtn_toggled                  (GtkToggleToolButton *toggletoolbutton,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = toggletoolbutton;
  GtkCheckMenuItem* menu = GTK_CHECK_MENU_ITEM( lookup_widget(mainWindow,
                                                    "open_directory_tree") );
  gtk_check_menu_item_set_active( menu, !appSettings.showDirTree );
}

static void on_bookmark_item_activate (GtkMenuItem* menu, gpointer user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET( user_data );
  GtkWidget* file_browser = get_current_file_browser( mainWindow );
  char* path = (char*)g_object_get_data( menu, "path" );
  if( !path )
    return;

  switch( appSettings.openBookmarkMethod )
  {
    case 1: /* current tab */
      file_browser_chdir( file_browser, path, TRUE );
      break;
    case 3: /* new window */
      mainWindow = create_mainWindow();
      gtk_widget_show( mainWindow );
      case 2: /* new tab */
      add_new_tab( mainWindow, path, appSettings.showDirTree );
      break;
  }
}

GtkWidget* create_bookmark_menu_item ( GtkWidget* mainWindow,
                                       const char* path,
                                       GdkPixbuf* icon )
{
  GtkWidget* folder_image;
  GtkWidget* menu_item;

  menu_item = gtk_image_menu_item_new_with_label(path);
  g_object_set_data( menu_item, "path", path );
  folder_image = gtk_image_new_from_pixbuf( icon );
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menu_item),
                                 folder_image);
  g_signal_connect( menu_item, "activate",
                    G_CALLBACK( on_bookmark_item_activate ), mainWindow );
  return menu_item;
}

GtkWidget* create_bookmark_menu ( GtkWidget* mainWindow )
{
  GtkWidget* bookmark_menu;
  GtkWidget* add_to_bookmark;
  GtkWidget* add_image;
  GtkWidget* edit_bookmark;
  GtkWidget* edit_image;
  GtkWidget* menu_item;
  GdkPixbuf* icon;
  GList* l;

  bookmark_menu = gtk_menu_new ();
  add_to_bookmark = gtk_image_menu_item_new_with_mnemonic (_("_Add to Bookmarks"));
  gtk_container_add (GTK_CONTAINER (bookmark_menu), add_to_bookmark);

  add_image = gtk_image_new_from_stock ("gtk-add", GTK_ICON_SIZE_MENU);
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (add_to_bookmark), add_image);

  g_signal_connect( add_to_bookmark, "activate",
                    G_CALLBACK( on_add_to_bookmark_activate ),
                    mainWindow );

  edit_bookmark = gtk_image_menu_item_new_with_mnemonic (_("_Edit Bookmarks"));
  gtk_container_add (GTK_CONTAINER (bookmark_menu), edit_bookmark);

  edit_image = gtk_image_new_from_stock ("gtk-edit", GTK_ICON_SIZE_MENU);
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (edit_bookmark), edit_image);

  g_signal_connect( edit_bookmark, "activate",
                    G_CALLBACK( on_edit_bookmark_activate ),
                    mainWindow );


  menu_item = gtk_separator_menu_item_new ();
  gtk_container_add (GTK_CONTAINER (bookmark_menu), menu_item);

  icon = get_folder_icon16();
  for( l = appSettings.bookmarks; l; l = l->next ) {
    menu_item = create_bookmark_menu_item( mainWindow,
                                           (char*)l->data,
                                           icon );
    gtk_container_add (GTK_CONTAINER (bookmark_menu), menu_item);
  }
  gtk_widget_show_all( bookmark_menu );
  return bookmark_menu;
}



void
on_go_btn_clicked                      (GtkToolButton   *toolbutton,
                                        gpointer         user_data)
{
  GtkWidget* mainWindow = GTK_WIDGET(toolbutton);
  on_addressBar_activate(mainWindow,
                         GTK_ENTRY(lookup_widget(mainWindow, "addressBar")));
}

