#include "filebrowser.h"
#include <gtk/gtk.h>
#include <string.h>

#include "glade_support.h"

#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <glib/gstdio.h>
#include <string.h>

#include <sys/types.h>
#include <grp.h> /* Query group name */
#include <pwd.h> /* Query user name */
#include <time.h>

/* MIME type processing library from FreeDesktop.org */
#include "xdgmime.h"

#include "mainwindow.h"
#include "mainwindowui.h"

#include "filepropertiesui.h"
#include "fileproperties.h"
#include "fileicon.h"
#include "mimeaction.h"
#include "inputdialog.h"
#include "foldercontent.h"
#include "fileoperation.h"
#include "mimedescription.h"
#include "appchooserdlg.h"

#include "filebrowser.h"
#include "ptktreemodelsort.h"

#include "settings.h"

static GQuark FileBrowserDataId = 0;

static GdkDragAction clipboard_action = GDK_ACTION_DEFAULT;

static guint folder_view_auto_scroll_timer = 0;
static GtkDirectionType folder_view_auto_scroll_direction = 0;

/*  Drag & Drop/Clipboard targets  */
static GtkTargetEntry dragTargets[]={
  {"text/uri-list", 0 , 0 }
};

/* If set to FALSE, all selection change in folderView is prevented. */
static gboolean can_folder_view_sel_change = TRUE;

/* Utility functions */
static gboolean update_folder_view_visible_region( FileBrowserData* data );

static GtkWidget* create_folder_view( FileBrowserViewMode view_mode,
                                      FileBrowserData* data );

static void init_list_view( GtkTreeView* list_view );

static void init_dir_tree( FileBrowserData* data );

static void file_browser_create_dir_tree( FileBrowserData* data );

static GList* folder_view_get_selected_items( FileBrowserData* data,
                                              GtkTreeModel** model );

static gpointer file_browser_chdir_thread( FileBrowserData* data );

static gboolean file_browser_chdir_complete( FileBrowserData* data );

/* Return a list of selected filenames (in full path, in UTF-8)
*  The returned list should be freed by calling:
*    g_list_foreach( file_list, (GFunc)g_free, NULL );
*    g_list_free( file_list );
*/
static GList*
file_browser_get_selected_files( FileBrowserData* data );

static void file_browser_open_selected_files_with_app( FileBrowserData* data,
                                                       const char* app_desktop );


static GtkWidget* file_browser_get_popup_menu_for_mime_type( FileBrowserData* data,
                                                             const char* mime_type );

/* Get GtkTreePath of the item at coordinate x, y */
static GtkTreePath*
folder_view_get_tree_path_at_pos( FileBrowserData* data, int x, int y );

/* sort functions of folder view */
static gint sort_files_by_name  (GtkTreeModel *model,
                           GtkTreeIter *a,
                           GtkTreeIter *b,
                           gpointer user_data);

static gint sort_files_by_size  (GtkTreeModel *model,
                                 GtkTreeIter *a,
                                 GtkTreeIter *b,
                                 gpointer user_data);

static gint sort_files_by_time  (GtkTreeModel *model,
                                 GtkTreeIter *a,
                                 GtkTreeIter *b,
                                 gpointer user_data);


static gboolean filter_files  (GtkTreeModel *model,
                               GtkTreeIter *it,
                               gpointer user_data);

/* sort functions of dir tree */
static gint sort_dir_tree_by_name  (GtkTreeModel *model,
                                    GtkTreeIter *a,
                                    GtkTreeIter *b,
                                    gpointer user_data);

/* signal handlers */
static void
on_folder_view_item_activated           (PtkIconView     *iconview,
                                         GtkTreePath     *path,
                                         FileBrowserData* data);
static void
on_folder_view_row_activated           (GtkTreeView     *tree_view,
                                        GtkTreePath     *path,
                                        GtkTreeViewColumn* col,
                                        FileBrowserData* data);
static void
on_folder_view_item_sel_change (PtkIconView *iconview,
                               FileBrowserData* data);
static gboolean
on_folder_view_key_press_event          (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        FileBrowserData* data);
static gboolean
on_folder_view_button_press_event       (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        FileBrowserData* data);
static gboolean
on_folder_view_button_release_event       (GtkWidget       *widget,
                                          GdkEventButton  *event,
                                          FileBrowserData* data);
static void
on_folder_viewScroll_scrolled       (GtkAdjustment *adjust,
                                        FileBrowserData* data);
static void
on_folder_view_size_allocate            (GtkWidget       *widget,
                                        GdkRectangle    *allocation,
                                        FileBrowserData* data);
static void
on_dir_tree_sel_changed           (GtkTreeSelection *treesel,
                                        FileBrowserData* data);
static void
on_dir_tree_row_expanded        (GtkTreeView     *treeview,
                                 GtkTreeIter     *iter,
                                 GtkTreePath     *path,
                                 FileBrowserData* data);

static void
on_folder_view_drag_data_received (GtkWidget        *widget,
                                  GdkDragContext   *drag_context,
                                  gint              x,
                                  gint              y,
                                  GtkSelectionData *data,
                                  guint             info,
                                  guint             time,
                                  gpointer          user_data);

static void
on_folder_view_drag_data_get (GtkWidget        *widget,
                                  GdkDragContext   *drag_context,
                                  GtkSelectionData *sel_data,
                                  guint             info,
                                  guint             time,
                                  FileBrowserData  *data);

static gboolean
on_tab_drag_motion (GtkWidget     *widget,
                                  GdkDragContext *drag_context,
                                  gint            x,
                                  gint            y,
                                  guint           time,
                                  FileBrowserData* data );


static void
on_folder_view_drag_begin           (GtkWidget      *widget,
                                    GdkDragContext *drag_context,
                                    gpointer        user_data);
static void
on_folder_view_drag_begin           (GtkWidget      *widget,
                                    GdkDragContext *drag_context,
                                    gpointer        user_data);

static gboolean
on_folder_view_drag_motion (GtkWidget      *widget,
                            GdkDragContext *drag_context,
                            gint            x,
                            gint            y,
                            guint           time,
                            FileBrowserData* data );

static gboolean
on_folder_view_drag_leave ( GtkWidget      *widget,
                            GdkDragContext *drag_context,
                            guint           time,
                            FileBrowserData* data );
static gboolean
on_folder_view_drag_drop    (GtkWidget      *widget,
                            GdkDragContext *drag_context,
                            gint            x,
                            gint            y,
                            guint           time,
                            FileBrowserData* data );
static void
on_folder_view_drag_end             (GtkWidget      *widget,
                                    GdkDragContext *drag_context,
                                    gpointer        user_data);


/* Signal handlers for popup menu */
static void
on_popup_open_activate                       (GtkMenuItem     *menuitem,
                                            gpointer         user_data);
static void
on_popup_open_with_another_activate          (GtkMenuItem     *menuitem,
                                            gpointer         user_data);
static void
on_file_property_activate              (GtkMenuItem     *menuitem,
                                            gpointer         user_data);
static void
on_popup_run_app                       (GtkMenuItem     *menuitem,
                                        FileBrowserData* data);
static void
on_popup_open_in_new_tab_activate      ( GtkMenuItem *menuitem,
                                         FileBrowserData* data );
static void
on_popup_cut_activate                        (GtkMenuItem     *menuitem,
                                              gpointer         user_data);
static void
on_popup_copy_activate                       (GtkMenuItem     *menuitem,
                                              gpointer         user_data);
static void
on_popup_paste_activate                      (GtkMenuItem     *menuitem,
                                              gpointer         user_data);
static void
on_popup_delete_activate                     (GtkMenuItem     *menuitem,
                                              gpointer         user_data);
static void
on_popup_rename_activate                      (GtkMenuItem     *menuitem,
                                               gpointer         user_data);
static void
on_popup_new_folder_activate                  (GtkMenuItem     *menuitem,
                                               gpointer         user_data);
static void
on_popup_new_text_file_activate               (GtkMenuItem     *menuitem,
                                               gpointer         user_data);
static void
on_popup_file_property_activate               (GtkMenuItem     *menuitem,
                                               gpointer         user_data);



/* Destructor of FileBrowserData */

static void file_browser_data_free( FileBrowserData* data )
{
  if( data->folder_content )
    folder_content_list_unref( data->folder_content );
  g_free( data );
}

/* File Browser API */

FileBrowserData* file_browser_new( GtkWidget* mainWindow,
                                   FileBrowserViewMode view_mode,
                                   gboolean open_dir_tree,
                                   gboolean create_tab_label )
{
  GtkWidget* folderView;
  GtkTreeSelection* tree_sel;

  GtkAdjustment *adjust;
  GtkWidget* close_btn;
  GtkWidget* evt_box;

  FileBrowserData* data = g_new0(FileBrowserData, 1);
  data->mainWindow = mainWindow;
  data->folderPane = gtk_hpaned_new ();

  if( ! FileBrowserDataId )
    FileBrowserDataId = g_quark_from_static_string("FileBrowserData");

  g_object_set_qdata_full(
                G_OBJECT(data->folderPane),
                FileBrowserDataId,
                data,
                (GDestroyNotify)file_browser_data_free );

  gtk_paned_set_position (GTK_PANED (data->folderPane), 160);

  if( open_dir_tree ) {
    file_browser_create_dir_tree( data );
  }

  data->folderViewScroll = gtk_scrolled_window_new (NULL, NULL);
  gtk_paned_pack2 (GTK_PANED (data->folderPane),
                        data->folderViewScroll, TRUE, TRUE);
  gtk_scrolled_window_set_policy (
                        GTK_SCROLLED_WINDOW (data->folderViewScroll),
                        GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  gtk_scrolled_window_set_shadow_type (
                        GTK_SCROLLED_WINDOW (data->folderViewScroll),
                        GTK_SHADOW_IN);

  data->view_mode = view_mode;
  folderView = create_folder_view( view_mode, data );
  data->folderView = folderView;

  gtk_container_add (GTK_CONTAINER (data->folderViewScroll),
                     folderView);
  gtk_widget_show_all (data->folderPane);

  adjust = gtk_scrolled_window_get_vadjustment(
                      GTK_SCROLLED_WINDOW(data->folderViewScroll) );
  g_signal_connect( G_OBJECT(adjust), "value-changed",
                  G_CALLBACK(on_folder_viewScroll_scrolled), data );

  adjust = gtk_scrolled_window_get_hadjustment(
                      GTK_SCROLLED_WINDOW(data->folderViewScroll) );
  g_signal_connect( G_OBJECT(adjust), "value-changed",
                  G_CALLBACK(on_folder_viewScroll_scrolled), data );


/* Create tab label */
  if( create_tab_label ){
    evt_box = gtk_event_box_new ();
    gtk_event_box_set_visible_window (GTK_EVENT_BOX(evt_box), FALSE);

    data->tab_label = gtk_hbox_new( FALSE, 0 );

    data->tab_icon = gtk_image_new_from_pixbuf (get_folder_icon16());
    gtk_box_pack_start( GTK_BOX(data->tab_label),
                        data->tab_icon, FALSE, FALSE, 4 );

    data->tab_text = gtk_label_new( "" );
    gtk_box_pack_start( GTK_BOX(data->tab_label),
                        data->tab_text, FALSE, FALSE, 4 );

    close_btn = gtk_button_new ();
    gtk_button_set_focus_on_click (GTK_BUTTON (close_btn), FALSE);
    gtk_button_set_relief( GTK_BUTTON (close_btn), GTK_RELIEF_NONE );
    gtk_container_add (GTK_CONTAINER (close_btn),
      gtk_image_new_from_icon_name("gtk-close", GTK_ICON_SIZE_MENU) );
    gtk_widget_set_size_request ( close_btn, 20, 20);
    gtk_box_pack_start ( GTK_BOX(data->tab_label),
                         close_btn,FALSE,FALSE, 0 );
    data->close_btn = close_btn;

    gtk_widget_show_all( data->tab_label );

    gtk_container_add (GTK_CONTAINER (evt_box), data->tab_label );
    data->tab_label = evt_box;

    gtk_widget_set_events ( evt_box, GDK_ALL_EVENTS_MASK );
    gtk_drag_dest_set ( evt_box, GTK_DEST_DEFAULT_ALL,
        dragTargets,
        sizeof(dragTargets)/sizeof(GtkTargetEntry),
        GDK_ACTION_DEFAULT|GDK_ACTION_COPY|GDK_ACTION_MOVE|GDK_ACTION_LINK);
    g_signal_connect ((gpointer) evt_box, "drag-motion",
                            G_CALLBACK (on_tab_drag_motion),
                            data );
  }

  return data;
}


guint initial_update_folder( FileBrowserData* data )
{
  GtkTreePath *first_iter_path = gtk_tree_path_new_from_indices(0, -1);
  if( first_iter_path )
  {
    if( data->view_mode == FBVM_ICON_VIEW ) {
      ptk_icon_view_scroll_to_path( PTK_ICON_VIEW(data->folderView),
                                    first_iter_path, TRUE, 0, 0 );
    }
    else if( data->view_mode == FBVM_LIST_VIEW ) {
      gtk_tree_view_scroll_to_cell ( GTK_TREE_VIEW(data->folderView),
                                    first_iter_path,
                                    NULL, TRUE, 0, 0 );
    }
    gtk_tree_path_free( first_iter_path );
  }
  gtk_widget_show( GTK_WIDGET(data->folderView) );
  return FALSE;
}

static gboolean dir_tree_chdir( FileBrowserData* data,
                                const char* folder_path )
{
  GtkTreeView* dirTree = data->dirTree;
  GtkTreeStore* tree;
  GtkTreeModel* model;
  GtkTreeIter it, parent_it;
  GtkTreePath* tree_path = NULL;
  gchar *name;
  gchar **dirs, **dir;
  gboolean found;

  if( !folder_path || *folder_path != '/' )
    return FALSE;

  dirs = g_strsplit( folder_path + 1, "/", -1 );

  if( !dirs )
    return FALSE;

  model = gtk_tree_view_get_model( dirTree );

  if( ! gtk_tree_model_iter_children ( model, &parent_it, NULL ) )
    return FALSE;

  /* special case: root dir */
  if( ! dirs[0] ) {
    it = parent_it;
    tree_path = gtk_tree_model_get_path ( model, &parent_it );
    goto _found;
  }

  for( dir = dirs; *dir; ++dir ){
    if( ! gtk_tree_model_iter_children ( model, &it, &parent_it ) )
      return FALSE;

    found = FALSE;
    if( tree_path )
      gtk_tree_path_free( tree_path );
    tree_path = gtk_tree_model_get_path( model, &it );
    do{
      if( !gtk_tree_model_get_iter( model, &it, tree_path ) )
        break;
      gtk_tree_model_get( model, &it, COL_DIRTREE_TEXT, &name, -1 );

      if(!name)
        continue;
      if( 0 == strcmp( name, *dir ) ){
        gtk_tree_view_expand_row (dirTree, tree_path, FALSE);
        gtk_tree_model_get_iter( model, &parent_it, tree_path );
        g_free( name );
        found = TRUE;
        break;
      }
      g_free( name );
      gtk_tree_path_next (tree_path);
    }while( TRUE );
    if( ! found ){
      return FALSE; /* Error! */
    }
  }
_found:
  g_strfreev( dirs );
  gtk_tree_selection_select_path (
          gtk_tree_view_get_selection(dirTree), tree_path );

  /*
  gtk_tree_view_scroll_to_cell ( dirTree, tree_path, 0, TRUE, 0.5, 0.5 );
  */
  gtk_tree_path_free( tree_path );
  return TRUE;
}

/*
static void folderView_chdir( FileBrowserData* data,
                              const char* folder_path )
{
  PtkIconView* folderView = data->folderView;
}
*/

void file_browser_chdir( GtkWidget* file_browser,
                         const char* folder_path,
                         gboolean  addHistory )
{
  FileBrowserData* data = file_browser_get_data(file_browser);

  GtkWidget* folderView = data->folderView;
  GtkTreeView* dirTree = data->dirTree;
  GtkEntry* addressBar;
  GtkStatusbar* statusBar;
  GdkCursor* cursor;

  char* path_end;
  int ctxid = 0;
  char *cwd;
  char* upath = folder_path? strdup( folder_path ) : NULL;
  char* path;
  gboolean is_current;

  FolderContent* folder_content;

  if( ! upath )
    return;

  /* remove redundent '/' */
  if( strcmp( upath, "/" ) )
  {
    path_end = upath + strlen(upath) - 1;
    for( ; path_end > path; --path_end ){
      if( *path_end != '/' )
        break;
      else
        *path_end = '\0';
    }
  }

  path = g_filename_from_utf8( upath, -1, NULL, NULL, NULL );
  if( ! g_file_test( path, (G_FILE_TEST_IS_DIR) ) )
  {
    errorMessage( GTK_WINDOW(data->mainWindow), _("Directory does'nt exist!") );
    g_free( path );
    g_free( upath );
    return;
  }
  else
    g_free( path );

  is_current = (get_current_file_browser(data->mainWindow)==file_browser);

  if( addHistory ){
    if( ! data->curHistory || strcmp( data->curHistory->data, upath ) ){
      /* Has forward history */
      if( data->curHistory && data->curHistory->next )  {
        g_list_foreach ( data->curHistory->next, (GFunc)g_free, NULL );
        g_list_free( data->curHistory->next );
        data->curHistory->next = NULL;
      }
      /* Add path to history if there is no forward history */
      data->history = g_list_append( data->history, upath );
      data->curHistory = g_list_last( data->history );
    }
  }

  if( data->view_mode == FBVM_ICON_VIEW ) {
    ptk_icon_view_set_model( PTK_ICON_VIEW(folderView), NULL );
  }
  else if( data->view_mode == FBVM_LIST_VIEW ) {
    gtk_tree_view_set_model( GTK_TREE_VIEW(folderView), NULL );
  }
  if( data->folder_content )
    folder_content_list_unref( data->folder_content );

  data->dir_name = strrchr( (char*)data->curHistory->data, '/' );
  if( data->dir_name[1] != '\0' )
    ++data->dir_name;

  /* Update UI */
  if( is_current ){
    addressBar = GTK_ENTRY( lookup_widget( data->mainWindow,
                            "addressBar") );
    statusBar = GTK_STATUSBAR( lookup_widget( data->mainWindow,
                               "statusBar") );
    /* Update UI */
    update_mainWindow_ui( file_browser );
    gtk_statusbar_push( statusBar, 0, _("Loading..."));
  }

  if( ! addHistory )
    g_free( upath );

  cursor = gdk_cursor_new( GDK_WATCH );
  gdk_window_set_cursor ( data->mainWindow->window, cursor );
  gdk_cursor_unref( cursor );
  gtk_widget_set_sensitive( data->folderPane, FALSE );

  g_thread_create ( (GThreadFunc)file_browser_chdir_thread,
                     data, FALSE, NULL );
}

gpointer file_browser_chdir_thread( FileBrowserData* data )
{
  char* upath = (char*)data->curHistory->data;
  char* path;

  path = g_filename_from_utf8( upath, -1, NULL, NULL, NULL );
  data->folder_content = folder_content_list_get( path );
  g_free( path );

  g_idle_add( (GSourceFunc)file_browser_chdir_complete, data );
}

gboolean file_browser_chdir_complete( FileBrowserData* data )
{
  GtkTreeSortable* sortable;
  gboolean is_current;
  GtkTreeIterCompareFunc sort_funcs[] = { &sort_files_by_name,
                                          &sort_files_by_size,
                                          &sort_files_by_time };
  int sort_col;

  is_current = (get_current_file_browser(data->mainWindow)==data->folderPane);

  /* Set up a sorter to folder content model */
  if( data->list_sorter )
    g_object_unref( G_OBJECT(data->list_sorter) );

  data->list_sorter = ptk_tree_model_sort_new_with_model(
      GTK_TREE_MODEL(data->folder_content->list ));

  sortable = GTK_TREE_SORTABLE(data->list_sorter);
  gtk_tree_sortable_set_sort_func( sortable, COL_FILE_NAME,
                                   sort_funcs[ data->sort_order ], data,
                                   NULL );

  if(data->sort_order == FB_SORT_BY_NAME) {
    sort_col = COL_FILE_NAME;
  }
  else {
    COL_FILE_STAT;
  }
  gtk_tree_sortable_set_sort_column_id( sortable,
                                        sort_col,
                                        data->sort_type );

  /* Set up a filter to folder content model */
  if( data->list_filter )
    g_object_unref( G_OBJECT(data->list_filter) );

  data->list_filter = gtk_tree_model_filter_new(
      data->list_sorter, NULL );
  gtk_tree_model_filter_set_visible_func(
      GTK_TREE_MODEL_FILTER(data->list_filter),
  filter_files, data, NULL );

  data->fileCount = gtk_tree_model_iter_n_children(
      data->list_filter, NULL );

  if( data->view_mode == FBVM_ICON_VIEW ) {
    ptk_icon_view_set_model( PTK_ICON_VIEW(data->folderView),
                             GTK_TREE_MODEL( data->list_filter ) );
  }
  else if( data->view_mode == FBVM_LIST_VIEW ) {
    gtk_tree_view_set_model( GTK_TREE_VIEW(data->folderView),
                             GTK_TREE_MODEL( data->list_filter ) );
  }

  /* If dir tree is displayed */
  if( data->show_dir_tree ){
    init_dir_tree( data );
    dir_tree_chdir( data, (char*)data->curHistory->data );
  }

  gdk_window_set_cursor ( data->mainWindow->window, NULL );
  gtk_widget_set_sensitive( data->folderPane, TRUE );

  /* Update UI */
  if( is_current ) {
    update_mainWindow_ui( data->folderPane );
    gtk_widget_grab_focus( data->folderView );
  }

  g_idle_add( (GSourceFunc)initial_update_folder, data );
  return FALSE;
}

const char* file_browser_get_cwd( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  if( ! data->curHistory )
    return NULL;
  return (const char*)data->curHistory->data;
}

gboolean file_browser_can_back( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  /* there is back history */
  return ( data->curHistory && data->curHistory->prev );
}

void file_browser_go_back( GtkWidget* file_browser )
{
  const char* path;
  FileBrowserData* data = file_browser_get_data( file_browser );
  /* there is no back history */
  if ( ! data->curHistory || ! data->curHistory->prev )
    return;
  path = (const char*)data->curHistory->prev->data;
  data->curHistory = data->curHistory->prev;
  file_browser_chdir( file_browser, path, FALSE );
}

gboolean file_browser_can_forward( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  /* If there is forward history */
  return ( data->curHistory && data->curHistory->next );
}

void file_browser_go_forward( GtkWidget* file_browser )
{
  const char* path;
  FileBrowserData* data = file_browser_get_data( file_browser );
  /* If there is no forward history */
  if ( ! data->curHistory || ! data->curHistory->next )
    return;
  path = (const char*)data->curHistory->next->data;
  data->curHistory = data->curHistory->next;
  file_browser_chdir( file_browser, path, FALSE );
}

PtkIconView* file_browser_get_folder_view( GtkWidget* file_browser )
{

}

GtkTreeView* file_browser_get_dir_tree( GtkWidget* file_browser )
{

}

FileBrowserData* file_browser_get_data( GtkWidget* file_browser )
{
  return (FileBrowserData*)g_object_get_qdata(
                                                        G_OBJECT(file_browser),
                                                        FileBrowserDataId );
}

void file_browser_select_all( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  GtkTreeSelection* tree_sel;
  if( data->view_mode == FBVM_ICON_VIEW ) {
    ptk_icon_view_select_all( PTK_ICON_VIEW(data->folderView) );
  }
  else if( data->view_mode == FBVM_LIST_VIEW ) {
    tree_sel = gtk_tree_view_get_selection( GTK_TREE_VIEW(data->folderView) );
    gtk_tree_selection_select_all( tree_sel );
  }
}

static gboolean
invert_selection                    (GtkTreeModel* model,
                                     GtkTreePath *path,
                                     GtkTreeIter* it,
                                     FileBrowserData* data )
{
  GtkTreeSelection* tree_sel;
  if( data->view_mode == FBVM_ICON_VIEW ) {
    if( ptk_icon_view_path_is_selected ( PTK_ICON_VIEW(data->folderView), path ) )
      ptk_icon_view_unselect_path ( PTK_ICON_VIEW(data->folderView), path );
    else
      ptk_icon_view_select_path ( PTK_ICON_VIEW(data->folderView), path );
  }
  else if( data->view_mode == FBVM_LIST_VIEW ) {
    tree_sel = gtk_tree_view_get_selection( GTK_TREE_VIEW(data->folderView) );
    if( gtk_tree_selection_path_is_selected ( tree_sel, path ) )
      gtk_tree_selection_unselect_path ( tree_sel, path );
    else
      gtk_tree_selection_select_path ( tree_sel, path );
  }
  return FALSE;
}

void file_browser_invert_selection( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  GtkTreeModel* model;
  if( data->view_mode == FBVM_ICON_VIEW ) {
    model = ptk_icon_view_get_model( PTK_ICON_VIEW(data->folderView) );
  }
  else if( data->view_mode == FBVM_LIST_VIEW ) {
    model = gtk_tree_view_get_model( GTK_TREE_VIEW(data->folderView) );
  }
  gtk_tree_model_foreach ( model,
   (GtkTreeModelForeachFunc)invert_selection, data );
}

/* signal handlers */

static gboolean item_activated( FileBrowserData* data )
{
  file_browser_open_selected_files_with_app( data, NULL );
  return FALSE;
}

void
on_folder_view_item_activated           (PtkIconView     *iconview,
                                         GtkTreePath     *path,
                                         FileBrowserData* data)
{
  g_idle_add( (GSourceFunc)item_activated, data );
}

void
on_folder_view_row_activated           (GtkTreeView     *tree_view,
                                        GtkTreePath     *path,
                                        GtkTreeViewColumn* col,
                                        FileBrowserData* data)
{
  g_idle_add( (GSourceFunc)item_activated, data );
}

static gboolean on_folder_view_update_sel( FileBrowserData* data )
{
  GList* sel_files;
  GList* sel;
  GtkTreeIter it;
  GtkTreeSelection* tree_sel;
  GtkTreeModel* model;
  struct stat* pstat;

  data->n_sel_files = 0;
  data->sel_size = 0;

  sel_files = folder_view_get_selected_items( data, &model );

  for( sel = sel_files; sel; sel = g_list_next( sel ) )
  {
    if( gtk_tree_model_get_iter( model, &it, (GtkTreePath*)sel->data ) )
    {
//      gtk_tree_model_get( model, &it, COL_FILE_STAT, &pstat, -1 );
//      if( pstat ){
//        data->sel_size += pstat->st_size;
//      }
      ++data->n_sel_files;
    }
  }

  g_list_foreach( sel_files,
                  (GFunc)gtk_tree_path_free,
                  NULL );
  g_list_free( sel_files );
  return FALSE;
}

/*
* I don't know why, but calculate selected items in this handler will
* cause crash. So I delay the operation and put it in an idle handler.
*/
void on_folder_view_item_sel_change (PtkIconView *iconview,
                                     FileBrowserData* data)
{
  g_idle_add( (GSourceFunc)on_folder_view_update_sel, data );
}


gboolean
on_folder_view_key_press_event          (GtkWidget       *widget,
                                         GdkEventKey     *event,
                                         FileBrowserData* data)
{

  return FALSE;
}



gboolean
on_folder_view_button_release_event       (GtkWidget       *widget,
                                          GdkEventButton  *event,
                                          FileBrowserData* data)
{
  GtkTreeSelection* tree_sel;
  GtkTreePath* tree_path;
  if( data->view_mode == FBVM_LIST_VIEW && !can_folder_view_sel_change ) {
    can_folder_view_sel_change = TRUE;
    tree_sel = gtk_tree_view_get_selection( GTK_TREE_VIEW(widget) );
    gtk_tree_view_get_path_at_pos( GTK_TREE_VIEW(widget),
                                   event->x, event->y, &tree_path, NULL, NULL, NULL);
    if( tree_path ) {
      if( gtk_tree_selection_path_is_selected( tree_sel, tree_path ) ) {
        gtk_tree_selection_unselect_all( tree_sel );
        gtk_tree_selection_select_path( tree_sel, tree_path );
      }
      gtk_tree_path_free( tree_path );
    }
  }
  return FALSE;
}

gboolean
on_folder_view_button_press_event       (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        FileBrowserData* data)
{
  GtkTreeModel* model;
  GtkTreePath *tree_path;
  GtkTreeIter it;
  gchar *file_name;
  const gchar *mime_type;
  gchar* dir_path;
  struct stat *file_stat;
  GtkWidget* popup = NULL;
  GtkTreeSelection* tree_sel;

  if( event->type == GDK_BUTTON_PRESS )
  {
    if( data->view_mode == FBVM_ICON_VIEW ) {
      tree_path = ptk_icon_view_get_path_at_pos( PTK_ICON_VIEW(widget),
                                                event->x, event->y);

      if( !tree_path && event->button == 3 ){
        ptk_icon_view_unselect_all ( PTK_ICON_VIEW(widget) );
      }

      model = ptk_icon_view_get_model( PTK_ICON_VIEW(widget));
    }
    else if( data->view_mode == FBVM_LIST_VIEW ) {
      model = gtk_tree_view_get_model( GTK_TREE_VIEW(widget));
      gtk_tree_view_get_path_at_pos( GTK_TREE_VIEW(widget),
                   event->x, event->y, &tree_path, NULL, NULL, NULL);
      tree_sel = gtk_tree_view_get_selection( GTK_TREE_VIEW(widget) );

      /* If click on a selected row */
      if( tree_path &&
          event->type == GDK_BUTTON_PRESS &&
          !(event->state & (GDK_SHIFT_MASK|GDK_CONTROL_MASK)) &&
          gtk_tree_selection_path_is_selected (tree_sel, tree_path) )  {
        can_folder_view_sel_change = FALSE;
      }
      else
        can_folder_view_sel_change = TRUE;
    }

    /* middle button */
    if( event->button == 2 && tree_path )
    {
      gtk_tree_model_get_iter( model, &it, tree_path );
      gtk_tree_model_get( model, &it,
                              COL_FILE_NAME, &file_name,
                              COL_FILE_STAT, &file_stat,
                              COL_FILE_TYPE, &mime_type, -1 );
      if( S_ISDIR(file_stat->st_mode)  )  {
        dir_path = g_build_filename( (char*)data->curHistory->data,
                                                      file_name, NULL );
        add_new_tab( data->mainWindow, dir_path, (data->dirTree != NULL) );
        g_free( dir_path );
      }
      g_free( file_name );
      /*g_free( mime_type ); is not needed, since COL_FILE_TYPE
        is stored with "G_POINTER" type, not G_STRING */
    }
    else if( event->button == 3 )
    {
      if( data->view_mode == FBVM_ICON_VIEW ) {
        if( tree_path &&
            !ptk_icon_view_path_is_selected (PTK_ICON_VIEW(widget),
                                             tree_path) )
        {
          ptk_icon_view_unselect_all ( PTK_ICON_VIEW(widget) );
          ptk_icon_view_select_path( PTK_ICON_VIEW(widget), tree_path );
        }
      }

      if( tree_path )
      {
        gtk_tree_model_get_iter( model, &it, tree_path );
        gtk_tree_model_get( model, &it,
                            COL_FILE_TYPE, &mime_type, -1 );
      }
      else
      {
        mime_type = XDG_MIME_TYPE_DIRECTORY;
      }
      popup = file_browser_get_popup_menu_for_mime_type(data, mime_type);
      gtk_menu_popup( GTK_MENU(popup), NULL, NULL,
                      NULL, NULL, 3, event->time );
    }

    if( tree_path );
      gtk_tree_path_free( tree_path );
  }
  return FALSE;
}

void
on_folder_viewScroll_scrolled       (GtkAdjustment *adjust,
                                    FileBrowserData* data)
{
  update_folder_view_visible_region(data);
  /*
  * Because of some strange behavior of gtk+, I have to install an idle
  * handler here to ensure all visible items are really upadted.
  * FIXME: This is inefficient and should be improved in the future.
  */
  g_idle_add( (GSourceFunc)update_folder_view_visible_region, data );
}

void
on_folder_view_size_allocate            (GtkWidget       *widget,
                                        GdkRectangle    *allocation,
                                        FileBrowserData* data)
{
  GtkTreeModel* list;
  if( data->view_mode == FBVM_ICON_VIEW ) {
    list = ptk_icon_view_get_model( PTK_ICON_VIEW(data->folderView) );
  }
  else if( data->view_mode == FBVM_LIST_VIEW ) {
    list = gtk_tree_view_get_model( GTK_TREE_VIEW(data->folderView) );
  }
  if( list )    /* If the folder has been loaded, update visible region */
    g_idle_add( (GSourceFunc)update_folder_view_visible_region, data );
}

char* dir_path_from_tree_path( GtkTreeModel* model, GtkTreePath* path )
{
  gint* idx = gtk_tree_path_get_indices( path );
  gint depth = gtk_tree_path_get_depth( path );
  GtkTreeIter it, subit, child, *parent;
  char dir_path[1024] = "";
  gchar* name;
  int i;

  if( !idx )
    return NULL;

  for( i = 0, parent = NULL; i < depth; ++i )
  {
    gtk_tree_model_iter_nth_child( model, &child, parent, idx[i] );
    name = NULL;
    gtk_tree_model_get( model, &child, COL_DIRTREE_TEXT, &name, -1 );
    if( ! name )
      return NULL;

    if( i > 1 )
      strcat( dir_path, "/" );
    strcat( dir_path, name );
    g_free( name );

    subit = child;
    parent = &subit;
  }
  return g_strdup( dir_path );
}

void on_dir_tree_row_expanded        (GtkTreeView     *treeview,
                                      GtkTreeIter     *iter,
                                      GtkTreePath     *tree_path,
                                      FileBrowserData* data)
{
  gint* idx;
  GtkTreeIter it, subit, child, *parent;
  GtkTreeRowReference* row_ref, *del_ref;
  FolderContent* content;
  GtkTreePath* real_path;

  char* dir_path = NULL;
  gchar* name;
  GtkTreeModel* model;
  GtkTreeModel* store;
  GtkTreeModelFilter* filter;

  model = gtk_tree_view_get_model(treeview );
  filter = GTK_TREE_MODEL_FILTER(model);

  if( !gtk_tree_model_iter_children( model, &child, iter ) ) {
    return;
  }
  gtk_tree_model_get( model, &child, COL_DIRTREE_TEXT, &name, -1 );
  if( name ){
    g_free( name );
    return;
  }

  real_path = gtk_tree_model_filter_convert_path_to_child_path( filter, tree_path );
  if( real_path )
  {
    store = gtk_tree_model_filter_get_model( GTK_TREE_MODEL_FILTER(model) );
    dir_path = dir_path_from_tree_path( store, real_path );
    if( ! dir_path ) {
      return;
    }

    row_ref = gtk_tree_row_reference_new( store, real_path );
    content = folder_content_tree_get( dir_path, row_ref );

    gtk_tree_model_get_iter( model, &it, tree_path );
    if( gtk_tree_model_iter_children( model, &child, &it ) ) {
      gtk_tree_model_filter_convert_iter_to_child_iter( model, &it, &child );
      gtk_tree_store_remove( GTK_TREE_STORE(store), &it );
    }
    gtk_tree_path_free( real_path );
    g_free( dir_path );
  }
  return;
}

static gboolean on_dir_tree_update_sel ( FileBrowserData* data )
{
  GtkTreeModel* model;
  GtkTreeIter it;
  GtkTreePath* tree_path;
  char* dir_path = NULL;
  GtkTreeSelection* tree_sel;
  tree_sel = gtk_tree_view_get_selection( data->dirTree );

  if( gtk_tree_selection_get_selected( tree_sel, &model, &it ) )
  {
    tree_path = gtk_tree_model_get_path ( model, &it );
    dir_path = dir_path_from_tree_path( model, tree_path );
    gtk_tree_path_free( tree_path );
  }

  if( dir_path ){
    if( strcmp( dir_path, (char*)data->curHistory->data ) ) {
      file_browser_chdir( data->folderPane, dir_path, TRUE );
    }
    g_free( dir_path );
  }
  return FALSE;
}

void
on_dir_tree_sel_changed           (GtkTreeSelection *treesel,
                                   FileBrowserData* data)
{
  g_idle_add( (GSourceFunc)on_dir_tree_update_sel, data );
}

static void get_file_perm_string( char* perm, mode_t mode )
{
  perm[0] = S_ISDIR( mode ) ? 'd' : (S_ISLNK(mode) ? 'l' : '-');
  perm[1] = (mode & S_IRUSR) ? 'r' : '-';
  perm[2] = (mode & S_IWUSR) ? 'w' : '-';
  perm[3] = (mode & S_IXUSR) ? 'x' : '-';
  perm[4] = (mode & S_IRGRP) ? 'r' : '-';
  perm[5] = (mode & S_IWGRP) ? 'w' : '-';
  perm[6] = (mode & S_IXGRP) ? 'x' : '-';
  perm[7] = (mode & S_IROTH) ? 'r' : '-';
  perm[8] = (mode & S_IWOTH) ? 'w' : '-';
  perm[9] = (mode & S_IXOTH) ? 'x' : '-';
  perm[10] = '\0';
}

static void get_file_size_string( char* buf, guint64 size )
{
  char* unit;
  if( size > ((guint64)1)<<30 ) {
    if( size > ((guint64)1)<<40 ) {
      size >>= 40;
      unit = "TB";
    }
    else {
      size >>= 30;
      unit = "GB";
    }
  }
  else if( size > (1<<20) ) {
    size >>= 20;
    unit = "MB";
  }
  else if( size > (1<<10) ) {
    size >>= 10;
    unit = "KB";
  }
  else {
    unit = size > 1 ? "Bytes" : "Byte";
  }
  sprintf( buf, "%llu %s", size, unit );
}

gboolean update_folder_view_visible_region( FileBrowserData* data )
{
  GtkWidget* folder_view = data->folderView;
  GtkTreeModel* model;
  GtkTreeModelSort* model_sorter;
  GtkListStore* list;
  GtkTreePath *start_path, *end_path;
  GtkTreeIter it;
  GtkTreeIter it2;
  GtkTreePath* real_path;
  GtkTreePath* sorter_path;
  char* file_name;
  const char* mime = NULL;
  char* file_path;
  char* ufile_path;
  char* psize;
  char size[32];
  char* owner;
  char perm[16];
  char time[40];
  const char* desc = NULL;

  struct stat* pstat;
  GdkPixbuf* mime_icon;
  int update_count = 0;
  static struct passwd* puser = NULL;
  static struct group* pgroup = NULL;
  static uid_t cached_uid = 0;
  static gid_t cached_gid = 0;

  if( data->view_mode == FBVM_ICON_VIEW ) {
    if( !ptk_icon_view_get_visible_range ( PTK_ICON_VIEW(folder_view),
                                           &start_path, &end_path ) )
        return;
    model = ptk_icon_view_get_model( PTK_ICON_VIEW(folder_view) );
  }
  else if( data->view_mode == FBVM_LIST_VIEW ) {
    if( !gtk_tree_view_get_visible_range ( GTK_TREE_VIEW(folder_view),
                                           &start_path, &end_path ) )
      return;
    model = gtk_tree_view_get_model( GTK_TREE_VIEW(folder_view) );
  }

  /*
    NOTE:It seems that this is a bug of gtk+ 2.8.
    gtk_tree_view_get_visible_range sometimes returns invalid paths.
    This shouldn't happen according to the document.
    So, bug report to gtk+ team is needed.
    Before this problem can be fixed, we do some check ourselves.
  */
  if( (!end_path || gtk_tree_path_get_depth (end_path) <= 0)
      || (!start_path || gtk_tree_path_get_depth (start_path) <= 0) )  {
    if( start_path )
      gtk_tree_path_free( start_path );
    if( end_path )
      gtk_tree_path_free( end_path );
    return;
  }

  model_sorter = PTK_TREE_MODEL_SORT( gtk_tree_model_filter_get_model(
                                            GTK_TREE_MODEL_FILTER(model) ) );
  list = GTK_LIST_STORE( ptk_tree_model_sort_get_model(model_sorter) );

  for( ; gtk_tree_path_compare( start_path, end_path ) <= 0;
         gtk_tree_path_next ( start_path ) )
  {
    sorter_path = gtk_tree_model_filter_convert_path_to_child_path (
                            GTK_TREE_MODEL_FILTER(model), start_path );
    if( !sorter_path ) {
      g_print( "invalid sorter_path\n" );
      continue;
    }
    real_path = ptk_tree_model_sort_convert_path_to_child_path (
                            model_sorter, sorter_path );
    if( !real_path ) {
      g_print( "invalid real_path\n" );
      continue;
    }
    g_free( sorter_path );

    if( gtk_tree_model_get_iter ( list, &it, real_path ) )
    {
      gtk_tree_model_get ( list, &it,
                            COL_FILE_NAME, &file_name,
                            COL_FILE_TYPE, &mime,
                            COL_FILE_STAT, &pstat,
                            COL_FILE_SIZE, &psize, -1 );

      if( !mime )
      {
        /*
        Variable mime always points to a ``constant string'',
        so no string copy is needed. The string ptrs are always valid.
        */
        ufile_path = g_build_filename( (char*)data->curHistory->data,
                                        file_name, NULL );
        if( ufile_path )  {
          file_path = g_filename_from_utf8( ufile_path, -1, NULL, NULL, NULL );
          g_free( ufile_path );
          mime = xdg_mime_get_mime_type_for_file(file_path, NULL);
        }
        else{
          file_path = NULL;
        }
        mime = xdg_mime_get_mime_type_for_file(file_path, NULL);
        if( !mime )
          mime = XDG_MIME_TYPE_UNKNOWN;

        ++update_count;

        gtk_list_store_set ( list, &it,
                              COL_FILE_TYPE, mime, -1 );

        mime_icon = NULL;

        /*
        * Thumbnail support.
        * FIXME: This should be optional
        */
        if( appSettings.showThumbnail &&
            mime != XDG_MIME_TYPE_UNKNOWN &&
            0 == strncmp(mime, "image/", 6) &&
            pstat->st_size < appSettings.maxThumbSize )
        {
          if( file_path ) {
            mime_icon = gdk_pixbuf_new_from_file_at_scale ( file_path, 32, 32,
                                                            TRUE, NULL );
          }
        }

        if( file_path )
          g_free( file_path );

        if( !mime_icon )
          mime_icon = get_mime_icon( folder_view, mime );

        g_free( file_name );

        gtk_list_store_set ( list, &it,
                              COL_FILE_ICON, mime_icon, -1 );

      }

      if( !psize )
      {
        if( data->view_mode == FBVM_LIST_VIEW )
        {
          get_file_size_string( size, pstat->st_size );
          /* user:group */
          if( pstat->st_uid != cached_uid ) {
            cached_uid = pstat->st_uid;
            puser = getpwuid( cached_uid );
          }
          if( pstat->st_gid != cached_gid ) {
            cached_gid = pstat->st_gid;
            pgroup = getgrgid( cached_gid );
          }
          owner = g_strdup_printf ( "%s:%s", puser ? puser->pw_name : "",
                                    pgroup ? pgroup->gr_name : "" );
          get_file_perm_string( perm, pstat->st_mode );
          strftime( time, sizeof(time),
                    "%Y-%m-%d %H:%M", localtime( &pstat->st_mtime ) );

          if( S_ISLNK( pstat->st_mode ) )
            desc = get_mime_description( "inode/symlink" );
          else
            desc = get_mime_description( mime );

          gtk_list_store_set ( list, &it,
                                COL_FILE_SIZE, size,
                                COL_FILE_PERM, perm,
                                COL_FILE_DESC, desc,
                                COL_FILE_OWNER, owner,
                                COL_FILE_MTIME, time, -1 );
          g_free( owner );
        }
      }
    }

    g_free( real_path );
  }
  gtk_tree_path_free( start_path );
  gtk_tree_path_free( end_path );
  return FALSE;
}

void file_browser_update_mime_icons( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data(file_browser);
  update_folder_view_visible_region( data );
}

static gint sort_files_by_name  (GtkTreeModel *model,
                           GtkTreeIter *a,
                           GtkTreeIter *b,
                           gpointer user_data)
{
  FileBrowserData* data = (FileBrowserData*)user_data;
  struct stat *stat_a, *stat_b;
  gchar* name_a, *name_b;
  gint ret;
  gtk_tree_model_get( model, a, COL_FILE_NAME, &name_a,
                               COL_FILE_STAT, &stat_a, -1 );

  if( !name_a )
    return 0;
  gtk_tree_model_get( model, b, COL_FILE_NAME, &name_b,
                               COL_FILE_STAT, &stat_b, -1 );
  if( !name_b ){
    g_free( name_a );
    return 0;
  }
  ret = S_ISDIR(stat_a->st_mode) - S_ISDIR(stat_b->st_mode);
  if( ret )
    return data->sort_type == GTK_SORT_ASCENDING ? -ret : ret;
  ret = g_ascii_strcasecmp( name_a , name_b );
  g_free( name_a );
  g_free( name_b );
  return ret;
}

static gint sort_files_by_size  (GtkTreeModel *model,
                                 GtkTreeIter *a,
                                 GtkTreeIter *b,
                                 gpointer user_data)
{
  FileBrowserData* data = (FileBrowserData*)user_data;
  struct stat *stat_a, *stat_b;
  gint ret;
  gtk_tree_model_get( model, a, COL_FILE_STAT, &stat_a, -1 );
  gtk_tree_model_get( model, b, COL_FILE_STAT, &stat_b, -1 );
  ret = S_ISDIR(stat_a->st_mode) - S_ISDIR(stat_b->st_mode);
  if( ret )
    return data->sort_type == GTK_SORT_ASCENDING ? -ret : ret;
  ret = (stat_a->st_size - stat_b->st_size);
  return ret ? ret : sort_files_by_name( model, a, b, user_data );
}

static gint sort_files_by_time  (GtkTreeModel *model,
                                 GtkTreeIter *a,
                                 GtkTreeIter *b,
                                 gpointer user_data)
{
  FileBrowserData* data = (FileBrowserData*)user_data;
  struct stat *stat_a, *stat_b;
  gint ret;
  gtk_tree_model_get( model, a, COL_FILE_STAT, &stat_a, -1 );
  gtk_tree_model_get( model, b, COL_FILE_STAT, &stat_b, -1 );
  ret = S_ISDIR(stat_a->st_mode) - S_ISDIR(stat_b->st_mode);
  if( ret )
    return data->sort_type == GTK_SORT_ASCENDING ? -ret : ret;
  ret = (stat_a->st_mtime - stat_b->st_mtime);
  return ret ? ret : sort_files_by_name( model, a, b, user_data );
}


gboolean filter_files  (GtkTreeModel *model,
                        GtkTreeIter *it,
                        gpointer user_data)
{
  gboolean ret;
  gchar* name;
  FileBrowserData* data = (FileBrowserData*)user_data;

  if ( data && data->show_hidden_files )
    return TRUE;

  gtk_tree_model_get( model, it, COL_FILE_NAME, &name, -1 );

  if( name && name[0] == '.' )
    ret = FALSE;
  else
    ret = TRUE;
  g_free( name );
  return ret;
}


static gint sort_dir_tree_by_name  (GtkTreeModel *model,
                                    GtkTreeIter *a,
                                    GtkTreeIter *b,
                                    gpointer user_data)
{
  gchar* name_a, *name_b;
  gint ret;
  gtk_tree_model_get( model, a, COL_DIRTREE_TEXT, &name_a, -1 );
  if( !name_a )
    return -1;
  gtk_tree_model_get( model, b, COL_DIRTREE_TEXT, &name_b, -1 );
  if( !name_b ) {
    g_free( name_a );
    return 1;
  }
  ret = g_ascii_strcasecmp( name_a, name_b );
  g_free( name_a );
  g_free( name_b );
  return ret;
}

void init_dir_tree( FileBrowserData* data )
{
  static GtkTreeStore* tree = NULL;
  GtkTreeView* dirTreeView = data->dirTree;
  GtkTreeSortable* sortable;
  GtkTreeModelFilter* filter;
  GtkTreeIter* parent = NULL;
  GtkTreeIter it, subit;
  GtkTreePath* iter_path;
  GtkTreeViewColumn* col;
  GtkCellRenderer* renderer;
  gchar* file_name;
  struct stat file_stat;

  GdkPixbuf* folder_icon = get_folder_icon16();

  if( gtk_tree_view_get_model( dirTreeView ) )
    return;

  if( !tree ){
   tree = gtk_tree_store_new( NUM_COL_DIRTREE, GDK_TYPE_PIXBUF, G_TYPE_STRING );
    gtk_tree_store_append ( tree, &it, parent );
    gtk_tree_store_set ( tree, &it, COL_DIRTREE_ICON, folder_icon,
                                            COL_DIRTREE_TEXT, "/", -1 );
    gtk_tree_store_append ( tree, &subit, &it );
  }
  sortable = GTK_TREE_SORTABLE(tree);
  gtk_tree_sortable_set_sort_func( sortable, COL_DIRTREE_TEXT,
                                   sort_dir_tree_by_name, NULL, NULL );

  gtk_tree_sortable_set_sort_column_id( sortable,
                                        COL_DIRTREE_TEXT,
                                        data->sort_type );

  filter = gtk_tree_model_filter_new( tree, NULL );
  gtk_tree_model_filter_set_visible_func( filter, filter_files, data, NULL );

  gtk_tree_view_set_model( dirTreeView, GTK_TREE_MODEL( filter ) );
  data->tree_filter = filter;

  iter_path = gtk_tree_path_new_from_indices (0, -1);
  gtk_tree_view_expand_row ( dirTreeView, iter_path, FALSE );
  gtk_tree_path_free( iter_path );

  if( ! gtk_tree_view_get_column(dirTreeView, 0) ){
    col = gtk_tree_view_column_new ();

    renderer = gtk_cell_renderer_pixbuf_new();
    gtk_tree_view_column_pack_start( col, renderer, FALSE );
    gtk_tree_view_column_set_attributes( col, renderer, "pixbuf", COL_DIRTREE_ICON, NULL );

    renderer = gtk_cell_renderer_text_new();
    gtk_tree_view_column_pack_start( col, renderer, TRUE );
/*  g_object_set( renderer, "editable", TRUE, NULL ); */
    gtk_tree_view_column_set_attributes( col, renderer, "text", COL_DIRTREE_TEXT, NULL );

    gtk_tree_view_append_column ( dirTreeView, col );
  }
}

static gboolean can_sel_change ( GtkTreeSelection *selection,
                                 GtkTreeModel *model,
                                 GtkTreePath *path,
                                 gboolean path_currently_selected,
                                 gpointer data)
{
  return can_folder_view_sel_change;
}


static GtkWidget* create_folder_view( FileBrowserViewMode view_mode,
                                      FileBrowserData* data )
{
  GtkWidget* folder_view;
  GtkTreeSelection* tree_sel;
  GtkCellRenderer* renderer;
  switch( view_mode )
  {
    case FBVM_ICON_VIEW:
      folder_view = ptk_icon_view_new ();
      ptk_icon_view_set_selection_mode ( PTK_ICON_VIEW(folder_view),
                                         GTK_SELECTION_MULTIPLE);

      ptk_icon_view_set_pixbuf_column ( PTK_ICON_VIEW(folder_view), COL_FILE_ICON );
      ptk_icon_view_set_text_column ( PTK_ICON_VIEW(folder_view), COL_FILE_NAME );

      ptk_icon_view_set_column_spacing( PTK_ICON_VIEW(folder_view), 4 );
      ptk_icon_view_set_item_width ( PTK_ICON_VIEW(folder_view), 108 );

      ptk_icon_view_set_enable_search( PTK_ICON_VIEW(folder_view), TRUE );
      ptk_icon_view_set_search_column( PTK_ICON_VIEW(folder_view), COL_FILE_NAME );

      gtk_cell_layout_clear (GTK_CELL_LAYOUT (folder_view));

      renderer = gtk_cell_renderer_pixbuf_new ();
      /* add the icon renderer */
      g_object_set (G_OBJECT (renderer),
                    "follow_state", TRUE,
                    NULL );
      gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (folder_view), renderer, FALSE);
      gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (folder_view), renderer,
                                     "pixbuf", COL_FILE_ICON);

      /* add the name renderer */
      renderer = gtk_cell_renderer_text_new ();
      g_object_set (G_OBJECT (renderer),
                    "wrap-mode", PANGO_WRAP_CHAR,
                    "wrap-width", 108,
                    "xalign", 0.5,
                    "yalign", 0.0,
                    NULL);
      gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (folder_view), renderer, TRUE);
      gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (folder_view), renderer,
                                     "text", COL_FILE_NAME);

      ptk_icon_view_enable_model_drag_source (
          PTK_ICON_VIEW(folder_view),
          (GDK_CONTROL_MASK|GDK_BUTTON1_MASK|GDK_BUTTON3_MASK),
          dragTargets,
          sizeof(dragTargets)/sizeof(GtkTargetEntry),
          GDK_ACTION_DEFAULT|GDK_ACTION_COPY|GDK_ACTION_MOVE|GDK_ACTION_LINK);

      ptk_icon_view_enable_model_drag_dest (
          PTK_ICON_VIEW(folder_view),
          dragTargets,
          sizeof(dragTargets)/sizeof(GtkTargetEntry),
          GDK_ACTION_DEFAULT|GDK_ACTION_COPY|GDK_ACTION_MOVE|GDK_ACTION_LINK);

      g_signal_connect ((gpointer) folder_view,
                         "item_activated",
                         G_CALLBACK (on_folder_view_item_activated),
                         data );

      g_signal_connect ((gpointer) folder_view,
                         "key_press_event",
                         G_CALLBACK (on_folder_view_key_press_event),
                         data);
      g_signal_connect_after ((gpointer) folder_view,
                         "selection-changed",
                         G_CALLBACK (on_folder_view_item_sel_change),
                         data );

      break;
    case FBVM_LIST_VIEW:
      folder_view = gtk_tree_view_new ();
      init_list_view( GTK_TREE_VIEW(folder_view) );
      tree_sel = gtk_tree_view_get_selection( GTK_TREE_VIEW(folder_view) );
      gtk_tree_selection_set_mode( tree_sel, GTK_SELECTION_MULTIPLE);

      gtk_tree_view_enable_model_drag_source (
          GTK_TREE_VIEW(folder_view),
      (GDK_CONTROL_MASK|GDK_BUTTON1_MASK|GDK_BUTTON3_MASK),
      dragTargets,
      sizeof(dragTargets)/sizeof(GtkTargetEntry),
      GDK_ACTION_DEFAULT|GDK_ACTION_COPY|GDK_ACTION_MOVE|GDK_ACTION_LINK);

      gtk_tree_view_enable_model_drag_dest (
          GTK_TREE_VIEW(folder_view),
      dragTargets,
      sizeof(dragTargets)/sizeof(GtkTargetEntry),
      GDK_ACTION_DEFAULT|GDK_ACTION_COPY|GDK_ACTION_MOVE|GDK_ACTION_LINK);

      g_signal_connect ((gpointer) folder_view,
                        "row_activated",
                        G_CALLBACK (on_folder_view_row_activated),
                        data );

      g_signal_connect_after ((gpointer) tree_sel,
                        "changed",
                        G_CALLBACK (on_folder_view_item_sel_change),
                        data );

      gtk_tree_selection_set_select_function( tree_sel,
                                              can_sel_change, data, NULL );
      break;
  }

  g_signal_connect ((gpointer) folder_view,
                     "button-press-event",
                     G_CALLBACK (on_folder_view_button_press_event),
                     data);
  g_signal_connect ((gpointer) folder_view,
                     "button-release-event",
                     G_CALLBACK (on_folder_view_button_release_event),
                     data);
  g_signal_connect ((gpointer) folder_view,
                     "size_allocate",
                     G_CALLBACK (on_folder_view_size_allocate),
                     data);

  /* init drag & drop support */

  g_signal_connect ((gpointer) folder_view, "drag-data-received",
                     G_CALLBACK (on_folder_view_drag_data_received),
                     data );

  g_signal_connect ((gpointer) folder_view, "drag-data-get",
                     G_CALLBACK (on_folder_view_drag_data_get),
                     data );

  g_signal_connect ((gpointer) folder_view, "drag-begin",
                     G_CALLBACK (on_folder_view_drag_begin),
                     data );

  g_signal_connect ((gpointer) folder_view, "drag-motion",
                      G_CALLBACK (on_folder_view_drag_motion),
                      data );

  g_signal_connect ((gpointer) folder_view, "drag-leave",
                      G_CALLBACK (on_folder_view_drag_leave),
                      data );

  g_signal_connect ((gpointer) folder_view, "drag-drop",
                     G_CALLBACK (on_folder_view_drag_drop),
                     data );

  g_signal_connect ((gpointer) folder_view, "drag-end",
                     G_CALLBACK (on_folder_view_drag_end),
                     data );

  return folder_view;
}


static void init_list_view( GtkTreeView* list_view )
{
  GtkTreeViewColumn* col;
  GtkCellRenderer *renderer;
  GtkCellRenderer *pix_renderer;

  int cols[] = { COL_FILE_NAME, COL_FILE_SIZE, COL_FILE_DESC,
                 COL_FILE_PERM, COL_FILE_OWNER, COL_FILE_MTIME };
  int i;

  const char* titles[] = { N_("Name"), N_("Size"), N_("Type"),
            N_("Permission"), N_("Owner:Group"), N_("Last Modification") };

  for( i = 0; i < G_N_ELEMENTS(cols); ++i ) {
    col = gtk_tree_view_column_new ();
    gtk_tree_view_column_set_resizable (col, TRUE);

    renderer = gtk_cell_renderer_text_new();

    if( i == 0 ) { /* First column */
      gtk_object_set( GTK_OBJECT(renderer), "ellipsize", PANGO_ELLIPSIZE_END, NULL );

      pix_renderer = gtk_cell_renderer_pixbuf_new();
      gtk_tree_view_column_pack_start( col, pix_renderer, FALSE );
      gtk_tree_view_column_set_attributes( col, pix_renderer, "pixbuf",
                                           COL_FILE_ICON, NULL );
      gtk_tree_view_column_set_expand (col, TRUE);
      gtk_tree_view_column_set_min_width(col, 200);
    }

    gtk_tree_view_column_pack_start( col, renderer, TRUE );
    gtk_tree_view_column_set_attributes( col, renderer, "text", cols[i], NULL );
    gtk_tree_view_append_column ( list_view, col );
    gtk_tree_view_column_set_title( col, _( titles[i] ) );
  }

  col = gtk_tree_view_get_column( list_view, 2 );
  gtk_tree_view_column_set_sizing( col, GTK_TREE_VIEW_COLUMN_FIXED );
  gtk_tree_view_column_set_fixed_width (col, 120);

  /*
  col = gtk_tree_view_get_column( list_view, 0 );
  gtk_tree_view_column_set_clickable (col, TRUE);
  gtk_tree_view_column_set_sort_column_id (col, COL_FILE_NAME);

  col = gtk_tree_view_get_column( list_view, 1 );
  gtk_tree_view_column_set_clickable (col, TRUE);
  gtk_tree_view_column_set_sort_column_id (col, COL_FILE_STAT);

  col = gtk_tree_view_get_column( list_view, 5 );
  gtk_tree_view_column_set_clickable (col, TRUE);
  gtk_tree_view_column_set_sort_column_id (col, COL_FILE_STAT);
  */

  gtk_tree_view_set_rules_hint ( list_view, TRUE );
}

void file_browser_refresh( GtkWidget* file_browser )
{
  file_browser_chdir( file_browser,
                           file_browser_get_cwd( file_browser ),
                           FALSE );
}

int file_browser_get_n_files( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  return data->fileCount;
}

GList* folder_view_get_selected_items( FileBrowserData* data,
                                       GtkTreeModel** model )
{
  GtkTreeSelection* tree_sel;
  if( data->view_mode == FBVM_ICON_VIEW ) {
    *model = ptk_icon_view_get_model( PTK_ICON_VIEW(data->folderView) );
    return ptk_icon_view_get_selected_items( PTK_ICON_VIEW(data->folderView) );
  }
  else if( data->view_mode == FBVM_LIST_VIEW ) {
    tree_sel = gtk_tree_view_get_selection( GTK_TREE_VIEW(data->folderView) );
    return gtk_tree_selection_get_selected_rows( tree_sel, model );
  }
  return NULL;
}


static char* folder_view_get_drop_dir( FileBrowserData* data, int x, int y )
{
  GtkTreePath* tree_path = NULL;
  GtkTreeModel *model;
  GtkTreeViewColumn* col;
  GtkTreeIter it;
  char* ufilename;
  char* udest_path = NULL;
  struct stat* pstat;

  if( data->view_mode == FBVM_ICON_VIEW ) {
    ptk_icon_view_widget_to_icon_coords ( PTK_ICON_VIEW(data->folderView),
                                          x, y, &x, &y );
    tree_path = folder_view_get_tree_path_at_pos(data, x, y);
    model = ptk_icon_view_get_model( PTK_ICON_VIEW(data->folderView) );
  }
  else if( data->view_mode == FBVM_LIST_VIEW ) {
    gtk_tree_view_get_path_at_pos( GTK_TREE_VIEW(data->folderView), x, y,
                                   NULL, &col, NULL, NULL );
    if( col == gtk_tree_view_get_column(GTK_TREE_VIEW(data->folderView), 0) )
    {
      gtk_tree_view_get_dest_row_at_pos ( GTK_TREE_VIEW(data->folderView), x, y,
                                          &tree_path, NULL );
      model = gtk_tree_view_get_model( GTK_TREE_VIEW(data->folderView) );
    }
  }
  if( tree_path )
  {
    gtk_tree_model_get_iter( model, &it, tree_path );
    gtk_tree_model_get( model, &it, COL_FILE_NAME, &ufilename,
                        COL_FILE_STAT, &pstat, -1 );
    if( ufilename )
    {
      if( S_ISDIR( pstat->st_mode ) )
      {
        udest_path = g_build_filename( (char*)data->curHistory->data,
                                        ufilename, NULL );
      }
      else  /* Drop on a file, not folder */
      {
        /* Return current directory */
        udest_path = g_strdup((char*)data->curHistory->data);
      }
      g_free( ufilename );
    }
    gtk_tree_path_free( tree_path );
  }
  else
  {
    return g_strdup((char*)data->curHistory->data);
  }
  return udest_path;
}

void on_folder_view_drag_data_received (GtkWidget        *widget,
                                       GdkDragContext   *drag_context,
                                       gint              x,
                                       gint              y,
                                       GtkSelectionData *data,
                                       guint             info,
                                       guint             time,
                                       gpointer          user_data)
{
  gchar **list, **puri;
  GList* files = NULL;
  FileOperation* file_operation;
  FileOperationType file_action = FO_MOVE;
  FileBrowserData* file_browser_data = (FileBrowserData*)user_data;
  char* dest_dir;
  char* file_path;

  /*  Don't call the default handler  */
  g_signal_stop_emission_by_name( widget, "drag-data-received" );

  if ((data->length >= 0) && (data->format == 8))  {
    puri = list = gtk_selection_data_get_uris( data );
    if( puri )
    {
      if( 0 == (drag_context->action  &
          (GDK_ACTION_MOVE|GDK_ACTION_COPY|GDK_ACTION_LINK)) )
      {
        drag_context->action = GDK_ACTION_MOVE;
      }
      gtk_drag_finish (drag_context, TRUE, FALSE, time);

      while( *puri ){
        if( **puri == '/' ) {
          file_path = g_strdup( *puri );
        }
        else {
          file_path = g_filename_from_uri(*puri, NULL, NULL);
        }

        if( file_path ){
          files = g_list_prepend( files, file_path );
        }
        ++puri;
      }
      g_strfreev( list );

      switch( drag_context->action )
      {
        case GDK_ACTION_COPY:
          file_action = FO_COPY;
          break;
        case GDK_ACTION_LINK:
          file_action = FO_LINK;
          break;
      }

      if( files ) {
        dest_dir = folder_view_get_drop_dir( file_browser_data, x, y );
        /* g_print( "dest_dir = %s\n", dest_dir ); */
        file_operation = file_operation_new( files, dest_dir, file_action, NULL,
                                  GTK_WINDOW(file_browser_data->mainWindow) );
        g_free( dest_dir );
      }
      return;
    }
  }
  gtk_drag_finish (drag_context, FALSE, FALSE, time);
}

void on_folder_view_drag_data_get (GtkWidget        *widget,
                                  GdkDragContext   *drag_context,
                                  GtkSelectionData *sel_data,
                                  guint             info,
                                  guint             time,
                                  FileBrowserData  *data)
{
  GdkAtom type = gdk_atom_intern( "text/uri-list", FALSE );
  gchar* uri;
  GString* uri_list = g_string_sized_new( 8192 );
  GList* sels = file_browser_get_selected_files( data );
  GList* sel;

  drag_context->actions = GDK_ACTION_MOVE|GDK_ACTION_COPY|GDK_ACTION_LINK;
  drag_context->suggested_action = GDK_ACTION_MOVE;

  /*  Don't call the default handler  */
  g_signal_stop_emission_by_name( widget, "drag-data-get" );

  for( sel = sels; sel; sel = g_list_next(sel) )
  {
    uri = g_filename_to_uri( (char*)sel->data, NULL, NULL );

    g_string_append( uri_list, uri );
    g_free( uri );

    g_string_append( uri_list, "\r\n" );
  }

  g_list_foreach( sels, (GFunc)g_free, NULL );
  g_list_free( sels );

  gtk_selection_data_set ( sel_data, type, 8, ( guchar*)uri_list->str, uri_list->len+1 );
  g_string_free( uri_list, TRUE );
}


void on_folder_view_drag_begin      (GtkWidget      *widget,
                                    GdkDragContext *drag_context,
                                    gpointer        user_data)
{
  /*  Don't call the default handler  */
  g_signal_stop_emission_by_name ( widget, "drag-begin" );

  gtk_drag_set_icon_stock ( drag_context, GTK_STOCK_DND_MULTIPLE, 1, 1 );
}

static GtkTreePath*
folder_view_get_tree_path_at_pos( FileBrowserData* data, int x, int y )
{
  GtkScrolledWindow* scroll;
  GtkAdjustment *vadj;
  gint vy;
  gdouble vpos;
  GtkTreePath *tree_path;

  if( data->view_mode == FBVM_ICON_VIEW ) {
    tree_path = ptk_icon_view_get_path_at_pos( PTK_ICON_VIEW(data->folderView), x, y );
  }
  else if( data->view_mode == FBVM_LIST_VIEW ) {
    gtk_tree_view_get_path_at_pos( GTK_TREE_VIEW(data->folderView), x, y,
                                   &tree_path, NULL, NULL, NULL );
  }
  return tree_path;
}

gboolean on_folder_view_auto_scroll( GtkScrolledWindow* scroll )
{
  GtkAdjustment *vadj;
  gdouble vpos;

  vadj = gtk_scrolled_window_get_vadjustment( scroll ) ;
  vpos = gtk_adjustment_get_value( vadj );

  if( folder_view_auto_scroll_direction == GTK_DIR_UP )
  {
    vpos -= vadj->step_increment;
    if( vpos > vadj->lower )
      gtk_adjustment_set_value ( vadj, vpos );
    else
      gtk_adjustment_set_value ( vadj, vadj->lower );
  }
  else
  {
    vpos += vadj->step_increment;
    if( (vpos + vadj->page_size) < vadj->upper )
      gtk_adjustment_set_value ( vadj, vpos );
    else
      gtk_adjustment_set_value ( vadj, (vadj->upper - vadj->page_size) );
  }

  return TRUE;
}

gboolean on_folder_view_drag_motion (GtkWidget      *widget,
                                    GdkDragContext *drag_context,
                                    gint            x,
                                    gint            y,
                                    guint           time,
                                    FileBrowserData* data )
{
  GtkScrolledWindow* scroll;
  GtkAdjustment *hadj, *vadj;
  gint vx, vy;
  gdouble hpos, vpos;
  GtkTreeModel* model;
  GtkTreePath *tree_path;
  GtkTreeViewColumn* col;
  GtkTreeIter it;
  gchar *file_name;
  gchar* dir_path;
  struct stat *file_stat;
  GdkDragAction suggested_action;

  /*  Don't call the default handler  */
  g_signal_stop_emission_by_name ( widget, "drag-motion" );

  scroll = GTK_SCROLLED_WINDOW( gtk_widget_get_parent ( widget ) );

  vadj = gtk_scrolled_window_get_vadjustment( scroll ) ;
  vpos = gtk_adjustment_get_value( vadj );

  if( y < 32 ){
    /* Auto scroll up */
    if( ! folder_view_auto_scroll_timer ) {
      folder_view_auto_scroll_direction = GTK_DIR_UP;
      folder_view_auto_scroll_timer = g_timeout_add(
                                        150,
                                        (GSourceFunc)on_folder_view_auto_scroll,
                                        scroll );
    }
  }
  else if( y > (widget->allocation.height - 32 ) ) {
    if( ! folder_view_auto_scroll_timer ) {
      folder_view_auto_scroll_direction = GTK_DIR_DOWN;
      folder_view_auto_scroll_timer = g_timeout_add(
                                        150,
                                        (GSourceFunc)on_folder_view_auto_scroll,
                                        scroll );
    }
  }
  else if( folder_view_auto_scroll_timer ) {
    g_source_remove( folder_view_auto_scroll_timer );
    folder_view_auto_scroll_timer = 0;
  }

  tree_path = NULL;
  if( data->view_mode == FBVM_ICON_VIEW )
  {
    ptk_icon_view_widget_to_icon_coords(  PTK_ICON_VIEW(widget), x, y, &x, &y );
    tree_path = ptk_icon_view_get_path_at_pos( PTK_ICON_VIEW(widget), x, y );
    model = ptk_icon_view_get_model( PTK_ICON_VIEW(widget) );
  }
  else
  {
    if( gtk_tree_view_get_path_at_pos( GTK_TREE_VIEW(widget), x, y,
                                       NULL, &col, NULL, NULL ) )
    {
      if( gtk_tree_view_get_column (GTK_TREE_VIEW(widget), 0) == col ) {
        gtk_tree_view_get_dest_row_at_pos (GTK_TREE_VIEW(widget), x, y,
                                           &tree_path, NULL);
        model = gtk_tree_view_get_model( GTK_TREE_VIEW(widget) );
      }
    }
  }

  if( tree_path )
  {
    if( gtk_tree_model_get_iter( model, &it, tree_path ) )
    {
      gtk_tree_model_get( model, &it, COL_FILE_STAT, &file_stat, -1 );
      if( ! file_stat || ! S_ISDIR(file_stat->st_mode) )
      {
        gtk_tree_path_free( tree_path );
        tree_path = NULL;
      }
    }
  }

  if( data->view_mode == FBVM_ICON_VIEW )
  {
    ptk_icon_view_set_drag_dest_item ( PTK_ICON_VIEW(widget),
                                        tree_path, PTK_ICON_VIEW_DROP_INTO );
  }
  else if( data->view_mode == FBVM_LIST_VIEW )
  {
    gtk_tree_view_set_drag_dest_row( GTK_TREE_VIEW(widget),
                                      tree_path,
                                      GTK_TREE_VIEW_DROP_INTO_OR_AFTER );
  }

  if( tree_path )
    gtk_tree_path_free( tree_path );

  suggested_action = (drag_context->actions  & GDK_ACTION_MOVE) ?
                        GDK_ACTION_MOVE : drag_context->suggested_action;
  gdk_drag_status ( drag_context, suggested_action, time );

  return TRUE;
}

gboolean on_folder_view_drag_leave ( GtkWidget      *widget,
                                     GdkDragContext *drag_context,
                                     guint           time,
                                     FileBrowserData* data )
{
  /*  Don't call the default handler  */
  g_signal_stop_emission_by_name( widget, "drag-leave" );

  if( folder_view_auto_scroll_timer ) {
    g_source_remove( folder_view_auto_scroll_timer );
    folder_view_auto_scroll_timer = 0;
  }
}


gboolean on_folder_view_drag_drop   (GtkWidget      *widget,
                                    GdkDragContext *drag_context,
                                    gint            x,
                                    gint            y,
                                    guint           time,
                                    FileBrowserData* data )
{
  GdkAtom target = gdk_atom_intern( "text/uri-list", FALSE );
  /*  Don't call the default handler  */
  g_signal_stop_emission_by_name( widget, "drag-drop" );

  gtk_drag_get_data ( widget, drag_context, target, time );
  return TRUE;
}


void on_folder_view_drag_end        (GtkWidget      *widget,
                                     GdkDragContext *drag_context,
                                     gpointer        user_data)
{
  FileBrowserData* data = (FileBrowserData*)user_data;
  if( folder_view_auto_scroll_timer ) {
    g_source_remove( folder_view_auto_scroll_timer );
    folder_view_auto_scroll_timer = 0;
  }
  if( data->view_mode == FBVM_ICON_VIEW ) {
    ptk_icon_view_set_drag_dest_item( PTK_ICON_VIEW(widget), NULL, 0 );
  }
  else if( data->view_mode == FBVM_LIST_VIEW ) {
    gtk_tree_view_set_drag_dest_row( GTK_TREE_VIEW(widget), NULL, 0 );
  }
}

gboolean on_tab_drag_motion (GtkWidget      *widget,
                             GdkDragContext *drag_context,
                             gint            x,
                             gint            y,
                             guint           time,
                             FileBrowserData* data )
{
  /* TODO: Add a timeout here and don't set current page immediately */
  GtkNotebook* notebook = GTK_NOTEBOOK( lookup_widget( data->mainWindow,
                                                   "folderNoteBook") );
  gint idx = gtk_notebook_page_num ( GTK_NOTEBOOK(notebook),
                                     data->folderPane );
  gtk_notebook_set_current_page( notebook, idx );
}

void file_browser_rename_selected_file( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data(file_browser);
  GtkWidget* input_dlg;
  GtkLabel* prompt;
  GList* items;
  GtkTreeModel* model;
  GtkTreeIter it;
  GtkTreeSelection* tree_sel;
  gchar* dir;
  gchar* file_name;
  gchar* ufile_name;
  gchar* to_path;
  gchar* ufull_path;
  gchar* from_path;

  if( gtk_widget_is_focus ( GTK_WIDGET(data->folderView) ) ){
    items = folder_view_get_selected_items(data, &model);
    if( !items )
      return;
    gtk_tree_model_get_iter( model, &it, (GtkTreePath*)items->data );
    gtk_tree_model_get( model, &it, 1, &ufile_name, -1);
    g_list_foreach( items, (GFunc)gtk_tree_path_free, NULL );
    g_list_free( items );
  }
/*  else if( gtk_widget_is_focus ( GTK_WIDGET(data->dirTree) ) ){
    dir = g_path_get_dirname ( (char*)data->curHistory->data );
    file_name = g_strdup(
        (char*)data->curHistory->data + strlen(dir) );
    tree_sel = gtk_tree_view_get_selection( data->dirTree );
    gtk_tree_selection_get_selected (tree_sel, &model, &it );
  }
  else
    return;
*/

  ufull_path = g_build_filename( (char*)data->curHistory->data,
                                  ufile_name, NULL );
  from_path = g_filename_from_utf8( ufull_path, -1,
                                    NULL, NULL, NULL );
  g_free( ufull_path );

  input_dlg = input_dialog_new( _("Rename File"),
                                _("Please input new file name:"),
                                ufile_name,
                                GTK_WINDOW(data->mainWindow) );
  g_free(ufile_name);

  while( gtk_dialog_run( GTK_DIALOG(input_dlg)) == GTK_RESPONSE_OK )
  {
    ufile_name = input_dialog_get_text( input_dlg );
    ufull_path = g_build_filename( (char*)data->curHistory->data,
                                   ufile_name, NULL );
    g_free( ufile_name );
    to_path = g_filename_from_utf8( ufull_path, -1,
                                    NULL, NULL, NULL );
    g_free( ufull_path );
    if( g_file_test( to_path, G_FILE_TEST_EXISTS ) )
    {
      prompt = GTK_LABEL( input_dialog_get_label( input_dlg ) );
      gtk_label_set_text( prompt,
                          _("The file name you specified has existed.\n"
                              "Please input a new one:") );
      g_free( to_path );
      continue;
    }
    g_rename( from_path, to_path );
    g_free( from_path );
    g_free( to_path );
    break;
  }

  gtk_widget_destroy( input_dlg );

/*
  This seems to be buggy? Is it a bug of GTK+ 2.8?
  Everytime my program crashes here.
*/
/*  gtk_widget_grab_focus (GTK_WIDGET(data->folderView));
  ptk_icon_view_set_cursor( data->folderView,
                            (GtkTreePath*)items->data,
                            NULL, TRUE );
*/
}

gboolean file_browser_can_paste( GtkWidget* file_browser )
{
  return FALSE;
}

void file_browser_paste( GtkWidget* file_browser )
{
  FileBrowserData* data;
  GtkClipboard* clip = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
  GdkAtom target = gdk_atom_intern( "text/uri-list", FALSE );
  gchar **uri_list, **puri;
  GtkSelectionData* sel_data = NULL;
  GList* files = NULL;
  GList* sel_files, *sel;
  gchar* file_path;
  gchar* dest_dir;
  FileOperation* file_operation;
  FileOperationType action;

  sel_data = gtk_clipboard_wait_for_contents( clip, target );
  if( ! sel_data ){
    g_warning("no data in clipboard!\n");
    return;
  }

  if( (sel_data->length >= 0) && (sel_data->format == 8)) {
    puri = uri_list = gtk_selection_data_get_uris( sel_data );
    while( *puri ){
      file_path = g_filename_from_uri(*puri, NULL, NULL);
      if( file_path ){
        files = g_list_prepend( files, file_path );
      }
      ++puri;
    }
    g_strfreev( uri_list );
    gtk_selection_data_free( sel_data );

    data = file_browser_get_data( file_browser );
    /*
    * If only one item is selected and the item is a
    * directory, paste the files in that directory;
    * otherwise, paste the file in current directory.
    */
    if( sel_files = file_browser_get_selected_files(data) )  {
      dest_dir = (char*)sel_files->data;
    }
    else{
      dest_dir = (char*)data->curHistory->data;
    }
    if( clipboard_action == GDK_ACTION_MOVE )
      action = FO_MOVE;
    else
      action = FO_COPY;
    file_operation = file_operation_new( files,
                                         dest_dir,
                                         action,
                                         NULL,
                                         GTK_WINDOW(data->mainWindow) );
    if( sel_files ){
      g_list_foreach( sel_files, (GFunc)g_free, NULL);
      g_list_free( sel_files );
    }
  }
}

gboolean file_browser_can_cut_or_copy( GtkWidget* file_browser )
{
  return FALSE;
}


static void clipboard_get_data (GtkClipboard *clipboard,
                                GtkSelectionData *selection_data,
                                guint info,
                                gpointer user_data)
{
  FileBrowserData* data = (FileBrowserData*)user_data;
  GdkAtom target = gdk_atom_intern( "text/uri-list", FALSE );
  GList* sel;
  gchar* uri;
  GString* list = g_string_sized_new( 8192 );

  for( sel = data->clipboard_file_list;
       sel; sel = g_list_next(sel) )
  {
    uri = g_filename_to_uri( (char*)sel->data, NULL, NULL );
    g_string_append( list, uri );
    g_free( uri );
    g_string_append( list, "\r\n" );
  }

  g_print("copy: %s\n", list->str);

  gtk_selection_data_set ( selection_data, target, 8,
                           ( guchar*)list->str,
                           list->len + 1 );
  g_string_assign( list, "" );

  for( sel = data->clipboard_file_list;
       sel; sel = g_list_next(sel) )
  {
    g_string_append( list, (char*)sel->data );
    g_string_append( list, "\r\n" );
  }

  g_string_free( list, TRUE );
}

static void clipboard_clean_data (GtkClipboard *clipboard,
                                  gpointer user_data)
{
/*  FileBrowserData* data = (FileBrowserData*)user_data;
  g_print("clean clipboard!\n");
  if( data->clipboard_file_list )
  {
    g_list_foreach( data->clipboard_file_list,
                    (GFunc)g_free, NULL );
    g_list_free( data->clipboard_file_list );
    data->clipboard_file_list = NULL;
  }
*/  clipboard_action = GDK_ACTION_DEFAULT;
}

void file_browser_cut( GtkWidget* file_browser )
{
  /* What "cut" and "copy" do are the same.
  *  The only difference is clipboard_action = GDK_ACTION_MOVE.
  */
  file_browser_copy( file_browser );
  clipboard_action = GDK_ACTION_MOVE;
}

void file_browser_copy( GtkWidget* file_browser )
{
  GtkClipboard* clip = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
  GtkTargetList* target_list = gtk_target_list_new(NULL, 0);
  GList* target;
  gint i, n_targets;
  GtkTargetEntry* targets;
  GtkTargetPair* pair;
  FileBrowserData* data = file_browser_get_data( file_browser );

  gtk_target_list_add_text_targets( target_list, 0 );
  n_targets = g_list_length( target_list->list ) + 1;

  targets = g_new0( GtkTargetEntry, n_targets );
  target = target_list->list;
  for(  i = 0; target; ++i, target = g_list_next(target) ) {
    pair = (GtkTargetPair*)target->data;
    targets[i].target = gdk_atom_name (pair->target);
  }
  targets[i].target = "text/uri-list";

  gtk_target_list_unref (target_list);

  if( data->clipboard_file_list )
  {
    g_list_foreach( data->clipboard_file_list,
                    (GFunc)g_free, NULL );
    g_list_free( data->clipboard_file_list );
  }
  data->clipboard_file_list = file_browser_get_selected_files( data );

  if( data->clipboard_file_list )
  {
    gtk_clipboard_set_with_data (
              clip, targets, n_targets,
              clipboard_get_data,
              clipboard_clean_data,
              (gpointer)file_browser_get_data(file_browser));
  }
  g_free( targets );

  clipboard_action = GDK_ACTION_COPY;
}

void file_browser_can_delete( GtkWidget* file_browser )
{

}

void file_browser_delete( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data(file_browser);
  GtkWidget* dlg;
  GList* file_list;
  gchar* file_path;
  gint ret;
  GList* sel;
  FileOperation* file_operation;

  if( ! data->n_sel_files )
    return;

  dlg = gtk_message_dialog_new( GTK_WINDOW(data->mainWindow),
                                GTK_DIALOG_MODAL,
                                GTK_MESSAGE_WARNING,
                                GTK_BUTTONS_YES_NO,
                                _("Really delete selected files?"));
  ret = gtk_dialog_run( GTK_DIALOG(dlg) );
  gtk_widget_destroy( dlg );
  if( ret != GTK_RESPONSE_YES ){
    return;
  }

  file_list = file_browser_get_selected_files( data );
  for( sel = file_list; sel; sel = g_list_next(sel) ) {
    file_path = g_filename_from_utf8( (char*)sel->data,
                                      -1, NULL, NULL, NULL );

    file_operation = file_operation_delete_file(file_path, NULL,
                                              GTK_WINDOW(data->mainWindow) );

    g_free( file_path );
  }

  g_list_foreach( file_list, (GFunc)g_free, NULL );
  g_list_free( file_list );
}

/* Return a list of selected filenames (in full path, in UTF-8) */
static GList*
file_browser_get_selected_files( FileBrowserData* data )
{
  GList* sel_files;
  GList* sel;
  GList* file_list = NULL;
  GtkTreeModel* model;
  GtkTreeIter it;
  gchar* file_name;

  sel_files = folder_view_get_selected_items(data, &model);

  if( ! sel_files )
    return NULL;

  for( sel = sel_files; sel; sel = g_list_next(sel) ) {
    gtk_tree_model_get_iter( model, &it, (GtkTreePath*)sel->data );
    gtk_tree_model_get( model, &it, COL_FILE_NAME, &file_name, -1 );
    file_list = g_list_append( file_list,
                               g_build_filename(
                               (char*)data->curHistory->data,
                               file_name, NULL ) );
  }
  g_list_foreach( sel_files,
                  (GFunc)gtk_tree_path_free,
                  NULL);
  g_list_free( sel_files );
  return file_list;
}

void file_browser_open_selected_files( GtkWidget* file_browser )
{
  FileBrowserData* data;
  data = file_browser_get_data(file_browser);
  file_browser_open_selected_files_with_app( data, NULL );
}

void file_browser_open_selected_files_with_app( FileBrowserData* data,
                                                const char* app_desktop )

{
  GList* sel_items, *item;
  GtkTreeIter it;
  struct stat *pstat;
  gchar* file_name;
  gchar* new_path;
  gchar* mime;
  gchar *app, *exec, *cmd, *file_list[2] = {NULL, NULL};
  GtkTreeModel* list;
  GError* err;
  g_print("open !!\n");
  sel_items = folder_view_get_selected_items(data, &list);

  for( item = sel_items; item; item = item->next )
  {
    if( ! gtk_tree_model_get_iter( list, &it, (GtkTreePath*)item->data ) )
      continue;

    gtk_tree_model_get( list, &it, COL_FILE_NAME, &file_name,
                        COL_FILE_STAT, &pstat, -1 );
    if( file_name ) {
      new_path = g_build_filename( (char*)data->curHistory->data,
                                    file_name, NULL );
      g_free( file_name );

      if( new_path )  {
        if( ! app_desktop && pstat && S_ISDIR(pstat->st_mode) )
        {
          file_browser_chdir( data->folderPane, new_path, TRUE );
        }
        else{
          /* If this file is an executable file, run it. */
          if( !app_desktop && g_file_test(new_path,
                   G_FILE_TEST_IS_EXECUTABLE) )
          {
            err = NULL;
            if( ! g_spawn_command_line_async (new_path, &err) )  {
              errorMessage( data->mainWindow, err->message );
              g_error_free( err );
            }
            return;
          }

          gtk_tree_model_get( list, &it, COL_FILE_TYPE, &mime, -1);

          if( app_desktop ) {
            app = g_strdup( app_desktop );
          }
          else {
            app = get_default_app_for_mime_type(mime);
          }

          if( app )
          {
            /* Check whether this is an app desktop file or just a command line */
            if( g_str_has_suffix (app, ".desktop") ) {
              exec = get_exec_from_desktop_file(app);
              app_desktop = app;
            }
            else {
              if( ! strchr( app, '%' ) )  { /* No filename parameters */
                exec = g_strconcat( app, " %f", NULL );
              }
              else {
                exec = g_strdup( app );
              }
              app_desktop = NULL;
            }

            if( exec ) {
              file_list[0] = new_path;
              file_list[1] = NULL;
              cmd = translate_app_exec_to_command_line(exec, app_desktop, file_list);

              if( cmd ) {
                g_print( "Debug: Execute %s\n", cmd );
                g_spawn_command_line_async( cmd, NULL );
                g_free( cmd );
              }
              g_free( app );
              g_free( exec );
            }
          }
        }
       g_free( new_path );
       break;
      }
    }
  }
  g_list_foreach( sel_items, (GFunc)gtk_tree_path_free, NULL );
  g_list_free( sel_items );
}

/* Signal handlers for popup menu */
static void on_popup_selection_done( GtkMenuShell* popup, gpointer data )
{
  GList* additional_items = (GList*)g_object_get_data( G_OBJECT( popup ),
                                                       "additional_items");
  if( additional_items ){
    g_list_foreach( additional_items, (GFunc)gtk_widget_destroy, NULL );
    g_list_free( additional_items );
  }
}

static GtkWidget* file_browser_create_basic_popup_menu()
{
  GtkWidget* popup;
  GtkWidget *open;
  GtkWidget *open_with;
  GtkWidget *seperator1;
  GtkWidget *open_with_another;
  GtkWidget *seperator2;
  GtkWidget *cut;
  GtkWidget *copy;
  GtkWidget *paste;
  GtkWidget *delete;
  GtkWidget *rename;
  GtkWidget *image365;
  GtkWidget *seperator3;
  GtkWidget *create_new;
  GtkWidget *image366;
  GtkWidget *create_new_menu;
  GtkWidget *new_folder;
  GtkWidget *image367;
  GtkWidget *new_text_file;
  GtkWidget *image368;
  GtkWidget *property;
  GtkWidget *image369;
  GtkAccelGroup *accel_group;
  accel_group = gtk_accel_group_new ();

  popup = gtk_menu_new ();

  open = gtk_image_menu_item_new_from_stock ("gtk-open", accel_group);
  gtk_widget_show (open);
  gtk_container_add (GTK_CONTAINER (popup), open);

  open_with = gtk_menu_item_new_with_mnemonic (_("Open _with..."));
  gtk_widget_show (open_with);
  gtk_container_add (GTK_CONTAINER (popup), open_with);
  g_object_set_data(G_OBJECT(popup), "open_with", open_with );

  seperator2 = gtk_separator_menu_item_new ();
  gtk_widget_show (seperator2);
  gtk_container_add (GTK_CONTAINER (popup), seperator2);
  gtk_widget_set_sensitive (seperator2, FALSE);

  cut = gtk_image_menu_item_new_from_stock ("gtk-cut", accel_group);
  gtk_widget_show (cut);
  gtk_container_add (GTK_CONTAINER (popup), cut);

  copy = gtk_image_menu_item_new_from_stock ("gtk-copy", accel_group);
  gtk_widget_show (copy);
  gtk_container_add (GTK_CONTAINER (popup), copy);

  paste = gtk_image_menu_item_new_from_stock ("gtk-paste", accel_group);
  gtk_widget_show (paste);
  gtk_container_add (GTK_CONTAINER (popup), paste);

  delete = gtk_image_menu_item_new_from_stock ("gtk-delete", accel_group);
  gtk_widget_show (delete);
  gtk_container_add (GTK_CONTAINER (popup), delete);

  rename = gtk_image_menu_item_new_with_mnemonic (_("_Rename"));
  gtk_widget_show (rename);
  gtk_container_add (GTK_CONTAINER (popup), rename);
  gtk_widget_add_accelerator (rename, "activate", accel_group,
                              GDK_F2, (GdkModifierType) 0,
                              GTK_ACCEL_VISIBLE);

  image365 = gtk_image_new_from_stock ("gtk-edit", GTK_ICON_SIZE_MENU);
  gtk_widget_show (image365);
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (rename), image365);

  seperator3 = gtk_separator_menu_item_new ();
  gtk_widget_show (seperator3);
  gtk_container_add (GTK_CONTAINER (popup), seperator3);
  gtk_widget_set_sensitive (seperator3, FALSE);

  create_new = gtk_image_menu_item_new_with_mnemonic (_("_Create New"));
  gtk_widget_show (create_new);
  gtk_container_add (GTK_CONTAINER (popup), create_new);
  g_object_set_data( G_OBJECT(popup), "create_new", create_new );

  image366 = gtk_image_new_from_stock ("gtk-new", GTK_ICON_SIZE_MENU);
  gtk_widget_show (image366);
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (create_new), image366);

  create_new_menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (create_new), create_new_menu);

  new_folder = gtk_image_menu_item_new_with_mnemonic (_("_Folder"));
  gtk_widget_show (new_folder);
  gtk_container_add (GTK_CONTAINER (create_new_menu), new_folder);

  image367 = gtk_image_new_from_stock ("gtk-directory", GTK_ICON_SIZE_MENU);
  gtk_widget_show (image367);
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (new_folder), image367);

  new_text_file = gtk_image_menu_item_new_with_mnemonic (_("_Text File"));
  gtk_widget_show (new_text_file);
  gtk_container_add (GTK_CONTAINER (create_new_menu), new_text_file);

  image368 = gtk_image_new_from_stock ("gtk-edit", GTK_ICON_SIZE_MENU);
  gtk_widget_show (image368);
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (new_text_file), image368);

  property = gtk_image_menu_item_new_with_mnemonic (_("_Property"));
  gtk_widget_show (property);
  gtk_container_add (GTK_CONTAINER (popup), property);
  gtk_widget_add_accelerator (property, "activate", accel_group,
                              GDK_Return, (GdkModifierType) GDK_MOD1_MASK,
                              GTK_ACCEL_VISIBLE);

  image369 = gtk_image_new_from_stock ("gtk-info", GTK_ICON_SIZE_MENU);
  gtk_widget_show (image369);
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (property), image369);

  g_signal_connect ((gpointer) popup, "selection-done",
                     G_CALLBACK (on_popup_selection_done),
                     NULL );
  g_signal_connect ((gpointer) open, "activate",
                     G_CALLBACK (on_popup_open_activate),
                     popup);
  g_signal_connect ((gpointer) cut, "activate",
                     G_CALLBACK (on_popup_cut_activate),
                     popup);
  g_signal_connect ((gpointer) copy, "activate",
                     G_CALLBACK (on_popup_copy_activate),
                     popup);
  g_signal_connect ((gpointer) paste, "activate",
                     G_CALLBACK (on_popup_paste_activate),
                     popup);
  g_signal_connect ((gpointer) delete, "activate",
                     G_CALLBACK (on_popup_delete_activate),
                     popup);
  g_signal_connect ((gpointer) rename, "activate",
                     G_CALLBACK (on_popup_rename_activate),
                     popup);
  g_signal_connect ((gpointer) new_folder, "activate",
                     G_CALLBACK (on_popup_new_folder_activate),
                     popup);
  g_signal_connect ((gpointer) new_text_file, "activate",
                     G_CALLBACK (on_popup_new_text_file_activate),
                     popup);
  g_signal_connect ((gpointer) property, "activate",
                     G_CALLBACK (on_popup_file_property_activate),
                     popup);
  return popup;
}

/* Retrive popup menu for selected files */
GtkWidget* file_browser_get_popup_menu_for_mime_type( FileBrowserData* data,
                                                      const char* mime_type )
{
  static GtkWidget* popup = NULL;
  GtkWidget *open;
  GtkWidget *open_in_new_tab;
  GtkWidget *open_with;
  GtkWidget *open_with_menu;
  GtkWidget *seperator;
  GtkWidget *open_with_another;

  GtkWidget *image365;
  GtkWidget *create_new;
  GtkWidget *image366;
  GtkWidget *create_new_menu;
  GtkWidget *new_folder;
  GtkWidget *image367;
  GtkWidget *new_text_file;
  GtkWidget *image368;

  GtkWidget *app_menu_item;
  GList* additional_items = NULL;
  gboolean is_dir;

  char **apps, **app;
  char* app_name;

  GdkPixbuf* app_icon;
  GtkWidget* app_img;

  if( ! popup ){
    popup = file_browser_create_basic_popup_menu();
  }

  g_object_set_data( G_OBJECT(popup), "FileBrowserData", data );

  /*  g_print("mime_type: %s!\n", mime_type); */

  open_with = GTK_WIDGET( g_object_get_data( G_OBJECT(popup), "open_with") );
  create_new = GTK_WIDGET( g_object_get_data( G_OBJECT(popup), "create_new") );
  /* Add some special menu items */
  if( mime_type == XDG_MIME_TYPE_DIRECTORY )
  {
    open_in_new_tab = gtk_menu_item_new_with_mnemonic (_("Open in New _Tab"));
    gtk_widget_show (open_in_new_tab);
    gtk_menu_shell_insert ( GTK_MENU_SHELL(popup), open_in_new_tab, 1 );
    additional_items = g_list_prepend( additional_items, open_in_new_tab );

    is_dir = TRUE;
    gtk_widget_show( create_new );

    g_signal_connect ((gpointer) open_in_new_tab, "activate",
                       G_CALLBACK (on_popup_open_in_new_tab_activate),
                       data);
  }
  else{
    is_dir = FALSE;
    gtk_widget_show( open_with );
    gtk_widget_hide( create_new );
  }

  /*  Add all of the apps  */

  open_with_menu = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (open_with), open_with_menu);

  apps = get_all_apps_for_mime_type( mime_type );
  if( ! apps )  /* fallback for text files */
  {
   if( 0 == strncmp( mime_type, "text/", 5 ) ) {
      apps = get_all_apps_for_mime_type( "text/plain" );
    }
  }

  if( apps ){
    for( app = apps; *app; ++app ){
      if( (app - apps) == 1 ) /* Add a separator after default app */
      {
        seperator = gtk_separator_menu_item_new ();
        gtk_widget_show (seperator);
        gtk_container_add (GTK_CONTAINER (open_with_menu), seperator);
        gtk_widget_set_sensitive (seperator, FALSE);
      }

      app_name = get_app_name_from_desktop_file( *app );
      if( app_name ){
        app_menu_item = gtk_image_menu_item_new_with_label ( app_name );
        g_free( app_name );
      }
      else{
        app_menu_item = gtk_image_menu_item_new_with_label ( *app );
      }
      g_object_set_data_full( G_OBJECT(app_menu_item), "desktop_file",
                              *app, g_free );
      app_icon = get_app_icon_from_desktop_file( *app, 16 );
      app_img = gtk_image_new_from_pixbuf( app_icon );
      gtk_image_menu_item_set_image ( app_menu_item, app_img );
      /*
      *  Little trick: remove the string from string vector so that g_strfreev()
      *  will not free the string *app.
      */
      *app = NULL;

      gtk_widget_show(app_menu_item );
      gtk_container_add (GTK_CONTAINER (open_with_menu), app_menu_item );

      g_signal_connect( G_OBJECT(app_menu_item), "activate",
                        G_CALLBACK(on_popup_run_app), (gpointer)data );
    }
    g_strfreev( apps );
  }
  seperator = gtk_separator_menu_item_new ();
  gtk_widget_show (seperator);
  gtk_container_add (GTK_CONTAINER (open_with_menu), seperator);
  gtk_widget_set_sensitive (seperator, FALSE);

  open_with_another = gtk_menu_item_new_with_mnemonic (_("_Open with another program"));
  gtk_widget_show (open_with_another);
  gtk_container_add (GTK_CONTAINER (open_with_menu), open_with_another);
  g_signal_connect ((gpointer) open_with_another, "activate",
                    G_CALLBACK (on_popup_open_with_another_activate),
                    data);


  g_object_set_data( G_OBJECT(popup), "additional_items",
                     (gpointer)additional_items );

  return popup;
}


void
on_popup_open_activate                       (GtkMenuItem     *menuitem,
                                              gpointer         user_data)
{
  FileBrowserData* data;
  data = (FileBrowserData*)g_object_get_data(G_OBJECT(user_data),
                                             "FileBrowserData");
  file_browser_open_selected_files_with_app( data, NULL );
}

void
on_popup_open_with_another_activate          (GtkMenuItem     *menuitem,
                                              gpointer         user_data)
{
  GtkWidget* dlg;
  char* app = NULL;
  GtkTreeModel* model;
  GtkTreeIter it;
  const mime_type;
  GList* sel_files;
  FileBrowserData* data = (FileBrowserData*)user_data;

  sel_files = folder_view_get_selected_items(data, &model);
  gtk_tree_model_get_iter( model, &it, (GtkTreePath*)sel_files->data );
  gtk_tree_model_get( model, &it, COL_FILE_TYPE, &mime_type, -1 );
  g_list_foreach( sel_files, (GFunc)gtk_tree_path_free, NULL );
  g_list_free( sel_files );

  dlg = app_chooser_dialog_new(GTK_WINDOW(data->mainWindow), mime_type);

  if( gtk_dialog_run( GTK_DIALOG(dlg) ) == GTK_RESPONSE_OK )
  {
    app = app_chooser_dialog_get_selected_app(dlg);
    if( app ) {
      /* The selected app is set to default action */
      if( app_chooser_dialog_get_set_default(dlg) ){
        set_default_app_for_mime_type( mime_type, app );
      }
      file_browser_open_selected_files_with_app( data, app );
      g_free( app );
    }
  }
  gtk_widget_destroy( dlg );
}

void on_popup_run_app( GtkMenuItem *menuitem, FileBrowserData* data )
{
  char* desktop_file = g_object_get_data( G_OBJECT(menuitem), "desktop_file");
  if( !desktop_file )
    return;
  file_browser_open_selected_files_with_app( data, desktop_file );
}

void on_popup_open_in_new_tab_activate( GtkMenuItem *menuitem,
                                        FileBrowserData* data )
{
  GList* sel;
  struct stat file_stat;
  GList* sel_files = file_browser_get_selected_files( data );
  if( sel_files ) {
    for( sel = sel_files; sel; sel = sel->next ) {
      stat( (char*)sel->data, &file_stat );
      if( S_ISDIR(file_stat.st_mode) ) {
        add_new_tab( data->mainWindow,
                    (char*)sel->data, data->show_dir_tree);
      }
    }
    g_list_foreach( sel_files, (GFunc)g_free, NULL );
    g_list_free( sel_files );
  }
  else {
    add_new_tab( data->mainWindow,
                 (char*)data->curHistory->data, data->show_dir_tree);
  }
}

void
on_popup_cut_activate                      (GtkMenuItem     *menuitem,
                                            gpointer         user_data)
{
  GObject* popup = G_OBJECT(user_data);
  FileBrowserData* data = (FileBrowserData*)g_object_get_data(popup,
                                                         "FileBrowserData" );
  if( data )
    file_browser_cut( data->folderPane );
}


void
on_popup_copy_activate                       (GtkMenuItem     *menuitem,
                                            gpointer         user_data)
{
  GObject* popup = G_OBJECT(user_data);
  FileBrowserData* data = (FileBrowserData*)g_object_get_data(popup,
                                                      "FileBrowserData" );
  if( data )
    file_browser_copy( data->folderPane );
}


void
on_popup_paste_activate                      (GtkMenuItem     *menuitem,
                                              gpointer         user_data)
{
  GObject* popup = G_OBJECT(user_data);
  FileBrowserData* data = (FileBrowserData*)g_object_get_data(popup,
                                                        "FileBrowserData" );
  if( data )
    file_browser_paste( data->folderPane );
}


void
on_popup_delete_activate                     (GtkMenuItem     *menuitem,
                                              gpointer         user_data)
{
  GObject* popup = G_OBJECT(user_data);
  FileBrowserData* data = (FileBrowserData*)g_object_get_data(popup,
                                                      "FileBrowserData" );
  if( data )
    file_browser_delete( data->folderPane );
}

void
on_popup_rename_activate                      (GtkMenuItem     *menuitem,
                                               gpointer         user_data)
{
  GObject* popup = G_OBJECT(user_data);
  FileBrowserData* data = (FileBrowserData*)g_object_get_data(popup,
                                                      "FileBrowserData" );
 if( data )
    file_browser_rename_selected_file( data->folderPane );
}

void
on_popup_new_folder_activate                  (GtkMenuItem     *menuitem,
                                               gpointer         user_data)
{
  GObject* popup = G_OBJECT(user_data);
  FileBrowserData* data = (FileBrowserData*)g_object_get_data(popup,
                                                  "FileBrowserData" );
/*
  if( data )
    file_browser( data->folderPane );
*/
}
void
on_popup_new_text_file_activate               (GtkMenuItem     *menuitem,
                                               gpointer         user_data)
{

}

void
on_popup_file_property_activate               (GtkMenuItem     *menuitem,
                                               gpointer         user_data)
{
  GtkWidget* dlg;
  GObject* popup = G_OBJECT(user_data);
  GList* sel_files;
  FileBrowserData* data = (FileBrowserData*)g_object_get_data(popup,
                                                          "FileBrowserData" );

  if( data ) {
    sel_files = file_browser_get_selected_files( data );
    dlg = create_filePropertiesDlg();
    gtk_window_set_transient_for( GTK_WINDOW(dlg),
                                  GTK_WINDOW(data->mainWindow) );

    if( !sel_files ){
      sel_files = g_list_append( sel_files, g_strdup((char*)data->curHistory->data) );
    }
    file_properties_dlg_init( dlg, sel_files );

    gtk_widget_show( dlg );
  }
}

void file_browser_show_hidden_files( GtkWidget* file_browser,
                                     gboolean show )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  data->show_hidden_files = show;
  gtk_tree_model_filter_refilter( GTK_TREE_MODEL_FILTER(data->list_filter) );
  if( data->tree_filter )
    gtk_tree_model_filter_refilter( GTK_TREE_MODEL_FILTER(data->tree_filter) );
}

void file_browser_create_dir_tree( FileBrowserData* data )
{
  GtkWidget* dirTree;
  GtkTreeSelection* dirTreeSel;
  data->dirTreeScroll = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (data->dirTreeScroll);

  gtk_paned_pack1 (GTK_PANED (data->folderPane),
                   data->dirTreeScroll,
                   FALSE, FALSE);

  gtk_scrolled_window_set_policy (
      GTK_SCROLLED_WINDOW(data->dirTreeScroll),
  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  gtk_scrolled_window_set_shadow_type (
      GTK_SCROLLED_WINDOW (data->dirTreeScroll),
  GTK_SHADOW_IN);

  dirTree = gtk_tree_view_new ();
  gtk_tree_view_set_headers_visible( GTK_TREE_VIEW(dirTree), FALSE );
  gtk_widget_show ( dirTree );
  gtk_container_add (GTK_CONTAINER (data->dirTreeScroll),
                     dirTree );

  gtk_widget_set_size_request ( dirTree, 40, -1);

  dirTreeSel = gtk_tree_view_get_selection( GTK_TREE_VIEW(dirTree) );

  g_signal_connect ( dirTreeSel, "changed",
                     G_CALLBACK (on_dir_tree_sel_changed),
                     data);

  g_signal_connect ( dirTree, "row-expanded",
                     G_CALLBACK (on_dir_tree_row_expanded),
                     data);

  data->dirTree = GTK_TREE_VIEW ( dirTree );

  /* This causes serious problems because of some bugs of GTK+,
     so I disabled it currently */
  /*
  gtk_tree_view_enable_model_drag_source ( GTK_TREE_VIEW(dirTree),
                                           (GDK_CONTROL_MASK|GDK_BUTTON1_MASK|GDK_BUTTON3_MASK),
                                           dragTargets,
                                           sizeof(dragTargets)/sizeof(GtkTargetEntry),
                                           GDK_ACTION_DEFAULT|GDK_ACTION_COPY|GDK_ACTION_MOVE|GDK_ACTION_LINK);
  */

  gtk_tree_view_enable_model_drag_dest ( GTK_TREE_VIEW(dirTree),
                                         dragTargets,
                                         sizeof(dragTargets)/sizeof(GtkTargetEntry),
                                         GDK_ACTION_DEFAULT|GDK_ACTION_COPY|GDK_ACTION_MOVE|GDK_ACTION_LINK);

}

void file_browser_show_dir_tree( GtkWidget* file_browser,
                                 gboolean show )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  data->show_dir_tree = show;
  if( show )
  {
    if( ! data->dirTreeScroll) {
      file_browser_create_dir_tree( data );
      init_dir_tree( data );
      dir_tree_chdir( data, (char*)data->curHistory->data );
    }
    gtk_widget_show( data->dirTreeScroll );
  }
  else
  {
    gtk_widget_hide( data->dirTreeScroll );
  }
}

void file_browser_sort_by_name( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  GtkTreeSortable* sortable;
  if( data->sort_order == FB_SORT_BY_NAME )
    return;
  data->sort_order = FB_SORT_BY_NAME;
  if( data->list_sorter )
  {
    sortable = GTK_TREE_SORTABLE(data->list_sorter);
    gtk_tree_sortable_set_sort_func( sortable, COL_FILE_NAME,
                                    sort_files_by_name, data,
                                    NULL );
    gtk_tree_sortable_set_sort_column_id( sortable,
                                          COL_FILE_NAME,
                                          data->sort_type );
    update_folder_view_visible_region(data);
  }
}

void file_browser_sort_by_size( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  GtkTreeSortable* sortable;
  if( data->sort_order == FB_SORT_BY_SIZE )
    return;
  data->sort_order = FB_SORT_BY_SIZE;
  if( data->list_sorter )
  {
    sortable = GTK_TREE_SORTABLE(data->list_sorter);
    gtk_tree_sortable_set_sort_func( sortable, COL_FILE_STAT,
                                    sort_files_by_size, data,
                                    NULL );
    gtk_tree_sortable_set_sort_column_id( sortable,
                                          COL_FILE_STAT,
                                          data->sort_type );
    update_folder_view_visible_region(data);
  }
}

void file_browser_sort_by_time( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  GtkTreeSortable* sortable;
  if( data->sort_order == FB_SORT_BY_TIME )
    return;
  data->sort_order = FB_SORT_BY_TIME;
  if( data->list_sorter )
  {
    sortable = GTK_TREE_SORTABLE(data->list_sorter);

    gtk_tree_sortable_set_sort_func( sortable, COL_FILE_STAT,
                                    sort_files_by_time, data,
                                    NULL );

    gtk_tree_sortable_set_sort_column_id( sortable,
                                          COL_FILE_STAT,
                                          data->sort_type );
    update_folder_view_visible_region(data);
  }
}

void file_browser_sort_ascending( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  GtkTreeSortable* sortable;
  gint col;
  GtkSortType order;
  if( data->sort_type != GTK_SORT_ASCENDING )
  {
    data->sort_type = GTK_SORT_ASCENDING;
    sortable = GTK_TREE_SORTABLE(data->list_sorter);
    gtk_tree_sortable_get_sort_column_id ( sortable, &col, &order );
    gtk_tree_sortable_set_sort_column_id ( sortable, col, GTK_SORT_ASCENDING );
  }
  update_folder_view_visible_region(data);
}

void file_browser_sort_descending( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  GtkTreeSortable* sortable;
  gint col;
  GtkSortType order;
  if( data->sort_type != GTK_SORT_DESCENDING )
  {
    data->sort_type = GTK_SORT_DESCENDING;
    sortable = GTK_TREE_SORTABLE(data->list_sorter);
    gtk_tree_sortable_get_sort_column_id ( sortable, &col, &order );
    gtk_tree_sortable_set_sort_column_id ( sortable, col, GTK_SORT_DESCENDING );
  }
  update_folder_view_visible_region(data);
}

void file_browser_view_as_icons( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  if( data->view_mode == FBVM_ICON_VIEW )
    return;
  data->view_mode = FBVM_ICON_VIEW;
  gtk_widget_destroy( data->folderView );
  data->folderView = create_folder_view( FBVM_ICON_VIEW, data );
  ptk_icon_view_set_model( PTK_ICON_VIEW(data->folderView), data->list_filter );
  gtk_widget_show( data->folderView );
  gtk_container_add( GTK_CONTAINER( data->folderViewScroll ), data->folderView );
  g_idle_add( (GSourceFunc)update_folder_view_visible_region, data);
}

void file_browser_view_as_list ( GtkWidget* file_browser )
{
  FileBrowserData* data = file_browser_get_data( file_browser );
  if( data->view_mode == FBVM_LIST_VIEW )
    return;
  data->view_mode = FBVM_LIST_VIEW;
  gtk_widget_destroy( data->folderView );
  data->folderView = create_folder_view( FBVM_LIST_VIEW, data );
  gtk_tree_view_set_model( GTK_TREE_VIEW(data->folderView), data->list_filter );
  gtk_widget_show( data->folderView );
  gtk_container_add( GTK_CONTAINER( data->folderViewScroll ), data->folderView );
  g_idle_add( (GSourceFunc)update_folder_view_visible_region, data);
}

