/* $Id: tcp.c,v 1.15 2001/01/05 01:10:51 keiji Exp $ */
/*
** tcp.c - tcp part of pakemon
** Author : Keiji Takeda
**
** pakemon is an abbreviation of "Packet Monster", a simple packet
** monitoring misuse detector.
** Copyright (C) 1999, 2000 Keiji Takeda <keiji@sfc.keio.ac.jp>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "util.h"
#include "signature.h"

/************************************************
  Function: tcp_callback
  Purpose: handle captured (and reconstructed)
           tcp
 ***********************************************/

void tcp_callback (struct tcp_stream *a_tcp, void ** dummy ) {

  struct ip_info ip_packet; /* pakemon.h */

  struct ip ip_header;

  struct in_addr buf_in_addr;

  char buf_IP[16];

  int buf_number;

  /*** initialize ip_packet with 0 ***/
  memset(&ip_packet, 0, sizeof(ip_packet));

  /*** Extract IPs ***/
  strcpy( ip_packet.src_IP, inet_ntoa ( *(struct in_addr *)&( (a_tcp->addr).saddr) ) );

  strcpy( ip_packet.dest_IP, inet_ntoa ( *(struct in_addr *)&( (a_tcp->addr).daddr) ) );

  memset( &ip_header, 0, sizeof(struct ip) );

  ip_header.ip_p = IPPROTO_TCP;

  (ip_header.ip_dst).s_addr = (a_tcp->addr).daddr;

  (ip_header.ip_src).s_addr = (a_tcp->addr).saddr;

  ip_packet.ip_header = &ip_header;

  /*** Extract Port Numbers ***/
  ip_packet.src_number = (a_tcp->addr).source;

  ip_packet.dest_number = (a_tcp->addr).dest;

  if ( a_tcp->nids_state == NIDS_JUST_EST )
    {
      a_tcp->client.collect++;

      a_tcp->server.collect++;

      ip_packet.payload = "[CONNECTION ESTABLISHED]";

      ip_packet.payload_len = strlen( ip_packet.payload ); 

      ip_packet.endof_payload = ip_packet.payload + ip_packet.payload_len; 

      if(pkmprm.verbose_mode)
	dump_packet_info( ip_packet, local_netaddr, local_netmask);

      return;
    }
  else if ( (a_tcp->nids_state == NIDS_CLOSE) || (a_tcp->nids_state == NIDS_RESET)
 || (a_tcp->nids_state == NIDS_TIMED_OUT)  )
    {

      struct report *i_report, *j_report;

      char report_id[DEFAULT_BUFFER_SIZE];

      i_report = head_report;

      j_report = i_report;

      sprintf( report_id, "%s%lu%s%lu", 
	       ip_packet.src_IP, ip_packet.src_number,
	       ip_packet.dest_IP, ip_packet.dest_number );
      
      /*** check if the report already exists ***/
      while(i_report != (struct report *)NULL )
	{
	  if( !strcmp( i_report->report_id, report_id ) )
	    {
	      if( i_report == head_report )
		head_report = i_report->next_report;

	      j_report->next_report = i_report->next_report;

	      free_check( i_report,"tcp_call_back:i_report" );

	      i_report = j_report->next_report;

	    }

	  j_report = i_report;

	  i_report = i_report->next_report;

	}

      /*** report is not found ***/

      ip_packet.payload = "[CONNECTION CLOSED]";

      ip_packet.payload_len = strlen(ip_packet.payload); 

      ip_packet.endof_payload = ip_packet.payload + ip_packet.payload_len; 

      if(pkmprm.verbose_mode)
	dump_packet_info( ip_packet, local_netaddr, local_netmask);
      
      return;

    }
  else if ( a_tcp->nids_state == NIDS_DATA )
    {

      struct half_stream *hlf;

      if ( a_tcp->client.count_new )
	{

	  hlf = &a_tcp->client;

	  /* swap IPs */
	  strcpy( buf_IP, ip_packet.src_IP );
	  strcpy( ip_packet.src_IP, ip_packet.dest_IP );
	  strcpy( ip_packet.dest_IP, buf_IP );

	  buf_in_addr = (ip_packet.ip_header)->ip_src;
	  (ip_packet.ip_header)->ip_src = (ip_packet.ip_header)->ip_dst;
	  (ip_packet.ip_header)->ip_dst = buf_in_addr;
	  
	  /* swap ports */
	  buf_number = ip_packet.src_number;
	  ip_packet.src_number  = ip_packet.dest_number;
	  ip_packet.dest_number = buf_number;

	}
      else
	hlf = &a_tcp->server;

      ip_packet.payload = hlf->data;

      ip_packet.payload_len = a_tcp->read;

      ip_packet.endof_payload = hlf->data + a_tcp->read;

      read_tcp( &ip_packet );

      if(pkmprm.verbose_mode)
	dump_packet_info( ip_packet, local_netaddr, local_netmask);
      
      nids_discard(a_tcp, hlf->count - hlf->offset - sig_len.max);

    }

  return;

} /* tcp_callback */

/*****************************************************
  Function: read_tcp
  Purpose: read payload of a tcp packet
           read_payload cannot be used since segment maybe
	   overlapped.
  Arguments:struct ip_info *ip_packet : IP information package.
******************************************************/
void read_tcp(struct ip_info *ip_packet)
{

  char report_id[DEFAULT_BUFFER_SIZE];

  int i;

  FILE *alert_log = (FILE *)NULL ;

  FILE *dump_log = (FILE *)NULL ;

  /* pointer to a data stores record to be reported */
  struct report *i_report, *j_report;

  u_char *capital; /* buffer to store capitalized payload */

  c_node * detect_sig = (c_node *)NULL;

  s_node **s_tree_index;

  c_node **c_tree_index;

  time_t detection_time;

  /*** pattern match with signature ***/

  s_tree_index = malloc_check( sizeof( s_node *), "read_tcp: s_tree_index" );

  c_tree_index = malloc_check( sizeof( c_node *), "read_tcp: c_tree_index" );

  capital = malloc_check( ( ip_packet->payload_len + 1 ) * sizeof(u_char), "read_tcp: capital" );

  if( 
     ( s_tree_index == ( s_node **)NULL ) ||
     ( c_tree_index == ( c_node **)NULL ) ||
     ( capital == (u_char *)NULL )
     )
    {
      fputs( "Memory allocation error in read_tcp.",stderr );
      return;
    }

  *s_tree_index = tcp_tree;

  *c_tree_index = (c_node *)NULL;

  /* making capital string */
  for( i = 0; i < ip_packet->payload_len; i++ )
    {
      *(capital + i) = toupper( *( ip_packet->payload + i ) );
    }

  /* termination of data array, just for emergency */
  *( capital + ip_packet->payload_len ) = (u_char)NULL; 

  while( (detect_sig = detect_signature( ip_packet, capital, s_tree_index, c_tree_index ))
	 != (c_node *)NULL )
    {

      i_report = head_report;

      j_report = i_report;

      sprintf( report_id, "%s%lu%s%lu%d", 
	       ip_packet->src_IP, ip_packet->src_number,
	       ip_packet->dest_IP, ip_packet->dest_number,
	       detect_sig->sig_id );

      /*** check if the report already exists ***/
      while( i_report != (struct report *)NULL )
	{
	      
	  if( !strcmp(i_report->report_id, report_id) )
	      break;

	  j_report = i_report;

	  i_report = i_report->next_report;

	}

      /*** report is found then skip ***/

      if(i_report != (struct report *)NULL )
	{

	  *c_tree_index = (*c_tree_index)->next;
      
	  if( *c_tree_index == (c_node *)NULL )
	    *s_tree_index = (*s_tree_index)->next;

	  continue;

	}

      /*** report is not found ***/

      /*** create new report ***/
      if( head_report == (struct report *)NULL )
	{

	  /*** create new report ***/
	  i_report = (struct report *)malloc_check( sizeof(struct report), "read_tcp:i_report"  );
	      
	  /*** head report is the new record ***/
	  head_report = i_report;

	  /* store the file name in the report record */
	  strcpy( i_report->report_id, report_id );
	      
	  i_report->next_report = (struct report *)NULL;

	} else if( i_report == (struct report *)NULL ) {
	      
	  /*** add new node ***/
	  /*** create new record ***/
	  i_report =(struct report *)malloc_check(sizeof(struct report), "read_tcp:i_report2");

	  /* store the connection_id in the report record */
	  strcpy(i_report->report_id, report_id);

	  i_report->next_report = (struct report *)NULL;

	  /* update j_report to point the new report */
	  j_report->next_report = i_report;

	}
      
      *c_tree_index = (*c_tree_index)->next;
      
      if( *c_tree_index == (c_node *)NULL )
	{
	  *s_tree_index = (*s_tree_index)->next;
	}
      
      /* print aleart message */
      time( &detection_time );
      
      if( pkmprm.std_alert )
	{

	  printf("ALERT %sTCP %s(%lu)->%s(%lu)%s \n", 
		 ctime( &detection_time ),
		 ip_packet->src_IP, ip_packet->src_number,
		 ip_packet->dest_IP, ip_packet->dest_number,
		 detect_sig->name
		 );
	
	  /* make beep sound */
	  putchar('\007');
	  fflush(stdout);
	  
	}

      if( pkmprm.use_syslog )					    
	{
	  syslog(LOG_ALERT,
		 "ALERT TCP %s(%lu)->%s(%lu) %s\n", 
		 ip_packet->src_IP, ip_packet->src_number,
		 ip_packet->dest_IP, ip_packet->dest_number,
		 detect_sig->name
		 );
	}
      else if( pkmprm.alert_log_name != (char *)NULL )
	{
	  if( ( alert_log = fopen( pkmprm.alert_log_name, "a"  )) != NULL)
	    {
	      fprintf( alert_log, "\nTCP %s(%lu)->%s(%lu) %s",
		       ip_packet->src_IP, ip_packet->src_number,
		       ip_packet->dest_IP, ip_packet->dest_number,
		       ctime( &detection_time ) 
		       );

	      fprintf( alert_log, "%s\n", detect_sig->name );

	      fclose( alert_log );

	    }
	}

      if(pkmprm.dump_log_mode == LOG_NONE )
	  continue;

      if ( pkmprm.dump_log_name == (char *)NULL )
	{
	  fprintf(stderr, "No dump_log_name.");
	  continue;
	}
	  
      if( ( dump_log = fopen( pkmprm.dump_log_name, "a" ) ) == (FILE *)NULL )
	{
	  fprintf(stderr, "Failed to open dump log file.");
	  continue;
	}

      /* log detected trafic */
      fprintf( dump_log, "\nTCP %s(%lu)->%s(%lu) %s",
	       ip_packet->src_IP, ip_packet->src_number,
	       ip_packet->dest_IP, ip_packet->dest_number,
	       ctime( &detection_time ) 
	       );

      fprintf( dump_log, "%s (TCP %lu %lu)\n", 
	      detect_sig->name,
	      ((s_node *)detect_sig->s_node)->src_number,
	      ((s_node *)detect_sig->s_node)->dest_number
	      );

      if( pkmprm.dump_log_mode == LOG_TEXT )
	text_dump_payload( dump_log, ip_packet, TERM_WIDTH );
      
      if( pkmprm.dump_log_mode == LOG_BIN )
	bin_dump_payload( dump_log, ip_packet );

      fclose( dump_log );

    } /* while */

  free_check( s_tree_index,"read_tcp:s_tree_index" );
  free_check( c_tree_index, "read_tcp:c_tree_index" );
  free_check( capital, "read_tcp:capital" );
  return;

} /* read_tcp */




