/* $Id: signature.c,v 1.12 2000/10/23 11:21:06 keiji Exp $ */
/*
** signature.c : signature part of pakemon.
**
** Pakemon is an abbreviation of "Packet Monster", a simple packet
** monitoring misuse detector.
**
** Copyright (C) 1999, 2000 Keiji Takeda <keiji@sfc.keio.ac.jp>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "signature.h"
#include "util.h"

/****************************************************************
  Function: get_next_str
  Purpose: Utility for signature perser just skip a string.
  Argument:
            char **reader : pointer for signature reader
	    int mas : maximum index
  Returns :
            int index for next string 
*****************************************************************/
int get_next_str( char **reader, int max ){

  int i;

  char *next;

  i = 0;
  while ( isspace( **reader ) )
    {
      (*reader)++;

      if ( i++ >= max )
	{
	  fputs("Signature Format Error", stderr );
	  final( 0 );
	}
    }

  i = 0;

  next = *reader;

  while ( !isspace( *next ) )
    {
      next++;
      i++;
      if ( i >= max )
	{
	  fputs( "Signature Format Error", stderr );
	  final( 0 );
	}
    }
  return i;
} /* get_next_str */

/****************************************************************
  Function: new_s_node
  Purpose: allocate and initialize new service node
  Argument: u_long service_number
  Returns : s_node * : pointer to the new node
*****************************************************************/
s_node *new_s_node( u_long src_number, u_long dest_number )
{

  s_node *new_node = malloc_check( sizeof(s_node), "new_s_node:new_node" );
  
  if( new_node == (s_node *)NULL )
    {
      fputs( "Memory allocation error for a service node.",stderr );
      final( 0 );
    }

  new_node->src_number = src_number;

  new_node->dest_number = dest_number;

  new_node->c_tree = (c_node *)NULL;

  new_node->next = (s_node *)NULL;

  return new_node;

} /* new_s_node */

/****************************************************************
  Function: new_c_node
  Purpose: allocate and initialize new content node
  Argument: 
  Returns : c_node * : pointer to the new node
*****************************************************************/
c_node *new_c_node(s_node *service,  u_int sig_id, u_char *pattern, u_int pattern_len, char * name, u_char pattern_mode )
{

  u_char namelen;

  char * re_err;

  c_node *new_node = malloc_check( sizeof(c_node), "new_c_node:new_node" );

  if( new_node == (c_node *)NULL )
    {
      fputs("Memory allocation error.", stderr );
      final( 0 );
    }

  new_node->s_node = service;

  new_node->sig_id = sig_id;

  new_node->pattern_mode = pattern_mode;

  if( new_node->pattern_mode == REGEXP ){

    new_node->pattern = (u_char *)NULL;

    new_node->pattern_len = 0;

    new_node->delta1 = (u_char *)NULL;

    new_node->delta2 = (u_char *)NULL;

    if( (const char *)re_err = re_compile_pattern( pattern, strlen( pattern ),&(new_node->reg_pattern))){
      fprintf( stderr, "new_c_node:re_compile_pattern %s\n", re_err );
      final(0);
    }

    (new_node->reg_pattern).fastmap = (char *) malloc_check( 256, "new_c_node:fastmap");

    if( re_compile_fastmap(&( new_node->reg_pattern )))
      {
	fputs ( "new_c_node:re_compile_fastmap", stderr );
	final( 0 );
      }

  } else {

    new_node->pattern = (u_char *)malloc_check( sizeof(u_char) * ( pattern_len+1 ), "new_c_node:new_node->pattern" );
    
    if( new_node->pattern == (u_char *)NULL)
      {
	fputs("Memory allocation failure for new_nonode->pattern", stderr);
	final( 0 );
      }

    memcpy( new_node->pattern, pattern, pattern_len );

    memset( new_node->pattern + pattern_len, 0, 1 );

    new_node->pattern_len = pattern_len;

    new_node->delta1 = bm_init_delta1( pattern, (u_char)pattern_len );

    new_node->delta2 = bm_init_delta2( pattern, (u_char)pattern_len );

    memset( &(new_node->reg_pattern), 0, sizeof(struct re_pattern_buffer ) );

  }

  new_node->next = (c_node *)NULL;
  
  namelen = sizeof(char) * ( strlen(name) );

  new_node->name = (char *)malloc_check( namelen + sizeof(char), "new_c_node:new_node->name" );

  if( new_node->name == (char *)NULL )
    {
      fputs("Memory allocation failure for new_node->name", stderr);
      final( 0 );
    }

  memcpy( new_node->name, name, namelen );

  memset( new_node->name + namelen, 0, 1);

  return new_node;

} /* new_c_node*/

/****************************************************************
  Function: get_c_tree
  Purpose: search c_tree for a specific protocol and service 
  Argument: char *sig_file :signature file name
  Returns : int : number of signature
                  -1 : could not open the file
*****************************************************************/
s_node * get_s_node( char *protocol, u_long src_number, u_long dest_number )
{

  s_node **s_tree_top = (s_node **)NULL;

  s_node *s_tree_index = (s_node *)NULL;

  s_node *s_tree_prev = (s_node *)NULL;

  /* tcp_tree, udp_tree, icmp_tree top node for each tree */

  if( !strncmp( protocol, "tcp", 3) )
    s_tree_top = &tcp_tree;
  else if(!strncmp( protocol, "udp", 3) )
    s_tree_top = &udp_tree;
  else if(!strncmp( protocol, "icmp", 4) )
    s_tree_top = &icmp_tree;
  else
    {
      fprintf( stderr, "Unknown protocol name %s", protocol );
      final( 0 );
    }

  if( *s_tree_top == (s_node *)NULL )
    {

      *s_tree_top = new_s_node( src_number, dest_number );

      s_tree_index = *s_tree_top;

    }
  else
    {
      /* start from s_tree_top */
      s_tree_index = *s_tree_top;

      /* search s_tree ordered by destination number */

      while( ( s_tree_index != (s_node *)NULL )
	     && ( s_tree_index->dest_number < dest_number ))
	{
		  
	  /* save current position */
	  s_tree_prev = s_tree_index;

	  /* forward */
	  s_tree_index = s_tree_index->next;

	} /* while */

      if( s_tree_index == (s_node *)NULL )
	{
	  /* create new node */
	  s_tree_prev->next = new_s_node( src_number, dest_number );

	  /* set current node */
	  s_tree_index = s_tree_prev->next;

	}
      else if( s_tree_index->dest_number != dest_number )
	{

	  if( s_tree_index == *s_tree_top )
	    {

	      /* create new node as top */
	      s_tree_prev = new_s_node( src_number, dest_number ); 

	      /* for temporary */
	      s_tree_prev->next = s_tree_index;

	      *s_tree_top = s_tree_prev;

	      s_tree_index = *s_tree_top;

	    }
	  else
	    {

	      /* create new node */
	      s_tree_prev->next = new_s_node( src_number, dest_number );

	      /* link from new node to next */
	      ( s_tree_prev->next )->next = s_tree_index;

	      /* set current node */
	      s_tree_index = s_tree_prev->next; 

	    } /* else s_tree */

	} /* else if s_tree->service_number */

    } /* else */

  while( ( s_tree_index != (s_node *)NULL )
	 && ( s_tree_index->dest_number == dest_number )
	 && ( s_tree_index->src_number < src_number ) )
    {
      /* save current position */
      s_tree_prev = s_tree_index;

      /* forward */
      s_tree_index = s_tree_index->next;

    } /* while */

  if( s_tree_index == (s_node *)NULL ) /* last node */
    {

      /* create new node */
      s_tree_prev->next = new_s_node( src_number, dest_number );

      /* set current node */
      s_tree_index = s_tree_prev->next;

    }
  else if( ( s_tree_index->dest_number != dest_number ) || ( s_tree_index->src_number != src_number ) )
    {

      /* create new node */
      s_tree_prev->next = new_s_node( src_number, dest_number );

      /* link from new node to next */
      ( s_tree_prev->next )->next = s_tree_index;

      /* set current node */
      s_tree_index = s_tree_prev->next; 

    } /* else s_tree */

  return ( s_tree_index );

}/* get_s_node */

/****************************************************************
  Function: init_signature data
  Purpose: initialize signature table reading from a signature file
  Argument: char *sig_file :signature file name
  Returns : int : number of signature
                  0 : initialization failure
*****************************************************************/
int init_signature(char *sig_file)
{
  
  FILE *fp;

  char line_buff[LARGE_BUF_SIZE]; /* line to read */

  char *sig_reader, *endof_sig, *line_end; /* pointers to read the line */

  int line_len, sig_id, read_len, pattern_len;

  char name_buf[STD_BUF_SIZE]; /* buffer for signature name */
  
  s_node *s_tree_index;

  c_node *c_tree_index;

  char *protocol;

  u_char pattern[LARGE_BUF_SIZE]; /* signature pattern data */

  u_long src_number = 0, dest_number = 0; /* service(port) numbers */

  u_char pattern_mode = CASE_SENSE;

  endof_sig = (char *)NULL;

  /*** open file ***/
  if( (FILE *)NULL == ( fp = fopen( sig_file, "r")) ){
    fprintf(stderr, "Signature file %s could not be opend.\n", sig_file );
    return ( 0 );
  }

  sig_len.max = sig_len.min = 0;

  sig_id = 0;

  /* process each line */
  while ( fgets( line_buff, LARGE_BUF_SIZE, fp ) != (char *)NULL  )
    {

      /* line_len = length of new line */
      line_len = strlen( line_buff );

      if( line_len >= LARGE_BUF_SIZE - 1 )
	{
	  fprintf( stderr, "Signature Format Error: A line is more than %d chars.\n%s" ,LARGE_BUF_SIZE -1, line_buff );
	  return ( 0 );
	}

      line_end = line_buff + line_len;

      sig_reader = line_buff; 

      /*********** comment line or NULL ***********/

      if ( (*sig_reader == '#')||(*sig_reader == '\n') ) continue;

      /*********** Signature Name **********/

      read_len = get_next_str( &sig_reader, line_len );

      if ( read_len >= STD_BUF_SIZE )
	{
	  fprintf( stderr, "Signature Format Error: signature name must be less than %d chars.\n%s\n" ,STD_BUF_SIZE, line_buff );
	  return ( 0 );
	}

      memcpy( name_buf, sig_reader, read_len );

      memset( name_buf + read_len, '\0', 1 );

      sig_reader += read_len;

      /*********** Protocol Name **********/

      read_len = get_next_str( &sig_reader, 
			       line_len - ( sig_reader - line_buff));

      if ( read_len >= STD_BUF_SIZE )
	{
	  fprintf( stderr, "Signature Format Error: protocol name must be less than %d chars.\n%s\n" ,STD_BUF_SIZE, line_buff );
	  return ( 0 );
	}

      protocol = sig_reader; /* point to protocol name */

      sig_reader += read_len;

      /*********** Service Number( Port Number ) source **********/

      read_len = 
	get_next_str( &sig_reader, line_len - ( sig_reader - line_buff) );

      if( *sig_reader == '*' )
	{
	  src_number = ULONG_MAX;
	}
      else
	{
	  src_number = strtoul( sig_reader, (char **)NULL, 10 );

	  if( (src_number == ULONG_MAX) && (errno == ERANGE)) 
	    {
	      fprintf(stderr, "Invalid Service Number %s in a Signature Definition\n%s\n", sig_reader,line_buff );
	      return ( 0 );
	    }
	}

      sig_reader += read_len;

      /*********** Service Number( Port Number ) destination **********/

      read_len = 
	get_next_str( &sig_reader, line_len - ( sig_reader - line_buff) );

      if( *sig_reader == '*' )
	{
	  dest_number = ULONG_MAX ; /* ULONG_MAX is wild card */
	}
      else
	{
	  dest_number = strtoul( sig_reader, (char **)NULL, 10 );

	  if( (src_number == ULONG_MAX) && (errno == ERANGE) ) 
	    {
	      fprintf(stderr, "Invalid Service Number %s in a Signature Definition\n%s\n", sig_reader,line_buff );
	      return( 0 );
	    }
	}

      sig_reader += read_len;

      /*********** Payload **********/

      read_len = get_next_str( &sig_reader, line_len - ( sig_reader - line_buff) );

      if( *sig_reader == '\"')
	{

	  /********** case sensitive payload description (start with double quote) **********/

	  endof_sig = index( sig_reader + 1, '\"' );

	  /* skip escape sequences */
	  while( (*(endof_sig - 1) == '\\') && ( endof_sig +1 < line_end -1 )) 
	    endof_sig = index( endof_sig + 1, '\"');

	  if( ( endof_sig > line_end - 1 ) || ( endof_sig == (char *)NULL ) )
	    {
	      fputs("Invalid Payload Description. No closing quote\n", stderr );
	      fputs( line_buff, stderr );
	      return ( 0 );
	    }

	  pattern_mode = CASE_SENSE;

	}
      else if( *sig_reader == '\'') 
	{

	  /********** case insensitive payload description (start with single quote) **********/

	  endof_sig = index( sig_reader + 1, '\'' );

	  /* skip escape sequences */
	  while( *(endof_sig - 1) == '\\' ) 
	    endof_sig = index( endof_sig +1, '\'');

	  if( ( endof_sig > line_buff + line_len) || ( endof_sig == (char *)NULL ) )
	    {
	      fputs("Invalid Payload Description. No closing quote.", stderr );
	      fputs( line_buff, stderr );
	      return ( 0 );
	    }

	  pattern_mode = CASE_INSENSE;

	}
      else if( *sig_reader == '<') 
	{

	  endof_sig = index( sig_reader + 1, '>' );

	  /* skip escape sequences */
	  while( *(endof_sig - 1) == '\\' ) 
	    endof_sig = index( endof_sig +1, '>');

	  if( ( endof_sig > line_buff + line_len) || ( endof_sig == (char *)NULL ) )
	    {
	      fputs("Invalid Payload Description. No closing quote.", stderr );
	      fputs( line_buff, stderr );
	      return ( 0 );
	    }

	  pattern_mode = REGEXP;

	}else{

	  /********** no payload description **********/

	  fputs("No Payload Definition", stderr );
	  fputs(line_buff, stderr );
	  return ( 0 );
	}

      pattern_len = make_pattern( pattern, LARGE_BUF_SIZE, sig_reader + 1, endof_sig - 1, pattern_mode ); /* exclude quotes */

      s_tree_index = get_s_node( protocol, src_number, dest_number );

      if( s_tree_index->c_tree == ( c_node *)NULL )
	{

	  s_tree_index->c_tree = new_c_node( s_tree_index, sig_id, pattern, pattern_len, name_buf, pattern_mode );

	  ( s_tree_index->c_tree )->next = (c_node *)NULL;

	  c_tree_index = s_tree_index->c_tree;
	  
	}
      else
	{

	  c_tree_index = s_tree_index->c_tree;

	  while( c_tree_index->next != ( c_node *)NULL )
	    c_tree_index = c_tree_index->next;

	  c_tree_index->next = new_c_node( s_tree_index, sig_id, pattern, pattern_len, name_buf, pattern_mode );

	  (c_tree_index->next)->next = ( c_node * )NULL;

	  c_tree_index = c_tree_index->next;

	}
      
      /* check minimum length */
      if( (sig_len.min == 0) || ( sig_len.min > c_tree_index->pattern_len ) )
	sig_len.min = c_tree_index->pattern_len;
	  
      if( (sig_len.max == 0) || ( sig_len.max < c_tree_index->pattern_len ) )
	sig_len.max = c_tree_index->pattern_len;

      sig_id++;
    
    }/* while */

  fclose(fp);
  

#ifdef DEBUG

  puts("===== tcp_tree =====");

  dump_s_tree( tcp_tree );

  puts("===== udp_tree =====");

  dump_s_tree( udp_tree );

  puts("===== icmp_tree =====");

  dump_s_tree( icmp_tree );

#endif

  return sig_id;

} /* init_signature */

/****************************************************************
  Function: make_pattern
  Purpose: make pattern data to match from a pattern description
  Argument: 
           u_char *pattern : pattern data to make
           char *sig_reader : reading position
	   char *endof_sig : end point of reading
	   u_char pattern_mode : CASE_INSENSE CASE_SENSE REGEXP
  Returns : 
           int length of pattern
*****************************************************************/
int make_pattern( u_char *pattern, u_int max_pattern_size, char *sig_reader, char *endof_sig, u_char pattern_mode )
{

  u_char *sig_writer;

  char letter;

  int read_length;
  
  memset( pattern, 0, max_pattern_size ); /* initialize pattern buffer */

  if( pattern_mode == REGEXP )
    {
      read_length = endof_sig - sig_reader +1;

      if( read_length > max_pattern_size )
	{
	  fputs(  "Too large pattern description", stderr );
	  final( 0 );
	}

      memcpy( pattern, sig_reader, read_length );

      return read_length;
    }

  for( sig_writer = pattern; sig_reader <= endof_sig; sig_reader++ )
    {

      letter = *sig_reader; /* get a letter */

      if ( letter == '\\' ) /* escape sequence */
	{

	  if( sig_reader == endof_sig )
	    {
	      fputs("Invalid Escape Sequence at the last of pattern description.", stderr );
	      final( 0 );
	    }
	  
	  letter = *( ++sig_reader );

	  switch( letter )
	    {
	    case 'a':
	      {
		*sig_writer = '\a';
		break;
	      }
	    case 'b':
	      {
		*sig_writer = '\b';
		break;
	      }
	    case 'f':
	      {
		*sig_writer = '\f';
		break;
	      }
	    case 'n':
	      {
		*sig_writer = '\n';
		break;
	      }
	    case 'r':
	      {
		*sig_writer = '\r';
		break;
	      }
	    case 't':
	      {
		*sig_writer = '\t';
		break;
	      }
	    case 'v':
	      {
		*sig_writer = '\v';
		break;
	      }
	    case '\\':
	      {
		*sig_writer = '\\';
		break;
	      }
	    case '\?':
	      {
		*sig_writer = '\?';
		break;
	      }
	    case '\'':
	      {
		*sig_writer = '\'';
		break;
	      }
	    case '\"':
	      {
		*sig_writer = '\"';
		break;
	      }

	      /* for sexadecimal binary expression */
	    case 'x':
	      {
		char bin_buffer[3];

		if( isxdigit( *( ++sig_reader ) ) )
		  {
		    if( isxdigit( *( sig_reader + 1 ) ) )
		      {
			sprintf( bin_buffer, "%c%c%c", 
				 *sig_reader, 
				 *(sig_reader + 1 ),
				 (char)NULL );
			sig_reader += 2;
		      }
		    else
		      {
			sprintf( bin_buffer, "%c%c", 
				 *sig_reader, 
				 (char)NULL );
			sig_reader += 1;
		      }

		    *sig_writer = 
		      (u_char)strtol( bin_buffer, (char **)NULL, 16);
		      
		    if( pattern_mode != CASE_SENSE )
		      *sig_writer = toupper( *sig_writer );
		    
		  } /* if isxdigit */
		break;
	      }
	    default:
	      {
		fputs("Invalid Escape Sequence in pattern description.", stderr );
		final( 0 );
	      }

	    } /* switch */

	  sig_writer++;
	}
      else if ( letter == '|' ) /* when get the first '|' read binary data */
	{

	  char *endof_bin = index( sig_reader + 1, '|' );

	  /* skip escape sequences */

	  while( *( endof_bin - 1) == '\\' )         /* escape sequence */
	    endof_bin = index( endof_bin + 1, '|'); /* find next | */

	  /***** closing | not found *****/
	  
	  if( ( endof_bin > endof_sig ) || ( endof_bin == (char *)NULL ) )
	    {
	      fputs("Invalid Payload Description.No closing |.", stderr );
	      final( 0 );
	    }
	      
	  sig_writer += make_bin_pattern( sig_writer, sig_reader + 1, endof_bin - 1, pattern_mode );

	  if( endof_bin == endof_sig )
	    sig_reader = endof_bin;
	  else
	    sig_reader = endof_bin + 1;
	}
      else                    /*** read text ***/
	{

	  *sig_writer = ( pattern_mode == CASE_SENSE ) ? 
	    letter : toupper( letter );
	  sig_writer++;
	  continue;
	}

    } /* for sig_reader */

  *sig_writer = (u_char)NULL;

  return sig_writer - pattern; /* legnth of pattern exclude last \0 */

} /* make_pattern */

/****************************************************************
  Function: make_bin_pattern
  Purpose: make binary pattern defined between ||.
  Argument: 
           u_char *sig_writer writing pointer
           char *bin_reader : reading position
	   char *endof_sig : end point of reading
	   u_char case_sensitive : TRUE when case sensitive
  Returns : 
           pointer to next writing position
*****************************************************************/
int make_bin_pattern( u_char *sig_writer, char *bin_reader, char *endof_bin, u_char pattern_mode )
{

  char bin_buffer[3];

  u_char *original = sig_writer;

  while( bin_reader <= endof_bin )
    {

      /* skip spaces */
      while( isspace( *bin_reader) ) 
	bin_reader++;

      if( isxdigit( *( bin_reader ) ) )
	{
	  if( isxdigit( *( bin_reader + 1 ) ) )
	    {
	      sprintf( bin_buffer, "%c%c%c", 
		       *bin_reader, 
		       *(bin_reader + 1 ),
		       (char)NULL );
	      bin_reader += 2;
	    }
	  else
	    {
	      sprintf( bin_buffer, "%c%c", 
		       *bin_reader, 
		       (char)NULL );
	      bin_reader += 1;
	    }

	  *sig_writer = 
	    (u_char)strtol( bin_buffer, (char **)NULL, 16);

	  if( pattern_mode != CASE_SENSE )
	    *sig_writer = toupper( *sig_writer );
	  
	  sig_writer++;

	}
      else
	{
	  fputs("Signature Format Error : Invalid Binary Expression. ", stderr );
	  final( 0 );
	}

    } /* while */

  return (sig_writer - original); /* length of pattern */

} /* make_bin_pattern */

/************************************************
  Function: detect_signature
  Purpose: detect signatures in payload
  Argument:
       struct ip_info *ip_packet
       u_char *capital
       s_node **s_tree_index
       c_node **c_tree_index
  Returns:
 ***********************************************/
c_node *detect_signature( struct ip_info *ip_packet, u_char *capital, s_node **s_tree_index,  c_node **c_tree_index )
{

  u_char *payload = (char *)NULL;

  while( ( *s_tree_index != (s_node *)NULL ) &&
	 ( (*s_tree_index)->dest_number <= ip_packet->dest_number ) )
    {

      /* number ULONG_MAX is used for wild card */

      while( ( *s_tree_index != (s_node *)NULL ) &&

	     ( ( (*s_tree_index)->dest_number == ULONG_MAX ) || 
	       ( (*s_tree_index)->dest_number == ip_packet->dest_number ) )   &&

	     ( ( (*s_tree_index)->src_number == ULONG_MAX ) ||
	       ( (*s_tree_index)->src_number == ip_packet->src_number )) )
	{

	  if( *c_tree_index == (c_node*)NULL )
	    *c_tree_index = (*s_tree_index)->c_tree;

	  while( *c_tree_index != (c_node *)NULL )
	    {

	      if( (*c_tree_index)->pattern_mode == REGEXP )
		{
		  if( re_search( &((*c_tree_index)->reg_pattern),
				 ip_packet->payload,
				 ip_packet->payload_len,
				 0,
				 ip_packet->payload_len,
				 0) >= 0 )
		    return ( *c_tree_index ); 

		}
	      else
		{

		  if( ((*c_tree_index)->pattern_len == 0) )
		    return ( *c_tree_index ); 

		  if( (*c_tree_index)->pattern_mode == CASE_SENSE )
		    payload = ip_packet->payload;

		  else if( (*c_tree_index)->pattern_mode == CASE_INSENSE )
		    payload = capital;

		  /* Quadratic Algorithm 
		     
		     if( qd_match( payload,
		     ip_packet->payload_len,
		     (*c_tree_index)->pattern,
		     (*c_tree_index)->pattern_len
		     ) )
		     return ( *c_tree_index ); 
		  */

		  /* Booyer-Moore Algorithm */
		  
		  if( bm_match ( payload,
				 ip_packet->payload_len,
				 (*c_tree_index)->pattern,
				 (*c_tree_index)->pattern_len,
				 (*c_tree_index)->delta1,
				 (*c_tree_index)->delta2
				 ) )
		    return ( *c_tree_index ); /* return when match */
		}

	      *c_tree_index = (*c_tree_index)->next;

	    } /* while c_tree */
	      
	  *s_tree_index = (*s_tree_index)->next;

	} /* while s_tree */
	  
      if(*s_tree_index != (s_node *)NULL)
	*s_tree_index = (*s_tree_index)->next;
    }

  return (c_node *)NULL;

} /* detect_signature */

/************************************************
  Function: dump_c_tree
  Purpose: dump all contents under the tree
  Argument:
  Returns:
 ***********************************************/
int dump_c_tree(s_node * s_tree )
{

  c_node * c_tree = ( c_node * )NULL;

  if(s_tree == (s_node *)NULL )
    {

      fprintf(stderr, "dump_c_tree: s_tree == NULL\n");

      return (-1);
    }
  
  c_tree = s_tree->c_tree;

  while( c_tree != (c_node *)NULL )
    {

      int i;
      printf("s_node[");

      if(((s_node *)c_tree->s_node)->src_number == ULONG_MAX ) 
	if(((s_node *)c_tree->s_node)->dest_number == ULONG_MAX )
	  printf("* *");
	else
	  printf("* %lu", ((s_node *)c_tree->s_node)->dest_number);
      else if (((s_node *)c_tree->s_node)->dest_number == ULONG_MAX )
	  printf("%lu *", ((s_node *)c_tree->s_node)->src_number);
      else
	  printf("%lu %lu", ((s_node *)c_tree->s_node)->src_number,
		 ((s_node *)c_tree->s_node)->dest_number );

      printf( "] c_node[%d] %s (%d) ", 
	      c_tree->sig_id,
	      c_tree->name,
	      c_tree->pattern_len
	      );

      if( c_tree->pattern_mode == CASE_SENSE )
	printf("\"");
      else
	printf("\'");

      for( i=0; i < c_tree->pattern_len; i++ )
	{
	  if ( isprint( *(c_tree->pattern + i)  ) )
	    putchar( *(c_tree->pattern + i) );
	  else
	    printf( "\\x%x", *( c_tree->pattern + i ) );
	}

      if( c_tree->pattern_mode == CASE_SENSE )
	puts("\"");
      else
	puts("\'");

      c_tree = c_tree->next;
    }

  return (0);

} /* dump_c_tree */

/************************************************
  Function: dump_s_tree
  Purpose: dump all signatures under the tree
  Argument:
  Returns:
 ***********************************************/
void dump_s_tree(s_node * s_tree )
{

  while( s_tree != (s_node *)NULL )
    {

      dump_c_tree( s_tree );

      s_tree = s_tree->next;
      
    }

} /* dump_sig_tree */

/************************************************
  Function: free_s_tree
  Purpose: free memory for each s_node
 ***********************************************/
void free_s_tree( s_node * s_tree)
{

  if( s_tree == (s_node *)NULL )
    return;

  if( s_tree->next != (s_node *)NULL )
    free_s_tree( s_tree->next );

  if( s_tree->c_tree != (c_node *)NULL )
    free_c_tree( s_tree->c_tree );

  free_check( s_tree, "free_s_tree:s_tree" );

  return;

} /*** free_s_tree ***/

/************************************************
  Function: free_c_tree
  Purpose: free memory for each c_node
 ***********************************************/
void free_c_tree( c_node * c_tree)
{

  if( c_tree->next != (c_node *)NULL )
    free_c_tree( c_tree->next );

  if( c_tree->pattern )
    free_check( c_tree->pattern, "free_c_tree:c_tree->pattern" );
  if( c_tree->name )
    free_check( c_tree->name, "free_c_tree:c_tree->name" );
  if( c_tree->delta1 )
    free_check( c_tree->delta1, "free_c_tree:c_tree->delta1" );
  if( c_tree->delta2 )
    free_check( c_tree->delta2, "free_c_tree:c_tree->delta2" );
  if( (c_tree->reg_pattern).translate )
    free_check( (c_tree->reg_pattern).translate, "free_c_tree:(c_tree->reg_pattern).translate" );
  if( (c_tree->reg_pattern).fastmap )
    free_check( (c_tree->reg_pattern).fastmap, "free_c_tree:(c_tree->reg_pattern).fastmap" );
  if( c_tree )
    free_check( c_tree, "free_c_tree:c_tree" );

  return;

} /*** free_c_tree ***/



