<?php
// Call LocaleUtilTest::main() if this source file is executed directly.
if (!defined('PHPUnit_MAIN_METHOD')) {
    define('PHPUnit_MAIN_METHOD', 'LocaleUtilTest::main');
}



require_once "testConf.php";
require_once ROOT_PATH."/lib/confs/sysConf.php";
require_once 'LocaleUtil.php';


/**
 * Test class for LocaleUtil.
 * Generated by PHPUnit on 2007-09-17 at 10:42:17.
 */
class LocaleUtilTest extends PHPUnit_Framework_TestCase {

	private $sysConf;

    /**
     * Runs the test methods of this class.
     *
     * @access public
     * @static
     */
    public static function main() {
        require_once 'PHPUnit/TextUI/TestRunner.php';

        $suite  = new PHPUnit_Framework_TestSuite('LocaleUtilTest');
        $result = PHPUnit_TextUI_TestRunner::run($suite);
    }

    /**
     * Sets up the fixture, for example, opens a network connection.
     * This method is called before a test is executed.
     *
     * @access protected
     */
    protected function setUp() {
    	$this->sysConf = new SysConf();
    }

    /**
     * Tears down the fixture, for example, closes a network connection.
     * This method is called after a test is executed.
     *
     * @access protected
     */
    protected function tearDown() {
    }

	/**
	 * Test that the sysConf is set by the LocaleUtil constructor.
	 */
	public function testSysConfSet() {
    	$localeUtil = LocaleUtil::getInstance();

		// Verify that LocaleUtil has got a sysConf instance
		$sysConf = $localeUtil->getSysConf();
		$this->assertTrue($sysConf instanceof sysConf);
	}

    public function testFormatDate() {
    	$localeUtil = LocaleUtil::getInstance();

		$this->sysConf->dateFormat = "d-m-Y";
		$localeUtil->setSysConf($this->sysConf);

    	$date = '2007-12-22';
    	$expected = '22-12-2007';
    	$this->assertEquals($expected, $localeUtil->formatDate($date));

    	// If input is invalid simply output input without attempting to convert
    	$date = 'No date';
    	$this->assertEquals($date, $localeUtil->formatDate($date));

		// Try some other formats
		$this->sysConf->dateFormat = "F jS, Y";
		$localeUtil->setSysConf($this->sysConf);

    	$date = '2007-9-18';
    	$expected = 'September 18th, 2007';
    	$this->assertEquals($expected, $localeUtil->formatDate($date));

    	// Passing a custom date format to method should ignore system configuration
    	$date = '2007-9-18';
    	$expected = '18th September, 2007';
    	$this->assertEquals($expected, $localeUtil->formatDate($date, "jS F, Y"));

    	// With no date format specified. Should use return input without changes.
		$this->sysConf->dateFormat = "";
		$localeUtil->setSysConf($this->sysConf);

    	$date = '2007-12-22';
    	$expected = $date;
    	$this->assertEquals($expected, $localeUtil->formatDate($date));
    }

    public function testFormatTime() {
    	$localeUtil = LocaleUtil::getInstance();

		$this->sysConf->timeFormat = "H:i";
		$localeUtil->setSysConf($this->sysConf);

    	$time = '13:01';
    	$expected = '13:01';
    	$this->assertEquals($expected, $localeUtil->formatTime($time));

    	// If input is invalid simply output input without attempting to convert
    	$time = '-';
    	$this->assertEquals($time, $localeUtil->formatTime($time));

    	// Passing a custom date format to method should ignore system configuration
    	$date = '18:35';
    	$expected = '06:35 PM';
    	$this->assertEquals($expected, $localeUtil->formatTime($date, "h:i A"));

    	// With no date format specified. Should use return input without changes.
		$this->sysConf->dateFormat = "";
		$localeUtil->setSysConf($this->sysConf);

    	$time = '13:01';
    	$expected = $time;
    	$this->assertEquals($time, $localeUtil->formatTime($time));
    }

    public function testFormatDateTime() {
    	$localeUtil = LocaleUtil::getInstance();

		$this->sysConf->dateFormat = "d/m/Y";
		$this->sysConf->timeFormat = "H:i";
		$localeUtil->setSysConf($this->sysConf);

    	$time = '2007-12-22 13:01';
    	$expected = '22/12/2007 13:01';
    	$this->assertEquals($expected, $localeUtil->formatDateTime($time));

    	// If input is invalid simply output input without attempting to convert
    	$time = 'a - b';
    	$this->assertEquals($time, $localeUtil->formatDateTime($time));

    	// Passing a custom date format to method should ignore system configuration
    	$date = '2007-9-18 18:35';
    	$expected = '18th September, 2007 06:35 PM';
    	$this->assertEquals($expected, $localeUtil->formatDate($date, "jS F, Y h:i A"));

    	// With no date format specified. Should use return input without changes.
		$this->sysConf->dateFormat = "";
		$localeUtil->setSysConf($this->sysConf);

    	$time = '2007 12 22 13:01';
    	$expected = $time;
    	$this->assertEquals($time, $localeUtil->formatDateTime($time));

    }

    public function testConvertToStandardTimeFormat() {
    	$localeUtil = LocaleUtil::getInstance();

		$this->sysConf->timeFormat = "H:i";
		$localeUtil->setSysConf($this->sysConf);

    	$time = '13:01';
    	$expected = '13:01';
    	$this->assertEquals($expected, $localeUtil->convertToStandardTimeFormat($time));

    	// If input is invalid return null
    	$time = '-';
    	$this->assertNull($localeUtil->convertToStandardTimeFormat($time));

    	// Passing a custom date format to method should ignore system configuration
    	$expected = '18:35';
    	$date = '06:35 PM';
    	$this->assertEquals($expected, $localeUtil->convertToStandardTimeFormat($date, "h:i A"));

    	// With no date format specified. Should return null.
		$this->sysConf->timeFormat = "";
		$localeUtil->setSysConf($this->sysConf);

    	$time = '13:01';
    	$this->assertNull($localeUtil->convertToStandardTimeFormat($time));
    }

    public function testConvertToStandardDateFormat() {
    	$localeUtil = LocaleUtil::getInstance();

		$this->sysConf->dateFormat = "d-m-Y";
		$localeUtil->setSysConf($this->sysConf);

    	$expected = '2007-12-22';
    	$date = '22-12-2007';
    	$this->assertEquals($expected, $localeUtil->convertToStandardDateFormat($date));

    	// If input is invalid return null
    	$date = 'No date';
    	$this->assertNull($localeUtil->convertToStandardDateFormat($date));

    	// Passing a custom date format to method should ignore system configuration
    	$expected = '2007-09-18';
    	$date = '09-18-2007';
    	$this->assertEquals($expected, $localeUtil->convertToStandardDateFormat($date, "m-d-Y"));

    	// With no date format specified. Should return null.
		$this->sysConf->dateFormat = "";
		$localeUtil->setSysConf($this->sysConf);

    	$date = '2007-12-22';
    	$this->assertNull($localeUtil->convertToStandardDateFormat($date));
    }

    public function testConvertToStandardDateTimeFormat() {
    	$localeUtil = LocaleUtil::getInstance();

		$this->sysConf->dateFormat = "d/m/Y";
		$this->sysConf->timeFormat = "H:i";
		$localeUtil->setSysConf($this->sysConf);

    	$expected = '2007-12-22 13:01';
    	$time = '22/12/2007 13:01';

    	$this->assertEquals($expected, $localeUtil->convertToStandardDateTimeFormat($time));

    	// If input is invalid return null
    	$time = 'a - b';
    	$this->assertNull($localeUtil->convertToStandardDateTimeFormat($time));

    	// Passing a custom date format to method should ignore system configuration
    	$expected = '2007-09-18 18:35';
    	$date= '09-18-2007 06:35 PM';
    	$this->assertEquals($expected, $localeUtil->convertToStandardDateTimeFormat($date, "m-d-Y h:i A"));

    	// With no date format specified. Should return null.
		$this->sysConf->dateFormat = "";
		$localeUtil->setSysConf($this->sysConf);

    	$time = '2007 12 22 13:01';
    	$expected = $time;
    	$this->assertNull($localeUtil->convertToStandardDateTimeFormat($time));
    }

    public function testConvertToXpDateFormat() {
    	$formatStr = 'Y-m-d';
    	$xpFormatStr = 'yyyy-MM-dd';
    	$this->assertEquals($xpFormatStr, LocaleUtil::convertToXpDateFormat($formatStr));

    	$formatStr = 'y-m-d';
    	$xpFormatStr = 'yy-MM-dd';
    	$this->assertEquals($xpFormatStr, LocaleUtil::convertToXpDateFormat($formatStr));

    	$formatStr = 'Y-n-d';
    	$xpFormatStr = 'yyyy-M-dd';
    	$this->assertEquals($xpFormatStr, LocaleUtil::convertToXpDateFormat($formatStr));

    	$formatStr = 'Y-m-j';
    	$xpFormatStr = 'yyyy-MM-d';
    	$this->assertEquals($xpFormatStr, LocaleUtil::convertToXpDateFormat($formatStr));

    	$formatStr = 'H:i:s';
    	$xpFormatStr = 'HH:mm:ss';
    	$this->assertEquals($xpFormatStr, LocaleUtil::convertToXpDateFormat($formatStr));

    	$formatStr = 'h:i:s';
    	$xpFormatStr = 'hh:mm:ss';
    	$this->assertEquals($xpFormatStr, LocaleUtil::convertToXpDateFormat($formatStr));

    	$formatStr = 'G:i:s';
    	$xpFormatStr = 'H:mm:ss';
    	$this->assertEquals($xpFormatStr, LocaleUtil::convertToXpDateFormat($formatStr));

    	$formatStr = 'g:i:s';
    	$xpFormatStr = 'h:mm:ss';
    	$this->assertEquals($xpFormatStr, LocaleUtil::convertToXpDateFormat($formatStr));

    	$formatStr = 'Y-m-d H:i';
    	$xpFormatStr = 'yyyy-MM-dd HH:mm';
    	$this->assertEquals($xpFormatStr, LocaleUtil::convertToXpDateFormat($formatStr));

    }

    public function testFormatMoney() {
        $localeUtil = LocaleUtil::getInstance();
        $this->assertEquals('1121.12', $localeUtil->formatMoney(1121.12121));
        $this->assertEquals('0.12', $localeUtil->formatMoney(0.12121));
        $this->assertEquals('1212121212.00', $localeUtil->formatMoney(1212121212));
        $this->assertEquals('0.00', $localeUtil->formatMoney(0));
        $this->assertEquals('0.00', $localeUtil->formatMoney('abc'));
        $this->assertEquals('0.00', $localeUtil->formatMoney(''));
        $this->assertEquals('0.00', $localeUtil->formatMoney(null));
    }
}

// Call LocaleUtilTest::main() if this source file is executed directly.
if (PHPUnit_MAIN_METHOD == 'LocaleUtilTest::main') {
    LocaleUtilTest::main();
}
?>
