/*
 * Copyright (c) 2013 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <util.h>
#include <utmpx.h>
#include <stdlib.h>
#include <syslog.h>
#include <sys/param.h>

#include "openvpn-plugin.h"

static void
logwtmpx_ext(line, username, host)
	char *line;
	const char *username;
	const char *host;
{
	char cmd[PATH_MAX];

	(void)snprintf(cmd, sizeof(cmd),
		       "%s/logwtmpx %s %s %s",
			BINDIR, line, username, host);
			
	system(cmd);

	return;
}

static const char *
get_env(name, envp)
	const char *name;
	const char *envp[];
{
	int i;
	int namelen;
	const char *cp;

	if (envp == NULL)
		return NULL;

	namelen = strlen(name);
	for (i = 0; envp[i] != NULL; i++) {
		if (strncmp(envp[i], name, namelen) != 0)
			continue;

		cp = envp[i] + namelen;
		if (*cp != '=')
			continue;

		return cp + 1;
	}

	return NULL;
}

OPENVPN_EXPORT openvpn_plugin_handle_t
openvpn_plugin_open_v1 (mask, argv, envp)
	unsigned int *mask;
	const char *argv[];
	const char *envp[];
{
	char *context;

	context = strdup("openvpn-acct-wtpmx"); /* Placeholder */
	
	*mask = 
	    OPENVPN_PLUGIN_MASK (OPENVPN_PLUGIN_CLIENT_DISCONNECT) |
	    OPENVPN_PLUGIN_MASK (OPENVPN_PLUGIN_LEARN_ADDRESS) |
#ifdef DEBUG
	    OPENVPN_PLUGIN_MASK (OPENVPN_PLUGIN_UP) |
	    OPENVPN_PLUGIN_MASK (OPENVPN_PLUGIN_DOWN) |
	    OPENVPN_PLUGIN_MASK (OPENVPN_PLUGIN_ROUTE_UP) |
	    OPENVPN_PLUGIN_MASK (OPENVPN_PLUGIN_IPCHANGE) |
	    OPENVPN_PLUGIN_MASK (OPENVPN_PLUGIN_TLS_VERIFY) |
	    OPENVPN_PLUGIN_MASK (OPENVPN_PLUGIN_AUTH_USER_PASS_VERIFY) |
	    OPENVPN_PLUGIN_MASK (OPENVPN_PLUGIN_CLIENT_CONNECT_V2) |
	    OPENVPN_PLUGIN_MASK (OPENVPN_PLUGIN_TLS_FINAL) |
#endif /* DEBUG */
	    0;

	return (openvpn_plugin_handle_t)context;
}

#ifdef DEBUG
static void
dump_env(type, argv, envp)
	const int type;
	const char *argv[]; 
	const char *envp[];
{
	size_t i;

	switch (type) {
	case OPENVPN_PLUGIN_UP:
		syslog(LOG_INFO, "OPENVPN_PLUGIN_UP");
		break;
	case OPENVPN_PLUGIN_DOWN:
		syslog(LOG_INFO, "OPENVPN_PLUGIN_DOWN");
		break;
	case OPENVPN_PLUGIN_ROUTE_UP:
		syslog(LOG_INFO, "OPENVPN_PLUGIN_ROUTE_UP");
		break;
	case OPENVPN_PLUGIN_IPCHANGE:
		syslog(LOG_INFO, "OPENVPN_PLUGIN_IPCHANGE");
		break;
	case OPENVPN_PLUGIN_TLS_VERIFY:
		syslog(LOG_INFO, "OPENVPN_PLUGIN_TLS_VERIFY");
		break;
	case OPENVPN_PLUGIN_AUTH_USER_PASS_VERIFY:
		syslog(LOG_INFO, "OPENVPN_PLUGIN_AUTH_USER_PASS_VERIFY");
		break;
	case OPENVPN_PLUGIN_CLIENT_CONNECT_V2:
		syslog(LOG_INFO, "OPENVPN_PLUGIN_CLIENT_CONNECT_V2");
		break;
	case OPENVPN_PLUGIN_CLIENT_DISCONNECT:
		syslog(LOG_INFO, "OPENVPN_PLUGIN_CLIENT_DISCONNECT");
		break;
	case OPENVPN_PLUGIN_LEARN_ADDRESS:
		syslog(LOG_INFO, "OPENVPN_PLUGIN_LEARN_ADDRESS");
		break;
	case OPENVPN_PLUGIN_TLS_FINAL:
		syslog(LOG_INFO, "OPENVPN_PLUGIN_TLS_FINAL");
		break;
	default:
		syslog(LOG_INFO, "OPENVPN_PLUGIN_?");
		break;
	}

	for (i = 0; argv[i] != NULL; ++i)
		syslog(LOG_INFO, "argv[%d] = \"%s\"", i, argv[i]);

	for (i = 0; envp[i] != NULL; ++i)
		syslog(LOG_INFO, "envp[%d] = \"%s\"", i, envp[i]);

	return;
}
#endif /* DEBUG */

OPENVPN_EXPORT int
openvpn_plugin_func_v1(handle, type, argv, envp)
	openvpn_plugin_handle_t handle;
	const int type;
	const char *argv[];
	const char *envp[];
{
	const char *remote_addr;
	const char *private_addr;
	const char *remote_user;
	char line[_UTX_LINESIZE];
	int login;

#ifdef DEBUG
	dump_env(type, argv, envp);
#endif /* DEBUG */

	switch(type) {
	case OPENVPN_PLUGIN_LEARN_ADDRESS: 
		login = 1;
		break;
	case OPENVPN_PLUGIN_CLIENT_DISCONNECT:
		login = 0;
		break;
	default:
		return OPENVPN_PLUGIN_FUNC_SUCCESS;
		break;
	}

	if ((remote_addr = get_env("untrusted_ip", envp)) == NULL)
		remote_addr = "(unknown)";
	if ((remote_user = get_env("username", envp)) == NULL)
		remote_user = "(unknown)";
	if ((private_addr = get_env("ifconfig_pool_remote_ip", envp)) == NULL)
		private_addr = "(unknown)";
	snprintf(line, sizeof(line), "vpn/%s", private_addr);

	if (login) {
		syslog(LOG_INFO, "openvpn login: %s on %s from %s",
		       remote_user, private_addr, remote_addr);
		if (geteuid() == 0)
			logwtmpx(line, remote_user, 
				 remote_addr, 0, LOGIN_PROCESS);
		else
			logwtmpx_ext(line, remote_user,remote_addr);
	} else {
		syslog(LOG_INFO, "openvpn logout: %s on %s from %s",
		       remote_user, private_addr, remote_addr);
		if (geteuid() == 0)
			logwtmpx(line, "", "", 0, LOGIN_PROCESS);
		else
			logwtmpx_ext(line, "", "");
	}

	return OPENVPN_PLUGIN_FUNC_SUCCESS;
}

OPENVPN_EXPORT void
openvpn_plugin_close_v1(handle)
	openvpn_plugin_handle_t handle;
{
	struct plugin_context *context;

	context = (struct plugin_context *)handle;
	free(context);

	return;
}
