/* Copyright 2003 by James M. Kretchmar
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice and this
 * permission notice appear in all copies and in supporting
 * documentation.  No representation is made about the suitability of
 * this software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdio.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <ctype.h>
#include "oak.h"

int main(int argc, char **argv)
{
  oak_matchline *m;
  FILE *logfile;
  char buff[LINE];
  int i, x, y, ret, first, prescan;
  char *host, *replacement, *logfilename, *configfile;
  struct stat statbuf;
  ino_t inode;
  off_t endptr, curptr;
  oak_matchlist *global_matchlist;
  oak_queuelist *global_queuelist;
  int global_queuelist_size, global_matchlist_size;
  int debug;
  struct sigaction sigact;
  
  oak_global_init(&g);

  configfile=NULL;
  if (argc>0) {
    argv++;
    argc--;
  }
  while (argc>0) {
    if (!strcmp(argv[0], "-c")) {
      if (argc<2) {
	fprintf(stderr, "Too few arguments to -c\n");
	oak_print_usage(1);
	exit(1);
      }
      configfile=argv[1];
      argv+=2;
      argc-=2;
    } else if (!strcmp(argv[0], "-d")) {
      oak_global_set_debug(&g);
      argv++;
      argc--;
    } else if (!strcmp(argv[0], "-v")) {
      printf("Oak version %s\n", OAK_VERSION_STRING);
      exit(0);
    } else if (!strcmp(argv[0], "-h")) {
      oak_print_usage(0);
      exit(0);
    } else {
      fprintf(stderr, "Uknown argument\n");
      oak_print_usage(1);
      exit(1);
    }
  }

  if (!configfile) {
    oak_print_usage(1);
    exit(1);
  }

  /* read in the config file */
  oak_readconfig(configfile);

  if (oak_global_is_debug(&g)) {
    debug=1;
    printf("DEBUG: starting oak\n");
  } else {
    debug=0;
  }

  /* setup the signal handler */
  if (debug) printf("DEBUG: creating signal catcher\n");
  sigact.sa_sigaction=sig_handler;
  sigemptyset(&sigact.sa_mask);
  sigact.sa_flags=SA_SIGINFO;
  sigaction(SIGTERM, &sigact, NULL);
  sigaction(SIGINT, &sigact, NULL);
  sigaction(SIGHUP, &sigact, NULL);
  sigaction(SIGCHLD, &sigact, NULL);

  /* set some useful local variables */
  global_matchlist=oak_global_get_matchlist(&g);
  global_matchlist_size=oak_list_get_size(global_matchlist);
  global_queuelist=oak_global_get_queuelist(&g);
  global_queuelist_size=oak_list_get_size(global_queuelist);
  logfilename=oak_global_get_logfile(&g);

  /* do we have any prescaning queues? */
  prescan=0;
  for (i=0; i<global_queuelist_size; i++) {
    if (oak_queue_is_prescan(oak_list_get_element(global_queuelist, i))) prescan=1;
  }

  /* open the logfile */
  if (debug) printf("DEBUG: opening log file\n");
  if ((logfile=fopen(logfilename, "r"))==NULL) {
    perror("could not open logfile");
    exit(1);
  }

  /* save the file pointer at the end of the file */
  endptr=lseek(fileno(logfile), 0, SEEK_END);
  if (debug) printf("DEBUG: ending file pointer is %i\n", (int) endptr);
  /* if we're prescanning zoom back to the beginning, otherwise stay at the end */
  if (prescan) lseek(fileno(logfile), 0, SEEK_SET);
  
  /* cache the inode */
  ret=fstat(fileno(logfile), &statbuf);
  if (ret) {
    perror("Could not stat logfile");
    exit(3);
  }
  inode=statbuf.st_ino;
  
  /* deamonize it here if not debug */
  if (!debug) {
    for(i=0; i<getdtablesize(); i++) {
      if (i!=fileno(logfile)) close(i);
    }

    i=fork();
    if (i == -1) {
      fprintf(stderr, "couldn't fork, going to die\n");
      exit(2);
    } else if (i != 0) {
      /* parent exits */
      exit(0);
    }

    setsid();
    chdir("/");
    umask(0);
  }

  /* main loop */
  if (debug) printf("DEBUG: entering main loop\n");
  if (prescan) {
    first=1;
  } else {
    if (debug) printf("DEBUG: skipping prescanning\n");
    first=0;
  }
  curptr=0;
  replacement=malloc(LINE);
  memset(replacement, 0, LINE);
  while(1) {
    sleep(1);

    /* read a line and check it against all the matchlines */
    clearerr(logfile);
    while (fgets(buff, LINE, logfile)) {
      curptr+=strlen(buff);
      host=oak_text_get_host_from_log(buff);
      if (!host) continue;
      /* if (debug) printf("DEBUG: found a new line in the infile from host %s.\n", host); */

      if (!oak_global_host_is_ok(&g, host)) {
	if (debug) printf("DEBUG: host %s is being filtered, skipping message.\n", host);
	free(host);
	continue;
      }

      for (i=0; i<global_matchlist_size; i++) {
	m=oak_list_get_element(global_matchlist, i);
	if (oak_matchline_string_matches(m, buff, replacement)) {
	  oak_queuelist *ql;
	  if (debug) printf("DEBUG: Message matches matchline, replacement is %s\n", replacement?replacement:"NULL");

	  ql=oak_matchline_get_queuelist(m);
	  x=oak_list_get_size(ql);
	  for (y=0; y<x; y++) {
	    oak_queue *q;
	    q=oak_list_get_element(ql, y);
	    if (first && !oak_queue_is_prescan(q)) continue;
	    if (!oak_queue_is_locking(q)) {
	      oak_queue_add_msg(q, replacement, host);
	    } else {
	      if (!oak_matchline_locked_for_queue(m, q)) {
		oak_queue_add_msg(q, replacement, host);
		oak_matchline_set_lock(m, q);
	      } else {
		if (debug) printf("DEBUG: discarding message due to lock\n");
	      }
	    }
	  }
	  break; /* don't process any more matchlines if found */
	}
      }

      if (first && (curptr>=endptr)) {
	if (debug) printf("DEBUG: past the ending filepointer now\n");
	first=0;
      }
      free(host);
    }

    /* check if it's time to hit any of the fire actions, OR fire anyway if the
     * global quit flag is set ... meaning it's time to send out the last SOS
     * before the ship sinks.
     */
    for (i=0; i<global_queuelist_size; i++) {
        oak_queue *q;
	q=oak_list_get_element(global_queuelist, i);
	if (oak_fire_is_time(oak_queue_get_fire(q)) || oak_global_is_quit(&g)) {
	  if (oak_queue_get_nummsgs(q) > 0) {
	    int r,s;
	    oak_list *al;
	    if (debug) printf("DEBUG: executing actions on queue %s with %i messages\n",
			      oak_queue_get_name(q),
			      oak_queue_get_nummsgs(q));
	    al=oak_queue_get_actionlist(q);
	    s=oak_list_get_size(al);
	    for (r=0; r<s; r++) {
	      oak_action_execute(oak_list_get_element(al, r), q);
	    }
	    oak_queue_delete_msgs(q);
	  } else {
	    if (debug) printf("DEBUG: skipping actions on queue %s with no messages\n", oak_queue_get_name(q));
	  }
	}
    }

    /* Exit if it's time to do so */
    if (oak_global_is_quit(&g)) exit(0);

    /* Check that the file hasn't moved.  If we get an error, wait 10
     * seconds and try again.
     */
    ret=stat(logfilename, &statbuf);
    if (ret) {
      while (1) {
	sleep(10);
	ret=stat(logfilename, &statbuf);
	if (!ret) break;  /* would like to do something more drastic? */
      }
    }
    if (inode!=statbuf.st_ino) {
      if (debug) printf("DEBUG: New log file detected, closing old log file.\n");
      fclose(logfile);
      if (debug) printf("DEBUG: Opening new log file.\n");
      if ((logfile=fopen(logfilename, "r"))==NULL) {
	/* some kind of warning, or at least loop */
	exit(1);
      }
      inode=statbuf.st_ino;
    }

  }
  
}

void oak_print_usage(int err)
{
  FILE *file;

  if (err) {
    file=stderr;
  } else {
    file=stdout;
  }
  fprintf(file, "Oak version %s\n", OAK_VERSION_STRING);
  fprintf(file, "\n");
  fprintf(file, "   oak [-v] [-h] [-d] -c <configfile>\n");
  fprintf(file, "       -v    print version number and exit\n");
  fprintf(file, "       -h    print this help message and exit\n");
  fprintf(file, "       -d    run in debugging mode\n");
  fprintf(file, "\n");
  fprintf(file, "You must specify a configuration file with -c\n");
  fprintf(file, "\n");
  fprintf(file, "Please visit http://www.ktools.org/ for information on updates\n");
  fprintf(file, "and improvements.  Send email to bug-oak@ktools.org to report\n");
  fprintf(file, "any problems.\n");
}

void sig_handler(int sig, siginfo_t *si, void *data)
{
  pid_t pid;
  int stat;

  if (sig==SIGTERM || sig==SIGHUP || sig==SIGINT) {
    oak_global_set_quit(&g);
  } else if (sig==SIGCHLD) {
    for (;;) {
      pid=waitpid(-1, &stat, WNOHANG);
      if (pid<=0) break;
    }
  }
    
}
