/*
 * ntesla - a program for designing Tesla Coils
 * 
 * Copyright (C) 1997  Steven A. Falco
 * 
 * Email contact: sfalco@worldnet.att.net
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

static char *COPYRIGHT = "Copyright (C) 1997 Steven A. Falco";

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#ifdef unix
#include <unistd.h>
#define CLEAR_SCREEN	fprintf(stderr, "\033[H\033[J");
#define O_BINARY	0
#endif

#ifdef _DOS
#include <io.h>
#include <graph.h>
#define CLEAR_SCREEN	_clearscreen(_GCLEARSCREEN);
#endif

#ifndef PI
#define PI		3.14159265358979323846
#endif
#ifndef M_SQRT2
#define M_SQRT2		1.41421356237309504880
#endif
#ifndef M_SQRT1_2
#define M_SQRT1_2	0.70710678118654752440
#endif
#ifndef W_OK
#define W_OK		02
#endif
#ifndef R_OK
#define R_OK		04
#endif

#define PARM_FILE       "ntesla.prm"
#define TMP_FILE	"ntesla.tmp"
#define VERSION         1.7
#define BS              512
#define SB		32	/* small buffer */
#define ROWS            30
#define COLS            80
#define MAXPRITURN	22
#define PERMITTIVITY	8.85e-12
#define SPACEFACTOR	0.95
#define SPOOLFACTOR	0.60
#define EXIT(r)		close(Fd_current_parms); exit(r);

/* Print macros
 *
 * P is for immediate printing, S is for store-and-print-later.
 * E is for error printing.  SHOW_ERROR will display any errors from the
 * previous entry, then clear the error.
 */
#define P fprintf(stderr,
#define S if(Row==ROWS) P "SCREEN FULL!\n"); else sprintf(Display[Row++],
#define E sprintf(Errorbuf, 
#define SHOW_ERROR if(Errorbuf[0]) { P Errorbuf); Errorbuf[0] = 0; }

/* primary_type */
#define SHUNTED		1
#define NONSHUNTED	0

/* primary_gap */
#define ROTARY		1
#define FIXED		0

/* cap_type */
#define FLAT		1
#define ROLLED		0

/* units */
#define INCH		1
#define METRIC		0

/* conversions */
typedef struct {
	double factor;
	char *unit;
} CVT;

/* Stored form for wire */
typedef struct {
	double thick;	/* Thickness in inches */
	int    gauge;	/* B&S Gauge */
} WIRE;

/* Capacitance structure */
typedef struct {
	char	*dielectric_name;
	double	dielectric_constant;
	double	breakdown;
} CAP_TABLE;

#define TOINCH		0	/* double */
#define TOFEET		1	/* double */
#define TOCM		2	/* double */
#define TOMM		3	/* double */
#define TOMETER		4	/* double */
#define TOBS		5	/* int: B&S gauge */
#define TOWMM		6	/* double: Wire mm */

#ifdef GLOBALS
EXTERN CVT Cvt[] = {
	{  1.00,	"in"     },	/* 0 */
	{  1.0 / 12.0,  "feet"   },	/* 1 */
	{  2.54,	"cm"     },	/* 2 */
	{ 25.40,	"mm"     },	/* 3 */
	{  0.0254,	"meters" },	/* 4 */
	{  0.00,	"B&S"    },	/* 5 -special */
	{ 25.40,	"mm"     }	/* 6 -special */
};

/* breakdowns derated to 1/3 for pulse service */
CAP_TABLE Cap_table[] = {
	{ "Polyethylene",	2.26,	 1200.0 / 3.0 },
	{ "Polypropylene",	2.25,	  650.0 / 3.0 },
	{ "Glass",		8.41,	  410.0 / 3.0 },
	{ "Plexiglass",		3.45,	  990.0 / 3.0 }
};
#else
EXTERN CVT Cvt[];
EXTERN CAP_TABLE Cap_table[];
#endif

/* Tesla coil parameters */
typedef struct {
	double version;		/* version of saved parameters */
	int    units;		/* inch=1 or metric=0 */
	char   printer[BS];	/* printer control string */
	char   separate[BS];	/* for separating printed output */
	double primary_volts;	/* primary voltage (V) */
	double primary_amps;	/* primary current (A) */
	double primary_va;	/* primary va (VA) */
	int    primary_type;	/* primary shunted=1 (neon) or not=0 (pig) */
	int    primary_gap;	/* rotary=1 or static=0 */
	double primary_cap;	/* primary capacitance (uF) */
	int    cap_type;	/* flat=1 or rolled=0 */
	int    cap_dielectric;	/* LDPE=1 */
	double cap_length;	/* length (in) */
	double cap_width;	/* width (in) */
	double cap_gap;		/* interplate spacing (in) */
	double cap_diameter;	/* rolled cap only (in) */
	int    cap_plates;	/* number of plates */
	double primary_thick;	/* primary tubing thickness (in) */
	double primary_spacing;	/* primary tubing spacing (in) */
	double primary_inner;	/* primary inner diameter (in) */
	int    primary_turns;	/* primary turns (unitless) */
	WIRE   secondary_wire;	/* secondary wire */
	double secondary_diam;	/* secondary diameter (in) */
	double secondary_leng;	/* secondary length (in) */
	double secondary_lfreq;	/* secondary loaded frequency (kHz) */
	double terminal_minor;	/* top terminal minor diameter (in) */
	double terminal_major;	/* top terminal major diameter (in) */
	double terminal_cap;	/* top terminal capacitance (pF) */
	double misc_amp;	/* misc menu amps */
	double misc_volt;	/* misc menu volts */
	double misc_res;	/* misc menu ohms */
	double misc_freq;	/* misc menu frequency */
	double misc_cap;	/* misc menu capacitance */
	double misc_ind;	/* misc menu inductance */
	double xf_freq;		/* line frequency */
	double spool_wd;	/* wire diameter on spool */
	double spool_cd;	/* core diameter on spool */
	double spool_len;	/* length of spool */
	WIRE   spool_wire;	/* wire on spool */
} TCPARMS;

/* Interpolator structure */
typedef struct {
	double x_axis;
	double y_axis;
} INTERPOLATE, *INTERPOLATE_P;

/* Globals */
EXTERN char Display[ROWS][COLS];
EXTERN char Errorbuf[COLS];
EXTERN int Row, Col;
EXTERN TCPARMS Tcparms;
EXTERN int Fd_current_parms;
EXTERN char *Filename;
EXTERN int Dialectric_count;

#if unix
EXTERN char Tmpname[BS];
#endif

/* some global buffers for unit conversions */
EXTERN char Ub0[SB];
EXTERN char Ub1[SB];
EXTERN char Ub2[SB];
EXTERN char Ub3[SB];
EXTERN char Ub4[SB];
EXTERN char Ub5[SB];

/* function prototypes */

/* main.c */
extern void main_menu( void );
extern void set_environment( void );
extern void save_parms( void );
extern void print_buf( void );
extern void general_instructions( void );
extern void intro( void );
extern void init( void );
extern char *UnitOut( void *, int , char *);
extern void UnitIn( double, void *, int );
extern void main_help( void );
extern void env_help( void );
extern void help_instructions( void );
extern char *show_sep( void );
extern char *get_sep( char *, int );
extern void dump_all( void );

/* interp.c */
extern double binSearch( int , INTERPOLATE_P , double );
extern double findSolFF( double );
extern double findMedhurstF( double );
extern double findQFactor( double );
extern void restrict( void );

/* cap.c */
extern void capgen( int );
extern void capacitor( void );
extern double calc_cap( void );
extern double calc_voltage( void );
extern double toroid_cap( void );
extern void cap_help( void );
extern void toroidgen( int );
extern void toroid( void );
extern void toroid_help( void );

/* trans.c */
extern void transgen( int );
extern void transformer( void );
extern void trans_help( void );

/* misc.c */
extern void misc( void );
extern void misc_help( void );
extern void findC( void );
extern void findL( void );
extern void findF( void );
extern void findA( void );
extern void findV( void );
extern void findR( void );

/* spiral.c */
extern double spiralInd( double , double , double );
extern void prigen( int );
extern void primary( void );
extern void pri_help( void );

/* solonoid.c */
extern double selfCapMedhurst( double , double );
extern double res_freq( double , double );
extern void solonoidParms( double, double, double );
extern void secgen( int );
extern void secondary( void );
extern void sec_help( void );

/* wiretab.c */
extern double gauge_to_inch( int );
extern int inch_to_gauge( double );
extern double inch_to_ohm( double );
