/*
 * dcc.c: Things dealing client to client connections. 
 *
 * Written By Troy Rollo <troy@cbme.unsw.oz.au> 
 *
 * Copyright (c) 1991, 1992 Troy Rollo.
 * Copyright (c) 1992-2000 Matthew R. Green.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "irc.h"
IRCII_RCSID("@(#)$Id: dcc.c,v 1.116 2000/05/11 02:56:21 mrg Exp $");

#ifdef ESIX
# include <lan/net_types.h>
#endif /* ESIX */

#if defined(ISC30) && defined(_POSIX_SOURCE)
# undef _POSIX_SOURCE
#include <sys/stat.h>
# define _POSIX_SOURCE
#else
# include <sys/stat.h>
#endif /* ICS30 || _POSIX_SOURCE */

#ifdef HAVE_WRITEV
#include <sys/uio.h>
#endif

#ifdef HAVE_DIRENT_H
# include <dirent.h>
#endif

#include "talkd.h"
#include "server.h"
#include "ircaux.h"
#include "whois.h"
#include "lastlog.h"
#include "ctcp.h"
#include "dcc.h"
#include "hook.h"
#include "vars.h"
#include "window.h"
#include "output.h"
#include "newio.h"
#include "irccrypt.h"
#include "screen.h"

/* ninja specific includes */
#include "dma.h"
#include "ircterm.h"
#include "hosts.h"
#include "tabkey.h"
#include "ndcc.h"
#include "ninja.h"
#include "friends.h"
#include "parse.h"

static	void	dcc_chat _((u_char *));
static	void	dcc_chat_rename _((u_char *));
/* ninja has a better dcc_filesend
 * 
static	void	dcc_filesend _((u_char *));
 */
static	int	dcc_filesend2 _((u_char *, u_char *));

static	void	dcc_getfile _((u_char *));
/* ninja also has some nice stuff */
static	void	dcc_getbynumber _((int));
static	void	dcc_getbyptr _((DCC_list *));
static	void	dcc_getall _((u_char *));

static	void	dcc_close _((u_char *));
/* more nice ninja stuff */
static	void	dcc_closebynumber _((int));
static	void	dcc_closebyptr _((DCC_list *));
static	void	dcc_closeall _((void));

static	void	dcc_talk _((u_char *));
static	void	dcc_tmsg _((u_char *));
static	void	dcc_rename _((u_char *));
static	void	dcc_summon _((u_char *));
static	void	dcc_send_raw _((u_char *));
static	void	process_incoming_chat _((DCC_list *));
static	void	process_outgoing_file _((DCC_list *));
static	void	process_incoming_file _((DCC_list *));
static	void	process_incoming_talk _((DCC_list *));
static	void	process_incoming_raw _((DCC_list *));
static	void	process_incoming_listen _((DCC_list *));

/* ninja specific functions */
static	void	dcc_getfile_resume _((u_char *));
static	void	dcc_getfile_resume_demanded _((u_char *, u_char *, u_char *, u_char *));
static	void	dcc_getfile_resume_start _((u_char *, u_char *, u_char *, u_char *));

#ifndef O_BINARY
#define O_BINARY 0
#endif /* O_BINARY */

struct
{
	u_char	*name;	/* *MUST* be in ALL CAPITALS */
	int	uniq; /* minimum length to be a unique command */
	void	(*function) _((u_char *));
}	dcc_commands[] =
{
	{ UP("CHAT"),	2, dcc_chat },
	{ UP("LIST"),	1, dcc_list },
	{ UP("SEND"),	2, dcc_sendfiles },
	{ UP("GET"),	1, dcc_getfile },
	{ UP("CLOSE"),	2, dcc_close },
	{ UP("TALK"),	2, dcc_talk },
	{ UP("TMSG"),	2, dcc_tmsg },
	{ UP("RENAME"),	3, dcc_rename },
	{ UP("SUMMON"),	2, dcc_summon },
	{ UP("RAW"),	2, dcc_send_raw },
   	{ UP("RESUME"),	3, dcc_getfile_resume },
	{ NULL,		0, (void (*) _((u_char *))) NULL }
};

	u_char	*dcc_types[] =
{
	UP("<null>"),
	UP("CHAT"),
	UP("SEND"),
	UP("GET"),
	UP("TALK"),
	UP("SUMMON"),
	UP("RAW_LISTEN"),
	UP("RAW"),
	NULL
};

/* this is such a fucking kludge */

struct	deadlist
{
	DCC_list *it;
	struct deadlist *next;
}	*deadlist = NULL;

extern	int	in_ctcp_flag;
extern	u_char	FAR MyHostName[];
extern	struct	in_addr	MyHostAddr;
extern int dgets_errno;
extern	FILE	*awayfile;
/*
 * eww global?  why?
static	off_t	filesize = 0;
 *
 * ninja uses these for global counting of get/send
 */
int	dcc_send_count = 0;
int	dcc_get_count = 0;

DCC_list	*ClientList = NULL;

static	void	add_to_dcc_buffer _((DCC_list *, u_char *));
static	void	dcc_really_erase _((void));
static	void	dcc_add_deadclient _((DCC_list *));
static	int	dcc_open _((DCC_list *));
/*
static	u_char	*dcc_time _((time_t));
 */

/*
 * dcc_searchlist searches through the dcc_list and finds the client
 * with the the flag described in type set.
 */
DCC_list *
dcc_searchlist(name, user, type, flag, othername, filesize)
	u_char	*name,
		*user;
	int	type,
		flag;
	u_char	*othername;
	off_t	filesize;
{
   DCC_list **Client, *NewClient;
   
   for (Client = (&ClientList); *Client ; Client = (&(**Client).next))
     {
	if ((((**Client).flags&DCC_TYPES) != type))
	  continue;
	if (my_stricmp(user, (**Client).user) != 0)
	  continue;
	if ((!name || (my_stricmp(name, (**Client).description) == 0)) ||
	     (othername && (**Client).othername
	      && (my_stricmp(othername, (**Client).othername)) == 0))
	  return *Client;
     }
   if (!flag)
     return NULL;

   *Client = NewClient = (DCC_list *) dma_Malloc(sizeof(DCC_list));
   NewClient->flags = type;
   NewClient->read = NewClient->write = NewClient->file = -1;
   NewClient->filesize = filesize;
   /*
    * setting stuff to 0 doesn't make much difference because
    * dma_Malloc() already does memst(x, 0, len) on malloc'd memory
    *
   NewClient->next = (DCC_list *) 0;
   NewClient->user = NewClient->description = NewClient->othername = NULL;
   NewClient->bytes_read = NewClient->bytes_sent = 0L;
   NewClient->starttime = 0;
   NewClient->buffer = 0;
   NewClient->locport = 0;
   NewClient->resume_offset = 0;
   NewClient->minimum_speed = 0.0;
    */
   dma_strcpy(&NewClient->description, name);
   dma_strcpy(&NewClient->user, user);
   dma_strcpy(&NewClient->othername, othername);
   time(&NewClient->lasttime);
   return NewClient;
}

static	void
dcc_add_deadclient(client)
	DCC_list *client;
{
   struct deadlist *new;
   
   new = (struct deadlist *) dma_malloc(sizeof(struct deadlist));
   new->next = deadlist;
   new->it = client;
   deadlist = new;
}

/*
 * dcc_erase searches for the given entry in the dcc_list and
 * removes it
 */
void
dcc_erase(Element)
	DCC_list	*Element;
{
   DCC_list	**Client;
   
   for (Client = &ClientList; *Client; Client = &(**Client).next)
     if (*Client == Element)
       {
	  *Client = Element->next;
	  new_close(Element->write);
	  new_close(Element->read);
	  new_close(Element->file);
	  dma_Free(&Element->description);
	  dma_Free(&Element->user);
	  dma_Free(&Element->othername);
	  dma_Free(&Element->buffer);
	  dma_Free(&Element);
	  return;
       }
}

static	void
dcc_really_erase()
{
   struct deadlist *dies;
   
   while ((dies = deadlist) != NULL)
     {
	deadlist = deadlist->next;
	dcc_erase(dies->it);
	dma_Free(&dies);
     }
}

/*
 * Set the descriptor set to show all fds in Client connections to
 * be checked for data.
 */
void
set_dcc_bits(rd, wd)
	fd_set	*rd, *wd;
{
   DCC_list	*Client;
   
   for (Client = ClientList; Client != NULL; Client = Client->next)
     {
#ifdef DCC_CNCT_PEND
	if (Client->write != -1 && (Client->flags & DCC_CNCT_PEND))
	  FD_SET(Client->write, wd);
#endif /* DCC_CNCT_PEND */
	if (Client->read != -1)
	  FD_SET(Client->read, rd);
     }
}

/*
 * Check all DCCs for data, and if they have any, perform whatever
 * actions are required.
 */
void
dcc_check(rd, wd)
	fd_set	*rd,
		*wd;
{
   DCC_list	**Client, *tmp;
   struct	timeval	time_out;
   int	previous_server;
   int	lastlog_level;
   
   previous_server = from_server;
   from_server = (-1);
   time_out.tv_sec = time_out.tv_usec = 0;
   lastlog_level = set_lastlog_msg_level(LOG_DCC);
   for (Client = (&ClientList); *Client != NULL && !break_io_processing;)
     {
#ifdef NON_BLOCKING_CONNECTS
	/*
	 * run all connect-pending sockets.. suggested by deraadt@theos.com
	 */
	if ((*Client)->flags & DCC_CNCT_PEND)
	  {
	     struct sockaddr_in	remaddr;
	     int	rl = sizeof(remaddr);
	     
	     if (getpeername((*Client)->read, (struct sockaddr *)&remaddr, &rl) != -1)
	       {
		  if ((*Client)->flags & DCC_OFFER)
		    {
		       (*Client)->flags &= ~DCC_OFFER;
		       save_message_from();
		       message_from((*Client)->user, LOG_DCC);
		       dcc_get_count++;
		       if (((*Client)->flags & DCC_TYPES) != DCC_RAW)
			 put_info("DCC: %s connection with %s[%s:%d] established. (1)",
				  dcc_types[(*Client)->flags&DCC_TYPES], (*Client)->user,
				  ninja_host(remaddr.sin_addr), ntohs(remaddr.sin_port));
		       /* if its a chat, add a tab key */
		       if (((*Client)->flags & DCC_TYPES) == DCC_CHAT)
			 add_tab_key(0, "msg =", (*Client)->user);
		       else if (((*Client)->flags & DCC_TYPES) == DCC_FILEREAD)
			 {
			    /* not sure, but maybe we should O_EXCL? */
			    (*Client)->file = open((*Client)->description, O_WRONLY | O_TRUNC | O_CREAT, 0600);
			    if ((*Client)->file == -1)
			      {
				 tmp = *Client;
				 Client = (&(**Client).next);
				 dcc_erase(tmp);
				 continue;
			      }
			 }
		       restore_message_from();
		    }
		  (*Client)->starttime = time(NULL);
		  (*Client)->flags &= ~DCC_CNCT_PEND;
		  set_blocking((*Client)->read);
		  if ((*Client)->read != (*Client)->write)
		    set_blocking((*Client)->write);
	       } /* else we're not connected yet */
	  }
#endif /* NON_BLOCKING_CONNECTS */
	if ((*Client)->read != -1 && FD_ISSET((*Client)->read, rd))
	  {
	     switch((*Client)->flags & DCC_TYPES)
	       {
		case DCC_CHAT:
		  process_incoming_chat(*Client);
		  break;
		case DCC_RAW_LISTEN:
		  process_incoming_listen(*Client);
		  break;
		case DCC_RAW:
		  process_incoming_raw(*Client);
		  break;
		case DCC_FILEOFFER:
		  process_outgoing_file(*Client);
		  break;
		case DCC_FILEREAD:
		  process_incoming_file(*Client);
		  break;
		case DCC_TALK:
		  process_incoming_talk(*Client);
		  break;
	       }
	  }
	if ((*Client)->flags & DCC_DELETE)
	  {
	     dcc_add_deadclient(*Client);
	     Client = (&(**Client).next);
	  }
	else
	  Client = (&(**Client).next);
     }
   (void) set_lastlog_msg_level(lastlog_level);
   dcc_really_erase();
   from_server = previous_server;
}

/*
 * Process a DCC command from the user.
 */
void
process_dcc(args)
	u_char	*args;
{
   u_char	*command;
   int	i;
   size_t	len;

   if (!(command = next_arg(args, &args)))
     return;
   len = my_strlen(command);
   upper(command);
   for (i = 0; dcc_commands[i].name != NULL; i++)
     {
	if (!my_strncmp(dcc_commands[i].name, command, len))
	  {
	     if (len < dcc_commands[i].uniq)
	       {
		  put_error("DCC: command not unique: %s", command );
		  return;
	       }
	     save_message_from();
	     message_from((u_char *) 0, LOG_DCC);
	     dcc_commands[i].function(args);
	     restore_message_from();
	     return;
	  }
     }
   put_error("DCC: Unknown command: %s", command);
}

static	int
dcc_open(Client)
	DCC_list	*Client;
{
   u_char    *user, *Type;
   struct	sockaddr_in	localaddr;
   struct	in_addr 	myip;
   int	sla;
   int	old_server;
#ifndef NON_BLOCKING_CONNECTS
   struct	sockaddr_in	remaddr;
   int	rl = sizeof(remaddr);
#endif /* NON_BLOCKING_CONNECTS */

   user = Client->user;
   old_server = from_server;
   if (-1 == from_server)
     from_server = get_window_server(0);
   myip.s_addr = server_list[from_server].local_addr.s_addr;
   if (myip.s_addr == 0 || myip.s_addr == htonl(0x7f000001))
     myip.s_addr = MyHostAddr.s_addr;
   Type = dcc_types[Client->flags & DCC_TYPES];
   if (Client->flags & DCC_OFFER)
     {
#ifdef DCC_CNCT_PEND
	Client->flags |= DCC_CNCT_PEND;
#endif /* DCC_CNCT_PEND */
	if ((Client->write = connect_by_number(Client->remport,
					       UP(inet_ntoa(Client->remote)), 1)) < 0)
	  {
	     save_message_from();
	     message_from(user, LOG_DCC);
	     if ((Client->write) != -5)
	       put_error("DCC: Unable to create connection: %s", errno ? strerror(errno) : "Unknown Host");
	     restore_message_from();
	     dcc_erase(Client);
	     from_server = old_server;
	     return 0;
	  }
	Client->read = Client->write;
	/* if (Client->resume_offset == 0) */
	  Client->bytes_read = Client->bytes_sent = 0L;
	Client->flags |= DCC_ACTIVE;
#ifndef NON_BLOCKING_CONNECTS
	Client->flags &= ~DCC_OFFER;
	Client->starttime = time(NULL);
	if (getpeername(Client->read, (struct sockaddr *)&remaddr, &rl) == -1)
	  {
	     save_message_from();
	     message_from(user, LOG_DCC);
	     put_error("DCC: getpeername failed: %s", strerror(errno));
	     restore_message_from();
	     dcc_erase(Client);
	     from_server = old_server;
	     return 0;
	  }
	if ((Client->flags & DCC_TYPES) != DCC_RAW)
	  {
	     if (my_strcmp(Type, "GET") == 0)
	       {
		  if ((Client->file = open(Client->description, O_WRONLY | O_TRUNC | O_CREAT, 0600)) == -1)
		    {
		       save_message_from();
		       message_from(user, LOG_DCC);
		       put_error("DCC: Unable to open %s: %s", Client->description, strerror(errno));
		       restore_message_from();
		       dcc_erase(Client);
		       from_server = old_server;
		       return 0;
		    }
		  /*
		   * this is already done when we connect..
		   * 
		   dcc_get_count++;
		   */
	       }
	     /*
	      * this is already done too?
	      * 
	     if (my_strcmp(Type, "CHAT") == 0)
	       add_tab_key(0, "msg =", user);
	      */
	     save_message_from();
	     message_from(user, LOG_DCC);
	     put_info("DCC: %s connection with %s[%s:%d] established. (2)",
		      Type, user, ninja_host(remaddr.sin_addr),
		      ntohs(remaddr.sin_port));
	     restore_message_from();
	  }
#endif /* NON_BLOCKING_CONNECTS */
	from_server = old_server;
	return 1;
     }
   else
     {
#ifdef DCC_CNCT_PEND
	Client->flags |= DCC_WAIT|DCC_CNCT_PEND;
#else
	Client->flags |= DCC_WAIT;
#endif /* DCC_CNCT_PEND */
	if ((Client->read = connect_by_number(0, empty_string, 1)) < 0)
	  {
	     save_message_from();
	     message_from(user, LOG_DCC);
	     if (Client->read != -5)
	       put_error("DCC: Unable to create connection: %s",
			 errno ? strerror(errno) : "Unknown Host");
	     restore_message_from();
	     dcc_erase(Client);
	     from_server = old_server;
	     return 0;
	  }
	sla = sizeof(struct sockaddr_in);
	getsockname(Client->read, (struct sockaddr *) &localaddr, &sla);
	if (Client->flags & DCC_TWOCLIENTS)
	  {
	     /* patch to NOT send pathname across */
	     u_char	*nopath;
	     
	     if ((Client->flags & DCC_FILEOFFER) &&
		 (nopath = my_rindex(Client->description, '/')))
	       nopath++;
	     else
	       nopath = Client->description;

	     /*
	      * XXX
	      * should make the case below for the filesize into
	      * generic off_t2str() function, or something.  this
	      * cast is merely a STOP-GAP measure.
	      */
	     if (Client->filesize)
	       send_ctcp(ctcp_type[in_ctcp_flag], user, UP("DCC"),
			 "%s %s %lu %u %ld", Type, nopath,
			 (u_long) ntohl(myip.s_addr),
			 (unsigned)ntohs(localaddr.sin_port),
			 (long)Client->filesize);
	     else
	       send_ctcp(ctcp_type[in_ctcp_flag], user, UP("DCC"),
			 "%s %s %lu %u", Type, nopath,
			 (u_long) ntohl(myip.s_addr),
			 (unsigned) ntohs(localaddr.sin_port));
	     /* hmmm.. lets not print this.. for file offers.. */
	     if (!(Client->flags & DCC_FILEOFFER))
	       {
		  save_message_from();
		  message_from(user, LOG_DCC);
		  put_info("DCC: Sent %s [%s:%u] request to %s",
			   Type,
			   ninja_host(myip),
			   (unsigned) ntohs(localaddr.sin_port),
			   user);
		  restore_message_from();
	       }
	  }
	/*
	 * Is this where dcc times are fucked up??  - phone
	 * Yes, it was..  and they are all hunky dory now..
	 */
	Client->starttime = 0;
	from_server = old_server;
	return 2;
     }
}

static void
dcc_chat(args)
	u_char	*args;
{
   u_char	*user;
   DCC_list	*Client;
   
   if ((user = next_arg(args, &args)) == NULL)
     {
	usage("dcc chat", "<nickname>");
	return;
     }
   Client = dcc_searchlist(UP("chat"), user, DCC_CHAT, 1, (u_char *) 0, 0);
   if ((Client->flags&DCC_ACTIVE) || (Client->flags&DCC_WAIT))
     {
	put_error("DCC: A previous CHAT to %s exists.", user);
	return;
     }
   Client->flags |= DCC_TWOCLIENTS;
   dcc_open(Client);
}

u_char	*
dcc_raw_listen(iport)
	u_int	iport;
{
   DCC_list	*Client;
   u_char	PortName[10];
   struct	sockaddr_in locaddr;
   u_char	*RetName = NULL;
   int	size;
   int	lastlog_level;
   u_short	port = (u_short) iport;

   lastlog_level = set_lastlog_msg_level(LOG_DCC);
   if (port && port < 1025)
     {
	put_error("DCC: Cannot bind to a privileged port");
	(void) set_lastlog_msg_level(lastlog_level);
	return NULL;
     }
   sprintf(CP(PortName), "%d", port);
   Client = dcc_searchlist(UP("raw_listen"), PortName, DCC_RAW_LISTEN, 1, (u_char *) 0, 0);
   if (Client->flags & DCC_ACTIVE)
     {
	put_error("DCC: A previous RAW_LISTEN on %s exists.", PortName);
	(void) set_lastlog_msg_level(lastlog_level);
	return RetName;
     }
   bzero((char *) &locaddr, sizeof(locaddr));
   locaddr.sin_family = AF_INET;
   locaddr.sin_addr.s_addr = htonl(INADDR_ANY);
   locaddr.sin_port = htons(port);
   if (0 > (Client->read = getTCPSock()))
     {
	dcc_erase(Client);
	put_error("DCC: socket() failed: %s", strerror(errno));
	(void) set_lastlog_msg_level(lastlog_level);
	return RetName;
     }
   set_socket_options(Client->read);
   if (bind(Client->read, (struct sockaddr *) &locaddr, sizeof(locaddr)) == -1)
     {
	dcc_erase(Client);
	put_error("DCC: Could not bind port: %s", strerror(errno));
	(void) set_lastlog_msg_level(lastlog_level);
	return RetName;
     }
   listen(Client->read, 4);
   size = sizeof(locaddr);
   Client->starttime = time((time_t *) 0);
   getsockname(Client->read, (struct sockaddr *) &locaddr, &size);
   Client->write = ntohs(locaddr.sin_port);
   Client->flags |= DCC_ACTIVE;
   sprintf(CP(PortName), "%d", Client->write);
   dma_strcpy(&Client->user, PortName);
   dma_strcpy(&RetName, PortName);
   (void) set_lastlog_msg_level(lastlog_level);
   return RetName;
}

u_char	*
dcc_raw_connect(host, iport)
	u_char	*host;
	u_int	iport;
{
   DCC_list	*Client;
   u_char	PortName[10];
   struct	in_addr	address;
   u_char	*RetName = (u_char *) 0;
   int	lastlog_level;
   u_short	port = (u_short)iport;
   
   lastlog_level = set_lastlog_msg_level(LOG_DCC);
   if (!resolve_host(&address, host))
     {
	put_error("DCC: Unknown host: %s", host);
	(void) set_lastlog_msg_level(lastlog_level);
	return RetName;
     }
   sprintf(CP(PortName), "%d", port); /* safe */
   Client = dcc_searchlist(host, PortName, DCC_RAW, 1, (u_char *) 0, 0);
   if (Client->flags & DCC_ACTIVE)
     {
	put_error("DCC: A previous RAW to %s on %s exists", host, PortName);
	(void) set_lastlog_msg_level(lastlog_level);
	return RetName;
     }
   Client->remport = port;
   bcopy((char *) &address, (char *) &Client->remote, sizeof(address));
   Client->flags = DCC_OFFER | DCC_RAW;
   if (!dcc_open(Client))
     return RetName;
   sprintf(CP(PortName), "%d", Client->read);
   dma_strcpy(&Client->user, PortName);
   if (do_hook(DCC_RAW_LIST, "%s %s E %d", PortName, host, port))
     put_info("DCC: RAW connection to %s on %s via %d established.",
	      host, PortName, port);
   dma_strcpy(&RetName, PortName);
   (void) set_lastlog_msg_level(lastlog_level);
   return RetName;
}

char    *talk_errors[] =
{
   "<No Error>",
     "User not logged in",
     "Connection failed",
     "Remote host does not recognize us",
     "Your party is refusing messages",
     "Unknown request",
     "Unknown protocol version",
     "Unable to decipher your address",
     "Unable to decipher return address" /* How the hell does it get
					  back then? */
};

static	void
dcc_talk(args)
	u_char	*args;
{
   u_char	*user;
   u_char	*host;
   int	status;
   
   DCC_list	*Client;

#ifdef DAEMON_UID
   if (getuid() == DAEMON_UID)
     {
	put_error("DCC: You are not permitted to use TALK");
	return;
     }
#endif /* DAEMON_UID */
   if ((user = next_arg(args, &args)) == NULL)
     {
	/* put_info("You must supply a user[@host] for DCC TALK");
	 */
	usage("dcc talk", "<user>[@<host>]");
	return;
     }
   if ((host = my_index(user, '@')) != NULL)
     *(host++) = '\0';
   else
     host = MyHostName;
   Client = dcc_searchlist(host, user, DCC_TALK, 1, (u_char *) 0, 0);
   if (Client->flags & DCC_ACTIVE || Client->flags & DCC_WAIT)
     {
	put_error("DCC: A previous TALK to %s@%s:%d exists.", user, host, 
		  ntohs(Client->remport));
	return;
     }
   if (host != MyHostName)
     {
	if (!resolve_host(&(Client->remote), host))
	  {
	     put_error("DCC: Unable to resolve %s", host);
	     dcc_erase(Client);
	     return;
	  }
     }
   else
     bcopy((char *) &MyHostAddr, (char *) &(Client->remote),
	   sizeof(struct in_addr));
   if ((Client->file = connect_by_number(-1, empty_string, 0)) < 0)
     {
	if (Client->file != -5)
	  put_error("DCC: Unable to create TALK connection: %s",
		    errno ? strerror(errno) : "Unknown Host");
	dcc_erase(Client);
	return;
     }
   put_info("DCC: Checking for invitation on caller's machine");
   if (!(status = send_talk_control(Client, DCC_TALK_CHECK)))
     {
	dcc_erase(Client);
	put_info("DCC: TALK connection timed out.");
	return;
     }
   if (--status || (Client->read = connect_by_number(Client->remport,
						     UP(inet_ntoa(Client->remote)), 0)) < 0)
     {
	put_info("DCC: Inviting %s@%s for a TALK.", Client->user, Client->description);
	if ((Client->read = connect_by_number(0, empty_string, 0)) == -1 ||
	    !send_talk_control(Client, DCC_TALK_INVITE) ||
	    !(status=send_talk_control(Client, DCC_TALK_ANNOUNCE)))
	  {
	     send_talk_control(Client, DCC_TALK_DELETE_LOCAL);
	     dcc_erase(Client);
	     return;
	  }
	if (--status)
	  {
	     put_error("DCC: TALK: %s", talk_errors[status]);
	     dcc_erase(Client);
	     return;
	  }
	put_info("DCC: Waiting for your party to respond...");
	Client->flags |= DCC_WAIT;
     }
   else
     {
	put_info("DCC: TALK connection to %s@%s:%d established.", Client->user, Client->description,
		 ntohs(Client->remport));
	time(&(Client->starttime));
	Client->write = Client->read;
	send(Client->write,  "\008\025\027", 3, 0);
	recv(Client->read, CP(Client->talkchars), 3, 0);
	Client->bytes_read = Client->bytes_sent = 3;
	Client->flags |= DCC_ACTIVE;
     }
}

static	void
dcc_summon(args)
	u_char	*args;
{
   u_char	*user;
   u_char	*host;
   DCC_list *Client;
   
   if (0 == (user = next_arg(args, &args)))
     {
	/* put_info("You must supply a user[@host] for DCC SUMMON");
	 */
	usage("dcc summon", "<user>[@<host>]");
	return;
     }
   if (0 != (host = my_index(user, '@')))
     *host++ = '\0';
   else
     host = MyHostName;
   
   Client = dcc_searchlist(host, user, DCC_SUMMON, 1, (u_char *) 0, 0);
   
   if (host != MyHostName)
     {
	if (!resolve_host(&(Client->remote), host))
	  {
	     put_error("DCC: Unable to resolve %s", host);
	     dcc_erase(Client);
	     return;
	  }
     }
   else
     bcopy((char *) &MyHostAddr, (char *) &(Client->remote),
	   sizeof(struct in_addr));
   if ((Client->file = connect_by_number(-1, empty_string, 0)) < 0)
     {
	if (Client->file != -5)
	  put_error("DCC: Unable to create SUMMON connection: %s",
		    errno ? strerror(errno) : "Unknown Host");
	return;
     }
   if (0 == send_talk_control(Client, DCC_TALK_SUMMON))
     put_error("DCC: SUMMON: connection timed out");
   send_talk_control(Client, DCC_TALK_DELETE_SUMMON);
   dcc_erase(Client);
}

int    
send_talk_control(Client, MessageType)
	DCC_list	*Client;
	int     MessageType;
{
   CTL_MSG	Message;
   CTL_RESPONSE	Response;
   static	long	 SeqNum = 0;
   struct	sockaddr_in *sin;
   struct	sockaddr_in SockAddr;
   struct	timeval time_out;
   fd_set	selset;
   int	i;
   int	dummy;
   
   Message.vers = TALK_VERSION;
   Message.id_num = htonl(SeqNum);
   SeqNum++; /* Not in the htonl because on some machines it's a macro */
   dummy = sizeof(SockAddr);
   getsockname(Client->file, (struct sockaddr *) &SockAddr, &dummy);
   Message.ctl_addr = (*(struct sockaddr *) &SockAddr);
   if (Client->read > 0)
     {
	getsockname(Client->read, (struct sockaddr *) &SockAddr,
		    &dummy);
	SockAddr.sin_addr=MyHostAddr;
     }
   Message.addr = (*(struct sockaddr *) &SockAddr);
   my_strncpy(Message.l_name, username, NAME_SIZE);
   Message.l_name[NAME_SIZE - 1] = '\0';
   my_strncpy(Message.r_name, Client->user, NAME_SIZE);
   Message.r_name[NAME_SIZE - 1] = '\0';
   Message.r_tty[0] = '\0';
   Message.pid = getpid();
   SockAddr.sin_addr = Client->remote;
   SockAddr.sin_port = htons(518);
   switch(MessageType)
     {
      case DCC_TALK_CHECK:
	Message.type = LOOK_UP;
	break;
      case DCC_TALK_INVITE:
	Message.type = LEAVE_INVITE;
	SockAddr.sin_addr = MyHostAddr;
	break;
      case DCC_TALK_ANNOUNCE:
	Message.type = ANNOUNCE;
	break;
      case DCC_TALK_DELETE_LOCAL:
	SockAddr.sin_addr = MyHostAddr;
      case DCC_TALK_DELETE_REMOTE:
	Message.type = DELETE;
	break;
      case DCC_TALK_SUMMON:
	my_strcpy(Message.l_name, "I:");
	my_strcat(Message.l_name, get_server_nickname(from_server));
	Message.type = ANNOUNCE;
	break;
      case DCC_TALK_DELETE_SUMMON:
	my_strcpy(Message.l_name, "I:");
	my_strcat(Message.l_name, get_server_nickname(from_server));
	Message.type = DELETE;
	break;
     }
   
   /* this fixes some broken stuff */
   sin = (struct sockaddr_in *)&(Message.ctl_addr);
   sin->sin_family = htons(AF_INET);
   sin = (struct sockaddr_in *)&(Message.addr);
   sin->sin_family = htons(AF_INET);
   
   for (i = 0; i < 3; i++)
     {
	if (sendto(Client->file, (char *) &Message, sizeof(Message), 0,
		   (struct sockaddr *) &SockAddr,
		   sizeof(SockAddr))!=sizeof(Message))
	  {
	     perror("sendto");
	     return 0;
	  }
	time_out.tv_sec = 10;
	time_out.tv_usec = 0;
	FD_ZERO(&selset);
	FD_SET(Client->file, &selset);
	switch(select(Client->file+1, &selset, NULL, NULL, &time_out))
	  {
	   case -1:
	     perror("select");
	     return 0;
	   case 1:
	     do
	       {
		  recv(Client->file, (char *) &Response, sizeof(Response), 0);
		  FD_ZERO(&selset);
		  FD_SET(Client->file, &selset);
		  time_out.tv_sec = 0;
		  time_out.tv_usec = 0;
	       }
	     while (select(Client->file + 1, &selset, NULL, NULL,
			   &time_out) > 0);
	     if (Response.type != Message.type)
	       continue;
	     if (LOOK_UP == Response.type &&
		 SUCCESS == Response.answer)
	       {
		  SockAddr = (*(struct sockaddr_in *)
			      &Response.addr);
		  Client->remote = SockAddr.sin_addr;
		  Client->remport = ntohs(SockAddr.sin_port);
	       }
	     return Response.answer + 1;
	  }
     }
   return 0;
}

/*
 * this is not used by Ninja IRC
 * 
static	void
dcc_filesend(args)
	u_char	*args;
{
	u_char	*user;
	u_char	*filename,
		*fullname;
	DCC_list *Client;
	u_char	FileBuf[BIG_BUFFER_SIZE+1];
	struct	stat	stat_buf;

#ifdef  DAEMON_UID
	if (DAEMON_UID == getuid())
	{
		put_info("You are not permitted to use DCC to exchange files");
		return;
	}
#endif / * DAEMON_UID * /
	if (0 == (user = next_arg(args, &args)) ||
	    0 == (filename = next_arg(args, &args)))
	{
		put_info("You must supply a nickname and filename for DCC SEND");
		return;
	}
	if (IS_ABSOLUTE_PATH(filename))
	{
		my_strcpy(FileBuf, filename);
	}
	else if (*filename == '~')
	{
		if (0 == (fullname = expand_twiddle(filename)))
		{
			put_error("Unable to expand %s", filename);
			return;
		}
		my_strcpy(FileBuf, fullname);
		dma_Free(&fullname);
	}
	else
	{
		getcwd(CP(FileBuf), sizeof(FileBuf));
		my_strcat(FileBuf, "/");
		my_strcat(FileBuf, filename);
	}
	if (0 != access(CP(FileBuf), R_OK))
	{
		put_error("Cannot access %s", FileBuf);
		return;
	}
	stat_file(CP(FileBuf), &stat_buf);
/ * some unix didn't have this ???? * /
#ifdef S_IFDIR
	if (stat_buf.st_mode & S_IFDIR)
	{
		put_error("Cannot send a directory");
		return;
	}
#endif / * S_IFDER * /
	if (scanstr(FileBuf, UP("/etc/")))
	{
		put_error("Send request rejected");
		return;
	}
	if ((int) my_strlen(FileBuf) >= 7 && 0 == my_strcmp(FileBuf + my_strlen(FileBuf) - 7, "/passwd"))
	{
		put_error("Send request rejected");
		return;
	}
	filesize = stat_buf.st_size;
	Client = dcc_searchlist(FileBuf, user, DCC_FILEOFFER, 1, filename, filesize);
	if ((Client->file = open(CP(Client->description), O_RDONLY | O_BINARY)) == -1)
	{
		put_info("Unable to open %s: %s\n", Client->description,
			errno ? strerror(errno) : "Unknown Host");
		new_close(Client->read);
		Client->read = Client->write = (-1);
		Client->flags |= DCC_DELETE;
		return;
	}
	filesize = 0;
	if ((Client->flags & DCC_ACTIVE) || (Client->flags & DCC_WAIT))
	{
		put_info("A previous DCC SEND:%s to %s exists", FileBuf, user);
		return;
	}
	Client->flags |= DCC_TWOCLIENTS;
	dcc_open(Client);
}
 */

static	void
dcc_getfile(args)
	u_char	*args;
{
   u_char	*user;
   u_char	*filename;
   DCC_list	*Client;
   int		number;
   
#ifdef  DAEMON_UID
   if (DAEMON_UID == getuid())
     {
	put_error("DCC: You are not permitted to exchange files.");
	return;
     }
#endif /* DAEMON_UID */
   if (0 == (user = next_arg(args, &args)))
     {
	/* put_info("You must supply a nickname for DCC GET");
	 */
	usage("dcc get", "<#>, or *, or <nick> *, or <nick> [<file 1> .. <file n>]");
	return;
     }
   
   /* get by number */
   if (isdigit(*user) && (number = my_atoi(user)) != 0)
     {
	dcc_getbynumber(number);
	return;
     }
   else if (*user == '*')
     {
	dcc_getall(UNULL);
	return;
     }
   filename = next_arg(args, &args);
   if (filename && *filename == '*')
     {
	dcc_getall(user);
	return;
     }
   while (1)
     {
	if (0 == (Client = dcc_searchlist(filename, user, DCC_FILEREAD, 0, (u_char *) 0, 0)))
	  {
	     if (filename)
	       put_error("DCC: %s isn't trying to send you \"%s\".", user, filename);
	     else
	       {
		  put_error("DCC: %s isn't trying to send you anything.", user);
		  return;
	       }
	  }
	else
	  dcc_getbyptr(Client);
	filename = next_arg(args, &args);
	if (!filename)
	  return;
     }
}

void
register_dcc_offer(user, type, description, address, port, size)
	u_char	*user;
	u_char	*type;
	u_char	*description;
	u_char	*address;
	u_char	*port;
	u_char	*size;
{
   DCC_list	*Client;
   int	CType;
   u_char	*c;
   u_long	TempLong;
   unsigned	TempInt;
   int	do_auto = 0;	/* used in dcc chat collisions */
   u_char	*cmd = (u_char *) 0;
   int	lastlog_level;
   /* ninja extensions vars */
   Friend *friend = NULL;
   off_t filesize;
   
   lastlog_level = set_lastlog_msg_level(LOG_DCC);
   if (0 != (c = my_rindex((description), '/')))
     description = c + 1;
   if ('.' == *description)
     *description = '_';
   if (size && *size)
     filesize = my_atol(size);  /* atoi() isn't big enuff */
   else
     filesize = 0;
   dma_strcpy(&cmd, type);
   upper(cmd);
   if (!my_strcmp(cmd, "CHAT"))
     CType = DCC_CHAT;
#ifndef  DAEMON_UID
   else if (!my_strcmp(cmd, "SEND"))
#else
   else if (!my_strcmp(cmd, "SEND") && DAEMON_UID != getuid())
#endif /* DAEMON_UID */
       CType = DCC_FILEREAD;
   else if (!my_strcmp(cmd, "RESUME"))
     {
	dcc_getfile_resume_demanded(user, description, address, port);
	goto out;
     }
   else if (!my_strcmp(cmd, "ACCEPT"))
     {
	dcc_getfile_resume_start(user, description, address, port);
	goto out;
     }
   else
     {
	put_error("DCC: Unknown request: %s (%s) received from %s", type, description, user);
	goto out;
     }
   Client = dcc_searchlist(description, user, CType, 1, (u_char *) 0, filesize);
   filesize = 0;
   if (Client->flags & DCC_WAIT)
     {
	dcc_erase(Client);
	if (DCC_CHAT == CType)
	  {
	     Client = dcc_searchlist(description, user, CType, 1, (u_char *) 0, 0);
	     do_auto = 1;
	  }
	else
	  {
	     put_error("DCC: %s collision for %s:%s", type, user, description);
	     send_ctcp_reply(user, UP("DCC"), "DCC: %s collision occured while connecting to %s (%s)", type, nickname,
			     description);
	     goto out;
	  }
     }
   if (Client->flags & DCC_ACTIVE)
     {
	put_error("DCC: Received %s request from %s while previous session still active.", type, user);
	goto out;
     }
   Client->flags |= DCC_OFFER;
   sscanf(CP(address), "%lu", &TempLong);
   Client->remote.s_addr = htonl(TempLong);
   sscanf(CP(port), "%u", &TempInt);
   Client->remport = TempInt;
   if (TempInt < 1024)
     {
	put_error("DCC: %s (%s) request from %s rejected [port = %d]", type, description, user, TempInt);
	dcc_erase(Client);
	goto out;
     }
   if ((u_long) 0 == TempLong || 0 == Client->remport)
     {
	put_error("DCC: %s (%s) request from %s ignored because it had a null port or address.",
		  type, description, user);
	dcc_erase(Client);
	goto out;
     }
   if (do_auto)
     {
	put_info("DCC: CHAT already requested by %s, connecting...", user);
	dcc_chat(user);
     }
   /*
    * XXX
    * should make the case below for the filesize into
    * generic off_t2str() function, or something.  this
    * cast is merely a STOP-GAP measure.
    */
   else if (Client->filesize)
     {	
	u_char mfmt[] = "DCC: %s (%s %s) request received from %s[%s:%s]";

	put_info(mfmt, type, description, ninja_size(Client->filesize), 
		 user, ninja_host(Client->remote), port);
	/* if we are away, log it */
	if (server_list[from_server].away_set)
	  add_to_awaylog(mfmt, type, description, ninja_size(Client->filesize),
			 user, ninja_host(Client->remote), port);
	
	/* is this coming from a friend? */
	friend = check_friend_autoget(user, FromUser, FromHost);
	/* if we're autogetting from this friend or we are autogetting from everyone,
	 * do the autoget stuff..
	 */
	if (get_int_var(DCC_AUTOGET_VAR) || friend)
	  {
	     u_char *ffn, dbuf[1024];
	     struct stat sb;
	     int statret;
	     
	     getcwd(dbuf, sizeof(dbuf)-1);
	     dbuf[sizeof(dbuf)-1] = '\0';
	     ffn = dma_Malloc(my_strlen(dbuf) + my_strlen(description) + 2);
	     if (!ffn)
	       {
		  put_error("DCC: can't allocate memory for checking for local file");
		  goto out;
	       }
	     statret = stat(ffn, &sb);
	     dma_Free(&ffn);
	     if (statret == -1)
	       {
		  if (friend)
		    {
		       u_char fmt[] = "DCC: Automatically getting \"%s\" from your friend, %s(%s)!";
		       
		       put_info(fmt, description, user, friend->nick);
		       if (server_list[from_server].away_set)
			 add_to_awaylog(fmt, description, user, friend->nick);
		    }
		  else
		    {
		       u_char fmt[] = "DCC: Automatically getting \"%s\" from %s";
		       
		       put_info(fmt, description, user);
		       if (server_list[from_server].away_set)
			 add_to_awaylog(fmt, description, user);
		    }
		  dcc_getbyptr(Client);
	       }
	     else
	       {
		  if (Client->filesize > sb.st_size)
		    {
		       u_char gfargs[256];
		       
		       if (friend)
			 {
			    u_char fmt[] = "DCC: Automatically resuming transfer of \"%s\" at %s from your friend, %s(%s)!";
			    
			    put_info(fmt, description, ninja_size(sb.st_size), user, friend->nick);
			    if (server_list[from_server].away_set)
			      add_to_awaylog(fmt, description, ninja_size(sb.st_size), user, friend->nick);
			 }
		       else
			 {
			    u_char fmt[] = "DCC: Automatically resuming transfer of \"%s\" at %s.";
			    
			    put_info(fmt, description, ninja_size(sb.st_size));
			    if (server_list[from_server].away_set)
			      add_to_awaylog(fmt, description, ninja_size(sb.st_size));
			 }
		       /* XXX nasty hack! */
		       snprintf(CP(gfargs), sizeof(gfargs)-1, "%s %s", user, description);
		       gfargs[sizeof(gfargs)-1] = '\0';
		       dcc_getfile_resume(gfargs);
		    }
		  else
		    {
		       u_char fmt[] = "DCC: Automatically closing \"%s\" because it exists.";
		       
		       put_info(fmt, description);
		       if (server_list[from_server].away_set)
			 add_to_awaylog(fmt, description);
		       dcc_closebyptr(Client);
		    }
	       }
	  }
	else
	  {
	     u_char gfargs[256];
	     
	     /* XXX nasty hack! */
	     snprintf(CP(gfargs), sizeof(gfargs)-1, "%s %s", user, description);
	     gfargs[sizeof(gfargs)-1] = '\0';
	     add_tab_key(1, "DCC GET ", gfargs);
	  }
     }
   else
     {
	u_char mfmt[] = "DCC: %s (%s) request received from %s[%s:%s]";
	
	put_info(mfmt, type, description,
		 user, ninja_host(Client->remote), port);
	if (server_list[from_server].away_set)
	  add_to_awaylog(mfmt, type, description,
			 user, ninja_host(Client->remote), port);
	
	/* is this coming from a friend? */
	friend = check_friend_autoget(user, FromUser, FromHost);
	/* if we're autogetting from this friend or we are autogetting from everyone,
	 * do the autoget stuff..
	 */
	if (get_int_var(DCC_AUTOGET_VAR) || friend)
	  {
	     if (friend)
	       {
		  u_char fmt[] = "DCC: Automatically accepting chat from your friend, %s(%s)!";
		  
		  put_info(fmt, user, friend->nick);
		  if (server_list[from_server].away_set)
		    add_to_awaylog(fmt, user, friend->nick);
	       }
	     else
	       {
		  u_char fmt[] = "DCC: Automatically accepting chat from %s";
		  
		  put_info(fmt, user);
		  if (server_list[from_server].away_set)
		    add_to_awaylog(fmt, user);
	       }
	     dcc_chat(user);
	  }
	else
	  add_tab_key(1, "DCC CHAT ", user);
     }
out:
   set_lastlog_msg_level(lastlog_level);
   dma_Free(&cmd);
}

static	void
process_incoming_chat(Client)
	DCC_list	*Client;
{
   struct	sockaddr_in	remaddr;
   int	sra;
   u_char	tmp[BIG_BUFFER_SIZE+1];
   u_char	tmpuser[IRCD_BUFFER_SIZE];
   u_char	*s, *bufptr;
   long	bytesread;
   int	old_timeout;
   size_t	len;
   
   save_message_from();
   message_from(Client->user, LOG_DCC);
   if (Client->flags & DCC_WAIT)
     {
	sra = sizeof(struct sockaddr_in);
	Client->write = accept(Client->read, (struct sockaddr *)
			       &remaddr, &sra);
#if defined(ESIX) || defined(_Windows)
	mark_socket(Client->write);
#endif /* ESIX */
	new_close(Client->read);
	Client->read = Client->write;
	Client->flags &= ~DCC_WAIT;
	Client->flags |= DCC_ACTIVE;
	put_info("DCC: chat connection to %s[%s:%d] established.", Client->user,
		 ninja_host(remaddr.sin_addr), ntohs(remaddr.sin_port));
	add_tab_key(0, "msg =", Client->user);
	Client->starttime = time(NULL);
	goto out;
     }
   s = Client->buffer;
   bufptr = tmp;
   if (s && *s)
     {
	len = my_strlen(s);
	my_strncpy(tmp, s, len);
	bufptr += len;
     }
   else
     len = 0;
   old_timeout = dgets_timeout(1);
   bytesread = dgets(bufptr, (int)((BIG_BUFFER_SIZE/2) - len), Client->read, (u_char *) 0);
   (void) dgets_timeout(old_timeout);
   switch (bytesread)
     {
      case -1:
	add_to_dcc_buffer(Client, bufptr);
	if (Client->buffer && (my_strlen(Client->buffer) > BIG_BUFFER_SIZE/2))
	  {
	     dma_Free(&Client->buffer);
	     put_error("DCC: dropped long CHAT message from %s", Client->user);
	  }
	break;
      case 0:
	put_error("DCC: CHAT connection to %s lost: %s", Client->user, dgets_errno == -1 ? "Remote end closed connection" : strerror(dgets_errno));
	new_close(Client->read);
	Client->read = Client->write = -1;
	Client->flags |= DCC_DELETE;
	break;
      default:
	dma_Free(&Client->buffer);
	len = my_strlen(tmp);
	if (len > BIG_BUFFER_SIZE/2)
	  len = BIG_BUFFER_SIZE/2;
	Client->bytes_read += len;
	*tmpuser = '=';
	strmcpy(tmpuser+1, Client->user, IRCD_BUFFER_SIZE-2);
	s = do_ctcp(tmpuser, nickname, tmp);
	s[my_strlen(s) - 1] = '\0';	/* remove newline */
	if (s && *s)
	  {
	     s[BIG_BUFFER_SIZE/2-1] = '\0';	/* XXX XXX: stop dcc long messages, stupid but "safe"? */
	     if (do_hook(DCC_CHAT_LIST, "%s %s", Client->user, s))
	       {
		  if (awayfile != NULL)
		    add_to_awaylog("=%s= %s", Client->user, s);
		  put_it("=%s= %s", Client->user, s);
	       }
	     add_tab_key(0, "msg =", Client->user);
	     if (beep_on_level & LOG_DCC)
	       {
		  if (away_set)
		    beep_em(get_int_var(BEEP_WHEN_AWAY_VAR));
		  else
		    beep_em(1);
	       }
	  }
     }
out:
   restore_message_from();
}

static	void
process_incoming_listen(Client)
	DCC_list	*Client;
{
   struct	sockaddr_in	remaddr;
   int	sra;
   u_char	FdName[10];
   DCC_list	*NewClient;
	int	new_socket;
   struct	hostent	*hp;
   u_char	*Name;
   
   sra = sizeof(struct sockaddr_in);
   new_socket = accept(Client->read, (struct sockaddr *) &remaddr,
		       &sra);
   if (0 != (hp = gethostbyaddr((char *)&remaddr.sin_addr,
				sizeof(remaddr.sin_addr), remaddr.sin_family)))
     Name = (u_char *)hp->h_name;
   else
     Name = (u_char *)inet_ntoa(remaddr.sin_addr);
#if defined(ESIX) || defined(_Windows)
   mark_socket(new_socket);
#endif /* ESIX */
   sprintf(CP(FdName), "%d", new_socket); /* safe */
   NewClient = dcc_searchlist(Name, FdName, DCC_RAW, 1, (u_char *) 0, 0);
   NewClient->starttime = time((time_t *) 0);
   NewClient->read = NewClient->write = new_socket;
   NewClient->remote = remaddr.sin_addr;
   NewClient->remport = remaddr.sin_port;
   NewClient->flags |= DCC_ACTIVE;
   NewClient->bytes_read = NewClient->bytes_sent = 0L;
   save_message_from();
   message_from(NewClient->user, LOG_DCC);
   if (do_hook(DCC_RAW_LIST, "%s %s N %d", NewClient->user,
	       NewClient->description,
	       Client->write))
     put_info("DCC: RAW connection to %s on %s via %d established.",
	      NewClient->description,
	      NewClient->user,
	      Client->write);
   restore_message_from();
}

static	void
process_incoming_raw(Client)
	DCC_list	*Client;
{
   u_char	tmp[BIG_BUFFER_SIZE+1];
   u_char	*s, *bufptr;
   long	bytesread;
   int     old_timeout;
   size_t	len;
   
   save_message_from();
   message_from(Client->user, LOG_DCC);
   
   s = Client->buffer;
   bufptr = tmp;
   if (s && *s)
     {
	len = my_strlen(s);
	my_strncpy(tmp, s, len);
	bufptr += len;
     }
   else
     len = 0;
   old_timeout = dgets_timeout(1);
   switch(bytesread = dgets(bufptr, (int)((BIG_BUFFER_SIZE/2) - len), Client->read, (u_char *) 0))
     {
      case -1:
	add_to_dcc_buffer(Client, bufptr);
	if (Client->buffer && (my_strlen(Client->buffer) > BIG_BUFFER_SIZE/2))
	  {
	     dma_Free(&Client->buffer);
	     put_error("DCC: dropping long RAW message from %s", Client->user);
	  }
	break;
      case 0:
	if (do_hook(DCC_RAW_LIST, "%s %s C",
		    Client->user, Client->description))
	  put_error("DCC: RAW connection to %s on %s lost.",
		    Client->user, Client->description);
	new_close(Client->read);
	Client->read = Client->write = -1;
	Client->flags |= DCC_DELETE;
	(void) dgets_timeout(old_timeout);
	break;
      default:
	dma_Free(&Client->buffer);
	len = my_strlen(tmp);
	if (len > BIG_BUFFER_SIZE / 2)
	  len = BIG_BUFFER_SIZE / 2;
	tmp[len - 1] = '\0';
	Client->bytes_read += len;
	if (do_hook(DCC_RAW_LIST, "%s %s D %s",
		    Client->user, Client->description, tmp))
	  put_info("DCC: RAW data on %s from %s: %s",
		   Client->user, Client->description, tmp);
	(void) dgets_timeout(old_timeout);
     }
   restore_message_from();
}

static	void
process_incoming_talk(Client)
	DCC_list	*Client;
{
   struct	sockaddr_in	remaddr;
   int	sra;
   u_char	tmp[BIG_BUFFER_SIZE+1];
   u_char	*s, *bufptr;
   long	bytesread;
   int     old_timeout;
   size_t	len;
   
   save_message_from();
   message_from(Client->user, LOG_DCC);
   if (Client->flags & DCC_WAIT)
     {
	sra = sizeof(struct sockaddr_in);
	Client->write = accept(Client->read, (struct sockaddr *)
			       &remaddr, &sra);
#if defined(ESIX) || defined(_Windows)
	mark_socket(Client->write);
#endif /* ESIX */
	new_close(Client->read);
	Client->read = Client->write;
	Client->flags &= ~DCC_WAIT;
	Client->flags |= DCC_ACTIVE;
	send_talk_control(Client, DCC_TALK_DELETE_LOCAL);
	new_close(Client->file);
	send(Client->write, "\010\025\027", 3, 0);
	recv(Client->read, CP(Client->talkchars), 3, 0);
	Client->bytes_read = Client->bytes_sent = 3;
	put_info("DCC: TALK connection to %s@%s:%d established.", Client->user,
		 ninja_host(remaddr.sin_addr), ntohs(remaddr.sin_port));
	goto out;
     }
   s = Client->buffer;
   bufptr = tmp;
   if (s && *s)
     {
	len = my_strlen(s);
	my_strncpy(tmp, s, len);
	bufptr += len;
     }
   else
     len = 0;
   old_timeout = dgets_timeout(1);
   switch(bytesread = dgets(bufptr, (int)len - (BIG_BUFFER_SIZE/2), Client->read,
			    Client->talkchars))
     {
      case -2:
	return;
      case -1:
	add_to_dcc_buffer(Client, bufptr);
	if (Client->buffer && (my_strlen(Client->buffer) > BIG_BUFFER_SIZE/2))
	  {
	     dma_Free(&Client->buffer);
	     put_info("DCC: dropping long TALK message from %s", Client->user);
	     goto out;
	  }
	break;
      case 0:
	put_info("DCC: TALK connection to %s lost", Client->user);
	new_close(Client->read);
	Client->read=Client->write = -1;
	Client->flags |= DCC_DELETE;
	(void) dgets_timeout(old_timeout);
	break;
      default:
	dma_Free(&Client->buffer);
	len = my_strlen(tmp);
	if (len > BIG_BUFFER_SIZE/2)
	  len = BIG_BUFFER_SIZE/2;
	tmp[len + 1] = '\0';
	Client->bytes_read += len;
	if (awayfile)
	  add_to_awaylog("@%s@ %s", Client->user, tmp);
	if (do_hook(TALK_LIST, "%s %s", Client->user, tmp))
	  put_it("@%s@ %s", Client->user, tmp);
	add_tab_key(0, "msg @", Client->user);
	(void) dgets_timeout(old_timeout);
     }
out:
   restore_message_from();
}

static	void
process_outgoing_file(Client)
	DCC_list	*Client;
{
   struct	sockaddr_in	remaddr;
   int	sra;
   u_char	tmp[8193];
   u_32int	bytesrecvd;
   int	bytesread;
   int	BlockSize;
   
   save_message_from();
   message_from(Client->user, LOG_DCC);
   if (Client->flags & DCC_WAIT)
     {
	sra = sizeof(struct sockaddr_in);
	Client->write = accept(Client->read,
			       (struct sockaddr *) &remaddr, &sra);
#if defined(ESIX) || defined(_Windows)
	mark_socket(Client->write);
#endif /* ESIX */
	new_close(Client->read);
	Client->read = Client->write;
	Client->flags &= ~DCC_WAIT;
	Client->flags |= DCC_ACTIVE;
	/* this will break resuming
	 *
	Client->bytes_sent = 0L;
	 */
	Client->starttime = time(NULL);
	if (Client->bytes_sent)
	  put_info("DCC: SEND connection to %s[%s:%d] established, resuming at %s.", Client->user,
		   ninja_host(remaddr.sin_addr), ntohs(remaddr.sin_port), ninja_size(Client->bytes_sent));
	else
	  put_info("DCC: SEND connection to %s[%s:%d] established.", Client->user,
		   ninja_host(remaddr.sin_addr), ntohs(remaddr.sin_port));
	dcc_send_count++;
	/* open file file!!! */
	if ((Client->file = open(Client->description, O_RDONLY)) == -1)
	  {
	     put_error("DCC: Unable to open %s: %s", Client->description, errno ? strerror(errno) : "Unknown error");
	     close(Client->read);
	     Client->read = Client->write = (-1);
	     Client->flags |= DCC_DELETE;
	     return;
	  }
	/* if we're resuming seek to the resume offset.. */
	if (Client->bytes_sent)
	  lseek(Client->file, Client->bytes_sent, SEEK_SET);
     }
   else 
     { 
	if ((bytesread = recv(Client->read, (char *) &bytesrecvd, sizeof(u_32int), 0)) < sizeof(u_32int))
	  {
#ifdef _Windows
	     int	recv_error;
	     
	     recv_error = WSAGetLastError();
	     if (bytesread == -1 &&
		 recv_error == WSAEWOULDBLOCK ||
		 recv_error == WSAEINTR)
	       goto out;
#endif /* _Windows */
	     
	     put_error("DCC: SEND: of %s to %s lost: %s", strip_path(Client->description), Client->user, strerror(errno));
	     new_close(Client->read);
	     Client->read = Client->write = (-1);
	     Client->flags |= DCC_DELETE;
	     new_close(Client->file);
	     goto out;
	  }
	else
	  if (ntohl(bytesrecvd) != Client->bytes_sent)
	    goto out;
     }
   BlockSize = get_int_var(DCC_BLOCK_SIZE_VAR);
   if (BlockSize > sizeof(tmp)-1)
     BlockSize = sizeof(tmp)-1;
   else if (BlockSize < 16)
     BlockSize = 16;
   bytesread = read(Client->file, tmp, sizeof(tmp)-1);
   if (bytesread > 0)
     {
	send(Client->write, CP(tmp), (size_t)bytesread, 0);
	Client->bytes_sent += bytesread;
     }
   else if (bytesread < 0)
     {
	put_error("DCC: read() error from file: %s", strerror(errno));
        Client->flags |= DCC_DELETE;
     }
   else
     {
	/*
	 * We do this here because lame Ultrix doesn't let us
	 * call put_it() with a float.  Perhaps put_it() should
	 * be fixed properly, and this kludge removed ..
	 * sometime....  -phone jan, 1993.
	 */
	
	u_char	lame_ultrix[10];	/* should be plenty */
	time_t	xtime = time(NULL) - Client->starttime;
	double	sent = (double)Client->bytes_sent;
	/* ninja extensions */
	u_char	rate_unit[2] = "\0\0";

	/* resumed ? */
	sent -= (double)Client->resume_offset;
	/* bytes -> bytes/sec */
	if (xtime <= 0)
	  xtime = 1;
	sent /= (double)xtime;
	/* find the rate unit and change bytes/sec -> unit/sec */
	if (sent > 1048756)
	  {
	     sent /= 1048756.0;
	     rate_unit[0] = 'M';
	  }
	else if (sent > 1024)
	  {
	     sent /= 1024.0;
	     rate_unit[0] = 'k';
	  }
	
	sprintf(CP(lame_ultrix), "%.3g", sent);
	put_info("DCC: SEND of %s to %s completed in %s at %s%sB/s",
		 strip_path(Client->description), Client->user, ninja_etime(xtime), lame_ultrix, rate_unit);
	new_close(Client->read);
	Client->read = Client->write = -1;
	Client->flags |= DCC_DELETE;
	new_close(Client->file);
     }
out:
   restore_message_from();
}

static	void
process_incoming_file(Client)
	DCC_list	*Client;
{
   u_char	tmp[8193];
   u_32int	bytestemp;
   int	bytesread;
   int	BlockSize;
   
   BlockSize = get_int_var(DCC_BLOCK_SIZE_VAR);
   if (BlockSize > sizeof(tmp)-1)
     BlockSize = sizeof(tmp)-1;
   else if (BlockSize < 512)
     BlockSize = 512;
     
   if ((bytesread = recv(Client->read, CP(tmp), BlockSize, 0)) <= 0)
     {
	/*
	 * We do this here because lame Ultrix doesn't let us
	 * call put_it() with a float.  Perhaps put_it() should
	 * be fixed properly, and this kludge removed ..
	 * sometime....  -phone jan, 1993.
	 */
	
	u_char	lame_ultrix[10];        /* should be plenty */
	time_t	xtime = time(NULL) - Client->starttime;
	double	sent = (double)Client->bytes_read;
	/* ninja extensions */
	u_char	rate_unit[2] = "\0\0";

#ifdef _Windows
	  {
	     int	recv_error;
	     recv_error = WSAGetLastError();
	     if (bytesread == -1 &&
		 recv_error == WSAEWOULDBLOCK ||
		 recv_error == WSAEINTR)
	       return;
	  }
#endif /* _Windows */

#ifdef NON_BLOCKING_CONNECTS
	/* wait, recv() may fail if we couldn't connect! and non-blocking is on */
	if (bytesread == -1)
	  {
	     put_error("DCC: Unable to create connection: %s", errno ? strerror(errno) : "Unknown Host");
	     Client->flags |= DCC_DELETE;
	     return;
	  }
#endif
	
	/* resumed ? */
	sent -= (double)Client->resume_offset;
	/* bytes -> bytes/sec */
	if (xtime <= 0)
	  xtime = 1;
	sent /= (double)xtime;
	/* find rate unit and convert bytes -> unit */
	if (sent > 1048756)
	  {
	     sent /= 1048756.0;
	     rate_unit[0] = 'M';
	  }
	else if (sent > 1024)
	  {
	     sent /= 1024.0;
	     rate_unit[0] = 'k';
	  }

	sprintf(CP(lame_ultrix), "%.3g", sent);
	save_message_from();
	message_from(Client->user, LOG_DCC);
	put_info("DCC: GET of %s from %s completed in %s at %s%sB/s",
		 strip_path(Client->description), Client->user, ninja_etime(xtime), lame_ultrix, rate_unit);
	restore_message_from();
	new_close(Client->read);
	new_close(Client->file);
	Client->read = Client->write = (-1);
	Client->flags |= DCC_DELETE;
	return;
     }
   write(Client->file, tmp, (size_t)bytesread);
   Client->bytes_read += bytesread;
   bytestemp = htonl(Client->bytes_read);
   send(Client->write, (char *)&bytestemp, sizeof(u_32int), 0);
}

/* flag == 1 means show it.  flag == 0 used by redirect */

void
dcc_message_transmit(user, text, type, flag)
	u_char	*user;
	u_char	*text;
	int	type,
		flag;
{
   DCC_list	*Client;
   u_char	tmp[BIG_BUFFER_SIZE+1];
   u_char	nickbuf[128];
   u_char	thing = '\0';
   u_char	*host = (u_char *) 0;
   crypt_key	*key;
   u_char	*line = NULL;
   int	lastlog_level;
   int	list = 0;
   size_t	len;
   
   lastlog_level = set_lastlog_msg_level(LOG_DCC);
   switch(type)
     {
      case DCC_TALK:
	if ((host = my_index(user, '@')) != NULL)
	  *(host++) = '\0';
	thing = '@';
	list = SEND_TALK_LIST;
	break;
      case DCC_CHAT:
	host = UP("chat");
	thing = '=';
	list = SEND_DCC_CHAT_LIST;
	break;
      case DCC_RAW:
	host = next_arg(text, &text);
	if (!host)
	  {
	     put_error("DCC: No host specified for RAW.");
	     goto out1;
	  }
	break;
     }
   save_message_from();
   message_from(user, LOG_DCC);
   if (!(Client = dcc_searchlist(host, user, type, 0, (u_char *) 0, 0)) || !(Client->flags&DCC_ACTIVE))
     {
	put_error("DCC: No active %s:%s connection for %s.", dcc_types[type], host ? host : (u_char *) "<any>", user);
	goto out;
     }
#ifdef DCC_CNCT_PEND
   /*
    * XXX - should make this buffer
    * XXX - just for dcc chat ?  maybe raw dcc too.  hmm.
    */
   if (Client->flags & DCC_CNCT_PEND)
     {
	put_error("DCC: %s:%s to %s is still connecting...", dcc_types[type], host ? host : (u_char *) "<any>", user);
	goto out;
     }
#endif /* DCC_DCNT_PEND */
   strmcpy(tmp, text, BIG_BUFFER_SIZE);
   if (type == DCC_CHAT) {
      nickbuf[0] = '=';
      strmcpy(nickbuf+1, user, sizeof(nickbuf) - 2);
      
      if ((key = is_crypted(nickbuf)) == 0 || (line = crypt_msg(tmp, key, 1)) == 0)
	line = tmp;
      
      add_tab_key(0, "msg =", Client->user);
   }
   else if (type == DCC_TALK)
     add_tab_key(0, "msg @", Client->user);
   else
     line = tmp;
#ifdef HAVE_WRITEV
     {
	struct iovec iov[2];
	
	iov[0].iov_base = CP(line);
	iov[0].iov_len = len = my_strlen(line);
	iov[1].iov_base = "\n";
	iov[1].iov_len = 1;
	len++;
	(void)writev(Client->write, iov, 2);
     }
#else
   /* XXX XXX XXX THIS IS TERRIBLE! XXX XXX XXX */
# define CRYPT_BUFFER_SIZE (IRCD_BUFFER_SIZE - 50)    /* XXX XXX FROM: crypt.c XXX XXX */
   strmcat(line, "\n", (size_t)((line == tmp) ? BIG_BUFFER_SIZE : CRYPT_BUFFER_SIZE));
   len = my_strlen(line);
   (void)send(Client->write, line, len, 0);
#endif
   Client->bytes_sent += len;
   if (flag && type != DCC_RAW) {
      if (do_hook(list, "%s %s", Client->user, text))
	put_it("=> %c%s%c %s", thing, Client->user, thing, text);
   }
out:   
   restore_message_from();
out1:   
   set_lastlog_msg_level(lastlog_level);
   return;
}

void
dcc_chat_transmit(user,	text)
	u_char	*user;
	u_char	*text;
{
   dcc_message_transmit(user, text, DCC_CHAT, 1);
}

static	void
dcc_tmsg(args)
	u_char	*args;
{
   u_char	*user;
   
   if (!(user = next_arg(args, &args)))
     {
	int	lastlog_level;
	
	lastlog_level = set_lastlog_msg_level(LOG_DCC);
	/*
	 * ninja handles this a little differently
	 * 
	put_info("You must specify a connection for a DCC TMSG");
	 */
	usage("dcc tmsg", "<user> <message>");
	(void) set_lastlog_msg_level(lastlog_level);
	return;
     }
   dcc_message_transmit(user, args, DCC_TALK, 1);
}

static	void
dcc_send_raw(args)
	u_char	*args;
{
   u_char	*name;
   
   if (!(name = next_arg(args, &args)))
	{
	   int	lastlog_level;
	   
	   lastlog_level = set_lastlog_msg_level(LOG_DCC);
	   /*
	    * ninja handles this a little differently
	    *
	   put_info("No name specified for DCC RAW");
	    */
	   usage("dcc raw", "<name> <host> <message>");
	   (void) set_lastlog_msg_level(lastlog_level);
	   return;
	}
   dcc_message_transmit(name, args, DCC_RAW, 1);
}

/*
 * dcc_time: Given a time value, it returns a string that is in the
 * format of "hours:minutes:seconds month day year" .  Used by 
 * dcc_list() to show the start time.
 * 
 * -----------
 * NINJA DOES NOT USE THIS
 * ----------
 *
static	u_char	*
dcc_time(the_time)
	time_t	the_time;
{
   struct	tm	*btime;
   u_char	*buf;
   static	char	*months[] = 
     {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
     };
   
   btime = localtime(&the_time);
   buf = (u_char *) malloc(22);
   if (sprintf(CP(buf), "%-2.2d:%-2.2d:%-2.2d %s %-2.2d %d", btime->tm_hour,
	       btime->tm_min, btime->tm_sec, months[btime->tm_mon],
	       btime->tm_mday, btime->tm_year + 1900))
     return buf;
   else
     return empty_string;
}
 */

void
dcc_list(args)
	u_char	*args;
{
   DCC_list	*Client;
   /*
    * ircII's format
   static	char	*format = "%-7.7s %-9.9s %-10.10s %-20.20s %-8.8s %-8.8s %s";
   unsigned	flags;
    */
   int	lastlog_level;
   /* ninja formats */
   u_char file_fmt[128], ofmt[128], wait_fmt[128];
   int count = 1, arglen = MAX(10, current_screen->co - 70);
   
   /* send/get format */
   snprintf(file_fmt, sizeof(file_fmt)-1, "%%-2.2d %%-10.10s %%9.9s:%%-%d.%ds Up %%s, %%s%%%% at %%s %%sB/s", arglen, arglen);
   file_fmt[sizeof(file_fmt)-1] = '\0';
   
   /* chat/raw/etc format */
   snprintf(ofmt, sizeof(ofmt)-1, "%%-2.2d %%-10.10s %%9.9s:%%-%d.%ds Up %%s, %%s sent, %%s read.", arglen, arglen);
   ofmt[sizeof(ofmt)-1] = '\0';
   
   /* waiting/offered format */
   snprintf(wait_fmt, sizeof(wait_fmt)-1, "%%-2.2d %%-10.10s %%9.9s:%%-%d.%ds %%s", arglen, arglen);
   wait_fmt[sizeof(wait_fmt)-1] = '\0';
   
   lastlog_level = set_lastlog_msg_level(LOG_DCC);
   put_raw("## Type            With:Args%s Status", strfill(' ', MAX(6, arglen - 4)));

   for (Client = ClientList ; Client != NULL ; Client = Client->next, count++)
     {
	char outstr[256];

	if (Client->flags & DCC_WAIT)
	  snprintf(outstr, sizeof(outstr)-1, wait_fmt, count, dcc_types[Client->flags & DCC_TYPES], Client->user, strip_path(Client->description), "Waiting for connection...");
	else if (Client->flags & DCC_OFFER)
	  snprintf(outstr, sizeof(outstr)-1, wait_fmt, count, dcc_types[Client->flags & DCC_TYPES], Client->user, strip_path(Client->description), "Offered...");
#ifdef DCC_CNCT_PEND
	else if (Client->flags & DCC_CNCT_PEND)
	  snprintf(outstr, sizeof(outstr)-1, wait_fmt, count, dcc_types[Client->flags & DCC_TYPES], Client->user, strip_path(Client->description), "Connecting...");
#endif	
	else if (Client->flags & DCC_ACTIVE)
	  {
	     time_t eltime = (time(NULL) - Client->starttime);
	     
	     if ((Client->flags & DCC_TYPES) == DCC_FILEREAD || (Client->flags & DCC_TYPES) == DCC_FILEOFFER)
	       {
		  char perc[16], ratestr[16], rate_unit[2] = "\0\0";
		  double ratio, rate, bytes;
		  
		  if ((Client->flags & DCC_TYPES) == DCC_FILEOFFER)
		    bytes = (double)Client->bytes_sent;
		  else
		    bytes = (double)Client->bytes_read;
		  
		  /* calculate % done */
		  ratio = 100 * ((double)bytes / (double)Client->filesize);
		  
		  /* calculate the speed */
		  rate = (double)(bytes - Client->resume_offset);
		  rate /= (double)(eltime > 0 ? eltime : 1);
		  
		  /* convert bytes/sec -> units/sec and set rate_unit */
		  if (rate > 1048576)
		    {
		       rate /= 1048576.0;
		       rate_unit[0] = 'M';
		    }
		  else if (rate > 1024)
		    {
		       rate /= 1024.0;
		       rate_unit[0] = 'k';
		    }
		  
		  /* convert #'s to strings */
		  snprintf(perc, sizeof(perc)-1, "%.3g", ratio);
		  perc[sizeof(perc)-1] = '\0';
		  snprintf(ratestr, sizeof(ratestr)-1, "%.3g", rate);
		  ratestr[sizeof(ratestr)-1] = '\0';
		  
		  /* show the send or get line */
		  snprintf(outstr, sizeof(outstr)-1, file_fmt, count, dcc_types[Client->flags & DCC_TYPES], Client->user, strip_path(Client->description),
			  ninja_etime(eltime), perc, ratestr, rate_unit);
	       }
	     else /* some other active thing */
	       {
		  u_char *this_sucks = NULL;
		  dma_strcpy(&this_sucks, ninja_size(Client->bytes_sent));
		  snprintf(outstr, sizeof(outstr)-1, 
			   ofmt, count, dcc_types[Client->flags & DCC_TYPES], Client->user, 
			   strip_path(Client->description), ninja_etime(eltime),
			   this_sucks, ninja_size(Client->bytes_read));
		  dma_Free(&this_sucks);
	       }
	  }
	outstr[current_screen->co-2] = '\0';
	put_raw("%s", outstr);
     }
   (void) set_lastlog_msg_level(lastlog_level);
}

static	void
dcc_close(args)
	u_char	*args;
{
   DCC_list	*Client;
   /*
   unsigned	flags;
    */
   u_char	*Type;
   u_char	*user;
   u_char	*description;
   int	CType;
   u_char	*cmd = NULL;
   int	lastlog_level;
   /* ninja extension */
   int	number;

   lastlog_level = set_lastlog_msg_level(LOG_DCC);
   if (!(Type = next_arg(args, &args)))
     {
	/* put_info("you must specify a type and nick for DCC CLOSE"); */
	usage("dcc close", "*, <#>, or <type> <nick> [<filename>]");
	goto out;
     }
   if (*Type == '*')
     {
	dcc_closeall();
	goto out;
     }
   if (isdigit(*Type))
     {
	number = my_atoi(Type);
	if (number > 0 /* && number <= count_dccs() */ )
	  dcc_closebynumber(number);
	goto out;
     }
   if (!(user=next_arg(args, &args)))
     {
	usage("dcc close", "*, <#>, or <type> <nick> [<filename>]");
	goto out;
     }
   description = next_arg(args, &args);
   dma_strcpy(&cmd, Type);
   upper(cmd);
   for (CType = 0; dcc_types[CType] != NULL; CType++)
     if (!my_strcmp(cmd, dcc_types[CType]))
       break;
   dma_Free(&cmd);
   if (!dcc_types[CType])
     {
	put_error("DCC: Unknown type: %s", Type);
	goto out;
     }
   
   if (0 == (Client = dcc_searchlist(description, user, CType, 0, description, 0)))
     {
	put_error("DCC: No %s:%s to %s found.", Type,
		 description ? description : (u_char *) "<any>", user);
	goto out;
     }
   
   dcc_closebyptr(Client);
   /*
    * this should be dcc_closebyptr
    * 
	flags = Client->flags;
	if (flags & DCC_DELETE)
	  goto out;
	if ((flags & DCC_WAIT) || (flags & DCC_ACTIVE))
	  {
	     new_close(Client->read);
	     if (Client->file)
	       new_close(Client->file);
	  }
	put_info("DCC %s:%s to %s closed", Type,
		 description ? description : (u_char *) "<any>", user);
	dcc_erase(Client);
    */
out:
   (void) set_lastlog_msg_level(lastlog_level);
}

/* this depends on dcc_rename() setting loglevel */
static void
dcc_chat_rename(args)
	u_char	*args;
{
   DCC_list	*Client;
   u_char	*user;
   u_char	*temp;
   
   if (!(user = next_arg(args, &args)) || !(temp = next_arg(args, &args)))
     {
	/* put_error("DCC: You must specify a current CHAT connection, and a new name for it."); */
	usage("dcc rename -chat", "<user> <new name>");
	return;
     }
   if (dcc_searchlist(UP("chat"), temp, DCC_CHAT, 0, (u_char *) 0, 0))
     {
	put_error("DCC: You already have a CHAT connection with %s, unable to rename.", temp);
	return;
     }
   if ((Client = dcc_searchlist(UP("chat"), user, DCC_CHAT, 0, (u_char *) 0, 0)))
     {
	dma_Free(&(Client->user));
	dma_strcpy(&(Client->user), temp);
	put_info("DCC: CHAT connection with %s renamed to %s", user, temp);
     }
   else
     put_error("DCC: No CHAT connection with %s", user);
}


static	void
dcc_rename(args)
	u_char	*args;
{
   DCC_list	*Client;
   u_char	*user;
   u_char	*description;
   u_char	*newdesc;
   u_char	*temp;
   int	lastlog_level;
   
   lastlog_level = set_lastlog_msg_level(LOG_DCC);
   if ((user = next_arg(args, &args)) && my_strnicmp(user, UP("-chat"), my_strlen(user)) == 0)
     {
	dcc_chat_rename(args);
	goto out;
     }
   if (!user || !(temp = next_arg(args, &args)))
     {
	/* put_error("DCC: You must specify a nick and new filename for DCC RENAME"); */
	usage("dcc rename", "<nick> [<old file name>] <new file name>");
	goto out;
     }
   if ((newdesc = next_arg(args, &args)) != NULL)
     description = temp;
   else
     {
	newdesc = temp;
	description = NULL;
     }
   if ((Client = dcc_searchlist(description, user, DCC_FILEREAD, 0, (u_char *) 0, 0)))
     {
	if (!(Client->flags & DCC_OFFER))
	  {
	     put_error("DCC: Too late to rename that file");
	     goto out;
	  }
	dma_Free(&(Client->description));
	dma_strcpy(&(Client->description), newdesc);
	put_info("DCC: File %s from %s renamed to %s",
		 description ? description : (u_char *) "<any>", user, newdesc);
     }
   else
     put_error("DCC: No file %s from %s found",
	      description ? description : (u_char *) "<any>", user);
out:
   (void) set_lastlog_msg_level(lastlog_level);
}

/*
 * close_all_dcc:  We call this when we create a new process so that
 * we don't leave any fd's lying around, that won't close when we
 * want them to..
 */

void
close_all_dcc()
{
	DCC_list *Client;

	while ((Client = ClientList))
		dcc_erase(Client);
}

static	void
add_to_dcc_buffer(Client, buf)
	DCC_list	*Client;
	u_char	*buf;
{
	if (buf && *buf)
	{
		if (Client->buffer)
			dma_strcat(&Client->buffer, buf);
		else
			dma_strcpy(&Client->buffer, buf);
	}
}

void
dcc_getbynumber(number)
   int number;
{
   int i;
   DCC_list *dcctmp;

   for (i = 1, dcctmp = ClientList; dcctmp != NULL; dcctmp = dcctmp->next, i++)
      if (i == number)
	 break;
   if (dcctmp == NULL)
     {
	put_error("DCC: Session #%d does not exist!", number);
	return;
     }
   dcc_getbyptr(dcctmp);
}

void
dcc_getbyptr(Client)
   DCC_list *Client;
{
   if ((Client->flags & DCC_ACTIVE) || (Client->flags & DCC_WAIT))
     {
	put_info("DCC: You're already getting \"%s\" from %s.", Client->description, Client->user);
	return;
     }
   if (!(Client->flags & DCC_OFFER))
     {
	put_error("I'm little a teapot! *holy shit*");
	irc_exit("I'm little a teapot! *holy shit*");
/*
 * NOT REACHED
	dcc_erase(Client);
	return;
 */
     }
   Client->flags |= DCC_TWOCLIENTS;
   if (!dcc_open(Client))
      return;
   return;
}

void
dcc_getall(user)
    u_char *user;
{
   DCC_list *Client;

   for (Client = ClientList; Client != NULL; Client = Client->next)
     if (((Client->flags & DCC_TYPES) == DCC_FILEREAD) && (Client->flags & DCC_OFFER))
     {
	if (user == NULL)
	  dcc_getbyptr(Client);
	else if (my_stricmp(user, Client->user) == 0)
	  dcc_getbyptr(Client);
     }
}

/*
 * Usage: /DCC RESUME <nick> [file]
 */
void
dcc_getfile_resume(args)
   u_char *args;
{
   u_char *user;
   u_char *filename = NULL;
   DCC_list *Client;
   struct stat sb;

   if (!(user = next_arg(args, &args)))
     {
	usage("dcc resume", "<nickname> <filename>");
	return;
     }

   if (args && *args)
     {
	filename = next_arg(args, &args);
     }

   if (!(Client = dcc_searchlist(filename, user, DCC_FILEREAD, 0, NULL, 0)))
     {
	if (filename)
	   put_error("DCC: %s isn't trying to send you \"%s\".", user, filename);
	else
	   put_error("DCC: %s isn't trying to send you anything.", user);
	return;
     }

   if (stat(Client->description, &sb) == -1)
     {
	u_char tb1[512];
	
	put_error("DCC: You tried to resume a file that does not exist, starting from the beginning.");
	snprintf(tb1, sizeof(tb1)-1, "%s %s", user, filename ? filename : empty_string);
	tb1[sizeof(tb1)-1] = '\0';
	dcc_getfile(tb1);
	return;
     }

   if ((Client->flags & DCC_ACTIVE) || (Client->flags & DCC_WAIT))
     {
	put_error("DCC: A previous GET:%s from %s exists!", filename ? filename : UP("<any>"), user);
	return;
     }

   Client->bytes_sent = 0L;
   Client->bytes_read = sb.st_size;

   doing_privmsg = in_ctcp_flag = 0;
   send_ctcp(ctcp_type[in_ctcp_flag], user, "DCC", "RESUME %s %hd %d",
	     Client->description, Client->remport, sb.st_size);
}

/*
 * When the peer demands DCC RESUME
 * We send out a DCC ACCEPT
 */
void
dcc_getfile_resume_demanded(user, filename, port, offset)
    u_char *user, *filename, *port, *offset;
{
   DCC_list *Client;

   if ((Client = dcc_searchlist(filename, user, DCC_FILEOFFER, 0, port, 0)) == NULL || !offset)
     {
	put_error("DCC: Invalid resume of \"%s\" requested by %s", filename, user);
	return;
     }
   Client->bytes_sent = atoi(offset);
   Client->resume_offset = atoi(offset);
   Client->bytes_read = 0L;

   doing_privmsg = in_ctcp_flag = 0;

   send_ctcp(ctcp_type[in_ctcp_flag], user, "DCC", "ACCEPT %s %s %s", filename, port, offset);
}

/*
 * When we get the DCC ACCEPT
 * We start the connection
 */
void
dcc_getfile_resume_start(nick, filename, port, offset)
    u_char *nick, *filename, *port, *offset;
{
   DCC_list *Client;
   u_char *fullname = NULL;

   if (!(Client = dcc_searchlist(filename, nick, DCC_FILEREAD, 0, port, 0)))
      return;
   Client->flags |= DCC_TWOCLIENTS;
   
   /* dcc_open handles its own errors */
   if (!dcc_open(Client))
      return;
   
   /* if we have a tilde, expand the path, if it fails get out */
   if (*Client->description == '~')
     { 
	fullname = expand_twiddle(Client->description);
	if (!fullname)
	  {
	     put_error("Unable to expand %s", Client->description);
	     close(Client->read);
	     Client->read = -1;
	     Client->flags |= DCC_DELETE;
	     return;
	  }
     }
   else
     dma_strcpy(&fullname, Client->description);
   
   Client->resume_offset = atoi(offset);
   if (!(Client->file = open(fullname ? fullname : Client->description, O_WRONLY | O_APPEND, 0644)))
     {
	put_error("Unable to open %s: %s", Client->description, errno ? strerror(errno) : "<No Error>");
	close(Client->read);
	Client->read = -1;
	Client->flags |= DCC_DELETE;
     }
   if (fullname)
     dma_Free(&fullname);
}

/* this helps the code below... */
static u_char *gbl_dcc_sf_name = UNULL;
int
dcc_sf_selectent(entry)
   SCANDIR_A3ATYPE entry;
{
   /* no dir entries beginning with dots. 
    *
    * and must match static global gbl_dcc_sf_name
    * 
    */
   return (*(entry->d_name) != '.'
	   && (!gbl_dcc_sf_name
	       || (gbl_dcc_sf_name 
		   && cs_match(gbl_dcc_sf_name, UP(entry->d_name)))));
}

/*
 * dcc send argument processor.
 * this function takes a comma delimited list of nicknames any number of file masks
 * and scans for the file mask.  then sends each of the nicknames a SEND request for each of the files
 * it's nice to be able to specify masks, but it's not recommended to send * really.
 * it would be better to just package the stuff up and send the one file.. it's more work tho for a lazy
 * ass.
 */
void
dcc_sendfiles(args)
    u_char *args;
{
   u_char *user, *ptr, *filename = NULL, *fullname = NULL;
   u_char buf[1024], path[1024], buf2[1024];
   int numfiles = 0, n;
   off_t totsize = 0;
   struct stat stbuf;
   struct dirent **dirfiles;

#ifdef	DAEMON_UID
   if (DAEMON_UID == getuid())
     {
	put_error("You are not permitted to use DCC to exchange files");
	return;
     }
#endif
   buf2[0] = '\0';
   /* check to make sure we have arguments, if not, tell the usage. */
   if ((user = next_arg(args, &args)) == NULL || !(args && *args))
     {
	usage("dcc send", "<nick 1,nick 2,..,nick n> <mask 1> [<mask 2> .. <mask n>]");
	return;
     }
   save_message_from();
   message_from(user, LOG_DCC);
   /* this is currently rather broken..
    *  numfiles = ninja_chkfiles(&args, buf, &totsize);
    */
   while ((ptr = next_arg(args, &args)) != NULL)
     {
	if (strlen(ptr) < 1)
	   continue;
	if (*ptr == '/')
	   strncpy(path, ptr, sizeof(path)-1);
	else if (*ptr == '~')
	  {
	     if ((fullname = expand_twiddle(ptr)) == NULL)
		continue;
	     strncpy(path, fullname, sizeof(path));
	     dma_Free(&fullname);
	  }
	else
	  {
	     if (getcwd(path, sizeof(path)-1) == NULL)
	       {
		  put_error("DCC: Unable to get cwd: %s", strerror(errno));
		  restore_message_from();
		  return;
	       }
	     strmcat(path, "/", sizeof(path)-1);
	     strmcat(path, ptr, sizeof(path)-1);
	  }
	dma_strcpy(&fullname, path);
	if ((filename = rindex(path, '/')) != NULL)
	  *filename++ = '\0';
	memset(&stbuf, 0, sizeof(stbuf));
	if (stat_file(fullname, &stbuf) != -1)	/* we found it already */
	  {
	     if (stbuf.st_mode & S_IFDIR)
	       {
		  put_info("DCC: To send a directory, use %s/*", fullname);
		  dma_Free(&fullname);
		  continue;
	       }
	     if (dcc_filesend2(user, fullname) > 0)
	       {
		  numfiles++;
		  totsize += stbuf.st_size;
		  strmcat(buf2, strip_path(fullname), sizeof(buf2) - 1);
	       }
	     else
	       {
		  /* put_error("DCC: Unable to send \"%s\".", fullname); */
		  dma_Free(&fullname);
		  continue;
	       }
	  }
	else if (my_index(filename, '*')	/* not found.. check for wildcards */
		 || my_index(filename, '?'))
	  {
	     dma_strcpy(&gbl_dcc_sf_name, filename);
	     n = scandir(path, &dirfiles,
			 (int (*) _((SCANDIR_A3ATYPE))) dcc_sf_selectent,
			 (int (*) _((const void *, const void *)))compar);
	     dma_Free(&gbl_dcc_sf_name);
	     if (n < 0)
	       {
		  put_error("DCC: scandir() ran out of memory!");
		  dma_Free(&fullname);
		  continue;
	       }
	     else if (n == 0)	/* no files in the directory!! */
	       {
		  put_error("DCC: No files found matching \"%s\".", filename);
		  dma_Free(&fullname);
		  restore_message_from();
		  continue;
	       }
	     else
	       {
		  while (n--)
		    {
		       /*
			* this is already handled by our selectent.. 
			* 
		       if (wild_match(filename, dirfiles[n]->d_name))
			 {
			*/
			    snprintf(buf, sizeof(buf), "%s/%s", path, dirfiles[n]->d_name);
			    if (dcc_filesend2(user, buf) > 0)
			      {
				 numfiles++;
				 stat_file(buf, &stbuf);
				 totsize += stbuf.st_size;
				 if (buf2[0] != '\0')
				   strmcat(buf2, " ", sizeof(buf2) - 1);
				 strmcat(buf2, dirfiles[n]->d_name, sizeof(buf2) - 1);
			      }
			    else
			      {
				 /* put_error("DCC: Unable to send \"%s\".", buf); */
				 dma_Free(&fullname);
				 continue;
			      }
		       /*
			 }
			*/
		    }
	       }
	  }
	else	/* not found and no wildcards */
	  {
	     put_error("DCC: Unable to send file \"%s\": %s", filename, strerror(errno));
	     dma_Free(&fullname);
	     continue;
	  }
     }
   dma_Free(&fullname);
   if (numfiles)
     put_info("DCC: Sending %s %d file%s (%s) [%s]", user, numfiles, numfiles == 1 ? "" : "s", buf2,
	      ninja_size(totsize));
   /* else
    *  put_error("No files matching \"%s\" were found.", filename);
    */
   restore_message_from();
}

int
dcc_filesend2(u_char *user, u_char *filename)
{
   DCC_list *Client;
   struct stat stat_buf;
   int /* old_display, */ total = 0;
   u_char *ptr, *ptr2 = NULL, *free_ptr;
   u_char *errstr1 = UP("DCC: Unable to send file \"%s\": %s");
   u_char *errstr2 = UP("DCC: Unable to send file \"%s\" to \"%s\": %s");

   if (0 != access(filename, R_OK))
     {
	put_error(errstr1, strip_path(filename), strerror(errno));
	return -1;
     }
   stat_file(filename, &stat_buf);
   if (stat_buf.st_mode & S_IFDIR)
     {
	put_error(errstr1, strip_path(filename), "no directories allowed.");
	return -1;
     }
   /* make a copy becuz strsep modifies the string */
   dma_strcpy(&ptr2, user);
   free_ptr = ptr2;
   while ((ptr = my_strsep(&ptr2, ",")))
     {
	Client = dcc_searchlist(filename, ptr, DCC_FILEOFFER, 1, filename, stat_buf.st_size);
	if ((Client->flags & DCC_ACTIVE) || (Client->flags & DCC_WAIT))
	  {
	     put_error(errstr2, strip_path(filename), ptr, "you are already sending it.");
	     continue;
	  }
	Client->flags |= DCC_TWOCLIENTS;
	/* this is a nasty hack
	old_display = window_display;
	window_display = 0;
	 */
	dcc_open(Client);
	/*
	window_display = old_display;
	/ * end nasty hack */
	total++;
     }
   dma_Free(&free_ptr);
   return total;
}

void
dcc_closebynumber(number)
   int number;
{
   DCC_list *Client;
   int count;

   for (count = 1, Client = ClientList; Client != NULL; Client = Client->next, count++)
     {
	if (count == number)
	  {
	     dcc_closebyptr(Client);
	     return;
	  }
     }
   if (Client == NULL)
     {
	put_error("DCC: Session #%d doesn't exist!", number);
	return;
     }
}

void
dcc_closebyptr(Client)
   DCC_list *Client;
{
   char *type = dcc_types[Client->flags & DCC_TYPES];

   save_message_from();
   message_from(Client->user, LOG_DCC);
   if (Client->flags & DCC_DELETE)
     {
	put_error("DCC: %s session (%s) with %s is already marked for deletion.", type, strip_path(Client->description),
	    Client->user);
	restore_message_from();
	return;
     }
   put_info("DCC: %s session (%s) with %s closed.", type, strip_path(Client->description), Client->user);
   dcc_erase(Client);
   restore_message_from();
}

void
dcc_closeall(void)
{
   DCC_list *Client;

   for (Client = ClientList; Client != NULL; Client = Client->next)
      dcc_closebyptr(Client);
}
